"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserWebSocketNsp = exports.room = exports.userEventNames = void 0;
const enums_1 = require("../lib/access-model/v2/enums");
const uuid_1 = __importDefault(require("uuid"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const app_auth_1 = require("../app.auth");
const app_config_1 = require("../app.config");
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const dal_manager_1 = require("../dal/dal.manager");
const restapi_1 = require("../lib/es/models/restapi");
const terminalWebRtc_1 = require("../lib/es/models/terminalWebRtc");
const messageBroker_server_to_app_pub_1 = require("../messageBroker/messageBroker.server-to-app.pub");
const messageBroker_server_to_app_sub_1 = require("../messageBroker/messageBroker.server-to-app.sub");
const messageBroker_server_to_device_pub_1 = require("../messageBroker/messageBroker.server-to-device.pub");
exports.userEventNames = {
    connection: "connection",
    disconnect: "disconnect",
    activitytimeline: "activitytimeline",
    controlpanelhealth: "controlpanelhealth",
    organizationemergency: "organizationemergency",
    notification: "notification",
    terminalconnection: "terminalconnection",
    roomIn: "roomin",
    roomOut: "roomout",
    visitorListChanged: "visitorlistchanged",
    remoteAccessGrantsChanged: "remoteaccessgrantschanged",
};
exports.room = {
    systemStatus: "system",
    activity: "activity",
    visitor: "visitor",
};
class UserWebSocketNsp {
    constructor(nsp) {
        this._connectedUsers = {};
        this._nsp = nsp;
        this._nsp.clientsWithSessionId = {};
        this._nsp.use((socket, next) => {
            try {
                if (!socket.handshake.auth.token) {
                    socket.emit("WEBSOCKET_AUTHENTICATION_TOKEN_MISSING");
                    next(new Error("message: token missing"));
                    return;
                }
                else {
                    jsonwebtoken_1.default.verify(socket.handshake.auth.token, app_config_1.appConfig.jwtSecret);
                    next();
                }
            }
            catch (error) {
                socket.emit("WEBSOCKET_AUTHENTICATION_FAILED");
                next(error);
                return;
            }
        });
        this._nsp.on(exports.userEventNames.connection, async (socket) => {
            const allowedMaxConnectionPerUserSocket = app_config_1.appConfig.allowedMaxConnectionPerUserSocket;
            const userToken = new app_auth_1.UserJwtPayload(jsonwebtoken_1.default.decode(socket.handshake.auth.token));
            const organizationId = userToken.getOrganizationId();
            if (!organizationId) {
                app_logs_1.logger.warn(`Organization Id for token ${socket.handshake.query.token} at nsp connection is null! Socket connection is closing!`);
                app_logs_1.logger.warn(`Verified user token is: ${JSON.stringify(userToken, null, 4)}`);
                socket.emit("USER_AUTHENTICATION_TOKEN_MISSING_ORGANIZATION_ID");
                socket.disconnect();
                return;
            }
            const socketOrigin = (socket.handshake.query.socketOrigin ?? restapi_1.SocketOrigin.Mobile);
            if (!this._connectedUsers[userToken.userId]) {
                this._connectedUsers[userToken.userId] = {};
            }
            const userSocketList = this._connectedUsers[userToken.userId][socketOrigin];
            if (socketOrigin === restapi_1.SocketOrigin.Web) {
                if (userSocketList?.length + 1 > allowedMaxConnectionPerUserSocket) {
                    app_logs_1.logger.warn(`User[${userToken.userId}] has more than ${allowedMaxConnectionPerUserSocket} socket connection. Socket connection closed.`);
                    socket.emit("USER_HAS_TOO_MANY_CONNECTIONS");
                    socket.disconnect();
                    return;
                }
            }
            else if (socketOrigin === restapi_1.SocketOrigin.Mobile) {
                if (userSocketList?.length > 0) {
                    app_logs_1.logger.warn(`User[${userToken.userId}] already has a mobile socket connection. Socket connection closed.`);
                    socket.emit("USER_HAS_TOO_MANY_CONNECTIONS");
                    socket.disconnect();
                    return;
                }
            }
            socket.userFilters = [];
            const sessionId = uuid_1.default.v4();
            socket.sessionId = sessionId;
            socket.handshake.userToken = userToken;
            socket.organizationId = organizationId;
            if (this._connectedUsers[userToken.userId][socketOrigin]) {
                this._connectedUsers[userToken.userId][socketOrigin].push(sessionId);
            }
            else {
                this._connectedUsers[userToken.userId][socketOrigin] = [sessionId];
            }
            socket.nsp.clientsWithSessionId[sessionId] = socket;
            socket.amqpServerToAppSubs = [];
            const userSubscriber = new messageBroker_server_to_app_sub_1.AmqpServerToAppSub(socket, organizationId, [organizationId + ".*." + userToken.userId, organizationId]);
            await userSubscriber.init();
            socket.amqpServerToAppSubs.push(userSubscriber);
            socket.emit("ready");
            socket.on(exports.userEventNames.disconnect, async (reason) => {
                const socketRef = socket.nsp.clientsWithSessionId[sessionId];
                if (socketRef && socketRef.amqpServerToAppSubs) {
                    await Promise.all(socketRef.amqpServerToAppSubs.map(async (amqpChannel) => await amqpChannel.closeConnection()));
                    delete socketRef.amqpServerToAppSubs;
                }
                delete socket.nsp.clientsWithSessionId[sessionId];
                if (this._connectedUsers[userToken.userId][socketOrigin]) {
                    this._connectedUsers[userToken.userId][socketOrigin] = this._connectedUsers[userToken.userId][socketOrigin].filter((sessionId) => sessionId !== socket.sessionId);
                }
                else {
                    this._connectedUsers[userToken.userId][socketOrigin] = [];
                }
            });
            socket.on(exports.userEventNames.roomIn, async (organizationId, roomName, data) => {
                let bindingKeys = [];
                const roomType = roomName.split(".")[0];
                let isVisitor = false;
                switch (roomType) {
                    case exports.room.systemStatus:
                        bindingKeys = [organizationId + "." + exports.room.systemStatus + ".*"];
                        break;
                    case exports.room.activity:
                        data.recordReasons = data.recordReasons ? data.recordReasons.map((rr) => parseInt(rr)) : undefined;
                        socket.userFilters.push(data);
                        data.accessControlPointIds?.forEach((acpId) => {
                            bindingKeys.push(this.bindingKeyGenerator(organizationId, acpId, data));
                        });
                        break;
                    case exports.room.visitor:
                        bindingKeys = [organizationId + "." + exports.room.visitor + ".*"];
                        isVisitor = true;
                        break;
                    default:
                        app_logs_1.logger.error("Unknown room name");
                        throw new Error("Unknown room name");
                }
                const userSubscriber = socket.amqpServerToAppSubs.find((a) => a.organizationId === organizationId);
                if (userSubscriber) {
                    await userSubscriber.bindKeys(bindingKeys);
                }
                else {
                    app_logs_1.logger.warn("No connection for roomin request, new connection is establishing...");
                    const userSubscriber = new messageBroker_server_to_app_sub_1.AmqpServerToAppSub(socket, organizationId, [`${organizationId}.*.${userToken.userId}`]);
                    await userSubscriber.init();
                    socket.amqpServerToAppSubs.push(userSubscriber);
                    await userSubscriber.bindKeys(bindingKeys);
                }
                if (roomType === exports.room.activity) {
                    await userSubscriber.newActivitiyConnectionSendLogs(socket.userFilters[socket.userFilters.length - 1]);
                }
                if (isVisitor) {
                    this.getVisitorRegistrationPoint(organizationId, userToken.userId);
                }
            });
            socket.on(exports.userEventNames.roomOut, async (organizationId, roomName, data) => {
                let bindingKeys = [];
                const roomType = roomName.split(".")[0];
                switch (roomType) {
                    case exports.room.systemStatus:
                        bindingKeys = [organizationId + "." + exports.room.systemStatus + ".*"];
                        break;
                    case exports.room.activity:
                        const index = socket.userFilters.findIndex((uf) => uf.widgetId === data.widgetId);
                        socket.userFilters.splice(index, 1);
                        data.accessControlPointIds.forEach((acpId) => {
                            bindingKeys.push(this.bindingKeyGenerator(organizationId, acpId, data));
                        });
                        bindingKeys.filter((item, index) => bindingKeys.indexOf(item) !== index);
                        break;
                    case exports.room.visitor:
                        bindingKeys = [organizationId + "." + exports.room.visitor + ".*"];
                        break;
                    default:
                        app_logs_1.logger.error("Unknown room name");
                        throw new Error("Unknown room name");
                }
                const userSubscriber = socket.amqpServerToAppSubs.find((a) => a.organizationId === organizationId);
                if (userSubscriber) {
                    await userSubscriber.unbindKeys(bindingKeys);
                }
            });
            socket.on(terminalWebRtc_1.WEBRTC_OFFER, async (proxyTerminalId, sdp) => {
                const device = await dal_manager_1.dbManager.accessRedisCache.getDeviceStateCache({ organizationId, deviceId: proxyTerminalId });
                if (!device || device.i === false) {
                    socket.emit(terminalWebRtc_1.WEBRTC_ANSWER, proxyTerminalId, null);
                }
                else {
                    messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(proxyTerminalId, {
                        e: enums_1.EnumsV2.ServerToDeviceMessageType.WebRTC,
                        p: {
                            m: sdp,
                            t: "offer",
                            p: sessionId,
                            u: userToken.userId,
                        },
                    }, 10000);
                }
            });
            socket.on(terminalWebRtc_1.WEBRTC_NEW_ICE_CANDIDATE, async (proxyTerminalId, candidate) => {
                const device = await dal_manager_1.dbManager.accessRedisCache.getDeviceStateCache({ organizationId, deviceId: proxyTerminalId });
                if (!device || device.i === false) {
                    socket.emit(terminalWebRtc_1.WEBRTC_NEW_ICE_CANDIDATE, proxyTerminalId, false);
                }
                else {
                    messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(proxyTerminalId, {
                        e: enums_1.EnumsV2.ServerToDeviceMessageType.WebRTC,
                        p: {
                            m: candidate,
                            t: "new-ice-candidate",
                            p: sessionId,
                            u: userToken.userId,
                        },
                    }, 10000);
                }
            });
            socket.on(terminalWebRtc_1.WEBRTC_ANSWER, async (proxyTerminalId, answer) => {
                const device = await dal_manager_1.dbManager.accessRedisCache.getDeviceStateCache({ organizationId, deviceId: proxyTerminalId });
                if (!device || device.i === false) {
                    socket.emit(terminalWebRtc_1.WEBRTC_ANSWER, proxyTerminalId, false);
                }
                else {
                    messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(proxyTerminalId, {
                        e: enums_1.EnumsV2.ServerToDeviceMessageType.WebRTC,
                        p: {
                            m: answer,
                            t: "answer",
                            p: sessionId,
                            u: userToken.userId,
                        },
                    }, 10000);
                }
            });
            socket.on(terminalWebRtc_1.WEBRTC_HANG_UP, async (proxyTerminalId, hangup) => {
                const device = await dal_manager_1.dbManager.accessRedisCache.getDeviceStateCache({ organizationId, deviceId: proxyTerminalId });
                if (!device || device.i === false) {
                    socket.emit(terminalWebRtc_1.WEBRTC_HANG_UP, proxyTerminalId, false);
                }
                else {
                    messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(proxyTerminalId, {
                        e: enums_1.EnumsV2.ServerToDeviceMessageType.WebRTC,
                        p: {
                            m: hangup,
                            t: "hang-up",
                            p: sessionId,
                            u: userToken.userId,
                        },
                    }, 10000);
                }
            });
        });
    }
    bindingKeyGenerator(organizationId, acpId, data) {
        return (organizationId +
            "." +
            exports.room.activity +
            "." +
            acpId +
            "." +
            (data.direction === app_enums_1.enums.AccessDirection.Entrance ? data.direction : data.direction === app_enums_1.enums.AccessDirection.Exit ? data.direction : "*") +
            "." +
            (data.status === app_enums_1.enums.AccessReportFilterAccessResultType.Success ? true : data.status === app_enums_1.enums.AccessReportFilterAccessResultType.Fail ? false : "*"));
    }
    async getVisitorRegistrationPoint(organizationId, userId) {
        let data = await dal_manager_1.dbManager.accessVisitor.listRegistrationPointsDetailed(organizationId, userId);
        this.sendVisitorRegistrationPoint(organizationId, data);
    }
    sendControlPanelHealthItem(organizationId, data) {
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + exports.room.systemStatus + ".*", {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.controlpanelhealth,
            p: data,
        });
    }
    sendControlPanelConnectionStatus(organizationId, data) {
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + exports.room.systemStatus + ".*", {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.terminalconnection,
            p: data,
        });
    }
    sendRegionEmergencyNotification(params) {
        const receiverOnlineUserIds = params.receiverUserIds.filter((receiverUserId) => this._connectedUsers[receiverUserId]?.[restapi_1.SocketOrigin.Web]?.length > 0);
        receiverOnlineUserIds.forEach((userId) => {
            messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(params.organizationId + ".*." + userId, {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.regionemergency,
                p: {
                    s: params.state,
                    rid: params.regionId,
                    n: params.regionName,
                },
            });
        });
    }
    sendTerminalEmergencyNotification(params) {
        const receiverOnlineUserIds = params.receiverUserIds.filter((receiverUserId) => this._connectedUsers[receiverUserId]?.[restapi_1.SocketOrigin.Web]?.length > 0);
        receiverOnlineUserIds.forEach((userId) => {
            messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(params.organizationId + ".*." + userId, {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.terminalemergency,
                p: {
                    s: params.state,
                    did: params.deviceId,
                    n: params.deviceName,
                },
            });
        });
    }
    sendNewNotification(organizationId, userId, notification) {
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + ".*." + userId, {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.notification,
            p: notification,
        });
    }
    sendVisitorListChangedNotification(organizationId) {
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + exports.room.visitor + ".*", {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.visitorListChanged,
            p: null,
        });
    }
    sendRemoteAccessGrantsChanged(organizationId, userId) {
        if (userId) {
            messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + ".*.*", {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.remoteAccessGrantsChanged,
                p: null,
            });
        }
        else {
            messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + ".*." + userId, {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.remoteAccessGrantsChanged,
                p: null,
            });
        }
    }
    async sendRemoteStateChanged(organizationId, data) {
        const userIds = await dal_manager_1.dbManager.accessAccessControlPoint.getUsersHasRightWithAcpId(organizationId, data.accessControlPointId, { config: true });
        for (let userId of userIds) {
            messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + ".*." + userId, {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.accesscontrolpointstatechanged,
                p: data,
            });
        }
    }
    sendVisitorRegistrationPoint(organizationId, data) {
        data.forEach((element) => {
            messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + exports.room.visitor + ".*", {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.visitorRegistrationDeviceStatusChanged,
                p: {
                    deviceId: element.tabletDeviceId,
                    deviceName: element.deviceName,
                    registrationPointId: element.id,
                    isConnected: element.isConnected,
                },
            });
        });
    }
}
exports.UserWebSocketNsp = UserWebSocketNsp;
