"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmqpServerToTabletSub = void 0;
const app_logs_1 = require("../app.logs");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messagebroker_models_1 = require("./messagebroker.models");
class AmqpServerToTabletSub {
    constructor(socket, deviceId) {
        this.onMessage = async (msg) => {
            if (msg === null) {
                return;
            }
            try {
                let data = JSON.parse(msg.content.toString());
                let response = {
                    e: data.e === messagebroker_models_1.MessageBrokerModels.ServerToTabletMessageTypes.QrCodeRequired || messagebroker_models_1.MessageBrokerModels.ServerToTabletMessageTypes.EndVisitByQr
                        ? messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.QrCodeTimeout
                        : messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.ApprovalTimeout,
                    s: data.s,
                };
                try {
                    this._socket.send(JSON.stringify(data), (err) => {
                        if (err) {
                            throw err;
                        }
                    });
                    let callbackPromise = new Promise((resolve, reject) => {
                        this._rpcCallbackRegistry.set(data.s, resolve);
                        let wait = setTimeout(() => {
                            clearTimeout(wait);
                            reject("timeout");
                        }, 60 * 1000);
                    });
                    response = (await callbackPromise);
                }
                catch (error) {
                    app_logs_1.logger.error(error);
                }
                this._channel.sendToQueue(msg.properties.replyTo, response, { correlationId: msg.properties.correlationId });
                this._channel.ack(msg);
            }
            catch (error) {
                app_logs_1.logger.error("[rpc-server-to-tablet.sub] error while consuming msg %s", error.message || error);
                this._channel.nack(msg, false, false);
            }
        };
        this._channel = null;
        this._queue = null;
        this._rpcResponseQueue = null;
        this._socket = socket;
        this._deviceId = deviceId;
        this._rpcCallbackRegistry = new Map();
    }
    get deviceId() {
        return this._deviceId;
    }
    async init() {
        return new Promise((resolve, reject) => {
            this._connection = messageBroker_manager_1.messageBrokerManager.connection;
            this._channel = this._connection.createChannel({
                json: true,
                name: "server-to-tablet.sub",
                setup: async (channel) => {
                    app_logs_1.logger.debug("[server-to-tablet.sub] channel is establishing...");
                    [this._queue] = await Promise.all([
                        channel.assertQueue("rpc_" + this._deviceId, { exclusive: true, autoDelete: true }),
                        channel.assertExchange(messagebroker_models_1.MessageBrokerNames.serverToTablet.exchangeName, "direct", {
                            durable: true,
                        }),
                        channel.prefetch(1),
                    ]);
                    await channel.bindQueue(this._queue.queue, messagebroker_models_1.MessageBrokerNames.serverToTablet.exchangeName, this._deviceId);
                    await channel.consume(this._queue.queue, this.onMessage, { noAck: false });
                },
            });
            this._channel.on("error", (err) => app_logs_1.logger.error("[server-to-tablet.sub] error while creating channel: ", err.message ?? err));
            this._channel.on("connect", () => app_logs_1.logger.info("[server-to-tablet.sub] channel created"));
            this._channel.on("close", () => app_logs_1.logger.info("[server-to-tablet.sub] channel closed"));
            this._channel.once("connect", () => resolve());
        });
    }
    async closeChannel() {
        await this._channel.close();
    }
    async onSocketMessage(msg) {
        const response = JSON.parse(msg);
        const responsePromise = this._rpcCallbackRegistry.get(response.s);
        if (responsePromise) {
            this._rpcCallbackRegistry.delete(response.s);
            responsePromise(response);
        }
    }
}
exports.AmqpServerToTabletSub = AmqpServerToTabletSub;
