"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.amqpServerToTabletPub = void 0;
const uuid_1 = __importDefault(require("uuid"));
const app_logs_1 = require("../app.logs");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messagebroker_models_1 = require("./messagebroker.models");
class AmqpServerToTabletPub {
    constructor() {
        this._rpcCallbackRegistry = {};
        this.onRpcMessage = async (msg) => {
            if (msg === null) {
                return;
            }
            try {
                let data = JSON.parse(msg.content.toString());
                if (this._rpcCallbackRegistry[msg.properties.correlationId]) {
                    this._rpcCallbackRegistry[msg.properties.correlationId](null, data);
                    this._rpcCallbackRegistry[msg.properties.correlationId] = undefined;
                }
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
        };
        this._channel = null;
        this._rpcReponseQueue = null;
    }
    async init() {
        return new Promise((resolve, reject) => {
            this._connection = messageBroker_manager_1.messageBrokerManager.connection;
            this._channel = this._connection.createChannel({
                json: true,
                name: "server-to-tablet.pub",
                setup: async (channel) => {
                    app_logs_1.logger.debug("[server-to-tablet.pub] channel is establishing...");
                    [this._rpcReponseQueue] = await Promise.all([
                        channel.assertQueue(undefined, {
                            exclusive: true,
                            autoDelete: true,
                        }),
                        channel.assertExchange(messagebroker_models_1.MessageBrokerNames.serverToTablet.exchangeName, "direct", {
                            durable: true,
                        }),
                    ]);
                    await channel.consume(this._rpcReponseQueue.queue, this.onRpcMessage, { noAck: true });
                },
            });
            this._channel.on("error", (err) => app_logs_1.logger.error("[server-to-tablet.pub] error while creating channel: ", err));
            this._channel.on("connect", () => app_logs_1.logger.info("[server-to-tablet.pub] channel created"));
            this._channel.on("close", () => app_logs_1.logger.info("[server-to-tablet.pub] channel closed"));
            this._channel.once("connect", () => resolve());
        });
    }
    async sendToExchange(exchange, routingKey, packet, options) {
        return new Promise((resolve, reject) => {
            this._channel.publish(exchange, routingKey, packet, options, (err, ok) => {
                if (err) {
                    app_logs_1.logger.error("[server-to-report.pub] Broker rejected message error: ", err);
                    reject();
                }
                else {
                    resolve();
                }
            });
        });
    }
    async sentToTablet(tabletDeviceId, packet, expiration) {
        try {
            await this.sendToExchange(messagebroker_models_1.MessageBrokerNames.serverToTablet.exchangeName, tabletDeviceId, packet, { expiration });
        }
        catch (err) {
            app_logs_1.logger.error("[server-to-report.pub] Error while trying to publish to DelayedExchange: " + err);
        }
    }
    sendToRpcQueue(deviceId, packet, expiration, callback) {
        let correlationId = uuid_1.default.v4();
        this._rpcCallbackRegistry[correlationId] = callback;
        setTimeout(() => {
            if (this._rpcCallbackRegistry[correlationId]) {
                this._rpcCallbackRegistry[correlationId](new Error("Message expired"), null);
                this._rpcCallbackRegistry[correlationId] = undefined;
            }
        }, expiration + 5000);
        this._channel.publish(messagebroker_models_1.MessageBrokerNames.serverToTablet.exchangeName, deviceId, packet, {
            correlationId: correlationId,
            replyTo: this._rpcReponseQueue.queue,
            expiration: expiration,
        });
    }
}
exports.amqpServerToTabletPub = new AmqpServerToTabletPub();
