"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.amqpServerToReportPub = void 0;
const uuid_1 = __importDefault(require("uuid"));
const app_logs_1 = require("../app.logs");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messagebroker_models_1 = require("./messagebroker.models");
class AmqpServerToReportPub {
    constructor() {
        this._rpcCallbackRegistry = {};
        this.onRpcMessage = (msg) => {
            const callback = this._rpcCallbackRegistry[msg.properties.correlationId];
            if (callback) {
                this._rpcCallbackRegistry[msg.properties.correlationId] = undefined;
                const response = JSON.parse(msg.content.toString());
                if (response.error) {
                    this._channel.nack(msg, false, false);
                    callback(new Error(response.error), null);
                }
                else {
                    app_logs_1.logger.info("[server-to-report.pub] Report is prepared in " + response.duration + " ms");
                    callback(null, response);
                    this._channel.ack(msg);
                }
            }
            else {
                this._channel.nack(msg, false, false);
            }
        };
    }
    async init() {
        return new Promise((resolve, reject) => {
            this._connection = messageBroker_manager_1.messageBrokerManager.connection;
            this._channel = this._connection.createChannel({
                json: true,
                name: "server-to-report.pub",
                setup: async (channel) => {
                    app_logs_1.logger.debug("[server-to-report.pub] channel is creating...");
                    [this._rpcReplyQueue] = await Promise.all([
                        channel.assertQueue(undefined, {
                            exclusive: true,
                            durable: false,
                            autoDelete: true,
                        }),
                        channel.assertQueue(messagebroker_models_1.MessageBrokerNames.serverToReport.queue, { durable: true }),
                        channel.assertExchange(messagebroker_models_1.MessageBrokerNames.serverToReport.exchange, "topic", {
                            durable: true,
                        }),
                        channel.prefetch(5),
                    ]);
                    await channel.bindQueue(messagebroker_models_1.MessageBrokerNames.serverToReport.queue, messagebroker_models_1.MessageBrokerNames.serverToReport.exchange, messagebroker_models_1.MessageBrokerNames.serverToReport.reportBindingKey);
                    await channel.consume(this._rpcReplyQueue.queue, this.onRpcMessage, { noAck: false });
                },
            });
            this._channel.on("error", (err) => app_logs_1.logger.error("[server-to-report.pub] error while creating channel: ", err));
            this._channel.on("connect", () => app_logs_1.logger.info("[server-to-report.pub] channel created"));
            this._channel.on("close", () => app_logs_1.logger.info("[server-to-report.pub] channel closed"));
            this._channel.once("connect", () => resolve());
        });
    }
    async sendToExchange(exchange, routingKey, packet, options) {
        return new Promise((resolve, reject) => {
            this._channel.publish(exchange, routingKey, packet, options, (err, ok) => {
                if (err) {
                    app_logs_1.logger.error("[server-to-report.pub] Broker rejected message error: ", err);
                    reject();
                }
                else {
                    resolve();
                }
            });
        });
    }
    async sendScheduledJobRequest(data) {
        await this.sendToExchange(messagebroker_models_1.MessageBrokerNames.serverToReport.exchange, messagebroker_models_1.MessageBrokerNames.serverToReport.reportRoutingKey, data, {
            persistent: true,
            contentType: "application/json",
            headers: {
                "trigger-type": "scheduled",
            },
        });
    }
    async sendInstantReportRequest(packet, expiration, callback) {
        const correlationId = uuid_1.default.v4();
        this._rpcCallbackRegistry[correlationId] = callback;
        setTimeout(() => {
            if (this._rpcCallbackRegistry[correlationId]) {
                this._rpcCallbackRegistry[correlationId] = undefined;
                app_logs_1.logger.warn("Report request has been expired!");
                callback(new Error("Timeout"), null);
            }
        }, expiration);
        this.sendToExchange(messagebroker_models_1.MessageBrokerNames.serverToReport.exchange, messagebroker_models_1.MessageBrokerNames.serverToReport.reportRoutingKey, packet, {
            correlationId: correlationId,
            replyTo: this._rpcReplyQueue.queue,
            expiration,
            headers: {
                "trigger-type": "instant",
            },
        });
    }
}
exports.amqpServerToReportPub = new AmqpServerToReportPub();
