"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmqpServerToAppSub = exports.amqpServerToAppSubUserEventNames = void 0;
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const dal_access_cache_1 = require("../dal/access/access-interfaces/dal.access.cache");
const dal_manager_1 = require("../dal/dal.manager");
const dal_db_armon_schema_1 = require("../dal/db/armon/dal.db.armon.schema");
const messageBroker_manager_1 = require("./messageBroker.manager");
const predefined_permissions_1 = require("../dal/db/predefined/predefined.permissions");
const api_securityhelper_1 = require("../api/api.securityhelper");
const terminalWebRtc_1 = require("../lib/es/models/terminalWebRtc");
exports.amqpServerToAppSubUserEventNames = {
    newlog: "newlog",
    activitytimeline: "activitytimeline",
    controlpanelhealth: "controlpanelhealth",
    terminalemergency: "terminalemergency",
    regionemergency: "regionemergency",
    notification: "notification",
    terminalconnection: "terminalconnection",
    visitorListChanged: "visitorlistchanged",
    remoteAccessGrantsChanged: "remoteaccessgrantschanged",
    organizationroleschanged: "organizationroleschanged",
    organizationSettingsChanged: "organizationsettingschanged",
    accesscontrolpointstatechanged: "accesscontrolpointstatechanged",
    systemhealth: "systemhealth",
    visitorRegistrationDeviceStatusChanged: "visitorRegistrationDeviceStatusChanged",
    terminalCameras: "terminalCameras",
    webRTC: "webRtc",
    webRTCIceServers: "webrtc-iceservers",
};
class AmqpServerToAppSub {
    constructor(socket, organizationId, bindingKeys) {
        this._isConnected = false;
        this.onMessage = async (msg) => {
            if (msg === null) {
                return;
            }
            try {
                let data = JSON.parse(msg.content.toString());
                switch (data.e) {
                    case exports.amqpServerToAppSubUserEventNames.newlog:
                        await this.newLog(data.oId, data.p);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.notification:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.notification, data.p, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.organizationroleschanged:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.organizationroleschanged, {}, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.organizationSettingsChanged:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.organizationSettingsChanged, {}, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.controlpanelhealth:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.controlpanelhealth, data.p, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.systemhealth:
                        await this.newSystemStatusLog(data.oId, data.p);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.regionemergency:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.regionemergency, data.p, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.terminalemergency:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.terminalemergency, data.p, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.visitorListChanged:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.visitorListChanged, {}, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.visitorRegistrationDeviceStatusChanged:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.visitorRegistrationDeviceStatusChanged, data.p, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.remoteAccessGrantsChanged:
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.remoteAccessGrantsChanged, {}, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.terminalconnection:
                        await this.newConnectionEvent(data.p);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.accesscontrolpointstatechanged:
                        let sendData = {
                            a: data.p.accessControlPointId,
                            n: data.p.accessControlPointName,
                            u: data.p.fullname,
                            sl: data.p.stateList.map((state) => {
                                return {
                                    s: state.state,
                                    d: state.direction,
                                };
                            }),
                        };
                        this._socket.emit(exports.amqpServerToAppSubUserEventNames.accesscontrolpointstatechanged, sendData, this.onSocketResult);
                        break;
                    case exports.amqpServerToAppSubUserEventNames.webRTC:
                        {
                            const payload = data.p;
                            if (this._socket.sessionId === payload.p) {
                                switch (payload.t) {
                                    case "answer":
                                        this._socket.emit(terminalWebRtc_1.WEBRTC_ANSWER, payload.d, payload.m);
                                        break;
                                    case "hang-up":
                                        this._socket.emit(terminalWebRtc_1.WEBRTC_HANG_UP, payload.d, payload.m);
                                        break;
                                    case "new-ice-candidate":
                                        this._socket.emit(terminalWebRtc_1.WEBRTC_NEW_ICE_CANDIDATE, payload.d, payload.m);
                                        break;
                                    case "offer":
                                        this._socket.emit(terminalWebRtc_1.WEBRTC_OFFER, payload.d, payload.m);
                                        break;
                                }
                            }
                        }
                        break;
                    case exports.amqpServerToAppSubUserEventNames.webRTCIceServers:
                        {
                            const payload = data.p;
                            app_logs_1.logger.debug("webrtc refreshed");
                            this._socket.emit(exports.amqpServerToAppSubUserEventNames.webRTCIceServers, payload);
                        }
                        break;
                }
                this._channel.ack(msg);
            }
            catch (error) {
                app_logs_1.logger.error("[server-to-app.sub] error while consuming msg %s", error.message || error);
                this._channel.nack(msg, false, false);
            }
        };
        this.newLog = async (organizationId, logItem, widgetId) => {
            let token = this._socket.handshake.userToken;
            const { widgetIdList, activity } = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                let widgetIds = [];
                if (logItem.o) {
                    if (logItem.o !== dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId) {
                        let identityDetail = await dal_manager_1.dbManager.accessIdentity.getIdentityDetailed(organizationId, token.userId, {
                            userId: logItem.o,
                            hasOrganizationWideRight: true,
                            hasIdentityFullWrite: true,
                            trx,
                        });
                        widgetIds = !widgetId ? this.checkFilter(logItem, identityDetail) : [widgetId];
                        if (widgetIds.length === 0) {
                            return {
                                widgetIdList: widgetIds,
                                activity: null,
                            };
                        }
                        if (!token.isPermittedForOrganization(organizationId, [predefined_permissions_1.Permissions.identity.getReadBasic(), predefined_permissions_1.Permissions.accessLog.getReportDetailed()])) {
                            let canSee = await dal_manager_1.dbManager.accessUser.canUserSeeOtherUsersAccess({
                                organizationId: organizationId,
                                accessControlPointId: logItem.a,
                                requesterUserId: token.userId,
                                userId: logItem.o,
                                trx,
                            });
                            if (!canSee) {
                                return {
                                    widgetIdList: widgetIds,
                                    activity: null,
                                };
                            }
                        }
                    }
                    else {
                        widgetIds = !widgetId
                            ? this.checkFilter(logItem, {
                                accessRights: [],
                                credentials: [],
                                id: dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId,
                                organizationProfile: null,
                                organizationUnits: [],
                                publicKey: null,
                                userGroups: [],
                            })
                            : [widgetId];
                    }
                }
                let t = !logItem.di ? app_enums_1.enums.CardActivityTimelineItemType.AccessAttempt : app_enums_1.enums.CardActivityTimelineItemType.ExitButton;
                let activity = {
                    i: logItem.id,
                    u: new Date(logItem.u),
                    t: t,
                    a: t === app_enums_1.enums.CardActivityTimelineItemType.AccessAttempt
                        ? {
                            i: uuid_1.default.v1(),
                            r: logItem.r,
                            d: logItem.d,
                            u: logItem.o,
                            n: logItem.on,
                            ai: logItem.a,
                            an: logItem.an,
                            di: logItem.di,
                            ir: logItem.ir,
                            sid: logItem.sid,
                            re: logItem.re,
                            cid: logItem.cid,
                            s: logItem.s,
                            v: logItem.v,
                            cx: logItem.cx,
                        }
                        : null,
                    e: t === app_enums_1.enums.CardActivityTimelineItemType.ExitButton
                        ? {
                            i: uuid_1.default.v1(),
                            d: logItem.d,
                            di: logItem.di,
                            ai: logItem.a,
                            an: logItem.an,
                        }
                        : null,
                };
                if (t == app_enums_1.enums.CardActivityTimelineItemType.AccessAttempt) {
                    if (logItem.o === dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId && (logItem.c || logItem.cx?.length)) {
                        let unknownCaption = [];
                        unknownCaption.push({
                            text: [logItem.c],
                        });
                        activity.a.cl = unknownCaption;
                        activity.a.u = null;
                    }
                    else if (logItem.o) {
                        const { caption: captionLines } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: logItem.o, trx });
                        activity.a.cl = captionLines;
                    }
                }
                return {
                    widgetIdList: widgetIds,
                    activity,
                };
            });
            if (activity) {
                for (let widgetId of widgetIdList) {
                    activity.wi = widgetId;
                    this._socket.emit(exports.amqpServerToAppSubUserEventNames.activitytimeline, activity, this.onSocketResult);
                }
            }
        };
        this.newConnectionEvent = async (data) => {
            this._socket.emit(exports.amqpServerToAppSubUserEventNames.terminalconnection, data, this.onSocketResult);
            await this.newSystemStatusLog(data.oId, data.i);
        };
        this.newSystemStatusLog = async (organizationId, deviceId) => {
            let deviceStatus;
            let cacheValue = await dal_manager_1.dbManager.accessRedisCache.getValue(dal_access_cache_1.CacheConstantKeys.DeviceStatus + organizationId);
            if (!cacheValue) {
                deviceStatus = await dal_manager_1.dbManager.accessDevice.getDeviceStatus(organizationId);
            }
            else {
                deviceStatus = JSON.parse(cacheValue);
            }
            let systemHealth = {
                did: deviceId,
                c: deviceStatus.connected,
                d: deviceStatus.disconnected,
                w: {
                    c: deviceStatus.warning.critical,
                    h: deviceStatus.warning.high,
                    m: deviceStatus.warning.medium,
                },
            };
            this._socket.emit(exports.amqpServerToAppSubUserEventNames.systemhealth, systemHealth, this.onSocketResult);
        };
        this._channel = null;
        this._queue = null;
        this._bindingKeys = [];
        this._socket = socket;
        this._organizationId = organizationId;
        this._id = uuid_1.default.v4();
        this._connection = messageBroker_manager_1.messageBrokerManager.socketAmqpConnection;
        this._bindingKeys.push(...bindingKeys);
    }
    async init() {
        this._channel = await this._connection.createConfirmChannel();
        await this.assertStructure();
        this._isConnected = true;
        this._channel.on("error", (err) => app_logs_1.logger.error("[server-to-app.pub] error while creating channel: ", err));
        this._channel.on("close", () => app_logs_1.logger.info("[server-to-app.pub] channel closed"));
    }
    async assertStructure() {
        app_logs_1.logger.debug("[server-to-app.sub] channel is establishing...");
        [this._exhange, this._queue] = await Promise.all([
            this._channel.assertExchange(AmqpServerToAppSub._exchangeName, "topic", {
                durable: true,
            }),
            this._channel.assertQueue(uuid_1.default.v4(), {
                exclusive: true,
                autoDelete: true,
            }),
        ]);
        for (const bindingKey of this._bindingKeys) {
            await this._channel.bindQueue(this._queue.queue, AmqpServerToAppSub._exchangeName, bindingKey);
        }
        await this._channel.consume(this._queue.queue, this.onMessage, { noAck: false });
        app_logs_1.logger.debug("[server-to-app.sub] channel is established");
    }
    async closeConnection() {
        this._isConnected = false;
        await this.unbindKeys(this._bindingKeys);
        await this._channel.deleteQueue(this._queue.queue);
        await this._channel.close();
    }
    async bindKeys(bindingKeys) {
        this._bindingKeys.push(...bindingKeys);
        try {
            if (this._isConnected) {
                for (const bindingKey of bindingKeys) {
                    await this._channel.bindQueue(this._queue.queue, this._exhange.exchange, bindingKey);
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error("Failed to bind a queue after roomin to a socket: " + error);
        }
    }
    async unbindKeys(bindingKeys) {
        this._bindingKeys.push(...bindingKeys);
        if (this._isConnected) {
            this._bindingKeys = this._bindingKeys.filter((keys) => !bindingKeys.includes(keys));
            for (const bindingKey of bindingKeys) {
                await this._channel.unbindQueue(this._queue.queue, this._exhange.exchange, bindingKey);
            }
        }
    }
    get organizationId() {
        return this._organizationId;
    }
    onSocketResult(err) {
        if (err) {
            app_logs_1.logger.error("This user agent should be not here actually %j", err);
        }
    }
    checkFilter(logItem, identityDetail) {
        let widgetIdList = [];
        for (let data of this._socket.userFilters) {
            let filter = this._socket.userFilters;
            let userCheck = this.isRequestedUserHasFilterToSeeThisUserLog(filter, identityDetail, logItem);
            let logCheck = this.isRequestedUserHasFilterToSeeThisLog(filter, logItem);
            if (userCheck && logCheck) {
                widgetIdList.push(data.widgetId);
            }
        }
        return widgetIdList;
    }
    checkStatus(filter, status) {
        return (filter.status !== app_enums_1.enums.AccessReportFilterAccessResultType.All &&
            ((filter.status === app_enums_1.enums.AccessReportFilterAccessResultType.Fail && status) || (filter.status === app_enums_1.enums.AccessReportFilterAccessResultType.Success && !status)));
    }
    checkAcpId(filter, acpId) {
        return filter.accessControlPointIds && !filter.accessControlPointIds.includes(acpId);
    }
    checkDirection(filter, direction) {
        return filter.direction && filter.direction !== app_enums_1.enums.AccessDirection.Entrance && filter.direction !== app_enums_1.enums.AccessDirection.Exit && filter.direction !== direction;
    }
    checkRecordReason(filter, reason) {
        return filter.recordReasons && !filter.recordReasons.includes(reason);
    }
    checkIfAnyFilter(filter) {
        return ((!filter.userIds || filter.userIds.length === 0) &&
            (!filter.organizationUnitIds || filter.organizationUnitIds.length === 0) &&
            (!filter.userGroupIds || filter.userGroupIds.length === 0));
    }
    checkUserIds(filter, userId) {
        return filter.userIds && filter.userIds.includes(userId);
    }
    checkOrganizationUnitIds(filter, organizationUnitIds) {
        return filter.organizationUnitIds && filter.organizationUnitIds.length > 0 && !(filter.organizationUnitIds.filter((value) => organizationUnitIds.includes(value)).length > 0);
    }
    checkUserGroupIds(filter, userGroups) {
        return filter.userGroupIds && filter.userGroupIds.filter((value) => userGroups.includes(value)).length > 0;
    }
    isRequestedUserHasFilterToSeeThisUserLog(filter, identityDetail, logItem) {
        let check = false;
        let userOrganizationUnits = identityDetail.organizationUnits.map((ou) => ou.organizationUnitId);
        let userGroups = identityDetail.userGroups.map((ug) => ug.id);
        if (this.checkIfAnyFilter(filter) ||
            this.checkUserIds(filter, logItem.o) ||
            this.checkOrganizationUnitIds(filter, userOrganizationUnits) ||
            this.checkUserGroupIds(filter, userGroups)) {
            check = true;
        }
        return check;
    }
    isRequestedUserHasFilterToSeeThisLog(filter, logItem) {
        let check = true;
        if (this.checkRecordReason(filter, logItem.r) || this.checkDirection(filter, logItem.d) || this.checkAcpId(filter, logItem.a) || this.checkStatus(filter, logItem.s)) {
            check = false;
        }
        return check;
    }
    async newActivitiyConnectionSendLogs(filter) {
        let token = this._socket.handshake.userToken;
        let req = {
            auth_token: token,
        };
        let authResult = (0, api_securityhelper_1.authorizeForAccessReport)(req, this._organizationId);
        if (await dal_manager_1.dbManager.accessAccessLog.canUserSeeUnkownUserAccessLogs(token.getOrganizationId(), token.userId, null)) {
            filter.userIds ? filter.userIds.push(dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId) : (filter.userIds = [dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId]);
        }
        let logItems = await dal_manager_1.dbManager.accessLog.getAccessLogs(this.organizationId, {
            take: 10,
            direction: filter.direction,
            accessControlPointIds: filter.accessControlPointIds,
            userIds: filter.userIds,
            status: filter.status,
            recordReasons: filter.recordReasons,
            userGroupIds: filter.userGroupIds,
            userOrganizationUnitIds: filter.organizationUnitIds ? filter.organizationUnitIds : authResult.permittedUnitIds,
        });
        for (let logItem of logItems) {
            await this.newLog(this.organizationId, logItem, filter.widgetId);
        }
    }
}
exports.AmqpServerToAppSub = AmqpServerToAppSub;
AmqpServerToAppSub._exchangeName = "server-to-app";
