"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.amqpEventSub = exports.AmqpEventSub = exports.amqpServerEvents = void 0;
const cluster_1 = __importDefault(require("cluster"));
const app_config_1 = require("../app.config");
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const business_device_1 = require("../business/business.device");
const dal_constants_1 = require("../dal/dal.constants");
const dal_manager_1 = require("../dal/dal.manager");
const ws_usernsp_1 = require("../ws/ws.usernsp");
const messageBroker_cli_1 = require("./messageBroker.cli");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messageBroker_server_to_app_pub_1 = require("./messageBroker.server-to-app.pub");
const messageBroker_server_to_app_sub_1 = require("./messageBroker.server-to-app.sub");
const messageBroker_server_to_device_pub_1 = require("./messageBroker.server-to-device.pub");
const messagebroker_models_1 = require("./messagebroker.models");
const dal_utils_1 = require("../dal/dal.utils");
const app_constants_1 = require("../app.constants");
exports.amqpServerEvents = {
    connectionClosed: "connection.closed",
    connectionCreated: "connection.created",
};
class AmqpEventSub {
    constructor() {
        this.onMessage = async (msg) => {
            if (msg === null) {
                return;
            }
            try {
                if (msg.fields.routingKey === exports.amqpServerEvents.connectionCreated) {
                    const device = await dal_manager_1.dbManager.accessDevice.getConnectedDeviceInfo(msg.properties.headers.user);
                    if (device) {
                        app_logs_1.logger.debug("Connect Event: " + msg.properties.headers.user);
                        await (0, business_device_1.updateDeviceConnectionStatus)({
                            organizationId: device.organizationId,
                            deviceId: device.id,
                            isConnected: true,
                            isRestart: process.uptime() < app_constants_1.serverRestartThresholdInSeconds,
                        });
                        await dal_manager_1.dbManager.accessRedisCache.setValue(dal_constants_1.DalConstants.NewDataCacheType.DeviceRabbitMQConnectionName + "_" + msg.properties.headers.name, JSON.stringify({ id: device.id, organizationId: device.organizationId }));
                        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(device.organizationId + "." + ws_usernsp_1.room.systemStatus + "." + device.id, {
                            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.terminalconnection,
                            p: {
                                i: device.id,
                                c: device.model,
                                n: device.name,
                                s: app_enums_1.enums.TerminalConnectionStatusType.Connected,
                                oId: device.organizationId,
                            },
                        });
                        this.registerOnTerminalChange(device.id, (() => {
                            app_logs_1.logger.debug(`Terminal change for device ${device.id}`);
                            messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(device.id, {
                                e: messagebroker_models_1.ServerToDeviceMessageType.GetChangesRequired,
                            });
                        }).bind(this));
                    }
                }
                else if (msg.fields.routingKey === exports.amqpServerEvents.connectionClosed) {
                    const cacheValue = await dal_manager_1.dbManager.accessRedisCache.getValue(dal_constants_1.DalConstants.NewDataCacheType.DeviceRabbitMQConnectionName + "_" + msg.properties.headers.name);
                    let deviceInfo = null;
                    if (cacheValue) {
                        deviceInfo = JSON.parse(cacheValue);
                    }
                    if (!deviceInfo) {
                        app_logs_1.logger.debug(`There is no data with connection name ${msg.properties.headers.name} in Redis, probably not a device`);
                        this._channel.ack(msg);
                        return;
                    }
                    const device = await dal_manager_1.dbManager.accessDevice.getConnectedDeviceInfo(deviceInfo.id);
                    if (device) {
                        await (0, business_device_1.updateDeviceConnectionStatus)({
                            organizationId: device.organizationId,
                            deviceId: device.id,
                            isConnected: false,
                            isRestart: process.uptime() < app_constants_1.serverRestartThresholdInSeconds,
                        });
                        app_logs_1.logger.debug("Disconnect Event: " + device.id);
                        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(device.organizationId + "." + ws_usernsp_1.room.systemStatus + "." + device.id, {
                            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.terminalconnection,
                            p: {
                                i: device.id,
                                c: device.model,
                                n: device.name,
                                s: app_enums_1.enums.TerminalConnectionStatusType.Lost,
                                oId: device.organizationId,
                            },
                        });
                        this.unregisterOnTerminalChange(device.id);
                    }
                }
                this._channel.ack(msg);
            }
            catch (error) {
                app_logs_1.logger.error("[amqp-event.sub] error while consuming msg %s", error.message || error);
                this._channel.nack(msg, false, false);
            }
        };
        this._channel = null;
        this._onTerminalChangeListeners = new Map();
    }
    async init() {
        if (cluster_1.default.isMaster) {
            await this.listenPg();
        }
        return new Promise((resolve, reject) => {
            this._connection = messageBroker_manager_1.messageBrokerManager.connection;
            this._channel = this._connection.createChannel({
                json: true,
                name: "event.sub",
                setup: async (channel) => {
                    app_logs_1.logger.debug("[event.sub] channel is establishing...");
                    await Promise.all([
                        channel.assertQueue(messagebroker_models_1.MessageBrokerNames.event.queue, {
                            durable: true,
                        }),
                        channel.prefetch(1),
                    ]);
                    await channel.bindQueue(messagebroker_models_1.MessageBrokerNames.event.queue, messagebroker_models_1.MessageBrokerNames.event.exchange, "connection.created");
                    await channel.bindQueue(messagebroker_models_1.MessageBrokerNames.event.queue, messagebroker_models_1.MessageBrokerNames.event.exchange, "connection.closed");
                },
            });
            this._channel.on("error", (err) => app_logs_1.logger.error("[event.sub] error while creating channel: ", err));
            this._channel.on("connect", async () => {
                app_logs_1.logger.info("[event.sub] channel created");
                await Promise.all([
                    (0, messageBroker_cli_1.purgeQueueMessages)({
                        hostname: app_config_1.appConfig.amqpClient.hostname,
                        queueName: messagebroker_models_1.MessageBrokerNames.event.queue,
                        port: app_config_1.appConfig.amqpClient.managementApiPort,
                        authorization: app_config_1.appConfig.amqpManagementApiAuth,
                    }),
                ]);
                await this._channel.addSetup(async (channel) => {
                    await channel.consume(messagebroker_models_1.MessageBrokerNames.event.queue, this.onMessage);
                    app_logs_1.logger.debug("[event.sub] queue is listening...");
                });
            });
            this._channel.on("close", () => app_logs_1.logger.info("[event.sub] channel closed"));
            this._channel.once("connect", () => resolve());
        });
    }
    async listenPg() {
        const client = await (0, dal_utils_1.listenClientWrapperWithRetry)("APP-WEB MASTER", dal_manager_1.dbManager.poolMain);
        client.on("notification", (async (msg) => {
            let listener = null;
            switch (msg.channel) {
                case "terminal_changes":
                    listener = this._onTerminalChangeListeners.get(msg.payload);
                    if (listener) {
                        listener();
                    }
                    break;
                case "region_state_changes":
                    listener = this._onRegionStateChangeListeners;
                    if (listener) {
                        listener(msg.payload);
                    }
                    break;
                case "access_rule_applied":
                    listener = this._onAccessRuleAppliedListeners;
                    if (listener) {
                        listener(msg.payload);
                    }
                    break;
                case "access_capacity_based_rule_applied":
                    listener = this._onAccessCapacityBasedRuleAppliedListeners;
                    if (listener) {
                        listener(msg.payload);
                        app_logs_1.logger.debug(msg.payload);
                    }
                    break;
                default:
                    break;
            }
        }).bind(this));
        client.on("error", (err) => {
            app_logs_1.logger.error("LISTEN APP-WEB MASTER connection error: " + err);
        });
        client.on("end", () => {
            app_logs_1.logger.info("LISTEN APP-WEB MASTER connection terminated... Restarting...");
            client.release();
            this.listenPg();
        });
        client.query("LISTEN terminal_changes");
        client.query("LISTEN region_state_changes");
        client.query("LISTEN access_rule_applied");
        client.query("LISTEN access_capacity_based_rule_applied");
        app_logs_1.logger.info("LISTEN APP-WEB MASTER connections started.");
    }
    async updateDeviceConnectionStatusesAndNotifyDevices() {
        (0, messageBroker_cli_1.refreshTerminalConnectionStates)({
            hostname: app_config_1.appConfig.amqpClient.hostname,
            port: app_config_1.appConfig.amqpClient.managementApiPort,
            authorization: app_config_1.appConfig.amqpManagementApiAuth,
        });
    }
    registerOnTerminalChange(deviceId, onTerminalChange) {
        if (cluster_1.default.isMaster) {
            this._onTerminalChangeListeners.set(deviceId, onTerminalChange);
        }
    }
    unregisterOnTerminalChange(deviceId) {
        if (cluster_1.default.isMaster) {
            this._onTerminalChangeListeners.delete(deviceId);
        }
    }
    registerOnRegionStateChange(onRegionStateChanged) {
        if (cluster_1.default.isMaster) {
            this._onRegionStateChangeListeners = onRegionStateChanged;
        }
    }
    registerOnAccessRuleAppliedChange(onAccessRuleApplied) {
        if (cluster_1.default.isMaster) {
            this._onAccessRuleAppliedListeners = onAccessRuleApplied;
        }
    }
    registerOnAccessCapacityBasedRuleAppliedChange(onAccessCapacityBasedRuleApplied) {
        if (cluster_1.default.isMaster) {
            this._onAccessCapacityBasedRuleAppliedListeners = onAccessCapacityBasedRuleApplied;
        }
    }
}
exports.AmqpEventSub = AmqpEventSub;
exports.amqpEventSub = new AmqpEventSub();
