"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.amqpDeviceToServerSub = void 0;
const app_enums_1 = require("../app.enums");
const app_httpserver_1 = require("../app.httpserver");
const app_logs_1 = require("../app.logs");
const app_util_1 = require("../app.util");
const business_device_1 = require("../business/business.device");
const business_notification_1 = require("../business/business.notification");
const business_system_1 = require("../business/business.system");
const cli_queries_1 = require("../dal/access/psql/cli-queries");
const dal_constants_1 = require("../dal/dal.constants");
const dal_manager_1 = require("../dal/dal.manager");
const dal_utils_1 = require("../dal/dal.utils");
const restapi_1 = require("../lib/es/models/restapi");
const ws_usernsp_1 = require("../ws/ws.usernsp");
const messageBroker_manager_1 = require("./messageBroker.manager");
const messageBroker_notification_pub_1 = require("./messageBroker.notification.pub");
const messageBroker_server_to_app_pub_1 = require("./messageBroker.server-to-app.pub");
const messageBroker_server_to_app_sub_1 = require("./messageBroker.server-to-app.sub");
const messagebroker_models_1 = require("./messagebroker.models");
const enums_1 = require("../lib/access-model/v2/enums");
const dal_access_psql_log_1 = require("../dal/access/psql/dal.access.psql.log");
const app_constants_1 = require("../app.constants");
class AmqpDeviceToServerSub {
    constructor() {
        this.onMessageDeprecated = async (msg) => {
            this.onMessage(msg, false);
        };
        this.onMessageSafeQueue = async (msg) => {
            this.onMessage(msg, true);
        };
        this.onMessage = async (msg, safe) => {
            if (msg === null) {
                return;
            }
            try {
                const data = JSON.parse(msg.content.toString());
                const organizationId = msg.properties.headers.o;
                const deviceId = msg.properties.headers.d;
                switch (data.e) {
                    case messagebroker_models_1.DeviceToServerMessageTypes.AccessLog: {
                        const { logItem, insertedLog } = await this.onAccessLog(msg, organizationId, data.p, safe);
                        await this.notifyUsersAndDevicesForAccessLog({ organizationId, logItem, insertedLog });
                        break;
                    }
                    case messagebroker_models_1.DeviceToServerMessageTypes.CounterSensorEvent:
                        await this.onCounterSensor(msg, organizationId, data.p, safe);
                        await this.checkCounterNotificationFromCache(organizationId, data.p);
                        break;
                    case messagebroker_models_1.DeviceToServerMessageTypes.EmergencyStateChanged:
                        await this.onEmergency(msg, organizationId, deviceId, safe, data.p);
                        break;
                    case messagebroker_models_1.DeviceToServerMessageTypes.StatusSensorStateChanged:
                        await this.onStatusSensorState(msg, organizationId, (data.p || data.pp), safe);
                        await this.notifyUsersForStatusSensorLog(organizationId, (data.p || data.pp));
                        break;
                    case messagebroker_models_1.DeviceToServerMessageTypes.TamperEvent:
                        await this.onTamperswitch(msg, organizationId, data.p, safe);
                        break;
                    case messagebroker_models_1.DeviceToServerMessageTypes.SystemHealth:
                        const systemHealth = await this.onSystemHealth(msg, organizationId, deviceId, data.p);
                        await this.notifyUsersAboutSystemHealth(organizationId, deviceId, systemHealth);
                        break;
                    case messagebroker_models_1.DeviceToServerMessageTypes.DeviceOfAdapterConnectionState:
                        await this.onDeviceOfAdapterConnectionState(msg, organizationId, data.p);
                        break;
                    case messagebroker_models_1.DeviceToServerMessageTypes.WebRTC:
                        await this.onWebRTCMessage(msg, organizationId, deviceId, data.p);
                        break;
                    default:
                        app_logs_1.logger.error("[device-to-server.sub] unknown msg %s", data.e);
                        this._channel.ack(msg);
                        break;
                }
            }
            catch (error) {
                app_logs_1.logger.error("[device-to-server.sub] error while consuming msg %s", error.message || error);
            }
        };
        this._channel = null;
    }
    async init() {
        return new Promise((resolve, reject) => {
            this._connection = messageBroker_manager_1.messageBrokerManager.connection;
            this._channel = this._connection.createChannel({
                json: true,
                name: "device-to-server.sub",
                setup: async (channel) => {
                    app_logs_1.logger.debug("[device-to-server.sub] channel is establishing...");
                    await Promise.all([
                        channel.assertExchange(messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, "direct", { durable: true }),
                        channel.assertQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, {
                            durable: true,
                        }),
                        channel.assertExchange(messagebroker_models_1.MessageBrokerNames.deviceToServer.exchange, "topic", {
                            durable: true,
                        }),
                        channel.assertQueue(messagebroker_models_1.MessageBrokerNames.deviceToServer.queue, {
                            durable: true,
                        }),
                        channel.assertQueue(messagebroker_models_1.MessageBrokerNames.deviceToServer.queueWithDeadLetter, {
                            durable: true,
                            deadLetterExchange: messagebroker_models_1.MessageBrokerNames.deadLetter.exchange,
                            deadLetterRoutingKey: messagebroker_models_1.MessageBrokerNames.deadLetter.deviceBindingKey,
                        }),
                        channel.prefetch(3),
                    ]);
                    await channel.bindQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, messagebroker_models_1.MessageBrokerNames.deadLetter.deviceBindingKey);
                    await channel.consume(messagebroker_models_1.MessageBrokerNames.deviceToServer.queueWithDeadLetter, this.onMessageSafeQueue, { noAck: false });
                    await channel.consume(messagebroker_models_1.MessageBrokerNames.deviceToServer.queue, this.onMessageDeprecated, { noAck: false });
                },
            });
            this._channel.on("error", (err) => app_logs_1.logger.error("[device-to-server.sub] error while creating channel: ", err));
            this._channel.on("connect", () => app_logs_1.logger.info("[device-to-server.sub] channel created"));
            this._channel.on("close", () => app_logs_1.logger.info("[device-to-server.sub] channel closed"));
            this._channel.on("connect", () => resolve());
        });
    }
    async onWebRTCMessage(msg, organizationId, deviceId, payload) {
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + ".*." + payload.u, { e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.webRTC, p: { d: deviceId, ...payload } });
        this._channel.ack(msg);
    }
    async onDeviceOfAdapterConnectionState(msg, organizationId, payload) {
        try {
            if (payload.s === app_enums_1.enums.DeviceOfAdapterConnectionState.Connected) {
                app_logs_1.logger.debug("adpConnect Event: " + payload.d);
                const device = await dal_manager_1.dbManager.accessDevice.getConnectedDeviceInfo(payload.d);
                if (device) {
                    messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(device.organizationId + "." + ws_usernsp_1.room.systemStatus + "." + device.id, {
                        e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.terminalconnection,
                        p: {
                            i: device.id,
                            c: device.model,
                            n: device.name,
                            s: app_enums_1.enums.TerminalConnectionStatusType.Connected,
                            oId: device.organizationId,
                        },
                    });
                    await (0, business_device_1.updateDeviceConnectionStatus)({
                        organizationId: device.organizationId,
                        deviceId: device.id,
                        isConnected: true,
                        isRestart: process.uptime() < app_constants_1.serverRestartThresholdInSeconds,
                    });
                    await dal_manager_1.dbManager.accessRedisCache.setValue(payload.d, JSON.stringify({
                        id: device.id,
                        model: device.model,
                        name: device.name,
                        organizationId: device.organizationId,
                    }));
                }
            }
            else {
                app_logs_1.logger.debug("adpDisconnect Event: " + payload.d);
                let cacheValue = await dal_manager_1.dbManager.accessRedisCache.getValue(payload.d);
                if (cacheValue) {
                    let device = JSON.parse(cacheValue);
                    messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(device.organizationId + "." + ws_usernsp_1.room.systemStatus + ".*", {
                        e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.terminalconnection,
                        p: {
                            i: device.id,
                            c: device.model,
                            n: device.name,
                            s: app_enums_1.enums.TerminalConnectionStatusType.Lost,
                            oId: device.organizationId,
                        },
                    });
                    await (0, business_device_1.updateDeviceConnectionStatus)({
                        organizationId: device.organizationId,
                        deviceId: device.id,
                        isConnected: false,
                        isRestart: process.uptime() < app_constants_1.serverRestartThresholdInSeconds,
                    });
                    await dal_manager_1.dbManager.accessRedisCache.delValue(payload.d);
                }
            }
            this._channel.ack(msg);
        }
        catch (error) {
            app_logs_1.logger.error("[device-to-server.sub] error while consuming msg %s", error.message || error);
            this._channel.nack(msg, false, false);
        }
    }
    async onAccessLog(msg, organizationId, deviceAccessLog, safe) {
        try {
            const [logItem, insertedLog] = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const logItem = await (0, business_device_1.updateLogFromDevice)({ organizationId, deviceLog: deviceAccessLog, logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant });
                const insertedLog = await dal_manager_1.dbManager.accessLog.addAccessLogFromDevice(organizationId, logItem, trx);
                const now = new Date();
                if (insertedLog.rg && insertedLog.rg.length > 0) {
                    for (const regionLog of insertedLog.rg) {
                        let regionNotification = {
                            i: regionLog.i,
                            s: regionLog.s.valueOf(),
                            ee: regionLog.ee,
                            xe: regionLog.xe,
                            o: insertedLog.o,
                            t: insertedLog.u,
                        };
                        await (0, cli_queries_1.upsertRegionStates)(organizationId, [{ accessControlPointId: insertedLog.a, log: regionNotification }], trx);
                        if (regionLog.ru && regionLog.ru.c) {
                            await (0, cli_queries_1.upsertAccessRuleHistory)(organizationId, {
                                date: now,
                                accessRuleId: regionLog.ru.i,
                                userId: insertedLog.o,
                                count: regionLog.ru.c,
                                accessControlPointId: insertedLog.a,
                                actionDateISO: insertedLog.u,
                                startDateTime: regionLog.ru.sd,
                                endDateTime: regionLog.ru.ed,
                                weekDay: regionLog.ru.wd,
                            }, trx);
                        }
                        if (regionLog.rbu && regionLog.rbu.length) {
                            await dal_manager_1.dbManager.accessUserGroup.updateUserGroupLimitRules({
                                organizationId,
                                rulesToUpdate: regionLog.rbu.map((rbu) => {
                                    return {
                                        userGroupId: rbu.g,
                                        accessRuleId: rbu.i,
                                        count: rbu.c >= 0 ? rbu.c : 0,
                                    };
                                }),
                                accessControlPointId: insertedLog.a,
                                actionDateISO: insertedLog.u,
                                trx,
                            });
                        }
                        if (regionLog.rti) {
                            await (0, cli_queries_1.updateUserRegionTicketUnits)(organizationId, {
                                userId: insertedLog.o,
                                accessControlPointId: insertedLog.a,
                                decrementUnit: regionLog.rti.c,
                                regionTicketId: regionLog.rti.i,
                            }, trx);
                        }
                    }
                }
                return [logItem, insertedLog];
            });
            this._channel.ack(msg);
            if (insertedLog.d && insertedLog.d === enums_1.EnumsV2.AccessDirection.Entrance && insertedLog.s === true) {
                await (0, dal_access_psql_log_1.assignAutoShift)(organizationId, { credentialOwnerUserId: insertedLog.o, generationTime: insertedLog.u, redisCache: dal_manager_1.dbManager.accessRedisCache, logId: insertedLog.id });
            }
            return { logItem, insertedLog };
        }
        catch (error) {
            try {
                if (!safe) {
                    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                        return (0, cli_queries_1.insertUnhandledDeviceLog)({
                            organizationId,
                            deviceId: deviceAccessLog.cid,
                            type: messagebroker_models_1.DeviceToServerMessageTypes.AccessLog,
                            logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant,
                            data: deviceAccessLog,
                            note: { error: error.message ?? error },
                            trx,
                        });
                    });
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming Access Log, saved to unhandled_device_log table %s", error.message || error);
                }
                else {
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming Access Log, sended to deadLetter %s", error.message || error);
                }
            }
            catch (error) {
                app_logs_1.logger.error("[device-to-server.sub] error while writing to unhandledDeviceLogs", error.message || error);
            }
            finally {
                this._channel.nack(msg, false, false);
                throw new Error(error);
            }
        }
    }
    async onCounterSensor(msg, organizationId, packet, safe) {
        try {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const counterSensorLogId = await dal_manager_1.dbManager.accessLog.insertOnlineCounterLog({ organizationId, log: packet, trx });
                if (packet.l && counterSensorLogId) {
                    await dal_manager_1.dbManager.accessLog.updateAccessLogForSensorData({ organizationId, logId: packet.l, counterSensorLogId, trx });
                }
            });
            this._channel.ack(msg);
        }
        catch (error) {
            try {
                if (!safe) {
                    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                        return (0, cli_queries_1.insertUnhandledDeviceLog)({
                            organizationId,
                            deviceId: packet.ci,
                            type: messagebroker_models_1.DeviceToServerMessageTypes.CounterSensorEvent,
                            logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant,
                            data: packet,
                            note: { error: error.message ?? error },
                            trx,
                        });
                    });
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming Counter Log, saved to unhandled_device_log table %s", error.message || error);
                }
                else {
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming Counter Log, sended to deadLetter %s", error.message || error);
                }
            }
            catch (error) {
                app_logs_1.logger.error("[device-to-server.sub] error while writing to unhandledDeviceLogs", error.message || error);
            }
            finally {
                this._channel.nack(msg, false, false);
                throw new Error(error);
            }
        }
    }
    async onEmergency(msg, organizationId, deviceId, safe, deviceState) {
        if (deviceState.s === dal_constants_1.DalConstants.libEnumsV1.EmergencyState.Normal) {
            this._channel.ack(msg);
            return;
        }
        try {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const state = (0, app_util_1.deviceEmergencyCodeToAppwebEmergencyCode)(deviceState.s);
                const [terminalInfo, regionListForEmergencyCheck] = await Promise.all([
                    dal_manager_1.dbManager.accessRedisCache.getDeviceNotificationInfoCache({ organizationId, deviceId }),
                    dal_manager_1.dbManager.accessDevice.getRegionIdNamePairsForDeviceEmergency({ organizationId, deviceId, trx }),
                ]);
                if (regionListForEmergencyCheck.length) {
                    const terminalIdsAlreadyNotified = [];
                    for (const regionIdNamePair of regionListForEmergencyCheck) {
                        const deviceIdsOfRegion = (await dal_manager_1.dbManager.accessRegion.getRegionTerminalIds(organizationId, regionIdNamePair.regionId)).filter((dior) => !terminalIdsAlreadyNotified.includes(dior));
                        terminalIdsAlreadyNotified.push(...deviceIdsOfRegion);
                        (0, business_system_1.handleRegionEmergency)({
                            organizationId,
                            regionId: regionIdNamePair.regionId,
                            regionName: regionIdNamePair.regionName,
                            state,
                            relatedTerminalIds: deviceIdsOfRegion,
                        });
                    }
                }
                else {
                    await (0, business_system_1.handleTerminalEmergency)({ organizationId: organizationId, terminalName: terminalInfo.n, terminalId: deviceId, state });
                }
            });
            this._channel.ack(msg);
        }
        catch (error) {
            try {
                if (!safe) {
                    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                        return (0, cli_queries_1.insertUnhandledDeviceLog)({
                            organizationId,
                            deviceId,
                            type: messagebroker_models_1.DeviceToServerMessageTypes.EmergencyStateChanged,
                            logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant,
                            data: {},
                            note: { error: error.message ?? error },
                            trx,
                        });
                    });
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming emergency Log, saved to unhandled_device_log table %s", error.message || error);
                }
                else {
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming emergency Log, sended to deadLetter %s", error.message || error);
                }
            }
            catch (error) {
                app_logs_1.logger.error("[device-to-server.sub] error while writing to unhandledDeviceLogs", error.message || error);
            }
            finally {
                this._channel.nack(msg, false, false);
                throw new Error(error);
            }
        }
    }
    async notifyUsersAndDevicesForAccessLog(params) {
        const { logItem, insertedLog, organizationId } = params;
        if (logItem.v && logItem.rg && logItem.rg.length > 0) {
            setTimeout(() => {
                this.terminateVisit(logItem);
            }, 100, this);
        }
        dal_manager_1.dbManager.accessPacs2.onAccessLogsChanged({
            reason: app_enums_1.enums.RecalculateWorkReason.NewAccessLog,
            organizationId: organizationId,
            userId: logItem.o,
            acpId: logItem.a,
            timestamp: new Date(logItem.u),
            direction: logItem.d,
        });
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.activity + "." + (logItem.a ? logItem.a : "*") + "." + (logItem.d ? logItem.d : "*") + "." + (logItem.s ? logItem.s : "*"), {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.newlog,
            p: insertedLog,
            oId: organizationId,
        });
        await this.checkAccessNotificationFromCache(insertedLog);
    }
    async onStatusSensorState(msg, organizationId, log, safe) {
        try {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const statusSensorLogId = await dal_manager_1.dbManager.accessLog.uploadStatusSensorLogsPg({ organizationId, sensorLogs: [log], trx })[0];
                if (log.l && statusSensorLogId) {
                    await dal_manager_1.dbManager.accessLog.updateAccessLogForSensorData({ organizationId, logId: log.l, statusSensorLogId, trx });
                }
            });
            this._channel.ack(msg);
        }
        catch (error) {
            try {
                if (!safe) {
                    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                        return (0, cli_queries_1.insertUnhandledDeviceLog)({
                            organizationId,
                            deviceId: log.ci,
                            type: messagebroker_models_1.DeviceToServerMessageTypes.StatusSensorStateChanged,
                            logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant,
                            data: log,
                            note: { error: error.message ?? error },
                            trx,
                        });
                    });
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming status sensor Log, saved to unhandled_device_log table %s", error.message || error);
                }
                else {
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming status sensor Log, sended to deadLetter %s", error.message || error);
                }
            }
            catch (error) {
                app_logs_1.logger.error("[device-to-server.sub] error while writing to unhandledDeviceLogs", error.message || error);
            }
            finally {
                this._channel.nack(msg, false, false);
                throw new Error(error);
            }
        }
    }
    async notifyUsersForStatusSensorLog(organizationId, log) {
        if (log.f) {
            (0, business_notification_1.handDoorOpenedByForceNotifications)({
                accessPointId: log.a,
                dateTime: new Date(log.u),
                logId: log.l,
                organizationId,
                reason: log.r,
                terminalId: log.ci,
            });
        }
        else {
            await this.checkStatusNotificationFromCache(organizationId, log);
        }
    }
    async onTamperswitch(msg, organizationId, packet, safe) {
        try {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                await dal_manager_1.dbManager.accessLog.uploadTamperSwitchLogsPg({
                    organizationId: organizationId,
                    tamperSwitchLogs: [packet],
                    trx,
                });
            });
            this._channel.ack(msg);
        }
        catch (error) {
            try {
                if (!safe) {
                    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                        return (0, cli_queries_1.insertUnhandledDeviceLog)({
                            organizationId,
                            deviceId: packet.ci,
                            type: messagebroker_models_1.DeviceToServerMessageTypes.TamperEvent,
                            logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant,
                            data: packet,
                            note: { error: error.message ?? error },
                            trx,
                        });
                    });
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming tamper switch log, saved to unhandled_device_log table %s", error.message || error);
                }
                else {
                    app_logs_1.logger.error("[device-to-server.sub] error while consuming tamper switch log, sended to deadLetter %s", error.message || error);
                }
            }
            catch (error) {
                app_logs_1.logger.error("[device-to-server.sub] error while writing to unhandledDeviceLogs", error.message || error);
            }
            finally {
                this._channel.nack(msg, false, false);
                throw new Error(error);
            }
        }
    }
    async checkAccessNotificationFromCache(log) {
        if (log.mp) {
            for (const pattern of log.mp) {
                const accessNotifications = await dal_manager_1.dbManager.accessRedisCache.getValue(pattern);
                if (accessNotifications) {
                    const notifications = JSON.parse(accessNotifications);
                    for (const notificationId of notifications) {
                        const result = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                            const accessNotification = await dal_manager_1.dbManager.accessNotifications.getAccessNotification({ notificationId, organizationId: log.oId, trx });
                            if (!(0, dal_utils_1.checkTimeRange)(accessNotification.options)) {
                                return null;
                            }
                            const receiverFilter = await dal_manager_1.dbManager.accessNotifications.getNotificationReceiverFilter({ notificationId, organizationId: log.oId, trx });
                            if (accessNotification.type === restapi_1.NotificationType.IdentityAccess) {
                                const instanceData = {
                                    i: log.id,
                                    on: log.on,
                                    a: log.a,
                                    r: log.r,
                                    d: log.d,
                                    t: new Date(log.u),
                                    s: log.sid,
                                };
                                const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
                                    createdT: new Date(),
                                    notificationId,
                                    organizationId: log.oId,
                                    receiverFilterId: receiverFilter.id,
                                    trx,
                                    instanceData,
                                });
                                return {
                                    notificationEventId,
                                    instanceData,
                                };
                            }
                            else if (accessNotification.type === restapi_1.NotificationType.SelfIdentityAccess) {
                                const bindingKeys = [];
                                const filteredUserIds = (await trx.query((0, dal_utils_1.generateUserFilterQuery)(log.oId, receiverFilter.filter.id, bindingKeys), bindingKeys)).rows.map((r) => r.userId);
                                if (filteredUserIds.includes(log.o)) {
                                    const instanceData = {
                                        m: accessNotification.settings?.message,
                                        d: accessNotification.options.direction,
                                        r: log.r,
                                        a: log.a,
                                        receiverUserIds: [log.o],
                                    };
                                    const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
                                        createdT: new Date(),
                                        notificationId,
                                        organizationId: log.oId,
                                        trx,
                                        instanceData,
                                    });
                                    return { notificationEventId, instanceData };
                                }
                                else {
                                    return null;
                                }
                            }
                            else {
                                return null;
                            }
                        });
                        if (result?.notificationEventId) {
                            (0, messageBroker_notification_pub_1.publishToNotificationService)({
                                a: 0,
                                i: result.notificationEventId,
                                n: app_enums_1.enums.AmqpMessageCode.Empty,
                                o: log.oId,
                                v: "1",
                                d: result.instanceData,
                            });
                        }
                    }
                }
            }
        }
    }
    async terminateVisit(log) {
        let activeVisitsOfUser = await dal_manager_1.dbManager.accessVisitor.listActiveVisitsOfVisitorByUserId(log.oId, log.o);
        if (!activeVisitsOfUser || activeVisitsOfUser.length < 1) {
            return;
        }
        let terminateVisits = false;
        for (const region of log.rg) {
            if (region.s != app_enums_1.enums.AntiPassbackState.Out) {
                continue;
            }
            const regionCached = await dal_manager_1.dbManager.accessRedisCache.getRegionCache({ organizationId: log.oId, regionId: region.i });
            if (!regionCached || !regionCached.v || !regionCached.t) {
                continue;
            }
            terminateVisits = true;
            break;
        }
        if (terminateVisits) {
            let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(log.oId);
            for (const visitId of activeVisitsOfUser) {
                if (!process.env.TERMINATE_VISIT_IMMEDIATELY) {
                    setTimeout(async () => {
                        await dal_manager_1.dbManager.accessVisitor.terminateVisit(log.oId, visitId, organizationVisitorModuleSettings, dal_constants_1.DalConstants.SystemUserId);
                        app_httpserver_1.httpServer.sendVisitorListChangedNotification(log.oId);
                    }, 60 * 1000);
                }
                else if (process.env.TERMINATE_VISIT_IMMEDIATELY === "true") {
                    await dal_manager_1.dbManager.accessVisitor.terminateVisit(log.oId, visitId, organizationVisitorModuleSettings, dal_constants_1.DalConstants.SystemUserId);
                    app_httpserver_1.httpServer.sendVisitorListChangedNotification(log.oId);
                }
            }
        }
    }
    async checkStatusNotificationFromCache(organizationId, log) {
        if (log.mp) {
            for (const pattern of log.mp) {
                const statusNotifications = await dal_manager_1.dbManager.accessRedisCache.getValue(pattern);
                if (statusNotifications) {
                    const notificationIds = JSON.parse(statusNotifications);
                    for (const notificationId of notificationIds) {
                        const notificationEventId = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                            const receiverFilter = await dal_manager_1.dbManager.accessNotifications.getNotificationReceiverFilter({ notificationId, organizationId, trx });
                            return dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
                                createdT: new Date(),
                                notificationId,
                                organizationId,
                                receiverFilterId: receiverFilter.id,
                                trx,
                                instanceData: {
                                    l: log.l,
                                    a: log.a,
                                    r: log.r,
                                    t: new Date(log.u),
                                },
                            });
                        });
                        (0, messageBroker_notification_pub_1.publishToNotificationService)({
                            a: 0,
                            i: notificationEventId,
                            n: app_enums_1.enums.AmqpMessageCode.Empty,
                            o: organizationId,
                            v: "1",
                            d: {
                                l: log.l,
                                a: log.a,
                                r: log.r,
                                t: new Date(log.u),
                            },
                        });
                    }
                }
            }
        }
    }
    async checkCounterNotificationFromCache(organizationId, log) {
        if (log.mp) {
            for (const pattern of log.mp) {
                const counterNotifications = await dal_manager_1.dbManager.accessRedisCache.getValue(pattern);
                if (counterNotifications) {
                    const notificationIds = JSON.parse(counterNotifications);
                    for (const notificationId of notificationIds) {
                        const notificationEventId = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                            const receiverFilter = await dal_manager_1.dbManager.accessNotifications.getNotificationReceiverFilter({ notificationId, organizationId, trx });
                            return dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
                                createdT: new Date(),
                                notificationId,
                                organizationId,
                                receiverFilterId: receiverFilter.id,
                                trx,
                                instanceData: {
                                    l: log.l,
                                    a: log.a,
                                    r: log.r,
                                    t: new Date(log.u),
                                },
                            });
                        });
                        (0, messageBroker_notification_pub_1.publishToNotificationService)({
                            a: 0,
                            i: notificationEventId,
                            n: app_enums_1.enums.AmqpMessageCode.Empty,
                            o: organizationId,
                            v: "1",
                            d: {
                                l: log.l,
                                a: log.a,
                                r: log.r,
                                t: new Date(log.u),
                            },
                        });
                    }
                }
            }
        }
    }
    async onSystemHealth(msg, organizationId, deviceId, packet) {
        try {
            const did = packet.did || deviceId;
            const systemHealth = await (0, business_device_1.uploadDeviceSystemStatusLogs)(organizationId, did, [packet]);
            this._channel.ack(msg);
            return systemHealth;
        }
        catch (error) {
            this._channel.nack(msg, false, false);
            app_logs_1.logger.error("[device-to-server.sub] error while consuming Access Log, sended to deadLetter %s", error.message || error);
            throw new Error(error);
        }
    }
    async notifyUsersAboutSystemHealth(organizationId, deviceId, deviceHealth) {
        if (deviceHealth) {
            messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.systemStatus + ".*", {
                e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.systemhealth,
                p: deviceId,
                oId: organizationId,
            });
        }
    }
}
exports.amqpDeviceToServerSub = new AmqpDeviceToServerSub();
