"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.onDeadLetterDeviceSystemStatusLogs = exports.onDeadLetterTamperSwitch = exports.onDeadLetterStatusSensorLog = exports.onDeadLetterCounterSensorLog = exports.onDeadLetterAccessLog = exports.onDeadLetterMessage = exports.amqpToDeadLetterInit = void 0;
const enums_1 = require("../lib/access-model/v2/enums");
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const business_device_1 = require("../business/business.device");
const cli_queries_1 = require("../dal/access/psql/cli-queries");
const messagebroker_models_1 = require("./messagebroker.models");
const dal_access_psql_log_1 = require("../dal/access/psql/dal.access.psql.log");
let channel;
let pool;
let redisClient;
const appTag = "[amqp-to-cli.sub]";
async function amqpToDeadLetterInit(connection, initClients) {
    return new Promise(async (resolve, reject) => {
        channel = connection.createChannel({
            json: true,
            name: "amqp-to-cli.sub",
            setup: async (channel) => {
                app_logs_1.logger.debug("channel is establishing...", appTag);
                await Promise.all([
                    channel.prefetch(1),
                    channel.assertExchange(messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, "direct", { durable: true }),
                    channel.assertQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, {
                        durable: true,
                    }),
                ]);
                await channel.bindQueue(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, messagebroker_models_1.MessageBrokerNames.deadLetter.exchange, messagebroker_models_1.MessageBrokerNames.deadLetter.deviceBindingKey);
                await channel.consume(messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue, exports.onDeadLetterMessage, { noAck: false });
            },
        });
        const clients = await initClients();
        pool = clients.pool;
        redisClient = clients.redis;
        channel.on("error", (err) => app_logs_1.logger.error("[amqp-to-cli.sub] error while creating channel: ", err));
        channel.on("connect", () => app_logs_1.logger.info("[amqp-to-cli.sub] channel created"));
        channel.on("close", () => app_logs_1.logger.info("[amqp-to-cli.sub] channel closed"));
        channel.once("connect", () => resolve());
    });
}
exports.amqpToDeadLetterInit = amqpToDeadLetterInit;
const onDeadLetterMessage = async (msg) => {
    try {
        app_logs_1.logger.debug("Dead Letter Message Received");
        const data = JSON.parse(msg.content.toString());
        const organizationId = msg.properties.headers.o;
        const deviceId = msg.properties.headers.d;
        if (msg === null) {
            return;
        }
        if (data.e === messagebroker_models_1.DeviceToServerMessageTypes.AccessLog && data.p === null) {
            app_logs_1.logger.debug("Access log payload is empty. Discarding message.", appTag);
            channel.nack(msg, false, false);
            return;
        }
        const organizationExists = await (0, cli_queries_1.systemTransaction)(pool, async (trx) => (await trx.query(`SELECT id FROM public."organizationList" WHERE id = $1`, [organizationId])).rowCount);
        if (!organizationExists) {
            app_logs_1.logger.warn(`Organization[${organizationId}] is not exist. Discarding message...`);
            channel.nack(msg, false, false);
            return;
        }
        switch (data.e) {
            case messagebroker_models_1.DeviceToServerMessageTypes.AccessLog: {
                const deadLetterResult = await onDeadLetterAccessLog({
                    msg,
                    organizationId,
                    deviceId,
                    deviceAccessLog: data.p,
                    redisClient,
                    pool,
                });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.CounterSensorEvent: {
                const deadLetterResult = await onDeadLetterCounterSensorLog({ msg, organizationId, packet: data.p, pool });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.StatusSensorStateChanged: {
                const deadLetterResult = await onDeadLetterStatusSensorLog({ msg, organizationId, log: (data.p || data.pp), pool });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.TamperEvent: {
                const deadLetterResult = await onDeadLetterTamperSwitch({ msg, organizationId, packet: data.p, pool });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.SystemHealth: {
                const deadLetterResult = await onDeadLetterDeviceSystemStatusLogs({
                    msg,
                    organizationId,
                    deviceId,
                    packet: data.p,
                    pool,
                    redisClient,
                });
                if (deadLetterResult.isSuccess) {
                    channel.ack(msg);
                }
                else {
                    channel.nack(msg, false, deadLetterResult.requeueMessage);
                }
                break;
            }
            case messagebroker_models_1.DeviceToServerMessageTypes.EmergencyStateChanged:
            case messagebroker_models_1.DeviceToServerMessageTypes.WebRTC:
                app_logs_1.logger.debug("Discarding WebRtc or Emergency Message...");
                channel.ack(msg);
                break;
            default:
                app_logs_1.logger.error("[amqp-to-cli.sub] unknown msg %s", data.e);
                await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                    return (0, cli_queries_1.insertUnhandledDeviceLog)({
                        organizationId,
                        deviceId,
                        logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                        type: data.e,
                        note: { message: "dead letter received an unknown message type" },
                        data: data.p,
                        trx,
                    });
                });
                channel.nack(msg, false, true);
                break;
        }
        app_logs_1.logger.debug("Offline Log Processed");
    }
    catch (error) {
        app_logs_1.logger.error("[amqp-to-cli.sub] error while consuming msg %s", error.message || error);
        channel.nack(msg, false, true);
    }
};
exports.onDeadLetterMessage = onDeadLetterMessage;
async function onDeadLetterAccessLog(params) {
    try {
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            const logItem = await (0, business_device_1.updateLogFromDeviceWithClient)({
                organizationId: params.organizationId,
                deviceLog: params.deviceAccessLog,
                logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                redisCache: params.redisClient,
                trx,
            });
            const insertedLog = await (0, cli_queries_1.addAccessLogFromDevice)({ organizationId: params.organizationId, log: logItem, trx });
            if (insertedLog && insertedLog.rg && insertedLog.rg.length > 0) {
                for (const regionLog of insertedLog.rg) {
                    const regionNotification = {
                        i: regionLog.i,
                        s: regionLog.s.valueOf(),
                        ee: regionLog.ee,
                        xe: regionLog.xe,
                        o: insertedLog.o,
                        t: insertedLog.u,
                    };
                    await (0, cli_queries_1.upsertRegionStates)(params.organizationId, [{ accessControlPointId: insertedLog.a, log: regionNotification }], trx);
                    if (regionLog.ru && regionLog.ru.c) {
                        await (0, cli_queries_1.upsertAccessRuleHistory)(params.organizationId, {
                            date: new Date(),
                            accessRuleId: regionLog.ru.i,
                            userId: insertedLog.o,
                            count: regionLog.ru.c,
                            accessControlPointId: insertedLog.a,
                            actionDateISO: insertedLog.u,
                            startDateTime: regionLog.ru.sd,
                            endDateTime: regionLog.ru.ed,
                        }, trx);
                    }
                    if (regionLog.rti) {
                        await (0, cli_queries_1.updateUserRegionTicketUnits)(params.organizationId, {
                            userId: insertedLog.o,
                            accessControlPointId: insertedLog.a,
                            decrementUnit: regionLog.rti.c,
                            regionTicketId: regionLog.rti.i,
                        }, trx);
                    }
                }
                if (insertedLog.ir !== true && insertedLog.d && insertedLog.d === enums_1.EnumsV2.AccessDirection.Entrance && insertedLog.s === true) {
                    await (0, dal_access_psql_log_1.assignAutoShift)(params.organizationId, {
                        credentialOwnerUserId: insertedLog.o,
                        generationTime: insertedLog.u,
                        redisCache: params.redisClient,
                        logId: insertedLog.id,
                    }, trx);
                }
            }
            if (insertedLog) {
                await (0, cli_queries_1.onAccessLogsChanged)({
                    reason: app_enums_1.enums.RecalculateWorkReason.OfflineLogReceived,
                    organizationId: params.organizationId,
                    userId: logItem.o,
                    acpId: logItem.a,
                    timestamp: new Date(logItem.u),
                    direction: logItem.d,
                    trx,
                });
            }
        });
        return {
            isSuccess: true,
        };
    }
    catch (error) {
        app_logs_1.logger.debug("Error while consuming dead letter access log: " + error);
        if (error.message.startsWith(`invalid byte sequence for encoding "UTF8"`)) {
            app_logs_1.logger.warn(`Corrupt credential data from deviceId[${params.deviceId}] accessPoint[${params.deviceAccessLog.a}]`);
            return {
                isSuccess: false,
                requeueMessage: false,
            };
        }
        else {
            await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
                await (0, cli_queries_1.insertUnhandledDeviceLog)({
                    organizationId: params.organizationId,
                    deviceId: params.deviceId,
                    logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                    type: messagebroker_models_1.DeviceToServerMessageTypes.AccessLog,
                    data: params.deviceAccessLog,
                    note: { error: error.message ?? error },
                    trx,
                });
            });
            return {
                isSuccess: false,
                requeueMessage: false,
            };
        }
    }
}
exports.onDeadLetterAccessLog = onDeadLetterAccessLog;
async function onDeadLetterCounterSensorLog(params) {
    try {
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            const counterSensorLogIds = await (0, cli_queries_1.uploadCounterLogs)({ organizationId: params.organizationId, logs: [params.packet], trx });
            if (params.packet.l && counterSensorLogIds.length > 0) {
                await (0, cli_queries_1.updateAccessLogForSensorData)({ organizationId: params.organizationId, logId: params.packet.l, counterSensorLogId: counterSensorLogIds[0], trx });
            }
        });
        return {
            isSuccess: true,
        };
    }
    catch (error) {
        app_logs_1.logger.debug("Error while consuming dead letter access log: " + error);
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            await (0, cli_queries_1.insertUnhandledDeviceLog)({
                organizationId: params.organizationId,
                deviceId: params.packet.ci,
                logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                type: messagebroker_models_1.DeviceToServerMessageTypes.CounterSensorEvent,
                data: params.packet,
                note: { error: error.message ?? error },
                trx,
            });
        });
        return {
            isSuccess: false,
            requeueMessage: false,
        };
    }
}
exports.onDeadLetterCounterSensorLog = onDeadLetterCounterSensorLog;
async function onDeadLetterStatusSensorLog(params) {
    try {
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            await (0, cli_queries_1.uploadStatusSensorLogsPg)({ organizationId: params.organizationId, sensorLogs: [params.log], trx });
        });
        return {
            isSuccess: true,
        };
    }
    catch (error) {
        app_logs_1.logger.debug("Error while consuming dead letter access log: " + error);
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            return (0, cli_queries_1.insertUnhandledDeviceLog)({
                organizationId: params.organizationId,
                deviceId: params.log.ci,
                logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                type: messagebroker_models_1.DeviceToServerMessageTypes.StatusSensorStateChanged,
                data: params.log,
                note: { error: error.message ?? error },
                trx,
            });
        });
        return {
            isSuccess: false,
            requeueMessage: false,
        };
    }
}
exports.onDeadLetterStatusSensorLog = onDeadLetterStatusSensorLog;
async function onDeadLetterTamperSwitch(params) {
    try {
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            await (0, cli_queries_1.uploadTamperSwitchLogsPg)({ organizationId: params.organizationId, tamperSwitchLogs: [params.packet], trx });
        });
        return {
            isSuccess: true,
        };
    }
    catch (error) {
        app_logs_1.logger.debug("Error while consuming dead letter access log: " + error);
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            return (0, cli_queries_1.insertUnhandledDeviceLog)({
                organizationId: params.organizationId,
                logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                type: messagebroker_models_1.DeviceToServerMessageTypes.TamperEvent,
                deviceId: params.packet.ci,
                data: params.packet,
                note: { error: error.message ?? error },
                trx,
            });
        });
        return {
            isSuccess: false,
            requeueMessage: false,
        };
    }
}
exports.onDeadLetterTamperSwitch = onDeadLetterTamperSwitch;
async function onDeadLetterDeviceSystemStatusLogs(params) {
    try {
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            await (0, cli_queries_1.uploadDeviceSystemStatusLogsPg)({ organizationId: params.organizationId, deviceId: params.deviceId, deviceLogs: [params.packet], trx, redis: params.redisClient });
        });
        return {
            isSuccess: true,
        };
    }
    catch (error) {
        app_logs_1.logger.debug("Error while consuming dead letter access log: " + error);
        await (0, cli_queries_1.systemTransaction)(params.pool, async (trx) => {
            return (0, cli_queries_1.insertUnhandledDeviceLog)({
                organizationId: params.organizationId,
                logReceiveMethod: app_enums_1.enums.LogReceiveMethod.DeadLetter,
                type: messagebroker_models_1.DeviceToServerMessageTypes.SystemHealth,
                deviceId: params.deviceId,
                data: params,
                note: { error: error.message ?? error },
                trx,
            });
        });
        return {
            isSuccess: false,
            requeueMessage: false,
        };
    }
}
exports.onDeadLetterDeviceSystemStatusLogs = onDeadLetterDeviceSystemStatusLogs;
