"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.emailRegex = exports.wiegandDataFormatter = exports.uuidV4Hyphen = exports.validatePgPassFile = exports.baseConfigJoi = exports.joiToString = exports.joiFileExistsValidation = exports.getEpoch = exports.uuidV4DashedHex = exports.uuidV4Base64 = exports.uuidV4Hex = exports.validateTls = void 0;
const tls_1 = __importDefault(require("tls"));
const net_1 = __importDefault(require("net"));
const joi_1 = __importDefault(require("joi"));
const os_1 = __importDefault(require("os"));
const crypto_1 = __importDefault(require("crypto"));
const assert_1 = __importDefault(require("assert"));
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
const terminal_1 = require("../es/models/terminal");
const defaultRabbitmqClientSsl = process.env.ARMON_CONFIG_DIRECTORY ? path_1.default.join(process.env.ARMON_CONFIG_DIRECTORY, "ssl") : path_1.default.join("/", "var", "opt", "armon", "webapp", "certs");
function validateTls(caFile, certFile, keyFile) {
    const sca = new tls_1.default.TLSSocket(new net_1.default.Socket(), { cert: caFile });
    const scert = new tls_1.default.TLSSocket(new net_1.default.Socket(), { cert: certFile });
    const ca = sca.getCertificate();
    const cert = scert.getCertificate();
    assert_1.default.deepStrictEqual(ca.subject, cert.issuer, new Error("ca.subject and cert.issuer are imcompatable"));
    const rnd = crypto_1.default.randomBytes(50);
    const key = keyFile;
    const sign = crypto_1.default.createSign("RSA-SHA256");
    sign.update(rnd);
    const signature = sign.sign(key);
    const publicKey = certFile;
    const verifier = crypto_1.default.createVerify("RSA-SHA256");
    verifier.update(rnd);
    const result = verifier.verify(publicKey, signature);
    if (!result) {
        throw new Error("Key is not valid.");
    }
    return cert.subject;
}
exports.validateTls = validateTls;
function uuid() {
    const rnds = crypto_1.default.randomBytes(16);
    rnds[6] = (rnds[6] & 0x0f) | 0x40;
    rnds[8] = (rnds[8] & 0x3f) | 0x80;
    return rnds;
}
function uuidV4Hex() {
    return uuid().toString("hex");
}
exports.uuidV4Hex = uuidV4Hex;
function uuidV4Base64() {
    return uuid().toString("base64").substring(0, 22);
}
exports.uuidV4Base64 = uuidV4Base64;
function uuidV4DashedHex() {
    const val = uuidV4Hex();
    return [val.substring(0, 8), val.substring(8, 12), val.substring(12, 16), val.substring(16, 20), val.substring(20)].join("-");
}
exports.uuidV4DashedHex = uuidV4DashedHex;
function getEpoch() {
    return new Date().getTime();
}
exports.getEpoch = getEpoch;
const joiFileExistsValidation = (value) => {
    const r = path_1.default.resolve(value);
    if (r) {
        if (!fs_1.default.existsSync(r)) {
            throw new Error(`file ${r} is not found!`);
        }
        return r;
    }
    else {
        return undefined;
    }
};
exports.joiFileExistsValidation = joiFileExistsValidation;
function joiToString(error) {
    return error.message + " " + JSON.stringify(error.details, null, 4);
}
exports.joiToString = joiToString;
exports.baseConfigJoi = {
    PGPASSFILE: joi_1.default.custom((value) => {
        const error = joi_1.default.custom(exports.joiFileExistsValidation).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.PGPASSFILE = value || getDefaultPgPassLocation();
        return process.env.PGPASSFILE;
    }).default(() => {
        process.env.PGPASSFILE = getDefaultPgPassLocation();
        return process.env.PGPASSFILE;
    }),
    PGHOST: joi_1.default.custom((value) => {
        const error = joi_1.default.string().hostname().validate(value).error;
        if (error) {
            throw error;
        }
        process.env.PGHOST = value;
        return value;
    }).default(() => {
        process.env.PGHOST = "127.0.0.1";
        return process.env.PGHOST;
    }),
    PGPORT: joi_1.default.custom((value) => {
        const error = joi_1.default.number().port().validate(value).error;
        if (error) {
            throw error;
        }
        process.env.PGPORT = value;
        return value;
    }).default(() => {
        process.env.PGPORT = "5432";
        return process.env.PGPORT;
    }),
    PGUSER: joi_1.default.custom((value) => {
        const error = joi_1.default.string().max(255).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.PGUSER = value;
        return value;
    }).default(() => {
        process.env.PGUSER = "armonapi";
        return process.env.PGUSER;
    }),
    PGSUPERUSER: joi_1.default.custom((value) => {
        const error = joi_1.default.string().max(255).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.PGSUPERUSER = value;
        return value;
    }).default(() => {
        process.env.PGSUPERUSER = "armonsuper";
        return process.env.PGSUPERUSER;
    }),
    PGDATABASE: joi_1.default.custom((value) => {
        const error = joi_1.default.string().max(255).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.PGDATABASE = value;
        return value;
    }).default(() => {
        process.env.PGDATABASE = "armon";
        return process.env.PGDATABASE;
    }),
    PGAPPNAME: joi_1.default.custom((value) => {
        process.env.PGAPPNAME = value;
        return value;
    }).default(() => {
        throw new Error("This field, PGAPPNAME, must be suplied, overwrite to your service's config");
    }),
    NODE_ENV: joi_1.default.custom((value) => {
        process.env.NODE_ENV = value;
        return value;
    }).default(() => {
        process.env.NODE_ENV = "production";
        return process.env.NODE_ENV;
    }),
    LOG_DIRECTORY: joi_1.default.custom((value) => {
        process.env.LOG_DIRECTORY = value;
        return value;
    }).default(() => {
        process.env.LOG_DIRECTORY =
            os_1.default.platform() === "win32" ? path_1.default.resolve(process.env.APPDATA, "armon", "webapp", "log") : path_1.default.resolve("/", "var", "opt", "armon", "webapp", "log");
        return process.env.LOG_DIRECTORY;
    }),
    LOG_LEVEL: joi_1.default.custom((value) => {
        process.env.LOG_LEVEL = value;
        return value;
    }).default(() => {
        process.env.LOG_LEVEL = process.env.NODE_ENV === "development" ? "debug" : "info";
        return process.env.LOG_LEVEL;
    }),
    LOG_LAST_DAY_COUNT: joi_1.default.custom((value) => {
        const error = joi_1.default.string().max(255).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.LOG_LAST_DAY_COUNT = value;
        return value;
    }).default(() => {
        process.env.LOG_LAST_DAY_COUNT = "60";
        return process.env.LOG_LAST_DAY_COUNT;
    }),
    AMQP_HOSTNAME: joi_1.default.custom((value) => {
        if (value != "aw-rabbitmq") {
            const error = joi_1.default.string().domain().validate(value).error;
            if (error) {
                throw error;
            }
        }
        process.env.AMQP_HOSTNAME = value;
        return value;
    }).default(() => {
        process.env.AMQP_HOSTNAME = "localhost";
        return process.env.AMQP_HOSTNAME;
    }),
    AMQP_MANAGEMENT_API_PORT: joi_1.default.custom((value) => {
        const amqpManagementApiPort = parseInt(value);
        const error = joi_1.default.number().port().validate(amqpManagementApiPort).error;
        if (error) {
            throw error;
        }
        process.env.AMQP_MANAGEMENT_API_PORT = value;
        return amqpManagementApiPort;
    }).default(() => {
        process.env.AMQP_MANAGEMENT_API_PORT = "15672";
        return parseInt(process.env.AMQP_MANAGEMENT_API_PORT);
    }),
    AMQP_VHOST: joi_1.default.custom((value) => {
        const error = joi_1.default.string().max(255).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.AMQP_VHOST = value;
        return value;
    }).default(() => {
        process.env.AMQP_VHOST = "/";
        return process.env.AMQP_VHOST;
    }),
    AMQP_PORT: joi_1.default.custom((value) => {
        const amqpPort = parseInt(value);
        const error = joi_1.default.number().port().validate(amqpPort).error;
        if (error) {
            throw error;
        }
        process.env.AMQP_PORT = value;
        return amqpPort;
    }).default(() => {
        process.env.AMQP_PORT = "5671";
        return parseInt(process.env.AMQP_PORT);
    }),
    AMQP_SERVER_URL: joi_1.default.custom((value) => {
        const error = joi_1.default.string()
            .uri({
            scheme: ["amqps"],
        })
            .validate(value).error;
        if (error) {
            throw error;
        }
        process.env.AMQP_SERVER_URL = value;
        return value;
    }).default(() => {
        process.env.AMQP_SERVER_URL = "amqps://localhost:5671/";
        return process.env.AMQP_SERVER_URL;
    }),
    AMQPS_CONN_HEARTBEAT: joi_1.default.custom((value) => {
        const heartbeat = parseInt(value);
        const error = joi_1.default.number().integer().min(1).max(10).validate(heartbeat).error;
        if (error) {
            throw error;
        }
        process.env.AMQPS_CONN_HEARTBEAT = value;
        return heartbeat;
    }).default(() => {
        process.env.AMQPS_CONN_HEARTBEAT = "10";
        return parseInt(process.env.AMQPS_CONN_HEARTBEAT);
    }),
    AMQPS_CLIENT_TLS_CA: joi_1.default.custom((value) => {
        const error = joi_1.default.custom(exports.joiFileExistsValidation).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.AMQPS_CLIENT_TLS_CA = value;
        return value;
    }).default(() => {
        if (os_1.default.platform() === "win32") {
            throw new Error("You must declare AMQPS_CLIENT_TLS_CA env.");
        }
        process.env.AMQPS_CLIENT_TLS_CA = path_1.default.join(defaultRabbitmqClientSsl, "chain.ca.crt.pem");
        return process.env.AMQPS_CLIENT_TLS_CA;
    }),
    AMQPS_CLIENT_TLS_CERT: joi_1.default.custom((value) => {
        const error = joi_1.default.custom(exports.joiFileExistsValidation).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.AMQPS_CLIENT_TLS_CERT = value;
        return value;
    }).default(() => {
        if (os_1.default.platform() === "win32") {
            throw new Error("You must declare AMQPS_CLIENT_TLS_CERT env.");
        }
        process.env.AMQPS_CLIENT_TLS_CERT = path_1.default.join(defaultRabbitmqClientSsl, "webappclient.crt.pem");
        return process.env.AMQPS_CLIENT_TLS_CERT;
    }),
    AMQPS_CLIENT_TLS_KEY: joi_1.default.custom((value) => {
        const error = joi_1.default.custom(exports.joiFileExistsValidation).validate(value).error;
        if (error) {
            throw error;
        }
        process.env.AMQPS_CLIENT_TLS_KEY = value;
        return value;
    }).default(() => {
        if (os_1.default.platform() === "win32") {
            throw new Error("You must declare AMQPS_CLIENT_TLS_KEY env.");
        }
        process.env.AMQPS_CLIENT_TLS_KEY = path_1.default.join(defaultRabbitmqClientSsl, "webappclient.key.pem");
        return process.env.AMQPS_CLIENT_TLS_KEY;
    }),
    REDIS_SERVER_URL: joi_1.default.custom((value) => {
        const error = joi_1.default.string()
            .uri({ scheme: ["redis", "rediss"] })
            .validate(value).error;
        if (error) {
            throw error;
        }
        process.env.REDIS_SERVER_URL = value;
        return value;
    }).default(() => {
        process.env.REDIS_SERVER_URL = "redis://localhost:6379/1";
        return process.env.REDIS_SERVER_URL;
    }),
};
function validatePgPassFile(pgPassFileLocation) {
    if (!pgPassFileLocation) {
        pgPassFileLocation = getDefaultPgPassLocation();
        process.env.PGPASSFILE = pgPassFileLocation;
    }
    if (!fs_1.default.existsSync(pgPassFileLocation)) {
        console.error(`PGPASSFILE not found in ${pgPassFileLocation} directory! 
			Please provide pgpassfile in the format according to the link https://www.postgresql.org/docs/12/libpq-pgpass.html
			IMPORTANT NOTE!!! 
			Both db user (generally armonapi) and superuser (generally armonsuper) user entries are necessary!`);
        process.exit(-1);
    }
    else {
        const pgpassFileContent = fs_1.default.readFileSync(pgPassFileLocation, { encoding: "utf8" });
        let lines = pgpassFileContent.split(os_1.default.EOL);
        lines = lines.filter((l) => l !== "");
        const dbUsers = [];
        for (const line of lines) {
            const dbUser = line.split(":")[3];
            if (!dbUsers.includes(dbUser)) {
                dbUsers.push(dbUser);
            }
        }
        if (!dbUsers.includes(process.env.PGSUPERUSER)) {
            console.error(`${process.env.PGSUPERUSER} user not supplied in pgpassfile in location ${pgPassFileLocation}!. Please check file contents and try again`);
            process.exit(-1);
        }
        if (os_1.default.platform() !== "win32") {
            const stats = fs_1.default.statSync(pgPassFileLocation);
            const unixFilePermissions = "0" + (stats.mode & parseInt("777", 8)).toString(8);
            if (unixFilePermissions !== "0600") {
                console.error(`Wrong permissions supplied for pgpassfile. Please run the command below as root:
				chmod 600 ${pgPassFileLocation}
				`);
            }
        }
        try {
            process.env.PGPASSFILE = pgPassFileLocation;
            if (lines.length !== 2) {
                throw Error("There are less or more pgpassfile entries then expected! \n 2 entries (one for standart, one for super user) expected!");
            }
            else {
                for (const line of lines) {
                    const lineContents = line.split(":");
                    if (!process.env.PGHOST) {
                        process.env.PGHOST = lineContents[0];
                    }
                    else if (process.env.PGHOST !== lineContents[0]) {
                        throw Error("HOST entries in PG PASSFILE content is different from each other! \n Please check PGPASSFILE contents on location " + pgPassFileLocation);
                    }
                    if (!process.env.PGPORT) {
                        process.env.PGPORT = lineContents[1];
                    }
                    else if (process.env.PGPORT !== lineContents[1]) {
                        throw Error("PORT entries in PG PASSFILE content is different from each other! \n Please check PGPASSFILE contents on location " + pgPassFileLocation);
                    }
                    if (!process.env.PGDATABASE) {
                        process.env.PGDATABASE = lineContents[2];
                    }
                    else if (process.env.PGDATABASE !== lineContents[2]) {
                        throw Error("DBNAME entries in PG PASSFILE content is different from each other! \n Please check PGPASSFILE contents on location " + pgPassFileLocation);
                    }
                    if (lineContents[3] === "armonsuper") {
                        process.env.PGSUPERUSER = "armonsuper";
                    }
                    else {
                        process.env.PGUSER = lineContents[3];
                    }
                }
            }
        }
        catch (error) {
            console.error(error);
            process.exit(-1);
        }
    }
    return true;
}
exports.validatePgPassFile = validatePgPassFile;
function getDefaultPgPassLocation() {
    return os_1.default.platform() !== "win32" ? path_1.default.resolve("/", "var", "lib", "postgresql", ".pgpass") : path_1.default.resolve(process.env.APPDATA, "postgresql", "pgpass.conf");
}
function uuidV4Hyphen(uuid) {
    return uuid.substr(0, 8) + "-" + uuid.substr(8, 4) + "-" + uuid.substr(12, 4) + "-" + uuid.substr(16, 4) + "-" + uuid.substr(20, 12);
}
exports.uuidV4Hyphen = uuidV4Hyphen;
function wiegandDataFormatter(data, format) {
    const cardId = parseInt(data.toString("hex"), 16);
    switch (format) {
        case terminal_1.CredentialDataFormat.W26:
            return parseInt(cardId.toString(2).padStart(26, "0").substring(1, 25));
            break;
        case terminal_1.CredentialDataFormat.W26a:
            return parseInt(cardId.toString(2).padStart(26, "0").substring(9, 25));
            break;
        case terminal_1.CredentialDataFormat.W34:
            return parseInt(cardId.toString(2).padStart(34, "0").substring(1, 33));
            break;
        case terminal_1.CredentialDataFormat.W34a:
            return parseInt(cardId.toString(2).padStart(34, "0").substring(9, 33));
            break;
        case terminal_1.CredentialDataFormat.W36:
            return parseInt(cardId.toString(2).padStart(36, "0").substring(17, 33));
            break;
        case terminal_1.CredentialDataFormat.W36a:
            return parseInt(cardId.toString(2).padStart(36, "0").substring(19, 35));
            break;
        case terminal_1.CredentialDataFormat.W37:
            return parseInt(cardId.toString(2).padStart(37, "0").substring(17, 36));
            break;
        case terminal_1.CredentialDataFormat.W37a:
            return parseInt(cardId.toString(2).padStart(37, "0").substring(20, 36));
            break;
        case terminal_1.CredentialDataFormat.W50:
            return parseInt(cardId.toString(2).padStart(50, "0").substring(17, 49));
            break;
        case terminal_1.CredentialDataFormat.W54:
            return parseInt(cardId.toString(2).padStart(54, "0").substring(1, 53));
            break;
        default:
            return cardId;
            break;
    }
}
exports.wiegandDataFormatter = wiegandDataFormatter;
exports.emailRegex = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
