"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseReceivingMessage = exports.prepareWebpResponseFailMessage = exports.createWebpResponseMessage = exports.createWebpCancelRequestMessage = exports.createWebpRequestMessage = exports.createPingMessage = exports.WEBRTC_HANG_UP = exports.WEBRTC_NEW_ICE_CANDIDATE = exports.WEBRTC_ANSWER = exports.WEBRTC_OFFER = void 0;
exports.WEBRTC_OFFER = "webrtc-offer";
exports.WEBRTC_ANSWER = "webrtc-answer";
exports.WEBRTC_NEW_ICE_CANDIDATE = "webrtc-new-ice-candidate";
exports.WEBRTC_HANG_UP = "webrtc-hang-up";
const PING_PONG = 0;
const WEBP_REQUEST = 1;
const WEBP_RESPONSE = 2;
const WEBP_FAIL_RESPONSE = 3;
const WEBP_CANCEL_REQUEST = 4;
const WEBP_RESPONSE_FIRST_CHUNK = 0;
const WEBP_RESPONSE_CONTINUED_CHUNK = 1;
const WEBP_RESPONSE_LAST_CHUNK = 2;
function createPingMessage(payload, textEncoder) {
    if (payload.length > 4000) {
        throw new Error("WebRTC Data Channel Protocol Error ping payload length must be less than 4000");
    }
    const protocolArr = new Uint8Array([PING_PONG]);
    const contentArr = textEncoder.encode(payload);
    const sendArr = new Uint8Array(protocolArr.length + contentArr.length);
    sendArr.set(protocolArr, 0);
    sendArr.set(contentArr, protocolArr.length);
    return sendArr.buffer;
}
exports.createPingMessage = createPingMessage;
function createWebpRequestMessage(chunkSize = 0, interval = 0, cameraId, requestId, textEncoder) {
    if (cameraId.length > 64) {
        throw new Error("WebRTC Data Channel Protocol Error webprequest cameraId length must be less than 48");
    }
    const protocolArr = new Uint8Array([WEBP_REQUEST, chunkSize, interval]);
    const cameraIdArr = textEncoder.encode(cameraId);
    const requestIdArr = textEncoder.encode(requestId);
    const sendArr = new Uint8Array(protocolArr.length + cameraIdArr.length + requestIdArr.length);
    sendArr.set(protocolArr, 0);
    sendArr.set(cameraIdArr, protocolArr.length);
    sendArr.set(requestIdArr, protocolArr.length + cameraIdArr.length);
    return sendArr.buffer;
}
exports.createWebpRequestMessage = createWebpRequestMessage;
function createWebpCancelRequestMessage(requestId, textEncoder) {
    if (requestId.length > 64) {
        throw new Error("WebRTC Data Channel Protocol Error webprequest cancel requestId length must be less than 48");
    }
    const protocolArr = new Uint8Array([WEBP_CANCEL_REQUEST]);
    const requestIdArr = textEncoder.encode(requestId);
    const sendArr = new Uint8Array(protocolArr.length + requestIdArr.length);
    sendArr.set(protocolArr, 0);
    sendArr.set(requestIdArr, protocolArr.length);
    return sendArr.buffer;
}
exports.createWebpCancelRequestMessage = createWebpCancelRequestMessage;
function createWebpResponseMessage(isFirst, isLast, requestId, webpId, data, textEncoder) {
    if (requestId.length !== 36) {
        throw new Error("WebRTC Data Channel Protocol Error requestid should 36 characters");
    }
    if (webpId.length !== 36) {
        throw new Error("WebRTC Data Channel Protocol Error webpid should 36 characters");
    }
    const protocolArr = new Uint8Array([WEBP_RESPONSE, isFirst ? WEBP_RESPONSE_FIRST_CHUNK : isLast ? WEBP_RESPONSE_LAST_CHUNK : WEBP_RESPONSE_CONTINUED_CHUNK]);
    const requestIdArr = textEncoder.encode(requestId);
    const webpIdArr = textEncoder.encode(webpId);
    const sendArr = new Uint8Array(protocolArr.length + requestIdArr.length + webpIdArr.length + data.length);
    sendArr.set(protocolArr, 0);
    sendArr.set(requestIdArr, protocolArr.length);
    sendArr.set(webpIdArr, protocolArr.length + requestIdArr.length);
    sendArr.set(data, protocolArr.length + requestIdArr.length + webpIdArr.length);
    return sendArr.buffer;
}
exports.createWebpResponseMessage = createWebpResponseMessage;
function prepareWebpResponseFailMessage(requestId, webpId, error, textEncoder) {
    if (requestId.length !== 36) {
        throw new Error("WebRTC Data Channel Protocol Error requestid should 36 characters");
    }
    if (webpId.length !== 36) {
        throw new Error("WebRTC Data Channel Protocol Error webpId should 36 characters");
    }
    const protocolArr = new Uint8Array([WEBP_FAIL_RESPONSE]);
    const requestIdArr = textEncoder.encode(requestId);
    const webpIdArr = textEncoder.encode(webpId);
    const errorMsgArr = textEncoder.encode(error);
    const sendArr = new Uint8Array(protocolArr.length + requestIdArr.length + webpId.length + errorMsgArr.length);
    sendArr.set(protocolArr, 0);
    sendArr.set(requestIdArr, protocolArr.length);
    sendArr.set(webpIdArr, protocolArr.length + requestIdArr.length);
    sendArr.set(errorMsgArr, protocolArr.length + webpIdArr.length);
    return sendArr.buffer;
}
exports.prepareWebpResponseFailMessage = prepareWebpResponseFailMessage;
function parseReceivingMessage(event, handlers, textDecoder) {
    const data = new Uint8Array(event.data);
    switch (data[0]) {
        case PING_PONG:
            return handlers.onPing(textDecoder.decode(data.slice(1)));
        case WEBP_REQUEST:
            return handlers.onWebpRequest(parseWebpRequestMessage(data, textDecoder));
        case WEBP_RESPONSE:
            return handlers.onWebpResponse(parseWebpResponse(data, textDecoder));
        case WEBP_FAIL_RESPONSE:
            return handlers.onWebpFailResponse(parseWebpFailResponse(data, textDecoder));
        case WEBP_CANCEL_REQUEST:
            return handlers.onWebpCancelRequest(parseWebpCancelRequest(data, textDecoder));
        default:
            throw new Error("Unknown protocol message");
    }
}
exports.parseReceivingMessage = parseReceivingMessage;
function parseWebpRequestMessage(data, textDecoder) {
    return {
        chunkSize: data[1],
        interval: data[2],
        cameraId: textDecoder.decode(data.slice(3, 39)),
        requestId: textDecoder.decode(data.slice(39)),
    };
}
function parseWebpResponse(data, textDecoder) {
    const requestId = textDecoder.decode(data.slice(2, 38));
    const webpId = textDecoder.decode(data.slice(38, 74));
    return {
        isFirst: data[1] === WEBP_RESPONSE_FIRST_CHUNK,
        isLast: data[1] === WEBP_RESPONSE_LAST_CHUNK,
        requestId,
        webpId,
        chunk: data.slice(74),
    };
}
function parseWebpFailResponse(data, textDecoder) {
    const requestId = textDecoder.decode(data.slice(1, 37));
    const webpId = textDecoder.decode(data.slice(37, 73));
    const error = textDecoder.decode(data.slice(73));
    return {
        requestId,
        webpId,
        error,
    };
}
function parseWebpCancelRequest(data, textDecoder) {
    const requestId = textDecoder.decode(data.slice(1, 37));
    return requestId;
}
