"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.Ziraat = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
require("pdfmake/build/vfs_fonts");
const app_config_1 = require("../../../app.config");
const moment_1 = __importDefault(require("moment"));
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("../../../app.enums");
const sharp_1 = __importDefault(require("sharp"));
const app_logs_1 = require("../../../app.logs");
const generator_1 = require("../../../business/report/generator");
const report_util_1 = require("../../../business/report/report.util");
const a_1 = require("../a");
const NOT_FOUND_IMAGE = "data:image/jpeg;base64,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";
class Ziraat extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._organizationId = request.organizationId;
        const filter = request.filter;
        this._options = {
            startUtc: filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateTimeRange)?.value?.start,
            endUtc: filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateTimeRange)?.value?.end,
        };
        this._locale = locale;
    }
    async generateReport() {
        const data = await (0, a_1.getAccessAndSensorLogs)({ organizationId: this._organizationId, withSnapshots: true, startUtc: this._options.startUtc, endUtc: this._options.endUtc });
        const locale = this._locale;
        const reportLogsData = [
            [
                { text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.DETAILS", locale }), style: "tableHeader" },
                { text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.SNAPSHOT", locale }), style: "tableHeader" },
            ],
        ];
        for (const rawLog of data.accessAndStatusSensorLogs) {
            let pdfRow = [];
            let logDetails = [];
            if (rawLog.type === 1) {
                const accessLog = rawLog.log;
                const user = data.userCaptions.find((uc) => uc.id === accessLog.o);
                if (user) {
                    const captions = (0, report_util_1.parseCaption)(user.caption);
                    logDetails.push({
                        columns: [
                            {
                                image: captions.columns[0].image,
                                width: report_util_1.ReportConstants.widthProfilePicture * 2,
                                height: report_util_1.ReportConstants.heightProfilePicture * 2,
                                style: "center",
                                margin: [20, 20, 20, 5],
                            },
                            {
                                text: captions.columns[1].text,
                                fontSize: 14,
                                color: "black",
                                margin: [40, 40, 0, 20],
                            },
                        ],
                    });
                }
                else {
                    logDetails.push({
                        columns: [
                            {
                                image: report_util_1.ReportConstants.emptyAvatar,
                                width: report_util_1.ReportConstants.widthProfilePicture * 2,
                                height: report_util_1.ReportConstants.heightProfilePicture * 2,
                                style: "center",
                                margin: [20, 20, 20, 5],
                            },
                            {
                                text: i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale }),
                                fontSize: 14,
                                color: "black",
                                margin: [40, 40, 0, 20],
                            },
                        ],
                    });
                }
                const m = (0, moment_1.default)(accessLog.u).locale(locale);
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.date", locale }) + ": " + m.format("L") + " " + m.format("HH:mm:ss"),
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.CABIN", locale }) + ": " + data.accessControlPoints.find((acp) => acp.id === accessLog.a).name,
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                let cellForStatus = this.createCellForAccessLogStatus(accessLog, this._locale);
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.DESCRIPTION", locale }) + ": " + (cellForStatus.log ?? cellForStatus.description) + "\n",
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.RESULT", locale }) + ": " + cellForStatus.statusInfo + "\n",
                            fontSize: 14,
                            color: cellForStatus.statusColor,
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
            }
            else {
                const sensorLog = rawLog.log;
                logDetails.push({
                    columns: [
                        {
                            image: report_util_1.ReportConstants.emptyAvatar,
                            width: report_util_1.ReportConstants.widthProfilePicture * 2,
                            height: report_util_1.ReportConstants.heightProfilePicture * 2,
                            style: "center",
                            margin: [20, 20, 20, 5],
                        },
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale }),
                            fontSize: 14,
                            color: "black",
                            margin: [40, 40, 0, 20],
                        },
                    ],
                });
                const m = (0, moment_1.default)(sensorLog.u).locale(locale);
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.date", locale }) + ": " + m.format("L") + " " + m.format("HH:mm:ss"),
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                const sensorLogAccessPointName = data.accessControlPoints.find((acp) => acp.id === sensorLog.a).name;
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.CABIN", locale }) + ": " + sensorLogAccessPointName,
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.DESCRIPTION", locale }) +
                                ": " +
                                i18n_1.default.__({ phrase: "CUSTOM-REPORT.SENSOR_UNKNOWN_ENTRANCE", locale: this._locale }, {
                                    an: sensorLogAccessPointName,
                                }),
                            fontSize: 14,
                            color: "black",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
                logDetails.push({
                    text: [
                        {
                            text: i18n_1.default.__({ phrase: "CUSTOM-REPORT.RESULT", locale }) +
                                ": " +
                                i18n_1.default.__({
                                    phrase: sensorLog.r === app_enums_1.enums.StatusSensorReason.UnknownReasonOpenAlthoughRelayIsMapped
                                        ? "CUSTOM-REPORT.OPENED_WITH_KEY"
                                        : report_util_1.ReportConstants.StatusSensorReasons.find((ssr) => ssr.type === sensorLog.r)?.description,
                                    locale: this._locale,
                                }),
                            fontSize: 14,
                            color: "#62A420",
                        },
                    ],
                    margin: [20, 3, 20, 0],
                });
            }
            pdfRow.push(logDetails);
            pdfRow.push({
                image: rawLog.snapshots.length
                    ? "data:image/jpeg;base64," + (await (0, sharp_1.default)(Buffer.from(rawLog.snapshots[0]), { failOnError: false }).toFormat("jpeg").toBuffer()).toString("base64")
                    : NOT_FOUND_IMAGE,
                width: 320,
                height: 180,
                style: "center",
            });
            reportLogsData.push(pdfRow);
        }
        this._dd = {
            footer: ((currentPage, pageCount) => {
                return {
                    columns: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale }) + ": " + (0, moment_1.default)().locale(locale).format("DD/MM/YYYY HH:mm") + " ",
                            alignment: "left",
                            fontSize: "10",
                            margin: [40, 0],
                        },
                        { text: currentPage.toString() + " / " + pageCount, alignment: "right", fontSize: "10", margin: [40, 0] },
                    ],
                };
            }),
            pageOrientation: "landscape",
            content: [
                data.organizationInfo.n,
                { text: i18n_1.default.__({ phrase: "PDF-REPORT.access_logs", locale }) + "\n", style: "header" },
                {
                    text: [
                        { text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_DATE", locale }) + " : ", style: "filterBold" },
                        {
                            text: (0, moment_1.default)(this._options.startUtc).locale(locale).format("DD/MM/YYYY HH:mm") + " - " + (0, moment_1.default)(this._options.endUtc).locale(locale).format("DD/MM/YYYY HH:mm"),
                            style: "filter",
                        },
                    ],
                },
                {
                    style: "tableStyle",
                    layout: {
                        fillColor: function (rowIndex, node, columnIndex) {
                            return rowIndex % 2 === 0 ? report_util_1.ReportConstants.evenRowColor : null;
                        },
                    },
                    table: {
                        headerRows: 1,
                        dontBreakRows: true,
                        keepWithHeaderRows: 1,
                        widths: ["auto", "auto"],
                        body: reportLogsData,
                    },
                },
            ],
            styles: {
                header: {
                    fontSize: 24,
                    bold: true,
                    margin: [0, 0, 0, 10],
                    alignment: "center",
                },
                tableStyle: {
                    margin: [0, 5, 0, 15],
                },
                tableHeader: {
                    bold: true,
                    fontSize: 18,
                    color: "black",
                },
                status: {
                    margin: [0, 15],
                    fontSize: 10,
                    color: "gray",
                },
                date: {
                    fontSize: 11,
                    color: "black",
                    alignment: "right",
                },
                organizationName: {
                    fontSize: 11,
                    bold: true,
                    alignment: "left",
                },
                filter: {
                    fontSize: 14,
                    color: "black",
                    alignment: "left",
                },
                filterBold: {
                    fontSize: 14,
                    color: "black",
                    bold: true,
                    alignment: "left",
                },
            },
            defaultStyle: {},
        };
        const res = await this.createPdf();
        return {
            filePath: res,
            format: app_enums_1.enums.ReportFormat.Pdf,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            notificationType: undefined,
            sendFileInAttachments: false,
            customReportName: i18n_1.default.__({ phrase: "CUSTOM-REPORT.NAME", locale }),
            reportContainsEmptyData: false,
            preparedReportId: this._fileId,
        };
    }
    createPdf() {
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(this._dd);
        return new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(this._filePath, result);
                app_logs_1.logger.debug("Report generated:" + this._filePath);
                resolve(this._filePath);
            });
        });
    }
    createCellForAccessLogStatus(record, locale) {
        let log;
        if (record.v) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_visitor", locale });
        }
        else if (record.ir) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_remote_access", locale });
        }
        else if (record.di) {
            log = i18n_1.default.__({ phrase: "PDF-REPORT.log_exit_button", locale });
        }
        let translationKey = (0, report_util_1.getDescriptionAccessLogItem)(record.r, record.d);
        let isSuccess = record.r === app_enums_1.enums.AccessLogReason.Success || (record.di && record.r === app_enums_1.enums.AccessLogReason.Manual);
        let credentialDescription = "";
        if (record.cx && record.cx.length > 0) {
            let visibleCredentialData = [];
            record.cx.forEach((credential) => {
                visibleCredentialData.push((0, report_util_1.getCredentialDataVisibility)(credential.t));
            });
            record.cx.forEach((credential, index) => {
                let credentialType = report_util_1.ReportConstants.CredentialTypes.find((item) => item.type === credential.t);
                credentialDescription += i18n_1.default.__({ phrase: "PDF-REPORT." + credentialType.description, locale }) + " ";
                if (credential.d && visibleCredentialData[index]) {
                    credentialDescription += "(" + credential.d + ")";
                }
            });
        }
        return {
            statusColor: isSuccess ? "#62A420" : "#E62700",
            description: record != null && record.an != null
                ? i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }, {
                    an: record.an.split("/")[0],
                })
                : i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }),
            log: log,
            statusInfo: (0, report_util_1.getAccessLogStatusInfo)(record.d, this._locale, isSuccess),
            credentialDescription: credentialDescription,
        };
    }
}
exports.Ziraat = Ziraat;
async function generateReport(request, locale) {
    let report = new Ziraat(request, locale);
    return await report.generateReport();
}
exports.generateReport = generateReport;
