"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TemsanPPermissionPersonnelCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const controller_report_1 = require("../../../api/pacs/v2/controllers/controller.report");
const sharp = require("sharp");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
            top: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.RightThick = {
            right: {
                style: "thick",
            },
        };
        Styling.AllThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thick",
            },
            bottom: {
                style: "thick",
            },
            top: {
                style: "thick",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    let report = new TemsanPPermissionPersonnelCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class TemsanPPermissionPersonnelCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._request = request;
        this._filter = request.filter;
        this._options = {
            date: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value).startOf("day"),
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
        this._ppermissionTypeGroups = [
            {
                groupName: "Görevli",
                ppermissionTypeIds: [
                    "7e0d63d9-9f31-4712-a743-9f85f049bbe3",
                    "224719c5-56e7-45db-ad0e-ab0f50e59368",
                    "0d25d0ea-0dae-41c6-ad08-2cdf514bfec8",
                    "fd8eae61-2af8-4f19-a0c9-863fd7037727",
                    "40756baa-08d0-4ecb-b90d-72d726e3ddc5",
                    "9db1f8eb-01ba-4826-8031-b64b7ffedd55",
                    "cab17205-22a9-4086-bb55-dd1da73dad48",
                    "2556c176-eb2b-4e9b-ac0c-4acd75db07d2",
                    "e1e72aaf-84da-463c-a487-a3cbd3861aa1",
                    "08c3129d-89bb-413f-bf96-6f3316cd4155",
                    "decddd64-ba28-4c1b-9392-703768fb3a28",
                    "37c6aa45-b975-4e89-b4fc-30167439a2e2",
                    "b85bcfe2-5a22-4ffa-b340-73379f33dfa9",
                    "f3705e57-27ff-4133-ac54-81fa9c7e3611",
                    "4c557400-de97-428b-9240-0de4cd6e9175",
                    "48821cf3-dd27-4382-821b-c030d6f7af81",
                    "0ec39d9e-bbbb-414a-9d23-feb97c38a8be",
                    "584408da-c207-4535-815d-51ae286cb339",
                    "ab05ab70-78fc-44e5-a83b-8a38e06f244c",
                    "9e5814bb-0109-4c8e-85ec-091a6a4bd2fc",
                    "00ad81d1-2263-46b4-af82-9047f4db17bd",
                    "43eb1f1e-b268-4b98-98e6-ae849e4151ff",
                    "b309b9b0-4883-471b-be5d-36a480db45e0",
                    "50412a28-62de-4387-9ff1-084100ebfbf3",
                    "2749316f-123a-4287-831b-50eee1fb03a8",
                    "74fa9a1e-e6e9-485b-8e0d-7c7f3058ebfd",
                    "c9619f0d-25fa-4e1e-a911-f75e333ce394",
                    "491d39c2-2f89-44fc-98b0-661fca5f0cef",
                    "0b69cf8c-1e6f-4faf-8304-53f68d67e292",
                    "c484d552-c18d-4e2f-839c-605cf6b78e4a",
                    "0078445a-e45a-4aff-8854-b46346465faa",
                    "5db2a163-6b95-4e92-ba17-268eb4fb164d",
                    "7374222e-eb6c-410b-ad89-6ec7cca20869",
                    "12a345cc-42f2-4bbd-a9c6-a1b13e072a63",
                    "5de8f823-a4b4-4f6a-89ea-cc1cddc92ebc",
                    "ad2f5df7-378a-4ca3-987e-781af3e07aca",
                    "95b4dc3c-7c3b-4b6f-8590-2f9794376280",
                    "24ca030a-215d-4cb5-a466-291f075e282a",
                    "67f3e828-df85-456a-bfa6-c36dad7d6c75",
                    "2d3f642e-e692-4c52-8a5d-2e35d7e99c8f",
                    "489313ac-b516-4110-b056-836698f21a95",
                    "d18da6ef-d293-4ca2-bf16-73fd4fb14ea7",
                    "efb3eca3-ae62-45c6-9242-fdc818040f6f",
                    "8cf26d07-dae0-4b9f-bbba-59bfd2ad6c78",
                    "f7890a18-4cdf-46eb-b8f4-1aeb8035f3b4",
                    "4310ee06-4f81-4817-ad51-c30dfd62599f",
                    "296ee4f7-1fc4-4098-ba5b-9c9038090ab5",
                    "00fd8417-ad92-4213-b35e-d6ca32657d3b",
                    "7d9f1344-79a6-44cd-980c-805ce4311930",
                    "532f87c7-f2e1-458d-b64a-1411288abebf",
                    "2ab7fcea-8561-4dbf-8f65-e708076a8a0d",
                    "76ba8f29-95b1-4aaa-984d-922f1bd71753",
                    "ea3f7dec-1271-486a-963a-2fb70d4d2c84",
                    "347a39c4-fa62-42a9-9e43-3e227d61c590",
                    "f8539eb3-41e6-4586-a3fd-f971579a5b5d",
                    "25a31641-28e1-413c-9e97-4af8f2955965",
                    "e24edc7a-9dfc-4b1c-b795-c4c7f2fc6c79",
                    "c4063a52-322e-4c46-935a-97450ea3b775",
                    "02fc4542-5848-4bdb-8d23-86964d93db2e",
                    "c30dcf9a-e4ab-4db0-a196-d2cedd6342eb",
                    "190b6c29-5c02-47eb-a7a8-041bccd5eb08",
                    "7e015bc1-665a-40ac-baa7-96b12df28af2",
                    "89c1e685-0b73-4643-a65d-abdca03341fa",
                    "9c232b78-43ba-41b9-9b82-cd63b777051e",
                    "182d858d-9425-477a-929a-d1191870ad1a",
                    "c821a27f-65a3-4bc3-8696-a4d01f133ef7",
                    "f5439a03-3e80-45ab-b2df-48bb318b85ac",
                    "374cc686-100b-49bd-a752-ec0baabba970",
                    "2051612f-5527-4989-bc7a-a8febce9a3cc",
                    "f993fedc-e43d-4ad6-aa66-b20ed8506d5d",
                    "f51935bd-3f71-46ea-8754-0fc1924d9c03",
                    "13caa4c1-6b8d-4275-a1ae-d7494627632e",
                    "a954f072-823a-438e-ae52-990c2a6c6ef1",
                    "df39aab4-4fc0-45b3-bd32-f8a6c3570529",
                    "a722d902-4e48-41ba-a9e3-3a5abd8baba3",
                    "9292afda-ae06-4f52-8a2c-a93c084fd1ba",
                    "feac8f02-909b-4c37-beae-c47abf9e26d7",
                    "864258f6-f180-4fc7-95d1-7b4800729bf4",
                    "4c88df92-6f98-4060-8974-fad22e9eaec3",
                    "60f77047-3f69-42a8-b1c8-563491bbcbc3",
                    "07e9498a-8970-4ea1-9acf-340031cf7ec7",
                    "1c94926d-3f83-4831-8e29-6022261b023a",
                    "e81739b0-fde0-4481-acb2-1441ad704d0e",
                    "31ac401e-0c52-409f-ac99-444fc7afa69b",
                    "08679fb3-4c3e-4f93-beff-c52516243f42",
                    "568127bf-5d9c-43b8-8a3d-2c89bfe3c78e",
                    "8a0493fe-d2c8-4200-bf82-b498cf2892b5",
                    "11785e48-d413-4490-8c89-aecaea02cc84",
                    "5d907762-6df4-4308-888c-5733798b1854",
                    "7b2033fc-2bdf-41ba-ab7c-b20c118460fe",
                    "3f44cf8f-004b-498d-8be6-54b8139ac58d",
                    "cf19fc30-e59d-4fdd-947d-aab2843f81f8",
                    "2d290a29-36f4-45e6-976f-307b5019e2b3",
                    "2a2f9069-4a32-44e5-a48a-1be04f7d0cf7",
                    "5accc8a8-dd0b-492a-92c6-51f32e6543b8",
                    "9939b6c9-a2ec-4c88-ae53-a21644815a8e",
                    "29952066-5776-4bb6-a36d-6e26a7081450",
                    "6a7977e2-36f5-4682-bf32-1c2b1969064d",
                    "0c190ea4-5e2f-43b6-a0e0-6b1eeb58bef9",
                    "9e995c4a-81e4-4ec5-b462-ea4d46b45283",
                    "ff28f52c-864a-4eae-b283-b2f203c51a52",
                    "3735db37-6fd3-4793-b5ed-bb1e78f15365",
                    "f4ff782f-d49d-4de0-b03f-375416f777af",
                    "f558c1f0-d251-4de5-83ac-96d65552c408",
                    "6b1eebc2-e94c-4f41-8fdf-be0751214f4b",
                    "10edecbc-003e-48eb-a339-c6cbe9ab67d9",
                    "14ac6f5b-0e38-43b8-a8fb-d7d96d6e9bea",
                    "e7f18470-a12a-4d20-856b-5ff5048747c3",
                    "4b2e58b6-d3d3-4c61-bffb-92ab51932af8",
                    "21a5f606-d2be-4910-b170-952dd52a5aa5",
                    "2c529fbc-482b-423b-b3f0-070ff78aa4de",
                    "b6ae4946-18ef-4a81-b0b7-d2ebd4111e75",
                    "e8abcc74-4fb1-444c-815c-0f3a2a7c3909",
                    "1f1be254-6313-4788-a8e8-65e63d92f10d",
                    "54dd0ee2-0518-4816-a075-d5fe244ca986",
                    "1c6b983b-6b0a-4561-8468-7096ee47f439",
                    "387e155a-9a2e-4288-adbf-4262336764fa",
                    "8e7d0c42-2cb8-4cd8-b324-695699d63957",
                    "4ae5c1f1-2352-4831-9d9d-e1b80011f717",
                    "10395e9c-e024-4491-b8a8-400efd614d6a",
                    "784f7e67-b566-4601-8162-dba4698368f6",
                    "91bf0cba-2c7d-4871-8a46-9a1e9704b7e7",
                    "91fe404b-36ac-4216-8a38-36d6574fbf3c",
                    "2fef92f4-f55f-4bd2-bb79-e8b74c74ed1c",
                    "7e7f7254-97ba-43d3-ac37-5ef210ecaf3b",
                    "efff1922-994f-4159-a9fb-5194c255317f",
                    "abe90ca6-5fc4-4805-bc59-7efe073c326a",
                    "2765908e-96c9-47b9-9b67-41794bd8ec61",
                    "09531c63-8039-4214-a284-cbc29ebdf662",
                    "de11adc3-70ac-4500-87a8-12b4bde463e8",
                    "28eea9ac-f048-49c4-836f-dd4e5e4ee5f6",
                    "e49ef077-fea2-4c94-b846-f32a2c4d1416",
                    "f85d3134-f28e-49b1-9ce0-37abae4c4cc5",
                    "43a86b23-93e4-4aef-9020-78da8326c42e",
                    "9ec7aa4e-0b48-4c08-be2d-95852ef1642b",
                    "14703fb8-1f7c-4678-9477-752afd91ee89",
                    "1d08112f-acb4-45ef-b35c-b7cf1979ee42",
                    "83ad8152-18be-4eb8-9728-b9bb2409e6ac",
                    "abf0943e-ed71-4e0d-a389-71443663c6a9",
                    "bc8de929-f022-448a-8295-1e3fae9c3aa7",
                    "efebc42a-ffdd-4ea9-86c6-7d1bfb44f239",
                    "1ce274e8-0a19-4a68-a9b7-64275a347651",
                    "53220dcc-564c-45d7-929c-cf63c40686e2",
                    "fa1ed754-928d-406c-b512-baa5e4a43c74",
                    "1a0d50b9-f3a1-49de-be21-810daf50cc80",
                    "a57cc59c-a3ba-4ca7-98b9-312c1abb1a05",
                    "4954e5e3-2a0d-4e83-9d5f-1a82b4065687",
                    "bbad3b5e-d3e5-49c2-aa39-cfb639c2792e",
                    "e38a619b-d2c8-4dfd-88ee-4a1e8d88faef",
                    "cdd5ab5b-dd09-4b52-b95d-3d9cb10c026e",
                    "1fc94fa9-078a-48f6-90f1-daf5b3e4abdc",
                    "deff04ed-9350-47e0-80a9-2d31bfbfae79",
                    "66324808-fa44-4edd-9e1f-37d35be57fa3",
                    "4c794351-9820-4eda-b73a-0ebe12274086",
                    "19169bdf-5ecc-49ba-82ec-365da3f1a954",
                    "054ef113-59c2-4b14-a5ba-e4bac1118a9a",
                    "94cb63b0-76f6-4152-ada6-1c2f060275ab",
                    "49346700-7082-4377-aef4-c3a42f72d6ba",
                    "73ee298d-2568-4d66-8985-8caf0922c2b3",
                    "d82a4b02-4213-40ff-aefb-9b99d8bf311f",
                    "2dd242c6-53f7-4d94-8b17-77df423067a9",
                    "750f8e7b-ad04-48b5-af51-aa1d4e062bf3",
                    "b5aa3cf0-3cfa-40b7-9e17-dc23ac9c053d",
                    "01a99920-14e6-404c-8e0f-65a50afc88db",
                    "7b14700a-66c6-481a-bb28-3355d65b2d34",
                    "02e4bc50-3068-4907-8663-dd078ccc9517",
                    "26fe856b-d610-4e16-96da-d4b8820ea031",
                ],
            },
            {
                groupName: "Hastanede",
                ppermissionTypeIds: ["90e72b0d-c6ad-4f89-8e19-e14d25e81e6d"],
            },
            {
                groupName: "Mazaret İzinli",
                ppermissionTypeIds: [
                    "aff7fbac-18bb-4baa-8ffe-3de027fce639",
                    "aac57c1f-6f4a-4591-baff-292fb6804012",
                    "fb045e21-ea2e-4a4b-ba1e-337b9aab76a5",
                    "2a1a2b06-ca6f-411b-9047-c59557bdea75",
                    "c02cbe47-ad6f-40ef-97e1-456dc5f045cf",
                    "ae262845-093b-4894-a010-6970b89f2229",
                    "2b107311-9bbc-4f7c-b990-d7100bbabd9b",
                    "6fde0e96-150c-4b83-a731-75110cf9cfaa",
                    "a6800149-615e-44e2-bab5-15ef6b628a7c",
                    "72d6b877-8c60-4a12-8bc5-1685b4ff5cbc",
                    "f85d1e35-2937-4fcb-bb40-50a24fdf9a64",
                    "6c3639cc-0316-410f-9836-706d22751c9b",
                    "872eebe2-cf51-4ad7-b82c-89390ce5270e",
                    "8b517be8-d546-4d55-9cc6-20a65c39f780",
                    "56de10ee-7b75-4047-b235-c48736d0e2e8",
                    "8fa57e39-5a5b-4b73-88a6-ccbffc8c9abf",
                    "05c1c0a2-c87b-4839-a613-da3695ce9171",
                    "0347ec0d-be95-48d1-ae10-8e1b3213c689",
                    "5ec0cd76-fd30-49c1-8f02-c9c277d8d672",
                    "1776b27c-3a02-4366-817a-d860bc4944dc",
                    "0f48dc87-d04d-4b16-81cc-38611a6a5b3e",
                ],
            },
            {
                groupName: "Raporlu",
                ppermissionTypeIds: ["0cda96b8-94ea-48f4-aded-06f57562e3c3"],
            },
            {
                groupName: "Yıllık İzinli",
                ppermissionTypeIds: ["af339686-c4d9-4590-9053-51d6317450bb"],
            },
            {
                groupName: "Ücretsiz İzinli",
                ppermissionTypeIds: ["49f1f979-c6ae-4a8f-8fcd-3f559e7835fa"],
            },
        ];
        this._userGroupIdsForPermission = {
            "GENEL MÜDÜRLÜK": [
                { scope: false, id: "7f2d581f-7676-4105-8f50-7ccdcdaec358" },
                { scope: true, id: "99a28327-64ed-4c6c-bc71-94f8ab453c03" },
            ],
            ANKARA: [
                { scope: false, id: "057d6b0b-7cc6-48a5-8ecc-1f5d0d7386aa" },
                { scope: true, id: "b7288e19-36fc-4e1d-aca4-a08a5742f339" },
            ],
            DİYARBAKIR: [
                { scope: false, id: "0eb161b8-63af-475c-8ced-8cab2bdbbc9b" },
                { scope: true, id: "728d8f6b-63d7-421c-b053-a368dc747672" },
            ],
        };
        this._userGroupIds = [
            { name: "GMKD", id: "7f2d581f-7676-4105-8f50-7ccdcdaec358" },
            { name: "GMKI", id: "99a28327-64ed-4c6c-bc71-94f8ab453c03" },
            { name: "ANKKD", id: "057d6b0b-7cc6-48a5-8ecc-1f5d0d7386aa" },
            { name: "ANKKI", id: "b7288e19-36fc-4e1d-aca4-a08a5742f339" },
            { name: "DYBKD", id: "0eb161b8-63af-475c-8ced-8cab2bdbbc9b" },
            { name: "DYBKI", id: "728d8f6b-63d7-421c-b053-a368dc747672" },
            { name: "ERKEK", id: "f7571c5f-f4fb-464a-8e32-2648ff376106" },
            { name: "KADIN", id: "a8e8083a-298e-455d-ab08-fc2ae3a66d75" },
        ];
    }
    async generateReport() {
        app_logs_1.logger.info("Generating custom report");
        if ((0, moment_1.default)(this._options.date).isSame((0, moment_1.default)(), "day")) {
            await (0, controller_report_1.refreshUsersSummaryForToday)({
                requesterUserId: this._options.requesterUserId,
                userGroupIds: this._userGroupIds.filter((ug) => ug.name !== "ERKEK" && ug.name !== "KADIN").map((i) => i.id),
                organizationId: this._options.organizationId,
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
        }
        this._usersWithUserGroups = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsForControlPanel(this._request.organizationId);
        this._data = await dal_manager_1.dbManager.accessPacs2.workSummaryOfFilteredEmployeesAtDate(this._options.requesterUserId, this._options.organizationId, {
            date: this._options.date.toDate(),
            paginationRequest: { take: 500, skip: 0 },
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "DESC",
            userFilter: {
                userGroupIds: this._userGroupIds.filter((ug) => ug.name !== "ERKEK" && ug.name !== "KADIN").map((i) => i.id),
            },
        });
        this._filePath = await this.generate();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "TEMSAN GÜNLÜK PERSONEL YOKLAMASI",
        };
    }
    async generate() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        const logo = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoFile(this._request.organizationId);
        if (logo) {
            const temsanLogoBase64 = "data:image/jpeg;base64," + (await sharp(Buffer.from(logo), { failOnError: false }).toFormat("png").toBuffer()).toString("base64");
            const imageId1 = this._wb.addImage({
                base64: temsanLogoBase64,
                extension: "png",
            });
            this._wsGeneral.addImage(imageId1, {
                tl: { col: 0, row: 1 },
                ext: { width: 240, height: 50 },
            });
        }
        this.addFilterToSheet(this._wsGeneral, 6, 1);
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 13;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                " TEMSAN GÜNLÜK PERSONEL YOKLAMASI " +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                this._options.date.clone().locale(this._locale).format("LL") +
                +"\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initGeneralSheet(this._wsGeneral, 10, 1);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Requester user not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        this.addRowsTotalPersonnel();
        this._generalRowIndex += 3;
        this.addRowsAbsentPersonnel();
        this.listPPermissions();
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.info("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "TEMSAN", border: Constants.Styling.AllThinButRightThick });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "GENEL MÜDÜRLÜK", border: Constants.Styling.AllThinButRightThick });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex, startColIndex + 1);
        startColIndex += 2;
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "İŞLETME MÜDÜRLÜKLERİ", border: Constants.Styling.AllThinButRightThick });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex, startColIndex + 4);
        startColIndex += 5;
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "TOPLAM", border: Constants.Styling.AllThinButRightThick });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex, startColIndex + 1);
        startColIndex = 1;
        this.addInitCell({ ws: wsRaw, row: ++startRowIndex, col: startColIndex++, value: "", border: Constants.Styling.AllThinButRightThick });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "", border: Constants.Styling.AllThinButRightThick });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex, startColIndex + 1);
        startColIndex += 2;
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "ANKARA", border: Constants.Styling.AllThin });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex, startColIndex + 1);
        startColIndex += 2;
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "DİYARBAKIR", border: Constants.Styling.AllThinButRightThick });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex, startColIndex + 1);
        startColIndex += 2;
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "", border: Constants.Styling.AllThinButRightThick });
        wsRaw.mergeCells(startRowIndex, startColIndex - 1, startRowIndex++, startColIndex + 1);
        startColIndex = 1;
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Cinsiyetlerine Göre Personel Sayısı", border: Constants.Styling.AllThinButRightThick });
        columns.push({ width: 40 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nDışı", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nİçi", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Toplam", border: Constants.Styling.AllThinButRightThick, backgroundColorCode: "C5C5C5" });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nDışı", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nİçi", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Toplam", border: Constants.Styling.AllThin, backgroundColorCode: "C5C5C5" });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nDışı", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nİçi", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Toplam", border: Constants.Styling.AllThinButRightThick, backgroundColorCode: "C5C5C5" });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nDışı", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Kapsam\r\nİçi", border: Constants.Styling.AllThin });
        columns.push({ width: 20 });
        this.addInitCell({ ws: wsRaw, row: startRowIndex, col: startColIndex++, value: "Toplam", border: Constants.Styling.AllThinButRightThick, backgroundColorCode: "C5C5C5" });
        columns.push({ width: 20 });
        wsRaw.columns = columns;
        wsRaw.views = [{ activeCell: "A1", showGridLines: false }];
    }
    listPPermissions() {
        let colIndex = 1;
        this._generalRowIndex++;
        const directorates = [...Object.keys(this._userGroupIdsForPermission)];
        directorates.forEach((elem) => {
            this.setDataCell({
                ws: this._wsGeneral,
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: elem,
                border: Constants.Styling.AllThinButRightThick,
                backgroundColorCode: "A5A5A5",
                isBold: true,
                inMiddle: true,
            });
            this._wsGeneral.mergeCells(this._generalRowIndex, colIndex - 1, this._generalRowIndex, colIndex + 1);
            colIndex += 2;
        });
        colIndex = 1;
        this._generalRowIndex += 1;
        let maxRowHeight = 0;
        for (let i = 0; i < this._ppermissionTypeGroups.length; i++) {
            colIndex = 1;
            const ppermissionGroup = this._ppermissionTypeGroups[i];
            this._generalRowIndex += maxRowHeight;
            maxRowHeight = 2;
            directorates.forEach((elem) => {
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex,
                    value: ppermissionGroup.groupName,
                    ws: this._wsGeneral,
                    backgroundColorCode: "C5C5C5",
                    border: Constants.Styling.AllThin,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex + 1,
                    value: "Açıklama",
                    ws: this._wsGeneral,
                    backgroundColorCode: "C5C5C5",
                    border: Constants.Styling.AllThin,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex + 2,
                    value: "Dönüş Tarihi",
                    ws: this._wsGeneral,
                    backgroundColorCode: "C5C5C5",
                    border: Constants.Styling.AllThinButRightThick,
                });
                const groupIds = [];
                const userIdsOfUserGroup = [];
                this._userGroupIdsForPermission[elem].forEach((item) => groupIds.push(item.id));
                this._usersWithUserGroups.filter((elem) => groupIds.includes(elem.userGroupId)).forEach((elem) => userIdsOfUserGroup.push(...elem.userIds));
                const usersData = this._data.items.filter((d) => userIdsOfUserGroup.includes(d.userId));
                const allAbsentPersonnelData = usersData.filter((ud) => ud.usedPermissions.length > 0);
                const userList = allAbsentPersonnelData.filter((pd) => pd.usedPermissions?.find((p) => ppermissionGroup.ppermissionTypeIds.includes(p.typeId)));
                if (!userList || userList.length === 0) {
                    this.setDataCell({
                        rowIndex: this._generalRowIndex + 1,
                        colIndex: colIndex,
                        value: "Yok",
                        ws: this._wsGeneral,
                    });
                    this.setDataCell({
                        rowIndex: this._generalRowIndex + 1,
                        colIndex: colIndex + 2,
                        value: "",
                        ws: this._wsGeneral,
                        border: Constants.Styling.RightThick,
                    });
                }
                for (let i = 0; i < userList.length; i++) {
                    const userData = userList[i];
                    this.setDataCell({
                        rowIndex: this._generalRowIndex + 1 + i,
                        colIndex: colIndex,
                        value: userData.userCaption[0].text[0],
                        ws: this._wsGeneral,
                        border: Constants.Styling.AllThin,
                    });
                    this.setDataCell({
                        rowIndex: this._generalRowIndex + 1 + i,
                        colIndex: colIndex + 1,
                        value: userData.usedPermissions.find((p) => ppermissionGroup.ppermissionTypeIds.includes(p.typeId)).name,
                        ws: this._wsGeneral,
                        border: Constants.Styling.AllThin,
                    });
                    this.setDataCell({
                        rowIndex: this._generalRowIndex + 1 + i,
                        colIndex: colIndex + 2,
                        value: userData.usedPermissions.find((p) => ppermissionGroup.ppermissionTypeIds.includes(p.typeId)).range.end.format("DD.MM.YYYY"),
                        ws: this._wsGeneral,
                        border: Constants.Styling.AllThinButRightThick,
                    });
                }
                maxRowHeight = userList.length >= maxRowHeight ? userList.length + 1 : maxRowHeight;
                colIndex += 3;
            });
            colIndex = 3;
            for (let i = 1; i <= maxRowHeight; i++) {
                this.setDataCell({
                    rowIndex: this._generalRowIndex + i,
                    colIndex: colIndex,
                    value: this._wsGeneral.getCell(this._generalRowIndex + i, colIndex).value,
                    ws: this._wsGeneral,
                    border: Constants.Styling.RightThick,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex + i,
                    colIndex: colIndex + 3,
                    value: this._wsGeneral.getCell(this._generalRowIndex + i, colIndex + 3).value,
                    ws: this._wsGeneral,
                    border: Constants.Styling.RightThick,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex + i,
                    colIndex: colIndex + 6,
                    value: this._wsGeneral.getCell(this._generalRowIndex + i, colIndex + 6).value,
                    ws: this._wsGeneral,
                    border: Constants.Styling.RightThick,
                });
            }
        }
        this._generalRowIndex += maxRowHeight;
        colIndex = 1;
        let maxAbsentHeight = 0;
        directorates.forEach((elem) => {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex,
                value: "Gelmeyen",
                ws: this._wsGeneral,
                backgroundColorCode: "C5C5C5",
                border: Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex + 1,
                value: "Açıklama",
                ws: this._wsGeneral,
                backgroundColorCode: "C5C5C5",
                border: Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex + 2,
                value: "Dönüş Tarihi",
                ws: this._wsGeneral,
                backgroundColorCode: "C5C5C5",
                border: Constants.Styling.AllThinButRightThick,
            });
            const groupIds = [];
            const userIdsOfUserGroup = [];
            this._userGroupIdsForPermission[elem].forEach((item) => groupIds.push(item.id));
            this._usersWithUserGroups.filter((elem) => groupIds.includes(elem.userGroupId)).forEach((elem) => userIdsOfUserGroup.push(...elem.userIds));
            const usersData = this._data.items.filter((d) => userIdsOfUserGroup.includes(d.userId));
            const allAbsentPersonnelData = usersData.filter((ud) => !ud.firstStart && !ud.normalWorkDuration && ud.expectedWorkDuration && ud.usedPermissions.length === 0);
            if (allAbsentPersonnelData.length > maxAbsentHeight) {
                maxAbsentHeight = allAbsentPersonnelData.length;
            }
            if (allAbsentPersonnelData.length === 0) {
                this.setDataCell({
                    rowIndex: this._generalRowIndex + 1,
                    colIndex: colIndex,
                    value: "Yok",
                    ws: this._wsGeneral,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex + 1,
                    colIndex: colIndex + 2,
                    value: "",
                    ws: this._wsGeneral,
                    border: Constants.Styling.RightThick,
                });
            }
            for (let i = 0; i < allAbsentPersonnelData.length; i++) {
                const userData = allAbsentPersonnelData[i];
                this.setDataCell({
                    rowIndex: this._generalRowIndex + 1 + i,
                    colIndex: colIndex,
                    value: userData.userCaption[0].text[0],
                    ws: this._wsGeneral,
                    border: Constants.Styling.AllThin,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex + 1 + i,
                    colIndex: colIndex + 1,
                    value: "",
                    ws: this._wsGeneral,
                    border: Constants.Styling.AllThin,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex + 1 + i,
                    colIndex: colIndex + 2,
                    value: "",
                    ws: this._wsGeneral,
                    border: Constants.Styling.AllThin,
                });
            }
            colIndex += 3;
        });
        colIndex = 3;
        for (let i = 1; i <= maxAbsentHeight; i++) {
            this.setDataCell({
                rowIndex: this._generalRowIndex + i,
                colIndex: colIndex,
                ws: this._wsGeneral,
                border: Constants.Styling.RightThick,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + i,
                colIndex: colIndex + 3,
                ws: this._wsGeneral,
                border: Constants.Styling.RightThick,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + i,
                colIndex: colIndex + 6,
                ws: this._wsGeneral,
                border: Constants.Styling.RightThick,
            });
        }
    }
    addRowsAbsentPersonnel() {
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex++,
            colIndex: colIndex,
            value: "İŞYERİ DIŞINDAKİ PERSONEL",
            ws: this._wsGeneral,
            backgroundColorCode: "F35A15",
        });
        const userGroupsIds = this._userGroupIds.filter((ug) => ug.name !== "ERKEK" && ug.name !== "KADIN").map((i) => i.id);
        for (let i = 0; i < this._ppermissionTypeGroups.length; i++) {
            const group = this._ppermissionTypeGroups[i];
            this.setDataCell({
                rowIndex: this._generalRowIndex + i,
                colIndex: colIndex,
                value: group.groupName,
                ws: this._wsGeneral,
                border: Constants.Styling.AllThinButRightThick,
            });
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex + this._ppermissionTypeGroups.length,
            colIndex: colIndex,
            value: "Gelmeyen",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThinButRightThick,
        });
        colIndex++;
        for (let i = 1; i <= userGroupsIds.length; i++) {
            const userGroupId = userGroupsIds[i - 1];
            const userIdsOfUserGroup = this._usersWithUserGroups.find((ug) => ug.userGroupId === userGroupId)?.userIds ?? [];
            const usersData = this._data.items.filter((d) => userIdsOfUserGroup.includes(d.userId));
            const allAbsentPersonnelData = usersData.filter((ud) => ud.usedPermissions.length > 0);
            for (let i = 0; i < this._ppermissionTypeGroups.length; i++) {
                const group = this._ppermissionTypeGroups[i];
                const userList = allAbsentPersonnelData.filter((pd) => pd.usedPermissions?.find((p) => group.ppermissionTypeIds.includes(p.typeId)));
                this.setDataCell({
                    rowIndex: this._generalRowIndex + i,
                    colIndex: colIndex,
                    value: userList.length,
                    ws: this._wsGeneral,
                    border: Constants.Styling.AllThin,
                });
            }
            const absentPersonnelData = usersData.filter((ud) => !ud.firstStart && !ud.normalWorkDuration && ud.expectedWorkDuration && ud.usedPermissions.length === 0);
            this.setDataCell({
                rowIndex: this._generalRowIndex + this._ppermissionTypeGroups.length,
                colIndex: colIndex,
                value: absentPersonnelData.length,
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
            colIndex++;
            if (i % 2 === 0) {
                for (let i = 0; i < this._ppermissionTypeGroups.length; i++) {
                    this.setDataCell({
                        rowIndex: this._generalRowIndex + i,
                        colIndex: colIndex,
                        value: {
                            formula: "SUM(" +
                                this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex - 2).address +
                                ":" +
                                this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex - 1).address +
                                ")",
                        },
                        ws: this._wsGeneral,
                        backgroundColorCode: "C5C5C5",
                        border: colIndex === 7 ? Constants.Styling.AllThin : Constants.Styling.AllThinButRightThick,
                    });
                }
                this.setDataCell({
                    rowIndex: this._generalRowIndex + this._ppermissionTypeGroups.length,
                    colIndex: colIndex,
                    value: {
                        formula: "SUM(" +
                            this._wsGeneral.getRow(this._generalRowIndex + this._ppermissionTypeGroups.length).getCell(colIndex - 2).address +
                            ":" +
                            this._wsGeneral.getRow(this._generalRowIndex + this._ppermissionTypeGroups.length).getCell(colIndex - 1).address +
                            ")",
                    },
                    ws: this._wsGeneral,
                    backgroundColorCode: "C5C5C5",
                    border: colIndex === 7 ? Constants.Styling.AllThin : Constants.Styling.AllThinButRightThick,
                });
                colIndex++;
            }
        }
        for (let i = 0; i <= this._ppermissionTypeGroups.length; i++) {
            const backgroundColorCode = "C5C5C5";
            this.setDataCell({
                rowIndex: this._generalRowIndex + i,
                colIndex: colIndex,
                value: {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex - 3).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex - 6).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex - 9).address +
                        ")",
                },
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + i,
                colIndex: colIndex + 1,
                value: {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex + 1 - 3).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex + 1 - 6).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex + 1 - 9).address +
                        ")",
                },
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + i,
                colIndex: colIndex + 2,
                value: {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex + 2 - 3).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex + 2 - 6).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + i).getCell(colIndex + 2 - 9).address +
                        ")",
                },
                ws: this._wsGeneral,
                backgroundColorCode: backgroundColorCode,
                border: Constants.Styling.AllThinButRightThick,
            });
        }
        this._generalRowIndex += this._ppermissionTypeGroups.length + 1;
        colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: "TOPLAM",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThinButRightThick,
        });
        for (let i = 1; i <= 12; i++) {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex,
                value: {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex - (this._ppermissionTypeGroups.length + 1)).getCell(colIndex).address +
                        ":" +
                        this._wsGeneral.getRow(this._generalRowIndex - 1).getCell(colIndex).address +
                        ")",
                },
                ws: this._wsGeneral,
                backgroundColorCode: i % 3 === 0 ? "C5C5C5" : undefined,
                border: i % 3 === 0 && i !== 6 ? Constants.Styling.AllThinButRightThick : Constants.Styling.AllThin,
            });
            colIndex++;
        }
        this._generalRowIndex++;
        colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: "MEVCUT PERSONEL SAYISI",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThinButRightThick,
        });
        for (let i = 1; i <= 12; i++) {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex,
                value: {
                    formula: this._wsGeneral.getRow(this._generalRowIndex - (this._ppermissionTypeGroups.length + 5)).getCell(colIndex).address +
                        "-" +
                        this._wsGeneral.getRow(this._generalRowIndex - 1).getCell(colIndex).address,
                },
                ws: this._wsGeneral,
                backgroundColorCode: "89B8EB",
                border: i % 3 === 0 && i !== 6 ? Constants.Styling.AllThinButRightThick : Constants.Styling.AllThin,
            });
            colIndex++;
        }
        this._generalRowIndex++;
    }
    addRowsTotalPersonnel() {
        let colIndex = 1;
        const userGroupsIds = this._userGroupIds.filter((ug) => ug.name !== "ERKEK" && ug.name !== "KADIN").map((i) => i.id);
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex,
            value: "Kadın",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThinButRightThick,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex + 1,
            colIndex: colIndex,
            value: "Erkek",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThinButRightThick,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex + 2,
            colIndex: colIndex++,
            value: "TOPLAM PERSONEL SAYISI",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThinButRightThick,
        });
        for (let i = 1; i <= userGroupsIds.length; i++) {
            const userGroupId = userGroupsIds[i - 1];
            const womenUserGroupId = this._userGroupIds.find((u) => u.name === "KADIN").id;
            const userIdsOfWomenUserGroup = this._usersWithUserGroups.find((ug) => ug.userGroupId === womenUserGroupId)?.userIds ?? [];
            const menUserGroupId = this._userGroupIds.find((u) => u.name === "ERKEK").id;
            const userIdsOfMenUserGroup = this._usersWithUserGroups.find((ug) => ug.userGroupId === menUserGroupId)?.userIds ?? [];
            const userIdsOfUserGroup = this._usersWithUserGroups.find((ug) => ug.userGroupId === userGroupId)?.userIds ?? [];
            const usersDataWomen = this._data.items.filter((d) => userIdsOfUserGroup.includes(d.userId) && userIdsOfWomenUserGroup.includes(d.userId));
            const usersDataMen = this._data.items.filter((d) => userIdsOfUserGroup.includes(d.userId) && userIdsOfMenUserGroup.includes(d.userId));
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex,
                value: (usersDataWomen?.length ?? 0),
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + 1,
                colIndex: colIndex,
                value: (usersDataMen?.length ?? 0),
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + 2,
                colIndex: colIndex++,
                value: ((usersDataWomen?.length ?? 0) + (usersDataMen?.length ?? 0)),
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
            });
            if (i % 2 === 0) {
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex,
                    value: {
                        formula: "SUM(" +
                            this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 2).address +
                            "+" +
                            this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 1).address +
                            ")",
                    },
                    ws: this._wsGeneral,
                    backgroundColorCode: "C5C5C5",
                    border: i === 4 ? Constants.Styling.AllThin : Constants.Styling.AllThinButRightThick,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex + 1,
                    colIndex: colIndex,
                    value: {
                        formula: "SUM(" +
                            this._wsGeneral.getRow(this._generalRowIndex + 1).getCell(colIndex - 2).address +
                            "+" +
                            this._wsGeneral.getRow(this._generalRowIndex + 1).getCell(colIndex - 1).address +
                            ")",
                    },
                    ws: this._wsGeneral,
                    backgroundColorCode: "C5C5C5",
                    border: i === 4 ? Constants.Styling.AllThin : Constants.Styling.AllThinButRightThick,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex + 2,
                    colIndex: colIndex,
                    value: {
                        formula: "SUM(" +
                            this._wsGeneral.getRow(this._generalRowIndex + 2).getCell(colIndex - 2).address +
                            "+" +
                            this._wsGeneral.getRow(this._generalRowIndex + 2).getCell(colIndex - 1).address +
                            ")",
                    },
                    ws: this._wsGeneral,
                    backgroundColorCode: "C5C5C5",
                    border: i === 4 ? Constants.Styling.AllThin : Constants.Styling.AllThinButRightThick,
                });
                colIndex++;
            }
        }
        for (let i = 0; i < 3; i++) {
            const backgroundColorCode = i === 2 ? "C5C5C5" : undefined;
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex,
                value: {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 3).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 6).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex).getCell(colIndex - 9).address +
                        ")",
                },
                ws: this._wsGeneral,
                backgroundColorCode: backgroundColorCode,
                border: i === 2 ? Constants.Styling.AllThinButRightThick : Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + 1,
                colIndex: colIndex,
                value: {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex + 1).getCell(colIndex - 3).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + 1).getCell(colIndex - 6).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + 1).getCell(colIndex - 9).address +
                        ")",
                },
                ws: this._wsGeneral,
                backgroundColorCode: backgroundColorCode,
                border: i === 2 ? Constants.Styling.AllThinButRightThick : Constants.Styling.AllThin,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex + 2,
                colIndex: colIndex,
                value: {
                    formula: "SUM(" +
                        this._wsGeneral.getRow(this._generalRowIndex + 2).getCell(colIndex - 3).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + 2).getCell(colIndex - 6).address +
                        "+" +
                        this._wsGeneral.getRow(this._generalRowIndex + 2).getCell(colIndex - 9).address +
                        ")",
                },
                ws: this._wsGeneral,
                backgroundColorCode: backgroundColorCode,
                border: i === 2 ? Constants.Styling.AllThinButRightThick : Constants.Styling.AllThin,
            });
            colIndex++;
        }
        this._generalRowIndex++;
    }
    setDataCell(params) {
        const c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value ?? "";
        c.alignment = {
            horizontal: params.inMiddle ? "center" : "left",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: params.isBold || false,
        };
        if (params.border) {
            c.border = Object.assign(c.border ?? {}, params.border);
        }
        if (params.numFmt) {
            c.numFmt = params.numFmt;
        }
        if (params.backgroundColorCode) {
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        }
        if (params.textColor) {
            c.font = {
                color: { argb: params.textColor },
            };
        }
    }
    addInitCell(params) {
        let c = params.ws.getCell(params.row, params.col);
        c.value = params.value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = params.border;
        if (params.textColor) {
            c.font = {
                color: { argb: params.textColor },
            };
        }
        if (params.backgroundColorCode) {
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        }
    }
    addFilterToSheet(ws, startRowIndex, startColIndex) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name.toUpperCase() + " GÜNLÜK PERSONEL YOKLAMASI", border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, this._options.date.clone().locale(this._locale).format("LL"), border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.TemsanPPermissionPersonnelCustomReport = TemsanPPermissionPersonnelCustomReport;
