"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TemsanMealCountCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const api_error_1 = require("../../../api/api.error");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
    Constants.ColorCodes = {
        ON_TIME: "7C9923",
        TOLERATED: "D56626",
        NOT_TOLERATED: "A11913",
        PERMISSION: "4C2AA8",
    };
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    const report = new TemsanMealCountCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class TemsanMealCountCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._requestFilter = request.filter;
        this._options = {
            startDateTime: (0, moment_1.default)(this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value)
                .startOf("day")
                .toDate(),
            endDateTime: (0, moment_1.default)(this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value)
                .endOf("day")
                .toDate(),
            organizationUnitIds: this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            filterOrganizationUnitMembersHierarchically: this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
        };
        this._userGroups = [{ id: null, name: "Diğer", count: 0 }];
        this._data = [];
    }
    async generateReport() {
        app_logs_1.logger.debug("Generating meal count custom report");
        if (this._options.organizationUnitIds?.length > 0) {
            this._organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._options.organizationUnitIds);
        }
        await this.getData();
        this._filePath = await this.generateMealCountCustomReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.PacsDailyWorkingHours,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? this._data.length === 0 : undefined,
        };
    }
    async generateMealCountCustomReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        for (const item of this._data) {
            const userGroupsOfUser = item.identity.userGroups.find((ug) => ug.name !== "ERKEK" && ug.name !== "KADIN");
            if (!userGroupsOfUser) {
                this._userGroups.find((ug) => !ug.id).count++;
            }
            else {
                if (this._userGroups.find((ug) => ug.id === userGroupsOfUser.id)) {
                    this._userGroups.find((ug) => ug.id === userGroupsOfUser.id).count++;
                }
                else {
                    this._userGroups.push({
                        count: 1,
                        id: userGroupsOfUser.id,
                        name: userGroupsOfUser.name,
                    });
                }
            }
        }
        for (const row of this._userGroups) {
            if (row.id) {
                this.addGeneralRow(row);
            }
        }
        const otherGroup = this._userGroups.find((ug) => !ug.id);
        if (otherGroup.count) {
            this.addGeneralRow(otherGroup);
        }
        this.addSumRow();
        this.addFilterToSheet(this._wsGeneral, 1, 1);
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready");
        return this._filePath;
    }
    async getData() {
        const _filter = {
            sortDateDesc: true,
            startUtc: this._options.startDateTime,
            endUtc: this._options.endDateTime,
            organizationUnitIds: this._options.organizationUnitIds,
            filterOrganizationUnitMembersHierarchically: this._options.filterOrganizationUnitMembersHierarchically,
        };
        const authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
            organizationId: this._request.organizationId,
            userId: this._request.requesterUserId,
            organizationUnitIds: this._options.organizationUnitIds,
            accessControlPointIds: null,
            userIds: null,
            userGroupIds: null,
        });
        if (!authResultForFilter.result) {
            throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
        }
        let filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: this._request.organizationId,
            userIds: _filter.userIds,
            organizationUnitIds: _filter.organizationUnitIds,
            userGroupIds: _filter.userGroupIds,
            filterOrganizationUnitMembersHierarchically: _filter.filterOrganizationUnitMembersHierarchically,
        });
        let filterUserIds = (filterUsers && filterUsers.length) > 0 ? filterUsers.map((f) => f.userId) : [];
        _filter.userIds = filterUserIds;
        const option = Object.assign(_filter, {
            take: app_config_1.appConfig.reportExportRowLimit,
            skip: null,
        });
        await dal_manager_1.dbManager.accessAccessLog.getAccessLogsReportNew(this._request.organizationId, option, async (rows) => {
            app_logs_1.logger.debug("[CustomReport]Fetch data %s rows", rows.items.length);
            for (const item of rows.items) {
                if (!this._data.find((i) => i.identity?.id === item.identity?.id) && item.identity?.id) {
                    this._data.push(item);
                }
            }
        });
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Kullanıcı Grubu", Constants.Styling.AllThin);
        columns.push({ width: 40 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Personel Sayısı", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    addGeneralRow(row) {
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.name,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.count,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
        });
        this._generalRowIndex++;
    }
    addSumRow() {
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: "TOPLAM",
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            isBold: true,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: this._userGroups.map((ug) => ug.count).reduce((accumulator, currentValue) => accumulator + currentValue),
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            isBold: true,
        });
        this._generalRowIndex += 2;
        colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex,
            value: "Mevcudu Veren İmza",
            ws: this._wsGeneral,
            isInRight: true,
        });
        this._generalRowIndex += 4;
    }
    addInitCell(ws, row, col, value, border) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    setDataCell(params) {
        const c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: params.isInRight ? "right" : "left",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: params.isBold || false,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt) {
            c.numFmt = params.numFmt;
        }
        if (params.backgroundColorCode) {
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        }
        if (params.isTextWhite) {
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
        }
    }
    addFilterToSheet(ws, startRowIndex, startColIndex) {
        const border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name + " Yemek Mevcudu", border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, "Tarih", border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, {
            formula: "DATE(" + (0, moment_1.default)(this._options.startDateTime).year() + "," + ((0, moment_1.default)(this._options.startDateTime).month() + 1) + "," + (0, moment_1.default)(this._options.startDateTime).date() + ")",
        }, border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 4);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 4);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 4);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, "Organizasyon Birimleri", border);
        const filter = this._organizationUnits
            ? this._organizationUnits.map((ou) => ou.name).join(", ") + "\n( Hiyerarşik Listeleme : " + (this._options.filterOrganizationUnitMembersHierarchically ? "Evet )" : "Hayır )")
            : "Hepsi";
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, filter, border);
        const row = ws.getRow(startRowIndex + 2);
        row.height = (this._organizationUnits?.length ?? 2) * 20;
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.TemsanMealCountCustomReport = TemsanMealCountCustomReport;
