"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TemsanDailyWorkingDetailsAtDateCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const business_pacs_report_1 = require("../../../business/pacs/business.pacs.report");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
    Constants.ColorCodes = {
        ON_TIME: "7C9923",
        TOLERATED: "D56626",
        NOT_TOLERATED: "A11913",
        PERMISSION: "4C2AA8",
    };
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    const report = new TemsanDailyWorkingDetailsAtDateCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class TemsanDailyWorkingDetailsAtDateCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._options = request.filter;
        this._requestFilter = request.filter;
        this._options = {
            date: (0, moment_1.default)(this._requestFilter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateDay).value).toDate(),
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "DESC",
            userFilter: {
                userGroupIds: [
                    "7f2d581f-7676-4105-8f50-7ccdcdaec358",
                    "99a28327-64ed-4c6c-bc71-94f8ab453c03",
                    "b7288e19-36fc-4e1d-aca4-a08a5742f339",
                    "057d6b0b-7cc6-48a5-8ecc-1f5d0d7386aa",
                    "728d8f6b-63d7-421c-b053-a368dc747672",
                    "0eb161b8-63af-475c-8ced-8cab2bdbbc9b",
                ],
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
                workStatusFilter: [dal_constants_1.DalConstants.WorkStatusFilter.LateStart, dal_constants_1.DalConstants.WorkStatusFilter.EarlyLeave],
            },
        };
    }
    async generateReport() {
        app_logs_1.logger.debug("Generating daily working details custom report");
        const option = Object.assign(this._options, {
            paginationRequest: {
                take: app_config_1.appConfig.reportExportRowLimit,
                skip: undefined,
            },
        });
        this._data = await dal_manager_1.dbManager.accessPacs2.workSummaryOfFilteredEmployeesAtDate(this._request.requesterUserId, this._request.organizationId, option);
        const userIds = this._data.items.map((i) => i.userId);
        const firstLastData = await (0, business_pacs_report_1.getDailySummaryReport)(this._request.organizationId, {
            date: new Date(this._options.date),
            userIds: userIds,
            status: this._options.userFilter.userOrganizationStatus,
            sortingMethod: this._options.sortType,
            pagination: {
                take: app_config_1.appConfig.reportExportRowLimit + 1,
                skip: 0,
            },
            hasOrganizationWide: undefined,
            permittedUnitIds: undefined,
        }, this._request.requesterUserId);
        this._items = firstLastData.items;
        this._workplans = firstLastData.workPlans;
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.PacsDailyWorkingHours,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? this._data.items.length === 0 : undefined,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_WORKING_DETAILS_REPORT", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                (0, moment_1.default)(this._options.date).format("DD/MM/YYYY") +
                "\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        for (const item of this._data.items) {
            this.addGeneralRow(item);
        }
        let filters = [];
        if (this._options.userFilter.userOrganizationStatus) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.IdentityStatusType[this._options.userFilter.userOrganizationStatus].toUpperCase(), locale: this._locale }));
        }
        if (this._options.userFilter.workStatusFilter) {
            this._options.userFilter.workStatusFilter.forEach((workStatusFilter) => {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.WORK_STATUS", locale: this._locale }) +
                    " : " +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.WorkStatusFilter[workStatusFilter].toUpperCase(), locale: this._locale }));
            });
        }
        if (this._options.userFilter.organizationUnitIds && this._options.userFilter.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._options.userFilter.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (this._options.userFilter.applyOrganizationUnitFilterHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userFilter.userGroupIds && this._options.userFilter.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userFilter.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames.map((ugn) => ugn.name).join(","));
        }
        if (this._options.userFilter.workPlanIds && this._options.userFilter.workPlanIds.length > 0) {
            let workplanNames = this._workplans.filter((wp) => this._options.userFilter.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: this._locale }) + " : " + workplanNames.join(","));
        }
        if (this._options.userFilter.userIds && this._options.userFilter.userIds.length > 0) {
            let userNames = this._items.filter((i) => this._options.userFilter.userIds.includes(i.user.id)).map((i) => i.user.fullname);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.join(","));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"), filters.length);
        this._wsGeneral.pageSetup.printArea = "A6:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready");
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.name_surname", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 35 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.UNIQUE_ID", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.FIRST_START", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.LAST_LEAVE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.STATUS", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPECTED_ENTRANCE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPECTED_EXIT", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ENTRANCE_PENALTY_TIME", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.MISSING_WORK", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.OVERTIME", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPECTED_WORK_DURATION", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.NORMAL_WORK_DURATION", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCOUNTED_DECLARED_WORK_DURATION", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCOUNTED_IGNORED_WORK_DURATION", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCOUNTED_OFFTIME_PERMISSION_DURATION", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EARLY_LEAVE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.LATE_LEAVE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EARLY_START", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.LATE_START", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.TOLERATED_EARLY_LEAVE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.TOLERATED_EARLY_START", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.TOLERATED_LATE_LEAVE", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.TOLERATED_LATE_START", locale: this._locale }), Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow(data) {
        const row = this._items.find((i) => i.user.id === data.userId);
        if (!row) {
            return;
        }
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.user.fullname.toUpperCase(),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.user.uniqueId,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.entranceDateTime ? (0, moment_1.default)(row.entranceDateTime).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.entranceDateTime ? Constants.TimeFormat : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.exitDateTime ? (0, moment_1.default)(row.exitDateTime).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.exitDateTime ? Constants.TimeFormat : null,
        });
        let statusText = row.entranceDateTime
            ? i18n_1.default.__({ phrase: "EXCEL-REPORT.ARRIVED", locale: this._locale })
            : i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_ARRIVED", locale: this._locale });
        let colorText = undefined;
        let hasPermissionNow;
        let permissionNowType;
        let hasPermissionInDay = row.ppermissions && row.ppermissions.length > 0;
        if (hasPermissionInDay) {
            row.ppermissions.forEach((permission) => {
                if ((0, moment_1.default)().isBetween(permission.startUtc, permission.endUtc)) {
                    hasPermissionNow = true;
                    permissionNowType = permission.ppermissionTypeName;
                }
            });
        }
        if (hasPermissionInDay) {
            colorText = Constants.ColorCodes.PERMISSION;
            if (hasPermissionNow) {
                statusText += "\n" + i18n_1.default.__({ phrase: "PDF-REPORT.day_off", locale: this._locale }) + "(" + permissionNowType + ")";
            }
            else if (hasPermissionInDay) {
                row.ppermissions.forEach((permission) => {
                    let interval = (0, moment_1.default)(permission.startUtc).locale(this._locale).format("DD/MM/YYYY HH:mm") + " - " + (0, moment_1.default)(permission.endUtc).locale(this._locale).format("DD/MM/YYYY HH:mm");
                    statusText += "\n" + permission.ppermissionTypeName + "\n(" + interval + ")";
                });
            }
        }
        else {
            colorText = this.findColorCodeByPerformance(data);
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: statusText,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: null,
            backgroundColorCode: colorText,
            isTextWhite: true,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.workPlan.startOfWorkUtc ? (0, moment_1.default)(row.workPlan.startOfWorkUtc).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.workPlan.startOfWorkUtc ? Constants.TimeFormat : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.workPlan.endOfWorkUtc ? (0, moment_1.default)(row.workPlan.endOfWorkUtc).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.workPlan.endOfWorkUtc ? Constants.TimeFormat : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.entrancePenaltyTimeInSeconds ? (Math.round(row.entrancePenaltyTimeInSeconds / 60) / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.entrancePenaltyTimeInSeconds
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.missingWorkInMinutes ? (row.missingWorkInMinutes / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.missingWorkInMinutes
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.extraWorkInMinutes ? (row.extraWorkInMinutes / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: row.extraWorkInMinutes
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.expectedWorkDuration ? (data.expectedWorkDuration / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.expectedWorkDuration
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.normalWorkDuration ? (data.normalWorkDuration / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.normalWorkDuration
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.accountedDeclaredWorkDuration ? (data.accountedDeclaredWorkDuration / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.accountedDeclaredWorkDuration
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.accountedIgnoredWorkDuration ? (data.accountedIgnoredWorkDuration / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.accountedIgnoredWorkDuration
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.accountedOffTimePermissionDuration ? (data.accountedOffTimePermissionDuration / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.accountedOffTimePermissionDuration
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.earlyLeave ? (data.earlyLeave / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.earlyLeave
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.lateLeave ? (data.lateLeave / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.lateLeave
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.earlyStart ? (data.earlyStart / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.earlyStart
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.lateStart ? (data.lateStart / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.lateStart
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.toleratedEarlyLeave ? (data.toleratedEarlyLeave / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.toleratedEarlyLeave
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.toleratedEarlyStart ? (data.toleratedEarlyStart / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.toleratedEarlyStart
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.toleratedLateLeave ? (data.toleratedLateLeave / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.toleratedLateLeave
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: data.toleratedLateStart ? (data.toleratedLateStart / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: data.toleratedLateStart
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this._generalRowIndex++;
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextWhite)
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
    }
    findColorCodeByPerformance(employeeData) {
        let colorCode = null;
        if (!employeeData.firstStart || employeeData.lateStart > 0) {
            colorCode = Constants.ColorCodes.NOT_TOLERATED;
        }
        else {
            if (employeeData.toleratedLateStart && employeeData.toleratedLateStart > 0) {
                colorCode = Constants.ColorCodes.TOLERATED;
            }
            else {
                colorCode = Constants.ColorCodes.ON_TIME;
            }
        }
        return colorCode;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_WORKING_DETAILS_REPORT", locale: this._locale }), border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, {
            formula: "DATE(" + (0, moment_1.default)(this._options.date).year() + "," + ((0, moment_1.default)(this._options.date).month() + 1) + "," + (0, moment_1.default)(this._options.date).date() + ")",
        }, border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, filter, border);
        if (filterlength) {
            let row = ws.getRow(startRowIndex + 2);
            row.height = filterlength * 15;
        }
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.TemsanDailyWorkingDetailsAtDateCustomReport = TemsanDailyWorkingDetailsAtDateCustomReport;
