"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const pg_1 = require("pg");
const app_enums_1 = require("../../app.enums");
const dal_constants_1 = require("../../dal/dal.constants");
const uuid_1 = __importDefault(require("uuid"));
const dal_db_armon_schema_1 = require("../../dal/db/armon/dal.db.armon.schema");
const cli_queries_1 = require("../../dal/access/psql/cli-queries");
const PentiOrganizationId = "f492ffcb-e516-4d03-b4bf-1532fa5cb210";
const DailyWorkingDetailtId = "8122d96f-2b80-4025-868f-126e26720f3b";
const WorkingDetailId = "525cf731-74b7-49b9-adcf-ec3fd51c11dd";
const PentiReportTemplate = {
    type: dal_constants_1.DalConstants.ReportTemplateType.CustomReport,
    name: "",
    filter: {},
    customReportId: "",
};
const ScheduledJobTemplate = {
    enabled: true,
    type: 19,
    interval: 0,
    firstExecutionDate: "",
    settings: {},
};
const ParentOrganizationUnitId = "c0db07bf-0be6-42a6-adc6-0dbdc3f207ef";
async function generateScheduledJobs(client, firstExecutionDate, generationTemplate) {
    const managerAndUnitIds = await generationTemplate.step1(client);
    for (const ids of managerAndUnitIds) {
        const { managerUserId, organizationUnitId } = ids;
        const reportTemplateId = await generationTemplate.step2(client, managerUserId, organizationUnitId);
        await generationTemplate.step3(client, firstExecutionDate, reportTemplateId, managerUserId);
    }
}
const DailyScheduledJobGenerator = {
    step1: async (client) => {
        const managerAndUnitIds = (await client.query(`
						SELECT  ou.id as "organizationUnitId",
								"managerUserId"
						FROM "${PentiOrganizationId}"."organizationUnits" as ou
						WHERE   "managerUserId" IS NOT NULL 
								AND "ancestorIds" ILIKE '%${ParentOrganizationUnitId}%'
								AND "managerUserId" != ALL( SELECT uop."userId"
															FROM "${PentiOrganizationId}"."userOrganizationProfiles" uop
															INNER JOIN "${PentiOrganizationId}"."userOrganizations" uo
																ON uop."userOrganizationId" = uo.id
															WHERE (  "extensionFields"->>'title' = 'C-LEVEL'
																	OR "extensionFields"->>'title' = 'DİREKTÖR'
																	OR "extensionFields"->>'title' = 'KIDEMLİ DİREKTÖR'
																	OR "extensionFields"->>'title' = 'CEO'
																	OR "extensionFields"->>'title' = 'PÇ CEO')
																	AND uo."isDisabled" = false
																	)
								AND ou."deletedAt" IS NULL
            `)).rows;
        return managerAndUnitIds;
    },
    step2: async (client, managerUserId, organizationUnitId) => {
        PentiReportTemplate.name = "Planlanmış Günlük Rapor";
        PentiReportTemplate.filter = {
            filters: [
                {
                    type: 11,
                    value: "2024-08-22",
                },
                {
                    type: 5,
                    value: true,
                },
                {
                    type: 4,
                    value: [organizationUnitId],
                },
                {
                    type: 3,
                },
                {
                    type: 2,
                },
                {
                    type: 1,
                },
            ],
        };
        const reportTemplateId = await addReportTemplate({
            ...PentiReportTemplate,
            organizationId: PentiOrganizationId,
            userId: managerUserId,
            trx: client,
            customReportId: DailyWorkingDetailtId,
        });
        return reportTemplateId;
    },
    step3: async (client, firstExecutionDate, reportTemplateId, managerUserId) => {
        ScheduledJobTemplate.interval = 2;
        ScheduledJobTemplate.settings = generateScheduledJobTemplateSettings(DailyWorkingDetailtId, reportTemplateId, managerUserId);
        ScheduledJobTemplate.firstExecutionDate = firstExecutionDate;
        await addScheduledJob({ ...ScheduledJobTemplate, organizationId: PentiOrganizationId, userId: managerUserId, trx: client });
    },
};
const WeeklyScheduledJobGenerator = {
    step1: async (client) => {
        const managerAndUnitIds = (await client.query(`
                    SELECT  ou.id as "organizationUnitId",
                            "managerUserId"
                    FROM "${PentiOrganizationId}"."organizationUnits" as ou
                    WHERE   "managerUserId" IS NOT NULL 
                            AND "ancestorIds" ILIKE '%${ParentOrganizationUnitId}%'
                            AND "managerUserId" != ALL( 
													SELECT uop."userId"
                                                    FROM "${PentiOrganizationId}"."userOrganizationProfiles" uop
													INNER JOIN "${PentiOrganizationId}"."userOrganizations" uo
														ON uop."userOrganizationId" = uo.id													
                                                    WHERE (  "extensionFields"->>'title' = 'C-LEVEL'
                                                            OR "extensionFields"->>'title' = 'CEO'
                                                            OR "extensionFields"->>'title' = 'PÇ CEO')
															AND uo."isDisabled" = false
															)
							AND ou."deletedAt" IS NULL
            `)).rows;
        managerAndUnitIds.push({ managerUserId: "b298836d-3ab3-4494-b709-c3cb0e05f101", organizationUnitId: "8b331e52-3cba-4ca4-bbe9-8e3c4f51c8dc" });
        managerAndUnitIds.push({ managerUserId: "57820638-a9d8-4c6d-8706-bef2caeb356a", organizationUnitId: "d030c149-97fb-4e13-951a-25d4c038a173" });
        return managerAndUnitIds;
    },
    step2: async (client, managerUserId, organizationUnitId) => {
        PentiReportTemplate.name = "Planlanmış Haftalık Rapor";
        PentiReportTemplate.filter = {
            filters: [
                {
                    type: 8,
                    value: {
                        startDay: "2024-08-15",
                        endDay: "2024-08-21",
                    },
                },
                {
                    type: 5,
                    value: true,
                },
                {
                    type: 4,
                    value: [organizationUnitId],
                },
                {
                    type: 3,
                },
                {
                    type: 2,
                },
                {
                    type: 1,
                },
            ],
        };
        const reportTemplateId = await addReportTemplate({
            ...PentiReportTemplate,
            organizationId: PentiOrganizationId,
            userId: managerUserId,
            trx: client,
            customReportId: WorkingDetailId,
        });
        return reportTemplateId;
    },
    step3: async (client, firstExecutionDate, reportTemplateId, managerUserId) => {
        ScheduledJobTemplate.interval = 3;
        ScheduledJobTemplate.settings = generateScheduledJobTemplateSettings(WorkingDetailId, reportTemplateId, managerUserId);
        ScheduledJobTemplate.firstExecutionDate = firstExecutionDate;
        await addScheduledJob({ ...ScheduledJobTemplate, organizationId: PentiOrganizationId, userId: managerUserId, trx: client });
    },
};
const MonthlyScheduledJobGenerator = {
    step1: async (client) => {
        const managerAndUnitIds = (await client.query(`
                    SELECT  ou.id as "organizationUnitId",
                            "managerUserId"
                    FROM "${PentiOrganizationId}"."organizationUnits" as ou
                    WHERE   "managerUserId" IS NOT NULL 
                            AND "ancestorIds" ILIKE '%${ParentOrganizationUnitId}%'
                            AND "managerUserId" = ANY( SELECT uop."userId"
                                                    FROM "${PentiOrganizationId}"."userOrganizationProfiles" uop
													INNER JOIN "${PentiOrganizationId}"."userOrganizations" uo
														ON uop."userOrganizationId" = uo.id														
                                                    WHERE "extensionFields"->>'title' = 'C-LEVEL' AND ("extensionFields"->>'title' <> 'CEO' OR "extensionFields"->>'title' <> 'PÇ CEO')
														  AND uo."isDisabled" = false
													)
							AND ou."deletedAt" IS NULL
            `)).rows;
        return managerAndUnitIds;
    },
    step2: async (client, managerUserId, organizationUnitId) => {
        PentiReportTemplate.name = "Planlanmış Aylık Rapor";
        PentiReportTemplate.filter = {
            filters: [
                {
                    type: 8,
                    value: {
                        startDay: "2024-08-15",
                        endDay: "2024-08-21",
                    },
                },
                {
                    type: 5,
                    value: true,
                },
                {
                    type: 4,
                    value: [organizationUnitId],
                },
                {
                    type: 3,
                },
                {
                    type: 2,
                },
                {
                    type: 1,
                },
            ],
        };
        const reportTemplateId = await addReportTemplate({
            ...PentiReportTemplate,
            organizationId: PentiOrganizationId,
            userId: managerUserId,
            trx: client,
            customReportId: WorkingDetailId,
        });
        return reportTemplateId;
    },
    step3: async (client, firstExecutionDate, reportTemplateId, managerUserId) => {
        ScheduledJobTemplate.interval = 4;
        ScheduledJobTemplate.settings = generateScheduledJobTemplateSettings(WorkingDetailId, reportTemplateId, managerUserId);
        ScheduledJobTemplate.firstExecutionDate = firstExecutionDate;
        await addScheduledJob({ ...ScheduledJobTemplate, organizationId: PentiOrganizationId, userId: managerUserId, trx: client });
    },
};
function generateScheduledJobTemplateSettings(customReportId, reportTemplateId, managerUserId) {
    return {
        format: 1,
        sendWhenEmpty: false,
        templateId: reportTemplateId,
        reportType: 99,
        customReportId: customReportId,
        notification: {
            receiverFilter: {
                roleIds: [],
                userIds: [managerUserId],
                groupIds: [],
                workPlanIds: [],
            },
            mediums: {
                web: true,
                email: false,
                pushNotification: false,
                sms: false,
            },
        },
    };
}
async function addReportTemplate(params) {
    const id = uuid_1.default.v4();
    await (params.trx ?? this._pgPool).query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}"
			(id, "createdT", name, "userId", type, filter, "customReportId")
		VALUES($1, $2, $3, $4, $5, $6, $7)
			`, [id, new Date(), params.name, params.userId, params.type, params.filter, params.customReportId]);
    return id;
}
async function addScheduledJob(params) {
    let notificationId = null;
    const createdT = new Date();
    if (params.settings?.notification?.receiverFilterId ||
        params.type === 31 ||
        params.type === 49 ||
        params.type === 32) {
        let receiverFilterId = null;
        if (params.type === 31 ||
            params.type === 49 ||
            params.type === 32) {
            receiverFilterId = params.settings.subjectedEmployeesFilter.id;
        }
        else {
            receiverFilterId = params.settings?.notification?.receiverFilterId;
        }
        notificationId = uuid_1.default.v4();
        let instanceContext = null;
        await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
				(id, "createdT", type, "createdByUserId", "receiverFilterId", email, sms, "pushNotification", web, "settings")
				VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)`, [
            notificationId,
            createdT,
            params.type,
            params.userId,
            receiverFilterId,
            params.settings.notification.mediums.email || false,
            params.settings.notification.mediums.sms || false,
            params.settings.notification.mediums.pushNotification || false,
            params.settings.notification.mediums.web || false,
            instanceContext,
        ]);
    }
    const id = params.id || uuid_1.default.v4();
    await params.trx.query(`INSERT INTO public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" 
		(id, "createdT", "organizationId", type, enabled, "createdByUserId", interval, 
		"firstExecutionDate", "nextExecutionDate", note, "notificationId")
		VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)`, [id, createdT, params.organizationId, params.type, true, params.userId, params.interval, params.firstExecutionDate, params.firstExecutionDate, params.note, notificationId]);
    if (params.type === 19) {
        const settings = params.settings;
        await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" 
		(id, "reportTemplateId", format, "sendWhenEmpty")
		VALUES ($1, $2, $3, $4 )`, [id, settings.templateId, settings.format, settings.sendWhenEmpty]);
    }
    else if ([32, 31, 49].includes(params.type)) {
        const settings = params.settings;
        await params.trx.query(`
			INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.pacsNotificationScheduledJobs}" 
			(id, "subjectedEmployeesFilterId")
			VALUES ($1, $2)`, [id, settings.subjectedEmployeesFilter.id]);
    }
}
async function main() {
    const pool = new pg_1.Pool({
        host: process.env.PGHOST,
        port: parseInt(process.env.PGPORT),
        database: process.env.PGDATABASE,
        user: process.env.PGSUPERUSER,
        application_name: "penti-scheduled-job-creator",
        max: 100,
    });
    await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
        await generateScheduledJobs(trx, "2024-09-02T09:00:00.000Z", DailyScheduledJobGenerator);
        await generateScheduledJobs(trx, "2024-09-02T09:00:00.000Z", WeeklyScheduledJobGenerator);
        await generateScheduledJobs(trx, "2024-09-01T10:00:00.000Z", MonthlyScheduledJobGenerator);
    });
}
main();
