"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.PENTIWorkingDetailsCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid = require("uuid");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const fs_1 = __importDefault(require("fs"));
const dal_constants_1 = require("../../../dal/dal.constants");
const enums_1 = require("../../../lib/access-model/v2/enums");
class PENTIWorkingDetailsCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._request = request;
        this._options = request.filter;
        this._currentDate = (0, moment_1.default)();
    }
    async generateReport() {
        this._filters = {
            userFilter: {
                organizationUnitIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                applyOrganizationUnitFilterHierarchically: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userGroupIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                userIds: this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                workPlanIds: this._options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
            },
            dateRange: {
                startDateTime: (0, moment_1.default)(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay)
                    .startOf("day")
                    .toDate(),
                endDateTime: (0, moment_1.default)(this._options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay)
                    .endOf("day")
                    .toDate(),
            },
        };
        if (this._request.reportTemplateId) {
            this._filters.dateRange.endDateTime = (0, moment_1.default)(this._filters.dateRange.endDateTime).subtract(1, "day").toDate();
        }
        let dataSummary = await dal_manager_1.dbManager.accessPacs2.getFirstInLastOutSummaryForUsers(this._request.organizationId, this._request.requesterUserId, {
            startDate: this._filters.dateRange.startDateTime,
            endDate: this._filters.dateRange.endDateTime,
            organizationUnitIds: this._filters.userFilter?.organizationUnitIds,
            organizationUnitHierarchically: this._filters.userFilter?.applyOrganizationUnitFilterHierarchically,
            userGroupIds: this._filters.userFilter?.userGroupIds,
            userIds: this._filters.userFilter?.userIds,
            workPlanIds: this._filters.userFilter?.workPlanIds,
            status: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "ASC",
            take: null,
            skip: null,
        });
        this._itemsSummary = dataSummary.items;
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "Çalışma Detayı Raporu",
            reportContainsEmptyData: this._request.reportTemplateId ? this._itemsSummary.length === 0 : undefined,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating Penti Günlük Çalışma Detayı excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: "tr" }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 10;
        this.initGeneralSheet(this._wsGeneral, 9, 1);
        let filters = [];
        if (this._filters.userFilter?.organizationUnitIds && this._filters.userFilter?.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._filters.userFilter?.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: "tr" }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (this._filters.userFilter?.applyOrganizationUnitFilterHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: "tr" }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: "tr" }));
            }
        }
        if (this._filters.userFilter?.userGroupIds && this._filters.userFilter?.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._filters.userFilter?.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: "tr" }) + " : " + userGroupNames.map((ugn) => ugn.name).join(", "));
        }
        if (this._filters.userFilter?.workPlanIds) {
            this._workplansFilter = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(this._request.organizationId, this._filters.userFilter?.workPlanIds);
        }
        if (this._filters.userFilter?.workPlanIds && this._filters.userFilter?.workPlanIds.length > 0) {
            let workplanNames = this._workplansFilter.items.filter((wp) => this._filters.userFilter?.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: "tr" }) + " : " + workplanNames.join(", "));
        }
        if (this._filters.userFilter?.userIds && this._filters.userFilter?.userIds.length > 0) {
            let identities = await dal_manager_1.dbManager.accessIdentity.listIdentity(this._request.organizationId, this._filters.userFilter?.userIds);
            let userNames = identities.items.filter((i) => this._filters.userFilter?.userIds.includes(i.id)).map((i) => i.fullname);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: "tr" }) + " : " + userNames.join(", "));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"));
        const userIds = this._itemsSummary.map((item) => item.userId);
        const filter = {
            dateRange: {
                startDateTime: this._filters.dateRange.startDateTime,
                endDateTime: this._filters.dateRange.endDateTime,
            },
            workStatusFilter: this._filters.userFilter?.workStatusFilter,
            userIds,
        };
        const userDatas = await dal_manager_1.dbManager.accessPacs2.getDailySummaryReportForMultipleUser(this._request.organizationId, filter);
        const [usersOrganizationUnits, usersExtensionFeilds] = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const [units, extensions] = await Promise.all([
                dal_manager_1.dbManager.accessUser.listUsersUnitsHierarchicallyOrdered(this._request.organizationId, userIds, trx),
                dal_manager_1.dbManager.accessUser.getUsersOrganizationProfileExtensionFields(this._request.organizationId, userIds, trx),
            ]);
            return [units, extensions];
        });
        let usersManualEntryLogCount = [];
        const filterOptions = {
            take: app_config_1.appConfig.reportExportRowLimit,
            endUtc: this._filters.dateRange.endDateTime,
            startUtc: this._filters.dateRange.startDateTime,
            userIds: userIds,
            direction: enums_1.EnumsV2.AccessDirection.Entrance,
            showOnlyManuallyInserted: true,
        };
        await dal_manager_1.dbManager.accessAccessLog.getAccessLogsReportNew(this._request.organizationId, filterOptions, async (rows) => {
            userIds.forEach((userId) => {
                const groupedByUtc = rows.items
                    .filter((item) => item.identity.id === userId)
                    .reduce((acc, item) => {
                    const date = this.formatUTCToDateString(item.utc);
                    if (!acc[date]) {
                        acc[date] = item;
                    }
                    return acc;
                }, {});
                usersManualEntryLogCount.push({
                    userId,
                    manualEntryLogCount: Object.values(groupedByUtc).length ?? 0,
                });
            });
        });
        for (let item of this._itemsSummary) {
            const userOrganizationUnits = usersOrganizationUnits.find((uou) => uou.userId === item.userId)?.units;
            const position = usersExtensionFeilds.find((uef) => uef.userId === item.userId)?.extensionFields?.position;
            const manualEntryLogCount = usersManualEntryLogCount.find((uef) => uef.userId === item.userId)?.manualEntryLogCount;
            const userData = userDatas.find((userData) => userData.userId === item.userId);
            let rowData = {
                userOrganizationUnits,
                userFullName: userData.userFullnameAndUniqueIds[0].fullName,
                position,
                physicallyInRegionDuration: 0,
                missingWorkInMinutes: 0,
                officeDaysCount: 0,
                lateArrivedDaysCount: userData.summary.lateArrivedDaysCount,
                earlyLeftDaysCount: userData.summary.earlyLeftDaysCount,
                manualEntryLogCount,
            };
            for (let dayInfo of userData.items) {
                rowData.physicallyInRegionDuration += dayInfo.physicallyInRegionDuration ?? 0;
                rowData.missingWorkInMinutes += dayInfo.missingWorkInMinutes ?? 0;
                rowData.officeDaysCount += dayInfo.physicallyInRegionDuration ? 1 : 0;
            }
            this.addGeneralRow(rowData);
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    formatUTCToDateString(date) {
        const year = date.getUTCFullYear();
        const month = String(date.getUTCMonth() + 1).padStart(2, "0");
        const day = String(date.getUTCDate()).padStart(2, "0");
        const formattedDate = `${year}-${month}-${day}`;
        return formattedDate;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.name_surname",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.DEPARTMENT",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.SECTION",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.SUBSECTION",
                cellWidth: 35,
            },
            {
                phrase: "CUSTOM-REPORT.POSITION",
                cellWidth: 35,
            },
            {
                phrase: "EXCEL-REPORT.WORKING_DURATION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.MISSING_WORK",
                cellWidth: 20,
            },
            {
                phrase: "CUSTOM-REPORT.OFFICE_WORKING_DAYS",
                cellWidth: 20,
            },
            {
                phrase: "CUSTOM-REPORT.LATE_ENTRIES",
                cellWidth: 20,
            },
            {
                phrase: "CUSTOM-REPORT.EARLY_EXITS",
                cellWidth: 20,
            },
            {
                phrase: "CUSTOM-REPORT.MANUAL_RECORDS",
                cellWidth: 20,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addGeneralRow(row) {
        let colIndex = 1;
        const rowData = [
            {
                value: row.userFullName.toUpperCase(),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            },
            {
                value: row.userOrganizationUnits[2]?.name ?? "",
            },
            {
                value: row.userOrganizationUnits[3]?.name ?? "",
            },
            {
                value: row.userOrganizationUnits[4]?.name ?? "",
            },
            {
                value: row.position ?? "",
            },
            {
                value: row.physicallyInRegionDuration ? (row.physicallyInRegionDuration / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.physicallyInRegionDuration
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: row.missingWorkInMinutes ? (row.missingWorkInMinutes / (24 * 60)) : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.missingWorkInMinutes
                    ? '[<0.000694444]"";[<0.0415][m]" ' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '";[h] "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                        ' " m "' +
                        i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                        '"'
                    : null,
            },
            {
                value: row.officeDaysCount && row.officeDaysCount != 0 ? row.officeDaysCount : "",
            },
            {
                value: row.lateArrivedDaysCount && row.lateArrivedDaysCount != 0 ? row.lateArrivedDaysCount : "",
            },
            {
                value: row.earlyLeftDaysCount && row.earlyLeftDaysCount != 0 ? row.earlyLeftDaysCount : "",
            },
            {
                value: row.manualEntryLogCount && row.manualEntryLogCount != 0 ? row.manualEntryLogCount : "",
            },
        ];
        colIndex = this.addCellsToRowWithOptions(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + "ÇALIŞMA DETAYI RAPORU", generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._currentDate.locale("tr").format("LL") + " " + this._currentDate.locale("tr").format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: false }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE_REPORT_BELONGS_TO", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        let beginDate = (0, moment_1.default)(this._filters.dateRange.startDateTime);
        let endDate = (0, moment_1.default)(this._filters.dateRange.endDateTime);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, beginDate.locale("tr").format("LL") + " - " + endDate.locale("tr").format("LL"), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 3);
        ws.mergeCells(startRowIndex + 4, startColIndex + 1, startRowIndex + 4, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 4, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: "tr" }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 4, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 4);
        row.height = filterlength * 25;
    }
}
exports.PENTIWorkingDetailsCustomReport = PENTIWorkingDetailsCustomReport;
async function generateReport(request, locale) {
    let report = new PENTIWorkingDetailsCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
