"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleGraduatesWithTemporaryAccess = exports.handleDeletedUsers = exports.syncUserChanges = exports.syncChanges = void 0;
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const bidb_1 = require("../models/bidb");
const utils_1 = require("../utils/utils");
const db_functions_1 = require("./db-functions");
const duplicate_org_unit_remover_1 = require("./duplicate-org-unit-remover");
const business_utils_1 = require("./business.utils");
const constants_1 = require("../utils/constants");
let armonOrganizationUnits = [];
let armonUserGroups = [];
const TAG = "[metu-sync][business.sync-changes.ts] ";
const syncChanges = async () => {
    await (0, cli_queries_1.systemTransaction)(utils_1.armonDBPool, async (trx) => {
        await (0, duplicate_org_unit_remover_1.duplicateOrganizationUnitRemover)(trx);
        const bidbGroupIds = await (0, db_functions_1.getBIDBGroupIds)();
        const bidbUnitNames = await (0, db_functions_1.getBIDBUnitNames)();
        armonUserGroups = await (0, db_functions_1.getArmonUserGroupIdNamePairs)(bidbGroupIds.map((m) => bidb_1.BIDBGroupIdArmonGroupNameMapping[m]), trx);
        armonOrganizationUnits = await (0, db_functions_1.getArmonOrganizationUnitIdNamePairs)(bidbUnitNames, trx);
        app_logs_1.logger.info(`${TAG}${armonOrganizationUnits.length} unit and ${armonUserGroups.length} group info loaded to memory from Armon DB !`);
        await (0, exports.handleDeletedUsers)(trx);
        await (0, exports.syncUserChanges)(trx);
        await (0, exports.handleGraduatesWithTemporaryAccess)(trx);
        app_logs_1.logger.info(`${TAG}Sync operation completed!`);
    });
};
exports.syncChanges = syncChanges;
const syncUserChanges = async (trx) => {
    const bidbUsers = await (0, cli_queries_1.systemTransaction)(utils_1.bidbDBPool, async (trx_bidb) => {
        return (0, db_functions_1.getBidbUsersWhomHasInsertOrUpdateOperations)(trx_bidb);
    });
    for (const bidbUser of bidbUsers) {
        app_logs_1.logger.info(`========================================================================================================`);
        app_logs_1.logger.info(`${TAG}User with unique id ${bidbUser.id} is processing!`);
        bidbUser.updateStatus = false;
        const existanceCheck = await (0, db_functions_1.checkArmonUserExistance)(trx, bidbUser.id);
        app_logs_1.logger.info(`${TAG}User info from BIDB database :${JSON.stringify(bidbUser, null, 4)}`);
        if (existanceCheck) {
            try {
                await (0, business_utils_1.updateExistingBidbUser)(bidbUser, armonOrganizationUnits, armonUserGroups, existanceCheck.userId, trx);
                bidbUser.updateStatus = true;
            }
            catch (error) {
                app_logs_1.logger.error(`${TAG}Update operation for user with uniqueId ${bidbUser.id} failed`);
                app_logs_1.logger.error(error);
            }
        }
        else {
            try {
                app_logs_1.logger.info(`${TAG}Importing user with uniqueId ${bidbUser.id}....`);
                await (0, business_utils_1.importNewBidbUser)(bidbUser, armonOrganizationUnits, armonUserGroups, trx);
                app_logs_1.logger.warn(`${TAG}User with uniqueId ${bidbUser.id} successfully imported to armon`);
            }
            catch (error) {
                app_logs_1.logger.error(`${TAG}Import operation for user with uniqueId ${bidbUser.id} failed`);
                app_logs_1.logger.error(error);
            }
        }
    }
    const handledUsers = bidbUsers.filter((f) => f.updateStatus === true);
    await (0, cli_queries_1.systemTransaction)(utils_1.bidbDBPool, async (trx) => {
        await (0, db_functions_1.clearUserUpdateOperationsFromBIDBDb)(trx, handledUsers.map((m) => {
            return {
                detail_updates: m.detail_updates.map((mm) => {
                    return {
                        kisi_id: mm.kisi_id,
                        rol_id: mm.rol_id,
                        islem_durumu: mm.islem_durumu,
                        guncellenme_tarihi: mm.guncellenme_tarihi,
                    };
                }),
                user_updates: m.user_updates.map((mm) => {
                    return {
                        id: mm.id,
                        islem_durumu: mm.islem_durumu,
                        guncellenme_tarihi: mm.guncellenme_tarihi,
                    };
                }),
                role_updates: m.role_updates.map((mm) => {
                    return {
                        kisi_id: mm.kisi_id,
                        rol_id: mm.rol_id,
                        islem_durumu: mm.islem_durumu,
                        guncellenme_tarihi: mm.guncellenme_tarihi,
                    };
                }),
            };
        }));
    });
    return;
};
exports.syncUserChanges = syncUserChanges;
const handleDeletedUsers = async (trx) => {
    const bidbDeletedUsers = await (0, cli_queries_1.systemTransaction)(utils_1.bidbDBPool, async (bidbtrx) => {
        return (0, db_functions_1.getBidbUsersWhomHasDeleteOperations)(bidbtrx);
    });
    for (const deletedUser of bidbDeletedUsers) {
        let deletedUserExistanceCheck = await (0, db_functions_1.checkArmonUserExistance)(trx, deletedUser.user_updates[0].id);
        if (deletedUserExistanceCheck) {
            if (!deletedUserExistanceCheck.isDisabled) {
                const lastUserUpdateOp = deletedUser.user_updates.sort((a, b) => {
                    return a.guncellenme_tarihi.localeCompare(b.guncellenme_tarihi);
                })[deletedUser.user_updates.length - 1];
                try {
                    await (0, business_utils_1.updateExistingBidbUser)({
                        id: lastUserUpdateOp.id,
                        ad: lastUserUpdateOp.ad,
                        soyad: lastUserUpdateOp.soyad,
                        eposta: lastUserUpdateOp.eposta,
                        details: [],
                        groups: [],
                        isDisabled: true,
                    }, armonOrganizationUnits, armonUserGroups, deletedUserExistanceCheck.userId, trx);
                    deletedUser.updateStatus = true;
                }
                catch (error) {
                    app_logs_1.logger.error(`${TAG}Update operation for user with uniqueId ${lastUserUpdateOp.id} failed`);
                    app_logs_1.logger.error(error);
                }
            }
            else {
                app_logs_1.logger.info(`User with unique id ${deletedUser.user_updates[0].id} is already disabled at Armon Database, skipping`);
                deletedUser.updateStatus = true;
            }
        }
        else {
            app_logs_1.logger.info(`User with unique id ${deletedUser.user_updates[0].id} is not present in Armon Database, skipping`);
            deletedUser.updateStatus = true;
        }
    }
    const handledUsers = bidbDeletedUsers.filter((f) => f.updateStatus === true);
    await (0, cli_queries_1.systemTransaction)(utils_1.bidbDBPool, async (trx) => {
        await (0, db_functions_1.clearUserUpdateOperationsFromBIDBDb)(trx, handledUsers.map((m) => {
            return {
                detail_updates: m.detail_updates.map((mm) => {
                    return {
                        kisi_id: mm.kisi_id,
                        rol_id: mm.rol_id,
                        islem_durumu: mm.islem_durumu,
                        guncellenme_tarihi: mm.guncellenme_tarihi,
                    };
                }),
                user_updates: m.user_updates.map((mm) => {
                    return {
                        id: mm.id,
                        islem_durumu: mm.islem_durumu,
                        guncellenme_tarihi: mm.guncellenme_tarihi,
                    };
                }),
                role_updates: m.role_updates.map((mm) => {
                    return {
                        kisi_id: mm.kisi_id,
                        rol_id: mm.rol_id,
                        islem_durumu: mm.islem_durumu,
                        guncellenme_tarihi: mm.guncellenme_tarihi,
                    };
                }),
            };
        }));
    });
};
exports.handleDeletedUsers = handleDeletedUsers;
const handleGraduatesWithTemporaryAccess = async (trx) => {
    const temporaryGraduates = await (0, db_functions_1.getTemporaryGraduates)(trx);
    const temporaryGraduatesToDisable = temporaryGraduates.filter((f) => f.dateDiff > constants_1.METU_TEMPORARY_GRADUATE_DAYS_TO_WAIT);
    app_logs_1.logger.info(`There are ${temporaryGraduatesToDisable.length} temporary graduates to process`);
    for (const tg of temporaryGraduatesToDisable) {
        await (0, business_utils_1.disableTemporaryGraduates)(trx, tg.userId);
    }
};
exports.handleGraduatesWithTemporaryAccess = handleGraduatesWithTemporaryAccess;
