"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCustomEndPoints = void 0;
var jwt = require("jsonwebtoken");
const app_config_1 = require("../../../app.config");
const app_auth_1 = require("../../../app.auth");
const app_logs_1 = require("../../../app.logs");
const api_validatorhelper_1 = require("../../../api/api.validatorhelper");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const ORGANIZATION_ID = "701531d6-7ce3-4dce-afb4-3303a8f49ad2";
const TeknokentA7_Met = "bd8ac7c6-088b-4030-b030-40277f0b58e0";
const TeknokentA7_Met_Kap = "036d1bc0-4091-4cef-b703-ff8e4a9ecc91";
var FilterPoint;
(function (FilterPoint) {
    FilterPoint[FilterPoint["TeknokentA7"] = 1] = "TeknokentA7";
    FilterPoint[FilterPoint["Met"] = 2] = "Met";
    FilterPoint[FilterPoint["Met_Kapali"] = 3] = "Met_Kapali";
})(FilterPoint || (FilterPoint = {}));
function addCustomEndPoints(app) {
    app.use("/teknokent", async (req, res, next) => {
        res.redirect("/app?d=TEKNOKENT&ssodisable=true");
    });
    app.post("/custom/teknopass/authorization", async (req, res, next) => {
        const authHeader = req.header("Authorization");
        if (!authHeader) {
            res.status(400).send({
                errorCode: 10000,
                message: `Authorization header is missing!`,
            });
            return;
        }
        try {
            let decodedToken = jwt.verify(authHeader.substring(7), app_config_1.appConfig.jwtSecret);
            req["auth_token"] = new app_auth_1.UserJwtPayload(decodedToken);
        }
        catch (error) {
            app_logs_1.logger.error(`[mke server-hook.ts] JWT validation error! ${error}`);
            if (error.name === "TokenExpiredError") {
                res.status(401).send({ errorCode: 10001, message: "JWT timed out, please refresh access token!" });
            }
            else if (error.name === "JsonWebTokenError") {
                res.status(400).send({
                    errorCode: 10002,
                    message: `JWT could not be validated, probably malformed, please refresh access token or login again!`,
                });
            }
            else {
                res.status(500).send({ errocCode: 10006, message: error.message });
                app_logs_1.logger.error(`Error while authenticating teknopass custom endpoint`);
                app_logs_1.logger.error(error);
            }
            return;
        }
        let reqBody = req.body;
        const validator = new api_validatorhelper_1.ValidatorHelper();
        try {
            validator.validateRegex("licensePlate", reqBody.licensePlate, false, /^(0[1-9]|[1-7][0-9]|8[01])(([A-Z])(\d{4,5})|([A-Z]{2})(\d{3,4})|([A-Z]{3})(\d{2,3}))$/);
            validator.validateEnum("filterPoint", true, FilterPoint, req.body.filterPoint);
            validator.finalize();
        }
        catch (error) {
            res.status(400).json({
                errorCode: 10003,
                message: "Request validation error",
                validationerrors: validator.errors,
            });
            return;
        }
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const userSearchResult = await dal_manager_1.dbManager.accessUser.getUserInfoByCredentialData(ORGANIZATION_ID, {
                credentialData: reqBody.licensePlate,
                credentialType: dal_constants_1.DalConstants.CredentialType.VehiclePlate,
            }, trx);
            if (!userSearchResult) {
                res.status(404).json({
                    errorCode: 10004,
                    message: `User with license plate ${reqBody.licensePlate} not found at Armon database`,
                });
                return;
            }
            const userDetails = await dal_manager_1.dbManager.accessIdentity.getIdentityDetailed(ORGANIZATION_ID, dal_constants_1.DalConstants.SystemUserId, {
                userId: userSearchResult.userId,
                hasIdentityFullWrite: true,
                hasOrganizationWideRight: true,
                trx,
            });
            let PERMITTED_USER_GROUP_IDS = [
                "0f2d31ec-b490-423b-ae78-7dcdfac0d277",
                "a3a2a144-36b0-4bb1-b03b-9a31694ea624",
                "dec25906-2f06-4f28-a649-e06662feba80",
                "e3d89dd5-d5fd-45d2-8ebd-de998067d6b3",
                "5915d6ac-ed28-48d2-8c77-1fa1cb5477a2",
            ];
            if (reqBody.filterPoint) {
                switch (reqBody.filterPoint) {
                    case FilterPoint.Met:
                        PERMITTED_USER_GROUP_IDS.push(TeknokentA7_Met);
                        break;
                    case FilterPoint.TeknokentA7:
                        let A7_ADDITIONAL_PERMITTED_IDS = [
                            "85fc62bf-69c6-485f-8547-0c979b69fdc1",
                            "6b4c8a29-1667-4540-aac8-e9a3b532ef73",
                            "e0ef5e9f-3e0e-4d7d-851f-8c5571bd8a3e",
                            "646cc39d-c276-44e5-a6e5-fa7c7404d1c2",
                            "a0a0ff1b-731c-412e-9a08-10e897eafb28",
                        ];
                        PERMITTED_USER_GROUP_IDS.push(...A7_ADDITIONAL_PERMITTED_IDS);
                        break;
                    case FilterPoint.Met_Kapali:
                        PERMITTED_USER_GROUP_IDS = [TeknokentA7_Met_Kap];
                        break;
                }
            }
            if (userDetails.userGroups.map((m) => m.id).some((s) => PERMITTED_USER_GROUP_IDS.includes(s))) {
                res.status(200).send();
            }
            else {
                res.status(401).json({
                    errorCode: 10005,
                    message: `User with license plate ${reqBody.licensePlate} is not authorized to pass`,
                });
            }
        });
    });
}
exports.addCustomEndPoints = addCustomEndPoints;
