"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = void 0;
const dal_manager_1 = require("../../../dal/dal.manager");
const moment = require("moment");
const uuid_1 = __importDefault(require("uuid"));
const path_1 = __importDefault(require("path"));
const app_config_1 = require("../../../app.config");
const exceljs_1 = __importDefault(require("exceljs"));
const app_enums_1 = require("../../../app.enums");
const dal_constants_1 = require("../../../dal/dal.constants");
const app_logs_1 = require("../../../app.logs");
const moment_range_1 = require("moment-range");
const generator_1 = require("../../../business/report/generator");
var constants;
(function (constants) {
    constants.sunday = {
        num: 0,
        name: "P",
    };
    constants.saturday = {
        num: 6,
        name: "C",
    };
    constants.months = {
        1: "Ocak",
        2: "Şubat",
        3: "Mart",
        4: "Nisan",
        5: "Mayıs",
        6: "Haziran",
        7: "Temmuz",
        8: "Ağustos",
        9: "Eylül",
        10: "Ekim",
        11: "Kasım",
        12: "Aralık",
    };
    constants.colorCodes = {
        red: "FF1818",
        blue: "1DAFF1",
        green: "20AB01",
        black: "000000",
        yellow: "FFFF00",
        orange: "FF973B",
        white: "FFFFFF",
    };
    constants.indexes = {
        USER_UNIQUE_ID: 0,
        USER_NAME: 1,
        USER_SURNAME: 2,
        MONTHLY_DATA: 3,
    };
    constants.workPlanToleraces = {
        HALF_DAY_IN_MINUTES: 240,
    };
    constants.periodColorsMatch = {
        S: constants.colorCodes.black,
        A: constants.colorCodes.green,
        G: constants.colorCodes.red,
    };
})(constants || (constants = {}));
class CustomMonthlyPerformanceRequestExcel {
    constructor(raw, validatorHelper) {
        validatorHelper.validateUUIDArray("organizationUnitIds", true, raw.organizationUnitIds);
        validatorHelper.validateUUIDArray("userGroupIds", true, raw.userGroupIds);
        validatorHelper.validateUUIDArray("userIds", true, raw.userIds);
        validatorHelper.validateUUIDArray("workPlanId", true, raw.workPlanId);
        validatorHelper.validateIsBoolean("organizationUnitHierachially", true, raw.organizationUnitHierachially);
        validatorHelper.validateDate("startDate", true, raw.startDate);
        validatorHelper.validateDate("endDate", true, raw.endDate);
        validatorHelper.finalize();
        this.startDate = new Date(raw.startDate);
        this.endDate = new Date(raw.endDate);
        this.organizationUnitIds = raw.organizationUnitIds;
        this.organizationUnitHierachially = raw.organizationUnitHierachially;
        this.userGroupIds = raw.userGroupIds;
        this.workPlanIds = raw.workPlanIds;
        this.userIds = raw.userIds;
    }
}
class ExcelReportCustomMonthlyPerformance extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wsMonthlyPerformanceRowNumber = 8;
        this._userPermissionIds = [];
        this._userPermissionPermissionTypeMapping = [];
        this._rowNumber = 1;
        this._request = request;
        this._filter = request.filter;
        let targetMonth = moment(this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "YYYY-MM");
        this._options = {
            organizationUnitHierachially: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            organizationUnitIds: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroupIds: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            workPlanIds: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
            userIds: [],
            startDate: targetMonth.startOf("month").toDate(),
            endDate: targetMonth.add(1, "month").startOf("month").toDate(),
        };
        this._wb = new exceljs_1.default.Workbook();
        this._wb.calcProperties.fullCalcOnLoad = true;
        this._wsMonthlyPerformance = this._wb.addWorksheet("Puantaj", {
            pageSetup: { paperSize: 9, orientation: "landscape" },
        });
        app_logs_1.logger.debug("Generating Custom Monthly Performance Report xlsx");
        this.initMonthlyPerformance();
    }
    async generateReport() {
        await this.setCustomMonthlyPerformanceData();
        this.finishCustomMonthlyPerformance();
        let fileId = uuid_1.default.v4();
        let filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, fileId + ".xlsx");
        await this._wb.xlsx.writeFile(filePath);
        app_logs_1.logger.debug("Report is ready -> " + filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: fileId,
            filePath: filePath,
            notificationType: undefined,
            customReportName: "Aylık Puantaj Raporu",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async setCustomMonthlyPerformanceData() {
        await dal_manager_1.dbManager.accessPacs2.customMonthlyPerformanceReportForGubretas({
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            dateRange: { startDateTime: this._options.startDate, endDateTime: this._options.endDate },
            workPlanIds: this._options.workPlanIds ? this._options.workPlanIds : null,
            userGroupIds: this._options.userGroupIds ? this._options.userGroupIds : null,
            userIds: this._options.userIds !== null ? this._options.userIds : null,
            organizationUnitIds: this._options.organizationUnitIds ? this._options.organizationUnitIds : null,
            applyOrganizationHierarchically: this._options.organizationUnitHierachially !== null ? this._options.organizationUnitHierachially : null,
            onData: this.onEmployeeData.bind(this),
        });
    }
    async onEmployeeData(rows) {
        this._userPermissionIds = [];
        for (const row of rows) {
            for (const dailyData of row[constants.indexes.MONTHLY_DATA]) {
                if (dailyData && dailyData.employeeDay && dailyData.employeeDay.p && dailyData.employeeDay.p.length > 0) {
                    dailyData.employeeDay.p.forEach((element) => {
                        this._userPermissionIds.push(element.p);
                    });
                }
            }
        }
        this._userPermissionPermissionTypeMapping = await dal_manager_1.dbManager.accessPacs2.getPermissionPermissionTypeIdMapping(this._request.organizationId, this._userPermissionIds);
        for (const row of rows) {
            this._userPermissionIds = [];
            let user = {
                uniqueId: row[constants.indexes.USER_UNIQUE_ID],
                name: row[constants.indexes.USER_NAME],
                surName: row[constants.indexes.USER_SURNAME],
                monthlyData: row[constants.indexes.MONTHLY_DATA],
                workDayCount: 0,
                sickLeaveCount: 0,
                annualPermissionCount: 0,
                weeklyLeaveCount: 0,
            };
            this.addEmployeeMonthDataRow(user, this._rowNumber);
            this._wsMonthlyPerformanceRowNumber += 2;
            this._rowNumber++;
        }
    }
    addEmployeeMonthDataRow(user, userIndex) {
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 1, userIndex, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 2, user.uniqueId, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3, user.name + " " + user.surName, null);
        let currentMonthStart = moment(this._options.startDate).startOf("month").format("YYYY-MM-DD");
        let nextMonthStart = moment(this._options.startDate).add("month", 1).startOf("month").format("YYYY-MM-DD");
        let currentDay = currentMonthStart;
        while (currentDay != nextMonthStart) {
            let dailyData = user.monthlyData.find((md) => md.date === currentDay);
            let cellData = {
                firstLineBackgroundColor: constants.colorCodes.white,
                firstLineTextColor: constants.colorCodes.black,
                firstLineData: "",
                secondLineBackgroundColor: constants.colorCodes.white,
                secondLineTextColor: constants.colorCodes.black,
                secondLineData: "",
            };
            if (dailyData && dailyData.employeeDay) {
                if ((!dailyData.employeeDay.s.ew || dailyData.employeeDay.s.ew === 0) && (!dailyData.employeeDay.p || !(dailyData.employeeDay.p.length > 0))) {
                    cellData.firstLineBackgroundColor = constants.colorCodes.white;
                    cellData.firstLineData = "";
                    cellData.firstLineTextColor = "";
                    cellData.secondLineBackgroundColor = constants.colorCodes.white;
                    cellData.secondLineData = "T";
                    cellData.secondLineTextColor = constants.colorCodes.red;
                    user.weeklyLeaveCount++;
                }
                else {
                    if (dailyData.employeeDay.p && dailyData.employeeDay.p.length > 0) {
                        let permissions = dailyData.employeeDay.p;
                        let permissionTotal = 0;
                        for (const iterator of permissions) {
                            permissionTotal += iterator.u;
                        }
                        if (permissionTotal > constants.workPlanToleraces.HALF_DAY_IN_MINUTES) {
                            let permissionTypeId = this._userPermissionPermissionTypeMapping.find((upptm) => upptm.id === dailyData.employeeDay.p[0].p).ppermissionTypeId;
                            switch (permissionTypeId) {
                                case "17ecd956-e08f-40ed-8bf8-9f4624823b8f":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.blue;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.blue;
                                    cellData.secondLineData = "Yİ";
                                    user.annualPermissionCount++;
                                    break;
                                case "d5b5044a-31c7-42df-9de2-82bcd1563a17":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "Gİ";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "f8e1538a-4b2d-4454-adf3-22d911237812":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "Dİ";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "9d8513cb-003f-4825-89ae-9dd9fb11bbc5":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "H";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "c466d2e0-eda6-4091-a391-9b13b41d4ab9":
                                    cellData.firstLineTextColor = constants.colorCodes.blue;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.white;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.blue;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.white;
                                    cellData.secondLineData = "İK";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "276f5b17-7832-44a7-a4b1-60f8f8731722":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "Eİ";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "7fd46ae0-c584-4dec-b269-35bd06d127e3":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "Öİ";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "ec415398-bb4d-4fda-ba9e-e8a8c8d9b0d4":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "İZ";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "8c5ed703-aafb-40bb-9ba0-89b9aa77a2f9":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "ÜİZ";
                                    user.weeklyLeaveCount++;
                                    break;
                                case "6fa5af9a-f038-490a-886f-368ea4e7951e":
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "SİZ";
                                    user.weeklyLeaveCount++;
                                    break;
                                default:
                                    cellData.firstLineTextColor = constants.colorCodes.red;
                                    cellData.firstLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.firstLineData = "";
                                    cellData.secondLineTextColor = constants.colorCodes.red;
                                    cellData.secondLineBackgroundColor = constants.colorCodes.yellow;
                                    cellData.secondLineData = "Mİ";
                                    user.weeklyLeaveCount++;
                                    break;
                            }
                            user.weeklyLeaveCount++;
                        }
                    }
                    if (dailyData &&
                        dailyData.employeeDay &&
                        dailyData.employeeDay.ar &&
                        dailyData.employeeDay.ar.length > 0 &&
                        dailyData.employeeDay.ar[0].a &&
                        dailyData.employeeDay.ar[0].a.length > 0) {
                        let firstInTime;
                        let workPlanSegmentOfWorkingHours = dailyData.employeeDay.ws.find((w) => w.wt === dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours);
                        if (workPlanSegmentOfWorkingHours) {
                            let workPlanWorkingHoursRange = new moment_range_1.DateRange(new Date(workPlanSegmentOfWorkingHours.s), new Date(workPlanSegmentOfWorkingHours.e));
                            for (const access of dailyData.employeeDay.ar[0].a) {
                                let accessRange = new moment_range_1.DateRange(new Date(access.s), new Date(access.e));
                                if (workPlanWorkingHoursRange.overlaps(accessRange)) {
                                    if (!firstInTime) {
                                        firstInTime = moment(access.s);
                                    }
                                    if (firstInTime && moment(access.s).isBefore(firstInTime)) {
                                        firstInTime = moment(access.s);
                                    }
                                }
                            }
                        }
                        if (firstInTime) {
                            let timePeriods = {
                                S: moment(dailyData.date + " 08:00", "YYYY-MM-DD hh:mm"),
                                A: moment(dailyData.date + " 16:00", "YYYY-MM-DD hh:mm"),
                                G: moment(dailyData.date + " 00:00", "YYYY-MM-DD hh:mm"),
                            };
                            let checkPeriod;
                            let diff;
                            for (let key of Object.keys(timePeriods)) {
                                let d = Math.abs(firstInTime.diff(timePeriods[key]));
                                if (diff === undefined || diff === null) {
                                    checkPeriod = key;
                                    diff = d;
                                    continue;
                                }
                                if (d < diff) {
                                    checkPeriod = key;
                                    diff = d;
                                }
                            }
                            cellData.firstLineTextColor = constants.periodColorsMatch[checkPeriod];
                            cellData.firstLineData = checkPeriod;
                        }
                        else {
                            cellData.firstLineTextColor = constants.colorCodes.red;
                            cellData.firstLineData = "";
                        }
                    }
                    else {
                        cellData.firstLineTextColor = constants.colorCodes.red;
                        cellData.firstLineData = "";
                    }
                    if (dailyData.employeeDay.s.n > 0) {
                        if (dailyData.employeeDay.s.n < dailyData.employeeDay.s.ew)
                            cellData.secondLineTextColor = constants.colorCodes.black;
                        cellData.secondLineData = "X";
                        user.workDayCount++;
                    }
                    else {
                        if (!dailyData.employeeDay.p || !(dailyData.employeeDay.p.length > 0)) {
                            cellData.secondLineData = "D";
                        }
                    }
                }
            }
            this.addDataCellWithStyle(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + moment(currentDay).get("date"), cellData);
            currentDay = moment(currentDay).add(1, "days").format("YYYY-MM-DD");
        }
        currentDay = currentMonthStart;
        while (currentDay != nextMonthStart) {
            let dailyData = user.monthlyData.find((md) => md.date === currentDay);
            let nextDayData = user.monthlyData.find((md) => md.date === moment(currentDay).add(1, "days").format("YYYY-MM-DD"));
            let cellData = {
                firstLineBackgroundColor: constants.colorCodes.white,
                firstLineTextColor: constants.colorCodes.black,
                firstLineData: "",
                secondLineBackgroundColor: constants.colorCodes.white,
                secondLineTextColor: constants.colorCodes.black,
                secondLineData: "",
            };
            if (dailyData && dailyData.employeeDay) {
                if (dailyData.employeeDay.s.ew === 0 && dailyData.employeeDay.s.e > 0) {
                    if (nextDayData && nextDayData.employeeDay.s.ew === 0) {
                        cellData.secondLineBackgroundColor = constants.colorCodes.green;
                        cellData.secondLineData = "X";
                        cellData.secondLineTextColor = constants.colorCodes.black;
                    }
                    else {
                        cellData.secondLineBackgroundColor = constants.colorCodes.red;
                        cellData.secondLineData = "X";
                        cellData.secondLineTextColor = constants.colorCodes.black;
                    }
                    this.addDataCellWithStyle(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + moment(currentDay).get("date"), cellData);
                }
            }
            currentDay = moment(currentDay).add(1, "days").format("YYYY-MM-DD");
        }
        let daysInMonth = moment(currentMonthStart).daysInMonth();
        let total = 0;
        if (user.workDayCount) {
            total += user.workDayCount;
        }
        if (user.sickLeaveCount) {
            total += user.sickLeaveCount;
        }
        if (user.weeklyLeaveCount) {
            total += user.weeklyLeaveCount;
        }
        if (user.annualPermissionCount) {
            total += user.annualPermissionCount;
        }
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 1, user.workDayCount ? user.workDayCount : null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 2, user.sickLeaveCount ? user.sickLeaveCount : null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 3, user.weeklyLeaveCount ? user.weeklyLeaveCount : null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 4, user.annualPermissionCount ? user.annualPermissionCount : null, constants.colorCodes.red);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 5, total);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 6, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 7, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 8, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 9, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 10, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 11, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 12, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 13, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 14, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 15, null);
        this.addDataCell(this._wsMonthlyPerformance, this._wsMonthlyPerformanceRowNumber, 3 + daysInMonth + 16, null);
    }
    addDataCell(ws, currentRowNumber, col, value, color) {
        let c = ws.getCell(currentRowNumber, col);
        c.value = value;
        c.font = {
            color: { argb: color },
            size: 11,
        };
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = {
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "medium",
            },
        };
        ws.mergeCells(currentRowNumber, col, currentRowNumber + 1, col);
    }
    addDataCellWithStyle(ws, currentRowNumber, col, value) {
        let c_first = ws.getCell(currentRowNumber, col);
        let c_second = ws.getCell(currentRowNumber + 1, col);
        c_first.value = value.firstLineData;
        c_first.font = {
            color: { argb: value.firstLineTextColor },
            bold: false,
            size: 8,
        };
        c_first.fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: value.firstLineBackgroundColor },
        };
        c_first.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c_first.border = {
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "dotted",
            },
        };
        c_second.value = value.secondLineData;
        c_second.font = {
            color: { argb: value.secondLineTextColor },
            bold: false,
            size: 8,
        };
        c_second.fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: value.secondLineBackgroundColor },
        };
        c_second.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c_second.border = {
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "medium",
            },
        };
    }
    async initMonthlyPerformance() {
        let organizationUnitsAndParents = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            organizationUnitsAndParents = await dal_manager_1.dbManager.accessPacs2.getOrganizationUnitsWithParentInfo(this._request.organizationId, this._options.organizationUnitIds);
        }
        let totalColNum = 1;
        let columns = [{ width: 4 }, { width: 7 }, { width: 18 }];
        const initHeaderValues = ["SIRA\r\nNO", "FAB.\r\nNO", "ADI VE SOYADI"];
        for (let index = 1; index <= initHeaderValues.length; index++) {
            let c = this._wsMonthlyPerformance.getCell(5, index);
            c.value = initHeaderValues[index - 1];
            c.font = {
                bold: true,
                size: 9,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "medium",
                },
                left: {
                    style: "medium",
                },
                right: {
                    style: "medium",
                },
                bottom: {
                    style: "medium",
                },
            };
            this._wsMonthlyPerformance.mergeCells(5, index, 7, index);
            totalColNum++;
        }
        let requestedMonth = moment(this._options.startDate, "YYYYM");
        let startOfMonth = moment(requestedMonth.startOf("month").toDate());
        let endOfMonth = moment(requestedMonth.endOf("month").startOf("day").toDate());
        let d = startOfMonth.clone();
        let prevMonth = constants.months[requestedMonth.subtract(1, "month").format("M")].toUpperCase();
        while (d.isSameOrBefore(endOfMonth)) {
            columns.push({ width: 2.5 });
            let c_f = this._wsMonthlyPerformance.getCell(5, totalColNum);
            c_f.border = {
                top: {
                    style: "medium",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "dotted",
                },
            };
            let c_s = this._wsMonthlyPerformance.getCell(6, totalColNum);
            c_s.value = d.date();
            c_s.font = {
                bold: true,
                size: 7,
            };
            c_s.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c_s.border = {
                top: {
                    style: "dotted",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "dotted",
                },
            };
            let c_t = this._wsMonthlyPerformance.getCell(7, totalColNum);
            if (d.weekday() === constants.saturday.num) {
                c_t.value = constants.saturday.name;
            }
            else if (d.weekday() === constants.sunday.num) {
                c_t.value = constants.sunday.name;
            }
            c_t.font = {
                bold: true,
                color: { argb: "FF0000" },
                size: 10,
            };
            c_t.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c_t.border = {
                top: {
                    style: "dotted",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "medium",
                },
            };
            totalColNum++;
            d = d.add(1, "day");
        }
        const lastHeaderValues = ["İŞ GÜN", "HASTA", "HFT.T\r\nGN.T.", "YIILIK\r\nÜCRE.\r\nİZİN", "TOPLAM", "VARD.\r\nZAMMI"];
        for (let index = 1; index <= lastHeaderValues.length; index++) {
            columns.push({ width: 6 });
            let c = this._wsMonthlyPerformance.getCell(5, totalColNum++);
            c.value = lastHeaderValues[index - 1];
            c.font = {
                size: 7,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "medium",
                },
                left: {
                    style: "medium",
                },
                right: {
                    style: "medium",
                },
                bottom: {
                    style: "medium",
                },
            };
            this._wsMonthlyPerformance.mergeCells(5, totalColNum - 1, 7, totalColNum - 1);
        }
        columns.push({ width: 3 });
        let c = this._wsMonthlyPerformance.getCell(5, totalColNum++);
        c.value = "VARDİYA\r\nARA ZAMMI";
        c.font = {
            size: 7,
        };
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "medium",
            },
            left: {
                style: "medium",
            },
            right: {
                style: "medium",
            },
            bottom: {
                style: "medium",
            },
        };
        this._wsMonthlyPerformance.mergeCells(5, totalColNum - 1, 6, totalColNum);
        c = this._wsMonthlyPerformance.getCell(7, totalColNum - 1);
        c.value = "25%";
        c.font = {
            size: 7,
        };
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "medium",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "medium",
            },
        };
        columns.push({ width: 3 });
        c = this._wsMonthlyPerformance.getCell(7, totalColNum++);
        c.value = "50%";
        c.font = {
            size: 7,
        };
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "medium",
            },
            bottom: {
                style: "medium",
            },
        };
        c = this._wsMonthlyPerformance.getCell(5, totalColNum++);
        c.value = "TOZ PRİMİ GÜN ADEDİ";
        c.font = {
            size: 7,
        };
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "medium",
            },
            left: {
                style: "medium",
            },
            right: {
                style: "medium",
            },
            bottom: {
                style: "medium",
            },
        };
        this._wsMonthlyPerformance.mergeCells(5, totalColNum - 1, 6, totalColNum + 3);
        for (let i = 1; i < 6; i++) {
            columns.push({ width: 3 });
            c = this._wsMonthlyPerformance.getCell(7, totalColNum - 1);
            c.value = i.toString();
            c.font = {
                size: 7,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "medium",
                },
            };
            totalColNum++;
        }
        const finishHeaderValues = ["YOL\r\nGÜN\r\nADEDİ", "YEMEK\r\nGÜN\r\nADEDİ", prevMonth + "\r\nAYI ÜRETİM\r\nPRİMİ"];
        for (let index = 1; index <= finishHeaderValues.length; index++) {
            columns.push({ width: 6 });
            let c = this._wsMonthlyPerformance.getCell(5, totalColNum - 1);
            c.value = finishHeaderValues[index - 1];
            c.font = {
                size: 7,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "medium",
                },
                left: {
                    style: "medium",
                },
                right: {
                    style: "medium",
                },
                bottom: {
                    style: "medium",
                },
            };
            this._wsMonthlyPerformance.mergeCells(5, totalColNum - 1, 7, totalColNum - 1);
            totalColNum++;
        }
        this._wsMonthlyPerformance.columns = columns;
        this._wsMonthlyPerformance.views = [{ state: "frozen", xSplit: 0, ySplit: 7, activeCell: "A8", showGridLines: false }];
        this._wsMonthlyPerformance.autoFilter = {
            from: "A5",
            to: "C5",
        };
        let departmentNames;
        if (organizationUnitsAndParents && organizationUnitsAndParents.length > 0) {
            departmentNames = organizationUnitsAndParents.map((ouap) => (ouap.parentName ? ouap.parentName : "Mevcut Değil")).join(",");
        }
        else {
            departmentNames = "Seçim Yapılmadı";
        }
        let unitNames;
        if (organizationUnitsAndParents && organizationUnitsAndParents.length > 0) {
            unitNames = organizationUnitsAndParents.map((ouap) => (ouap.name ? ouap.name : "Mevcut Değil")).join(",");
        }
        else {
            unitNames = "Seçim Yapılmadı";
        }
        this._wsMonthlyPerformance.pageSetup.printTitlesRow = "5:7";
        this._wsMonthlyPerformance.headerFooter.oddHeader =
            "&L" +
                "GÜBRETAŞ GÜBRE FABRİKALARI T.A.Ş.\nYarımca Tesisleri Müdürlüğü\n\n" +
                "&C" +
                constants.months[moment(this._options.startDate, "YYYYM").get("month") + 1] +
                " " +
                requestedMonth.get("year") +
                "  AYI\nDEVAM PUANTAJ CETVELİ\n\n" +
                "&R" +
                "Departman:" +
                departmentNames +
                "\n" +
                "Birim:" +
                unitNames +
                "\n\n";
        this._wsMonthlyPerformance.headerFooter.oddFooter = "Sayfa &P / &N" + "&R" + "&D  &T";
        this.addFilterSummaryToSheet(this._wsMonthlyPerformance, totalColNum, departmentNames, unitNames);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        this._wb.creator = this._requestedUserCaption;
    }
    addFilterSummaryToSheet(ws, colNum, departmens, units) {
        let c = ws.getCell(1, 1);
        c.value = "GÜBRETAŞ";
        c.alignment = {
            horizontal: "left",
        };
        c.font = {
            bold: true,
        };
        c.border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
        };
        c = ws.getCell(2, 1);
        c.value = "Gübre Fabrikaları T.A.Ş.";
        c.font = {
            italic: true,
        };
        c.border = {
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
        };
        c = ws.getCell(3, 1);
        c.value = "Yarımca Tesisleri Müdürlüğü";
        c.font = {
            italic: true,
        };
        c.border = {
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(1, 1, 1, 3);
        ws.mergeCells(2, 1, 2, 3);
        ws.mergeCells(3, 1, 3, 3);
        c = ws.getCell(1, colNum - 11);
        c.value = "DEPARTMAN NO";
        c.font = {
            bold: true,
            size: 9,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(1, colNum - 11, 1, colNum - 7);
        c = ws.getCell(1, colNum - 6);
        c.value = departmens;
        c.font = {
            size: 9,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(1, colNum - 6, 1, colNum - 2);
        c = ws.getCell(2, colNum - 11);
        c.value = "BÖLÜMÜ";
        c.font = {
            bold: true,
            size: 9,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(2, colNum - 11, 2, colNum - 7);
        c = ws.getCell(2, colNum - 6);
        c.value = units;
        c.font = {
            size: 9,
        };
        c.border = {
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(2, colNum - 6, 2, colNum - 2);
        let middleColNum = Math.round(colNum / 2);
        c = ws.getCell(2, middleColNum - 7);
        c.value = constants.months[new Date(this._options.startDate).getMonth() + 1] + " " + new Date(this._options.startDate).getFullYear() + "   AYI";
        c.font = {
            size: 12,
            bold: true,
        };
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(2, middleColNum - 7, 2, middleColNum + 7);
        c = ws.getCell(3, middleColNum - 7);
        c.value = "DEVAM PUANTAJ CETVELİ";
        c.font = {
            size: 12,
            bold: true,
        };
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(3, middleColNum - 7, 3, middleColNum + 7);
    }
    finishCustomMonthlyPerformance() {
        let rowNum = this._wsMonthlyPerformanceRowNumber;
        let colNum = 1;
        const initHeaderValues = ["", "", "YEKÜN"];
        for (let index = 1; index <= initHeaderValues.length; index++) {
            let c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
            c.value = initHeaderValues[index - 1];
            c.font = {
                bold: true,
                size: 12,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "medium",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "medium",
                },
            };
            this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum + 1, colNum);
            colNum++;
        }
        let requestedMonth = moment(this._options.startDate, "YYYYM");
        let startOfMonth = moment(requestedMonth.startOf("month").toDate());
        let endOfMonth = moment(requestedMonth.endOf("month").startOf("day").toDate());
        let d = startOfMonth.clone();
        while (d.isSameOrBefore(endOfMonth)) {
            let c_f = this._wsMonthlyPerformance.getCell(rowNum, colNum);
            c_f.value = d.date();
            c_f.font = {
                bold: true,
                size: 7,
            };
            c_f.alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            c_f.border = {
                top: {
                    style: "medium",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thin",
                },
            };
            let c_s = this._wsMonthlyPerformance.getCell(rowNum + 1, colNum++);
            c_s.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "medium",
                },
            };
            d = d.add(1, "day");
        }
        let formula = [];
        while (colNum <= 3 + requestedMonth.daysInMonth() + 16) {
            for (let tmpRow = 8; tmpRow < rowNum; tmpRow += 2) {
                formula.push(this._wsMonthlyPerformance.getCell(tmpRow, colNum).address);
            }
            let cellToAddFormula = this._wsMonthlyPerformance.getCell(rowNum, colNum);
            let cellColRow = cellToAddFormula.$col$row.split("$");
            cellToAddFormula.value = {
                formula: "SUM(" + cellColRow[1] + "8:" + cellColRow[1] + (this._wsMonthlyPerformanceRowNumber - 2) + ") ",
            };
            this._wsMonthlyPerformance.getCell(rowNum, colNum).alignment = {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            };
            this._wsMonthlyPerformance.getCell(rowNum, colNum).font = {
                bold: true,
                size: 8,
            };
            this._wsMonthlyPerformance.getCell(rowNum, colNum).border = {
                top: {
                    style: "medium",
                },
                left: {
                    style: "medium",
                },
                right: {
                    style: "medium",
                },
                bottom: {
                    style: "medium",
                },
            };
            this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum + 1, colNum);
            colNum++;
            formula = [];
        }
        this._wsMonthlyPerformanceRowNumber += 2;
        let firstCol = ["X", "D", "T Kırmızı", "X Yeşil", "X Kırmızı", "X Mavi", "Gİ", "Yİ", "Dİ", "H Kırmızı", "İK"];
        let secondCol = [
            "İşaretli Çalışılan Gün",
            "Devamsız",
            "Hafta Tatili",
            'Çalışılan 6. Gün "',
            'Çalışılan 7. Gün "',
            "Resmi veya Bayram Tatili",
            "Görevli",
            "Yıllık Ücretli İzin",
            "Doğum Ücretli İzni",
            "Hastalık, İstirahatli",
            "İş Kazası",
        ];
        let thirdCol = ["Eİ", "Öİ", "RT", "Mİ", "F.İz", "İZ", "A.Yeşil", "G.Kırmızı", "ÜİZ", "Asİz", "SİZ"];
        let fourthCol = [
            "Evlenme Ücretli İzni",
            "Ölüm Ücretli İzni",
            "Resmi Bayram Tatili",
            "Mazeret Ücretli İzin",
            "Doğal Facia İzni",
            "İdari Ücretli İzin",
            "Akşam Vardiyası",
            "Gece Vardiyası",
            "Ücretsiz İzin",
            "Askerlik İzni",
            "Sendika İzni",
        ];
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        colNum = 1;
        for (let i = 0; i < firstCol.length; i++) {
            let c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
            c.value = firstCol[i];
            c.alignment = {
                horizontal: "left",
                vertical: "middle",
                wrapText: true,
            };
            c.font = {
                size: 9,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "dotted",
                },
                bottom: {
                    style: "thin",
                },
            };
            this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum, colNum + 1);
            rowNum++;
        }
        colNum += 2;
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        for (let i = 0; i < secondCol.length; i++) {
            let c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
            c.value = secondCol[i];
            c.alignment = {
                horizontal: "left",
                vertical: "middle",
                wrapText: true,
            };
            c.font = {
                size: 9,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "dotted",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thin",
                },
            };
            rowNum++;
        }
        colNum += 3;
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        for (let i = 0; i < thirdCol.length; i++) {
            let c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
            c.value = thirdCol[i];
            c.alignment = {
                horizontal: "left",
                vertical: "middle",
                wrapText: true,
            };
            c.font = {
                size: 9,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "dotted",
                },
                bottom: {
                    style: "thin",
                },
            };
            this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum, colNum + 3);
            rowNum++;
        }
        colNum += 4;
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        for (let i = 0; i < fourthCol.length; i++) {
            let c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
            c.value = fourthCol[i];
            c.alignment = {
                horizontal: "left",
                vertical: "middle",
                wrapText: true,
            };
            c.font = {
                size: 9,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "dotted",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thin",
                },
            };
            this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum, colNum + 6);
            rowNum++;
        }
        colNum += 8;
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        let c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
        c.value =
            "Fabrika Müdürlüğünü\r\nYukarıdaki puantaj tanzim edilen işçilerin " +
                constants.months[requestedMonth.get("month") + 1] +
                " " +
                requestedMonth.get("year") +
                " ayına ait çalıştıkları gün ve miktarlar hizalarında gösterilmiştir. Tahakkuku emirlerinize arz olunur.";
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            size: 10,
        };
        this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum + 7, colNum + 12);
        colNum = 4 + requestedMonth.daysInMonth();
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
        c.value = "Puantajı Düzenleyen\r\nİ.K Uzm.Yard.";
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            size: 9,
        };
        this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum + 1, colNum + 3);
        colNum += 5;
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
        c.value = "Kontrol Eden\r\nİnsan Kaynakları Şefi";
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            size: 9,
        };
        this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum + 1, colNum + 3);
        colNum += 5;
        rowNum = this._wsMonthlyPerformanceRowNumber + 1;
        c = this._wsMonthlyPerformance.getCell(rowNum, colNum);
        c.value = "Tahakkuka alınması uygundur.\r\nYönetici";
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            size: 9,
        };
        this._wsMonthlyPerformance.mergeCells(rowNum, colNum, rowNum + 1, colNum + 5);
        this._wsMonthlyPerformanceRowNumber += firstCol.length + 1;
        colNum += 5;
        let lastCell = this._wsMonthlyPerformance.getCell(this._wsMonthlyPerformanceRowNumber, colNum);
        let lastCellColRow = lastCell.$col$row.split("$");
        this._wsMonthlyPerformance.pageSetup.printArea = "A5:" + lastCellColRow[1] + this._wsMonthlyPerformanceRowNumber;
    }
}
async function generateReport(request, locale) {
    let report = new ExcelReportCustomMonthlyPerformance(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
