"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GDZ_UtilityModule = exports.ADM_UtilityModule = exports.UtilityModule = exports.generateExcelCountIfFormulas = exports.totalWorkedWeekends = exports.totalWorkedHolidays = exports.substractWeeklyHolidaysForAnnualLeaves = exports.getRefectoryAccessCountOfUserBetweenDates = exports.generateEmployeeDataQuery = void 0;
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const constants_1 = require("./constants");
const luxon_1 = require("luxon");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const moment_1 = __importDefault(require("moment"));
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const dal_constants_1 = require("../../../dal/dal.constants");
function generateEmployeeDataQuery(organizationId, userFilter, startDate, startDateWithOffset, endDate, endDateWithOffset) {
    let bindings = userFilter.bindingKeys;
    const result = `
		DECLARE employee_day_cursor
		CURSOR FOR
		WITH employee_day_cte AS(
			SELECT	"userId",
					"uniqueId",
					fullname,
					COALESCE("extensionFields","extensionFields",'{}') as "extensionFields",
					"employmentStartUtc",
					"employmentEndUtc",
					jsonb_agg("data" ORDER BY data->>'d' ASC) "monthlyData",
					SUM(CASE WHEN (ordered.data->'s'->>'wd')::int > 0 AND (data->>'d')::timestamp with time zone::date >= $${userFilter.bindingKeys.length + 1} THEN 1 END)::integer as "countOfWorkingDays"
			FROM
			(
				SELECT  ed."userId",
						uop."uniqueId",
						uop."name" || ' ' || uop."surname" as fullname,
						uop."extensionFields"::jsonb,
						uop."employmentStartUtc",
						uop."employmentEndUtc",
						ed."data"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop
					ON uo.id = uop."userOrganizationId"
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS ed
					ON ed."userId" = uop."userId"
				INNER JOIN
					(${userFilter.query}) AS T3
					ON T3."userId" = ed."userId" AND ed."data" IS NOT NULL
				WHERE ed.date >= $${userFilter.bindingKeys.length + 3} AND ed.date <= $${userFilter.bindingKeys.length + 4}
				ORDER BY ed.date ASC
			) as ordered
			GROUP BY "userId", "uniqueId", fullname, "extensionFields"::jsonb, "employmentStartUtc", "employmentEndUtc"
		), permission_cte AS (
			SELECT  pp.id,
					upp."userId",
					pt.id as "typeId",
                    pp."dynamicFormData",
					tstzrange($${userFilter.bindingKeys.length + 1}, $${userFilter.bindingKeys.length + 4}) * tstzrange(pp."startDateTime", pp."endDateTime") as  "permissionRange",
					CASE    WHEN tstzrange($${userFilter.bindingKeys.length + 3}, $${userFilter.bindingKeys.length + 1}) && tstzrange(pp."startDateTime", pp."endDateTime") 
								 AND pp."requestDateTime" >=  $${userFilter.bindingKeys.length + 3} AND pp."requestDateTime" < $${userFilter.bindingKeys.length + 2}
							THEN tstzrange(pp."startDateTime", pp."endDateTime") * tstzrange($${userFilter.bindingKeys.length + 3} , $${userFilter.bindingKeys.length + 4})
							WHEN tstzrange($${userFilter.bindingKeys.length + 3}, $${userFilter.bindingKeys.length + 1}) && tstzrange(pp."startDateTime", pp."endDateTime") 
								 AND pp."requestDateTime" >=  $${userFilter.bindingKeys.length + 3} 
							THEN tstzrange(pp."startDateTime", pp."endDateTime") * tstzrange($${userFilter.bindingKeys.length + 3} , $${userFilter.bindingKeys.length + 2})
							WHEN  tstzrange($${userFilter.bindingKeys.length + 1}, $${userFilter.bindingKeys.length + 4}) && tstzrange(pp."startDateTime", pp."endDateTime")
								AND pp."requestDateTime" < $${userFilter.bindingKeys.length + 2}
							THEN tstzrange($${userFilter.bindingKeys.length + 1}, $${userFilter.bindingKeys.length + 4}) * tstzrange(pp."startDateTime", pp."endDateTime")
							WHEN  tstzrange($${userFilter.bindingKeys.length + 1}, $${userFilter.bindingKeys.length + 4}) &&  tstzrange(pp."startDateTime", pp."endDateTime")
								AND pp."requestDateTime" >= $${userFilter.bindingKeys.length + 2}
							THEN tstzrange($${userFilter.bindingKeys.length + 1}, $${userFilter.bindingKeys.length + 2}) * tstzrange(pp."startDateTime", pp."endDateTime")
					ELSE NULL END as "permissionIntersection"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" as pp
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" as upp
				ON pp.id = upp."ppermissionId"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" as pt
				ON  pt.id = pp."ppermissionTypeId"
			WHERE	pp.status = 1 
		)
		SELECT  edc.*,
				perms."permissions"
		FROM employee_day_cte as edc
		LEFT JOIN (
            SELECT 	
                "userId",
                jsonb_agg(
                    jsonb_build_object(
					'id',"id",
                    'typeId', "typeId",
                    'dynamicFormData', "dynamicFormData",
					'startDate', lower("permissionRange"),
					'endDate', upper("permissionRange"),
					'offsettingStartDate', lower("permissionIntersection"),
					'offsettingEndDate', upper("permissionIntersection"),
                    'usageDay', EXTRACT (DAY FROM (upper("permissionIntersection") - lower("permissionIntersection")))
                    )
                ) as permissions
            FROM permission_cte
            WHERE "permissionIntersection" IS NOT NULL
            GROUP BY "userId"
        ) perms
		ON perms."userId" = edc."userId"
		ORDER BY fullname ASC;
    `;
    bindings.push(startDate, endDateWithOffset, startDateWithOffset, endDate);
    return { query: result, bindings: bindings };
}
exports.generateEmployeeDataQuery = generateEmployeeDataQuery;
async function getRefectoryAccessCountOfUserBetweenDates(organizationId, userId, regionId, startDate, endDate, trx) {
    const result = (await trx.query(`
				SELECT COUNT(DISTINCT ("actionUtc"::date))::int as "visitsCountsOfRefectory"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" as al
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints}" as racp
					ON (al.log->>'a')::uuid = racp."accessControlPointId"
				WHERE racp."regionId" = $1
						AND al."actionUtc" >= $2
						AND al."actionUtc" <= $3
						AND al.log->>'o' = $4
				`, [regionId, startDate, endDate, userId])).rows[0]?.visitsCountsOfRefectory ?? 0;
    return result;
}
exports.getRefectoryAccessCountOfUserBetweenDates = getRefectoryAccessCountOfUserBetweenDates;
function substractWeeklyHolidaysForAnnualLeaves(user, permissionTypeId, feeCount) {
    const annualLeavesSubset = user.permissions?.filter((perm) => perm.typeId === permissionTypeId && perm.usageDay >= 5);
    annualLeavesSubset?.forEach((perm) => {
        if (luxon_1.DateTime.fromISO(perm.offsettingEndDate).weekday >= 5) {
            let current = luxon_1.DateTime.fromISO(perm.offsettingEndDate);
            let endOfWeek = current.endOf("week");
            while (current < endOfWeek) {
                const dayData = user.monthlyData.find((data) => luxon_1.DateTime.fromISO(data.d).equals(current));
                if (dayData && dayData.s.wd === 0) {
                    feeCount -= 1;
                }
                current = current.plus({ days: 1 });
            }
        }
    });
    return feeCount;
}
exports.substractWeeklyHolidaysForAnnualLeaves = substractWeeklyHolidaysForAnnualLeaves;
function totalWorkedHolidays(userMonthlyData) {
    let counter = 0;
    userMonthlyData.forEach((elem) => {
        const workPlanDuration = elem.s.wd;
        const expectedHoliday = elem.s.eh;
        const extraWorkingHour = elem.s.e;
        if (workPlanDuration === 0 && expectedHoliday > 0 && extraWorkingHour > 0) {
            counter++;
        }
    });
    return counter;
}
exports.totalWorkedHolidays = totalWorkedHolidays;
function totalWorkedWeekends(userMonthlyData) {
    let counter = 0;
    userMonthlyData.forEach((elem) => {
        const workPlanDuration = elem.s.wd;
        const expectedHoliday = elem.s.eh;
        const extraWorkingHour = elem.s.e;
        if (workPlanDuration === 0 && expectedHoliday === 0 && extraWorkingHour > 0) {
            counter++;
        }
    });
    return counter;
}
exports.totalWorkedWeekends = totalWorkedWeekends;
function generateExcelCountIfFormulas(beginCell, endCell, conditions) {
    return { formula: conditions.map((elem) => `COUNTIF(${beginCell}:${endCell},"${elem}")`).join("+") };
}
exports.generateExcelCountIfFormulas = generateExcelCountIfFormulas;
class UtilityModule {
    constructor(request, locale, workSheetRowIndex, workSheetColIndex, wsGeneral, refectoryRegionId, feeTypeRules) {
        this._filter = request.filter;
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch)?.value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
        this._daysTotal = Math.round(this._options.endDate.diff(this._options.startDate, "days").toObject().days);
        this._offsetDayCount = this._options.startDate.minus({ days: 1 }).day - 25;
        this._workSheetRowIndex = workSheetRowIndex;
        this._workSheetColIndex = workSheetColIndex;
        this._wsGeneral = wsGeneral;
        this._refectoryRegionId = refectoryRegionId;
        this._feeTypeRule = feeTypeRules;
    }
    getColumnLetter(index) {
        const ACode = "A".charCodeAt(0);
        const letters = [];
        while (index >= 0) {
            const remainder = index % 26;
            letters.unshift(String.fromCharCode(ACode + remainder));
            index = Math.floor(index / 26) - 1;
        }
        return letters.join("");
    }
    setCell(ws, row, col, value, border, font, alignment) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = alignment ?? {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = font ?? { bold: true };
        c.border = border;
        return c;
    }
    addInitCell(ws, addressStr, value, border, font, alignment) {
        const c = ws.getCell(addressStr);
        c.value = value;
        c.alignment = alignment ?? {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = font ?? { bold: true };
        c.border = border;
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        const c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
    isMonthContainEmploymentStart(user) {
        if (user.employmentStartUtc && luxon_1.Interval.fromDateTimes(this._options.startDate, this._options.endDate).contains(luxon_1.DateTime.fromJSDate(user.employmentStartUtc))) {
            return luxon_1.DateTime.fromJSDate(user.employmentStartUtc).toFormat("MM/dd/yyyy");
        }
        else {
            return "";
        }
    }
    fillTheSpaces(dayDiff) {
        const values = [];
        for (let x = 0; x < dayDiff - 1; x++) {
            values.push({ value: "" });
        }
        return values;
    }
    applyWorkingHolidayRule(vacation) {
        switch (vacation.type) {
            case app_enums_1.enums.VacationType.National:
                return { value: "ÇGT" };
            case app_enums_1.enums.VacationType.Religious:
                return { value: "ÇBT" };
            default:
                return { value: "ÇA" };
        }
    }
    applyHolidayRule(vacation) {
        switch (vacation.type) {
            case app_enums_1.enums.VacationType.National:
                return { value: "GT" };
            case app_enums_1.enums.VacationType.Religious:
                return { value: "BT" };
            default:
                return { value: "A" };
        }
    }
    applyWorkingWeekendsRule() {
        return { value: "ÇHT" };
    }
    applyWorkingDayRule(firstEntry, lastExit, firstInLastOut) {
        if (firstEntry || lastExit) {
            return { value: firstInLastOut };
        }
        else {
            return { value: firstInLastOut, backgroundColorCode: "FFFD0F0F" };
        }
    }
    generateDataQuery() {
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: this._options.organizationId,
            requesterUserId: this._options.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: this._options.userGroups,
                organizationUnitIds: this._options.organizationUnitIds,
                userIds: this._options.userIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
            },
            requiredOrganizationUnitWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            requiredOrganizationWidePermissions: ["e:u", "u:b", "g:r", "i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        let { query, bindings } = generateEmployeeDataQuery(this._options.organizationId, userFilter, this._options.startDate.toSQL(), this._options.startDate.minus({ days: this._offsetDayCount }).toSQL(), this._options.endDate.toSQL(), this._options.endDate.set({ day: 25 }).toSQL());
        return { query, bindings };
    }
    extraWorkRule(params) {
        const extraLeave = params.extraLeaveHours.reduce((prev, current) => (prev += current));
        if (extraLeave) {
            if (params.vacation) {
                params.totalExtraWorks.vacations += extraLeave;
                return this.applyWorkingHolidayRule(params.vacation);
            }
            else if (params.workPlanDuration <= 0) {
                params.totalExtraWorks.weekend += extraLeave > 7.5 ? 7.5 : extraLeave;
                return this.applyWorkingWeekendsRule();
            }
            else {
                params.totalExtraWorks.weekDay += extraLeave;
                return this.applyWorkingDayRule(params.firstEntry, params.lastExit, this.firstAndLastEntryFormatter(params.firstEntry, params.lastExit));
            }
        }
    }
    firstAndLastEntryFormatter(firstEntry, lastExit) {
        let firstInLastExitFormat = firstEntry ? (0, moment_1.default)(firstEntry).format("HH:mm") + "\n" : "__ :__" + "\n";
        firstInLastExitFormat += lastExit ? (0, moment_1.default)(lastExit).format("HH:mm") + "\n" : "__ :__";
        return firstInLastExitFormat;
    }
    set workSheetRowIndex(value) {
        this._workSheetRowIndex = value;
    }
    set workSheetColIndex(value) {
        this._workSheetColIndex = value;
    }
    get workSheetRowIndex() {
        return this._workSheetRowIndex;
    }
    get workSheetColIndex() {
        return this._workSheetColIndex;
    }
    get wsGeneral() {
        return this._wsGeneral;
    }
    set wsGeneral(value) {
        this._wsGeneral = value;
    }
}
exports.UtilityModule = UtilityModule;
class ADM_UtilityModule extends UtilityModule {
    constructor(request, locale, workSheetRowIndex, workSheetColIndex, wsGeneral, refectoryRegionId, feeTypeRules) {
        super(request, locale, workSheetRowIndex, workSheetColIndex, wsGeneral, refectoryRegionId, feeTypeRules);
        this._permissionTypeAbbrevations = constants_1.ADMTallyReportConstants["PERMISSION_TYPE_ABBREVATIONS"];
    }
    applyTallyReportRules(data, totalExtraWorks, extraLeaveHours, vacation, permissionTypeId) {
        const extraDayData = data.ex;
        const firstEntry = extraDayData.s;
        const lastExit = extraDayData.e;
        const workplanDuration = data.s.wd;
        if (extraLeaveHours?.length > 0) {
            const extraLeave = extraLeaveHours.reduce((prev, current) => (prev += current));
            if (extraLeave) {
                let firstInLastExitFormat = firstEntry ? (0, moment_1.default)(firstEntry).format("HH:mm") + "\n" : "__ :__" + "\n";
                firstInLastExitFormat += lastExit ? (0, moment_1.default)(lastExit).format("HH:mm") + "\n" : "__ :__";
                if (vacation) {
                    totalExtraWorks.vacations += extraLeave;
                    return this.applyWorkingHolidayRule(vacation);
                }
                else if (workplanDuration <= 0) {
                    totalExtraWorks.weekend += extraLeave > 7.5 ? 7.5 : extraLeave;
                    return this.applyWorkingWeekendsRule();
                }
                else {
                    totalExtraWorks.weekDay += extraLeave;
                    return this.applyWorkingDayRule(firstEntry, lastExit, firstInLastExitFormat);
                }
            }
        }
        else if (vacation) {
            if (permissionTypeId && constants_1.ADMTallyReportConstants.PRIORITIZED_LEAVES_OVER_HOLIDAYS.includes(permissionTypeId)) {
                return this.applyPermission(permissionTypeId);
            }
            else {
                return this.applyHolidayRule(vacation);
            }
        }
        else if (permissionTypeId) {
            return this.applyPermission(permissionTypeId);
        }
        else {
            if (workplanDuration > 0) {
                const filterWorkingWorkPlan = data.ws.filter((elem) => elem.wt === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours);
                const expectedStartTime = filterWorkingWorkPlan[0].s;
                const expectedEndTime = filterWorkingWorkPlan[filterWorkingWorkPlan.length - 1].e;
                let expectedWorkingRange = expectedStartTime ? (0, moment_1.default)(expectedStartTime).format("HH:mm") + "\n" : "__ :__" + "\n";
                expectedWorkingRange += expectedEndTime ? (0, moment_1.default)(expectedEndTime).format("HH:mm") + "\n" : "__ :__";
                return this.applyWorkingDayRule(firstEntry, lastExit, expectedWorkingRange);
            }
            else {
                if (vacation) {
                    return this.applyHolidayRule(vacation);
                }
                else {
                    return { value: "HT" };
                }
            }
        }
    }
    applyPermission(typeId) {
        return { value: this._permissionTypeAbbrevations[typeId] ?? "İznin kısaltması bulunamadı." };
    }
    addPermissionDescriptionsForAbbrv() {
        let colIndex = 2;
        this.setCell(this.wsGeneral, this.workSheetRowIndex + 1, colIndex++, constants_1.ADMTallyReportConstants.PERMISSION_DESCRIPTIONS.join("\n\n"), generator_1.ReportGenerator.Constants.Styling.AllThin, undefined, {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        });
        this.wsGeneral.mergeCells(this.workSheetRowIndex + 1, 2, this.workSheetRowIndex + 32, 13);
    }
    async applyFeeTypeRules(user, trx) {
        const feeCounts = [];
        const employmentEndUtc = luxon_1.DateTime.fromJSDate(user.employmentEndUtc);
        let feeCount = user.employmentEndUtc && luxon_1.DateTime.now().hasSame(employmentEndUtc, "month") && luxon_1.DateTime.now().hasSame(employmentEndUtc, "year")
            ? Math.round(employmentEndUtc.diff(this._options.startDate, "days").toObject().days)
            : this._daysTotal;
        let copyingBackWorkPlanDefinitionCounter = 0;
        if (luxon_1.DateTime.now() < this._options.endDate) {
            for (let i = user.monthlyData.length; i < this._daysTotal + this._offsetDayCount; i++) {
                if (copyingBackWorkPlanDefinitionCounter === 7) {
                    copyingBackWorkPlanDefinitionCounter = 0;
                }
                const data = user.monthlyData[user.monthlyData.length - 7 + copyingBackWorkPlanDefinitionCounter++];
                user.countOfWorkingDays += data?.s.wd > 0 ? 1 : 0;
            }
        }
        for (const feeTypeRule of constants_1.ADMFeeTypeRules) {
            let feeCountToBeCalculated = feeCount;
            const rule = feeTypeRule.rules.find((rule) => {
                if (rule.extensionFieldRules.every((extField) => {
                    return extField.validValues.includes(user.extensionFields[extField.key]);
                })) {
                    return rule;
                }
            });
            if (rule) {
                if (feeTypeRule.type === constants_1.feeType.YEMEK_UCRETI_NET || feeTypeRule.type === constants_1.feeType.YEMEK_UCRETI_NET_YEMEKHANE) {
                    feeCountToBeCalculated = await rule.calculate(user, feeCountToBeCalculated, this._options.organizationId, this._options.startDate.toSQL(), this._options.endDate.toSQL(), trx);
                }
                else {
                    feeCountToBeCalculated = await rule.calculate(user, feeCountToBeCalculated);
                }
                feeCounts.push(feeCountToBeCalculated > 0 ? feeCountToBeCalculated : 0);
            }
            else {
                feeCounts.push(0);
            }
        }
        return feeCounts;
    }
}
exports.ADM_UtilityModule = ADM_UtilityModule;
class GDZ_UtilityModule extends UtilityModule {
    constructor(request, locale, workSheetRowIndex, workSheetColIndex, wsGeneral, refectoryRegionId, feeTypeRules) {
        super(request, locale, workSheetRowIndex, workSheetColIndex, wsGeneral, refectoryRegionId, feeTypeRules);
        this._permissionTypeAbbrevations = constants_1.GEDIZTallyReportConstants["PERMISSION_TYPE_ABBREVATIONS"];
    }
    async applyFeeTypeRules(user, trx) {
        const feeCounts = [];
        const employmentEndUtc = luxon_1.DateTime.fromJSDate(user.employmentEndUtc);
        let feeCount = user.employmentEndUtc && luxon_1.DateTime.now().hasSame(employmentEndUtc, "month") && luxon_1.DateTime.now().hasSame(employmentEndUtc, "year")
            ? Math.round(employmentEndUtc.diff(this._options.startDate, "days").toObject().days)
            : this._daysTotal;
        let copyingBackWorkPlanDefinitionCounter = 0;
        if (luxon_1.DateTime.now() < this._options.endDate) {
            for (let i = user.monthlyData.length; i < this._daysTotal + this._offsetDayCount; i++) {
                if (copyingBackWorkPlanDefinitionCounter === 7) {
                    copyingBackWorkPlanDefinitionCounter = 0;
                }
                const data = user.monthlyData[user.monthlyData.length - 7 + copyingBackWorkPlanDefinitionCounter++];
                user.countOfWorkingDays += data?.s.wd > 0 ? 1 : 0;
            }
        }
        for (const feeTypeRule of constants_1.GDZFeeTypeRules) {
            let feeCountToBeCalculated = feeCount;
            const rule = feeTypeRule.rules.find((rule) => {
                if (rule.extensionFieldRules.every((extField) => {
                    return extField.validValues.includes(user.extensionFields[extField.key]);
                })) {
                    return rule;
                }
            });
            if (rule) {
                if (feeTypeRule.type === constants_1.feeType.YEMEK_UCRETI_NET) {
                    feeCountToBeCalculated = await rule.calculate(user, feeCountToBeCalculated, this._options.organizationId, this._options.startDate.toSQL(), this._options.endDate.toSQL(), trx);
                }
                else {
                    feeCountToBeCalculated = await rule.calculate(user, feeCountToBeCalculated);
                }
                feeCounts.push(feeCountToBeCalculated > 0 ? feeCountToBeCalculated : 0);
            }
            else {
                feeCounts.push(0);
            }
        }
        return feeCounts;
    }
    applyTallyReportRules(data, totalExtraWorks, extensionFields, extraLeaveHours, vacation, permissionTypeId) {
        const userWorkPlanRule = extensionFields ? extensionFields["org_workplan_rule"] : "";
        switch (userWorkPlanRule) {
            case "NOR1":
            case "NOR2":
                return this.applyTallyReportRulesForFirstWorkPlanRule(data, totalExtraWorks, extraLeaveHours, vacation, permissionTypeId);
            case "VAR1":
            case "VAR2":
                return this.applyTallyReportRulesForSecondWorkPlanRule(data, totalExtraWorks, extraLeaveHours, vacation, permissionTypeId);
            default:
                return this.applyTallyReportsRulesDefault(data, totalExtraWorks, extraLeaveHours, vacation, permissionTypeId);
        }
    }
    applyTallyReportsRulesDefault(data, totalExtraWorks, extraLeaveHours, vacation, permissionTypeId) {
        const extraDayData = data.ex;
        const firstEntry = extraDayData.s;
        const lastExit = extraDayData.e;
        const workPlanDuration = data.s.wd;
        if (extraLeaveHours?.length > 0) {
            return this.extraWorkRule({ extraLeaveHours, totalExtraWorks, workPlanDuration: workPlanDuration, firstEntry, lastExit, vacation });
        }
        else if (permissionTypeId) {
            return this.applyPermission(permissionTypeId);
        }
        else {
            if (workPlanDuration > 0) {
                return this.workingRule({ workPlanDuration, permissionTypeId, firstEntry, lastExit, workPlanDurationSegment: data.ws });
            }
            else {
                if (vacation) {
                    return this.applyHolidayRule(vacation);
                }
                else {
                    return { value: "HT" };
                }
            }
        }
    }
    applyTallyReportRulesForFirstWorkPlanRule(data, totalExtraWorks, extraLeaveHours, vacation, permissionTypeId) {
        const extraDayData = data.ex;
        const firstEntry = extraDayData.s;
        const lastExit = extraDayData.e;
        const workPlanDuration = data.s.wd;
        const dayOfWeek = (0, moment_1.default)(data.d).isoWeekday();
        if (extraLeaveHours?.length > 0) {
            return this.extraWorkRule({ extraLeaveHours, totalExtraWorks, workPlanDuration: workPlanDuration, firstEntry, lastExit, vacation });
        }
        else if (permissionTypeId) {
            return this.highPriorityPermissionRule({ permissionTypeId, workPlanDuration, dayOfWeek, vacation });
        }
        else {
            if (workPlanDuration > 0) {
                return this.workingRule({ workPlanDuration, permissionTypeId, firstEntry, lastExit, workPlanDurationSegment: data.ws });
            }
            else {
                if (vacation) {
                    return this.applyHolidayRule(vacation);
                }
                else {
                    return { value: "HT" };
                }
            }
        }
    }
    applyTallyReportRulesForSecondWorkPlanRule(data, totalExtraWorks, extraLeaveHours, vacation, permissionTypeId) {
        const extraDayData = data.ex;
        const firstEntry = extraDayData.s;
        const lastExit = extraDayData.e;
        const workPlanDuration = data.s.wd;
        const dayOfWeek = (0, moment_1.default)(data.d).isoWeekday();
        if (extraLeaveHours?.length > 0) {
            return this.extraWorkRule({ extraLeaveHours, totalExtraWorks, workPlanDuration: workPlanDuration, firstEntry, lastExit, vacation });
        }
        else if (permissionTypeId) {
            return this.highPriorityPermissionRule({ permissionTypeId, workPlanDuration, dayOfWeek, vacation });
        }
        else {
            if (workPlanDuration > 0) {
                return this.workingRule({ workPlanDuration, permissionTypeId, firstEntry, lastExit, workPlanDurationSegment: data.ws });
            }
            else {
                if (vacation) {
                    return this.applyHolidayRule(vacation);
                }
                else {
                    return { value: "HT" };
                }
            }
        }
    }
    highPriorityPermissionRule(params) {
        if (constants_1.GEDIZTallyReportConstants.LEAVE_TYPES_WITH_BUSSINESS_LOGIC.GROUP_1.includes(params.permissionTypeId)) {
            if (params.workPlanDuration === 0 && params.dayOfWeek === 6 && !params.vacation) {
                return this.applyPermission(params.permissionTypeId);
            }
            else if (params.vacation) {
                return this.applyHolidayRule(params.vacation);
            }
            else if (params.workPlanDuration === 0) {
                return { value: "HT" };
            }
            else {
                return this.applyPermission(params.permissionTypeId);
            }
        }
        else if (constants_1.GEDIZTallyReportConstants.LEAVE_TYPES_WITH_BUSSINESS_LOGIC.GROUP_2.includes(params.permissionTypeId)) {
            return this.applyPermission(params.permissionTypeId);
        }
    }
    highPriorityPermissionRuleSecond(params) {
        if (constants_1.GEDIZTallyReportConstants.LEAVE_TYPES_WITH_BUSSINESS_LOGIC.GROUP_1.includes(params.permissionTypeId) ||
            constants_1.GEDIZTallyReportConstants.LEAVE_TYPES_WITH_BUSSINESS_LOGIC.GROUP_3.includes(params.permissionTypeId)) {
            if (params.vacation) {
                return this.applyHolidayRule(params.vacation);
            }
            else {
                return this.applyPermission(params.permissionTypeId);
            }
        }
        else if (constants_1.GEDIZTallyReportConstants.LEAVE_TYPES_WITH_BUSSINESS_LOGIC.GROUP_2.includes(params.permissionTypeId)) {
            return this.applyPermission(params.permissionTypeId);
        }
    }
    workingRule(params) {
        if (params.workPlanDuration > 0) {
            if (params.permissionTypeId) {
                return this.applyPermission(params.permissionTypeId);
            }
            else {
                const filterWorkingWorkPlan = params.workPlanDurationSegment.filter((elem) => elem.wt === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours);
                const expectedStartTime = filterWorkingWorkPlan[0].s;
                const expectedEndTime = filterWorkingWorkPlan[filterWorkingWorkPlan.length - 1].e;
                return this.applyWorkingDayRule(params.firstEntry, params.lastExit, this.firstAndLastEntryFormatter(expectedStartTime, expectedEndTime));
            }
        }
    }
    applyPermission(typeId) {
        return {
            value: this._permissionTypeAbbrevations[typeId]?.abbr ?? "İznin kısaltması bulunamadı.",
            backgroundColorCode: this._permissionTypeAbbrevations[typeId]?.colorCode ?? "FFFFFFFF",
        };
    }
}
exports.GDZ_UtilityModule = GDZ_UtilityModule;
GDZ_UtilityModule.workPlanRuleFirstGroup = ["NOR1", "NOR2"];
GDZ_UtilityModule.workPlanRuleSecondGroup = ["VAR1", "VAR2"];
