"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ADMMonthlyTallyCustomReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const i18n_1 = __importDefault(require("i18n"));
const sharp_1 = __importDefault(require("sharp"));
const uuid_1 = __importDefault(require("uuid"));
const luxon_1 = require("luxon");
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_manager_1 = require("../../../dal/dal.manager");
const constants_1 = require("./constants");
const app_logs_1 = require("../../../app.logs");
const utils_1 = require("./utils");
class ADMMonthlyTallyCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._wb = new exceljs_1.default.Workbook();
        this._wsGeneral = this._wb.addWorksheet("ADM AYLIK PUANTAJ RAPORU", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsFilterSheet = this._wb.addWorksheet("Uygulanan Filtreler", {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._options = {
            startDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").startOf("month"),
            endDate: luxon_1.DateTime.fromFormat(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "yyyy-MM").endOf("month"),
            organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
            userGroups: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
            organizationUnitHierarchically: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
            userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            locale: this._locale,
        };
        this._daysTotal = Math.round(this._options.endDate.diff(this._options.startDate, "days").toObject().days);
        this._offsetDayCount = this._options.startDate.minus({ days: 1 }).day - 25;
        this._admUtilityModule = new utils_1.ADM_UtilityModule(request, locale, 1, 1, this._wsGeneral, constants_1.ADMRefectoryRegionId, constants_1.ADMFeeTypeRules);
    }
    async generateReport() {
        this._vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(this._options.organizationId);
        this._filePath = await this.generateADMMonthlyTallyCustomReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            customReportName: "ADM Aylık Puantaj Raporu",
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
        };
    }
    async generateADMMonthlyTallyCustomReport() {
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._wsFilterSheet.properties.defaultRowHeight = 15;
        const logo = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoFile(this._request.organizationId);
        if (logo) {
            const ADMLogoBase64 = "data:image/png;base64," + (await (0, sharp_1.default)(Buffer.from(logo), { failOnError: false }).toFormat("png").toBuffer()).toString("base64");
            const imageId1 = this._wb.addImage({
                base64: ADMLogoBase64,
                extension: "png",
            });
            this._wsGeneral.addImage(imageId1, {
                tl: { col: 1, row: 0 },
                ext: { width: 120, height: 60 },
            });
        }
        const filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            const organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._options.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(", "));
            if (this._options.organizationUnitHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroups && this._options.userGroups.length > 0) {
            let userGroupNames = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroups)).map((elem) => elem.name).join(", ");
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames);
        }
        if (this._options.userIds && this._options.userIds.length > 0) {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                let userNames = await dal_manager_1.dbManager.accessUser.getBasicUserInfoPg({ organizationId: this._request.organizationId, userIds: this._options.userIds, trx });
                filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.map((elem) => elem.fullname).join(","));
            });
        }
        this.addFilterToSheet(this._wsFilterSheet, 1, 1, filters.join("\r\n"));
        this.initGeneralSheet(this._wsGeneral);
        const requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._options.organizationId, [this._options.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            let { query, bindings } = this._admUtilityModule.generateDataQuery();
            await trx.query(query, bindings);
            let { rows } = await trx.query("FETCH 100 FROM employee_day_cursor");
            while (rows.length > 0) {
                await this.addGeneralRow(rows, trx);
                rows = (await trx.query("FETCH 100 FROM employee_day_cursor")).rows;
            }
            await trx.query(`CLOSE employee_day_cursor;`);
            this._admUtilityModule.addPermissionDescriptionsForAbbrv();
        });
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw) {
        let font = { name: "Rockwell Extra Bold", size: 14, bold: true, color: { argb: "FFFF6600" } };
        wsRaw.mergeCells("A1:C3");
        wsRaw.getCell("C3").border = generator_1.ReportGenerator.Constants.Styling.AllThin;
        wsRaw.mergeCells("A4:C4");
        wsRaw.getCell("A4").border = generator_1.ReportGenerator.Constants.Styling.AllThin;
        wsRaw.mergeCells("D4:M4");
        wsRaw.getCell("D4").border = generator_1.ReportGenerator.Constants.Styling.AllThin;
        this._admUtilityModule.addInitCell(wsRaw, "N4", i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEE_NUMBER", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.mergeCells("N4:Y4");
        wsRaw.mergeCells("Z4:AA4");
        this._admUtilityModule.addInitCell(wsRaw, "Z4", i18n_1.default.__({ phrase: "REPORT.MONTH", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.mergeCells("AB4:AF4");
        this._admUtilityModule.addInitCell(wsRaw, "AB4", this._options.startDate.month, generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.mergeCells("AG4:AJ4");
        this._admUtilityModule.addInitCell(wsRaw, "AG4", this._options.startDate.year, generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        this._admUtilityModule.addInitCell(wsRaw, "A6", "NO.", generator_1.ReportGenerator.Constants.Styling.AllThin, undefined, {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
            textRotation: 90,
        });
        this._admUtilityModule.setCell(wsRaw, 6, 2, "Sicil Numarası", generator_1.ReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFFCD5B4" },
        };
        this._admUtilityModule.setCell(wsRaw, 6, 3, "ADI VE SOYADI", generator_1.ReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFFCD5B4" },
        };
        this._admUtilityModule.setCell(wsRaw, 6, 4, "ÇALIŞAN ALT GRUBU", generator_1.ReportGenerator.Constants.Styling.AllThin).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFFCD5B4" },
        };
        wsRaw.getCell("E6").border = generator_1.ReportGenerator.Constants.Styling.AllThin;
        this._admUtilityModule.workSheetColIndex = 6;
        this._admUtilityModule.workSheetRowIndex = 6;
        for (let x = 26; x <= this._options.startDate.minus({ day: 1 }).day; x++) {
            this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex++, x, generator_1.ReportGenerator.Constants.Styling.AllThin).fill = {
                type: "gradient",
                gradient: "angle",
                degree: 90,
                stops: [
                    { position: 0, color: { argb: "FFFDEADB" } },
                    { position: 1, color: { argb: "FF3DA8C1" } },
                ],
            };
        }
        for (let i = 1; i <= this._daysTotal; i++) {
            this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex++, i, generator_1.ReportGenerator.Constants.Styling.AllThin).fill = {
                type: "gradient",
                gradient: "angle",
                degree: 90,
                stops: [
                    { position: 0, color: { argb: "FFFDEADB" } },
                    { position: 1, color: { argb: "FFFF6600" } },
                ],
            };
        }
        wsRaw.mergeCells("AK4:" + this._admUtilityModule.getColumnLetter(this._admUtilityModule.workSheetColIndex - 1) + 4);
        this._admUtilityModule.addInitCell(wsRaw, "AK4", "", generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.mergeCells("D5:" + this._admUtilityModule.getColumnLetter(this._admUtilityModule.workSheetColIndex - 1) + 5);
        this._admUtilityModule.addInitCell(wsRaw, "D5", "", generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        this._admUtilityModule.workSheetColIndex += 1;
        wsRaw.getCell("AJ5").border = generator_1.ReportGenerator.Constants.Styling.AllThin;
        const featureAlignment = {
            horizontal: "center",
            vertical: "bottom",
            wrapText: true,
            textRotation: 90,
        };
        wsRaw.mergeCells(this._admUtilityModule.workSheetRowIndex - 2, this._admUtilityModule.workSheetColIndex, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex);
        this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex, "NORMAL ÇALIŞMA GÜNÜ", generator_1.ReportGenerator.Constants.Styling.AllThin, undefined, featureAlignment).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFFFFF00" },
        };
        this._admUtilityModule.workSheetColIndex += 1;
        const permissionAndHolidayFont = { name: "Times New Roman", size: 7, bold: true };
        for (const name of constants_1.ADMTallyReportConstants.PERMISSION_NAMES) {
            wsRaw.mergeCells(this._admUtilityModule.workSheetRowIndex - 2, this._admUtilityModule.workSheetColIndex, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex);
            this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex, name, generator_1.ReportGenerator.Constants.Styling.AllThin, permissionAndHolidayFont, featureAlignment).fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "FFFDE9D9" },
            };
            this._admUtilityModule.workSheetColIndex += 1;
        }
        for (const name of constants_1.ADMTallyReportConstants.HOLIDAY_NAMES) {
            wsRaw.mergeCells(this._admUtilityModule.workSheetRowIndex - 2, this._admUtilityModule.workSheetColIndex, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex);
            this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex, name, generator_1.ReportGenerator.Constants.Styling.AllThin, permissionAndHolidayFont, featureAlignment).fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "FFFABF8F" },
            };
            this._admUtilityModule.workSheetColIndex += 1;
        }
        wsRaw.mergeCells(this._admUtilityModule.workSheetRowIndex - 2, this._admUtilityModule.workSheetColIndex, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex);
        this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex++, "TOPLAM GÜN", generator_1.ReportGenerator.Constants.Styling.AllThin, { name: "Times New Roman", size: 7, bold: true }, {
            horizontal: "center",
            vertical: "bottom",
            wrapText: true,
            textRotation: 90,
        }).fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFFFFF00" },
        };
        for (const extraWork of constants_1.ADMTallyReportConstants.EXTRA_WORK_NAMES) {
            wsRaw.mergeCells(this._admUtilityModule.workSheetRowIndex - 2, this._admUtilityModule.workSheetColIndex, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex);
            this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex, extraWork, generator_1.ReportGenerator.Constants.Styling.AllThin, { name: "Times New Roman", size: 10, bold: true }, featureAlignment).fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: "FFE26B0A" },
            };
            this._admUtilityModule.workSheetColIndex += 1;
        }
        for (const name of constants_1.ADMFeeTypeRules.map((elem) => elem.name)) {
            wsRaw.mergeCells(this._admUtilityModule.workSheetRowIndex - 2, this._admUtilityModule.workSheetColIndex, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex);
            this._admUtilityModule.setCell(wsRaw, this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex, name, generator_1.ReportGenerator.Constants.Styling.AllThin, permissionAndHolidayFont, featureAlignment);
            this._admUtilityModule.workSheetColIndex += 1;
        }
        this._admUtilityModule.workSheetColIndex -= 1;
        wsRaw.mergeCells("D1:" + this._admUtilityModule.getColumnLetter(this._admUtilityModule.workSheetColIndex) + 2);
        wsRaw.getCell("D1").fill = {
            type: "pattern",
            pattern: "solid",
            fgColor: { argb: "FFFDE9D9" },
        };
        this._admUtilityModule.addInitCell(wsRaw, "D1", i18n_1.default.__({ phrase: "CUSTOM-REPORT.GEDIZ.TALLY_RECORD", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, Object.assign({}, font, { color: { argb: "FF974706" } }));
        wsRaw.mergeCells("D3:" + this._admUtilityModule.getColumnLetter(this._admUtilityModule.workSheetColIndex - 1) + 3);
        this._admUtilityModule.addInitCell(wsRaw, "D3", "ADM ELEKTRİK DAĞITIM A.Ş ...............MÜDÜRLÜĞÜ/...............YÖNETİCİLİĞİ", generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.getCell(this._admUtilityModule.getColumnLetter(this._admUtilityModule.workSheetColIndex) + 3).border = generator_1.ReportGenerator.Constants.Styling.AllThin;
        this._admUtilityModule.addInitCell(wsRaw, this._admUtilityModule.getColumnLetter(this._admUtilityModule.workSheetColIndex) + 4, "İmza", generator_1.ReportGenerator.Constants.Styling.AllThin, font);
        wsRaw.mergeCells(4, this._admUtilityModule.workSheetColIndex + 1, 6, this._admUtilityModule.workSheetColIndex + 1);
        this._admUtilityModule.workSheetColIndex += 1;
        this._admUtilityModule.workSheetRowIndex += 1;
        wsRaw.findRow(4).height = 44.5;
        wsRaw.findRow(5).height = 25;
        wsRaw.findRow(6).height = 48.76;
        wsRaw.columns.forEach((column, index) => {
            if (index === 1) {
                column.width = 13;
            }
            else if (index === 2 || index === 3) {
                column.width = 22.14;
            }
            else {
                column.width = 12;
            }
        });
        wsRaw.views = [{ state: "frozen", ySplit: 6 }];
    }
    async addGeneralRow(userInfo, trx) {
        this._admUtilityModule.workSheetColIndex = 1;
        for (const user of userInfo) {
            const totalExtraWorks = {
                weekDay: 0,
                weekend: 0,
                vacations: 0,
            };
            const userOffsetPermissionRangeAndType = user.permissions?.map((elem) => {
                return {
                    typeId: elem.typeId,
                    range: luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(elem.offsettingStartDate).startOf("day"), luxon_1.DateTime.fromISO(elem.offsettingEndDate).plus({ day: 1 }).startOf("day")),
                };
            });
            const userPermissionRanges = user.permissions?.map((elem) => {
                return {
                    typeId: elem.typeId,
                    range: luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromISO(elem.startDate), luxon_1.DateTime.fromISO(elem.endDate).plus({ day: 1 }).startOf("day")),
                };
            });
            const excelValues = [
                { value: this._admUtilityModule.workSheetRowIndex - 6 },
                { value: user.uniqueId },
                { value: user.fullname },
                { value: user.extensionFields ? constants_1.ADM_EMPLOYEE_SUB_GROUPS[user.extensionFields["org_employee_subgroup"]] : "" },
                { value: this._admUtilityModule.isMonthContainEmploymentStart(user) },
                ...this._admUtilityModule.fillTheSpaces(luxon_1.DateTime.fromISO(user.monthlyData[0].d)
                    .diff(this._options.startDate.minus({ days: this._offsetDayCount }), "days")
                    .toObject().days + 1),
            ];
            user.monthlyData.forEach((data, index) => {
                if (index > 1) {
                    excelValues.push(...this._admUtilityModule.fillTheSpaces(luxon_1.DateTime.fromISO(data.d)
                        .diff(luxon_1.DateTime.fromISO(user.monthlyData[index - 1].d), "days")
                        .toObject().days));
                }
                let permissionTypeId = undefined;
                let extraLeaveHours = [];
                const employeeDayPermissions = data.p;
                const vacationExistency = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).contains(luxon_1.DateTime.fromJSDate((0, moment_1.default)(data.d).toDate())));
                if (employeeDayPermissions?.length > 0) {
                    if (userOffsetPermissionRangeAndType?.find((elem) => elem.range.contains(luxon_1.DateTime.fromISO(data.d)))) {
                        permissionTypeId = employeeDayPermissions[employeeDayPermissions.length - 1].pt;
                        extraLeaveHours = data.p
                            .filter((elem) => elem.pt === constants_1.ADMTallyReportConstants.EXTRA_PERMISSION_TYPE_ID)
                            .map((item) => (0, moment_1.default)(item.r.e).diff((0, moment_1.default)(item.r.s), "hour"));
                    }
                    else {
                        excelValues.push({ value: "Kullanılan izin bu ayın raporunda işlenmeyecektir." });
                        return;
                    }
                }
                excelValues.push(this._admUtilityModule.applyTallyReportRules(data, totalExtraWorks, extraLeaveHours, vacationExistency, permissionTypeId));
            });
            excelValues.push(...this.copyOffsetDaysFromPastDays(user.monthlyData[user.monthlyData.length - 1].d, user.monthlyData, totalExtraWorks, userPermissionRanges));
            if (user.employmentEndUtc && luxon_1.Interval.fromDateTimes(this._options.startDate, this._options.startDate).contains(luxon_1.DateTime.fromJSDate(user.employmentEndUtc))) {
                excelValues.push({ value: luxon_1.DateTime.fromJSDate(user.employmentEndUtc).toFormat("MM/dd/yyyy") });
            }
            else {
                excelValues.push({ value: "" });
            }
            const feeCounts = (await this._admUtilityModule.applyFeeTypeRules(user, trx)).map((elem) => {
                return {
                    value: elem,
                };
            });
            excelValues.push({ value: user.countOfWorkingDays });
            const currentColIndex = excelValues.length;
            Object.entries(constants_1.ADMTallyReportConstants.EXCEL_FORMULA_CONDITIONS).forEach(([key, value]) => {
                if (key === "YILLIK_I") {
                    excelValues.push({
                        value: {
                            formula: "SUMPRODUCT((F" +
                                this._admUtilityModule.workSheetRowIndex +
                                ":" +
                                this._admUtilityModule.getColumnLetter(currentColIndex - 3) +
                                this._admUtilityModule.workSheetRowIndex +
                                `="Y.İ")*1 +` +
                                "(F" +
                                this._admUtilityModule.workSheetRowIndex +
                                ":" +
                                this._admUtilityModule.getColumnLetter(currentColIndex - 3) +
                                this._admUtilityModule.workSheetRowIndex +
                                `="Y(Yarım)")*0.5)`,
                        },
                    });
                }
                else {
                    let beginCell = "";
                    if (key === "CGT" || key === "CHT" || key === "CBT") {
                        beginCell = this._admUtilityModule.getColumnLetter(5 + this._offsetDayCount) + this._admUtilityModule.workSheetRowIndex;
                    }
                    else {
                        beginCell = "F" + this._admUtilityModule.workSheetRowIndex;
                    }
                    const endCell = this._admUtilityModule.getColumnLetter(currentColIndex - 3) + this._admUtilityModule.workSheetRowIndex;
                    excelValues.push({ value: (0, utils_1.generateExcelCountIfFormulas)(beginCell, endCell, value) });
                }
            });
            excelValues.push(...[
                { value: this._daysTotal },
                { value: totalExtraWorks.weekDay },
                { value: totalExtraWorks.weekend },
                { value: totalExtraWorks.vacations },
                ...feeCounts,
                { value: "" },
            ]);
            excelValues.forEach((elem) => Object.assign(elem, {
                border: generator_1.ReportGenerator.Constants.Styling.AllThin,
                alignment: "center",
                bold: true,
            }));
            this.addCellsToRowWithOptions(excelValues, this._admUtilityModule.workSheetColIndex, this._admUtilityModule.workSheetRowIndex, this._wsGeneral);
            this._admUtilityModule.workSheetColIndex = 1;
            this._admUtilityModule.workSheetRowIndex += 1;
        }
    }
    copyOffsetDaysFromPastDays(lastDate, monthlyData, totalExtraWorks, userPermissionRanges) {
        let copyingBackWorkPlanDefinitionCounter = 0;
        let dayCount = 0;
        const cellValues = [];
        if (luxon_1.DateTime.now() < this._options.endDate && monthlyData.length >= 7) {
            for (let i = monthlyData.length; i < this._daysTotal + this._offsetDayCount; i++) {
                dayCount++;
                if (copyingBackWorkPlanDefinitionCounter === 7) {
                    copyingBackWorkPlanDefinitionCounter = 0;
                }
                const data = monthlyData[monthlyData.length - 7 + copyingBackWorkPlanDefinitionCounter++];
                const permission = userPermissionRanges?.find((elem) => elem.range.contains(luxon_1.DateTime.fromISO(lastDate).plus({ days: dayCount })));
                const vacation = this._vacations.items.find((elem) => luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(elem.startDateTime), luxon_1.DateTime.fromJSDate(elem.endDateTime)).contains(luxon_1.DateTime.fromISO(lastDate).plus({ days: dayCount })));
                this._wsGeneral.getCell(this._admUtilityModule.workSheetRowIndex, this._admUtilityModule.workSheetColIndex).fill = {
                    type: "pattern",
                    pattern: "solid",
                    fgColor: { argb: "FF8DB4E2" },
                };
                const cellValue = this._admUtilityModule.applyTallyReportRules(data, totalExtraWorks, undefined, vacation, permission?.typeId);
                if (!cellValue) {
                    cellValues.push({ value: "" });
                }
                else {
                    cellValue.backgroundColorCode = "FF8DB4E2";
                    cellValues.push(cellValue);
                }
            }
            return cellValues;
        }
        else {
            return this._admUtilityModule.fillTheSpaces(this._options.endDate.diff(luxon_1.DateTime.fromISO(lastDate), "days").toObject().days);
        }
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        const columns = [{ width: 45 }, { width: 45 }, { width: 45 }];
        this._admUtilityModule.addFilterCell(ws, startRowIndex, startColIndex, "AYLIK PUANTAJ RAPORU", constants_1.HeaderStyle.border, constants_1.HeaderStyle.font);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        this._admUtilityModule.addFilterCell(ws, startRowIndex + 1, startColIndex, "AİT OLDUĞU AY VE YIL", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleSecond.font);
        this._admUtilityModule.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, this._options.startDate.setLocale(this._locale).toFormat("LLLL yyyy"), constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleThird.font, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        this._admUtilityModule.addFilterCell(ws, startRowIndex + 2, startColIndex, "KURUM", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleSecond.font);
        this._admUtilityModule.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, "ADM", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleThird.font);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        if (filter.length > 0) {
            this._admUtilityModule.addFilterCell(ws, startRowIndex + 3, startColIndex, "UYGULANAN FILTRELER", constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleSecond.font);
            this._admUtilityModule.addFilterCell(ws, startRowIndex + 3, startColIndex + 1, filter, constants_1.HeaderStyleSecond.border, constants_1.HeaderStyleThird.font);
            ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
            let filterlength = filter.split("\r\n").length;
            let row = ws.getRow(startRowIndex + 3);
            row.height = filterlength * 30;
        }
        ws.columns = columns;
        ws.views = [{ state: "frozen", ySplit: 4, activeCell: "A1", showGridLines: false }];
    }
    addCellsToRow(values, colIndex, rowIndex, ws, border) {
        for (let i = 0; i < values.length; i++) {
            this._admUtilityModule.setCell(ws, rowIndex, colIndex++, values[i], border ?? generator_1.ReportGenerator.Constants.Styling.Dotted);
        }
        return colIndex;
    }
}
exports.ADMMonthlyTallyCustomReport = ADMMonthlyTallyCustomReport;
async function generateReport(request, locale) {
    const report = new ADMMonthlyTallyCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
