"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateHook = void 0;
const app_logs_1 = require("../../../app.logs");
const business_hooks_1 = require("../../../business/business.hooks");
const dal_manager_1 = require("../../../dal/dal.manager");
const restapi_1 = require("../../../lib/es/models/restapi");
const app_enums_1 = require("../../../app.enums");
const crm_queue_1 = require("../../common/crm/crm-queue");
const taskqueue_1 = require("../../../utils/taskqueue");
const keylocak_hooks_1 = require("../../common/keycloak/keylocak-hooks");
const crm_utils_1 = require("./crm/crm-utils");
const notificationQ = new taskqueue_1.TaskQueue(2000, "EnerjiSA Dagitim CRM");
function CreateHook() {
    return new EnerjiSADagitimHook();
}
exports.CreateHook = CreateHook;
const emailRegex = /^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$/;
const organizationId = "cf1a899a-bb3f-4379-9b25-4c6d7a89a80f";
const keyCloakSecretId = "1eeadb05-5411-4997-b1a2-8a093e57de78";
const crmSecretId = "d498a852-982a-49da-937d-d97bd5742864";
class EnerjiSADagitimHook extends business_hooks_1.OrganizationHookModels.IArmonHook {
    constructor() {
        super();
        this.beforeUserInsert = async function (params) {
            if (params.identity.organizationProfile.email) {
                const regexCheck = emailRegex.test(params.identity.organizationProfile.email.toLowerCase());
                if (!regexCheck) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.BAD_REQUEST,
                        errorMessage: `Email address is not valid! Please check email address provided!`,
                    };
                }
                const searchResult = await dal_manager_1.dbManager.accessIdentity.searchIdentityExact(organizationId, {
                    userOrganizationProfile: {
                        email: params.identity.organizationProfile.email,
                    },
                });
                if (searchResult.items.length) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.CONFLICT,
                        errorMessage: `There is already a${searchResult.items[0].isDisabled ? " passive" : "n active"} user registered with this mail address, therefore user cannot be added with this email address`,
                    };
                }
            }
            return {
                isValid: true,
                errorMessage: null,
            };
        };
        this.beforeUserUpdate = async function (params) {
            if (params.identity.organizationProfile.email) {
                const regexCheck = emailRegex.test(params.identity.organizationProfile.email.toLowerCase());
                if (!regexCheck) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.BAD_REQUEST,
                        errorMessage: `Email address is not valid! Please check email address provided!`,
                    };
                }
                const searchResult = await dal_manager_1.dbManager.accessIdentity.searchIdentityExact(organizationId, {
                    userOrganizationProfile: {
                        email: params.identity.organizationProfile.email,
                    },
                });
                let duplicateUser = searchResult.items.find((f) => f.id !== params.identity.id);
                if (duplicateUser) {
                    return {
                        isValid: false,
                        errorCode: app_enums_1.enums.HttpStatusCode.CONFLICT,
                        errorMessage: `There is already a${duplicateUser.isDisabled ? "passive" : "n active"} user registered with this mail address, therefore user cannot be updated with this email address`,
                    };
                }
            }
            return {
                isValid: true,
                errorMessage: null,
            };
        };
        this.afterIdentityUpsert = (0, keylocak_hooks_1.getKeyCloakAfterIdentityUpsertHook)(organizationId, keyCloakSecretId);
        this.beforeUpdateOrganizationSettings = async function (settings) {
            if (settings.webRtc?.iceServers?.length || settings.webRtc?.twilio?.enabled) {
                return {
                    isValid: false,
                    errorMessage: "ERRORS.CUSTOM.ENERJISA_WEBRTC",
                };
            }
            else {
                return {
                    isValid: true,
                    errorMessage: null,
                };
            }
        };
        this.afterGetOrganizationSettings = async function (settings) {
            if (settings.webRtc?.iceServers?.length || settings.webRtc?.twilio.enabled) {
                settings.webRtc.iceServers = [];
                settings.webRtc.twilio = {
                    enabled: false,
                    accountSid: null,
                    authToken: null,
                };
            }
        };
        this.sendNotificationOverride = async function (params) {
            if (params.medium === restapi_1.NotificationMedium.Web || params.medium === restapi_1.NotificationMedium.PushNotification) {
                return {
                    isImplemented: false,
                };
            }
            let transformResult = null;
            try {
                transformResult = await (0, crm_utils_1.transformNotificationToCRMParameters)(params.message.o, { instanceId: params.notification.instanceId, medium: params.medium, message: params.message }, params.trx);
            }
            catch (error) {
                app_logs_1.logger.error(`Error while transforming notification parameters to CRM parameters. Notification Instance Id: ${params.notification.instanceId}`);
                app_logs_1.logger.error(error);
                app_logs_1.logger.error(`Notification Message content: ${JSON.stringify(params.message)}`);
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.notification.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.CustomNotificationChannelError,
                        message: `Error while transforming notification parameters to CRM parameters. Error Details: ${error.toString()}`,
                    },
                    trx: params.trx,
                });
                return {
                    isImplemented: true,
                };
            }
            if (!transformResult.isImplemented) {
                return {
                    isImplemented: false,
                };
            }
            else {
                if (!transformResult.transformedParameters.reciever) {
                    app_logs_1.logger.error(`No valid recievers for notification via CRM. Notification Instance Id: ${params.notification.instanceId}`);
                    app_logs_1.logger.error(`Notification Message content: ${JSON.stringify(params.message)}`);
                    await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                        instanceId: params.notification.instanceId,
                        organizationId: params.organizationId,
                        state: 3,
                        actionNote: {
                            errorCode: app_enums_1.enums.ErrorCode.CustomNotificationChannelError,
                            message: `No valid recievers for notification via CRM`,
                        },
                        trx: params.trx,
                    });
                    return {
                        isImplemented: true,
                    };
                }
            }
            if (!transformResult.transformedParameters.campaignCode) {
                app_logs_1.logger.error(`Campaign Code could not be determined for CRM Notification with instance id: ${params.notification.instanceId}`);
                app_logs_1.logger.error(`Notification Message content: ${JSON.stringify(params.message)}`);
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.notification.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.CustomNotificationChannelError,
                        message: `Campaign Code could not be determined for CRM Notification. Please check notification instance data`,
                    },
                    trx: params.trx,
                });
                return {
                    isImplemented: true,
                };
            }
            notificationQ.push(new crm_queue_1.CRMNotificationTask({
                medium: params.medium,
                message: params.message,
                notification: params.notification,
                organizationId: params.organizationId,
            }, transformResult, crmSecretId));
            return {
                isImplemented: true,
            };
        };
    }
}
