"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.ExcelReportWorkSummaryOfEmployeeElimkoCustomReport = void 0;
const dal_manager_1 = require("../../../dal/dal.manager");
const i18n_1 = __importDefault(require("i18n"));
const moment = require("moment");
const uuid_1 = __importDefault(require("uuid"));
const path_1 = __importDefault(require("path"));
const app_config_1 = require("../../../app.config");
const exceljs_1 = __importDefault(require("exceljs"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const generator_1 = require("../../../business/report/generator");
const constants = {
    hsFmtTr: '[<0.000694444]"";[<0.0415][m]"dak";[h]"sa" m"dak"',
    hsFmtEng: '[<0.000694444]"";[<0.0415][m]"min";[h]"h" m"min"',
    date: "dd.mm.yyyy",
    time: "hh:mm:ss",
    momentDate: "DD-MM-YYYY",
};
class ExcelReportWorkSummaryOfEmployeeElimkoCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wsWorkSummaryCurrentRowNumber = 7;
        this._wsAccessLogsCurrentRowNumber = 7;
        this._request = request;
        this._options = request.filter;
        this._emptyReportCheck = true;
        let dateObject = this._options.filters.find((i) => i.type === 8);
        this._userId = this._options.filters.find((i) => i.type === 1).value[0];
        this._dataRange = {
            startDateTime: moment(dateObject.value.startDay, "YYYY-MM-DDTHH:mm:ss").toDate(),
            endDateTime: moment(dateObject.value.endDay, "YYYY-MM-DDTHH:mm:ss").toDate(),
        };
        this._wb = new exceljs_1.default.Workbook();
        this._wb.calcProperties.fullCalcOnLoad = true;
        this._wsWorkSummary = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKING_DURATION", locale }), {
            pageSetup: { paperSize: 9, orientation: "landscape" },
        });
        this._wsAccessLogs = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS", locale }), {
            pageSetup: { paperSize: 9, orientation: "landscape" },
        });
        this._wsPermissions = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.LEAVES", locale }), {
            pageSetup: { paperSize: 9, orientation: "landscape" },
        });
        this._accessLogIds = [];
        this._permissions = [];
        console.log("Generating Employee Personnel Work Report xlsx");
        this.initSheets();
    }
    async initSheets() {
        await this.initWorkSummary();
        this.initAccessLogs();
        this.initPermissions();
    }
    async initWorkSummary() {
        let locale = this._locale;
        let userCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._userId]);
        if (userCaption.length < 1) {
            throw new Error("Not found");
        }
        this._userCaption = userCaption[0].captionLines[0].text[0] + " / " + userCaption[0].captionLines[1].text[0];
        this.addFilterSummaryToSheet(this._wsWorkSummary, 2, 3);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._userId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        const headerValues = [
            i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPECTED_WORK", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.ACTUAL_WORK", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.EXTRA_WORK", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.INCOMPLETE_OVERTIME", locale }),
            i18n_1.default.__({ phrase: "PDF-REPORT.total_work_duration", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.PERMITTED_WORK", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.PUBLİC_HOLİDAY", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.TOTAL_WORK_AND_PERMISSION", locale }),
        ];
        for (let index = 1; index <= headerValues.length; index++) {
            let c = this._wsWorkSummary.getCell(5, index);
            c.value = headerValues[index - 1];
            c.font = {
                bold: true,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "bottom",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thin",
                },
            };
            this._wsWorkSummary.mergeCells(5, index, 6, index);
        }
        this._wsWorkSummary.views = [{ state: "frozen", xSplit: 0, ySplit: 6, activeCell: "A7", showGridLines: false }];
        for (let index = 0; index < headerValues.length; index++) {
            this._wsWorkSummary.columns[index].width = 14;
        }
        this._wsWorkSummary.autoFilter = {
            from: "A5",
            to: "H5",
        };
        this._wsWorkSummary.pageSetup.printTitlesRow = "5:6";
        this._wsWorkSummary.headerFooter.oddHeader =
            "&L" +
                this._userCaption +
                " / " +
                moment(this._options.filters[0].value.startDateTime).format(constants.momentDate) +
                " - " +
                moment(this._options.filters[0].value.startDateTime).format(constants.momentDate) +
                " " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKING_DURATION_REPORT", locale });
        this._wsWorkSummary.headerFooter.oddFooter = " &P / &N";
    }
    initAccessLogs() {
        let locale = this._locale;
        this.addFilterSummaryToSheet(this._wsAccessLogs, 2, 1);
        const headerValues = [
            i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS_ACCESS_POINT", locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.DIRECTION", locale }),
        ];
        for (let index = 1; index <= headerValues.length; index++) {
            let c = this._wsAccessLogs.getCell(5, index);
            c.value = headerValues[index - 1];
            c.font = {
                bold: true,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "bottom",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thin",
                },
            };
            this._wsAccessLogs.mergeCells(5, index, 6, index);
        }
        this._wsAccessLogs.views = [{ state: "frozen", xSplit: 0, ySplit: 6, activeCell: "A7", showGridLines: false }];
        this._wsAccessLogs.columns[0].width = 12;
        this._wsAccessLogs.columns[1].width = 12;
        this._wsAccessLogs.columns[2].width = 48;
        this._wsAccessLogs.columns[3].width = 12;
        this._wsAccessLogs.autoFilter = {
            from: "A5",
            to: "D5",
        };
        this._wsAccessLogs.pageSetup.printTitlesRow = "5:6";
        this._wsAccessLogs.headerFooter.oddHeader =
            "&L" +
                this._userCaption +
                " / " +
                moment(this._dataRange.startDateTime).format(constants.momentDate) +
                " - " +
                moment(this._dataRange.endDateTime).format(constants.momentDate) +
                " " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKING_DURATION_REPORT", locale });
        this._wsAccessLogs.headerFooter.oddFooter = " &P / &N";
    }
    addFilterSummaryToSheet(ws, mergeColKeys, mergeColValues) {
        let locale = this._locale;
        let c = ws.getCell(1, 1);
        c.value = i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEE", locale });
        c.alignment = {
            horizontal: "left",
        };
        c.border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        c = ws.getCell(1, mergeColKeys + 1);
        c.value = this._userCaption;
        c.alignment = {
            horizontal: "left",
        };
        c.font = {
            italic: true,
        };
        c.border = {
            right: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        c = ws.getCell(2, 1);
        c.value = i18n_1.default.__({ phrase: "EXCEL-REPORT.START_DATE", locale });
        c.alignment = {
            horizontal: "left",
        };
        c.border = {
            left: {
                style: "thin",
            },
        };
        c = ws.getCell(2, mergeColKeys + 1);
        let s = moment(this._dataRange.startDateTime);
        c.value = {
            formula: "DATE(" + s.year() + "," + (s.month() + 1) + "," + s.date() + ")",
        };
        c.numFmt = constants.date;
        c.font = {
            italic: true,
        };
        c.alignment = {
            horizontal: "left",
        };
        c.border = {
            right: {
                style: "thin",
            },
        };
        c = ws.getCell(3, 1);
        c.value = i18n_1.default.__({ phrase: "EXCEL-REPORT.END_DATE", locale });
        c.alignment = {
            horizontal: "left",
        };
        c.border = {
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        c = ws.getCell(3, mergeColKeys + 1);
        let e = moment(this._dataRange.endDateTime);
        c.value = {
            formula: "DATE(" + e.year() + "," + (e.month() + 1) + "," + e.date() + ")",
        };
        c.numFmt = constants.date;
        c.font = {
            italic: true,
        };
        c.alignment = {
            horizontal: "left",
        };
        c.border = {
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        ws.mergeCells(1, 1, 1, mergeColKeys);
        ws.mergeCells(2, 1, 2, mergeColKeys);
        ws.mergeCells(3, 1, 3, mergeColKeys);
        ws.mergeCells(1, 3, 1, mergeColValues + 3);
        ws.mergeCells(2, 3, 2, mergeColValues + 3);
        ws.mergeCells(3, 3, 3, mergeColValues + 3);
    }
    initPermissions() {
        this.addFilterSummaryToSheet(this._wsPermissions, 2, 3);
        const headerValues = ["İzin Başlangıç", "İzin Bitiş", "İzin Türe", "Kullanılan Süre"];
        for (let index = 1; index <= headerValues.length; index++) {
            let c = this._wsPermissions.getCell(5, index);
            c.value = headerValues[index - 1];
            c.font = {
                bold: true,
            };
            c.alignment = {
                horizontal: "center",
                vertical: "bottom",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thin",
                },
            };
            this._wsPermissions.mergeCells(5, index, 6, index);
            this._wsPermissions.columns[index - 1].width = 12;
        }
        this._wsPermissions.views = [{ state: "frozen", xSplit: 0, ySplit: 6, activeCell: "A7", showGridLines: false }];
    }
    async generateExcelReport() {
        await this.setWorkSummaryData();
        await this.setAccessLogData();
        let fileId = uuid_1.default.v4();
        let filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, fileId + ".xlsx");
        await this._wb.xlsx.writeFile(filePath);
        app_logs_1.logger.debug("File is ready -> " + filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: fileId,
            filePath: filePath,
            notificationType: undefined,
            customReportName: i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKING_DURATION", locale: this._locale }),
            sendFileInAttachments: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            reportContainsEmptyData: this._request.reportTemplateId ? this._emptyReportCheck : undefined,
        };
    }
    async setAccessLogData() {
        await dal_manager_1.dbManager.accessLogPacs.getAccessLogsById({
            dateRange: this._dataRange,
            onData: this.onAccessLogData.bind(this),
            organizationId: this._request.organizationId,
            accessLogIds: this._accessLogIds,
        });
        this._wsAccessLogs.pageSetup.printArea = "A5:H" + this._wsAccessLogsCurrentRowNumber;
    }
    onAccessLogData(rows) {
        for (const row of rows) {
            this._emptyReportCheck = false;
            let log = row[0];
            let date = moment(log.u);
            this.addAccessLogDataCell({
                formula: "DATE(" + date.year() + "," + (date.month() + 1) + "," + date.date() + ")",
            }, 1, constants.date);
            this.addAccessLogDataCell({
                formula: "TIME(" + date.hour() + "," + date.minute() + "," + date.second() + ")",
            }, 2, constants.time);
            this.addAccessLogDataCell(log.an, 3);
            this.addAccessLogDataCell(log.d === app_enums_1.enums.AccessDirection.Entrance ? "Giriş" : log.d === app_enums_1.enums.AccessDirection.Exit ? "Çıkış" : undefined, 4);
            this._wsAccessLogsCurrentRowNumber++;
        }
    }
    addAccessLogDataCell(value, col, numFmt) {
        this.addDataCell(this._wsAccessLogs, this._wsAccessLogsCurrentRowNumber, value, col, numFmt);
    }
    async setWorkSummaryData() {
        await dal_manager_1.dbManager.accessPacs2.workSummaryOfEmployeeInDateRangeAll({
            dateRange: this._dataRange,
            onData: this.onEmployeeDayData.bind(this),
            organizationId: this._request.organizationId,
            userId: this._userId,
        });
        let c = this._wsWorkSummary.getCell(this._wsWorkSummaryCurrentRowNumber, 1);
        c.value = "Toplam";
        c.font = { bold: true };
        c.alignment = {
            horizontal: "right",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addWorkSummaryTotalCells();
        this._wsWorkSummary.pageSetup.printArea = "A5:I" + this._wsWorkSummaryCurrentRowNumber;
    }
    addWorkSummaryTotalCells() {
        let locale = this._locale;
        let hsFmt;
        if (locale === "tr") {
            hsFmt = constants.hsFmtTr;
        }
        else {
            hsFmt = constants.hsFmtEng;
        }
        const maps = [{ B: 2 }, { C: 3 }, { D: 4 }, { E: 5 }, { F: 6 }, { G: 7 }, { H: 8 }, { I: 8 }];
        for (const m of maps) {
            let c = this._wsWorkSummary.getCell(this._wsWorkSummaryCurrentRowNumber, Object.values(m)[0]);
            c.value = {
                formula: "SUM(" + Object.keys(m)[0] + "7:" + Object.keys(m)[0] + (this._wsWorkSummaryCurrentRowNumber - 1) + ") ",
            };
            c.font = { bold: true };
            c.numFmt = hsFmt;
            c.alignment = {
                horizontal: "center",
                wrapText: true,
            };
            c.border = {
                top: {
                    style: "thin",
                },
                left: {
                    style: "thin",
                },
                right: {
                    style: "thin",
                },
                bottom: {
                    style: "thin",
                },
            };
        }
    }
    onEmployeeDayData(rows) {
        let locale = this._locale;
        let hsFmt;
        if (locale === "tr") {
            hsFmt = constants.hsFmtTr;
        }
        else {
            hsFmt = constants.hsFmtEng;
        }
        for (const row of rows) {
            this._emptyReportCheck = false;
            let date = moment(row[0]);
            let employeeDay = row[1];
            for (const c of employeeDay.c) {
                if (c.ad && c.ad.a) {
                    if (c.ad.a.si) {
                        this._accessLogIds.push(c.ad.a.si);
                    }
                    if (c.ad.a.ei) {
                        this._accessLogIds.push(c.ad.a.ei);
                    }
                }
            }
            if (row.p) {
                this._permissions.push(...row.p);
            }
            this.addWorkSummaryDataCell({
                formula: "DATE(" + date.year() + "," + (date.month() + 1) + "," + date.date() + ")",
            }, 1, constants.date);
            this.addWorkSummaryDataCell(employeeDay.s.ew / 1440, 2, hsFmt);
            this.addWorkSummaryDataCell(employeeDay.s.n / 1440, 3, hsFmt);
            this.addWorkSummaryDataCell(employeeDay.s.e / 1440, 4, hsFmt);
            this.addWorkSummaryDataCell(employeeDay.s.m / 1440, 5, hsFmt);
            this.addWorkSummaryDataCell({
                formula: "C" + this._wsWorkSummaryCurrentRowNumber + "+" + "D" + this._wsWorkSummaryCurrentRowNumber,
            }, 6, hsFmt);
            this.addWorkSummaryDataCell(employeeDay.s.ao / 1440, 7, hsFmt);
            this.addWorkSummaryDataCell(employeeDay.s.uh / 1440, 8, hsFmt);
            let total = employeeDay.s.n / 1440 + employeeDay.s.e / 1440 + employeeDay.s.ao / 1440 + employeeDay.s.uh / 1440;
            this.addWorkSummaryDataCell(total, 9, hsFmt);
            this._wsWorkSummaryCurrentRowNumber++;
        }
    }
    addWorkSummaryDataCell(value, col, numFmt) {
        this.addDataCell(this._wsWorkSummary, this._wsWorkSummaryCurrentRowNumber, value, col, numFmt);
    }
    addDataCell(ws, currentRowNumber, value, col, numFmt) {
        let c = ws.getCell(currentRowNumber, col);
        c.value = value;
        c.numFmt = numFmt;
        c.alignment = {
            horizontal: "center",
            vertical: "bottom",
            wrapText: true,
        };
        c.border = {
            top: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
    }
}
exports.ExcelReportWorkSummaryOfEmployeeElimkoCustomReport = ExcelReportWorkSummaryOfEmployeeElimkoCustomReport;
async function generateReport(request, locale) {
    let report = new ExcelReportWorkSummaryOfEmployeeElimkoCustomReport(request, locale);
    return report.generateExcelReport();
}
exports.generateReport = generateReport;
