"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getKeyCloakAfterIdentityUpsertHook = void 0;
const app_logs_1 = require("../../../app.logs");
const keycloak_api_1 = require("./keycloak-api");
const disableKeyCloakUser = async (integrator, armonIdentity, keyCloakCheckResult) => {
    if (keyCloakCheckResult.attributes?.source && keyCloakCheckResult.attributes?.source?.length && keyCloakCheckResult.attributes?.source[0] === integrator.keyCloakSourceIdentifer) {
        const updateUserResult = await integrator.updateUser({
            id: keyCloakCheckResult.id,
            attributes: {
                mobile_number: [armonIdentity.organizationProfile.phoneNumber],
                source: [integrator.keyCloakSourceIdentifer],
            },
            email: armonIdentity.organizationProfile.email,
            enabled: false,
            firstName: armonIdentity.organizationProfile.name,
            lastName: armonIdentity.organizationProfile.surname,
            username: armonIdentity.username,
        });
        if (!updateUserResult) {
            app_logs_1.logger.error(`Error while attempting to disable user with email address ${armonIdentity.organizationProfile.email} even if the source is ${integrator.keyCloakSourceIdentifer}`);
            app_logs_1.logger.error(updateUserResult.errorMessage);
            return false;
        }
    }
    else {
        app_logs_1.logger.warn(`Old email address ${armonIdentity.organizationProfile.email} of user is not created by ${integrator.keyCloakSourceIdentifer}, cannot disable user, therefore ignoring`);
    }
    const getClientIdResult = await integrator.getClientId();
    if (!getClientIdResult) {
        app_logs_1.logger.error(`Error while getting client id!`);
        return false;
    }
    await integrator.removeUserRoleMapping(keyCloakCheckResult.id, getClientIdResult);
    return true;
};
function getKeyCloakAfterIdentityUpsertHook(orgid, sid) {
    return async function (oldIdentity, newIdentity) {
        if (!newIdentity.organizationProfile.email) {
            app_logs_1.logger.warn(`Email field is required for KeyCloak integration, but not supplied in Armon, skipping user synchronization!`);
            return true;
        }
        let integratorInstance = await keycloak_api_1.KeyCloakAPI.getInstance(orgid, sid);
        if (!integratorInstance) {
            return false;
        }
        let checkUserResult;
        if (newIdentity.organizationProfile.isDisabled === true) {
            checkUserResult = await integratorInstance.getUserByEmail(newIdentity.organizationProfile.email);
            let disableResult = true;
            if (checkUserResult) {
                app_logs_1.logger.warn(`Armon user with email address ${oldIdentity.organizationProfile.email} disabled, attempting to also disable at KeyCloak`);
                disableResult = await disableKeyCloakUser(integratorInstance, newIdentity, checkUserResult);
            }
            else {
                app_logs_1.logger.warn(`Armon user with email address ${oldIdentity.organizationProfile.email} does not exists at KeyCloak, skipping`);
            }
            return disableResult;
        }
        else if (oldIdentity &&
            oldIdentity.organizationProfile.email &&
            newIdentity.organizationProfile.email &&
            oldIdentity.organizationProfile.email !== newIdentity.organizationProfile.email) {
            app_logs_1.logger.warn(`Email address of user ${newIdentity.organizationProfile.uniqueId} is updated from ${oldIdentity.organizationProfile.email} to ${newIdentity.organizationProfile.email} !!!!`);
            checkUserResult = await integratorInstance.getUserByEmail(oldIdentity.organizationProfile.email);
            if (checkUserResult) {
                app_logs_1.logger.warn(`Old email address ${oldIdentity.organizationProfile.email} of user ${newIdentity.organizationProfile.uniqueId} exists at KeyCloak, attempting to disable`);
                await disableKeyCloakUser(integratorInstance, oldIdentity, checkUserResult);
            }
            else {
                app_logs_1.logger.warn(`Armon user with email address ${oldIdentity.organizationProfile.email} does not exists at KeyCloak, skipping`);
            }
        }
        checkUserResult = await integratorInstance.getUserByEmail(newIdentity.organizationProfile.email);
        try {
            if (!checkUserResult) {
                app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} does not exist at KeyCloak, trying to create new user`);
                const addUserResult = await integratorInstance.addUser(newIdentity);
                const newUser = await integratorInstance.getUserByEmail(newIdentity.organizationProfile.email);
                const updatePasswordCallResult = await integratorInstance.updatePasswordCall(newUser.id);
                const getClientIdResult = await integratorInstance.getClientId();
                if (!getClientIdResult) {
                    app_logs_1.logger.error(`Error while getting client id!`);
                    return false;
                }
                const getKeyCloakRoleResult = await integratorInstance.getDefaultRole(getClientIdResult);
                if (!getKeyCloakRoleResult) {
                    app_logs_1.logger.error(`Error while getting client role!`);
                    return false;
                }
                const mapWithRoleResult = await integratorInstance.mapUserWithRole(newUser.id, getClientIdResult, getKeyCloakRoleResult);
                if (!addUserResult.success || !mapWithRoleResult.success || !updatePasswordCallResult.success || !mapWithRoleResult.success) {
                    app_logs_1.logger.error(addUserResult.errorMessage || mapWithRoleResult.errorMessage || updatePasswordCallResult.errorMessage || mapWithRoleResult.errorMessage);
                    return false;
                }
                app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully created at KeyCloak!`);
            }
            else {
                if (checkUserResult.attributes?.source && checkUserResult.attributes?.source.length && checkUserResult.attributes?.source[0] === integratorInstance.keyCloakSourceIdentifer) {
                    app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} previously sent to KeyCloak, updating!`);
                    const updateUserResult = await integratorInstance.updateUser({
                        id: checkUserResult.id,
                        attributes: {
                            mobile_number: [newIdentity.organizationProfile.phoneNumber],
                            source: [integratorInstance.keyCloakSourceIdentifer],
                        },
                        email: newIdentity.organizationProfile.email,
                        enabled: true,
                        firstName: newIdentity.organizationProfile.name,
                        lastName: newIdentity.organizationProfile.surname,
                        username: checkUserResult.username,
                    });
                    if (!checkUserResult.enabled) {
                        await integratorInstance.updatePasswordCall(checkUserResult.id);
                    }
                    if (!updateUserResult.success) {
                        app_logs_1.logger.error(`Newly added or updated Armon user with email address ${newIdentity.organizationProfile.email} could not be updated at KeyCloak!`);
                        app_logs_1.logger.error(`This case should not occur under normal circumstances!`);
                        return false;
                    }
                    app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully updated at KeyCloak!`);
                }
                else {
                    app_logs_1.logger.warn(`User exists at KeyCloak with email address ${newIdentity.organizationProfile.email} but source is not ${integratorInstance.keyCloakSourceIdentifer}. Can not add or modify user!`);
                }
                const userRoleMappingCheckResult = await integratorInstance.checkUserRoleMappings(checkUserResult.id);
                if (!userRoleMappingCheckResult) {
                    const getClientIdResult = await integratorInstance.getClientId();
                    if (!getClientIdResult) {
                        app_logs_1.logger.error(`Error while getting client id!`);
                        return false;
                    }
                    const getKeyCloakRoleResult = await integratorInstance.getDefaultRole(getClientIdResult);
                    if (!getKeyCloakRoleResult) {
                        app_logs_1.logger.error(`Error while getting client role!`);
                        return false;
                    }
                    const mapWithRoleResult = await integratorInstance.mapUserWithRole(checkUserResult.id, getClientIdResult, getKeyCloakRoleResult);
                    if (!mapWithRoleResult.success) {
                        app_logs_1.logger.error(`Cannot map user with email address ${newIdentity.organizationProfile.email} with role ${getKeyCloakRoleResult.name}!`);
                        return false;
                    }
                    app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully mapped with role at KeyCloak!`);
                }
            }
            app_logs_1.logger.info(`Armon user with email address ${newIdentity.organizationProfile.email} successfully synced to KeyCloak in accordance with predefined conditions!`);
            return true;
        }
        catch (error) {
            app_logs_1.logger.error(`Error while running KeyCloak user synchronization!!!`);
            app_logs_1.logger.error(error);
        }
    };
}
exports.getKeyCloakAfterIdentityUpsertHook = getKeyCloakAfterIdentityUpsertHook;
