"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CRMAPI = void 0;
const axios_1 = __importDefault(require("axios"));
const app_logs_1 = require("../../../app.logs");
const dal_access_psql_organization_1 = require("../../../dal/access/psql/dal.access.psql.organization");
const axios_cookiejar_support_1 = require("axios-cookiejar-support");
const tough_cookie_1 = require("tough-cookie");
const luxon_1 = require("luxon");
const TAG = `[crm-api.ts]`;
class CRMAPI {
    constructor() { }
    static async getInstance(organizationId, crmSecretId, trx) {
        if (!CRMAPI.instance) {
            CRMAPI.instance = new CRMAPI();
            const crmSecrets = await (0, dal_access_psql_organization_1.getOrganizationSecretById)(organizationId, crmSecretId, trx);
            if (crmSecrets) {
                CRMAPI.instance.baseUrl = crmSecrets.baseUrl;
                CRMAPI.instance.clientId = crmSecrets.clientId;
                CRMAPI.instance.scope = crmSecrets.scope;
                CRMAPI.instance.clientSecret = crmSecrets.clientSecret;
                CRMAPI.instance.grantType = crmSecrets.grantType;
                CRMAPI.instance.notificationTriggerPath = crmSecrets.notificationTriggerPath;
                CRMAPI.instance.authenticationPath = crmSecrets.authenticationPath;
                CRMAPI.instance.senderName = crmSecrets.senderName;
            }
            else {
                app_logs_1.logger.error(`CRM Secrets could not be obtained from database, operation aborted`);
                return null;
            }
            const cookieJar = new tough_cookie_1.CookieJar();
            CRMAPI.axiosClient = (0, axios_cookiejar_support_1.wrapper)(axios_1.default.create({
                jar: cookieJar,
                withCredentials: true,
            }));
        }
        return CRMAPI.instance;
    }
    async renewCRMToken() {
        const queryParams = new URLSearchParams();
        queryParams.append("grant_type", this.grantType);
        queryParams.append("scope", this.scope);
        try {
            const response = await CRMAPI.axiosClient({
                method: "POST",
                url: this.baseUrl + this.authenticationPath,
                headers: {
                    "Content-Type": "application/x-www-form-urlencoded",
                },
                data: queryParams,
                auth: {
                    username: this.clientId,
                    password: this.clientSecret,
                },
                timeout: 10000,
            });
            CRMAPI.CRMToken = response.data.access_token;
            CRMAPI.CRMTokenExpiration = luxon_1.DateTime.now().plus({ seconds: response.data.expires_in - 10 });
            app_logs_1.logger.info(`${TAG} CRM Token obtained successfully. Valid until ${CRMAPI.CRMTokenExpiration.toString()}`);
            return true;
        }
        catch (error) {
            app_logs_1.logger.error(`${TAG} Error while getting CRM Token`);
            app_logs_1.logger.error(error);
            console.log(error.response);
        }
        return false;
    }
    async sendMail(campaignCode, parameters, receivers, notificationInstanceId) {
        if (!CRMAPI.CRMToken || luxon_1.DateTime.now() > CRMAPI.CRMTokenExpiration) {
            const crmTokenResult = await this.renewCRMToken();
            if (!crmTokenResult) {
                app_logs_1.logger.error(`${TAG} Notification with instance id [${notificationInstanceId}] could not be sent via CRM, Bearer token cannot be obtained from CRM`);
                return {
                    isSuccess: false,
                    failReason: "Bearer token cannot be obtained from CRM",
                };
            }
        }
        const requestBody = {
            IV_SENDER: this.senderName,
            IV_CAMPAIGN_ID: campaignCode,
            IV_SEND_TIME: luxon_1.DateTime.now().toFormat("yyyyMMddHHmmss"),
            IT_EMAIL_PARAMS: {
                item: [
                    {
                        UNIQUE_PARAMETER: notificationInstanceId,
                        UNIQUE_ID: null,
                        TO: { item: receivers },
                        MESSAGE_VARIABLES: {
                            item: parameters,
                        },
                    },
                ],
            },
        };
        console.log(`${TAG} email request parameters: ${JSON.stringify(requestBody, null, 4)}`);
        try {
            const mailResponse = await CRMAPI.axiosClient({
                method: "POST",
                url: this.baseUrl + this.notificationTriggerPath,
                headers: {
                    Authorization: `Bearer ${CRMAPI.CRMToken}`,
                    "Content-Type": "application/json",
                },
                data: requestBody,
            });
            console.log(`${TAG} CRM email response: ${JSON.stringify(mailResponse.data, null, 4)}`);
            if (mailResponse.data?.ET_RESULTS?.item && mailResponse.data?.ET_RESULTS?.item?.length && mailResponse.data?.ET_RESULTS?.item[0]?.STATUS === "20") {
                app_logs_1.logger.info(`${TAG} Notification with instance id [${notificationInstanceId}] sent successfully via CRM`);
                return {
                    isSuccess: true,
                };
            }
            else {
                app_logs_1.logger.error(`${TAG} Notification with instance id [${notificationInstanceId}] could not be sent via CRM`);
                app_logs_1.logger.error(`${TAG} ${JSON.stringify(mailResponse.data, null, 4)}`);
                return {
                    isSuccess: false,
                    failReason: mailResponse.data,
                };
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error while sending mail through CRM`);
            app_logs_1.logger.error(error);
            console.log(error.response);
            return {
                isSuccess: false,
                failReason: error,
            };
        }
    }
    async sendSMS(campaignCode, parameters, receiver, notificationInstanceId) {
        if (!CRMAPI.CRMToken || luxon_1.DateTime.now() > CRMAPI.CRMTokenExpiration) {
            const crmTokenResult = await this.renewCRMToken();
            if (!crmTokenResult) {
                app_logs_1.logger.error(`${TAG} Notification with instance id [${notificationInstanceId}] could not be sent via CRM, Bearer token cannot be obtained from CRM`);
                return {
                    isSuccess: false,
                    failReason: "Bearer token cannot be obtained from CRM",
                };
            }
        }
        const requestBody = {
            IV_SENDER: this.senderName,
            IV_CAMPAIGN_ID: campaignCode,
            IV_SEND_TIME: luxon_1.DateTime.now().toFormat("yyyyMMddHHmmss"),
            IT_SMS_PARAMS: {
                item: [
                    {
                        UNIQUE_PARAMETER: notificationInstanceId,
                        UNIQUE_ID: null,
                        TELEPHONE: receiver,
                        MESSAGE_VARIABLES: {
                            item: parameters,
                        },
                    },
                ],
            },
        };
        console.log(`${TAG} CRM sms request parameters: ${JSON.stringify(requestBody, null, 4)}`);
        try {
            const smsResponse = await CRMAPI.axiosClient({
                method: "POST",
                url: this.baseUrl + this.notificationTriggerPath,
                headers: {
                    Authorization: `Bearer ${CRMAPI.CRMToken}`,
                    "Content-Type": "application/json",
                },
                data: requestBody,
            });
            console.log(`${TAG} CRM sms response: ${JSON.stringify(smsResponse.data, null, 4)}`);
            if (smsResponse.data?.ET_RESULTS?.item && smsResponse.data.ET_RESULTS.item.length && smsResponse.data.ET_RESULTS.item[0].STATUS === "20") {
                app_logs_1.logger.info(`${TAG} Notification with instance id [${notificationInstanceId}] sent successfully via CRM`);
                return {
                    isSuccess: true,
                };
            }
            else {
                app_logs_1.logger.error(`${TAG} Notification with instance id [${notificationInstanceId}] could not be sent via CRM`);
                app_logs_1.logger.error(`${TAG} ${JSON.stringify(smsResponse.data, null, 4)}`);
                return {
                    isSuccess: false,
                    failReason: smsResponse.data,
                };
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error while sending SMS through CRM`);
            app_logs_1.logger.error(error);
            console.log(error.response);
            return {
                isSuccess: false,
                failReason: error,
            };
        }
    }
}
exports.CRMAPI = CRMAPI;
