"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateHook = void 0;
const luxon_1 = require("luxon");
const app_logs_1 = require("../../../app.logs");
const business_hooks_1 = require("../../../business/business.hooks");
const business_visitor_1 = require("../../../business/visitor/business.visitor");
const dal_access_error_1 = require("../../../dal/access/dal.access.error");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const predefined_roles_1 = require("../../../dal/db/predefined/predefined.roles");
const restapi_1 = require("../../../lib/es/models/restapi");
const visitorRegistrationPointId = "74a5ca4c-1873-43cc-852d-aafe0cd4267d";
const terminateActiveVisitsExceed24HoursId = "80eb6555-99b1-459a-9509-3ae849eea494";
function CreateHook() {
    return new CoZoneHook();
}
exports.CreateHook = CreateHook;
class CoZoneHook extends business_hooks_1.OrganizationHookModels.IArmonHook {
    constructor() {
        super();
        this.beforePreregisterVisit = async function (params) {
            let isStandartUser = false;
            let visitedUser = params.visitFields.find((f) => f.name === "visitedPerson");
            if (params.visitorProfileId) {
                const activeVisits = (await params.trx.raw(`
					SELECT id, state
					FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}"
					WHERE "organizationVisitorProfileId" = ?
				`, [params.visitorProfileId])).rows;
                if (activeVisits.length > 0) {
                    if (activeVisits[0].state === restapi_1.OrganizationVisitorStates.Expected) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.VISITOR.CUSTOM.VISITOR_HAS_ACTIVE_PREREGISTERED_VISIT", null, false);
                    }
                    else if (activeVisits[0].state === restapi_1.OrganizationVisitorStates.Active) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.VISITOR.CUSTOM.VISITOR_HAS_ACTIVE_VISIT", null, false);
                    }
                }
            }
            if (!visitedUser) {
                const userRoleControlResult = (await params.trx.raw(`
					SELECT "typeId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" AS r
					INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
						ON uo."roleId" = r.id
					WHERE uo."deletedAt" IS NULL AND r."deletedAt" IS NULL
						AND uo."userId" = ?
					`, [params.requesterUserId])).rows;
                if (!userRoleControlResult.length || !(userRoleControlResult[0].typeId === predefined_roles_1.PredefinedRoles.StandartUserOrganization.id)) {
                    (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.CUSTOM.VISITED_PERSON_REQUIRED", null, false);
                }
                else {
                    visitedUser = {
                        name: "visitedPerson",
                        value: params.requesterUserId,
                    };
                    isStandartUser = true;
                }
            }
            const unitCheck = (await params.trx.raw(`
				SELECT COUNT(*)::smallint AS c FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					ON uoou."userOrganizationId" = uo.id
				WHERE uo."deletedAt" IS NULL AND uoou."deletedAt" IS NULL
					AND uo."userId" = ?
			`, [visitedUser.value])).rows;
            if (!unitCheck[0].c) {
                if (isStandartUser) {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.VISITOR.CUSTOM.VISITED_PERSON_MISSING_UNIT_SELF", null, false);
                }
                else {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.VISITOR.CUSTOM.VISITED_PERSON_MISSING_UNIT", null, false);
                }
            }
            return params.visitorProfileFields;
        };
        this.afterPreregisterVisit = async function (params) {
            const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(params.organizationId);
            const accessControlPoints = await params.trx.raw(`
				SELECT * FROM
				"${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
				WHERE "deletedAt" IS NULL
			`);
            const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId: params.organizationId, userId: params.requestUserId });
            try {
                const dbResult = await business_visitor_1.BusinessVisitor.newVisit(params.organizationId, params.requestUserId, {
                    visitFields: params.visitFields,
                    accessControlPointIds: accessControlPoints.rows.map((r) => r.id),
                    credentials: [],
                    force: true,
                    visitorProfileFields: params.visitorProfileFields,
                    visitorProfileId: params.visitorProfileId,
                    replacedVisitorProfileId: params.visitorProfileId,
                    visitId: params.expectedVisitId,
                    visitorRegistrationPointId,
                }, organizationVisitorModuleSettings, params.trx, locale);
                app_logs_1.logger.info("after preregistration promoted to visit: " + JSON.stringify(dbResult, null, 4));
            }
            catch (err) {
                app_logs_1.logger.error("Error while promoting preregistered visit to active visit");
                app_logs_1.logger.error(err);
            }
        };
        this.scheduledJobRoutine = this.customSceduledJobRoutine.bind(this);
    }
    async customSceduledJobRoutine(organizationId, scheduledJobId) {
        if (scheduledJobId === terminateActiveVisitsExceed24HoursId) {
            await this.terminateActiveVisitsExceed24Hours(organizationId);
        }
        else {
            app_logs_1.logger.error("Scheduled job id is not equal to only defined custom scheduled job for organization " + organizationId);
            app_logs_1.logger.error("Probably there is an error in scheduling mechanism");
        }
    }
    async terminateActiveVisitsExceed24Hours(organizationId) {
        const lastActiveVisitTimeBefTermination = luxon_1.DateTime.now().minus({ days: 1 }).toSQL();
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const { rows, rowCount } = await trx.query(`
				SELECT id as "visitId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}"
				WHERE "startUtc" < $1 and state = $2
				`, [lastActiveVisitTimeBefTermination, dal_constants_1.DalConstants.UnterminatedVisitState.Active]);
            if (rowCount === 0) {
                app_logs_1.logger.info("--------------------------------------");
                app_logs_1.logger.info("There is no active visit which exceeds 24 hour. Organization: " + organizationId);
            }
            else {
                const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
                for (const visit of rows) {
                    await dal_manager_1.dbManager.accessVisitor.terminateVisit(organizationId, visit.visitId, organizationVisitorModuleSettings, dal_constants_1.DalConstants.SystemUserId);
                }
            }
        });
    }
}
