"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.AkedasEmployeeLeaveCustomReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const dal_manager_1 = require("../../../dal/dal.manager");
const business_pacs_claim_1 = require("../../../business/pacs/business.pacs.claim");
const business_pacs_ppermission_1 = require("../../../business/pacs/business.pacs.ppermission");
const excel_report_refactory_1 = require("../../../business/report/excel.report.refactory");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
var Constants;
(function (Constants) {
    Constants.AnnualLeaveTypeId = "9af0f620-810f-4134-8dbd-44f46c1f6b1d";
})(Constants || (Constants = {}));
class AkedasEmployeeLeaveCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        const options = request.filter;
        this._reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
        this._filter = {
            dateRange: {
                startDateTime: new Date(options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay),
                endDateTime: new Date(options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay),
            },
            userFilter: {
                organizationUnitIds: options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                organizationUnitHierachially: options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userIds: options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                userGroupIds: options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                status: options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserStatus)?.value,
            },
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
    }
    getHeaders() {
        const headers = [
            {
                value: "Organizasyon Birim",
                width: 35,
            },
            {
                value: "İzin Yılı",
                width: 20,
            },
            {
                value: "İzin Ay",
                width: 20,
            },
            {
                value: "İzin Başlangıç",
                width: 20,
            },
            {
                value: "İzin Bitiş",
                width: 20,
            },
            {
                value: "İzin Türü",
                width: 20,
            },
            {
                value: "İzin Gün Sayısı",
                width: 20,
            },
            {
                value: "Kalan İzin",
                width: 20,
            },
            {
                value: "İzin Alma Tarihi",
                width: 20,
            },
            {
                value: "İzin Onaylanma Tarihi",
                width: 20,
            },
            {
                value: "Onay Veren Birim Müdürü (Ad-Soyad, Tarih, Saat)",
                width: 40,
            },
            {
                value: "Onaylayan IK Personeli (Ad-Soyad, Tarih, Saat)",
                width: 40,
            },
        ];
        return headers;
    }
    async generateReport() {
        let fileObj;
        const sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                headers: this.getHeaders(),
            },
        ];
        const reportInfoSheetData = {
            reportNameValue: "Akedaş İzin Kullanım Raporu",
            filters: {
                organizationUnitIds: this._filter.userFilter.organizationUnitIds,
                organizationUnitHierachially: this._filter.userFilter.organizationUnitHierachially,
                userIds: this._filter.userFilter.userIds,
                userGroupIds: this._filter.userFilter.userGroupIds,
                status: this._filter.userFilter.status,
            },
        };
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
                idBasedUserFilter: {
                    userIds: this._filter?.userFilter?.userIds,
                    userGroupIds: this._filter?.userFilter?.userGroupIds,
                    organizationUnitIds: this._filter?.userFilter?.organizationUnitIds,
                    userOrganizationStatus: this._filter?.userFilter?.status,
                    applyOrganizationUnitFilterHierarchically: this._filter?.userFilter?.organizationUnitHierachially,
                },
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                bindingKeys: [],
            });
            let qParams = userFilter.bindingKeys;
            qParams.push(this._filter.dateRange.endDateTime);
            qParams.push(this._filter.dateRange.startDateTime);
            let ppermissions = (await trx.query(`SELECT p.id, 
						p."ppermissionTypeId", 
						pt."id" as "ptId",
						pt."name",
						pt."isDailyScheduled",
						p."startDateTime", 
						p."endDateTime", 
						p."requestDateTime", 
						upp."userId", 
						jsonb_agg(
							jsonb_build_object(
							'approvalId', uppa.id,
							'approverUserId', uppa."approverUserId",
							'approvementDate', uppa."approvementDate",
							'organizationUnitId', uppa."organizationUnitId",
							'status', uppa."status",
							'approverFullname', uop."name" || ' ' || uop."surname"
							)		
						) AS approvements
					FROM "${this._request.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" p
					INNER JOIN "${this._request.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" upp ON upp."ppermissionId" = p."id"
					INNER JOIN(${userFilter.query}) uf ON uf."userId" = upp."userId"
					LEFT JOIN "${this._request.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" pt ON pt."id" = p."ppermissionTypeId" AND pt."deletedAt" IS NULL
					LEFT JOIN "${this._request.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements}" uppa ON p."id" = uppa."ppermissionId" AND upp."userId" = uppa."userId"
					LEFT JOIN "${this._request.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop ON uppa."approverUserId" = uop."userId"
					WHERE p."startDateTime" < $${qParams.length - 1} AND p."endDateTime" > $${qParams.length}
					GROUP BY 
						p.id, 
						p."ppermissionTypeId",
						pt."id",
						pt."name",
						pt."isDailyScheduled",
						p."startDateTime", 
						p."endDateTime", 
						p."requestDateTime", 
						upp."userId"
					ORDER BY p."startDateTime" DESC`, qParams)).rows;
            ppermissions = ppermissions.filter((row) => row.approvements?.every((a) => a.approvementDate && a.status === true));
            const userIds = new Set();
            const ppermissionTypeIds = new Set();
            ppermissions.forEach((pp) => {
                userIds.add(pp.userId);
                ppermissionTypeIds.add(pp.ppermissionTypeId);
            });
            await this._reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds: Array.from(userIds),
                reportInfoSheetData,
                sheetInfo: sheetInfo,
            });
            const ppermissionTypeUsage = await (0, business_pacs_claim_1.getUsersPPermissionsTypeUsageOfBetweenDates)(this._request.organizationId, Array.from(userIds), Constants.AnnualLeaveTypeId, trx);
            const organizationUnitsOfUsers = await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasicOfUsers(this._request.organizationId, Array.from(userIds), trx);
            const vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(this._request.organizationId);
            const ppermissionTypes = await dal_manager_1.dbManager.accessPacs.getPPermissionTypes(this._request.organizationId, Array.from(ppermissionTypeIds));
            const permissionTypeMap = new Map(ppermissionTypes.map((pt) => [pt.id, pt]));
            for (const ppermission of ppermissions) {
                const sheetInfo = [
                    {
                        sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                        data: [],
                    },
                ];
                ppermission["orgUnits"] = organizationUnitsOfUsers
                    .find((user) => user.userId === ppermission.userId)
                    .units.map((unit) => unit.name)
                    .join(", ");
                const permissionType = permissionTypeMap.get(ppermission.ppermissionTypeId);
                ppermission["leaveDuration"] = ppermission.isDailyScheduled
                    ? await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(this._request.organizationId, ppermission.ppermissionTypeId, ppermission.startDateTime, ppermission.endDateTime, vacations, ppermission.userId, permissionType)
                    : await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(this._request.organizationId, ppermission.ppermissionTypeId, ppermission.startDateTime, ppermission.endDateTime, vacations, permissionType);
                if (ppermission.ptId === Constants.AnnualLeaveTypeId) {
                    const foundUserPeriod = ppermissionTypeUsage.userPeriods.find((userPeriod) => userPeriod.userId === ppermission.userId);
                    const lastPeriod = foundUserPeriod.periods[0];
                    ppermission["leftLeaves"] = lastPeriod.remaining !== null ? lastPeriod.remaining / app_enums_1.enums.PPermissionUsageFactor.DailyPermissionUsageFactor : null;
                }
                let columns = this.addRow(ppermission);
                sheetInfo[0].data.push({
                    userId: ppermission.userId,
                    cells: columns,
                });
                this._reportFactory.appendData(sheetInfo);
            }
        });
        fileObj = await this._reportFactory.finalize(reportInfoSheetData);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: fileObj.fileId,
            filePath: fileObj.filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
        };
    }
    addRow(row) {
        const start = luxon_1.DateTime.fromJSDate(row.startDateTime).startOf("month");
        const end = luxon_1.DateTime.fromJSDate(row.endDateTime).endOf("month");
        const years = [];
        let currentYear = start.year;
        while (currentYear <= end.year) {
            years.push(currentYear);
            currentYear++;
        }
        const months = [];
        let current = start;
        while (current <= end) {
            months.push(current.setLocale(this._locale).toFormat("LLLL"));
            current = current.plus({ months: 1 });
        }
        const latestApprovaltDate = new Date(Math.max(...row.approvements?.map((approvement) => new Date(approvement.approvementDate).getTime())));
        let hrs = [];
        let unitManagers = [];
        row.approvements?.forEach((appr) => {
            if (appr.organizationUnitId) {
                unitManagers.push(appr);
            }
            else {
                hrs.push(appr);
            }
        });
        let unitManagersText = "";
        let hrsText = "";
        unitManagers.forEach((unitManager) => {
            unitManagersText += unitManager.approverFullname + " - " + luxon_1.DateTime.fromISO(unitManager.approvementDate).toFormat("dd.MM.yyyy HH:mm") + " \n";
        });
        hrs.forEach((hr) => {
            hrsText += hr.approverFullname + " - " + luxon_1.DateTime.fromISO(hr.approvementDate).toFormat("dd.MM.yyyy HH:mm") + " \n";
        });
        const rowData = [
            {
                value: row.orgUnits,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: years.join(", "),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: months.join(", "),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: row.isDailyScheduled
                    ? luxon_1.DateTime.fromJSDate(row.startDateTime).toFormat("dd.MM.yyyy")
                    : luxon_1.DateTime.fromJSDate(row.startDateTime).toFormat("dd.MM.yyyy HH:mm"),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: row.isDailyScheduled
                    ? luxon_1.DateTime.fromJSDate(row.endDateTime).toFormat("dd.MM.yyyy")
                    : luxon_1.DateTime.fromJSDate(row.endDateTime).toFormat("dd.MM.yyyy HH:mm"),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: row.name,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: row.isDailyScheduled ? row.leaveDuration : (row.leaveDuration / (24 * 60)),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: row.isDailyScheduled ? null : row.leaveDuration / (24 * 60) && this.getDurationFormatExcel(this._locale),
            },
            {
                value: row.leftLeaves ?? null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: luxon_1.DateTime.fromJSDate(row.requestDateTime).toFormat("dd.MM.yyyy"),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: luxon_1.DateTime.fromJSDate(latestApprovaltDate).toFormat("dd.MM.yyyy"),
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
            },
            {
                value: unitManagersText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
                alignment: {
                    wrapText: true,
                },
            },
            {
                value: hrsText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
                alignment: {
                    wrapText: true,
                },
            },
        ];
        return rowData;
    }
    getDurationFormatExcel(locale) {
        return ('[<0.000694444]"";[<0.0415][m]" ' +
            i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: locale }) +
            '";[h] "' +
            i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: locale }) +
            ' " m "' +
            i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: locale }) +
            '"');
    }
}
exports.AkedasEmployeeLeaveCustomReport = AkedasEmployeeLeaveCustomReport;
async function generateReport(request, locale) {
    let report = new AkedasEmployeeLeaveCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
