"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = exports.MerzigoDailyCustomReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const generator_1 = require("../../../business/report/generator");
const dal_manager_1 = require("../../../dal/dal.manager");
const excel_report_refactory_1 = require("../../../business/report/excel.report.refactory");
const report_util_1 = require("../../../business/report/report.util");
var Constants;
(function (Constants) {
    Constants.RaporLeaveId = "f03d5b04-921b-4151-a01a-583270ebe1fa";
})(Constants || (Constants = {}));
class MerzigoDailyCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._userIds = [];
        this._afterDatesLength = 0;
        this._sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                data: [],
                headers: [],
            },
        ];
        const options = request.filter;
        this._reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
        this._filter = {
            dateRange: {
                startDateTime: new Date(options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay),
                endDateTime: new Date(options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay),
            },
            userFilter: {
                organizationUnitIds: options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                organizationUnitHierachially: options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userIds: options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                userGroupIds: options.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                status: options.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserStatus)?.value,
            },
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
        this._sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                data: [],
                headers: this.getHeaders(this._filter.dateRange),
            },
        ];
    }
    async onPerEmployeeDataCallbackFunctionReader(rows) {
        app_logs_1.logger.debug("Fetch reader total data %s rows", rows.length);
        this._userIds = [...this._userIds, ...rows.map((item) => item.userId)];
        let userIds = rows.map((item) => item.userId);
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const usersWithUnits = await dal_manager_1.dbManager.accessUser.listUsersUnitsHierarchicallyOrdered(this._request.organizationId, this._userIds, trx);
            const usersProfiles = (await trx.query(`
				SELECT
				    uop."employmentStartUtc",
				    uop."employmentEndUtc",
				    uop."extensionFields"::text,
				    uop."userId",
				    string_agg(ug."name", ', ' ORDER BY ug."name") AS "userGroupNames"
				FROM "${this._request.organizationId}"."userOrganizationProfiles" uop
				LEFT JOIN "${this._request.organizationId}"."userGroupUserOrganizations" uguo
				    ON uguo."userOrganizationId" = uop."userOrganizationId"
				LEFT JOIN "${this._request.organizationId}"."userGroups" ug
				    ON ug."id" = uguo."userGroupId"
				WHERE uop."userId" = ANY($1)
				GROUP BY
				    uop."employmentStartUtc",
				    uop."employmentEndUtc",
				    uop."extensionFields"::text,
				    uop."userId"`, [userIds])).rows;
            for (const employeeDays of rows) {
                let generatedGeneralColumns = [];
                let foundUserWithUnits = usersWithUnits.find((userU) => userU.userId === employeeDays.userId);
                let foundProfile = usersProfiles.find((userP) => userP.userId === employeeDays.userId);
                generatedGeneralColumns.push({
                    value: foundUserWithUnits.units[1]?.name,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                generatedGeneralColumns.push({
                    value: foundUserWithUnits.units[1]?.name,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                generatedGeneralColumns.push({
                    value: foundUserWithUnits.units[foundUserWithUnits.units.length - 1]?.name,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                generatedGeneralColumns.push({
                    value: JSON.parse(foundProfile.extensionFields)?.workArea,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                generatedGeneralColumns.push({
                    value: JSON.parse(foundProfile.extensionFields)?.title,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                generatedGeneralColumns.push({
                    value: foundProfile.userGroupNames,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                generatedGeneralColumns.push({
                    value: foundProfile.employmentStartUtc,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                generatedGeneralColumns.push({
                    value: foundProfile.employmentEndUtc,
                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    alignment: "center",
                });
                employeeDays.days = employeeDays.days.sort((a, b) => new Date(a.d).getTime() - new Date(b.d).getTime());
                for (const day of employeeDays.days) {
                    if (day) {
                        generatedGeneralColumns.push(this.firstEntrance(day));
                        generatedGeneralColumns.push(this.lastOut(day));
                    }
                    else {
                        this._sheetInfo[0].data.push({
                            userId: employeeDays.userId,
                            cells: [
                                {
                                    value: i18n_1.default.__({ phrase: "EXCEL-REPORT.STILL_CALCULATING", locale: this._locale }),
                                    border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                                },
                            ],
                        });
                    }
                }
                while (generatedGeneralColumns.length < this._afterDatesLength) {
                    generatedGeneralColumns.push({
                        value: null,
                        border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                    });
                }
                generatedGeneralColumns.push(this.missingDayCount(employeeDays));
                generatedGeneralColumns.push(this.extraWorkDuration(employeeDays));
                generatedGeneralColumns.push(this.missingWorkDuration(employeeDays));
                generatedGeneralColumns.push(this.physicallyInRegionDuration(employeeDays));
                generatedGeneralColumns.push(this.breakDuration(employeeDays));
                generatedGeneralColumns.push(this.leaveUsedDayCount(employeeDays));
                generatedGeneralColumns.push(this.raporUsedDayCount(employeeDays));
                this._sheetInfo[0].data.push({
                    userId: employeeDays.userId,
                    cells: generatedGeneralColumns,
                });
            }
        });
    }
    getHeaders(daterange) {
        const headers = [
            {
                value: "Şirket",
                width: 30,
            },
            {
                value: "Bağlı Birim",
                width: 30,
            },
            {
                value: "Departman",
                width: 30,
            },
            {
                value: "Birim",
                width: 30,
            },
            {
                value: "Unvan",
                width: 30,
            },
            {
                value: "Kullanıcı Grubu \n (Yöneticiler)",
                width: 30,
            },
            {
                value: "İlk İşe Giriş Tarihi",
                width: 30,
            },
            {
                value: "Çıkış Tarihi",
                width: 30,
            },
        ];
        const start = luxon_1.DateTime.fromJSDate(daterange.startDateTime);
        const end = luxon_1.DateTime.fromJSDate(daterange.endDateTime);
        let current = start;
        while (current <= end) {
            headers.push({
                value: `Giriş \n ${current.toFormat("dd.MM.yyyy")} \n ${current.setLocale("tr").toFormat("cccc")}`,
                width: 30,
            });
            headers.push({
                value: `Çıkış \n ${current.toFormat("dd.MM.yyyy")} \n ${current.setLocale("tr").toFormat("cccc")}`,
                width: 30,
            });
            current = current.plus({ days: 1 });
        }
        this._afterDatesLength = headers.length;
        const remainingHeaders = [
            {
                value: "Eksik Gün",
                width: 35,
            },
            {
                value: "Fazla Çalışma Saati",
                width: 20,
            },
            {
                value: "Eksik Mesai Saati",
                width: 20,
            },
            {
                value: "Bölgede Geçirilen Süre",
                width: 20,
            },
            {
                value: "Kullanılan Toplam Mola Süresi",
                width: 20,
            },
            {
                value: "Kullanılan İzin",
                width: 20,
            },
            {
                value: "Raporlu Gün",
                width: 20,
            },
        ];
        return [...headers, ...remainingHeaders];
    }
    async generateReport() {
        let fileObj;
        const reportInfoSheetData = {
            reportNameValue: "Merzigo Çalışma Detayı Raporu",
            filters: {
                organizationUnitIds: this._filter.userFilter.organizationUnitIds,
                organizationUnitHierachially: this._filter.userFilter.organizationUnitHierachially,
                userIds: this._filter.userFilter.userIds,
                userGroupIds: this._filter.userFilter.userGroupIds,
                status: this._filter.userFilter.status,
                dateTimeRange: {
                    startDateTime: this._filter.dateRange.startDateTime,
                    endDateTime: this._filter.dateRange.endDateTime,
                },
            },
        };
        await dal_manager_1.dbManager.accessPacs2.getEmployeeDaysPerEmployeeFiltered(this._request.organizationId, this._request.requesterUserId, {
            startDate: new Date(this._filter.dateRange.startDateTime),
            endDate: new Date(this._filter.dateRange.endDateTime),
            organizationUnitIds: this._filter.userFilter?.organizationUnitIds,
            organizationUnitHierarchically: this._filter.userFilter?.organizationUnitHierachially,
            userGroupIds: this._filter.userFilter?.userGroupIds,
            userIds: this._filter.userFilter?.userIds,
            status: this._filter.userFilter.status,
            sortType: 1,
            sortOrder: "ASC",
            take: null,
            skip: null,
        }, this.onPerEmployeeDataCallbackFunctionReader.bind(this));
        await this._reportFactory.initialize({
            requesterUserId: this._request.requesterUserId,
            userIds: this._userIds,
            reportInfoSheetData,
            sheetInfo: this._sheetInfo,
            profileColumnsIndex: 6,
        });
        this._reportFactory.appendData(this._sheetInfo);
        fileObj = await this._reportFactory.finalize(reportInfoSheetData);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: fileObj.fileId,
            filePath: fileObj.filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
        };
    }
    firstEntrance(data) {
        let isDummyAccessLogForFirstIn = false;
        let firstInText;
        if (data.s.wd > 0 && data.s.pr == 0) {
            firstInText = i18n_1.default.__({ phrase: "PDF-REPORT.not_arrived", locale: this._locale });
        }
        const holidayIdsOfTheDay = new Set(data.h.map((h) => h.h));
        if (Array.from(holidayIdsOfTheDay).length > 0) {
            firstInText = "Resmi Tatil";
        }
        const leaveIdsOfTheDay = new Set(data.p.map((l) => l.pt));
        Array.from(leaveIdsOfTheDay).forEach((leave) => {
            if (leave === Constants.RaporLeaveId) {
                firstInText = "Raporlu";
            }
            else {
                firstInText = "Yıllık İzin";
            }
        });
        let firstEntryTime;
        data.ar.find((elem) => {
            let findingActualAccess = elem.a.find((innerElem) => innerElem.s && innerElem.si);
            if (findingActualAccess) {
                firstEntryTime = {
                    firstStart: findingActualAccess.s,
                    firstStartId: findingActualAccess.si,
                };
            }
            else {
                let findingDummyAccess = elem.a.find((innerElem) => innerElem.s);
                if (findingDummyAccess) {
                    firstEntryTime = {
                        firstStart: findingDummyAccess.s,
                        firstStartId: findingDummyAccess.si,
                    };
                }
            }
        });
        if (firstEntryTime && firstEntryTime.firstStart) {
            if (firstEntryTime.firstStartId) {
                firstInText = (0, report_util_1.formatDateForExcel)(firstEntryTime.firstStart);
            }
            else {
                isDummyAccessLogForFirstIn = true;
            }
        }
        return {
            value: isDummyAccessLogForFirstIn ? "__:__" : firstInText,
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            fillColor: isDummyAccessLogForFirstIn ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : this.getColorText(data),
            numFmt: isDummyAccessLogForFirstIn ? null : generator_1.ReportGenerator.Constants.TimeFormat,
            alignment: { horizontal: "center" },
        };
    }
    lastOut(data) {
        let isDummyAccessLogForLastOut = false;
        let lastOutText;
        if (data.s.wd > 0 && data.s.pr == 0) {
            lastOutText = i18n_1.default.__({ phrase: "PDF-REPORT.not_arrived", locale: this._locale });
        }
        const holidayIdsOfTheDay = new Set(data.h.map((h) => h.h));
        if (Array.from(holidayIdsOfTheDay).length > 0) {
            lastOutText = "Resmi Tatil";
        }
        const leaveIdsOfTheDay = new Set(data.p.map((l) => l.pt));
        Array.from(leaveIdsOfTheDay).forEach((leave) => {
            if (leave === Constants.RaporLeaveId) {
                lastOutText = "Raporlu";
            }
            else {
                lastOutText = "Yıllık İzin";
            }
        });
        let lastLeaveTime;
        data.ar.reverse();
        data.ar.find((elem) => {
            elem.a.reverse();
            let findingActualAccess = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
            if (findingActualAccess) {
                lastLeaveTime = {
                    lastLeave: findingActualAccess.e,
                    lastLeaveId: findingActualAccess.ei,
                };
            }
            else {
                let findingActualAccess = elem.a.find((innerElem) => innerElem.e);
                if (findingActualAccess) {
                    lastLeaveTime = {
                        lastLeave: findingActualAccess.e,
                        lastLeaveId: findingActualAccess.ei,
                    };
                }
            }
        });
        if (lastLeaveTime && lastLeaveTime.lastLeave) {
            if (lastLeaveTime.lastLeaveId) {
                lastOutText = (0, report_util_1.formatDateForExcel)(lastLeaveTime.lastLeave);
            }
            else {
                isDummyAccessLogForLastOut = true;
            }
        }
        return {
            value: isDummyAccessLogForLastOut ? "__:__" : lastOutText,
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            fillColor: isDummyAccessLogForLastOut ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : this.getColorText(data),
            numFmt: isDummyAccessLogForLastOut ? null : generator_1.ReportGenerator.Constants.TimeFormat,
            alignment: { horizontal: "center" },
        };
    }
    missingDayCount(data) {
        let missingDayCount = 0;
        data.days.forEach((day) => {
            if (day.s.wd > 0 && day.s.pr == 0) {
                missingDayCount++;
            }
        });
        return {
            value: missingDayCount,
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            alignment: { horizontal: "center" },
        };
    }
    extraWorkDuration(data) {
        let extraWorkDuration = 0;
        data.days.forEach((day) => {
            extraWorkDuration += day.s.e ?? null;
        });
        return {
            value: extraWorkDuration ? this.minutesToHHMM(extraWorkDuration) : null,
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            alignment: { horizontal: "center" },
        };
    }
    missingWorkDuration(data) {
        let missingWorkDuration = 0;
        data.days.forEach((day) => {
            missingWorkDuration += day.s.m ?? null;
        });
        return {
            value: missingWorkDuration ? `-${this.minutesToHHMM(missingWorkDuration)}` : null,
            alignment: { horizontal: "center" },
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
        };
    }
    physicallyInRegionDuration(data) {
        let physicallyInRegionDuration = 0;
        data.days.forEach((day) => {
            physicallyInRegionDuration += day.s.pr ?? null;
        });
        return {
            value: physicallyInRegionDuration ? this.minutesToHHMM(physicallyInRegionDuration) : null,
            alignment: { horizontal: "center" },
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
        };
    }
    breakDuration(data) {
        let breakDuration = 0;
        data.days.forEach((day) => {
            breakDuration += day.s.ufb + day.s.afb + day.s.ucb + day.s.acb;
        });
        return {
            value: breakDuration ? this.minutesToHHMM(breakDuration) : null,
            alignment: { horizontal: "center" },
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
        };
    }
    leaveUsedDayCount(data) {
        let leaveUsedDayCount = 0;
        data.days.forEach((day) => {
            if (day.p.length > 0) {
                day.p.forEach((leave) => {
                    if (leave.pt !== Constants.RaporLeaveId) {
                        leaveUsedDayCount++;
                    }
                });
            }
        });
        return {
            value: leaveUsedDayCount,
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            alignment: { horizontal: "center" },
        };
    }
    raporUsedDayCount(data) {
        let raporUsedDayCount = 0;
        data.days.forEach((day) => {
            if (day.p.length > 0) {
                day.p.forEach((leave) => {
                    if (leave.pt === Constants.RaporLeaveId) {
                        raporUsedDayCount++;
                    }
                });
            }
        });
        return {
            value: raporUsedDayCount,
            border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            alignment: { horizontal: "center" },
        };
    }
    minutesToHHMM(totalMinutes) {
        const sign = totalMinutes < 0 ? "-" : "";
        const abs = Math.abs(totalMinutes);
        const h = Math.floor(abs / 60);
        const m = abs % 60;
        return `${sign}${String(h).padStart(2, "0")}:${String(m).padStart(2, "0")}`;
    }
    getColorText(data) {
        let colorText = undefined;
        if (data.s.pr && data.s.pr > 0) {
            if (data.s.ls > 0) {
                colorText = "FF0000";
            }
            else {
                if (data.s.tls && data.s.tls > 0) {
                    colorText = "FFC000";
                }
                else {
                    colorText = "FFFFFF";
                }
            }
        }
        else {
            if (data.s.wd && data.s.wd > 0) {
                if (data.s.ew && data.s.ew > 0) {
                    colorText = "FFFFFF";
                }
            }
            else {
                colorText = "FFFF00";
            }
        }
        return colorText;
    }
}
exports.MerzigoDailyCustomReport = MerzigoDailyCustomReport;
async function generateReport(request, locale) {
    let report = new MerzigoDailyCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
