"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateReport = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_enums_1 = require("../../../app.enums");
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
const dal_manager_1 = require("../../../dal/dal.manager");
const app_config_1 = require("../../../app.config");
const uuid_1 = __importDefault(require("uuid"));
const Cursor = require("pg-cursor");
const app_logs_1 = require("../../../app.logs");
const generator_1 = require("../../../business/report/generator");
const dal_access_psql_common_1 = require("../../../dal/access/psql/dal.access.psql.common");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
var Constants;
(function (Constants) {
    Constants.Days = {
        sunday: 0,
        monday: 1,
        tuesday: 2,
        wednesday: 3,
        thursday: 4,
        friday: 5,
        saturday: 6,
    };
    Constants.DayNames = ["Pz", "Pt", "Sl", "Çr", "Pr", "Cm", "Ct"];
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.EquelizedSheetName = "Aylık Mahsuplaşma Raporu";
    Constants.ExtraWorkSheetName = "Aylık Fazla Mesai Raporu";
    Constants.RawSheetName = "Ham Veri";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
class SageMahsuplasmaReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._data = {};
        this._request = request;
        this._filter = request.filter;
        let dateMonth = (0, moment_1.default)(this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateMonth).value, "YYYY-MM");
        this._organizationId = request.organizationId;
        this._startDate = dateMonth.clone().startOf("month");
        this._endDate = dateMonth.clone().endOf("month");
        this._requesterUserId = request.requesterUserId;
        this._reportId = this._filter.reportId;
        this._pool = dal_manager_1.dbManager.poolMain;
    }
    async generateEquelizedMonth() {
        app_logs_1.logger.debug("Generating Sage Mock xlsx");
        const fileId = uuid_1.default.v4();
        let filename = path_1.default.join(app_config_1.appConfig.tmpDirectory, fileId + ".xlsx");
        if (fs_1.default.existsSync(filename)) {
            fs_1.default.unlinkSync(filename);
        }
        const wb = new exceljs_1.default.Workbook();
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._organizationId, [this._requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        wb.creator = this._requestedUserCaption;
        wb.calcProperties.fullCalcOnLoad = true;
        const wsEquelized = wb.addWorksheet(Constants.EquelizedSheetName);
        const wsExtraWork = wb.addWorksheet(Constants.ExtraWorkSheetName);
        const wsRaw = wb.addWorksheet(Constants.RawSheetName);
        this.addFilterSummaryToSheet(wsRaw, 1, 1);
        this.addFilterSummaryToSheet(wsExtraWork, 1, 1);
        this.addFilterSummaryToSheet(wsEquelized, 1, 1);
        let weekAddressesRaw = this.initRaw(wsRaw, 5, 1);
        let weekAddressesExtraWork = this.initRaw(wsExtraWork, 5, 1);
        app_logs_1.logger.debug("Starting to fetch data...");
        await this.fetchAllData(async (rows) => {
            app_logs_1.logger.debug("Fetch data %s rows", rows.dailyData.length);
            for (const row of rows.dailyData) {
                let d = this._startDate.clone();
                let personData = row[0];
                let personUniqueId = personData[0].uniqueId;
                personData.sort((d1, d2) => new Date(d1.date).getTime() - new Date(d2.date).getTime());
                for (const singleDay of personData) {
                    let isHoliday = singleDay.isHoliday > 0;
                    let date = (0, moment_1.default)(singleDay.date);
                    if (d < date && !(personUniqueId in this._data)) {
                        let fullName = singleDay.name + " " + singleDay.surname;
                        this._data[personUniqueId] = {
                            fullname: fullName.toLocaleUpperCase(),
                            sicilNo: singleDay.sicilNo,
                            title: singleDay.title,
                            userId: singleDay.userId,
                            days: [
                                {
                                    missing: 0,
                                    over: 0,
                                    isHoliday: false,
                                },
                            ],
                        };
                        d = d.add(1, "day");
                        while (d < date) {
                            this._data[personUniqueId].days.push({
                                missing: 0,
                                over: 0,
                                isHoliday: false,
                            });
                            d = d.add(1, "day");
                        }
                        this._data[personUniqueId].days.push({
                            missing: parseFloat((singleDay.missingWork / 60).toFixed(2)),
                            over: parseFloat((singleDay.extraWork / 60).toFixed(2)),
                            isHoliday: isHoliday,
                        });
                    }
                    else if (d < date && personUniqueId in this._data) {
                        while (d.date() < date.date()) {
                            this._data[personUniqueId].days.push({
                                missing: 0,
                                over: 0,
                                isHoliday: false,
                            });
                            d = d.add(1, "day");
                        }
                        this._data[personUniqueId].days.push({
                            missing: parseFloat((singleDay.missingWork / 60).toFixed(2)),
                            over: parseFloat((singleDay.extraWork / 60).toFixed(2)),
                            isHoliday: isHoliday,
                        });
                    }
                    else if (!(personUniqueId in this._data)) {
                        let fullName = singleDay.name + " " + singleDay.surname;
                        this._data[personUniqueId] = {
                            fullname: fullName.toLocaleUpperCase(),
                            sicilNo: singleDay.sicilNo,
                            title: singleDay.title,
                            userId: singleDay.userId,
                            days: [
                                {
                                    missing: parseFloat((singleDay.missingWork / 60).toFixed(2)),
                                    over: parseFloat((singleDay.extraWork / 60).toFixed(2)),
                                    isHoliday: isHoliday,
                                },
                            ],
                        };
                    }
                    else {
                        this._data[personUniqueId].days.push({
                            missing: parseFloat((singleDay.missingWork / 60).toFixed(2)),
                            over: parseFloat((singleDay.extraWork / 60).toFixed(2)),
                            isHoliday: isHoliday,
                        });
                    }
                    d = d.add(1, "day");
                }
                if (d <= this._endDate) {
                    while (d <= this._endDate) {
                        this._data[personUniqueId].days.push({
                            missing: 0,
                            over: 0,
                            isHoliday: false,
                        });
                        d = d.add(1, "day");
                    }
                }
            }
            return Promise.resolve();
        });
        let date = this._startDate.clone().add(-1, "day").endOf("day");
        let begin = this._startDate.clone().startOf("year");
        if (date.isSame(begin, "year")) {
            this._previousMonthsData = await dal_manager_1.dbManager.accessPacs2.getMonthlyReport(this._organizationId, this._requesterUserId, {
                dateFilter: {
                    group: 4,
                    range: {
                        startDateTime: begin.toDate(),
                        endDateTime: date.toDate(),
                    },
                },
            });
        }
        app_logs_1.logger.debug("End of fetching data...");
        this.initEquilized(wsEquelized, 5, 1);
        app_logs_1.logger.debug("init finished");
        this.setData(wsRaw, wsExtraWork, wsEquelized, weekAddressesExtraWork);
        app_logs_1.logger.debug("data det finished");
        await wb.xlsx.writeFile(filename);
        app_logs_1.logger.debug("File is ready");
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: fileId,
            filePath: filename,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "Aylık Mahsuplaşma",
            reportContainsEmptyData: this._request.reportTemplateId ? Object.keys(this._data).length === 0 : undefined,
        };
    }
    async fetchAllData(onData) {
        const client = await this._pool.connect();
        app_logs_1.logger.debug("Connected...");
        let query = `
        SELECT json_agg(
            json_build_object(
                'date', (ed."date")::date,
                'uniqueId', (uop."uniqueId")::text,
                'sicilNo', uop."extensionFields"->'personnelNumber',
                'name', (uop."name")::text,
                'surname', (uop."surname")::text,
                'userId', (uop."userId")::text,
                'title', uop."extensionFields"->'title',
                'extraWork', (ed.data->'s'->>'e')::integer,
                'missingWork', (ed.data->'s'->>'m')::integer,
                'isHoliday', (ed.data->'s'->>'eh')::integer
                )) as "dailyData"
        FROM "${this._organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" ed
        INNER JOIN "${this._organizationId}"."users" u ON ed."userId" = u."id" AND u."deletedAt" IS NULL
        INNER JOIN "${this._organizationId}"."userOrganizations" uo ON uo."userId" = ed."userId" AND uo."organizationId" = ed."organizationId" AND uo."deletedAt" IS NULL
        INNER JOIN "${this._organizationId}"."userOrganizationProfiles" uop ON uop."userOrganizationId" = uo.id AND uop."deletedAt" IS NULL
        `;
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            requesterUserId: this._requesterUserId,
            organizationId: this._organizationId,
            idBasedUserFilter: {},
            requiredOrganizationUnitWidePermissions: [],
            requiredOrganizationWidePermissions: [],
            bindingKeys: [],
        });
        let queryParams = userFilter.bindingKeys;
        let queryParamIndex = userFilter.bindingKeys.length;
        query += `
            INNER JOIN (${userFilter.query}) userFilter ON userFilter."userId" = ed."userId"
            WHERE ed."organizationId" = $${++queryParamIndex} AND ed.date >= $${++queryParamIndex} AND ed.date <= $${++queryParamIndex}
            GROUP BY  uop."uniqueId";
        `;
        queryParams.push(this._organizationId, this._startDate.toDate(), this._endDate.toDate());
        const cursor = client.query(new Cursor(query, queryParams, { rowMode: "array" }));
        let rows = {
            dailyData: [],
        };
        while (true) {
            try {
                rows.dailyData = await new Promise((resolve, reject) => {
                    cursor.read(100, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error(error);
                process.exit(1);
            }
            if (rows.dailyData.length < 100) {
                break;
            }
        }
        app_logs_1.logger.debug("Cursor end");
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
            process.exit(1);
        }
        app_logs_1.logger.debug("Cursor closed");
    }
    addFilterSummaryToSheet(ws, startRowIndex, startColIndex) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, "TÜBİTAK SAGE Çalışma Özeti", border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, "Başlangıç Tarihi", border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, {
            formula: "DATE(" + this._startDate.year() + "," + (this._startDate.month() + 1) + "," + this._startDate.date() + ")",
        }, border, { italic: true }, Constants.DateFormat);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex, "Bitiş Tarihi", border);
        this.addFilterCell(ws, startRowIndex + 2, startColIndex + 1, {
            formula: "DATE(" + this._endDate.year() + "," + (this._endDate.month() + 1) + "," + this._endDate.date() + ")",
        }, border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 1);
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        if (params.alignment) {
            c.alignment = params.alignment;
        }
        if (params.border) {
            c.border = params.border;
        }
    }
    initRaw(wsExtraWork, startRowIndex, startColIndex) {
        let weeks = [];
        let d = this._startDate.clone();
        let columns = [{ width: 16 }, { width: 32 }, { width: 32 }];
        this.addInitCell(wsExtraWork, startRowIndex, startColIndex, "Sicil No", Constants.Styling.AllThin);
        wsExtraWork.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex++);
        this.addInitCell(wsExtraWork, startRowIndex, startColIndex, "Adı Soyadı", Constants.Styling.AllThin);
        wsExtraWork.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex++);
        this.addInitCell(wsExtraWork, startRowIndex, startColIndex, "Ünvan", Constants.Styling.AllThin);
        wsExtraWork.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex);
        wsExtraWork.autoFilter = {
            from: "A5",
            to: "C5",
        };
        let currentWeek = 1;
        let mergeWeekStart = 0;
        while (d.isSameOrBefore(this._endDate)) {
            columns.push({ width: 4 });
            columns.push({ width: 4 });
            let week = {};
            if (columns.length === 5 || d.weekday() === Constants.Days.monday) {
                this.addInitCell(wsExtraWork, startRowIndex, columns.length - 1, currentWeek++ + ". Hafta", Constants.Styling.AllThin);
                if (mergeWeekStart > 0) {
                    wsExtraWork.mergeCells(startRowIndex, mergeWeekStart, startRowIndex, columns.length - 7);
                }
                mergeWeekStart = columns.length - 1;
            }
            this.addInitCell(wsExtraWork, startRowIndex + 1, columns.length - 1, d.date(), Constants.Styling.AllThin);
            wsExtraWork.mergeCells(startRowIndex + 1, columns.length - 1, startRowIndex + 1, columns.length);
            this.addInitCell(wsExtraWork, startRowIndex + 2, columns.length - 1, Constants.DayNames[d.weekday()], Constants.Styling.AllThin);
            wsExtraWork.mergeCells(startRowIndex + 2, columns.length - 1, startRowIndex + 2, columns.length);
            this.addInitCell(wsExtraWork, startRowIndex + 3, columns.length - 1, "E", Constants.Styling.AllThin);
            this.addInitCell(wsExtraWork, startRowIndex + 3, columns.length, "F", Constants.Styling.AllThin);
            if (d.weekday() === Constants.Days.sunday || d.isSame(this._endDate, "day")) {
                columns.push({ width: 8 });
                this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Hafta Sonu", Constants.Styling.AllThin);
                week.weekendOverWork = wsExtraWork.getColumn(columns.length).letter;
                wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
                columns.push({ width: 8 });
                this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Hafta İçi", Constants.Styling.AllThin);
                week.weekdayOverWork = wsExtraWork.getColumn(columns.length).letter;
                wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
                columns.push({ width: 8 });
                this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Resmi Tatil", Constants.Styling.AllThin);
                week.holidayWork = wsExtraWork.getColumn(columns.length).letter;
                wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
                columns.push({ width: 8 });
                this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Fazla Süreli Çalışma", Constants.Styling.AllThin);
                week.exceedOverWork = wsExtraWork.getColumn(columns.length).letter;
                wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
                columns.push({ width: 8 });
                this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Eksik Çalışma", Constants.Styling.AllThin);
                week.missingWork = wsExtraWork.getColumn(columns.length).letter;
                wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
                weeks.push(week);
            }
            d = d.add(1, "day");
        }
        if (mergeWeekStart > 0) {
            wsExtraWork.mergeCells(startRowIndex, mergeWeekStart, startRowIndex, columns.length - 5);
        }
        let montlyTotals = {};
        this.addInitCell(wsExtraWork, startRowIndex - 1, columns.length + 1, "Aylık Toplam", Constants.Styling.AllThin);
        wsExtraWork.mergeCells(startRowIndex - 1, columns.length + 1, startRowIndex - 1, columns.length + 5);
        columns.push({ width: 10 });
        this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Hafta Sonu", Constants.Styling.AllThin);
        montlyTotals.weekendOverWork = wsExtraWork.getColumn(columns.length).letter;
        wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 10 });
        this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Hafta İçi", Constants.Styling.AllThin);
        montlyTotals.weekdayOverWork = wsExtraWork.getColumn(columns.length).letter;
        wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 10 });
        this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Resmi Tatil", Constants.Styling.AllThin);
        montlyTotals.holidayWork = wsExtraWork.getColumn(columns.length).letter;
        wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 10 });
        this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Fazla Süreli Çalışma", Constants.Styling.AllThin);
        montlyTotals.exceedOverWork = wsExtraWork.getColumn(columns.length).letter;
        wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 10 });
        this.addInitCell(wsExtraWork, startRowIndex, columns.length, "Eksik Çalışma", Constants.Styling.AllThin);
        montlyTotals.missingWork = wsExtraWork.getColumn(columns.length).letter;
        wsExtraWork.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        wsExtraWork.columns = columns;
        wsExtraWork.views = [{ state: "frozen", xSplit: 3, ySplit: 8, activeCell: "D9", showGridLines: false }];
        return montlyTotals;
    }
    initEquilized(wsEquilized, startRowIndex, startColIndex) {
        let d = this._startDate.clone();
        let columns = [{ width: 16 }, { width: 32 }, { width: 32 }];
        this.addInitCell(wsEquilized, startRowIndex, startColIndex, "Sicil No", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex++);
        this.addInitCell(wsEquilized, startRowIndex, startColIndex, "Adı Soyadı", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex++);
        this.addInitCell(wsEquilized, startRowIndex, startColIndex, "Ünvan", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex++);
        wsEquilized.autoFilter = {
            from: "A5",
            to: "C5",
        };
        columns.push({ width: 15 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Önceki Aylardan Devreden Toplam Mazeretsiz Eksik Çalışma", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Net Hafta Sonu", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Mahsuplaşılan Hafta Sonu", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Net Hafta İçi", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Mahsuplaşılan Hafta İçi", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Net Resmi Tatil", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Mahsuplaşılan Resmi Tatil", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Net Fazla Süreli Çalışma", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Mahsuplaşılan Fazla Süreli Çalışma", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Net Eksik Çalışma", Constants.Styling.AllThin);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        columns.push({ width: 12 });
        this.addInitCell(wsEquilized, startRowIndex, columns.length, "Mahsuplaşılan Eksik Çalışma", Constants.Styling.AllThinButRightThick);
        wsEquilized.mergeCells(startRowIndex, columns.length, startRowIndex + 3, columns.length);
        this.addInitCell(wsEquilized, startRowIndex - 1, columns.length - 9, "Aylık Mahsuplaşma", Constants.Styling.AllThinButRightThick);
        wsEquilized.mergeCells(startRowIndex - 1, columns.length - 9, startRowIndex - 1, columns.length);
        wsEquilized.columns = columns;
        wsEquilized.views = [{ state: "frozen", xSplit: 3, ySplit: 8, activeCell: "D9", showGridLines: false }];
    }
    setData(wsRaw, wsExtraWork, wsEquelized, monthlyColumn) {
        let sortedIds = Object.keys(this._data).sort();
        let colIndex = 0;
        let rowIndex = 9;
        let holidayCells = [];
        for (let i = 0; i < sortedIds.length; i++) {
            let key = sortedIds[i];
            let data = this._data[sortedIds[i]];
            let row = wsRaw.getRow(rowIndex++);
            let values = [data.sicilNo, data.fullname, data.title];
            colIndex = 3;
            let monthlyFormulaOverWeekend = [];
            let monthlyFormulaOverWeekday = [];
            let monthlyFormulaMissingWork = [];
            let monthlyFormulaExceedOverWork = [];
            let monthlyFormulaOverHoliday = [];
            let formulaOverWeekend = [];
            let formulaOverWeekday = [];
            let formulaMissingWork = [];
            let formulaExceedOverWork = [];
            let formulaOverHoliday = [];
            let d = this._startDate.clone();
            let dayIndex = 0;
            while (d.isSameOrBefore(this._endDate)) {
                let weekday = d.weekday();
                let dayData = data.days[dayIndex];
                values[colIndex++] = dayData.missing;
                values[colIndex++] = dayData.over;
                if (dayData.isHoliday) {
                    let address = wsRaw.getColumn(colIndex).letter + (rowIndex - 1);
                    formulaOverHoliday.push(address);
                    formulaExceedOverWork.push(address);
                    holidayCells.push(address);
                    holidayCells.push(wsRaw.getColumn(colIndex - 1).letter + (rowIndex - 1));
                }
                else if (weekday === Constants.Days.saturday || weekday === Constants.Days.sunday) {
                    let address = wsRaw.getColumn(colIndex).letter + (rowIndex - 1);
                    formulaOverWeekend.push(address);
                    formulaExceedOverWork.push(address);
                }
                else if (weekday === Constants.Days.monday ||
                    weekday === Constants.Days.tuesday ||
                    weekday === Constants.Days.wednesday ||
                    weekday === Constants.Days.thursday ||
                    weekday === Constants.Days.friday) {
                    let address = wsRaw.getColumn(colIndex).letter + (rowIndex - 1);
                    formulaOverWeekday.push(address);
                    formulaExceedOverWork.push(address);
                    formulaMissingWork.push(wsRaw.getColumn(colIndex - 1).letter + (rowIndex - 1));
                }
                if (d.weekday() === Constants.Days.sunday || d.isSame(this._endDate, "day")) {
                    if (formulaOverWeekend.length > 0) {
                        values[colIndex] = {
                            formula: formulaOverWeekend.join("+"),
                        };
                    }
                    colIndex++;
                    if (formulaOverWeekday.length > 0) {
                        values[colIndex] = {
                            formula: formulaOverWeekday.join("+"),
                        };
                    }
                    colIndex++;
                    if (formulaOverHoliday.length > 0) {
                        values[colIndex] = {
                            formula: formulaOverHoliday.join("+"),
                        };
                    }
                    holidayCells.push(wsRaw.getColumn(colIndex + 1).letter + (rowIndex - 1));
                    colIndex++;
                    if (formulaExceedOverWork.length > 0) {
                        values[colIndex] = {
                            formula: formulaExceedOverWork.join("+"),
                        };
                    }
                    colIndex++;
                    if (formulaMissingWork.length > 0) {
                        values[colIndex] = {
                            formula: formulaMissingWork.join("+"),
                        };
                    }
                    colIndex++;
                    monthlyFormulaOverWeekend = monthlyFormulaOverWeekend.concat(formulaOverWeekend);
                    monthlyFormulaOverWeekday = monthlyFormulaOverWeekday.concat(formulaOverWeekday);
                    monthlyFormulaMissingWork = monthlyFormulaMissingWork.concat(formulaMissingWork);
                    monthlyFormulaExceedOverWork = monthlyFormulaExceedOverWork.concat(formulaExceedOverWork);
                    monthlyFormulaOverHoliday = monthlyFormulaOverHoliday.concat(formulaOverHoliday);
                    formulaOverWeekend = [];
                    formulaOverWeekday = [];
                    formulaMissingWork = [];
                    formulaOverHoliday = [];
                    formulaExceedOverWork = [];
                }
                d = d.add(1, "day");
                dayIndex++;
            }
            if (monthlyFormulaOverWeekend.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaOverWeekend.join("+"),
                };
            }
            colIndex++;
            if (monthlyFormulaOverWeekday.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaOverWeekday.join("+"),
                };
            }
            colIndex++;
            if (monthlyFormulaOverHoliday.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaOverHoliday.join("+"),
                };
            }
            holidayCells.push(wsRaw.getColumn(colIndex + 1).letter + (rowIndex - 1));
            colIndex++;
            if (monthlyFormulaExceedOverWork.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaExceedOverWork.join("+"),
                };
            }
            colIndex++;
            if (monthlyFormulaMissingWork.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaMissingWork.join("+"),
                };
            }
            row.values = values;
        }
        colIndex = 0;
        rowIndex = 9;
        holidayCells = [];
        for (let i = 0; i < sortedIds.length; i++) {
            let key = sortedIds[i];
            let data = this._data[sortedIds[i]];
            let row = wsExtraWork.getRow(rowIndex++);
            let values = [data.sicilNo, data.fullname, data.title];
            colIndex = 3;
            let monthlyFormulaOverWeekend = [];
            let monthlyFormulaOverWeekday = [];
            let monthlyFormulaMissingWork = [];
            let monthlyFormulaExceedOverWork = [];
            let monthlyFormulaOverHoliday = [];
            let formulaOverWeekend = [];
            let formulaOverWeekday = [];
            let formulaMissingWork = [];
            let formulaExceedOverWork = [];
            let formulaOverHoliday = [];
            let d = this._startDate.clone();
            let dayIndex = 0;
            while (d.isSameOrBefore(this._endDate)) {
                let weekday = d.weekday();
                let dayData = data.days[dayIndex];
                values[colIndex++] = dayData.missing;
                if (dayData.isHoliday) {
                    let address = wsExtraWork.getColumn(colIndex).letter + (rowIndex - 1);
                    let addressExtra = wsExtraWork.getColumn(colIndex + 1).letter + (rowIndex - 1);
                    values[colIndex++] = {
                        formula: "IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + " >8, 8," + "'" + Constants.RawSheetName + "'!" + addressExtra + ")",
                    };
                    formulaOverHoliday.push("IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + ">8,8," + "'" + Constants.RawSheetName + "'!" + addressExtra + ")");
                    formulaExceedOverWork.push("IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + "<8,0," + "'" + Constants.RawSheetName + "'!" + addressExtra + "-8)");
                    holidayCells.push(address);
                    holidayCells.push(addressExtra);
                }
                else if (weekday === Constants.Days.saturday || weekday === Constants.Days.sunday) {
                    let address = wsExtraWork.getColumn(colIndex).letter + (rowIndex - 1);
                    let addressExtra = wsExtraWork.getColumn(colIndex + 1).letter + (rowIndex - 1);
                    values[colIndex++] = {
                        formula: "IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + " >8, 8," + "'" + Constants.RawSheetName + "'!" + addressExtra + ")",
                    };
                    formulaOverWeekend.push("IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + ">8,8," + "'" + Constants.RawSheetName + "'!" + addressExtra + ")");
                    formulaExceedOverWork.push("IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + "<8,0," + "'" + Constants.RawSheetName + "'!" + addressExtra + "-8)");
                }
                else if (weekday === Constants.Days.monday ||
                    weekday === Constants.Days.tuesday ||
                    weekday === Constants.Days.wednesday ||
                    weekday === Constants.Days.thursday ||
                    weekday === Constants.Days.friday) {
                    let address = wsExtraWork.getColumn(colIndex).letter + (rowIndex - 1);
                    let addressExtra = wsExtraWork.getColumn(colIndex + 1).letter + (rowIndex - 1);
                    values[colIndex++] = {
                        formula: "IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + " >3, 3," + "'" + Constants.RawSheetName + "'!" + addressExtra + ")",
                    };
                    formulaOverWeekday.push("IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + ">3,3," + "'" + Constants.RawSheetName + "'!" + addressExtra + ")");
                    formulaExceedOverWork.push("IF(" + "'" + Constants.RawSheetName + "'!" + addressExtra + "<3,0," + "'" + Constants.RawSheetName + "'!" + addressExtra + "-3)");
                    formulaMissingWork.push(address);
                }
                if (d.weekday() === Constants.Days.sunday || d.isSame(this._endDate, "day")) {
                    if (formulaOverWeekend.length > 0) {
                        values[colIndex] = {
                            formula: formulaOverWeekend.join("+"),
                        };
                    }
                    colIndex++;
                    if (formulaOverWeekday.length > 0) {
                        values[colIndex] = {
                            formula: "IF(" + formulaOverWeekday.join("+") + " <5, 0," + formulaOverWeekday.join("+") + "-5)",
                        };
                    }
                    colIndex++;
                    if (formulaOverHoliday.length > 0) {
                        values[colIndex] = {
                            formula: formulaOverHoliday.join("+"),
                        };
                    }
                    holidayCells.push(wsExtraWork.getColumn(colIndex + 1).letter + (rowIndex - 1));
                    colIndex++;
                    if (formulaExceedOverWork.length > 0) {
                        if (formulaOverWeekday.length > 0) {
                            values[colIndex] = {
                                formula: "+IF(" + formulaOverWeekday.join("+") + "<5," + formulaOverWeekday.join("+") + ",5)",
                            };
                        }
                    }
                    colIndex++;
                    if (formulaMissingWork.length > 0) {
                        values[colIndex] = {
                            formula: formulaMissingWork.join("+"),
                        };
                    }
                    colIndex++;
                    monthlyFormulaOverWeekend = monthlyFormulaOverWeekend.concat(formulaOverWeekend);
                    monthlyFormulaOverWeekday = monthlyFormulaOverWeekday.concat(formulaOverWeekday);
                    monthlyFormulaMissingWork = monthlyFormulaMissingWork.concat(formulaMissingWork);
                    monthlyFormulaExceedOverWork = monthlyFormulaExceedOverWork.concat(formulaExceedOverWork);
                    monthlyFormulaOverHoliday = monthlyFormulaOverHoliday.concat(formulaOverHoliday);
                    formulaOverWeekend = [];
                    formulaOverWeekday = [];
                    formulaMissingWork = [];
                    formulaOverHoliday = [];
                    formulaExceedOverWork = [];
                }
                d = d.add(1, "day");
                dayIndex++;
            }
            if (monthlyFormulaOverWeekend.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaOverWeekend.join("+"),
                };
            }
            colIndex++;
            if (monthlyFormulaOverWeekday.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaOverWeekday.join("+"),
                };
            }
            colIndex++;
            if (monthlyFormulaOverHoliday.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaOverHoliday.join("+"),
                };
            }
            holidayCells.push(wsExtraWork.getColumn(colIndex + 1).letter + (rowIndex - 1));
            colIndex++;
            if (monthlyFormulaExceedOverWork.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaExceedOverWork.join("+"),
                };
            }
            colIndex++;
            if (monthlyFormulaMissingWork.length > 0) {
                values[colIndex] = {
                    formula: monthlyFormulaMissingWork.join("+"),
                };
            }
            row.values = values;
        }
        rowIndex = 9;
        for (let i = 0; i < sortedIds.length; i++) {
            let key = sortedIds[i];
            let data = this._data[sortedIds[i]];
            let colIndex = 4;
            wsEquelized.getCell(rowIndex, 1).value = data.sicilNo;
            wsEquelized.getCell(rowIndex, 2).value = data.fullname;
            wsEquelized.getCell(rowIndex, 3).value = data.title;
            let userPreviousData = this._previousMonthsData?.items.find((i) => i.user.id === data.userId);
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: userPreviousData && userPreviousData.editedWorkingDurations && userPreviousData.editedWorkingDurations.missedWorkDuration
                    ? parseFloat((userPreviousData.editedWorkingDurations.missedWorkDuration / 60).toFixed(2))
                    : 0,
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "'" + Constants.ExtraWorkSheetName + "'!" + monthlyColumn.weekendOverWork + rowIndex,
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "IF(OR(" +
                        wsEquelized.getCell(rowIndex, colIndex + 1).address +
                        ">0" +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex + 5).address +
                        ">0)," +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekendOverWork +
                        rowIndex +
                        "," +
                        "IF((" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekdayOverWork +
                        rowIndex +
                        ")>" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekendOverWork +
                        rowIndex +
                        ",0" +
                        "," +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekendOverWork +
                        rowIndex +
                        "-" +
                        "(" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekdayOverWork +
                        rowIndex +
                        "))" +
                        ")",
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "'" + Constants.ExtraWorkSheetName + "'!" + monthlyColumn.weekdayOverWork + rowIndex,
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "IF(" +
                        wsEquelized.getCell(rowIndex, colIndex + 3).address +
                        ">0," +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekdayOverWork +
                        rowIndex +
                        "," +
                        "IF((" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        ")>" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekdayOverWork +
                        rowIndex +
                        ",0" +
                        "," +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekdayOverWork +
                        rowIndex +
                        "-" +
                        "(" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        "))" +
                        ")",
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "'" + Constants.ExtraWorkSheetName + "'!" + monthlyColumn.holidayWork + rowIndex,
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "IF(OR(" +
                        wsEquelized.getCell(rowIndex, colIndex - 5).address +
                        ">0," +
                        wsEquelized.getCell(rowIndex, colIndex + 1).address +
                        ">0" +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex - 3).address +
                        ">0)," +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.holidayWork +
                        rowIndex +
                        "," +
                        "IF((" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekdayOverWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekendOverWork +
                        rowIndex +
                        ")>" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.holidayWork +
                        rowIndex +
                        ",0" +
                        "," +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.holidayWork +
                        rowIndex +
                        "-" +
                        "(" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekdayOverWork +
                        rowIndex +
                        "-" +
                        "'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.weekendOverWork +
                        rowIndex +
                        "))" +
                        ")",
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "'" + Constants.ExtraWorkSheetName + "'!" + monthlyColumn.exceedOverWork + rowIndex,
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "IF('" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        "<'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        "," +
                        0 +
                        ",'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.exceedOverWork +
                        rowIndex +
                        "-'" +
                        Constants.ExtraWorkSheetName +
                        "'!" +
                        monthlyColumn.missingWork +
                        rowIndex +
                        ")",
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "'" + Constants.ExtraWorkSheetName + "'!" + monthlyColumn.missingWork + rowIndex + " + D" + rowIndex,
                },
                ws: wsEquelized,
            });
            this.setDataCell({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: {
                    formula: "IF(SUM(" +
                        wsEquelized.getCell(rowIndex, colIndex - 4).address +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex - 6).address +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex - 8).address +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex - 10).address +
                        ")>" +
                        wsEquelized.getCell(rowIndex, colIndex - 2).address +
                        ",0," +
                        wsEquelized.getCell(rowIndex, colIndex - 2).address +
                        "-" +
                        "SUM(" +
                        wsEquelized.getCell(rowIndex, colIndex - 4).address +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex - 6).address +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex - 8).address +
                        "," +
                        wsEquelized.getCell(rowIndex, colIndex - 10).address +
                        ")" +
                        ")",
                },
                ws: wsEquelized,
            });
            rowIndex++;
        }
        for (const cellHoliday of holidayCells) {
            wsRaw.getCell(cellHoliday).fill = {
                type: "pattern",
                pattern: "darkGray",
                fgColor: { argb: "FFFFED30" },
            };
            wsExtraWork.getCell(cellHoliday).fill = {
                type: "pattern",
                pattern: "darkGray",
                fgColor: { argb: "FFFFED30" },
            };
        }
        this.arrangeBordersRaw(wsRaw, colIndex + 1, rowIndex, false);
        this.arrangeBordersRaw(wsExtraWork, colIndex + 1, rowIndex, false);
        this.arrangeBordersRaw(wsEquelized, 1 * 10 + 4, rowIndex, true);
    }
    arrangeBordersRaw(wsExtraWork, endColIndex, endRowIndex, sheetControl) {
        for (let r = 9; r < endRowIndex; r++) {
            for (let i = 1; i <= endColIndex; i++) {
                let c = wsExtraWork.getCell(r, i);
                if (sheetControl) {
                    if (i % 10 === 4 && i > 4) {
                        c.border = {
                            bottom: {
                                style: r === endRowIndex - 1 ? "thin" : "dotted",
                            },
                            right: {
                                style: "thick",
                            },
                        };
                    }
                    else {
                        c.border = {
                            bottom: {
                                style: r === endRowIndex - 1 ? "thin" : "dotted",
                            },
                            right: {
                                style: "thin",
                            },
                        };
                    }
                }
                else {
                    c.border = {
                        bottom: {
                            style: r === endRowIndex - 1 ? "thin" : "dotted",
                        },
                        right: {
                            style: "thin",
                        },
                    };
                }
            }
        }
    }
}
async function generateReport(request, locale) {
    let report = new SageMahsuplasmaReport(request, locale);
    return await report.generateEquelizedMonth();
}
exports.generateReport = generateReport;
