"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TurquoiseYachtExtraWorkCustomReport = exports.generateReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const controller_report_1 = require("../../../api/pacs/v2/controllers/controller.report");
var Constants;
(function (Constants) {
    Constants.TimeFormat = "HH:mm";
    Constants.DateFormat = "dd.mm.yyyy";
    Constants.DateTimeFormat = "dd.mm.yyyy HH:mm";
    let Styling;
    (function (Styling) {
        Styling.AllThin = {
            right: {
                style: "thin",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.Dotted = {
            right: {
                style: "dotted",
            },
            left: {
                style: "dotted",
            },
            bottom: {
                style: "dotted",
            },
            top: {
                style: "dotted",
            },
        };
        Styling.AllThinButRightThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
        };
        Styling.AllThick = {
            right: {
                style: "thick",
            },
            left: {
                style: "thick",
            },
            bottom: {
                style: "thick",
            },
            top: {
                style: "thick",
            },
        };
    })(Styling = Constants.Styling || (Constants.Styling = {}));
})(Constants || (Constants = {}));
async function generateReport(request, locale) {
    let report = new TurquoiseYachtExtraWorkCustomReport(request, locale);
    return report.generateReport();
}
exports.generateReport = generateReport;
class TurquoiseYachtExtraWorkCustomReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._request = request;
        this._filter = request.filter;
        this._options = {
            dateRange: {
                startDateTime: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.startDay)
                    .startOf("day")
                    .toDate(),
                endDateTime: (0, moment_1.default)(this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.DateRange).value.endDay)
                    .endOf("day")
                    .toDate(),
            },
            userFilter: {
                organizationUnitIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnit).value,
                organizationUnitHierachially: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.OrganizationUnitHierarchically).value,
                userGroupIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserGroup).value,
                userIds: this._filter.filters.find((i) => i.type === app_enums_1.enums.CustomOrganizationReportFilterType.UserIdFromFreeSearch).value,
                workPlanIds: this._filter.filters.find((f) => f.type === app_enums_1.enums.CustomOrganizationReportFilterType.WorkPlan).value,
            },
            organizationId: request.organizationId,
            requesterUserId: request.requesterUserId,
            locale: locale,
        };
    }
    async generateReport() {
        app_logs_1.logger.info("Generating custom report");
        if ((0, moment_1.default)(this._options.dateRange.endDateTime).isSame((0, moment_1.default)(), "day")) {
            await (0, controller_report_1.refreshUsersSummaryForToday)({
                requesterUserId: this._options.requesterUserId,
                applyOrganizationUnitFilterHierarchically: this._options.userFilter.organizationUnitHierachially,
                userGroupIds: this._options.userFilter.userGroupIds,
                organizationId: this._options.organizationId,
                organizationUnitIds: this._options.userFilter.organizationUnitIds,
                userIds: this._options.userFilter.userIds,
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
        }
        const dbResult = (await dal_manager_1.dbManager.accessPacs2.getFirstInLastOutSummaryForUsers(this._options.organizationId, this._options.requesterUserId, {
            startDate: this._options.dateRange.startDateTime,
            endDate: this._options.dateRange.endDateTime,
            organizationUnitIds: this._options.userFilter.organizationUnitIds,
            organizationUnitHierarchically: this._options.userFilter.organizationUnitHierachially,
            userGroupIds: this._options.userFilter.userGroupIds,
            userIds: this._options.userFilter.userIds,
            workPlanIds: this._options.userFilter.workPlanIds,
            status: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
            sortType: dal_constants_1.DalConstants.WorkSummarySortType.Name,
            sortOrder: "ASC",
            workStatusFilter: [],
            take: null,
            skip: null,
        })).items;
        this._userIds = dbResult.map((u) => u.userId);
        this._userDaysDetails = [];
        this._maxExtraWorkDays = 0;
        for (const userId of this._userIds) {
            const item = {
                userId: userId,
                days: await dal_manager_1.dbManager.accessPacs2.workSummaryOfEmployeeInDateRange(this._options.requesterUserId, this._options.organizationId, {
                    dateRange: this._options.dateRange,
                    paginationRequest: { take: 365, skip: 0 },
                    userId: userId,
                    sortType: dal_constants_1.DalConstants.WorkSummarySortType.Date,
                    sortOrder: "ASC",
                }),
            };
            this._userDaysDetails.push(item);
            const count = item.days.items.filter((d) => d.extraWorkDuration).length;
            this._maxExtraWorkDays = count > this._maxExtraWorkDays ? count : this._maxExtraWorkDays;
        }
        this._filePath = await this.generate();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.CustomReport,
            sendFileInAttachments: undefined,
            customReportName: "TURQUOISE YACHT & TURQUOISE SHIPYARD EMPLOYEE EXTRA WORK REPORT",
        };
    }
    async generate() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this.addFilterToSheet(this._wsGeneral, 1, 1);
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                " İZİN FORMU RAPORU " +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                (0, moment_1.default)(this._options.dateRange.startDateTime).locale(this._locale).format("LL") +
                "\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Requester user not found");
        }
        if (requestedUserCaption[0].captionLines && requestedUserCaption[0].captionLines[0] && requestedUserCaption[0].captionLines[0].text !== null) {
            this._requestedUserCaption = requestedUserCaption[0].captionLines[0].text[0];
        }
        else {
            this._requestedUserCaption = "";
        }
        this._wb.creator = this._requestedUserCaption;
        for (const item of this._userIds) {
            this.addGeneralRow(item);
        }
        this.addSumRow();
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.info("File is ready -> " + this._filePath);
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Sıra", Constants.Styling.AllThin);
        columns.push({ width: 10 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Sicil No", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "Adı Soyadı", Constants.Styling.AllThin);
        columns.push({ width: 35 });
        for (let i = 0; i < this._maxExtraWorkDays; i++) {
            this.addInitCell(wsRaw, startRowIndex, startColIndex++, "FM Tarihi", Constants.Styling.AllThin);
            columns.push({ width: 25 });
            this.addInitCell(wsRaw, startRowIndex, startColIndex++, "FM", Constants.Styling.AllThin);
            columns.push({ width: 10 });
        }
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "HAFTA İÇİ", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "HAFTA SONU", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "BAYRAM MESAİSİ", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell(wsRaw, startRowIndex, startColIndex++, "TOPLAM MESAİ", Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    addInitCell(ws, row, col, value, border) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
    }
    addGeneralRow(userId) {
        let colIndex = 1;
        const userDays = this._userDaysDetails.find((d) => d.userId === userId).days;
        const userCaption = userDays.items[0].userCaption;
        if (!(userDays?.items?.length > 0) || !(userCaption?.length > 0) || !userDays.items.find((d) => d.extraWorkDuration > 0)) {
            return;
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (this._generalRowIndex - 5),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (userCaption[1]?.text.join(" ") ?? ""),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: (userCaption[0]?.text.join(" ") ?? "").toUpperCase(),
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
        });
        let count = 0;
        let weekdaysFormula = [];
        let weekendsFormula = [];
        let holidaysFormula = [];
        for (const day of userDays.items) {
            if (day.extraWorkDuration) {
                if (day.expectedHolidayDuration) {
                    holidaysFormula.push(this._wsGeneral.getColumn(colIndex + 1).letter + this._generalRowIndex);
                }
                else if ((0, moment_1.default)(day.date).isoWeekday() > app_enums_1.enums.DayOfWeekForWorking.Friday) {
                    weekendsFormula.push(this._wsGeneral.getColumn(colIndex + 1).letter + this._generalRowIndex);
                }
                else {
                    weekdaysFormula.push(this._wsGeneral.getColumn(colIndex + 1).letter + this._generalRowIndex);
                }
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex++,
                    value: ((0, moment_1.default)(day.date).locale(this._options.locale).format("LL") +
                        ", " +
                        (0, moment_1.default)(day.date).locale(this._options.locale).day((0, moment_1.default)(day.date).weekday()).format("dddd")),
                    ws: this._wsGeneral,
                    border: Constants.Styling.Dotted,
                    isTextRed: (0, moment_1.default)(day.date).isoWeekday() > app_enums_1.enums.DayOfWeekForWorking.Friday,
                });
                this.setDataCell({
                    rowIndex: this._generalRowIndex,
                    colIndex: colIndex++,
                    value: day.extraWorkDuration ? (day.extraWorkDuration / (24 * 60)) : null,
                    ws: this._wsGeneral,
                    border: Constants.Styling.Dotted,
                    numFmt: day.extraWorkDuration
                        ? '[<0.000694444]"";[<0.0415][m]" ' +
                            i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                            '";[h] "' +
                            i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                            ' " m "' +
                            i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                            '"'
                        : null,
                    isTextRed: (0, moment_1.default)(day.date).isoWeekday() > app_enums_1.enums.DayOfWeekForWorking.Friday,
                });
                count++;
            }
        }
        for (let i = count; i < this._maxExtraWorkDays; i++) {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: "",
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                value: "",
                ws: this._wsGeneral,
                border: Constants.Styling.Dotted,
            });
        }
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: weekdaysFormula.length > 0 ? { formula: "=SUM(" + weekdaysFormula.join("+") + ")" } : "",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: weekdaysFormula.length > 0
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: weekendsFormula.length > 0 ? { formula: "=SUM(" + weekendsFormula.join("+") + ")" } : "",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: weekendsFormula.length > 0
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: holidaysFormula.length > 0 ? { formula: "=SUM(" + holidaysFormula.join("+") + ")" } : "",
            ws: this._wsGeneral,
            border: Constants.Styling.Dotted,
            numFmt: holidaysFormula.length > 0
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        const startAddress = this._wsGeneral.getColumn(colIndex - 3).letter + this._generalRowIndex;
        const endAddress = this._wsGeneral.getColumn(colIndex - 1).letter + this._generalRowIndex;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            value: {
                formula: "SUM(" + startAddress + ":" + endAddress + ")",
            },
            numFmt: '[<0.000694444]"";[<0.0415][m]" ' +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                '";[h] "' +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                ' " m "' +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                '"',
        });
        this._generalRowIndex++;
    }
    addSumRow() {
        let colIndex = 1;
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            value: "",
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            value: "",
        });
        this.setDataCell({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            ws: this._wsGeneral,
            border: Constants.Styling.AllThin,
            value: "",
        });
        for (let i = 0; i < this._maxExtraWorkDays; i++) {
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                value: "",
            });
            const startAddress = this._wsGeneral.getColumn(colIndex).letter + 6;
            const endAddress = this._wsGeneral.getColumn(colIndex).letter + (this._generalRowIndex - 1);
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                value: {
                    formula: "SUM(" + startAddress + ":" + endAddress + ")",
                },
                numFmt: '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"',
            });
        }
        for (let i = 0; i < 4; i++) {
            const startAddress = this._wsGeneral.getColumn(colIndex).letter + 6;
            const endAddress = this._wsGeneral.getColumn(colIndex).letter + (this._generalRowIndex - 1);
            this.setDataCell({
                rowIndex: this._generalRowIndex,
                colIndex: colIndex++,
                ws: this._wsGeneral,
                border: Constants.Styling.AllThin,
                value: {
                    formula: "SUM(" + startAddress + ":" + endAddress + ")",
                },
                numFmt: '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"',
            });
        }
        this._generalRowIndex++;
    }
    setDataCell(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextRed)
            c.font = {
                color: { argb: "FF0000" },
                bold: true,
            };
    }
    addFilterToSheet(ws, startRowIndex, startColIndex) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell(ws, startRowIndex, startColIndex, this._organizationBasic.name + " Fazla Mesai Raporu", border, { bold: true });
        this.addFilterCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)(this._options.dateRange.startDateTime).locale(this._locale).format("LL") + " - " + (0, moment_1.default)(this._options.dateRange.endDateTime).locale(this._locale).format("LL"), border, { italic: true }, Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
    }
    addFilterCell(ws, row, col, value, border, font, numFmt) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
    }
}
exports.TurquoiseYachtExtraWorkCustomReport = TurquoiseYachtExtraWorkCustomReport;
