"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_utils_1 = require("../../../dal.utils");
async function up(client, dbuser, dbsuperuser) {
    await fixOrganizationsTrigger.up(client, dbuser, dbsuperuser);
    await alterUsageTypeConstraint.up(client, dbuser, dbsuperuser);
    await userFilterView.up(client, dbuser, dbsuperuser);
    await userFilterResultFunction.up(client, dbuser, dbsuperuser);
    await updateFilterResultsFunction.up(client, dbuser, dbsuperuser);
    await changesOnUserOrganization.up(client, dbuser, dbsuperuser);
    await updateUserWorkPlanTrigger.up(client, dbuser, dbsuperuser);
    await updateUserGroupTrigger.up(client, dbuser, dbsuperuser);
    await updateUserUnitFilterUpdateTrigger.up(client, dbuser, dbsuperuser);
    await updateOrganizationUnitDeleteTrigger.up(client, dbuser, dbsuperuser);
    await updateUserFilterDeleteSettingsTriggerFunction.up(client, dbuser, dbsuperuser);
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
    await updateUserFilterDeleteSettingsTriggerFunction.down(client, dbuser, dbsuperuser);
    await updateOrganizationUnitDeleteTrigger.down(client, dbuser, dbsuperuser);
    await updateUserUnitFilterUpdateTrigger.down(client, dbuser, dbsuperuser);
    await updateUserGroupTrigger.down(client, dbuser, dbsuperuser);
    await updateUserWorkPlanTrigger.down(client, dbuser, dbsuperuser);
    await changesOnUserOrganization.down(client, dbuser, dbsuperuser);
    await updateFilterResultsFunction.down(client, dbuser, dbsuperuser);
    await userFilterResultFunction.down(client, dbuser, dbsuperuser);
    await userFilterView.down(client, dbuser, dbsuperuser);
    await alterUsageTypeConstraint.down(client, dbuser, dbsuperuser);
}
exports.down = down;
const fixOrganizationsTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_organization()
		RETURNS trigger
		LANGUAGE 'plpgsql'
			COST 100
			VOLATILE NOT LEAKPROOF
			AS $BODY$
			DECLARE
				transaction_id	uuid;
			BEGIN
				transaction_id := uuid_generate_v4();
				IF (TG_OP = 'UPDATE') THEN
					IF (NEW.settings->'webRtc' != OLD.settings->'webRtc') THEN
						INSERT INTO "___ORGANIZATION_ID___"."terminalChanges"
						("id", "transactionId", "deviceId", "actionT", "type", "data")
						SELECT uuid_generate_v4(), transaction_id, T."proxyTerminalId", now(), 3, '{}'::json FROM
						(SELECT DISTINCT "proxyTerminalId" FROM "___ORGANIZATION_ID___"."cameras") as T;
						PERFORM PG_NOTIFY('webrtc_ice_servers', JSON_BUILD_OBJECT('id', NEW.id)::text);								  		
					END IF;	
					IF (NEW."userFilterAccessRightActivation" IS TRUE AND NEW."userFilterAccessRightActivation" != OLD."userFilterAccessRightActivation") THEN
						WITH added_user_filters AS (
							SELECT  gen_random_uuid() as filter_id, "accessControlPointId", acp.name, 
									"remoteAccess", "read", "access", config, "grant", "snapshot", COUNT(uar."userId") as "userCount",
									array_agg(uar."userId") as users
							FROM "___ORGANIZATION_ID___"."userAccessRights" as uar	
							INNER JOIN "___ORGANIZATION_ID___"."accessControlPoints" as acp
								ON uar."accessControlPointId" = acp.id
							INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" as uo
								ON uar."userId" = uo."userId"
							INNER JOIN "___ORGANIZATION_ID___"."roles" as r
								ON uo."roleId" = r.id
							WHERE uar."deletedAt" IS NULL AND acp."deletedAt" IS NULL AND uo."isDisabled" IS FALSE
								  AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
							GROUP BY "accessControlPointId", acp.name, "read", "access", config, "grant", "snapshot", "remoteAccess"
							ORDER BY "accessControlPointId" DESC
						), filter_user_mapping AS(
							SELECT filter_id, "userId"
							FROM  added_user_filters as a
							CROSS JOIN unnest(users) as "userId"
						), inserted_filters AS (
							INSERT INTO "___ORGANIZATION_ID___".user_filter(
								id, name, description, combinator, "allIncluded", "usageType", "userCount")
							SELECT filter_id, name, 'Varsayılan ' || name || ' Filtresi', 2, false, 3, "userCount"
							FROM added_user_filters
						), inserted_filter_acp_mapping AS (
							INSERT INTO "___ORGANIZATION_ID___".user_filter_access_control_points(
								id, "filterId", "accessControlPointId", "read", "access", "remoteAccess", config, "grant", "snapshot")
							SELECT gen_random_uuid(), filter_id, "accessControlPointId", "read", "access", "remoteAccess", config, "grant", "snapshot" 
							FROM added_user_filters
						), inserted_filter_result AS (
							INSERT INTO "___ORGANIZATION_ID___".user_filter_user(
								"filterId", "userId")
							SELECT filter_id, "userId"
							FROM filter_user_mapping
						)
						INSERT INTO "___ORGANIZATION_ID___".filter_result(
							id,"filterId", "userId")
						SELECT gen_random_uuid(), filter_id, "userId"
						FROM filter_user_mapping;

						PERFORM "___ORGANIZATION_ID___".enable_user_filter_related_triggers();
					END IF;
				END IF;	
				PERFORM PG_NOTIFY('organization_settings_changed', TG_TABLE_SCHEMA);
			RETURN NEW;
			END;
			$BODY$;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const alterUsageTypeConstraint = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		    ALTER TABLE "___ORGANIZATION_ID___".user_filter
			DROP CONSTRAINT check_usage_type;
			ALTER TABLE "___ORGANIZATION_ID___".user_filter
			ADD CONSTRAINT check_usage_type CHECK ("usageType" IN(0,1,2,3,4));
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		    ALTER TABLE "___ORGANIZATION_ID___".user_filter
			DROP CONSTRAINT check_usage_type;
			ALTER TABLE "___ORGANIZATION_ID___".user_filter
			ADD CONSTRAINT check_usage_type CHECK ("usageType" IN(1,2,3,4));
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterView = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `	
		DROP VIEW IF EXISTS "___ORGANIZATION_ID___".vw_user_filter;

		CREATE VIEW "___ORGANIZATION_ID___".vw_user_filter
		AS
		SELECT  uf.id,
				jsonb_strip_nulls(jsonb_build_object(
									'id', uf.id,
									'name', uf.name,
									'description', uf.description,
									'combinator', uf.combinator,
									'usageType', uf."usageType",
									'allIncluded', uf."allIncluded",
									'groupIds', COALESCE(jsonb_agg(DISTINCT ufg."groupId") FILTER (WHERE ufg."groupId" IS NOT NULL), NULL::jsonb), 
									'roleIds', COALESCE(jsonb_agg(DISTINCT ufor."roleId") FILTER (WHERE ufor."roleId" IS NOT NULL), NULL::jsonb), 
									'userIds', COALESCE(jsonb_agg(DISTINCT ufu."userId") FILTER (WHERE ufu."userId" IS NOT NULL), NULL::jsonb), 
									'workPlanIds', COALESCE(jsonb_agg(DISTINCT ufw."workPlanId") FILTER (WHERE ufw."workPlanId" IS NOT NULL), NULL::jsonb), 
									'units', COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', ufn."unitId", 'hierarchically', ufn.hierarchically, 'roleIds', COALESCE (ufuragg."roleIds", '[]'::JSON))) 
											FILTER (WHERE ufn."unitId" IS NOT NULL), NULL::jsonb))
								)AS filter
		FROM "___ORGANIZATION_ID___".user_filter uf
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_group ufg ON uf.id = ufg."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role ufor ON uf.id = ufor."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_user ufu ON uf.id = ufu."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_work_plan ufw ON uf.id = ufw."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_unit ufn ON uf.id = ufn."filterId"
		LEFT JOIN ( SELECT	ufur."filterId",
							ufur."unitId",
							COALESCE(json_agg(ufur."roleId") FILTER (WHERE ufur."roleId" IS NOT NULL), NULL::json) AS "roleIds"
					FROM "___ORGANIZATION_ID___".user_filter_unit_role ufur
					GROUP BY ufur."filterId", ufur."unitId") ufuragg 
			ON ufuragg."unitId" = ufn."unitId" AND ufn."filterId" = ufuragg."filterId"
		GROUP BY uf.id;
		
		ALTER TABLE "___ORGANIZATION_ID___".vw_user_filter
			OWNER TO ${dbsuperuser};

		GRANT INSERT, SELECT, UPDATE, DELETE ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbuser};
		GRANT ALL ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbsuperuser};`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP VIEW IF EXISTS "___ORGANIZATION_ID___".vw_user_filter;

		CREATE VIEW "___ORGANIZATION_ID___".vw_user_filter
		AS
		SELECT  uf.id,
				jsonb_strip_nulls(jsonb_build_object(
									'id', uf.id,
									'name', uf.name,
									'description', uf.description,
									'combinator', uf.combinator,
									'usageType', uf."usageType",
									'allIncluded', uf."allIncluded",
									'groupIds', COALESCE(jsonb_agg(DISTINCT ufg."groupId") FILTER (WHERE ufg."groupId" IS NOT NULL), NULL::jsonb), 
									'roleIds', COALESCE(jsonb_agg(DISTINCT ufor."roleId") FILTER (WHERE ufor."roleId" IS NOT NULL), NULL::jsonb), 
									'userIds', COALESCE(jsonb_agg(DISTINCT ufu."userId") FILTER (WHERE ufu."userId" IS NOT NULL), NULL::jsonb), 
									'workPlanIds', COALESCE(jsonb_agg(DISTINCT ufw."workPlanId") FILTER (WHERE ufw."workPlanId" IS NOT NULL), NULL::jsonb), 
									'units', COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', ufn."unitId", 'hierarchically', ufn.hierarchically, 'roleIds', ufuragg."roleIds")) 
											FILTER (WHERE ufn."unitId" IS NOT NULL), NULL::jsonb))
								)AS filter
		FROM "___ORGANIZATION_ID___".user_filter uf
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_group ufg ON uf.id = ufg."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role ufor ON uf.id = ufor."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_user ufu ON uf.id = ufu."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_work_plan ufw ON uf.id = ufw."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_unit ufn ON uf.id = ufn."filterId"
		LEFT JOIN ( SELECT	ufur."filterId",
							ufur."unitId",
							COALESCE(json_agg(ufur."roleId") FILTER (WHERE ufur."roleId" IS NOT NULL), NULL::json) AS "roleIds"
					FROM "___ORGANIZATION_ID___".user_filter_unit_role ufur
					GROUP BY ufur."filterId", ufur."unitId") ufuragg 
			ON ufuragg."unitId" = ufn."unitId" AND ufn."filterId" = ufuragg."filterId"
		GROUP BY uf.id;
		
		ALTER TABLE "___ORGANIZATION_ID___".vw_user_filter
			OWNER TO ${dbsuperuser};

		GRANT INSERT, SELECT, UPDATE, DELETE ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbuser};
		GRANT ALL ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbsuperuser};`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterResultFunction = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `	
		CREATE TYPE "___ORGANIZATION_ID___".filter_info AS (
			id uuid,
			combinator smallint,
			"allIncluded" boolean
		);
		DROP FUNCTION "___ORGANIZATION_ID___".user_filter_result;
		CREATE FUNCTION "___ORGANIZATION_ID___".user_filter_result(
			filters "___ORGANIZATION_ID___".filter_info[],
			users "___ORGANIZATION_ID___".user_filter_user[],
			groups "___ORGANIZATION_ID___".user_filter_group[],
			roles "___ORGANIZATION_ID___".user_filter_organization_role[],
			units "___ORGANIZATION_ID___".user_filter_unit[],
			unit_roles "___ORGANIZATION_ID___".user_filter_unit_role[],
			workplans "___ORGANIZATION_ID___".user_filter_work_plan[]
		)
			RETURNS TABLE("filterId" uuid, "userId" uuid)
			LANGUAGE 'plpgsql'
			COST 100
			VOLATILE PARALLEL UNSAFE
			ROWS 1000
		AS $BODY$
			BEGIN
			RETURN QUERY
			WITH all_users AS (
				SELECT uo."userId"
				FROM "___ORGANIZATION_ID___"."userOrganizations" uo
					INNER JOIN "___ORGANIZATION_ID___"."roles" r
						ON uo."roleId" = r."id"
					WHERE uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
			),
			filter_data AS (
				SELECT * FROM unnest(filters)
			),
			user_data AS (
				SELECT * FROM unnest(users)
			),
			group_data AS (
				SELECT * FROM unnest(groups)
			),
			role_data AS (
				SELECT * FROM unnest(roles)
			),
			unit_data AS (
				SELECT * FROM unnest(units)
			),
			unit_role_data AS (
				SELECT * FROM unnest(unit_roles)
			),
			workplan_data AS (
				SELECT * FROM unnest(workplans)
			),
			filter_definition AS (
				SELECT
					fd.id,
					fd.combinator,
					fd."allIncluded",
					BOOL_OR(gd."filterId" IS NOT NULL) as has_group,
					BOOL_OR(rd."filterId" IS NOT NULL) as has_role,
					BOOL_OR(ud."filterId" IS NOT NULL) as has_unit,
					BOOL_OR(wd."filterId" IS NOT NULL) as has_work_plan
				FROM filter_data fd
				LEFT JOIN group_data gd
					ON fd."id" = gd."filterId"
				LEFT JOIN role_data rd
					ON fd."id" = rd."filterId"
				LEFT JOIN unit_data ud
					ON fd."id" = ud."filterId"
				LEFT JOIN workplan_data wd
					ON fd."id" = wd."filterId"
				GROUP BY fd.id, fd.combinator, fd."allIncluded"
			),
			group_users AS (
				SELECT DISTINCT ufg."filterId", uo."userId"
				FROM group_data ufg
				INNER JOIN "___ORGANIZATION_ID___"."userGroupUserOrganizations" uguo
					ON ufg."groupId" = uguo."userGroupId" AND uguo."deletedAt" IS NULL
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
					ON uguo."userOrganizationId" = uo.id
			),
			role_users AS (
				SELECT DISTINCT ufr."filterId", uo."userId"
				FROM role_data ufr
				INNER JOIN "___ORGANIZATION_ID___"."roles" r
					ON ufr."roleId" = r.id AND r."deletedAt" IS NULL
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
					ON ufr."roleId" = uo."roleId"
			),
			workplan_users AS (
				SELECT DISTINCT ufwp."filterId", uwp."userId"
				FROM workplan_data ufwp
				INNER JOIN "___ORGANIZATION_ID___"."workPlans" wp
					ON ufwp."workPlanId" = wp."id" AND wp."deletedAt" IS NULL
				INNER JOIN "___ORGANIZATION_ID___"."userWorkPlans" uwp
					ON ufwp."workPlanId" = uwp."workPlanId" AND tstzrange("startDateTime","endDateTime") @> now()
			),
			unit_users AS (
				SELECT DISTINCT ufu."filterId", uo."userId"
				FROM unit_data ufu
				LEFT JOIN unit_role_data ufur
					ON ufur."filterId" = ufu."filterId" AND ufur."unitId" = ufu."unitId"
				INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
					ON ou."deletedAt" IS NULL AND (ou.id = ufu."unitId" OR (ufu.hierarchically AND ou."ancestorIds" ILIKE '%' || ufu."unitId" || '%'))
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" uoou
					ON uoou."deletedAt" IS NULL AND ou."id" = uoou."organizationUnitId" AND
						(ufur."roleId" IS NULL OR ufur."roleId" = uoou."roleId")
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
					ON uo."id" = uoou."userOrganizationId" AND NOT uo."isDisabled" AND uo."deletedAt" IS NULL
			),
			filter_results AS (
				SELECT fd."id" as "filterId", all_users."userId"
				FROM all_users
				INNER JOIN filter_definition fd ON true
				LEFT JOIN group_users
					ON fd."id" = group_users."filterId" AND group_users."userId" = all_users."userId"
				LEFT JOIN unnest(users) as specific_users
					ON fd."id" = specific_users."filterId" AND specific_users."userId" = all_users."userId"
				LEFT JOIN role_users
					ON fd."id" = role_users."filterId" AND role_users."userId" = all_users."userId"
				LEFT JOIN workplan_users
					ON fd."id" = workplan_users."filterId" AND workplan_users."userId" = all_users."userId"
				LEFT JOIN unit_users
					ON fd."id" = unit_users."filterId" AND unit_users."userId" = all_users."userId"
				WHERE fd."allIncluded"
					OR specific_users."userId" IS NOT NULL
					OR (fd.combinator = 1 AND
						(fd.has_group OR fd.has_role OR fd.has_work_plan OR fd.has_unit) AND
						(NOT fd.has_group OR group_users."userId" IS NOT NULL) AND
						(NOT fd.has_role OR role_users."userId" IS NOT NULL) AND
						(NOT fd.has_work_plan OR workplan_users."userId" IS NOT NULL) AND
						(NOT fd.has_unit OR unit_users."userId" IS NOT NULL)
					)
					OR (fd.combinator = 2 AND
						(group_users."userId" IS NOT NULL OR role_users."userId" IS NOT NULL OR workplan_users."userId" IS NOT NULL OR unit_users."userId" IS NOT NULL)
					)
			),
			update_user_count AS (
				UPDATE "___ORGANIZATION_ID___".user_filter
				SET "userCount" = cnt
				FROM (
					SELECT fd.id as "filterId", COUNT(fr."userId") as cnt
					FROM filter_data fd
					LEFT JOIN filter_results fr
						ON fd.id = fr."filterId"
					GROUP BY fd.id
				) sq 
				WHERE id = sq."filterId"
			)
			SELECT * FROM filter_results;
			END;	
		$BODY$;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP FUNCTION "___ORGANIZATION_ID___".user_filter_result;
		CREATE FUNCTION "___ORGANIZATION_ID___".user_filter_result(filter_id uuid)
			RETURNS TABLE("userId" uuid)
			LANGUAGE 'plpgsql'
			COST 100
			VOLATILE NOT LEAKPROOF
		AS $BODY$
		BEGIN
			RETURN QUERY
			WITH all_users AS (
				SELECT uo."userId"
				FROM "___ORGANIZATION_ID___"."userOrganizations" uo
					INNER JOIN "___ORGANIZATION_ID___"."roles" r
						ON uo."roleId" = r."id"
					WHERE uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
			),
			filter_definition AS (
				SELECT
					uf.id, uf.combinator, uf."allIncluded",
					BOOL_OR(ufg."filterId" IS NOT NULL) as has_group,
					BOOL_OR(ufor."filterId" IS NOT NULL) as has_role,
					BOOL_OR(ufu."filterId" IS NOT NULL) as has_unit,
					BOOL_OR(ufw."filterId" IS NOT NULL) as has_work_plan
				FROM "___ORGANIZATION_ID___"."user_filter" uf
				LEFT JOIN "___ORGANIZATION_ID___"."user_filter_group" ufg
					ON uf.id = ufg."filterId"
				LEFT JOIN "___ORGANIZATION_ID___"."user_filter_organization_role" ufor
					ON uf.id = ufor."filterId"
				LEFT JOIN "___ORGANIZATION_ID___"."user_filter_unit" ufu
					ON uf.id = ufu."filterId"
				LEFT JOIN "___ORGANIZATION_ID___"."user_filter_work_plan" ufw
					ON uf.id = ufw."filterId"
				WHERE uf.id = filter_id
				GROUP BY uf.id
			),
			group_users AS (
				SELECT DISTINCT uo."userId"
				FROM "___ORGANIZATION_ID___"."user_filter_group" ufg
				INNER JOIN "___ORGANIZATION_ID___"."userGroupUserOrganizations" uguo
					ON ufg."groupId" = uguo."userGroupId" AND uguo."deletedAt" IS NULL
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
					ON uguo."userOrganizationId" = uo.id
				WHERE ufg."filterId" = filter_id
			),
			specific_users AS (
				SELECT DISTINCT ufu."userId"
				FROM "___ORGANIZATION_ID___"."user_filter_user" ufu
				WHERE ufu."filterId" = filter_id
			),
			role_users AS (
				SELECT DISTINCT uo."userId"
				FROM "___ORGANIZATION_ID___"."user_filter_organization_role" ufor
				INNER JOIN "___ORGANIZATION_ID___"."roles" r
					ON ufor."roleId" = r.id AND r."deletedAt" IS NULL
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
					ON ufor."roleId" = uo."roleId"
				WHERE ufor."filterId" = filter_id
			),
			workplan_users AS (
				SELECT DISTINCT uwp."userId"
				FROM "___ORGANIZATION_ID___"."user_filter_work_plan" ufwp
				INNER JOIN "___ORGANIZATION_ID___"."workPlans" wp
					ON ufwp."workPlanId" = wp."id" AND wp."deletedAt" IS NULL
				INNER JOIN "___ORGANIZATION_ID___"."userWorkPlans" uwp
					ON ufwp."workPlanId" = uwp."workPlanId" AND tstzrange("startDateTime","endDateTime") @> now()
				WHERE ufwp."filterId" = filter_id
			),
			unit_users AS (
				SELECT DISTINCT uo."userId"
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				LEFT JOIN "___ORGANIZATION_ID___"."user_filter_unit_role" ufur
					ON ufur."filterId" = ufu."filterId" AND ufur."unitId" = ufu."unitId"
				INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
					ON ou."deletedAt" IS NULL AND (ou.id = ufu."unitId" OR (ufu.hierarchically AND ou."ancestorIds" ILIKE '%' || ufu."unitId" || '%'))
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" uoou
					ON uoou."deletedAt" IS NULL AND ou."id" = uoou."organizationUnitId" AND
						(ufur."roleId" IS NULL OR ufur."roleId" = uoou."roleId")
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
					ON uo."id" = uoou."userOrganizationId" AND NOT uo."isDisabled" AND uo."deletedAt" IS NULL
				WHERE
					ufu."filterId" = filter_id
			),
			filter_result AS (
				SELECT all_users."userId"
				FROM all_users
				INNER JOIN filter_definition ON true
				LEFT JOIN group_users ON group_users."userId" = all_users."userId"
				LEFT JOIN specific_users ON specific_users."userId" = all_users."userId"
				LEFT JOIN role_users ON role_users."userId" = all_users."userId"
				LEFT JOIN workplan_users ON workplan_users."userId" = all_users."userId"
				LEFT JOIN unit_users ON unit_users."userId" = all_users."userId"
				WHERE filter_definition."allIncluded"
					OR specific_users."userId" IS NOT NULL
					OR (filter_definition.combinator = 1 AND
						(filter_definition.has_group OR filter_definition.has_role OR filter_definition.has_work_plan OR filter_definition.has_unit) AND
						(NOT filter_definition.has_group OR group_users."userId" IS NOT NULL) AND
						(NOT filter_definition.has_role OR role_users."userId" IS NOT NULL) AND
						(NOT filter_definition.has_work_plan OR workplan_users."userId" IS NOT NULL) AND
						(NOT filter_definition.has_unit OR unit_users."userId" IS NOT NULL)
					)
					OR (filter_definition.combinator = 2 AND
						(group_users."userId" IS NOT NULL OR role_users."userId" IS NOT NULL OR workplan_users."userId" IS NOT NULL OR unit_users."userId" IS NOT NULL)
					)
			),
			update_user_count AS (
				UPDATE "___ORGANIZATION_ID___".user_filter
				SET "userCount" = (SELECT COUNT(fr."userId") FROM filter_result fr)
				WHERE id = filter_id
			)
			SELECT *
			FROM filter_result;
		END;
		$BODY$;
		DROP TYPE "___ORGANIZATION_ID___".filter_info;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateFilterResultsFunction = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `	
		CREATE FUNCTION "___ORGANIZATION_ID___".update_filter_results("filterIds" uuid[])
			RETURNS VOID
			LANGUAGE 'plpgsql'
			COST 100
			VOLATILE PARALLEL UNSAFE
		AS $BODY$
			BEGIN
			IF ("filterIds" IS NULL OR array_length("filterIds",1) IS NULL) THEN
				RETURN;
			END IF;
			WITH filter_results AS (
				SELECT * FROM "___ORGANIZATION_ID___".user_filter_result(
					(SELECT array_agg((id, combinator, "allIncluded")::"___ORGANIZATION_ID___".filter_info)
					FROM "___ORGANIZATION_ID___"."user_filter"
					WHERE id = ANY("filterIds")),
					(SELECT array_agg(ufu)
					FROM "___ORGANIZATION_ID___"."user_filter_user" ufu
					WHERE "filterId" = ANY("filterIds")),
					(SELECT array_agg(ufg)
					FROM "___ORGANIZATION_ID___"."user_filter_group" ufg
					WHERE "filterId" = ANY("filterIds")),
					(SELECT array_agg(ufor)
					FROM "___ORGANIZATION_ID___"."user_filter_organization_role" ufor
					WHERE "filterId" = ANY("filterIds")),
					(SELECT array_agg(ufu)
					FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
					WHERE "filterId" = ANY("filterIds")),
					(SELECT array_agg(ufur)
					FROM "___ORGANIZATION_ID___"."user_filter_unit_role" ufur
					WHERE "filterId" = ANY("filterIds")),
					(SELECT array_agg(ufwp)
					FROM "___ORGANIZATION_ID___"."user_filter_work_plan" ufwp
					WHERE "filterId" = ANY("filterIds"))
				)
			),
			deleted AS (
				DELETE FROM "___ORGANIZATION_ID___".filter_result
				USING (
					SELECT *
					FROM "___ORGANIZATION_ID___".filter_result fr
					WHERE fr."filterId" = ANY("filterIds")
				) sq
				LEFT OUTER JOIN filter_results nfr
					ON sq."filterId" = nfr."filterId" AND sq."userId" = nfr."userId"
				WHERE filter_result.id = sq.id AND nfr."userId" IS NULL
			)
			INSERT INTO "___ORGANIZATION_ID___".filter_result
			SELECT gen_random_uuid(), "filterId", "userId"
			FROM filter_results as ufr
			ON CONFLICT DO NOTHING;
			END;
		$BODY$;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP FUNCTION "___ORGANIZATION_ID___".update_filter_results;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const changesOnUserOrganization = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `	
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_organization()
		RETURNS TRIGGER AS $$
		BEGIN
			IF (TG_OP = 'INSERT') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT uf.id))
				FROM "___ORGANIZATION_ID___".user_filter uf
				LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role as ufr
					ON uf.id = ufr."filterId"
				CROSS JOIN new_table as nt 
				INNER JOIN "___ORGANIZATION_ID___".roles as r 
					ON nt."roleId" = r.id
				WHERE nt."isDisabled" = false 
					AND nt."deletedAt" IS NULL 
					AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
					AND (uf."allIncluded" IS TRUE OR ufr."roleId" = nt."roleId")
					AND uf."usageType" = 3;
			ELSIF (TG_OP = 'UPDATE') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT uf.id))
				FROM "___ORGANIZATION_ID___"."user_filter" as uf
				LEFT JOIN "___ORGANIZATION_ID___"."user_filter_organization_role" as ufor
					ON uf.id = ufor."filterId"
				CROSS JOIN 
						(
						SELECT nt."roleId" as new_role,
							ot."roleId" as old_role
						FROM new_table as nt
						INNER JOIN old_table as ot
							ON  nt.id = ot.id 
								AND (ot."isDisabled" IS DISTINCT FROM nt."isDisabled" OR ot."deletedAt" IS DISTINCT FROM nt."deletedAt" OR ot."roleId" IS DISTINCT FROM nt."roleId")
						) subq
				WHERE 	uf."usageType" = 3 
						AND (
							(ufor."roleId" = subq.new_role OR ufor."roleId" = subq.old_role)
							OR (subq.new_role = subq.old_role AND uf."allIncluded" IS TRUE)												   
						);
							
				DELETE FROM "___ORGANIZATION_ID___".user_filter_user as ufu
				USING new_table as nt
				INNER JOIN old_table as ot
					ON nt.id = ot.id
				INNER JOIN "___ORGANIZATION_ID___".roles as r
					ON nt."roleId" = r."id"
				WHERE 	ufu."userId" = nt."userId"
						AND ((nt."isDisabled" = true AND ot."isDisabled" = false) OR (nt."deletedAt" IS NOT NULL AND ot."deletedAt" IS NULL))
						AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7');
			ELSIF (TG_OP = 'DELETE') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT uf.id))
				FROM "___ORGANIZATION_ID___".user_filter as uf
				LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role as ufor
					ON uf.id = ufor."filterId"
				CROSS JOIN (
							SELECT ot."roleId"
							FROM old_table as ot
							WHERE ot."deletedAt" IS NULL AND ot."isDisabled" IS FALSE
							) subq
				WHERE  uf."usageType" = 3
					AND (subq."roleId" = ufor."roleId" OR uf."allIncluded" IS TRUE);
			
				DELETE FROM "___ORGANIZATION_ID___".user_filter_user as ufu
				USING old_table as ot
				INNER JOIN "___ORGANIZATION_ID___".roles as r
					ON ot."roleId" = r."id"
				WHERE 	ufu."userId" = ot."userId"
						AND (ot."deletedAt" IS NULL AND ot."isDisabled" IS FALSE)
						AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7');
			END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_organization()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
			IF (TG_OP = 'INSERT') THEN
				SELECT array_agg(DISTINCT(uf.id)) into updated_filter_ids
				FROM "___ORGANIZATION_ID___".user_filter uf
				LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role as ufr
					ON uf.id = ufr."filterId"
				CROSS JOIN new_table as nt 
				INNER JOIN "___ORGANIZATION_ID___".roles as r 
					ON nt."roleId" = r.id
				WHERE nt."isDisabled" = false 
					AND nt."deletedAt" IS NULL 
					AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
					AND (uf."allIncluded" IS TRUE OR ufr."roleId" = nt."roleId")
					AND uf."usageType" = 3;
			ELSIF (TG_OP = 'UPDATE') THEN
				SELECT array_agg(DISTINCT(uf.id)) into updated_filter_ids
				FROM "___ORGANIZATION_ID___"."user_filter" as uf
				LEFT JOIN "___ORGANIZATION_ID___"."user_filter_organization_role" as ufor
					ON uf.id = ufor."filterId"
				CROSS JOIN 
						(
						SELECT nt."roleId" as new_role,
							ot."roleId" as old_role
						FROM new_table as nt
						INNER JOIN old_table as ot
							ON  nt.id = ot.id 
								AND (ot."isDisabled" IS DISTINCT FROM nt."isDisabled" OR ot."deletedAt" IS DISTINCT FROM nt."deletedAt" OR ot."roleId" IS DISTINCT FROM nt."roleId")
						) subq
				WHERE 	uf."usageType" = 3 
						AND (
							(ufor."roleId" = subq.new_role OR ufor."roleId" = subq.old_role)
							OR (subq.new_role = subq.old_role AND uf."allIncluded" IS TRUE)												   
						);
							
				DELETE FROM "___ORGANIZATION_ID___".user_filter_user as ufu
				USING new_table as nt
				INNER JOIN old_table as ot
					ON nt.id = ot.id
				INNER JOIN "___ORGANIZATION_ID___".roles as r
					ON nt."roleId" = r."id"
				WHERE 	ufu."userId" = nt."userId"
						AND ((nt."isDisabled" = true AND ot."isDisabled" = false) OR (nt."deletedAt" IS NOT NULL AND ot."deletedAt" IS NULL))
						AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7');
				
			ELSIF (TG_OP = 'DELETE') THEN
				SELECT array_agg(DISTINCT(uf.id)) into updated_filter_ids
				FROM "___ORGANIZATION_ID___".user_filter as uf
				LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role as ufor
					ON uf.id = ufor."filterId"
				CROSS JOIN (
							SELECT ot."roleId"
							FROM old_table as ot
							WHERE ot."deletedAt" IS NULL AND ot."isDisabled" IS FALSE
							) subq
				WHERE  uf."usageType" = 3
					AND (subq."roleId" = ufor."roleId" OR uf."allIncluded" IS TRUE);
			
				DELETE FROM "___ORGANIZATION_ID___".user_filter_user as ufu
				USING old_table as ot
				INNER JOIN "___ORGANIZATION_ID___".roles as r
					ON ot."roleId" = r."id"
				WHERE 	ufu."userId" = ot."userId"
						AND (ot."deletedAt" IS NULL AND ot."isDisabled" IS FALSE)
						AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7');
			END IF;
			IF (updated_filter_ids IS NOT NULL ) THEN 
				FOREACH filter_id IN ARRAY updated_filter_ids LOOP
					WITH filter_result AS (
						SELECT "userId"
						FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
					), deleted AS (
						DELETE FROM "___ORGANIZATION_ID___".filter_result
						WHERE  "filterId" = filter_id
								AND "userId" != ALL(SELECT "userId" FROM filter_result)
					)
					INSERT INTO "___ORGANIZATION_ID___".filter_result
					SELECT gen_random_uuid(), filter_id, "userId"
					FROM filter_result as ufr
					ON CONFLICT DO NOTHING;
				END LOOP;
			END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateUserWorkPlanTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_work_plan()
		RETURNS TRIGGER AS $$
		BEGIN
			IF (TG_OP = 'DELETE') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(uf.id))
				FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
				INNER JOIN old_table as ot
					ON ufwp."workPlanId" = ot."workPlanId"
				INNER JOIN "___ORGANIZATION_ID___".user_filter uf
					ON ufwp."filterId" = uf.id
				WHERE uf."usageType" = 3;
			ELSIF (TG_OP = 'INSERT') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(uf.id))
				FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
				INNER JOIN new_table as nt
					ON ufwp."workPlanId" = nt."workPlanId"
				INNER JOIN "___ORGANIZATION_ID___".user_filter uf
					ON ufwp."filterId" = uf.id
				WHERE uf."usageType" = 3;
			ELSE 
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(uf.id))
				FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
				INNER JOIN (
					SELECT "workPlanId"
					FROM new_table
					UNION
					SELECT "workPlanId"
					FROM old_table
				) updated_table
					ON ufwp."workPlanId" = updated_table."workPlanId"
				INNER JOIN "___ORGANIZATION_ID___".user_filter uf
					ON ufwp."filterId" = uf.id
				WHERE uf."usageType" = 3;
			END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_work_plan()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
			IF (TG_OP = 'DELETE') THEN
				SELECT ARRAY (
					SELECT uf.id 
					FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
					INNER JOIN old_table as ot
						ON ufwp."workPlanId" = ot."workPlanId"
					INNER JOIN "___ORGANIZATION_ID___".user_filter uf
						ON ufwp."filterId" = uf.id
					WHERE  uf."usageType" = 3
				) INTO updated_filter_ids;
			ELSIF (TG_OP = 'INSERT') THEN
				SELECT ARRAY (
					SELECT uf.id 
					FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
					INNER JOIN new_table as nt
						ON ufwp."workPlanId" = nt."workPlanId"
					INNER JOIN "___ORGANIZATION_ID___".user_filter uf
						ON ufwp."filterId" = uf.id
					WHERE  uf."usageType" = 3
				) INTO updated_filter_ids; 
			ELSE 
				SELECT ARRAY (
					SELECT uf.id 
					FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
					INNER JOIN (
						SELECT "workPlanId"
						FROM new_table
						UNION
						SELECT "workPlanId"
						FROM old_table
					) updated_table
						ON ufwp."workPlanId" = updated_table."workPlanId"
					INNER JOIN "___ORGANIZATION_ID___".user_filter uf
						ON ufwp."filterId" = uf.id
					WHERE uf."usageType" = 3
				) INTO updated_filter_ids;
			END IF;
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP 
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result) 
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId" 
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateUserGroupTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `	
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_group()
		RETURNS TRIGGER AS $$
		BEGIN
			IF (TG_OP = 'DELETE') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(uf.id))
				FROM "___ORGANIZATION_ID___".user_filter_group ufg
				INNER JOIN old_table as ot
					ON ufg."groupId" = ot."userGroupId"
				INNER JOIN "___ORGANIZATION_ID___".user_filter uf
					ON ufg."filterId" = uf.id
				WHERE uf."usageType" = 3;
			ELSIF (TG_OP = 'INSERT') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(uf.id))
				FROM "___ORGANIZATION_ID___".user_filter_group ufg
				INNER JOIN new_table as nt
					ON ufg."groupId" = nt."userGroupId"
				INNER JOIN "___ORGANIZATION_ID___".user_filter uf
					ON ufg."filterId" = uf.id
				WHERE uf."usageType" = 3;
			ELSE 
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(uf.id))
				FROM "___ORGANIZATION_ID___".user_filter_group ufg
				INNER JOIN (
					SELECT nt."userGroupId"
					FROM new_table as nt
					INNER JOIN old_table as ot
						ON nt.id = ot.id
					WHERE nt."deletedAt" IS DISTINCT FROM ot."deletedAt"
				) updated_table
					ON ufg."groupId" = updated_table."userGroupId"
				INNER JOIN "___ORGANIZATION_ID___".user_filter uf
					ON ufg."filterId" = uf.id
				WHERE uf."usageType" = 3;
			END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_group()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
			IF (TG_OP = 'DELETE') THEN
				SELECT ARRAY (
					SELECT uf.id 
					FROM "___ORGANIZATION_ID___".user_filter_group ufg
					INNER JOIN old_table as ot
						ON ufg."groupId" = ot."userGroupId"
					INNER JOIN "___ORGANIZATION_ID___".user_filter uf
						ON ufg."filterId" = uf.id
					WHERE  uf."usageType" = 3
				) INTO updated_filter_ids;
			ELSIF (TG_OP = 'INSERT') THEN
				SELECT ARRAY (
					SELECT uf.id 
					FROM "___ORGANIZATION_ID___".user_filter_group ufg
					INNER JOIN new_table as nt
						ON ufg."groupId" = nt."userGroupId"
					INNER JOIN "___ORGANIZATION_ID___".user_filter uf
						ON ufg."filterId" = uf.id
					WHERE  uf."usageType" = 3
				) INTO updated_filter_ids;
			ELSE 
				SELECT ARRAY (
					SELECT uf.id 
					FROM "___ORGANIZATION_ID___".user_filter_group ufg
					INNER JOIN (
						SELECT nt."userGroupId"
						FROM new_table as nt
						INNER JOIN old_table as ot
							ON nt.id = ot.id
						WHERE nt."deletedAt" IS DISTINCT FROM ot."deletedAt"
					) updated_table
						ON ufg."groupId" = updated_table."userGroupId"
					INNER JOIN "___ORGANIZATION_ID___".user_filter uf
						ON ufg."filterId" = uf.id
					WHERE uf."usageType" = 3
				) INTO updated_filter_ids;
			END IF;
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP 
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result) 
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId" 
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateUserUnitFilterUpdateTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_unit()
		RETURNS TRIGGER AS $$
		BEGIN
			IF (TG_OP = 'DELETE') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT ufu."filterId" ))
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
					SELECT ou."id" as "unitId", ou."ancestorIds"
					FROM old_table ot
					INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
						ON  ot."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL
				) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%')
				WHERE uf."usageType" = 3;
			ELSIF (TG_OP = 'INSERT') THEN
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT ufu."filterId" ))
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
					SELECT 	ou."id" as "unitId", ou."ancestorIds"
					FROM new_table as nt
					INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
						ON  nt."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL
				) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" || '%')
				WHERE uf."usageType" = 3;
			ELSE 
				PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT ufu."filterId" ))
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
					SELECT 	ou."id" as "unitId", 
							ou."ancestorIds"
					FROM old_table as ot
					INNER JOIN new_table as nt
						ON ot.id = nt.id
					INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
						ON  (ot."organizationUnitId" = ou."id" OR nt."organizationUnitId" = ou.id) 
							AND ou."deletedAt" IS NULL
					WHERE ot."deletedAt" IS DISTINCT FROM nt."deletedAt" OR ot."roleId" IS DISTINCT FROM nt."roleId"
				) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%')
				WHERE uf."usageType" = 3;
			END IF;
			RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_unit()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
			IF (TG_OP = 'DELETE') THEN
				SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
					SELECT 	ou."id" as "unitId", 
							ou."ancestorIds"
					FROM old_table as ot
					INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
						ON  ot."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL
				) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%')
				WHERE uf."usageType" = 3;
			ELSIF (TG_OP = 'INSERT') THEN
				SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
					SELECT 	ou."id" as "unitId", 
							ou."ancestorIds"
					FROM new_table as nt
					INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
						ON  nt."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL
				) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" || '%')
				WHERE uf."usageType" = 3;
			ELSE 
				SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
					SELECT 	ou."id" as "unitId", 
							ou."ancestorIds"
					FROM old_table as ot
					INNER JOIN new_table as nt
						ON ot.id = nt.id
					INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
						ON  (ot."organizationUnitId" = ou."id" OR nt."organizationUnitId" = ou.id) 
							AND ou."deletedAt" IS NULL
					WHERE ot."deletedAt" IS DISTINCT FROM nt."deletedAt" OR ot."roleId" IS DISTINCT FROM nt."roleId"
				) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%')
				WHERE uf."usageType" = 3;
			END IF;
		IF (updated_filter_ids IS NOT NULL) THEN
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP 
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result) 
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId" 
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateOrganizationUnitDeleteTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_del_or_up_organization_unit()
        RETURNS TRIGGER AS $$
		BEGIN
		IF (TG_OP = 'UPDATE') THEN
			DELETE FROM "___ORGANIZATION_ID___".user_filter_unit as ufu
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			WHERE ufu."unitId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";

			DELETE FROM "___ORGANIZATION_ID___".user_filter_unit_role as ufur
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			WHERE ufur."unitId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";

			PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT ufu."filterId"))
			FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
			INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
				ON uf.id = ufu."filterId"
			INNER JOIN (
				SELECT 	ot."id" as "unitId", 
						ot."ancestorIds" as old_ancestors,
						nt."ancestorIds" as new_ancestors
				FROM old_table as ot
				INNER JOIN new_table as nt
					ON ot.id = nt.id
			) subq
				ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND (old_ancestors ILIKE '%' || ufu."unitId" ||'%' OR new_ancestors ILIKE '%' || ufu."unitId" ||'%'))
			WHERE uf."usageType" = 3;
		ELSIF (TG_OP = 'DELETE') THEN
			PERFORM  "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT ufu."filterId"))
			FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
			INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
				ON uf.id = ufu."filterId"
			INNER JOIN (
				SELECT 	ot."id" as "unitId", ot."ancestorIds"
				FROM old_table as ot
				WHERE ot."deletedAt" IS NULL
			) subq
				ON ufu."unitId" <> subq."unitId" AND ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%'
			WHERE uf."usageType" = 3;
		END IF;
		RETURN NULL;
		END;
        $$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_del_or_up_organization_unit()
        RETURNS TRIGGER AS $$
		DECLARE 
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
		IF (TG_OP = 'UPDATE') THEN
			DELETE FROM "___ORGANIZATION_ID___".user_filter_unit as ufu
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			WHERE ufu."unitId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";

			DELETE FROM "___ORGANIZATION_ID___".user_filter_unit_role as ufur
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			WHERE ufur."unitId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";

			SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
			FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
			INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
				ON uf.id = ufu."filterId"
			INNER JOIN (
				SELECT 	ot."id" as "unitId", 
						ot."ancestorIds" as old_ancestors,
						nt."ancestorIds" as new_ancestors
				FROM old_table as ot
				INNER JOIN new_table as nt
					ON ot.id = nt.id
			) subq
				ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND (old_ancestors ILIKE '%' || ufu."unitId" ||'%' OR new_ancestors ILIKE '%' || ufu."unitId" ||'%'))
			WHERE uf."usageType" = 3;
		ELSIF (TG_OP = 'DELETE') THEN
			SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
			FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
			INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
				ON uf.id = ufu."filterId"
			INNER JOIN (
				SELECT 	ot."id" as "unitId", 
						ot."ancestorIds"
				FROM old_table as ot
				WHERE ot."deletedAt" IS NULL
			) subq
				ON ufu."unitId" <> subq."unitId" AND ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%'
			WHERE uf."usageType" = 3;
		END IF;
		IF (updated_filter_ids IS NOT NULL ) THEN 
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result)
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId"
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		END IF;
		RETURN NULL;
		END;
        $$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateUserFilterDeleteSettingsTriggerFunction = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings()
        RETURNS TRIGGER AS $$
        BEGIN
            PERFORM "___ORGANIZATION_ID___".update_filter_results(array_agg(DISTINCT "filterId"))
			FROM old_table as ot
			INNER JOIN "___ORGANIZATION_ID___"."user_filter" as uf
				ON ot."filterId" = uf.id
			WHERE uf."usageType" = 3;
        	RETURN NULL;
        END;
        $$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings()
        RETURNS TRIGGER AS $$
        DECLARE 
            filter RECORD;
        BEGIN
            FOR filter IN SELECT DISTINCT("filterId") as id
				FROM old_table as ot
                INNER JOIN "___ORGANIZATION_ID___"."user_filter" as uf
                	ON ot."filterId" = uf.id
                WHERE uf."usageType" = 3
            LOOP
                WITH filter_result AS (
                    SELECT "userId"
                    FROM "___ORGANIZATION_ID___".user_filter_result(filter.id)
                ), deleted AS (
                    DELETE FROM "___ORGANIZATION_ID___".filter_result
                    WHERE  "filterId" = filter.id
                            AND "userId" != ALL(SELECT "userId" FROM filter_result)
                )
                INSERT INTO "___ORGANIZATION_ID___".filter_result
                SELECT gen_random_uuid(), filter.id, "userId"
                FROM filter_result as ufr
                ON CONFLICT DO NOTHING;
            END LOOP;
        RETURN NULL;
        END;
        $$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
