"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_utils_1 = require("../../../dal.utils");
async function up(client, dbuser, dbsuperuser) {
    await updateUsersWithVisitorRoleAsStandardUser.up(client, dbuser, dbsuperuser);
    await createFilterResultTable.up(client, dbuser, dbsuperuser);
    await userFilterColumns.up(client, dbuser, dbsuperuser);
    await updateNotificationEventsUserFilter.up(client, dbuser, dbsuperuser);
    await removeUnusedUserFilters.up(client, dbuser, dbsuperuser);
    await updateUserFilterNameAndUsage.up(client, dbuser, dbsuperuser);
    await createUserFilterResultFunction.up(client, dbuser, dbsuperuser);
    await userUnitFilterUpdateTrigger.up(client, dbuser, dbsuperuser);
    await createUserGroupTrigger.up(client, dbuser, dbsuperuser);
    await createUserWorkPlanTrigger.up(client, dbuser, dbsuperuser);
    await updateUserOrganizationAfterUserDeleted.up(client, dbuser, dbsuperuser);
    await changesOnUserOrganization.up(client, dbuser, dbsuperuser);
    await userFilterView.up(client, dbuser, dbsuperuser);
    await userFilterAcpMappingsTableQueries.up(client, dbuser, dbsuperuser);
    await userFilterAcpMappingsTrigger.up(client, dbuser, dbsuperuser);
    await changesOnFilterResult.up(client, dbuser, dbsuperuser);
    await userFilterAccessRightActivationSettings.up(client, dbuser, dbsuperuser);
    await enableUserFilterRelatedTriggers.up(client, dbuser, dbsuperuser);
    await userFilterAccessRightActivationTrigger.up(client, dbuser, dbsuperuser);
    await userAccessRightsRemainingAuthorization.up(client, dbuser, dbsuperuser);
    await removeDisabledUsersAccessRights.up(client, dbuser, dbsuperuser);
    await removeDeletedAcpsUserAccessRights.up(client, dbuser, dbsuperuser);
    await updateUserFilterUserCount.up(client, dbuser, dbsuperuser);
    await userFilterDeleteSettingsTriggerFunction.up(client, dbuser, dbsuperuser);
    await userGroupUpdateTrigger.up(client, dbuser, dbsuperuser);
    await userFilterGroupDeleteTrigger.up(client, dbuser, dbsuperuser);
    await userRoleUpdateTrigger.up(client, dbuser, dbsuperuser);
    await userFilterOrganizationRoleTrigger.up(client, dbuser, dbsuperuser);
    await userFilterOrganizationUnitRoleTrigger.up(client, dbuser, dbsuperuser);
    await organizationUnitDeleteTrigger.up(client, dbuser, dbsuperuser);
    await userFilterOrganizationUnitDeleteTrigger.up(client, dbuser, dbsuperuser);
    await userFilterUserDeleteTrigger.up(client, dbuser, dbsuperuser);
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
    await userFilterUserDeleteTrigger.down(client, dbuser, dbsuperuser);
    await userFilterOrganizationUnitDeleteTrigger.down(client, dbuser, dbsuperuser);
    await organizationUnitDeleteTrigger.down(client, dbuser, dbsuperuser);
    await userFilterOrganizationUnitRoleTrigger.down(client, dbuser, dbsuperuser);
    await userFilterOrganizationRoleTrigger.down(client, dbuser, dbsuperuser);
    await userRoleUpdateTrigger.down(client, dbuser, dbsuperuser);
    await userFilterGroupDeleteTrigger.down(client, dbuser, dbsuperuser);
    await userGroupUpdateTrigger.down(client, dbuser, dbsuperuser);
    await userFilterDeleteSettingsTriggerFunction.down(client, dbuser, dbsuperuser);
    await userAccessRightsRemainingAuthorization.down(client, dbuser, dbsuperuser);
    await userFilterAccessRightActivationTrigger.down(client, dbuser, dbsuperuser);
    await enableUserFilterRelatedTriggers.down(client, dbuser, dbsuperuser);
    await userFilterAccessRightActivationSettings.down(client, dbuser, dbsuperuser);
    await changesOnFilterResult.down(client, dbuser, dbsuperuser);
    await userFilterAcpMappingsTrigger.down(client, dbuser, dbsuperuser);
    await userFilterAcpMappingsTableQueries.down(client, dbuser, dbsuperuser);
    await userFilterView.down(client, dbuser, dbsuperuser);
    await changesOnUserOrganization.down(client, dbuser, dbsuperuser);
    await updateUserOrganizationAfterUserDeleted.down(client, dbuser, dbsuperuser);
    await createUserWorkPlanTrigger.down(client, dbuser, dbsuperuser);
    await createUserGroupTrigger.down(client, dbuser, dbsuperuser);
    await userUnitFilterUpdateTrigger.down(client, dbuser, dbsuperuser);
    await createUserFilterResultFunction.down(client, dbuser, dbsuperuser);
    await userFilterColumns.down(client, dbuser, dbsuperuser);
    await createFilterResultTable.down(client, dbuser, dbsuperuser);
}
exports.down = down;
const updateUsersWithVisitorRoleAsStandardUser = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `

			WITH sq AS (
				SELECT uo.id AS uoid
				FROM  "___ORGANIZATION_ID___"."userOrganizations" uo
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizationProfiles" uop
					ON uop."userOrganizationId" = uo.id
				INNER JOIN "___ORGANIZATION_ID___"."roles" r
					ON uo."roleId" = r.id
				WHERE r."typeId" ='2a4ad5ff-1ced-45f8-992f-e19fc4e766d7'
			)
			UPDATE "___ORGANIZATION_ID___"."userOrganizations" uo
			SET "roleId" = (
				SELECT id 
				FROM "___ORGANIZATION_ID___"."roles"
				WHERE "typeId" = '28f5672e-cda7-4cd0-bbb4-82f2ee4b7ec3'
			)
			FROM sq
			WHERE uo.id = sq.uoid;

			WITH sq AS (
				SELECT uoou."userOrganizationId" AS uoid
				FROM  "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" uoou
				INNER JOIN "___ORGANIZATION_ID___"."userOrganizationProfiles" uop
					ON uop."userOrganizationId" = uoou."userOrganizationId"
				INNER JOIN "___ORGANIZATION_ID___"."roles" r
					ON uoou."roleId" = r.id
				WHERE r."typeId" = '9b8eebd7-057f-4021-ad6f-f2879733bdb9'
			)
			UPDATE "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" uoou
			SET "roleId" = (
				SELECT id 
				FROM "___ORGANIZATION_ID___"."roles"
				WHERE "typeId" = '62dc8044-0311-429b-805e-c86b91343c5b'
			)
			FROM sq
			WHERE uoou."userOrganizationId" = sq.uoid;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const removeDeletedAcpsUserAccessRights = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		UPDATE "___ORGANIZATION_ID___"."userAccessRights" as uar
		SET "deletedAt" = now()
		WHERE "accessControlPointId" = ANY (
			SELECT id
			FROM "___ORGANIZATION_ID___"."accessControlPoints"
			WHERE "deletedAt" IS NOT NULL
		) AND uar."deletedAt" IS NULL;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const removeDisabledUsersAccessRights = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		UPDATE "___ORGANIZATION_ID___"."userAccessRights" as uar
		SET "deletedAt" = now()
		WHERE id = ANY (
						SELECT uar.id
						FROM "___ORGANIZATION_ID___"."userAccessRights" as uar
						INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" as uo
							ON uar."userId" = uo."userId"
						WHERE uo."isDisabled" IS TRUE AND uar."deletedAt" IS NULL
			)
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userAccessRightsRemainingAuthorization = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_update_or_delete_user_access_rights()
		RETURNS TRIGGER AS $$
		BEGIN
			IF(TG_OP = 'DELETE') THEN
				IF (EXISTS ( SELECT ot."accessControlPointId"
							FROM old_table as ot
							INNER JOIN "___ORGANIZATION_ID___"."accessControlPoints" as acp
								ON ot."accessControlPointId" = acp.id 
							LEFT JOIN "___ORGANIZATION_ID___"."userAccessRights" as uar
								ON ot."accessControlPointId" = uar."accessControlPointId" AND uar."deletedAt" IS NULL
							WHERE acp."deletedAt" IS NULL
							GROUP BY ot."accessControlPointId"
							HAVING BOOL_OR( uar."grant" = true AND uar."read" = true AND uar."config" = true) IS FALSE OR BOOL_OR( uar."grant" = true AND uar."read" = true AND uar."config" = true) IS NULL
						)) THEN
				RAISE EXCEPTION 'At least 1 user must have grant, read and config rights.' USING ERRCODE = 'P0001';		
				END IF;
			ELSIF (TG_OP = 'UPDATE') THEN
				IF EXISTS ( SELECT 1 
							FROM
								(
									SELECT BOOL_OR(uar."grant" = true AND uar."read" = true AND uar."config" = true) as combination
									FROM new_table as nt
									INNER JOIN "___ORGANIZATION_ID___"."userAccessRights" as uar
										ON nt."accessControlPointId" = uar."accessControlPointId"
									WHERE uar."deletedAt" IS NULL
									GROUP BY uar."accessControlPointId" 
								) sq
							WHERE sq.combination = false
						) THEN
				RAISE EXCEPTION 'At least 1 user must have grant, read and config rights.' USING ERRCODE = 'P0001';		
				END IF;
			END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;



		CREATE TRIGGER after_update_user_access_rights
			AFTER UPDATE ON "___ORGANIZATION_ID___"."userAccessRights"
			REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_update_or_delete_user_access_rights();

		CREATE TRIGGER after_delete_user_access_rights
			AFTER DELETE ON "___ORGANIZATION_ID___"."userAccessRights"
			REFERENCING OLD TABLE AS old_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_update_or_delete_user_access_rights();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER  after_update_user_access_rights ON "___ORGANIZATION_ID___"."userAccessRights";
		DROP TRIGGER  after_delete_user_access_rights ON "___ORGANIZATION_ID___"."userAccessRights";
		DROP FUNCTION "___ORGANIZATION_ID___".after_update_or_delete_user_access_rights;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterAccessRightActivationTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_organization()
		RETURNS trigger
		LANGUAGE 'plpgsql'
			COST 100
			VOLATILE NOT LEAKPROOF
			AS $BODY$
			DECLARE
				transaction_id	uuid;
			BEGIN
				transaction_id := uuid_generate_v4();
				IF (TG_OP = 'UPDATE') THEN
					IF (NEW.settings->'webRtc' != OLD.settings->'webRtc') THEN
						INSERT INTO "___ORGANIZATION_ID___"."terminalChanges"
						("id", "transactionId", "deviceId", "actionDateISO", "type", "data")
						SELECT uuid_generate_v4(), transaction_id, T."proxyTerminalId", now(), 3, '{}'::json FROM
						(SELECT DISTINCT "proxyTerminalId" FROM "___ORGANIZATION_ID___"."cameras") as T;
						PERFORM PG_NOTIFY('webrtc_ice_servers', JSON_BUILD_OBJECT('id', NEW.id)::text);								  		
					END IF;	
					IF (NEW."userFilterAccessRightActivation" IS TRUE AND NEW."userFilterAccessRightActivation" != OLD."userFilterAccessRightActivation") THEN
						WITH added_user_filters AS (
							SELECT  gen_random_uuid() as filter_id, "accessControlPointId", acp.name, 
									"remoteAccess", "read", "access", config, "grant", "snapshot", COUNT(uar."userId") as "userCount",
									array_agg(uar."userId") as users
							FROM "___ORGANIZATION_ID___"."userAccessRights" as uar	
							INNER JOIN "___ORGANIZATION_ID___"."accessControlPoints" as acp
								ON uar."accessControlPointId" = acp.id
							INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" as uo
								ON uar."userId" = uo."userId"
							INNER JOIN "___ORGANIZATION_ID___"."roles" as r
								ON uo."roleId" = r.id
							WHERE uar."deletedAt" IS NULL AND acp."deletedAt" IS NULL AND uo."isDisabled" IS FALSE
								  AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
							GROUP BY "accessControlPointId", acp.name, "read", "access", config, "grant", "snapshot", "remoteAccess"
							ORDER BY "accessControlPointId" DESC
						), filter_user_mapping AS(
							SELECT filter_id, "userId"
							FROM  added_user_filters as a
							CROSS JOIN unnest(users) as "userId"
						), inserted_filters AS (
							INSERT INTO "___ORGANIZATION_ID___".user_filter(
								id, name, description, combinator, "allIncluded", "usageType", "userCount")
							SELECT filter_id, name, 'Varsayılan ' || name || ' Filtresi', 2, false, 3, "userCount"
							FROM added_user_filters
						), inserted_filter_acp_mapping AS (
							INSERT INTO "___ORGANIZATION_ID___".user_filter_access_control_points(
								id, "filterId", "accessControlPointId", "read", "access", "remoteAccess", config, "grant", "snapshot")
							SELECT gen_random_uuid(), filter_id, "accessControlPointId", "read", "access", "remoteAccess", config, "grant", "snapshot" 
							FROM added_user_filters
						), inserted_filter_result AS (
							INSERT INTO "___ORGANIZATION_ID___".user_filter_user(
								"filterId", "userId")
							SELECT filter_id, "userId"
							FROM filter_user_mapping
						)
						INSERT INTO "___ORGANIZATION_ID___".filter_result(
							id,"filterId", "userId")
						SELECT gen_random_uuid(), filter_id, "userId"
						FROM filter_user_mapping;

						PERFORM "___ORGANIZATION_ID___".enable_user_filter_related_triggers();
					END IF;
				END IF;	
				PERFORM PG_NOTIFY('organization_settings_changed', TG_TABLE_SCHEMA);
			RETURN NEW;
			END;
			$BODY$;
		
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_organization()
		RETURNS trigger
		LANGUAGE 'plpgsql'
			COST 100
			VOLATILE NOT LEAKPROOF
			AS $BODY$
			DECLARE
				transaction_id	uuid;
			BEGIN
				transaction_id := uuid_generate_v4();
				IF (TG_OP = 'UPDATE') THEN
					IF (NEW.settings->'webRtc' != OLD.settings->'webRtc') THEN
						INSERT INTO "___ORGANIZATION_ID___"."terminalChanges"
						("id", "transactionId", "deviceId", "actionDateISO", "type", "data")
						SELECT uuid_generate_v4(), transaction_id, T."proxyTerminalId", now(), 3, '{}'::json FROM
						(SELECT DISTINCT "proxyTerminalId" FROM "___ORGANIZATION_ID___"."cameras") as T;
						PERFORM PG_NOTIFY('webrtc_ice_servers', JSON_BUILD_OBJECT('id', NEW.id)::text);								  		
					END IF;	
				END IF;	
				PERFORM PG_NOTIFY('organization_settings_changed', TG_TABLE_SCHEMA);
			RETURN NEW;
			END;
			$BODY$;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const enableUserFilterRelatedTriggers = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".enable_user_filter_related_triggers()
		RETURNS VOID AS $$
		BEGIN
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" ENABLE TRIGGER after_insert_user_unit';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" ENABLE TRIGGER after_update_user_unit';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" ENABLE TRIGGER after_delete_user_unit';


			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userGroupUserOrganizations" ENABLE TRIGGER after_insert_user_group_user_organization';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userGroupUserOrganizations" ENABLE TRIGGER after_update_user_group_user_organization';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userGroupUserOrganizations" ENABLE TRIGGER after_delete_user_group_user_organization';


			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userWorkPlans" ENABLE TRIGGER after_insert_user_work_plan';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userWorkPlans" ENABLE TRIGGER after_update_user_work_plan';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userWorkPlans" ENABLE TRIGGER after_delete_user_work_plan';

			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."users" ENABLE TRIGGER after_update_user_deleted_statement';

			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userOrganizations" ENABLE TRIGGER after_update_user_organization';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."userOrganizations" ENABLE TRIGGER after_insert_user_organization';

			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" ENABLE TRIGGER after_insert_user_filter_access_control_points';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" ENABLE TRIGGER after_update_user_filter_access_control_points';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" ENABLE TRIGGER after_delete_user_filter_access_control_points';

			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."filter_result" ENABLE TRIGGER after_insert_filter_result';
			EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."filter_result" ENABLE TRIGGER after_delete_filter_result';

            EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_group" ENABLE TRIGGER after_delete_user_filter_group';
            EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_organization_role" ENABLE TRIGGER after_delete_user_filter_organization_role';
            EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_unit_role" ENABLE TRIGGER after_delete_user_filter_unit_role';
            EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_unit" ENABLE TRIGGER after_delete_user_filter_unit';
            EXECUTE 'ALTER TABLE "___ORGANIZATION_ID___"."user_filter_user" ENABLE TRIGGER after_delete_user_filter_user';
		END;
		$$ LANGUAGE plpgsql;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".enable_user_filter_related_triggers();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterAccessRightActivationSettings = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		ALTER TABLE "___ORGANIZATION_ID___".organizations
		ADD COLUMN "userFilterAccessRightActivation" boolean NOT NULL DEFAULT false;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		ALTER TABLE "___ORGANIZATION_ID___".organizations
		DROP COLUMN "userFilterAccessRightActivation"; `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const changesOnFilterResult = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_filter_result()
		RETURNS TRIGGER AS $$
		BEGIN
			IF (TG_OP = 'INSERT') THEN
				INSERT INTO "___ORGANIZATION_ID___"."userAccessRights"(
				id, "createdAt", "updatedAt", "deletedAt", "userId", "accessControlPointId", "remoteAccess", "markedAsFavorite", read, access, config, "grant", snapshot)
				SELECT 	gen_random_uuid(),
						now(), 
						now(), 
						NULL, 
						nt."userId", 
						ufacp."accessControlPointId", 
						ufacp."remoteAccess",
						NULL, 
						ufacp."read", 
						ufacp."access", 
						ufacp."config", 
						ufacp."grant", 
						ufacp."snapshot"
				FROM new_table as nt
				INNER JOIN "___ORGANIZATION_ID___".user_filter_access_control_points as ufacp
					ON nt."filterId" = ufacp."filterId"
				LEFT JOIN "___ORGANIZATION_ID___"."userAccessRights" as uar
					ON  uar."userId" = nt."userId" 
						AND ufacp."accessControlPointId" = uar."accessControlPointId"
						AND uar."deletedAt" IS NULL
				WHERE  uar.id IS NULL;
				
				UPDATE "___ORGANIZATION_ID___"."userAccessRights" as uar
				SET    "read" = (ufacp."read" OR uar.read),
					access = (ufacp."access" OR uar.access),
					"remoteAccess" = (ufacp."remoteAccess" OR uar."remoteAccess"),
					config = (ufacp.config OR uar.config),
					"grant" = (ufacp."grant" OR uar."grant"),
					snapshot = (ufacp.snapshot OR uar.snapshot)
				FROM new_table as nt
				INNER JOIN "___ORGANIZATION_ID___".user_filter_access_control_points as ufacp
					ON nt."filterId" = ufacp."filterId"
				WHERE uar."userId" = nt."userId" AND ufacp."accessControlPointId" = uar."accessControlPointId"
						AND uar."deletedAt" IS NULL;
			ELSIF (TG_OP = 'DELETE') THEN
				WITH will_be_effected_acps AS (
						SELECT ot."filterId",
								ufacp."accessControlPointId",
								ot."userId"
						FROM old_table as ot
						INNER JOIN "___ORGANIZATION_ID___".user_filter_access_control_points as ufacp
								ON ufacp."filterId" = ot."filterId"
					), acp_relations AS (
						SELECT  wbea."accessControlPointId",
								fr."userId",
								fr."filterId"
						FROM will_be_effected_acps as wbea
						INNER JOIN "___ORGANIZATION_ID___".filter_result as fr
							ON	fr."userId" = wbea."userId" AND wbea."filterId" <> fr."filterId"
						INNER JOIN "___ORGANIZATION_ID___".user_filter_access_control_points as ufacp
								ON ufacp."filterId" = fr."filterId"	
						WHERE wbea."accessControlPointId" = ufacp."accessControlPointId"
					), deleted_users AS (
						DELETE FROM "___ORGANIZATION_ID___"."userAccessRights" as uar
						USING will_be_effected_acps as wbea
						LEFT JOIN acp_relations as ar
							ON wbea."filterId" = ar."filterId"
						WHERE   ar."filterId" IS NULL
								AND uar."accessControlPointId" = wbea."accessControlPointId" 
								AND uar."userId" = wbea."userId"
					)
					UPDATE "___ORGANIZATION_ID___"."userAccessRights" as uar
					SET  	"updatedAt" = now(), 
							"read" = sub_q."read",
							"access" = sub_q.access,
							"remoteAccess" = sub_q."remoteAccess",
							"config" = sub_q.config,
							"grant" = sub_q."grant",
							"snapshot" = sub_q.snapshot
					FROM	(
								SELECT  ar."userId",
										ar."accessControlPointId",
										BOOL_OR(ufacp."read") as "read",
										BOOL_OR(ufacp.access) as "access",
										BOOL_OR(ufacp."remoteAccess") as "remoteAccess",
										BOOL_OR(ufacp.config) as "config",
										BOOL_OR(ufacp."grant") as "grant",
										BOOL_OR(ufacp.snapshot) as "snapshot"
								FROM acp_relations as ar
								INNER JOIN "___ORGANIZATION_ID___".user_filter_access_control_points as ufacp
									ON ar."accessControlPointId" = ufacp."accessControlPointId"
								WHERE ufacp."filterId" = ar."filterId"
								GROUP BY ar."userId", ar."accessControlPointId"
							) sub_q
					WHERE sub_q."accessControlPointId" = uar."accessControlPointId" 
						AND uar."userId" = sub_q."userId";		
			END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;

		CREATE TRIGGER after_insert_filter_result
			AFTER INSERT ON "___ORGANIZATION_ID___"."filter_result"
			REFERENCING NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_filter_result();

		CREATE TRIGGER after_delete_filter_result
			AFTER DELETE ON "___ORGANIZATION_ID___"."filter_result"
			REFERENCING OLD TABLE AS old_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_filter_result();
		
		ALTER TABLE "___ORGANIZATION_ID___"."filter_result" DISABLE TRIGGER after_insert_filter_result;
		ALTER TABLE "___ORGANIZATION_ID___"."filter_result" DISABLE TRIGGER after_delete_filter_result;

		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER  after_delete_filter_result ON "___ORGANIZATION_ID___".filter_result;
		DROP TRIGGER  after_insert_filter_result ON "___ORGANIZATION_ID___".filter_result;
		DROP FUNCTION "___ORGANIZATION_ID___".after_change_filter_result;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterAcpMappingsTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_filter_access_control_points()
		RETURNS TRIGGER AS $$
		BEGIN
			IF (TG_OP = 'INSERT') THEN			
				INSERT INTO "___ORGANIZATION_ID___"."userAccessRights"(
				id, "createdAt", "updatedAt", "deletedAt", "userId", "accessControlPointId", "remoteAccess", "markedAsFavorite", read, access, config, "grant", snapshot)
				SELECT 	gen_random_uuid(), 
						now(), 
						now(), 
						NULL, 
						fr."userId", 
						nt."accessControlPointId", 
						nt."remoteAccess",
						NULL, 
						nt."read", 
						nt."access", 
						nt."config", 
						nt."grant", 
						nt.snapshot
				FROM new_table as nt
				INNER JOIN "___ORGANIZATION_ID___".filter_result as fr
					ON nt."filterId" = fr."filterId"
				LEFT JOIN "___ORGANIZATION_ID___"."userAccessRights" as uar
					ON  uar."userId" = fr."userId" 
						AND nt."accessControlPointId" = uar."accessControlPointId"
						AND uar."deletedAt" IS NULL
				WHERE  uar.id IS NULL;
				
				UPDATE "___ORGANIZATION_ID___"."userAccessRights" as uar
				SET  "read" = (nt."read" OR uar.read),
							access = (nt."access" OR uar.access),
							"remoteAccess" = (nt."remoteAccess" OR uar."remoteAccess"),
							config = (nt.config OR uar.config),
							"grant" = (nt."grant" OR uar."grant"),
							snapshot = (nt.snapshot OR uar.snapshot)
				FROM new_table as nt
				INNER JOIN "___ORGANIZATION_ID___".filter_result as fr
					ON nt."filterId" = fr."filterId"
				WHERE uar."accessControlPointId" = nt."accessControlPointId" AND uar."userId" = fr."userId";
			ELSIF (TG_OP = 'DELETE') THEN
					WITH  will_be_updated_users AS (
						SELECT "userId",
								fr."filterId",
								ot."accessControlPointId"
						FROM old_table as ot
						INNER JOIN "___ORGANIZATION_ID___".filter_result as fr
								ON fr."filterId" = ot."filterId"
					), deleted_users AS (
						DELETE FROM "___ORGANIZATION_ID___"."userAccessRights" as uar
						USING will_be_updated_users as wbuu
						WHERE wbuu."accessControlPointId" = uar."accessControlPointId" AND uar."userId" = wbuu."userId" AND
							  (uar."userId" != ALL( SELECT fr."userId" 
							  						FROM "___ORGANIZATION_ID___".user_filter_access_control_points as ufacp
												 	INNER JOIN "___ORGANIZATION_ID___".filter_result as fr
												   		ON fr."filterId" = ufacp."filterId"
												    WHERE ufacp."accessControlPointId" = wbuu."accessControlPointId"
												   ))
					)
					UPDATE "___ORGANIZATION_ID___"."userAccessRights" as uar
					SET  	"updatedAt" = now(), 
							"read" = sub_q."read",
							"access" = sub_q.access,
							"remoteAccess" = sub_q."remoteAccess",
							"config" = sub_q.config,
							"grant" = sub_q."grant",
							"snapshot" = sub_q.snapshot
					FROM	(
								SELECT fr."userId",
									ufacp."accessControlPointId",
										BOOL_OR(ufacp."read") as "read",
										BOOL_OR(ufacp.access) as "access",
										BOOL_OR(ufacp."remoteAccess") as "remoteAccess",
										BOOL_OR(ufacp.config) as "config",
										BOOL_OR(ufacp."grant") as "grant",
										BOOL_OR(ufacp.snapshot) as "snapshot"
								FROM "___ORGANIZATION_ID___".user_filter_access_control_points as ufacp
								INNER JOIN "___ORGANIZATION_ID___".filter_result as fr
									ON ufacp."filterId" = fr."filterId"
								INNER JOIN "will_be_updated_users" AS wbuu
									ON fr."userId" = wbuu."userId"
										AND ufacp."accessControlPointId" = wbuu."accessControlPointId"
								GROUP BY fr."userId", ufacp."accessControlPointId"
							) sub_q
					WHERE sub_q."accessControlPointId" = uar."accessControlPointId" 
						AND uar."userId" = sub_q."userId";
			ELSE 
				UPDATE "___ORGANIZATION_ID___"."userAccessRights" as uar
				SET	"updatedAt"=now(), 
					"read" = sub_q."read",
					"access" = sub_q."access",
					"remoteAccess" = sub_q."remoteAccess",
					"config" = sub_q.config,
					"grant" = sub_q.grant,
					"snapshot" = sub_q.snapshot
				FROM (
						SELECT  "userId",
								ufacp."accessControlPointId",
								BOOL_OR(ufacp."read") as "read",
								BOOL_OR(ufacp.access) as "access",
								BOOL_OR(ufacp."remoteAccess") as "remoteAccess",
								BOOL_OR(ufacp.config) as "config",
								BOOL_OR(ufacp."grant") as "grant",
								BOOL_OR(ufacp.snapshot) as "snapshot"
						FROM "___ORGANIZATION_ID___"."user_filter_access_control_points" as ufacp
						INNER JOIN new_table as nt
							ON ufacp."accessControlPointId" = nt."accessControlPointId"
						INNER JOIN "___ORGANIZATION_ID___".filter_result as fr
							ON ufacp."filterId" = fr."filterId"
						GROUP BY "userId", ufacp."accessControlPointId"
				) sub_q
				WHERE uar."userId" = sub_q."userId" AND sub_q."accessControlPointId" = uar."accessControlPointId";
			END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;


		CREATE TRIGGER after_insert_user_filter_access_control_points
			AFTER INSERT ON "___ORGANIZATION_ID___"."user_filter_access_control_points"
			REFERENCING NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_filter_access_control_points();

		CREATE TRIGGER after_update_user_filter_access_control_points
			AFTER UPDATE ON "___ORGANIZATION_ID___"."user_filter_access_control_points"
			REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_filter_access_control_points();

		CREATE TRIGGER after_delete_user_filter_access_control_points
			AFTER DELETE ON "___ORGANIZATION_ID___"."user_filter_access_control_points"
			REFERENCING OLD TABLE AS old_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_filter_access_control_points();

		ALTER TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" DISABLE TRIGGER after_insert_user_filter_access_control_points;
		ALTER TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" DISABLE TRIGGER after_update_user_filter_access_control_points;
		ALTER TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" DISABLE TRIGGER after_delete_user_filter_access_control_points;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER  after_insert_user_filter_access_control_points ON "___ORGANIZATION_ID___"."user_filter_access_control_points";

 		DROP TRIGGER  after_update_user_filter_access_control_points ON "___ORGANIZATION_ID___"."user_filter_access_control_points";

 		DROP TRIGGER  after_delete_user_filter_access_control_points ON "___ORGANIZATION_ID___"."user_filter_access_control_points";	

		DROP FUNCTION "___ORGANIZATION_ID___".after_change_user_filter_access_control_points();	
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterAcpMappingsTableQueries = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE TABLE "___ORGANIZATION_ID___".user_filter_access_control_points
		(
			id UUID PRIMARY KEY,
			"filterId" UUID,
			"accessControlPointId" UUID,
			"read" BOOLEAN,
			"access" BOOLEAN,
			"remoteAccess" BOOLEAN,
			"config" BOOLEAN,
			"grant" BOOLEAN,
			"snapshot" BOOLEAN,
			CONSTRAINT user_filter_acp_user_filter_fk
			FOREIGN KEY ("filterId") REFERENCES "___ORGANIZATION_ID___"."user_filter" (id) ON DELETE CASCADE,
			CONSTRAINT user_filter_acp_acp_fk
			FOREIGN KEY ("accessControlPointId") REFERENCES "___ORGANIZATION_ID___"."accessControlPoints" (id) ON DELETE CASCADE,
			CONSTRAINT unique_user_filter_acp_filter_id_acp_id UNIQUE ("filterId", "accessControlPointId")
		);
		
		ALTER TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" OWNER TO ${dbsuperuser};
		GRANT INSERT, SELECT, DELETE, UPDATE ON TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" TO ${dbuser};
		GRANT ALL ON TABLE "___ORGANIZATION_ID___"."user_filter_access_control_points" TO ${dbsuperuser};
	`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = ` DROP TABLE "___ORGANIZATION_ID___".user_filter_access_control_points; `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterView = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `	
		DROP VIEW IF EXISTS "___ORGANIZATION_ID___".vw_user_filter;

		CREATE VIEW "___ORGANIZATION_ID___".vw_user_filter
		AS
		SELECT  uf.id,
				jsonb_strip_nulls(jsonb_build_object(
									'id', uf.id,
									'name', uf.name,
									'description', uf.description,
									'combinator', uf.combinator,
									'usageType', uf."usageType",
									'allIncluded', uf."allIncluded",
									'groupIds', COALESCE(jsonb_agg(DISTINCT ufg."groupId") FILTER (WHERE ufg."groupId" IS NOT NULL), NULL::jsonb), 
									'roleIds', COALESCE(jsonb_agg(DISTINCT ufor."roleId") FILTER (WHERE ufor."roleId" IS NOT NULL), NULL::jsonb), 
									'userIds', COALESCE(jsonb_agg(DISTINCT ufu."userId") FILTER (WHERE ufu."userId" IS NOT NULL), NULL::jsonb), 
									'workPlanIds', COALESCE(jsonb_agg(DISTINCT ufw."workPlanId") FILTER (WHERE ufw."workPlanId" IS NOT NULL), NULL::jsonb), 
									'units', COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', ufn."unitId", 'hierarchically', ufn.hierarchically, 'roleIds', ufuragg."roleIds")) 
											FILTER (WHERE ufn."unitId" IS NOT NULL), NULL::jsonb))
								)AS filter
		FROM "___ORGANIZATION_ID___".user_filter uf
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_group ufg ON uf.id = ufg."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role ufor ON uf.id = ufor."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_user ufu ON uf.id = ufu."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_work_plan ufw ON uf.id = ufw."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_unit ufn ON uf.id = ufn."filterId"
		LEFT JOIN ( SELECT	ufur."filterId",
							ufur."unitId",
							COALESCE(json_agg(ufur."roleId") FILTER (WHERE ufur."roleId" IS NOT NULL), NULL::json) AS "roleIds"
					FROM "___ORGANIZATION_ID___".user_filter_unit_role ufur
					GROUP BY ufur."filterId", ufur."unitId") ufuragg 
			ON ufuragg."unitId" = ufn."unitId" AND ufn."filterId" = ufuragg."filterId"
		GROUP BY uf.id;
		
		ALTER TABLE "___ORGANIZATION_ID___".vw_user_filter
			OWNER TO ${dbsuperuser};

		GRANT INSERT, SELECT, UPDATE, DELETE ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbuser};
		GRANT ALL ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbsuperuser};`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP VIEW IF EXISTS "___ORGANIZATION_ID___".vw_user_filter;

		CREATE VIEW "___ORGANIZATION_ID___".vw_user_filter
		AS
		SELECT	uf.id,
				json_build_object('id', uf.id, 'groupIds', COALESCE(jsonb_agg(DISTINCT ufg."groupId") FILTER (WHERE ufg."groupId" IS NOT NULL), NULL::jsonb), 'roleIds', COALESCE(jsonb_agg(DISTINCT ufor."roleId") FILTER (WHERE ufor."roleId" IS NOT NULL), NULL::jsonb), 'userIds', COALESCE(jsonb_agg(DISTINCT ufu."userId") FILTER (WHERE ufu."userId" IS NOT NULL), NULL::jsonb), 'workPlanIds', COALESCE(jsonb_agg(DISTINCT ufw."workPlanId") FILTER (WHERE ufw."workPlanId" IS NOT NULL), NULL::jsonb), 'units', COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', ufn."unitId", 'hierarchically', ufn.hierarchically, 'roleIds', ufuragg."roleIds")) FILTER (WHERE ufn."unitId" IS NOT NULL), NULL::jsonb)) AS filter
		FROM "___ORGANIZATION_ID___".user_filter uf
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_group ufg ON uf.id = ufg."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role ufor ON uf.id = ufor."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_user ufu ON uf.id = ufu."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_work_plan ufw ON uf.id = ufw."filterId"
		LEFT JOIN "___ORGANIZATION_ID___".user_filter_unit ufn ON uf.id = ufn."filterId"
		LEFT JOIN ( SELECT ufur."filterId",
					ufur."unitId",
					COALESCE(json_agg(ufur."roleId") FILTER (WHERE ufur."roleId" IS NOT NULL), NULL::json) AS "roleIds"
				FROM "___ORGANIZATION_ID___".user_filter_unit_role ufur
				GROUP BY ufur."filterId", ufur."unitId") ufuragg ON ufuragg."unitId" = ufn."unitId" AND ufn."filterId" = ufuragg."filterId"
		GROUP BY uf.id;

		ALTER TABLE "___ORGANIZATION_ID___".vw_user_filter
			OWNER TO ${dbsuperuser};

		GRANT INSERT, SELECT, UPDATE, DELETE ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbuser};
		GRANT ALL ON TABLE "___ORGANIZATION_ID___".vw_user_filter TO ${dbsuperuser};`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const changesOnUserOrganization = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_organization()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
		IF (TG_OP = 'INSERT') THEN
			SELECT array_agg(DISTINCT(uf.id)) into updated_filter_ids
			FROM "___ORGANIZATION_ID___".user_filter uf
			LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role as ufr
				ON uf.id = ufr."filterId"
			CROSS JOIN new_table as nt 
			INNER JOIN "___ORGANIZATION_ID___".roles as r 
				ON nt."roleId" = r.id
			WHERE nt."isDisabled" = false 
				AND nt."deletedAt" IS NULL 
				AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
				AND (uf."allIncluded" IS TRUE OR ufr."roleId" = nt."roleId")
				AND uf."usageType" = 3;
		ELSIF (TG_OP = 'UPDATE') THEN
			SELECT array_agg(DISTINCT(uf.id)) into updated_filter_ids
			FROM "___ORGANIZATION_ID___"."user_filter" as uf
			LEFT JOIN "___ORGANIZATION_ID___"."user_filter_organization_role" as ufor
				ON uf.id = ufor."filterId"
			CROSS JOIN 
					(
					SELECT nt."roleId" as new_role,
						ot."roleId" as old_role
					FROM new_table as nt
					INNER JOIN old_table as ot
						ON  nt.id = ot.id 
							AND (ot."isDisabled" IS DISTINCT FROM nt."isDisabled" OR ot."deletedAt" IS DISTINCT FROM nt."deletedAt" OR ot."roleId" IS DISTINCT FROM nt."roleId")
					) subq
			WHERE 	uf."usageType" = 3 
					AND (
						(ufor."roleId" = subq.new_role OR ufor."roleId" = subq.old_role)
						OR (subq.new_role = subq.old_role AND uf."allIncluded" IS TRUE)												   
					);
						
			DELETE FROM "___ORGANIZATION_ID___".user_filter_user as ufu
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			INNER JOIN "___ORGANIZATION_ID___".roles as r
				ON nt."roleId" = r."id"
			WHERE 	ufu."userId" = nt."userId"
					AND ((nt."isDisabled" = true AND ot."isDisabled" = false) OR (nt."deletedAt" IS NOT NULL AND ot."deletedAt" IS NULL))
					AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7');
			
		ELSIF (TG_OP = 'DELETE') THEN
			SELECT array_agg(DISTINCT(uf.id)) into updated_filter_ids
			FROM "___ORGANIZATION_ID___".user_filter as uf
			LEFT JOIN "___ORGANIZATION_ID___".user_filter_organization_role as ufor
				ON uf.id = ufor."filterId"
			CROSS JOIN (
						SELECT ot."roleId"
						FROM old_table as ot
						WHERE ot."deletedAt" IS NULL AND ot."isDisabled" IS FALSE
						) subq
			WHERE  uf."usageType" = 3
				AND (subq."roleId" = ufor."roleId" OR uf."allIncluded" IS TRUE);
		
			DELETE FROM "___ORGANIZATION_ID___".user_filter_user as ufu
			USING old_table as ot
			INNER JOIN "___ORGANIZATION_ID___".roles as r
				ON ot."roleId" = r."id"
			WHERE 	ufu."userId" = ot."userId"
					AND (ot."deletedAt" IS NULL AND ot."isDisabled" IS FALSE)
					AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7');
		END IF;
		IF (updated_filter_ids IS NOT NULL ) THEN 
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result)
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId"
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		END IF;
		RETURN NULL;
		END;

		$$ LANGUAGE plpgsql;

		CREATE OR REPLACE TRIGGER after_update_user_organization
			AFTER UPDATE ON "___ORGANIZATION_ID___"."userOrganizations"
			REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_organization();
			
		CREATE OR REPLACE TRIGGER after_insert_user_organization
			AFTER INSERT ON "___ORGANIZATION_ID___"."userOrganizations"
			REFERENCING  NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_organization();

		CREATE OR REPLACE TRIGGER after_delete_user_organization
			AFTER DELETE ON "___ORGANIZATION_ID___"."userOrganizations"
			REFERENCING OLD TABLE AS old_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_organization();
			
		ALTER TABLE "___ORGANIZATION_ID___"."userOrganizations" DISABLE TRIGGER after_update_user_organization;
		ALTER TABLE "___ORGANIZATION_ID___"."userOrganizations" DISABLE TRIGGER after_insert_user_organization;
		
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER after_delete_user_organization ON "___ORGANIZATION_ID___"."userOrganizations";
		DROP TRIGGER after_update_user_organization ON "___ORGANIZATION_ID___"."userOrganizations";
		DROP TRIGGER after_insert_user_organization ON "___ORGANIZATION_ID___"."userOrganizations";
		DROP FUNCTION "___ORGANIZATION_ID___".after_change_user_organization();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateUserOrganizationAfterUserDeleted = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_update_user_deleted_statement()
		RETURNS TRIGGER AS $$
		BEGIN

			UPDATE "___ORGANIZATION_ID___"."userOrganizations"
			SET "deletedAt" = NEW."deletedAt"
			WHERE "userId" = NEW.id;

		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;

		CREATE TRIGGER after_update_user_deleted_statement
			AFTER UPDATE ON "___ORGANIZATION_ID___"."users"
			FOR EACH ROW
			WHEN (OLD."deletedAt" IS DISTINCT FROM NEW."deletedAt" ) 
			EXECUTE FUNCTION "___ORGANIZATION_ID___".after_update_user_deleted_statement();

		ALTER TABLE "___ORGANIZATION_ID___"."users" DISABLE TRIGGER after_update_user_deleted_statement;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `	
		DROP TRIGGER  after_update_user_deleted_statement ON "___ORGANIZATION_ID___"."users";
    	DROP FUNCTION  "___ORGANIZATION_ID___".after_update_user_deleted_statement();`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const createUserWorkPlanTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_work_plan()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN

			IF (TG_OP = 'DELETE') THEN
				SELECT ARRAY (
								SELECT uf.id 
								FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
								INNER JOIN old_table as ot
									ON ufwp."workPlanId" = ot."workPlanId"
								INNER JOIN "___ORGANIZATION_ID___".user_filter uf
									ON ufwp."filterId" = uf.id
								WHERE  uf."usageType" = 3
				) INTO updated_filter_ids;
				
			ELSIF (TG_OP = 'INSERT') THEN
				SELECT ARRAY (
								SELECT uf.id 
								FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
								INNER JOIN new_table as nt
									ON ufwp."workPlanId" = nt."workPlanId"
								INNER JOIN "___ORGANIZATION_ID___".user_filter uf
									ON ufwp."filterId" = uf.id
								WHERE  uf."usageType" = 3
				) INTO updated_filter_ids; 
			ELSE 
				SELECT ARRAY (
								SELECT uf.id 
								FROM "___ORGANIZATION_ID___".user_filter_work_plan ufwp
								INNER JOIN 
										(
											SELECT "workPlanId"
											FROM new_table
											UNION
											SELECT "workPlanId"
											FROM old_table
										) updated_table
									ON ufwp."workPlanId" = updated_table."workPlanId"
								INNER JOIN "___ORGANIZATION_ID___".user_filter uf
									ON ufwp."filterId" = uf.id
								WHERE uf."usageType" = 3
				) INTO updated_filter_ids;
			END IF;
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP 
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result) 
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId" 
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;


		CREATE TRIGGER after_insert_user_work_plan
			AFTER INSERT ON "___ORGANIZATION_ID___"."userWorkPlans"
			REFERENCING NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_work_plan();
			
		CREATE TRIGGER after_update_user_work_plan
			AFTER UPDATE ON "___ORGANIZATION_ID___"."userWorkPlans"
			REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_work_plan();
			
		CREATE TRIGGER after_delete_user_work_plan
			AFTER DELETE ON "___ORGANIZATION_ID___"."userWorkPlans"
			REFERENCING OLD TABLE AS old_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_work_plan();

		ALTER TABLE "___ORGANIZATION_ID___"."userWorkPlans" DISABLE TRIGGER after_insert_user_work_plan;
		ALTER TABLE "___ORGANIZATION_ID___"."userWorkPlans" DISABLE TRIGGER after_update_user_work_plan;
		ALTER TABLE "___ORGANIZATION_ID___"."userWorkPlans" DISABLE TRIGGER after_delete_user_work_plan;
		
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER  after_insert_user_work_plan ON "___ORGANIZATION_ID___"."userWorkPlans";

		DROP TRIGGER  after_update_user_work_plan ON "___ORGANIZATION_ID___"."userWorkPlans";

		DROP TRIGGER  after_delete_user_work_plan ON "___ORGANIZATION_ID___"."userWorkPlans";

		DROP FUNCTION "___ORGANIZATION_ID___".after_change_user_work_plan();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const createUserGroupTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_group()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN

			IF (TG_OP = 'DELETE') THEN
				SELECT ARRAY (
								SELECT uf.id 
								FROM "___ORGANIZATION_ID___".user_filter_group ufg
								INNER JOIN old_table as ot
									ON ufg."groupId" = ot."userGroupId"
								INNER JOIN "___ORGANIZATION_ID___".user_filter uf
									ON ufg."filterId" = uf.id
								WHERE  uf."usageType" = 3
				) INTO updated_filter_ids;
				
			ELSIF (TG_OP = 'INSERT') THEN
				SELECT ARRAY (
								SELECT uf.id 
								FROM "___ORGANIZATION_ID___".user_filter_group ufg
								INNER JOIN new_table as nt
									ON ufg."groupId" = nt."userGroupId"
								INNER JOIN "___ORGANIZATION_ID___".user_filter uf
									ON ufg."filterId" = uf.id
								WHERE  uf."usageType" = 3
				) INTO updated_filter_ids; 
			ELSE 
				SELECT ARRAY (
								SELECT uf.id 
								FROM "___ORGANIZATION_ID___".user_filter_group ufg
								INNER JOIN 
										(
											SELECT nt."userGroupId"
											FROM new_table as nt
											INNER JOIN old_table as ot
												ON nt.id = ot.id
											WHERE nt."deletedAt" IS DISTINCT FROM ot."deletedAt"
										) updated_table
									ON ufg."groupId" = updated_table."userGroupId"
								INNER JOIN "___ORGANIZATION_ID___".user_filter uf
									ON ufg."filterId" = uf.id
								WHERE uf."usageType" = 3
				) INTO updated_filter_ids;
			END IF;
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP 
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result) 
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId" 
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;


		CREATE TRIGGER after_insert_user_group_user_organization
			AFTER INSERT ON "___ORGANIZATION_ID___"."userGroupUserOrganizations"
			REFERENCING NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_group();
			
		CREATE TRIGGER after_update_user_group_user_organization
			AFTER UPDATE ON "___ORGANIZATION_ID___"."userGroupUserOrganizations"
			REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_group();
			
		CREATE TRIGGER after_delete_user_group_user_organization
			AFTER DELETE ON "___ORGANIZATION_ID___"."userGroupUserOrganizations"
			REFERENCING OLD TABLE AS old_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_group();

		ALTER TABLE "___ORGANIZATION_ID___"."userGroupUserOrganizations" DISABLE TRIGGER after_insert_user_group_user_organization;
		ALTER TABLE "___ORGANIZATION_ID___"."userGroupUserOrganizations" DISABLE TRIGGER after_update_user_group_user_organization;
		ALTER TABLE "___ORGANIZATION_ID___"."userGroupUserOrganizations" DISABLE TRIGGER after_delete_user_group_user_organization;
		
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER  after_insert_user_group_user_organization ON "___ORGANIZATION_ID___"."userGroupUserOrganizations";

		DROP TRIGGER  after_update_user_group_user_organization ON "___ORGANIZATION_ID___"."userGroupUserOrganizations";

		DROP TRIGGER  after_delete_user_group_user_organization ON "___ORGANIZATION_ID___"."userGroupUserOrganizations";

		DROP FUNCTION "___ORGANIZATION_ID___".after_change_user_group();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userUnitFilterUpdateTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_change_user_unit()
		RETURNS TRIGGER AS $$
		DECLARE
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
			IF (TG_OP = 'DELETE') THEN
				SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
							SELECT 	ou."id" as "unitId", 
									ou."ancestorIds"
							FROM old_table as ot
							INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
								ON  ot."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL
							) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%')
				WHERE uf."usageType" = 3;
			ELSIF (TG_OP = 'INSERT') THEN
				SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
							SELECT 	ou."id" as "unitId", 
									ou."ancestorIds"
							FROM new_table as nt
							INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
								ON  nt."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL
							) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" || '%')
				WHERE uf."usageType" = 3;
			ELSE 
				SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
				FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
				INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
					ON uf.id = ufu."filterId"
				INNER JOIN (
							SELECT 	ou."id" as "unitId", 
									ou."ancestorIds"
							FROM old_table as ot
							INNER JOIN new_table as nt
								ON ot.id = nt.id
							INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
								ON  (ot."organizationUnitId" = ou."id" OR nt."organizationUnitId" = ou.id) 
									AND ou."deletedAt" IS NULL
							WHERE ot."deletedAt" IS DISTINCT FROM nt."deletedAt" OR ot."roleId" IS DISTINCT FROM nt."roleId"
							) subq
					ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%')
				WHERE   uf."usageType" = 3;
			END IF;
		IF (updated_filter_ids IS NOT NULL) THEN
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP 
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result) 
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId" 
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		END IF;
		RETURN NULL;
		END;
		$$ LANGUAGE plpgsql;


		CREATE TRIGGER after_insert_user_unit
			AFTER INSERT ON "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits"
			REFERENCING NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_unit();
			
		CREATE TRIGGER after_update_user_unit
			AFTER UPDATE ON "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits"
			REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_unit();
			
		CREATE TRIGGER after_delete_user_unit
			AFTER DELETE ON "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits"
			REFERENCING OLD TABLE AS old_table
			FOR EACH STATEMENT EXECUTE FUNCTION "___ORGANIZATION_ID___".after_change_user_unit();
		
		ALTER TABLE "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" DISABLE TRIGGER after_insert_user_unit;
		ALTER TABLE "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" DISABLE TRIGGER after_update_user_unit;
		ALTER TABLE "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" DISABLE TRIGGER after_delete_user_unit;
 		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER  after_insert_user_unit ON "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits";

		DROP TRIGGER  after_update_user_unit ON "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits";

		DROP TRIGGER  after_delete_user_unit ON "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits";

		DROP FUNCTION "___ORGANIZATION_ID___".after_change_user_unit();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateUserFilterUserCount = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		SELECT (SELECT COUNT("userId")::integer FROM "___ORGANIZATION_ID___".user_filter_result(uf.id))
		FROM "___ORGANIZATION_ID___".user_filter as uf;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const createUserFilterResultFunction = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
			CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".user_filter_result(filter_id uuid)
			RETURNS TABLE("userId" uuid)
			LANGUAGE 'plpgsql'
			COST 100
			VOLATILE NOT LEAKPROOF
			AS $BODY$
			BEGIN
				RETURN QUERY
				WITH all_users AS (
					SELECT uo."userId"
					FROM "___ORGANIZATION_ID___"."userOrganizations" uo
						INNER JOIN "___ORGANIZATION_ID___"."roles" r
							ON uo."roleId" = r."id"
						WHERE uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND (r."typeId" IS DISTINCT FROM '9b8eebd7-057f-4021-ad6f-f2879733bdb9' AND r."typeId" IS DISTINCT FROM '2a4ad5ff-1ced-45f8-992f-e19fc4e766d7')
				),
				filter_definition AS (
					SELECT
						uf.id, uf.combinator, uf."allIncluded",
						BOOL_OR(ufg."filterId" IS NOT NULL) as has_group,
						BOOL_OR(ufor."filterId" IS NOT NULL) as has_role,
						BOOL_OR(ufu."filterId" IS NOT NULL) as has_unit,
						BOOL_OR(ufw."filterId" IS NOT NULL) as has_work_plan
					FROM "___ORGANIZATION_ID___"."user_filter" uf
					LEFT JOIN "___ORGANIZATION_ID___"."user_filter_group" ufg
						ON uf.id = ufg."filterId"
					LEFT JOIN "___ORGANIZATION_ID___"."user_filter_organization_role" ufor
						ON uf.id = ufor."filterId"
					LEFT JOIN "___ORGANIZATION_ID___"."user_filter_unit" ufu
						ON uf.id = ufu."filterId"
					LEFT JOIN "___ORGANIZATION_ID___"."user_filter_work_plan" ufw
						ON uf.id = ufw."filterId"
					WHERE uf.id = filter_id
					GROUP BY uf.id
				),
				group_users AS (
					SELECT DISTINCT uo."userId"
					FROM "___ORGANIZATION_ID___"."user_filter_group" ufg
					INNER JOIN "___ORGANIZATION_ID___"."userGroupUserOrganizations" uguo
						ON ufg."groupId" = uguo."userGroupId" AND uguo."deletedAt" IS NULL
					INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
						ON uguo."userOrganizationId" = uo.id
					WHERE ufg."filterId" = filter_id
				),
				specific_users AS (
					SELECT DISTINCT ufu."userId"
					FROM "___ORGANIZATION_ID___"."user_filter_user" ufu
					WHERE ufu."filterId" = filter_id
				),
				role_users AS (
					SELECT DISTINCT uo."userId"
					FROM "___ORGANIZATION_ID___"."user_filter_organization_role" ufor
					INNER JOIN "___ORGANIZATION_ID___"."roles" r
						ON ufor."roleId" = r.id AND r."deletedAt" IS NULL
					INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
						ON ufor."roleId" = uo."roleId"
					WHERE ufor."filterId" = filter_id
				),
				workplan_users AS (
					SELECT DISTINCT uwp."userId"
					FROM "___ORGANIZATION_ID___"."user_filter_work_plan" ufwp
					INNER JOIN "___ORGANIZATION_ID___"."workPlans" wp
						ON ufwp."workPlanId" = wp."id" AND wp."deletedAt" IS NULL
					INNER JOIN "___ORGANIZATION_ID___"."userWorkPlans" uwp
						ON ufwp."workPlanId" = uwp."workPlanId" AND tstzrange("startDateTime","endDateTime") @> now()
					WHERE ufwp."filterId" = filter_id
				),
				unit_users AS (
					SELECT DISTINCT uo."userId"
					FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
					LEFT JOIN "___ORGANIZATION_ID___"."user_filter_unit_role" ufur
						ON ufur."filterId" = ufu."filterId" AND ufur."unitId" = ufu."unitId"
					INNER JOIN "___ORGANIZATION_ID___"."organizationUnits" ou
						ON ou."deletedAt" IS NULL AND (ou.id = ufu."unitId" OR (ufu.hierarchically AND ou."ancestorIds" ILIKE '%' || ufu."unitId" || '%'))
					INNER JOIN "___ORGANIZATION_ID___"."userOrganizationOrganizationUnits" uoou
						ON uoou."deletedAt" IS NULL AND ou."id" = uoou."organizationUnitId" AND
							(ufur."roleId" IS NULL OR ufur."roleId" = uoou."roleId")
					INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" uo
						ON uo."id" = uoou."userOrganizationId" AND NOT uo."isDisabled" AND uo."deletedAt" IS NULL
					WHERE
						ufu."filterId" = filter_id
				),
				filter_result AS (
					SELECT all_users."userId"
					FROM all_users
					INNER JOIN filter_definition ON true
					LEFT JOIN group_users ON group_users."userId" = all_users."userId"
					LEFT JOIN specific_users ON specific_users."userId" = all_users."userId"
					LEFT JOIN role_users ON role_users."userId" = all_users."userId"
					LEFT JOIN workplan_users ON workplan_users."userId" = all_users."userId"
					LEFT JOIN unit_users ON unit_users."userId" = all_users."userId"
					WHERE filter_definition."allIncluded"
						OR specific_users."userId" IS NOT NULL
						OR (filter_definition.combinator = 1 AND
							(filter_definition.has_group OR filter_definition.has_role OR filter_definition.has_work_plan OR filter_definition.has_unit) AND
							(NOT filter_definition.has_group OR group_users."userId" IS NOT NULL) AND
							(NOT filter_definition.has_role OR role_users."userId" IS NOT NULL) AND
							(NOT filter_definition.has_work_plan OR workplan_users."userId" IS NOT NULL) AND
							(NOT filter_definition.has_unit OR unit_users."userId" IS NOT NULL)
						)
						OR (filter_definition.combinator = 2 AND
							(group_users."userId" IS NOT NULL OR role_users."userId" IS NOT NULL OR workplan_users."userId" IS NOT NULL OR unit_users."userId" IS NOT NULL)
						)
				),
				update_user_count AS (
					UPDATE "___ORGANIZATION_ID___".user_filter
					SET "userCount" = (SELECT COUNT(fr."userId") FROM filter_result fr)
					WHERE id = filter_id
				)
				SELECT *
				FROM filter_result;
			END;
			$BODY$;
		
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `DROP FUNCTION "___ORGANIZATION_ID___".user_filter_result(uuid);`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateUserFilterNameAndUsage = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		-- NOTIFICATION 
		WITH notification_user_filter_names AS (
			SELECT  		CASE WHEN n.type = 1  THEN 'Erişim Bildirimi Filtresi ' || ranks
								WHEN n.type = 2  THEN 'Sensör Durumu Bildirim Filtresi ' || ranks
								WHEN n.type = 4  THEN 'Ziyaretçi Geldi Bildirimi Filtresi' || ranks
								WHEN n.type = 6  THEN 'İzin Onay Bildirimi Filtersi ' || ranks
								WHEN n.type = 10  THEN 'Yeni Geri Bildirim Filtersi ' || ranks
								WHEN n.type = 11  THEN 'Geri Bildirim Güncellendi Filtersi ' || ranks
								WHEN n.type = 15  THEN 'Duyuru Bildirimi Filtersi ' || ranks
								WHEN n.type = 19 THEN 'Planlanmış İş Bildirim Filtresi ' || ranks
								WHEN n.type = 27 THEN 'İzin Onay Bekliyor Bildirim Filtresi ' || ranks
								WHEN n.type = 31 THEN 'Bireysel İşe Başlamama Bildirim Filtresi ' || ranks
								WHEN n.type = 32 THEN 'Bireysel Eksik Çalışma Bildirim Filtresi ' || ranks
								WHEN n.type = 38 THEN 'Terminal Bağlandı Bildirim Filtresi ' || ranks
								WHEN n.type = 39 THEN 'Terminal Bağlantı Koptu Bildirim Filtresi ' || ranks
								WHEN n.type = 41 THEN 'Raporlama Servisi Hata Bildirim Filtresi ' || ranks
								WHEN n.type = 45 THEN 'Bireysel Erişim Bildirim Filtresi ' || ranks
								WHEN n.type = 46 THEN 'Kullanıcı Hesabı Oluşturuldu Bildirim Filtresi ' || ranks
								WHEN n.type = 49 THEN 'Bireysel İş Bırakmama Bildirim Filtresi ' || ranks
								ELSE ''
							END as name,
							"receiverFilterId"
					FROM
							(	
								SELECT subq.type,
									(rank() OVER (PARTITION BY subq.type ORDER BY subq."receiverFilterId" DESC))::text  as ranks,
									subq."receiverFilterId"
								FROM
								(
									SELECT n.type, n."receiverFilterId"
									FROM "___ORGANIZATION_ID___"."notification"  as n
									WHERE n."receiverFilterId" IS NOT NULL
									UNION
									SELECT n.type, n."receiverFilterId"
									FROM "___ORGANIZATION_ID___"."zz_notification"  as n
									WHERE n."receiverFilterId" IS NOT NULL
								)subq
								ORDER BY subq."receiverFilterId" DESC
							) as n
		)	
		UPDATE "___ORGANIZATION_ID___"."user_filter" as uf
		SET name = nufn.name, "usageType" = 2, description = nufn.name
		FROM notification_user_filter_names as nufn
		WHERE uf.id = nufn."receiverFilterId" AND uf.name IS NULL;

		-- CUSTOM REPORTS
		UPDATE "___ORGANIZATION_ID___"."user_filter" as uf
		SET name = cr.name, description = cr.name, "usageType" = 2
		FROM "___ORGANIZATION_ID___"."customReports" as cr
		WHERE uf.id = cr."userFilterId" AND uf.name IS NULL;

		-- Permissions
		UPDATE "___ORGANIZATION_ID___"."user_filter" as uf
		SET name = subq.name || ' Filtresi ' || ranks, "usageType" = 1, description = subq.name || ' Filtresi '
		FROM (
			SELECT  "userFilterId", 
					pt.name,
					(rank() OVER(PARTITION BY "ppermissionTypeId" ORDER BY "ppermissionTypeId" DESC))::text  as ranks
			FROM "___ORGANIZATION_ID___"."ppermission_type_rules" as ptr
			INNER JOIN "___ORGANIZATION_ID___"."ppermissionTypes" as pt
				ON ptr."ppermissionTypeId" = pt.id
			ORDER BY "ppermissionTypeId" DESC
		) subq
		WHERE uf.id = subq."userFilterId";
 		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const removeUnusedUserFilters = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		WITH unused_user_filters AS
		(
			SELECT uf.id 
			FROM (	
						SELECT DISTINCT("userFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."customReports"
						UNION 
						SELECT DISTINCT("receiverFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."notification"
						UNION
						SELECT DISTINCT("receiverFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."notification_event"
						UNION
						SELECT DISTINCT("subjectedEmployeesFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."pacs_notification_scheduled_job"
						UNION
						SELECT DISTINCT("userFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."ppermission_type_rules"
						UNION
						SELECT DISTINCT("userFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."preparedReports"
						UNION
						SELECT DISTINCT("receiverFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."zz_notification"
						UNION 
						SELECT DISTINCT("receiverFilterId") AS "filterId"
						FROM "___ORGANIZATION_ID___"."zz_notification_event"
			)subq
			RIGHT JOIN "___ORGANIZATION_ID___".user_filter as uf
				ON subq."filterId" = uf.id
			WHERE subq."filterId" IS NULL
		)
		DELETE FROM "___ORGANIZATION_ID___".user_filter as uf
		USING unused_user_filters uuf
		WHERE uf.id = uuf.id;
 		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const updateNotificationEventsUserFilter = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		UPDATE "___ORGANIZATION_ID___".notification_event as ne
		SET "receiverFilterId" = NULL
		FROM "___ORGANIZATION_ID___".notification as n
		WHERE n.id = ne."notificationId" AND n."receiverFilterId" IS DISTINCT FROM  ne."receiverFilterId";

		UPDATE "___ORGANIZATION_ID___".zz_notification_event as ne
		SET "receiverFilterId" = NULL
		FROM "___ORGANIZATION_ID___".zz_notification as n
		WHERE n.id = ne."notificationId" AND n."receiverFilterId" IS DISTINCT FROM  ne."receiverFilterId";

		UPDATE "___ORGANIZATION_ID___".zz_notification_event as ne
		SET "receiverFilterId" = NULL
		FROM "___ORGANIZATION_ID___".notification as n
		WHERE n.id = ne."notificationId" AND n."receiverFilterId" IS DISTINCT FROM  ne."receiverFilterId";

		UPDATE "___ORGANIZATION_ID___"."preparedReports" as pr
		SET "userFilterId" = NULL
		WHERE pr."userFilterId" != ALL(
										SELECT DISTINCT("receiverFilterId") FROM "___ORGANIZATION_ID___".notification as n 
										WHERE "receiverFilterId" IS NOT NULL
										UNION
										SELECT DISTINCT("receiverFilterId") FROM "___ORGANIZATION_ID___".zz_notification as n 
										WHERE "receiverFilterId" IS NOT NULL
									);
 		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterColumns = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		ALTER TABLE "___ORGANIZATION_ID___".user_filter
		ADD COLUMN name character varying(255),
		ADD COLUMN description character varying(255), 
		ADD COLUMN combinator smallint DEFAULT 2,
		ADD COLUMN "allIncluded" boolean NOT NULL DEFAULT false,
		ADD COLUMN "usageType" smallint NOT NULL DEFAULT 2,
		ADD COLUMN "userCount" integer NOT NULL DEFAULT 0,
		ADD CONSTRAINT check_combinator
		CHECK (combinator IN(1,2)),
		ADD CONSTRAINT check_usage_type
		CHECK ("usageType" IN(1,2,3,4));
 		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		ALTER TABLE "___ORGANIZATION_ID___".user_filter
		DROP COLUMN "name";

		ALTER TABLE "___ORGANIZATION_ID___".user_filter
		DROP COLUMN "description"; 

		ALTER TABLE "___ORGANIZATION_ID___".user_filter
		DROP COLUMN combinator;

		ALTER TABLE "___ORGANIZATION_ID___".user_filter
		DROP COLUMN "allIncluded";

		ALTER TABLE "___ORGANIZATION_ID___".user_filter
		DROP COLUMN "usageType";

		ALTER TABLE "___ORGANIZATION_ID___".user_filter
		DROP COLUMN "userCount";
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const createFilterResultTable = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
 		CREATE TABLE "___ORGANIZATION_ID___"."filter_result"
 		(
 			id uuid PRIMARY KEY,
 			"filterId" uuid,
 			"userId" uuid,
 			CONSTRAINT filter_result_user_filter_fk
 			FOREIGN KEY ("filterId") REFERENCES "___ORGANIZATION_ID___"."user_filter" (id) ON DELETE CASCADE,
 			CONSTRAINT filter_result_user_id_fk
 			FOREIGN KEY ("userId") REFERENCES "___ORGANIZATION_ID___"."users" (id) ON DELETE CASCADE,
 			CONSTRAINT unique_filter_id_user_id UNIQUE ("filterId", "userId")
 		);
 			ALTER TABLE "___ORGANIZATION_ID___"."filter_result" OWNER TO ${dbsuperuser};
 			GRANT INSERT, SELECT, DELETE, UPDATE ON TABLE "___ORGANIZATION_ID___"."filter_result" TO ${dbuser};
 			GRANT ALL ON TABLE "___ORGANIZATION_ID___"."filter_result" TO ${dbsuperuser};
 		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = ` DROP TABLE "___ORGANIZATION_ID___".filter_result; `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterDeleteSettingsTriggerFunction = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
        CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings()
        RETURNS TRIGGER AS $$
        DECLARE 
            filter RECORD;
        BEGIN
            FOR filter IN SELECT DISTINCT("filterId") as id FROM old_table as ot
                          INNER JOIN "___ORGANIZATION_ID___"."user_filter" as uf
                            ON ot."filterId" = uf.id
                          WHERE uf."usageType" = 3
            LOOP
                WITH filter_result AS (
                    SELECT "userId"
                    FROM "___ORGANIZATION_ID___".user_filter_result(filter.id)
                ), deleted AS (
                    DELETE FROM "___ORGANIZATION_ID___".filter_result
                    WHERE  "filterId" = filter.id
                            AND "userId" != ALL(SELECT "userId" FROM filter_result)
                )
                INSERT INTO "___ORGANIZATION_ID___".filter_result
                SELECT gen_random_uuid(), filter.id, "userId"
                FROM filter_result as ufr
                ON CONFLICT DO NOTHING;
            END LOOP;
        RETURN NULL;
        END;
        $$ LANGUAGE plpgsql;
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
        DROP FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings();
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userGroupUpdateTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
            CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_update_user_group()
            RETURNS TRIGGER AS $$
            BEGIN
                DELETE FROM "___ORGANIZATION_ID___".user_filter_group as ufg
                USING new_table as nt
                INNER JOIN old_table as ot
                    ON nt.id = ot.id
                WHERE ufg."groupId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";
            RETURN NULL;
            END;
            $$ LANGUAGE plpgsql;
            
            
            CREATE TRIGGER after_delete_user_group
            AFTER UPDATE ON "___ORGANIZATION_ID___"."userGroups"
            REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
            FOR EACH STATEMENT
            EXECUTE FUNCTION "___ORGANIZATION_ID___".after_update_user_group();
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
            DROP TRIGGER after_delete_user_group ON "___ORGANIZATION_ID___"."userGroups";
            DROP FUNCTION "___ORGANIZATION_ID___".after_update_user_group();
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterGroupDeleteTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `

        CREATE TRIGGER after_delete_user_filter_group
        AFTER DELETE ON "___ORGANIZATION_ID___"."user_filter_group"
        REFERENCING OLD TABLE AS old_table 
        FOR EACH STATEMENT
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings();

        ALTER TABLE "___ORGANIZATION_ID___"."user_filter_group" DISABLE TRIGGER after_delete_user_filter_group;
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
            DROP TRIGGER after_delete_user_filter_group ON "___ORGANIZATION_ID___"."user_filter_group";
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userRoleUpdateTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
        CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_update_role()
        RETURNS TRIGGER AS $$
        BEGIN
            DELETE FROM "___ORGANIZATION_ID___".user_filter_organization_role as ufor
            USING new_table as nt
            INNER JOIN old_table as ot
                ON nt.id = ot.id
            WHERE ufor."roleId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";

			DELETE FROM "___ORGANIZATION_ID___".user_filter_unit_role as ufur
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			WHERE ufur."roleId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";
        RETURN NULL;
        END;
        $$ LANGUAGE plpgsql;


        CREATE TRIGGER after_update_role
        AFTER UPDATE ON "___ORGANIZATION_ID___"."roles"
        REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
        FOR EACH STATEMENT
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_update_role();
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
        DROP TRIGGER after_update_role ON "___ORGANIZATION_ID___"."roles";
        DROP FUNCTION "___ORGANIZATION_ID___".after_update_role();
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterOrganizationRoleTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
        CREATE TRIGGER after_delete_user_filter_organization_role
        AFTER DELETE ON "___ORGANIZATION_ID___"."user_filter_organization_role"
        REFERENCING OLD TABLE AS old_table 
        FOR EACH STATEMENT
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings();

        ALTER TABLE "___ORGANIZATION_ID___"."user_filter_organization_role" DISABLE TRIGGER after_delete_user_filter_organization_role;
      `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
        DROP TRIGGER after_delete_user_filter_organization_role ON "___ORGANIZATION_ID___"."user_filter_organization_role";
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterOrganizationUnitRoleTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
        CREATE TRIGGER after_delete_user_filter_unit_role
        AFTER DELETE ON "___ORGANIZATION_ID___"."user_filter_unit_role"
        REFERENCING OLD TABLE AS old_table 
        FOR EACH STATEMENT
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings();

        ALTER TABLE "___ORGANIZATION_ID___"."user_filter_unit_role" DISABLE TRIGGER after_delete_user_filter_unit_role;		
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
        DROP TRIGGER after_delete_user_filter_unit_role ON "___ORGANIZATION_ID___"."user_filter_unit_role";
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const organizationUnitDeleteTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
        CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_del_or_up_organization_unit()
        RETURNS TRIGGER AS $$
		DECLARE 
			updated_filter_ids uuid[];
			filter_id uuid;
		BEGIN
		IF (TG_OP = 'UPDATE') THEN
			DELETE FROM "___ORGANIZATION_ID___".user_filter_unit as ufu
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			WHERE ufu."unitId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";

			DELETE FROM "___ORGANIZATION_ID___".user_filter_unit_role as ufur
			USING new_table as nt
			INNER JOIN old_table as ot
				ON nt.id = ot.id
			WHERE ufur."unitId" = nt.id AND nt."deletedAt" IS NOT NULL AND nt."deletedAt" IS DISTINCT FROM ot."deletedAt";

			SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
			FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
			INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
				ON uf.id = ufu."filterId"
			INNER JOIN (
						SELECT 	ot."id" as "unitId", 
								ot."ancestorIds" as old_ancestors,
								nt."ancestorIds" as new_ancestors
						FROM old_table as ot
						INNER JOIN new_table as nt
							ON ot.id = nt.id
						) subq
				ON ufu."unitId" = subq."unitId" OR (ufu.hierarchically IS TRUE AND (old_ancestors ILIKE '%' || ufu."unitId" ||'%' OR new_ancestors ILIKE '%' || ufu."unitId" ||'%'))
			WHERE uf."usageType" = 3;
		ELSIF (TG_OP = 'DELETE') THEN
			SELECT array_agg(DISTINCT(ufu."filterId")) into updated_filter_ids
			FROM "___ORGANIZATION_ID___"."user_filter_unit" ufu
			INNER JOIN "___ORGANIZATION_ID___"."user_filter" uf
				ON uf.id = ufu."filterId"
			INNER JOIN (
						SELECT 	ot."id" as "unitId", 
								ot."ancestorIds"
						FROM old_table as ot
						WHERE ot."deletedAt" IS NULL
						) subq
				ON ufu."unitId" <> subq."unitId" AND ufu.hierarchically IS TRUE AND subq."ancestorIds" ILIKE '%' || ufu."unitId" ||'%'
			WHERE uf."usageType" = 3;
		END IF;
		IF (updated_filter_ids IS NOT NULL ) THEN 
			FOREACH filter_id IN ARRAY updated_filter_ids LOOP
				WITH filter_result AS (
					SELECT "userId"
					FROM "___ORGANIZATION_ID___".user_filter_result(filter_id)
				), deleted AS (
					DELETE FROM "___ORGANIZATION_ID___".filter_result
					WHERE  "filterId" = filter_id
							AND "userId" != ALL(SELECT "userId" FROM filter_result)
				)
				INSERT INTO "___ORGANIZATION_ID___".filter_result
				SELECT gen_random_uuid(), filter_id, "userId"
				FROM filter_result as ufr
				ON CONFLICT DO NOTHING;
			END LOOP;
		END IF;
		RETURN NULL;
		END;
        $$ LANGUAGE plpgsql;


		CREATE TRIGGER after_update_organization_unit
		AFTER UPDATE ON "___ORGANIZATION_ID___"."organizationUnits"
		REFERENCING OLD TABLE AS old_table NEW TABLE AS new_table
		FOR EACH STATEMENT
		EXECUTE FUNCTION "___ORGANIZATION_ID___".after_del_or_up_organization_unit();

		CREATE TRIGGER after_delete_organization_unit
		AFTER DELETE ON "___ORGANIZATION_ID___"."organizationUnits"
		REFERENCING OLD TABLE AS old_table
		FOR EACH STATEMENT
		EXECUTE FUNCTION "___ORGANIZATION_ID___".after_del_or_up_organization_unit();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
			DROP TRIGGER after_update_organization_unit ON "___ORGANIZATION_ID___"."organizationUnits";
			DROP TRIGGER after_delete_organization_unit ON "___ORGANIZATION_ID___"."organizationUnits";
			DROP FUNCTION "___ORGANIZATION_ID___".after_del_or_up_organization_unit();
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterOrganizationUnitDeleteTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE TRIGGER after_delete_user_filter_unit
		AFTER DELETE ON "___ORGANIZATION_ID___"."user_filter_unit"
		REFERENCING OLD TABLE AS old_table 
		FOR EACH STATEMENT
		EXECUTE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings();

		ALTER TABLE "___ORGANIZATION_ID___"."user_filter_unit" DISABLE TRIGGER after_delete_user_filter_unit;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `
		DROP TRIGGER after_delete_user_filter_unit ON "___ORGANIZATION_ID___"."user_filter_unit";
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const userFilterUserDeleteTrigger = {
    up: async (client, dbuser, dbsuperuser) => {
        const q = `
		CREATE TRIGGER after_delete_user_filter_user
		AFTER DELETE ON "___ORGANIZATION_ID___"."user_filter_user"
		REFERENCING OLD TABLE AS old_table 
		FOR EACH STATEMENT
		EXECUTE FUNCTION "___ORGANIZATION_ID___".after_delete_user_filter_settings();

		ALTER TABLE "___ORGANIZATION_ID___"."user_filter_user" DISABLE TRIGGER after_delete_user_filter_user;
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client, dbuser, dbsuperuser) => {
        const q = `DROP TRIGGER after_delete_user_filter_user ON "___ORGANIZATION_ID___"."user_filter_user"; `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
