"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_utils_1 = require("../../../dal.utils");
const luxon_1 = require("luxon");
async function up(client) {
    const createPartitionedTableQuery = `
        DROP TABLE IF EXISTS "___ORGANIZATION_ID___"."access_logs_arvento_new" CASCADE;

        CREATE TABLE "___ORGANIZATION_ID___"."access_logs_arvento_new" (
            id uuid DEFAULT gen_random_uuid() NOT NULL,
            "accessLogId" uuid NOT NULL,
            "actionUtc" timestamptz NOT NULL,
            "updatedAt" timestamptz DEFAULT now() NOT NULL,
            "type" int4 NOT NULL,
            "isSynchronized" bool DEFAULT false NOT NULL,
            "accessControlPointId" uuid NOT NULL,
            "data" jsonb NULL,
            CONSTRAINT access_logs_arvento_new_pkey PRIMARY KEY (id, "actionUtc")
        ) PARTITION BY RANGE ("actionUtc");

        CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___"."access_logs_arvento_default"
        PARTITION OF "___ORGANIZATION_ID___"."access_logs_arvento_new"
        DEFAULT;
    `;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, createPartitionedTableQuery, "___ORGANIZATION_ID___");
    const organizationIds = await client.query(`
        SELECT id FROM public."organizationList"
    `);
    for (const orgid of organizationIds.rows.map((m) => m.id)) {
        const res = await client.query(`
            SELECT DATE_TRUNC('month', "actionUtc") AS "missingPartitionMonth", COUNT(*) AS c
            FROM "${orgid}"."access_logs_arvento"
            GROUP BY "missingPartitionMonth"
            ORDER BY "missingPartitionMonth" ASC
        `);
        for (const row of res.rows) {
            const month = luxon_1.DateTime.fromJSDate(row.missingPartitionMonth);
            const partitionStart = month.startOf("month").toISODate();
            const partitionEnd = month.plus({ months: 1 }).startOf("month").toISODate();
            const partitionName = `access_logs_arvento_p${month.toFormat("yyyyMM")}`;
            await client.query(`
                CREATE TABLE IF NOT EXISTS "${orgid}"."${partitionName}"
                PARTITION OF "${orgid}"."access_logs_arvento_new"
                FOR VALUES FROM ('${partitionStart}') TO ('${partitionEnd}')
            `);
        }
        await client.query(`
            INSERT INTO "${orgid}"."access_logs_arvento_new" (
                id, "accessLogId", "actionUtc", "updatedAt", "type", "isSynchronized", "accessControlPointId", "data"
            )
            SELECT id, "accessLogId", "actionUtc", "updatedAt", "type", "isSynchronized", "accessControlPointId", "data"
            FROM "${orgid}"."access_logs_arvento"
        `);
    }
    const replaceNewTableWithOldOneQuery = `
        DROP TABLE "___ORGANIZATION_ID___"."access_logs_arvento" CASCADE;
        ALTER TABLE "___ORGANIZATION_ID___"."access_logs_arvento_new" RENAME TO access_logs_arvento;
        ALTER TABLE "___ORGANIZATION_ID___"."access_logs_arvento" RENAME CONSTRAINT access_logs_arvento_new_pkey TO access_logs_arvento_pkey;
    `;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, replaceNewTableWithOldOneQuery, "___ORGANIZATION_ID___");
}
exports.up = up;
async function down(client) {
    const rollbackQuery = `
        ALTER TABLE "___ORGANIZATION_ID___"."access_logs_arvento" DROP CONSTRAINT IF EXISTS access_logs_arvento_pkey;

        ALTER TABLE "___ORGANIZATION_ID___"."access_logs_arvento" RENAME TO access_logs_arvento_backup;

        CREATE TABLE "___ORGANIZATION_ID___"."access_logs_arvento" (
            id uuid DEFAULT gen_random_uuid() NOT NULL,
            "accessLogId" uuid NOT NULL,
            "actionUtc" timestamptz NOT NULL,
            "updatedAt" timestamptz DEFAULT now() NOT NULL,
            "type" int4 NOT NULL,
            "isSynchronized" bool DEFAULT false NOT NULL,
            "accessControlPointId" uuid NOT NULL,
            "data" jsonb NULL,
            CONSTRAINT access_logs_arvento_pkey PRIMARY KEY (id)
        );

        INSERT INTO "___ORGANIZATION_ID___"."access_logs_arvento"
        SELECT id, "accessLogId", "actionUtc", "updatedAt", "type", "isSynchronized", "accessControlPointId", "data"
        FROM "___ORGANIZATION_ID___"."access_logs_arvento_backup";

        DROP TABLE "___ORGANIZATION_ID___"."access_logs_arvento_backup" CASCADE;
    `;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, rollbackQuery, "___ORGANIZATION_ID___");
}
exports.down = down;
