"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_utils_1 = require("../../../dal.utils");
async function up(client) {
    let query = `
    CREATE OR REPLACE FUNCTION public.clone_schema(source_schema text, dest_schema text, include_recs boolean) RETURNS void
    LANGUAGE plpgsql
    AS $$
	--  This function will clone all sequences, tables, data, views & functions from any existing schema to a new one
	-- SAMPLE CALL:
	-- SELECT clone_schema('public', 'new_schema', TRUE);
	
	DECLARE
	src_oid          oid;
	tbl_oid          oid;
	func_oid         oid;
	object           text;
	v_rec            record;
	buffer           text;
	srctbl           text;
	default_         text;
	column_          text;
	qry              text;
	dest_qry         text;
	is_partition     boolean;
	partition_col_name     text;
	partition_range_cond     text;
	parent_rel_table text;
	fk_query text;
	trigger_name_ text;
	trigger_timing_ text;
	trigger_events_ text;
	trigger_orientation_ text;
	trigger_action_ text;
	
	BEGIN
	
	-- Check that source_schema exists
	SELECT oid INTO src_oid
	  FROM pg_namespace
	 WHERE nspname = quote_ident(source_schema);
	IF NOT FOUND
	  THEN 
	  RAISE NOTICE 'source schema % does not exist!', source_schema;
	  RETURN ;
	END IF;
	/*
	-- Check that dest_schema does not yet exist
	PERFORM nspname 
	  FROM pg_namespace
	 WHERE nspname = quote_ident(dest_schema);
	IF NOT FOUND
	  THEN 
	  EXECUTE 'CREATE SCHEMA ' || quote_ident(dest_schema) ;
	  -- RAISE NOTICE 'dest schema % already exists!', dest_schema;
	  RETURN ;
	END IF;
	*/
	
	EXECUTE 'CREATE SCHEMA IF NOT EXISTS ' || quote_ident(dest_schema) ;
	
	-- Create sequences
	-- TODO: Find a way to make this sequence's owner is the correct table.
	FOR object IN
	  SELECT sequence_name::text 
		FROM information_schema.sequences
	   WHERE sequence_schema = quote_ident(source_schema)
	LOOP
	  EXECUTE 'CREATE SEQUENCE ' || quote_ident(dest_schema) || '.' || quote_ident(object);
	  srctbl := quote_ident(source_schema) || '.' || quote_ident(object);
	END LOOP;
	
	-- Create functions 
	FOR func_oid IN
	  SELECT oid
		FROM pg_proc 
	   WHERE pronamespace = src_oid
	
	LOOP      
	  SELECT pg_get_functiondef(func_oid) INTO qry;
	  SELECT regexp_replace (
				regexp_replace (qry, 
					''''|| source_schema || '''', 
					'''' ||dest_schema ||'''', 
					'g'), 
				'\\y'||quote_ident(source_schema)||'\\y', 
				quote_ident(dest_schema), 
				'g') 
			INTO dest_qry;
	  EXECUTE dest_qry;
	
	END LOOP;
	
	-- Create tables 
	FOR object IN
	  SELECT TABLE_NAME::text as tx 
		FROM information_schema.tables 
	   WHERE table_schema = quote_ident(source_schema)
		 AND table_type = 'BASE TABLE'
		ORDER BY tx  asc
	LOOP
	
	   SELECT relispartition INTO is_partition
		FROM pg_catalog.pg_class
		WHERE relname = object;
		
		IF NOT is_partition THEN
		
			  select 
				col.column_name into partition_col_name
			from   
				(select
					 partrelid,
					 partnatts,
					 case partstrat 
						  when 'l' then 'list' 
						  when 'r' then 'range' end as partition_strategy,
					 unnest(partattrs) column_index
				 from
					 pg_partitioned_table) pt 
			join   
				pg_class par 
			on     
				par.oid = pt.partrelid
			join
				information_schema.columns col
			on  
		quote_ident(col.table_schema) = par.relnamespace::regnamespace::text
				and col.table_name = par.relname
				and ordinal_position = pt.column_index
			WHERE par.relnamespace::regnamespace::text = quote_ident(source_schema) AND par.relname = object;
			
		
		  buffer := quote_ident(dest_schema) || '.' || quote_ident(object);
		  EXECUTE 'CREATE TABLE ' || buffer || ' (LIKE ' || quote_ident(source_schema) || '.' || quote_ident(object) 
			  || ' INCLUDING ALL)' || (CASE WHEN partition_col_name IS NOT NULL THEN ' PARTITION BY RANGE ('|| quote_ident(partition_col_name)||')' ELSE '' END);
	
		  FOR column_, default_ IN
			SELECT column_name::text, 
				   REPLACE(column_default::text, source_schema, quote_ident(dest_schema)) 
			  FROM information_schema.COLUMNS 
			 WHERE table_schema =  quote_ident(dest_schema)
			   AND TABLE_NAME = object 
			   AND column_default LIKE 'nextval(%' || quote_ident(source_schema) || '%::regclass)'
		  LOOP
			  EXECUTE 'ALTER TABLE ' || buffer || ' ALTER COLUMN ' || column_ || ' SET DEFAULT ' || default_;
		  END LOOP;
			
		  IF include_recs 
			THEN 
			-- Insert records from source table
			  EXECUTE 'INSERT INTO ' || buffer || ' SELECT * FROM ' || quote_ident(source_schema) || '.' || quote_ident(object) || ';';
		  END IF;

			IF (substr((SELECT current_setting('server_version_num')), 1, 2)::integer >= 15) THEN
				FOR trigger_name_ IN
				SELECT trigger_name::text FROM information_schema.TRIGGERS 
				WHERE event_object_schema=source_schema and event_object_table=object 
				GROUP BY trigger_name
				LOOP
					buffer := quote_ident(dest_schema) || '.' || quote_ident(object);
					EXECUTE (SELECT 
					REPLACE (
						REPLACE (
						REPLACE(
							REPLACE (
							REPLACE (
								REPLACE (pg_get_triggerdef(oid), 'public.', ''), 
							'notification.', ''), 
							'organization.', ''), 
						'EXECUTE PROCEDURE', 'EXECUTE FUNCTION'),
						'EXECUTE FUNCTION ', CASE WHEN POSITION ('after_insert_default' IN pg_get_triggerdef(oid)) > 0
							THEN 'EXECUTE FUNCTION public.' ELSE 'EXECUTE FUNCTION '|| quote_ident(dest_schema) ||'.' END), 
					' ON ', ' ON '|| quote_ident(dest_schema) ||'.') 
					FROM pg_trigger
					WHERE tgname = trigger_name_
					AND ( CASE WHEN source_schema = 'public' THEN 
						tgrelid::regclass::text = quote_ident(object) ELSE 
						tgrelid::regclass::text = quote_ident(source_schema) || '.' ||quote_ident(object) END)
					);
				END LOOP;
			END IF;
		  
		ELSE
			SELECT pg_get_expr(p.relpartbound, p.oid) into partition_range_cond from pg_class as p
				JOIN pg_namespace as ns ON (ns.oid = p.relnamespace)
				where relname = object AND ns.nspname = source_schema;
			--select pg_get_expr(relpartbound, oid) into partition_range_cond from pg_class where relispartition and relname = object;
			RAISE NOTICE 'partition_range_cond %', partition_range_cond;
			SELECT REPLACE (REPLACE(partrelid::regclass::text, 'notification.', ''), 'organization.', '')
			INTO parent_rel_table from pg_partitioned_table WHERE partdefid::regclass::text  = quote_ident(source_schema) || '.' || quote_ident(object);
			--select partrelid::regclass::text, partdefid::regclass::text, * from pg_partitioned_table  --where  (partrelid::regclass::text) like quote_ident(source_schema) || '.%'
			IF parent_rel_table IS NULL THEN
				SELECT p.relname into parent_rel_table FROM pg_inherits
					JOIN pg_class as p ON (inhparent=p.oid)
					JOIN pg_namespace as ns ON (ns.oid = p.relnamespace)
					WHERE ((inhrelid::regclass::text ilike '%.' || object) OR 
					(inhrelid::regclass::text = object)) AND ns.nspname = source_schema;
			END IF;
			RAISE NOTICE 'parent_rel_table % of %', parent_rel_table, object;
			EXECUTE 'CREATE TABLE '|| quote_ident(dest_schema) || '.' || quote_ident(object) ||' PARTITION OF '|| quote_ident(dest_schema) 
			|| '.' ||quote_ident(parent_rel_table)||' ' ||partition_range_cond || ';';
		END IF;
		IF (substr((SELECT current_setting('server_version_num')), 1, 2)::integer < 15) THEN
			FOR trigger_name_ IN
			SELECT trigger_name::text FROM information_schema.TRIGGERS 
			WHERE event_object_schema=source_schema and event_object_table=object 
			GROUP BY trigger_name
			LOOP
				buffer := quote_ident(dest_schema) || '.' || quote_ident(object);
				EXECUTE (SELECT 
				REPLACE (
					REPLACE (
					REPLACE(
						REPLACE (
						REPLACE (
							REPLACE (pg_get_triggerdef(oid), 'public.', ''), 
						'notification.', ''), 
						'organization.', ''), 
					'EXECUTE PROCEDURE', 'EXECUTE FUNCTION'),
					'EXECUTE FUNCTION ', CASE WHEN POSITION ('after_insert_default' IN pg_get_triggerdef(oid)) > 0
						THEN 'EXECUTE FUNCTION public.' ELSE 'EXECUTE FUNCTION '|| quote_ident(dest_schema) ||'.' END), 
				' ON ', ' ON '|| quote_ident(dest_schema) ||'.') 
				FROM pg_trigger
				WHERE tgname = trigger_name_
				AND ( CASE WHEN source_schema = 'public' THEN 
					tgrelid::regclass::text = quote_ident(object) ELSE 
					tgrelid::regclass::text = quote_ident(source_schema) || '.' ||quote_ident(object) END)
				);
			END LOOP;
		END IF;
	END LOOP;
	
	--  add FK constraint
	FOR qry IN
		SELECT 'ALTER TABLE ' || quote_ident(dest_schema) || '.' || quote_ident(rn.relname) 
		|| ' ADD CONSTRAINT ' || quote_ident(ct.conname) || ' ' || 
		  REPLACE( REPLACE (REPLACE (pg_get_constraintdef(ct.oid),'notification.', ''), 'organization.', ''),
		'REFERENCES ', CASE WHEN  POSITION( '.' IN pg_get_constraintdef(ct.oid))> 0 THEN 'REFERENCES '||quote_ident(dest_schema) ||'.'  ELSE 'REFERENCES public.' END) || ';'
		FROM pg_constraint ct
		JOIN pg_class rn ON rn.oid = ct.conrelid
		JOIN pg_namespace ns ON ns.oid = connamespace
	   WHERE connamespace = ns.oid
		 --AND rn.relkind = 'r'
		 AND rn.relispartition = false
		 AND ct.contype = 'f'
		 AND ct.conparentid = 0
		 AND ns.nspname = quote_ident(source_schema)
	  LOOP
		  RAISE NOTICE 'query for foreign key %',qry;
		EXECUTE qry;
	
	  END LOOP;
	
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."organizationList" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."userList" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."userOrganizationMapping" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."things" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."scheduledJobs" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."scheduledJobHistoryLogs" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."migrations" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."migrations_lock" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."emailTemplates" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."oAuthRevokedTokens" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."organizationAuthenticationMethods" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."userAuthenticationMethods" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."reportTemplates" CASCADE;';
	
	EXECUTE 'DROP FUNCTION IF EXISTS ' || quote_ident(dest_schema) || '."clone_schema" CASCADE;';
	EXECUTE 'DROP FUNCTION IF EXISTS ' || quote_ident(dest_schema) || '."disable_trigger" CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."oAuthRevokedTokens" 
		DROP CONSTRAINT IF EXISTS "oAuthRevokedTokens_userId_fkey",
		DROP CONSTRAINT IF EXISTS "oAuthRevokedTokens_userListId_fkey",
		ADD CONSTRAINT "oAuthRevokedTokens_userListId_fkey" 
		FOREIGN KEY ("userId") REFERENCES public."userList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."emailTemplates" 
		DROP CONSTRAINT IF EXISTS "emailTemplates_organizationId_fkey",
		DROP CONSTRAINT IF EXISTS "emailTemplates_organizationListId_fkey",
		ADD CONSTRAINT "emailTemplates_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."scheduledJobs" 
		DROP CONSTRAINT IF EXISTS "scheduledJobs_organizationId_fkey",
		DROP CONSTRAINT IF EXISTS "scheduledJobs_organizationListId_fkey",
		ADD CONSTRAINT "scheduledJobs_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."organizationAuthenticationMethods" 
		DROP CONSTRAINT IF EXISTS "organizationauthenticationmethods_organizationid_foreign",
		DROP CONSTRAINT IF EXISTS "organizationAuthenticationMethods_organizationListId_fkey",
		ADD CONSTRAINT "organizationAuthenticationMethods_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."userAuthenticationMethods" 
		DROP CONSTRAINT IF EXISTS "userauthenticationmethods_organizationid_foreign",
		DROP CONSTRAINT IF EXISTS "userAuthenticationMethods_userListId_fkey",
		ADD CONSTRAINT "userAuthenticationMethods_userListId_fkey" 
		FOREIGN KEY ("userId") REFERENCES public."userList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."userAuthenticationMethods" 
		DROP CONSTRAINT IF EXISTS "userauthenticationmethods_authmethodid_foreign",
		DROP CONSTRAINT IF EXISTS "userAuthenticationMethods_organizationAuthenticationMethodsId_fkey",
		ADD CONSTRAINT "userAuthenticationMethods_organizationAuthenticationMethodsId_fkey" 
		FOREIGN KEY ("authenticationMethodId") REFERENCES public."organizationAuthenticationMethods"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."reportTemplates" 
		DROP CONSTRAINT IF EXISTS "reportTemplates_organizationListId_fkey",
		ADD CONSTRAINT "reportTemplates_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."reportTemplates" 
		DROP CONSTRAINT IF EXISTS "reportTemplates_userListId_fkey",
		ADD CONSTRAINT "reportTemplates_userListId_fkey" 
		FOREIGN KEY ("userId") REFERENCES public."userList"(id) ON DELETE CASCADE;';
	
	RETURN; 
	
	END;
	$$;
    `;
    await client.query(query);
    const assignAutoShiftFixQuery = `
-- DROP FUNCTION "___ORGANIZATION_ID___".assign_auto_shift(uuid, uuid, uuid, date, date);

CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".assign_auto_shift(param_user_id uuid, param_rule_id uuid, log_id uuid, start_date date, end_date date)
 RETURNS void
 LANGUAGE plpgsql
AS $function$
	DECLARE al record;
			at_now timestamp with time zone;
			exclusive_lock record;
			users_to_be_assigned uuid[] := ARRAY[]::uuid[];
			users_to_be_updated uuid[] := ARRAY[]::uuid[];
	BEGIN

		IF ("param_rule_id" IS NOT NULL) THEN
			SELECT ARRAY_AGG(DISTINCT(uo."userId")) into users_to_be_assigned
			FROM "___ORGANIZATION_ID___"."auto_shift_rule_set" AS asrs
			INNER JOIN "___ORGANIZATION_ID___"."auto_shift_rule_set_group" AS asrsg
				ON asrs.id = asrsg."ruleSetId"
			INNER JOIN "___ORGANIZATION_ID___"."userGroupUserOrganizations" AS uguo
				ON uguo."userGroupId" = asrsg."groupId"
			INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" AS uo
				ON uguo."userOrganizationId" = uo.id AND uo."isDisabled" IS FALSE
			WHERE asrs.id = "param_rule_id";
		END IF;

		FOR al IN
			SELECT  sq2.wp_id, logs."id" as "logId", 
					arrsg."groupId", arrsr."regionId", logs."userId",
					logs."actionUtc" = MIN(logs."actionUtc") OVER ( PARTITION BY logs."userId", logs."actionUtc"::date, sq2.id) as is_first,
					sq2."useOnlyFirstAccess",
					(logs."actionUtc"::date + membership_from_date)::timestamp with time zone as mem_start,
					(logs."actionUtc"::date + membership_to_date)::timestamp with time zone as mem_end
			FROM
			(
				SELECT
					id,
					sq.wp_id,
					sq."useOnlyFirstAccess",
					sq."supportMobileCheckin",
					sq."supportManualLog",
					sq."startDateTime",
					sq."endDateTime",
					'00:00:00'::interval
						+ (INTERVAL '1 days' * COALESCE((_from->'days')::integer, 0))
						+ (INTERVAL '1 hours' * COALESCE((_from->'hours')::integer, 0))
						+ (INTERVAL '1 minutes' * COALESCE((_from->'minutes')::integer, 0)) as from_time,
					'00:00:00'::interval
						+ (INTERVAL '1 days' * COALESCE((_to->'days')::integer, 0))
						+ (INTERVAL '1 hours' * COALESCE((_to->'hours')::integer, 0))
						+ (INTERVAL '1 minutes' * COALESCE((_to->'minutes')::integer, 0)) as to_time,
					'00:00:00'::interval
						+ (INTERVAL '1 days' * COALESCE((_membership_from->'days')::integer, 0))
						+ (INTERVAL '1 hours' * COALESCE((_membership_from->'hours')::integer, 0))
						+ (INTERVAL '1 minutes' * COALESCE((_membership_from->'minutes')::integer, 0)) as membership_from_date,
					'00:00:00'::interval
						+ (INTERVAL '1 days' * COALESCE((_membership_to->'days')::integer, 0))
						+ (INTERVAL '1 hours' * COALESCE((_membership_to->'hours')::integer, 0))
						+ (INTERVAL '1 minutes' * COALESCE((_membership_to->'minutes')::integer, 0)) as membership_to_date
				FROM
				(
					SELECT
						arrs.id,
						"useOnlyFirstAccess",
						"supportMobileCheckin",
						"supportManualLog",
						"startDateTime",
						"endDateTime",
						(rrr->>'workPlanId')::uuid as wp_id,
						rrr->'checkRange'->'from' as _from,
						rrr->'checkRange'->'to' as _to,
						rrr->'membershipRange'->'from' as _membership_from,
						rrr->'membershipRange'->'to' as _membership_to
					FROM "___ORGANIZATION_ID___"."auto_shift_rule_set" arrs, unnest(rules) as rrr
					WHERE ((id = "param_rule_id" OR "param_rule_id" IS NULL) AND (arrs."deletedAt" IS NULL))
				) sq
			) sq2
			INNER JOIN "___ORGANIZATION_ID___"."auto_shift_rule_set_group" arrsg
				ON arrsg."ruleSetId" = sq2.id
			LEFT JOIN "___ORGANIZATION_ID___"."auto_shift_rule_set_region" arrsr
				ON arrsr."ruleSetId" = sq2.id
			INNER JOIN (
				SELECT
					alo.id,
					(alo.log->>'o')::uuid as "userId",
					alo."actionUtc",
					alo.log->>'a' = '06ba5170-d827-4ced-b356-f2e99afd3971' as "isMobileCheckin",
					alo.log->>'rm' as "logMethod",
					(alo.log->>'ir')::boolean as "isRemote",
					alo.log->>'cid' IS NOT NULL as "isDevice",
					(alo.log->>'iq')::boolean as "isQrTriggered",
					array_agg((rg->>'i')::uuid) FILTER (WHERE rg->>'i' IS NOT NULL) as regions
				FROM "___ORGANIZATION_ID___".access_logs AS alo
				LEFT JOIN jsonb_array_elements(COALESCE(alo.log->>'rg', '[]')::jsonb) rg
					ON 1=1
				WHERE   alo.log->>'d' = '1'
						AND "start_date" < alo."actionUtc" 
						AND (alo."actionUtc" < "end_date" OR "end_date" IS NULL) 
						AND ( (alo.log->>'o')::uuid = "param_user_id" OR (alo.log->>'o')::uuid = ANY(users_to_be_assigned))
				GROUP BY alo.id, alo.log->>'o', alo."actionUtc"
			) logs
				ON
					sq2.from_time <= (logs."actionUtc")::time AND (logs."actionUtc")::time < sq2.to_time
					AND ("param_user_id" IS NULL OR (logs."actionUtc" >= sq2."startDateTime" AND (logs."actionUtc" <= sq2."endDateTime" OR sq2."endDateTime" IS NULL)))
					AND (
						(logs."logMethod" = '5' AND sq2."supportManualLog")
						OR (logs."logMethod" = '4' AND ( sq2."supportMobileCheckin" OR NOT logs."isMobileCheckin"))
						OR (logs."isRemote" IS NULL AND (NOT logs."isDevice" OR arrsr."regionId" = ANY(logs.regions)))
						OR (logs."isRemote" AND logs."isQrTriggered" AND arrsr."regionId" = ANY(logs.regions))
					)
			INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" AS uo
				ON uo."userId" = logs."userId"
			INNER JOIN "___ORGANIZATION_ID___"."userGroupUserOrganizations" AS uguo
				ON uguo."userOrganizationId" = uo.id AND uguo."userGroupId" = arrsg."groupId"
			WHERE   (logs."actionUtc" <= uguo."deletedAt" OR uguo."deletedAt" IS NULL) 
					AND logs."actionUtc" >= uguo."createdAt" 
					AND uo."deletedAt" IS NULL
			ORDER BY logs."userId", logs."actionUtc" ASC
		LOOP 
			IF ( (log_id = al."logId" OR log_id IS NULL) AND (NOT al."useOnlyFirstAccess" OR al.is_first) )  THEN
					at_now :=clock_timestamp();
					SELECT * into exclusive_lock
					FROM "___ORGANIZATION_ID___"."userWorkPlans" uwp
					WHERE al."userId" = uwp."userId" FOR UPDATE;

					SELECT CASE 
								WHEN al."userId" = ANY(users_to_be_updated) THEN users_to_be_updated
								ELSE ARRAY_APPEND(users_to_be_updated, al."userId") 
							END users into users_to_be_updated;
					
					INSERT INTO "___ORGANIZATION_ID___"."userWorkPlans"
					(id, "userId", "workPlanId", "startDateTime", "endDateTime", "createdAt")
					SELECT gen_random_uuid(), al."userId", subbed_tbl."workPlanId", LOWER(subbed_tbl.subbed), UPPER(subbed_tbl.subbed), at_now
					FROM (
						SELECT uwp."workPlanId", tstzrange(uwp."startDateTime", uwp."endDateTime") - tstzrange(al.mem_start, null) as subbed
						FROM "___ORGANIZATION_ID___"."userWorkPlans" uwp
						WHERE uwp."userId" = al."userId" AND tstzrange(uwp."startDateTime", uwp."endDateTime") && tstzrange(al.mem_start, al.mem_end)
						UNION ALL
						SELECT uwp."workPlanId", tstzrange(uwp."startDateTime", uwp."endDateTime") - tstzrange(null, al.mem_end) as subbed
						FROM "___ORGANIZATION_ID___"."userWorkPlans" uwp
						WHERE uwp."userId" = al."userId" AND tstzrange(uwp."startDateTime", uwp."endDateTime") && tstzrange(al.mem_start, al.mem_end)
					) subbed_tbl
					WHERE NOT isEmpty(subbed_tbl.subbed)
					UNION ALL
					SELECT gen_random_uuid(), al."userId", al.wp_id, al.mem_start, al.mem_end, at_now;

					DELETE FROM  "___ORGANIZATION_ID___"."userWorkPlans"
					WHERE "userId" = al."userId"
					AND tstzrange("startDateTime", "endDateTime") && tstzrange(al.mem_start, al.mem_end)
					AND "createdAt" <> at_now;
			END IF;
		END LOOP;
		INSERT INTO "___ORGANIZATION_ID___"."recalculateWorkQueue"
			(id, "userId", "organizationId", reason, "startDate")
			SELECT gen_random_uuid(), u, '___ORGANIZATION_ID___', 32, start_date
			FROM unnest(users_to_be_updated) u;
		UPDATE "___ORGANIZATION_ID___"."employeeLiveDay" eld
			SET "updateRequiredReason" = "updateRequiredReason" | 32, "updateRequiredRaisedAt" = now()
			WHERE eld."userId" = ANY(users_to_be_updated);
	END;
$function$
;
	`;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, assignAutoShiftFixQuery, "___ORGANIZATION_ID___");
}
exports.up = up;
async function down(client) {
    let query = `
    CREATE OR REPLACE FUNCTION public.clone_schema(source_schema text, dest_schema text, include_recs boolean) RETURNS void
    LANGUAGE plpgsql
    AS $$
	--  This function will clone all sequences, tables, data, views & functions from any existing schema to a new one
	-- SAMPLE CALL:
	-- SELECT clone_schema('public', 'new_schema', TRUE);
	
	DECLARE
	src_oid          oid;
	tbl_oid          oid;
	func_oid         oid;
	object           text;
	v_rec            record;
	buffer           text;
	srctbl           text;
	default_         text;
	column_          text;
	qry              text;
	dest_qry         text;
	is_partition     boolean;
	partition_col_name     text;
	partition_range_cond     text;
	parent_rel_table text;
	fk_query text;
	trigger_name_ text;
	trigger_timing_ text;
	trigger_events_ text;
	trigger_orientation_ text;
	trigger_action_ text;
	
	BEGIN
	
	-- Check that source_schema exists
	SELECT oid INTO src_oid
	  FROM pg_namespace
	 WHERE nspname = quote_ident(source_schema);
	IF NOT FOUND
	  THEN 
	  RAISE NOTICE 'source schema % does not exist!', source_schema;
	  RETURN ;
	END IF;
	/*
	-- Check that dest_schema does not yet exist
	PERFORM nspname 
	  FROM pg_namespace
	 WHERE nspname = quote_ident(dest_schema);
	IF NOT FOUND
	  THEN 
	  EXECUTE 'CREATE SCHEMA ' || quote_ident(dest_schema) ;
	  -- RAISE NOTICE 'dest schema % already exists!', dest_schema;
	  RETURN ;
	END IF;
	*/
	
	EXECUTE 'CREATE SCHEMA IF NOT EXISTS ' || quote_ident(dest_schema) ;
	
	-- Create sequences
	-- TODO: Find a way to make this sequence's owner is the correct table.
	FOR object IN
	  SELECT sequence_name::text 
		FROM information_schema.sequences
	   WHERE sequence_schema = quote_ident(source_schema)
	LOOP
	  EXECUTE 'CREATE SEQUENCE ' || quote_ident(dest_schema) || '.' || quote_ident(object);
	  srctbl := quote_ident(source_schema) || '.' || quote_ident(object);
	END LOOP;
	
	-- Create functions 
	FOR func_oid IN
	  SELECT oid
		FROM pg_proc 
	   WHERE pronamespace = src_oid
	
	LOOP      
	  SELECT pg_get_functiondef(func_oid) INTO qry;
	  SELECT replace(qry, quote_ident(source_schema) || '.', quote_ident(dest_schema) || '.') INTO dest_qry;
	  EXECUTE dest_qry;
	
	END LOOP;
	
	-- Create tables 
	FOR object IN
	  SELECT TABLE_NAME::text as tx 
		FROM information_schema.tables 
	   WHERE table_schema = quote_ident(source_schema)
		 AND table_type = 'BASE TABLE'
		ORDER BY tx  asc
	LOOP
	
	   SELECT relispartition INTO is_partition
		FROM pg_catalog.pg_class
		WHERE relname = object;
		
		IF NOT is_partition THEN
		
			  select 
				col.column_name into partition_col_name
			from   
				(select
					 partrelid,
					 partnatts,
					 case partstrat 
						  when 'l' then 'list' 
						  when 'r' then 'range' end as partition_strategy,
					 unnest(partattrs) column_index
				 from
					 pg_partitioned_table) pt 
			join   
				pg_class par 
			on     
				par.oid = pt.partrelid
			join
				information_schema.columns col
			on  
		quote_ident(col.table_schema) = par.relnamespace::regnamespace::text
				and col.table_name = par.relname
				and ordinal_position = pt.column_index
			WHERE par.relnamespace::regnamespace::text = quote_ident(source_schema) AND par.relname = object;
			
		
		  buffer := quote_ident(dest_schema) || '.' || quote_ident(object);
		  EXECUTE 'CREATE TABLE ' || buffer || ' (LIKE ' || quote_ident(source_schema) || '.' || quote_ident(object) 
			  || ' INCLUDING ALL)' || (CASE WHEN partition_col_name IS NOT NULL THEN ' PARTITION BY RANGE ('|| quote_ident(partition_col_name)||')' ELSE '' END);
	
		  FOR column_, default_ IN
			SELECT column_name::text, 
				   REPLACE(column_default::text, source_schema, quote_ident(dest_schema)) 
			  FROM information_schema.COLUMNS 
			 WHERE table_schema =  quote_ident(dest_schema)
			   AND TABLE_NAME = object 
			   AND column_default LIKE 'nextval(%' || quote_ident(source_schema) || '%::regclass)'
		  LOOP
			  EXECUTE 'ALTER TABLE ' || buffer || ' ALTER COLUMN ' || column_ || ' SET DEFAULT ' || default_;
		  END LOOP;
			
		  IF include_recs 
			THEN 
			-- Insert records from source table
			  EXECUTE 'INSERT INTO ' || buffer || ' SELECT * FROM ' || quote_ident(source_schema) || '.' || quote_ident(object) || ';';
		  END IF;

			IF (substr((SELECT current_setting('server_version_num')), 1, 2)::integer >= 15) THEN
				FOR trigger_name_ IN
				SELECT trigger_name::text FROM information_schema.TRIGGERS 
				WHERE event_object_schema=source_schema and event_object_table=object 
				GROUP BY trigger_name
				LOOP
					buffer := quote_ident(dest_schema) || '.' || quote_ident(object);
					EXECUTE (SELECT 
					REPLACE (
						REPLACE (
						REPLACE(
							REPLACE (
							REPLACE (
								REPLACE (pg_get_triggerdef(oid), 'public.', ''), 
							'notification.', ''), 
							'organization.', ''), 
						'EXECUTE PROCEDURE', 'EXECUTE FUNCTION'),
						'EXECUTE FUNCTION ', CASE WHEN POSITION ('after_insert_default' IN pg_get_triggerdef(oid)) > 0
							THEN 'EXECUTE FUNCTION public.' ELSE 'EXECUTE FUNCTION '|| quote_ident(dest_schema) ||'.' END), 
					' ON ', ' ON '|| quote_ident(dest_schema) ||'.') 
					FROM pg_trigger
					WHERE tgname = trigger_name_
					AND ( CASE WHEN source_schema = 'public' THEN 
						tgrelid::regclass::text = quote_ident(object) ELSE 
						tgrelid::regclass::text = quote_ident(source_schema) || '.' ||quote_ident(object) END)
					);
				END LOOP;
			END IF;
		  
		ELSE
			SELECT pg_get_expr(p.relpartbound, p.oid) into partition_range_cond from pg_class as p
				JOIN pg_namespace as ns ON (ns.oid = p.relnamespace)
				where relname = object AND ns.nspname = source_schema;
			--select pg_get_expr(relpartbound, oid) into partition_range_cond from pg_class where relispartition and relname = object;
			RAISE NOTICE 'partition_range_cond %', partition_range_cond;
			SELECT REPLACE (REPLACE(partrelid::regclass::text, 'notification.', ''), 'organization.', '')
			INTO parent_rel_table from pg_partitioned_table WHERE partdefid::regclass::text  = quote_ident(source_schema) || '.' || quote_ident(object);
			--select partrelid::regclass::text, partdefid::regclass::text, * from pg_partitioned_table  --where  (partrelid::regclass::text) like quote_ident(source_schema) || '.%'
			IF parent_rel_table IS NULL THEN
				SELECT p.relname into parent_rel_table FROM pg_inherits
					JOIN pg_class as p ON (inhparent=p.oid)
					JOIN pg_namespace as ns ON (ns.oid = p.relnamespace)
					WHERE ((inhrelid::regclass::text ilike '%.' || object) OR 
					(inhrelid::regclass::text = object)) AND ns.nspname = source_schema;
			END IF;
			RAISE NOTICE 'parent_rel_table % of %', parent_rel_table, object;
			EXECUTE 'CREATE TABLE '|| quote_ident(dest_schema) || '.' || quote_ident(object) ||' PARTITION OF '|| quote_ident(dest_schema) 
			|| '.' ||quote_ident(parent_rel_table)||' ' ||partition_range_cond || ';';
		END IF;
		IF (substr((SELECT current_setting('server_version_num')), 1, 2)::integer < 15) THEN
			FOR trigger_name_ IN
			SELECT trigger_name::text FROM information_schema.TRIGGERS 
			WHERE event_object_schema=source_schema and event_object_table=object 
			GROUP BY trigger_name
			LOOP
				buffer := quote_ident(dest_schema) || '.' || quote_ident(object);
				EXECUTE (SELECT 
				REPLACE (
					REPLACE (
					REPLACE(
						REPLACE (
						REPLACE (
							REPLACE (pg_get_triggerdef(oid), 'public.', ''), 
						'notification.', ''), 
						'organization.', ''), 
					'EXECUTE PROCEDURE', 'EXECUTE FUNCTION'),
					'EXECUTE FUNCTION ', CASE WHEN POSITION ('after_insert_default' IN pg_get_triggerdef(oid)) > 0
						THEN 'EXECUTE FUNCTION public.' ELSE 'EXECUTE FUNCTION '|| quote_ident(dest_schema) ||'.' END), 
				' ON ', ' ON '|| quote_ident(dest_schema) ||'.') 
				FROM pg_trigger
				WHERE tgname = trigger_name_
				AND ( CASE WHEN source_schema = 'public' THEN 
					tgrelid::regclass::text = quote_ident(object) ELSE 
					tgrelid::regclass::text = quote_ident(source_schema) || '.' ||quote_ident(object) END)
				);
			END LOOP;
		END IF;
	END LOOP;
	
	--  add FK constraint
	FOR qry IN
		SELECT 'ALTER TABLE ' || quote_ident(dest_schema) || '.' || quote_ident(rn.relname) 
		|| ' ADD CONSTRAINT ' || quote_ident(ct.conname) || ' ' || 
		  REPLACE( REPLACE (REPLACE (pg_get_constraintdef(ct.oid),'notification.', ''), 'organization.', ''),
		'REFERENCES ', CASE WHEN  POSITION( '.' IN pg_get_constraintdef(ct.oid))> 0 THEN 'REFERENCES '||quote_ident(dest_schema) ||'.'  ELSE 'REFERENCES public.' END) || ';'
		FROM pg_constraint ct
		JOIN pg_class rn ON rn.oid = ct.conrelid
		JOIN pg_namespace ns ON ns.oid = connamespace
	   WHERE connamespace = ns.oid
		 --AND rn.relkind = 'r'
		 AND rn.relispartition = false
		 AND ct.contype = 'f'
		 AND ct.conparentid = 0
		 AND ns.nspname = quote_ident(source_schema)
	  LOOP
		  RAISE NOTICE 'query for foreign key %',qry;
		EXECUTE qry;
	
	  END LOOP;
	
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."organizationList" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."userList" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."userOrganizationMapping" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."things" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."scheduledJobs" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."scheduledJobHistoryLogs" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."migrations" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."migrations_lock" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."emailTemplates" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."oAuthRevokedTokens" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."organizationAuthenticationMethods" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."userAuthenticationMethods" CASCADE;';
	EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(dest_schema) || '."reportTemplates" CASCADE;';
	
	EXECUTE 'DROP FUNCTION IF EXISTS ' || quote_ident(dest_schema) || '."clone_schema" CASCADE;';
	EXECUTE 'DROP FUNCTION IF EXISTS ' || quote_ident(dest_schema) || '."disable_trigger" CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."oAuthRevokedTokens" 
		DROP CONSTRAINT IF EXISTS "oAuthRevokedTokens_userId_fkey",
		DROP CONSTRAINT IF EXISTS "oAuthRevokedTokens_userListId_fkey",
		ADD CONSTRAINT "oAuthRevokedTokens_userListId_fkey" 
		FOREIGN KEY ("userId") REFERENCES public."userList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."emailTemplates" 
		DROP CONSTRAINT IF EXISTS "emailTemplates_organizationId_fkey",
		DROP CONSTRAINT IF EXISTS "emailTemplates_organizationListId_fkey",
		ADD CONSTRAINT "emailTemplates_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."scheduledJobs" 
		DROP CONSTRAINT IF EXISTS "scheduledJobs_organizationId_fkey",
		DROP CONSTRAINT IF EXISTS "scheduledJobs_organizationListId_fkey",
		ADD CONSTRAINT "scheduledJobs_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."organizationAuthenticationMethods" 
		DROP CONSTRAINT IF EXISTS "organizationauthenticationmethods_organizationid_foreign",
		DROP CONSTRAINT IF EXISTS "organizationAuthenticationMethods_organizationListId_fkey",
		ADD CONSTRAINT "organizationAuthenticationMethods_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."userAuthenticationMethods" 
		DROP CONSTRAINT IF EXISTS "userauthenticationmethods_organizationid_foreign",
		DROP CONSTRAINT IF EXISTS "userAuthenticationMethods_userListId_fkey",
		ADD CONSTRAINT "userAuthenticationMethods_userListId_fkey" 
		FOREIGN KEY ("userId") REFERENCES public."userList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."userAuthenticationMethods" 
		DROP CONSTRAINT IF EXISTS "userauthenticationmethods_authmethodid_foreign",
		DROP CONSTRAINT IF EXISTS "userAuthenticationMethods_organizationAuthenticationMethodsId_fkey",
		ADD CONSTRAINT "userAuthenticationMethods_organizationAuthenticationMethodsId_fkey" 
		FOREIGN KEY ("authenticationMethodId") REFERENCES public."organizationAuthenticationMethods"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."reportTemplates" 
		DROP CONSTRAINT IF EXISTS "reportTemplates_organizationListId_fkey",
		ADD CONSTRAINT "reportTemplates_organizationListId_fkey" 
		FOREIGN KEY ("organizationId") REFERENCES public."organizationList"(id) ON DELETE CASCADE;';
	
	EXECUTE 'ALTER TABLE IF EXISTS ' || quote_ident(source_schema)|| '."reportTemplates" 
		DROP CONSTRAINT IF EXISTS "reportTemplates_userListId_fkey",
		ADD CONSTRAINT "reportTemplates_userListId_fkey" 
		FOREIGN KEY ("userId") REFERENCES public."userList"(id) ON DELETE CASCADE;';
	
	RETURN; 
	
	END;
	$$;
    `;
    await client.query(query);
}
exports.down = down;
