"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const pgmaintenance_db_utils_1 = require("../../../../components/pgmaintenance/db/pgmaintenance.db.utils");
async function up(client, dbuser, dbsuperuser) {
    await client.query(`
        
DROP FUNCTION IF EXISTS public.after_insert_default() CASCADE; -- With cascade, all default table triggers are dropped

DROP FUNCTION IF EXISTS public.fpartitioncreator(text, text, text);

DROP FUNCTION IF EXISTS public.fpartitioncretor_for_schema_notification_event(text, text, text);
        `);
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
    await client.query(`
        
CREATE OR REPLACE FUNCTION public.after_insert_default()
    RETURNS trigger
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE NOT LEAKPROOF
AS $BODY$
    DECLARE
    res text;
    t_name text;
        BEGIN
        t_name := REPLACE (TG_TABLE_NAME, '_default', '');
        res := jsonb_build_object(
            'tName', t_name,
            'sName', TG_TABLE_SCHEMA,
            'key', (SELECT col.column_name FROM
                (SELECT partrelid, partnatts, case partstrat when 'l' then 'list'
                                        when 'r' then 'range' end as partition_strategy,
                    UNNEST (partattrs) column_index FROM pg_partitioned_table) pt 
                JOIN pg_class par ON par.oid = pt.partrelid
                JOIN information_schema.columns col ON quote_ident(col.table_schema) = par.relnamespace::regnamespace::text
                AND col.table_name = par.relname
                AND ordinal_position = pt.column_index
                WHERE par.relnamespace::regnamespace::text = quote_ident(TG_TABLE_SCHEMA) 
                AND par.relname = t_name
                )
        )::text;
        PERFORM PG_NOTIFY('partition', res);
        RAISE NOTICE  'partition for: %', res;
        RETURN null;
        END;
    
$BODY$;

ALTER FUNCTION public.after_insert_default()
    OWNER TO ${dbsuperuser};
GRANT EXECUTE ON FUNCTION public.after_insert_default() TO PUBLIC;
GRANT EXECUTE ON FUNCTION public.after_insert_default() TO ${dbuser};
GRANT EXECUTE ON FUNCTION public.after_insert_default() TO ${dbsuperuser};
        `);
    const tablesWithPartitions = await (0, pgmaintenance_db_utils_1.findTablesWithPartitions)(client);
    for (const twp of tablesWithPartitions) {
        await client.query(`

CREATE TRIGGER "after_insert_${twp.table_name}_default"
AFTER INSERT
ON "${twp.schema}"."${twp.table_name}_default"
FOR EACH ROW
EXECUTE FUNCTION public.after_insert_default();
            `);
    }
    await client.query(`
CREATE OR REPLACE FUNCTION public.fpartitioncreator(
	org_id text,
	table_name text,
	partition_key text)
    RETURNS void
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE PARALLEL UNSAFE
AS $BODY$
            DECLARE 
                def_partition	text;
                i	timestamp with time zone;
                partition_to_create	text;
                previous_partition	text;
                next_partition	text;
                start_of_partition 	timestamp with time zone;
                end_of_partition 	timestamp with time zone;
            BEGIN
                EXECUTE 'ALTER TABLE '|| quote_ident (org_id) || '.' || quote_ident(table_name) || ' DISABLE TRIGGER ALL;';
                IF (table_name = 'notification_event') THEN
                    PERFORM public.fpartitioncretor_for_schema_notification_event(org_id, table_name, partition_key);
                    EXECUTE 'ALTER TABLE '|| quote_ident (org_id) || '.' || quote_ident(table_name) || ' ENABLE TRIGGER ALL;';
                    RETURN;
                ELSE
                    def_partition := quote_ident(org_id)|| '.' || quote_ident(table_name)|| '_default';
            
                    EXECUTE 'ALTER TABLE '|| quote_ident (org_id) || '.' || quote_ident(table_name) || ' DETACH PARTITION ' || def_partition || ';';
                        FOR i IN EXECUTE 'SELECT T.' || quote_ident(partition_key) || ' FROM ( 
                            SELECT '|| quote_ident(partition_key)||', ROW_NUMBER() OVER (PARTITION BY date_trunc('|| quote_literal ('month') || 
                            ' , '|| quote_ident(partition_key) || ')) AS row_num FROM ' || def_partition || ') AS T WHERE T.row_num = 1;'
                    LOOP
                        partition_to_create := quote_ident(table_name || '_p' || EXTRACT (YEAR FROM i) || lpad ( (EXTRACT (MONTH FROM i))::text, 2 , '0'));
                        previous_partition := quote_ident(table_name || '_p' || EXTRACT (YEAR FROM (i + interval '-1 month')) || lpad ( (EXTRACT (MONTH FROM (i + interval '-1 month')))::text, 2 , '0'));
                        next_partition := quote_ident(table_name || '_p' || EXTRACT (YEAR FROM (i + interval '1 month')) || lpad ( (EXTRACT (MONTH FROM (i + interval '1 month')))::text, 2 , '0'));
        
                        IF NOT EXISTS(
                            SELECT relname FROM pg_class 
                            WHERE relname = previous_partition AND 
                            relnamespace::regnamespace::text = quote_ident(org_id)) THEN
                            start_of_partition := date_trunc('month', i);
                        ELSE
                            start_of_partition := substring (split_part((select pg_get_expr (c.relpartbound, c.oid, true)
                                from pg_class c
                                where relname = previous_partition and 
                                (relnamespace::regnamespace::text = quote_ident(org_id))), 'TO', 2 ) from '\((.+)\)');
                        END IF;
        
                        IF NOT EXISTS(SELECT relname FROM pg_class WHERE relname = next_partition AND 
                            relnamespace::regnamespace::text = quote_ident(org_id)) THEN
                            end_of_partition := date_trunc('month', i) + interval '1 month';
                        ELSE
                            end_of_partition := substring (split_part((select pg_get_expr (c.relpartbound, c.oid, true)
                                from pg_class c
                                where relname = next_partition and 
                                (relnamespace::regnamespace::text = quote_ident(org_id))), 'TO', 1 ) from '\((.+)\)');
                        END IF;
        
                        IF NOT EXISTS(SELECT relname FROM pg_class WHERE relname = partition_to_create AND 
                            relnamespace::regnamespace::text = quote_ident(org_id)) THEN
                            EXECUTE 'CREATE TABLE IF NOT EXISTS ' || quote_ident(org_id) || '.' || quote_ident(partition_to_create)
                                || ' PARTITION OF ' || quote_ident (org_id) || '.' || quote_ident(table_name)
                                || ' FOR VALUES FROM (' || quote_literal (start_of_partition) ||') TO (' || quote_literal(end_of_partition) ||');';
                        END IF;
        
                        EXECUTE 'INSERT INTO ' || quote_ident(org_id) || '.' || quote_ident(partition_to_create)
                            || ' SELECT * FROM '|| def_partition ||' WHERE '|| 
                            quote_ident(partition_key) || ' >= ' || quote_literal(start_of_partition) || ' AND ' || 
                            quote_ident(partition_key) || ' < ' || quote_literal(end_of_partition) || ';';
        
                        EXECUTE 'DELETE FROM ' || def_partition || ' WHERE '|| 
                            quote_ident(partition_key) || ' >= ' || quote_literal(start_of_partition) || ' AND ' || 
                            quote_ident(partition_key) || ' < ' || quote_literal(end_of_partition) || ';';
        
                    END LOOP;
        
                    EXECUTE 'ALTER TABLE '|| quote_ident (org_id) || '.' || quote_ident(table_name) || ' ATTACH PARTITION ' || def_partition || ' DEFAULT;';
                    EXECUTE 'ALTER TABLE '|| quote_ident (org_id) || '.' || quote_ident(table_name) || ' ENABLE TRIGGER ALL;';
                    RETURN;
                END IF;
            END;
        
$BODY$;

ALTER FUNCTION public.fpartitioncreator(text, text, text)
    OWNER TO ${dbsuperuser};
GRANT EXECUTE ON FUNCTION public.fpartitioncreator(text, text, text) TO PUBLIC;
GRANT EXECUTE ON FUNCTION public.fpartitioncreator(text, text, text) TO ${dbuser};
GRANT EXECUTE ON FUNCTION public.fpartitioncreator(text, text, text) TO ${dbsuperuser};
        



CREATE OR REPLACE FUNCTION public.fpartitioncretor_for_schema_notification_event(
	org_id text,
	table_name text,
	partition_key text)
    RETURNS void
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE PARALLEL UNSAFE
AS $BODY$
		DECLARE 
						def_partition	text;
						i	timestamp without time zone;
						partition_to_create	text;
						previous_partition	text;
						next_partition	text;
						start_of_partition 	timestamp without time zone;
						end_of_partition 	timestamp without time zone;
					BEGIN
						def_partition := quote_ident(org_id)|| '.' || quote_ident(table_name)|| '_default';
						
						EXECUTE 'CREATE TEMPORARY TABLE ' || quote_ident('temp_' || org_id || '_' || table_name || '_default') || ' ON COMMIT DROP AS SELECT * FROM '
							|| def_partition || ';';
			
						EXECUTE 'CREATE TEMPORARY TABLE ' || quote_ident('temp_' || org_id || '_notification_instance') || ' ON COMMIT DROP AS SELECT * FROM '
							|| quote_ident(org_id) || '.notification_instance'||' WHERE "eventId" IN (SELECT id FROM '|| 
							quote_ident('temp_' || org_id || '_' || table_name || '_default') ||');';
						
						EXECUTE 'TRUNCATE ' || def_partition || ' CASCADE;';
						
						FOR i IN EXECUTE 'SELECT T.' || quote_ident(partition_key) || ' FROM ( 
							SELECT '|| quote_ident(partition_key)||', ROW_NUMBER() OVER (PARTITION BY date_trunc('|| quote_literal ('month') || 
							' , '|| quote_ident(partition_key) || ')) AS row_num FROM ' || quote_ident('temp_' || org_id || '_' || table_name || '_default') || ') AS T WHERE T.row_num = 1;'
						LOOP
							partition_to_create := quote_ident(table_name || '_p' || EXTRACT (YEAR FROM i) || lpad ( (EXTRACT (MONTH FROM i))::text, 2 , '0'));
							previous_partition := quote_ident(table_name || '_p' || EXTRACT (YEAR FROM (i + interval '-1 month')) || lpad ( (EXTRACT (MONTH FROM (i + interval '-1 month')))::text, 2 , '0'));
							next_partition := quote_ident(table_name || '_p' || EXTRACT (YEAR FROM (i + interval '1 month')) || lpad ( (EXTRACT (MONTH FROM (i + interval '1 month')))::text, 2 , '0'));
					
							IF NOT EXISTS(
								SELECT relname FROM pg_class 
								WHERE relname = previous_partition AND 
								relnamespace::regnamespace::text = quote_ident(org_id)) THEN
								start_of_partition := date_trunc('month', i);
							ELSE
								start_of_partition := substring (split_part((select pg_get_expr (c.relpartbound, c.oid, true)
									from pg_class c
									where relname = previous_partition and 
									(relnamespace::regnamespace::text = quote_ident(org_id))), 'TO', 2 ) from '\((.+)\)');
							END IF;
							
							IF NOT EXISTS(SELECT relname FROM pg_class WHERE relname = next_partition AND 
								relnamespace::regnamespace::text = quote_ident(org_id)) THEN
								end_of_partition := date_trunc('month', i) + interval '1 month';
							ELSE
								end_of_partition := substring (split_part((select pg_get_expr (c.relpartbound, c.oid, true)
									from pg_class c
									where relname = next_partition and 
									(relnamespace::regnamespace::text = quote_ident(org_id))), 'TO', 1 ) from '\((.+)\)');
							END IF;
							
							IF NOT EXISTS(SELECT relname FROM pg_class WHERE relname = partition_to_create AND 
								relnamespace::regnamespace::text = quote_ident(org_id)) THEN
								EXECUTE 'CREATE TABLE IF NOT EXISTS ' || quote_ident(org_id) || '.' || quote_ident(partition_to_create)
									|| ' PARTITION OF ' || quote_ident (org_id) || '.' || quote_ident(table_name)
									|| ' FOR VALUES FROM (' || quote_literal (start_of_partition) ||') TO (' || quote_literal(end_of_partition) ||');';
							END IF;
							
							EXECUTE 'INSERT INTO ' || quote_ident(org_id) || '.' || quote_ident(partition_to_create)
							|| ' SELECT * FROM '|| quote_ident('temp_' || org_id || '_' || table_name || '_default') ||' WHERE '|| 
							quote_ident(partition_key) || ' >= ' || quote_literal(start_of_partition) || ' AND ' || 
							quote_ident(partition_key) || ' < ' || quote_literal(end_of_partition) || ' ON CONFLICT DO NOTHING;';
							
						END LOOP;
			
						EXECUTE 'INSERT INTO ' || quote_ident(org_id) ||'.notification_instance SELECT * FROM '
						|| quote_ident('temp_' || org_id || '_notification_instance') || ' ON CONFLICT DO NOTHING;';
						
					RETURN;
					END;
		
$BODY$;

ALTER FUNCTION public.fpartitioncretor_for_schema_notification_event(text, text, text)
    OWNER TO ${dbsuperuser};
GRANT EXECUTE ON FUNCTION public.fpartitioncretor_for_schema_notification_event(text, text, text) TO PUBLIC;
GRANT EXECUTE ON FUNCTION public.fpartitioncretor_for_schema_notification_event(text, text, text) TO ${dbuser};
GRANT EXECUTE ON FUNCTION public.fpartitioncretor_for_schema_notification_event(text, text, text) TO ${dbsuperuser};

        `);
}
exports.down = down;
