"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const luxon_1 = require("luxon");
async function up(client, dbuser, dbsuperuser) {
    const partitionsWithWrongRangesDbResult = await client.query(`
    SELECT 
        sq.range_start,
        sq.range_end,
        sq.schema_name,
        jsonb_agg(
                    jsonb_build_object(
                        'base_table_name', sq.base_table_name,
                        'partition_name', sq.partition_name
                    ) 
            ) AS tables_to_process
    FROM
    (
        SELECT 
            split_part(pg_get_expr(pt.relpartbound, pt.oid, true),'''', 2)::timestamptz AS range_start,
            split_part(pg_get_expr(pt.relpartbound, pt.oid, true),'''', 4)::timestamptz AS range_end,
            replace (base_tb.relnamespace::regnamespace::text, '"', '') AS schema_name,
            base_tb.relname AS base_table_name,
            pt.relname AS partition_name
        FROM pg_class base_tb 
        JOIN pg_inherits i ON i.inhparent = base_tb.oid 
        JOIN pg_class pt ON pt.oid = i.inhrelid
        WHERE (base_tb.oid::regclass::text ilike '%access_logs'
        or base_tb.oid::regclass::text ilike '%user_region_states')
        and pt.relname NOT ILIKE '%_default'
    ) sq
    WHERE sq.range_start + interval '45 days' < sq.range_end
    GROUP BY sq.range_start, sq.range_end, sq.schema_name
    `);
    if (!partitionsWithWrongRangesDbResult.rowCount) {
        console.log(`There is no partition with wrong ranges, skipping recovery`);
    }
    else {
        for (const row of partitionsWithWrongRangesDbResult.rows) {
            if (row.tables_to_process.length !== 2 ||
                !row.tables_to_process.find((f) => f.base_table_name !== "access_logs") ||
                !row.tables_to_process.find((f) => f.base_table_name !== "user_region_states")) {
                console.error(`==============================ATTENTION=====================================`);
                console.error(`There are unexpected partitions for organization, please get in contact with development team`);
                return;
            }
            const firstMonthStart = luxon_1.DateTime.fromJSDate(row.range_start).toFormat("y-LL") + "-01 00:00:00";
            const firstMonthPostfix = luxon_1.DateTime.fromJSDate(row.range_start).toFormat("yLL");
            const secondMonthStart = luxon_1.DateTime.fromJSDate(row.range_start).plus({ months: 1 }).toFormat("y-LL") + "-01 00:00:00";
            const secondMonthPostfix = luxon_1.DateTime.fromJSDate(row.range_start).plus({ months: 1 }).toFormat("yLL");
            const secondMonthEnd = luxon_1.DateTime.fromJSDate(row.range_end).toFormat("y-LL") + "-01 00:00:00";
            await client.query(`

    -- Detach Partitions

    ALTER TABLE "${row.schema_name}"."user_region_states"
    DETACH PARTITION "${row.schema_name}"."${row.tables_to_process.find((f) => f.base_table_name === "user_region_states").partition_name}";

    ALTER TABLE "${row.schema_name}"."access_logs"
    DETACH PARTITION "${row.schema_name}"."${row.tables_to_process.find((f) => f.base_table_name === "access_logs").partition_name}";

    -- Rename partitions

    ALTER TABLE "${row.schema_name}"."${row.tables_to_process.find((f) => f.base_table_name === "access_logs").partition_name}"
    RENAME TO "access_logs_temp";

    ALTER TABLE "${row.schema_name}"."${row.tables_to_process.find((f) => f.base_table_name === "user_region_states").partition_name}"
    RENAME TO "user_region_states_temp";

    -- Create new partitions with correct ranges

    CREATE TABLE IF NOT EXISTS "${row.schema_name}".user_region_states_p${firstMonthPostfix} 
    PARTITION OF "${row.schema_name}".user_region_states
    FOR VALUES FROM ('${firstMonthStart}') TO ('${secondMonthStart}');

    CREATE TABLE IF NOT EXISTS "${row.schema_name}".user_region_states_p${secondMonthPostfix} 
    PARTITION OF "${row.schema_name}".user_region_states
    FOR VALUES FROM ('${secondMonthStart}') TO ('${secondMonthEnd}');

    CREATE TABLE IF NOT EXISTS "${row.schema_name}".access_logs_p${firstMonthPostfix} 
    PARTITION OF "${row.schema_name}".access_logs
    FOR VALUES FROM ('${firstMonthStart}') TO ('${secondMonthStart}');

    CREATE TABLE IF NOT EXISTS "${row.schema_name}".access_logs_p${secondMonthPostfix} 
    PARTITION OF "${row.schema_name}".access_logs
    FOR VALUES FROM ('${secondMonthStart}') TO ('${secondMonthEnd}');

    -- Temporarily disable trigger in order to prevent duplicate user_region_states entries

    ALTER TABLE "${row.schema_name}".access_logs
    DISABLE TRIGGER access_logs_after_insert_add_region_state_trigger;

    -- Restore access_logs

    INSERT INTO "${row.schema_name}".access_logs
    SELECT * FROM "${row.schema_name}".access_logs_temp;
    
    -- Re-enable previously disabled triggers

    ALTER TABLE "${row.schema_name}".access_logs
    ENABLE TRIGGER access_logs_after_insert_add_region_state_trigger;

    -- Restore user_region_states

    INSERT INTO "${row.schema_name}".user_region_states
    SELECT * FROM "${row.schema_name}".user_region_states_temp;

    -- Drop old partitions (*_temp)

    DROP TABLE "${row.schema_name}".access_logs_temp;
    DROP TABLE "${row.schema_name}".user_region_states_temp;
                `);
        }
    }
    const templateSchemaPartitionsToDrop = await client.query(`
    SELECT
        nmsp_parent.nspname AS schema,
        parent.relname      AS base_table,
        child.relname       AS partition_name
    FROM pg_inherits
        JOIN pg_class parent            ON pg_inherits.inhparent = parent.oid
        JOIN pg_class child             ON pg_inherits.inhrelid   = child.oid
        JOIN pg_namespace nmsp_parent   ON nmsp_parent.oid  = parent.relnamespace
        JOIN pg_namespace nmsp_child    ON nmsp_child.oid   = child.relnamespace
    WHERE nmsp_parent.nspname = 'organization'
        and child.relname not ilike '%_default'
        and child.relkind= 'r'
    `);
    for (const row of templateSchemaPartitionsToDrop.rows) {
        await client.query(`
            ALTER TABLE "${row.schema}"."${row.base_table}"
            DETACH PARTITION "${row.schema}"."${row.partition_name}";

            DROP TABLE "${row.schema}"."${row.partition_name}";
        `);
    }
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
}
exports.down = down;
