"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_utils_1 = require("../../../dal.utils");
const luxon_1 = require("luxon");
async function up(client, dbuser, dbsuperuser) {
    const currentPartitionName = luxon_1.DateTime.now().toFormat("yLL");
    const currentPartitionStart = luxon_1.DateTime.now().startOf("month").toFormat("y-LL") + "-01 00:00:00";
    const currentPartitionEnd = luxon_1.DateTime.now().startOf("month").plus({ months: 1 }).toFormat("y-LL") + "-01 00:00:00";
    const nextPartitionName = luxon_1.DateTime.now().plus({ months: 1 }).toFormat("yLL");
    const nextPartitionStart = luxon_1.DateTime.now().plus({ months: 1 }).startOf("month").toFormat("y-LL") + "-01 00:00:00";
    const nextPartitionEnd = luxon_1.DateTime.now().plus({ months: 1 }).startOf("month").plus({ months: 1 }).toFormat("y-LL") + "-01 00:00:00";
    const regionStateTableAndTrigger = `
    -- Table for holding the data itself
	CREATE TABLE "___ORGANIZATION_ID___"."user_region_states"
	(
		id uuid NOT NULL,
		"userId" uuid NOT NULL,
		"regionId" uuid NOT NULL,
		"accessControlPointId" uuid,
		"accessLogId" uuid,
        "actionUtc" timestamp with time zone,
        action integer,
		CONSTRAINT "user_region_states_pkey" PRIMARY KEY (id, "actionUtc"),
        CONSTRAINT "user_region_states_userId_fkey" FOREIGN KEY ("userId")
            REFERENCES "___ORGANIZATION_ID___".users (id) MATCH SIMPLE
            ON UPDATE CASCADE
            ON DELETE CASCADE,
        CONSTRAINT "user_region_states_regionId_fkey" FOREIGN KEY ("regionId")
            REFERENCES "___ORGANIZATION_ID___".regions (id) MATCH SIMPLE
            ON UPDATE CASCADE
            ON DELETE CASCADE,
        CONSTRAINT "user_region_states_accessControlPointId_fkey" FOREIGN KEY ("accessControlPointId")
            REFERENCES "___ORGANIZATION_ID___"."accessControlPoints" (id) MATCH SIMPLE
            ON UPDATE CASCADE
            ON DELETE SET NULL,
        CONSTRAINT "user_region_states_access_logs_fkey" FOREIGN KEY ("accessLogId", "actionUtc")
            REFERENCES "___ORGANIZATION_ID___".access_logs (id, "actionUtc") MATCH SIMPLE
            ON UPDATE CASCADE
            ON DELETE CASCADE,
        CONSTRAINT "user_region_states_action_check" CHECK (action = ANY ('{0,1,2}'::smallint[]))
	) PARTITION BY RANGE ("actionUtc");

    CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".user_region_states_default PARTITION OF "___ORGANIZATION_ID___".user_region_states DEFAULT; 
    CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".user_region_states_p${currentPartitionName} PARTITION OF "___ORGANIZATION_ID___".user_region_states
        FOR VALUES FROM ('${currentPartitionStart}') TO ('${currentPartitionEnd}');
    CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".user_region_states_p${nextPartitionName} PARTITION OF "___ORGANIZATION_ID___".user_region_states
        FOR VALUES FROM ('${nextPartitionStart}') TO ('${nextPartitionEnd}');
    
    CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".access_logs_p${currentPartitionName} PARTITION OF "___ORGANIZATION_ID___".access_logs
        FOR VALUES FROM ('${currentPartitionStart}') TO ('${currentPartitionEnd}');
    CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".access_logs_p${nextPartitionName} PARTITION OF "___ORGANIZATION_ID___".access_logs
        FOR VALUES FROM ('${nextPartitionStart}') TO ('${nextPartitionEnd}');
	
	ALTER TABLE IF EXISTS "___ORGANIZATION_ID___"."user_region_states" OWNER to ${dbsuperuser};
	GRANT DELETE, INSERT, SELECT, UPDATE ON TABLE "___ORGANIZATION_ID___"."user_region_states" TO ${dbuser};
	GRANT ALL ON TABLE "___ORGANIZATION_ID___"."user_region_states" TO ${dbsuperuser};

    -- Trigger function to add region state entry on access log insert
    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".access_logs_after_insert_add_region_state()
        RETURNS trigger
        LANGUAGE 'plpgsql'
        COST 100
        VOLATILE NOT LEAKPROOF
    AS $BODY$
    BEGIN
        INSERT INTO "___ORGANIZATION_ID___"."user_region_states"
        (id, "userId", "regionId", "accessControlPointId", "accessLogId", "actionUtc", action)
        SELECT gen_random_uuid(), (NEW.log->>'o')::uuid, (rgs->>'i')::uuid, (NEW.log->>'a')::uuid as "accessControlPointId", NEW.id, NEW."actionUtc", (rgs->>'s')::integer
        FROM jsonb_array_elements(NEW.log->'rg') as rgs
        WHERE NEW.log->>'rg' IS NOT NULL AND NEW.log->>'o' IS NOT NULL
        AND EXISTS(SELECT id FROM "___ORGANIZATION_ID___".regions WHERE id = (rgs->>'i')::uuid);
        RETURN NEW;
    END;
    $BODY$;

    ALTER FUNCTION "___ORGANIZATION_ID___".access_logs_after_insert_add_region_state() OWNER TO ${dbsuperuser};
    GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".access_logs_after_insert_add_region_state() TO ${dbuser};
    GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".access_logs_after_insert_add_region_state() TO ${dbsuperuser};

    -- Trigger itself
    CREATE OR REPLACE TRIGGER access_logs_after_insert_add_region_state_trigger
    AFTER INSERT
    ON "___ORGANIZATION_ID___"."access_logs"
    FOR EACH ROW
    EXECUTE FUNCTION "___ORGANIZATION_ID___".access_logs_after_insert_add_region_state();

    CREATE TRIGGER after_insert_user_region_states_default 
    AFTER INSERT 
        ON "___ORGANIZATION_ID___".user_region_states_default
    FOR EACH ROW 
        EXECUTE FUNCTION public.after_insert_default();
`;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, regionStateTableAndTrigger, "___ORGANIZATION_ID___");
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
    const regionStateTableAndTrigger = `
    -- Trigger itself
    DROP TRIGGER IF EXISTS access_logs_after_insert_add_region_state_trigger ON "___ORGANIZATION_ID___"."access_logs";
    -- Trigger function to add region state entry on access log insert
    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".access_logs_after_insert_add_region_state();
    -- Table for holding the data itself
	DROP TABLE IF EXISTS "___ORGANIZATION_ID___"."user_region_states";
`;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, regionStateTableAndTrigger, "___ORGANIZATION_ID___");
}
exports.down = down;
