"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_utils_1 = require("../../../dal.utils");
const dal_db_armon_schema_1 = require("../../armon/dal.db.armon.schema");
const predefined_permissions_1 = require("../../predefined/predefined.permissions");
async function up(client, dbuser, dbsuperuser) {
    const q = `
	ALTER TABLE "___ORGANIZATION_ID___"."regions" ADD COLUMN IF NOT EXISTS capacity INTEGER;
	ALTER TABLE "___ORGANIZATION_ID___"."userGroupAccessRuleSets" 
		ADD COLUMN IF NOT EXISTS "currentCapacityUsage" INTEGER,
		ADD COLUMN IF NOT EXISTS "lastUpdateAccessControlPointId" UUID,
		ADD COLUMN IF NOT EXISTS "lastActionDateISO" TIMESTAMP WITH TIME ZONE,
		ALTER COLUMN "userGroupId" SET NOT NULL,
		ALTER COLUMN "accessRuleSetId" SET NOT NULL,
		ADD CONSTRAINT "userGroupAccessRuleSets_userGroupId_accessRuleSetId_key" UNIQUE ("userGroupId", "accessRuleSetId"),
		ADD CONSTRAINT "userGroupAccessRuleSets_accessControlPointId_fkey" FOREIGN KEY ("lastUpdateAccessControlPointId") 
			REFERENCES "___ORGANIZATION_ID___"."accessControlPoints"(id) ON UPDATE CASCADE ON DELETE SET NULL;


	-- FUNCTION: "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets()
	-- DROP FUNCTION "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets();
	
	CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets()
		RETURNS trigger
		LANGUAGE 'plpgsql'
		COST 100
		VOLATILE NOT LEAKPROOF
	AS $BODY$
	DECLARE
		data_object		 	json;
		related_device_ids 	uuid[];
		res					text;
		BEGIN
			IF (
				(
					OLD."currentCapacityUsage" = NEW."currentCapacityUsage"
						AND OLD."lastUpdateAccessControlPointId" = NEW."lastUpdateAccessControlPointId"
						AND OLD."lastActionDateISO" = NEW."lastActionDateISO"
				) 
				OR   
				(
					NEW."currentCapacityUsage" = NULL
					AND NEW."lastUpdateAccessControlPointId" = NULL
					AND NEW."lastActionDateISO" = NULL
				) 
			)THEN
				RETURN NULL;
			END IF;
			SELECT ARRAY_AGG(T."deviceId")
			FROM
			(
				SELECT acp."deviceId", ROW_NUMBER() OVER (PARTITION BY acp."deviceId") AS row_num
					FROM "___ORGANIZATION_ID___"."accessControlPoints" AS acp
					INNER JOIN "___ORGANIZATION_ID___"."regionAccessControlPoints" AS racp ON racp."accessControlPointId" = acp."id"
					INNER JOIN "___ORGANIZATION_ID___"."accessRuleSets" AS ars ON ars."id" =  NEW."accessRuleSetId" AND racp."regionId" = ars."regionId"
					WHERE acp."deletedAt" IS NULL
						AND ars."deletedAt" IS NULL
						AND acp."organizationId" = ars."organizationId"
						AND CASE WHEN NEW."lastUpdateAccessControlPointId" IS NOT NULL THEN 
						acp."deviceId" != (SELECT "deviceId" FROM "___ORGANIZATION_ID___"."accessControlPoints" 
										WHERE "id" = NEW."lastUpdateAccessControlPointId")
						ELSE TRUE END
			) AS T
			WHERE T."row_num" = 1
			INTO related_device_ids;
			data_object := JSON_BUILD_OBJECT(
					'ruleId', (SELECT id FROM "___ORGANIZATION_ID___"."accessRules" WHERE "accessRuleSetId" = NEW."accessRuleSetId" AND type = 3 limit 1),
					'current', NEW."currentCapacityUsage",
					'groupId', NEW."userGroupId");
			res :=JSON_BUILD_OBJECT(
				'data', data_object,
				'relatedDeviceIds', related_device_ids,
				'organizationId', TG_TABLE_SCHEMA)::text;
			PERFORM PG_NOTIFY('access_capacity_based_rule_applied', res);
			RETURN NULL;
		END;
	$BODY$;
	
	ALTER FUNCTION "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets()
		OWNER TO ${dbsuperuser};
	
	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets() TO ${dbsuperuser};
	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets() TO PUBLIC;
	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets() TO ${dbuser};
	
	-- Trigger: after_up_user_group_access_rule_sets
	-- DROP TRIGGER after_up_user_group_access_rule_sets ON "___ORGANIZATION_ID___"."userGroupAccessRuleSets";
	CREATE TRIGGER after_up_user_group_access_rule_sets
		AFTER UPDATE 
		ON "___ORGANIZATION_ID___"."userGroupAccessRuleSets"
		FOR EACH ROW
		EXECUTE PROCEDURE "___ORGANIZATION_ID___".after_up_user_group_access_rule_sets();



	-- FUNCTION: "___ORGANIZATION_ID___".before_up_in_delete_user_group_access_rule_sets()

	-- DROP FUNCTION "___ORGANIZATION_ID___".before_up_in_delete_user_group_access_rule_sets();
	
	CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".before_up_in_delete_user_group_access_rule_sets()
		RETURNS trigger
		LANGUAGE 'plpgsql'
		COST 100
		VOLATILE NOT LEAKPROOF
	AS $BODY$
	DECLARE
		transaction_id	uuid;
		device_id_list  uuid[];
		BEGIN
			transaction_id := uuid_generate_v4();
			IF (TG_OP = 'DELETE') THEN
				SELECT ARRAY_AGG(T."deviceId")
				FROM
				(
					SELECT acp."deviceId", ROW_NUMBER() OVER (PARTITION BY acp."deviceId") AS row_num
						FROM "___ORGANIZATION_ID___"."accessControlPoints" AS acp
						INNER JOIN "___ORGANIZATION_ID___"."regionAccessControlPoints" AS racp ON racp."accessControlPointId" = acp."id"
						INNER JOIN "___ORGANIZATION_ID___"."accessRuleSets" AS ars ON ars."id" =  OLD."accessRuleSetId" AND racp."regionId" = ars."regionId"
						WHERE acp."deletedAt" IS NULL
							AND ars."deletedAt" IS NULL
							AND acp."organizationId" = ars."organizationId"
				) AS T
				WHERE T."row_num" = 1
				INTO device_id_list;
			ELSE
				IF (TG_OP = 'UPDATE' 
					AND OLD."id" = NEW."id" 
					AND OLD."userGroupId" = NEW."userGroupId" 
					AND OLD."accessRuleSetId" = NEW."accessRuleSetId") THEN
					RETURN NEW;
				END IF;
				SELECT ARRAY_AGG(T."deviceId")
					FROM
					(
						SELECT acp."deviceId", ROW_NUMBER() OVER (PARTITION BY acp."deviceId") AS row_num
						FROM "___ORGANIZATION_ID___"."accessControlPoints" AS acp
						INNER JOIN "___ORGANIZATION_ID___"."regionAccessControlPoints" AS racp ON racp."accessControlPointId" = acp."id"
						INNER JOIN "___ORGANIZATION_ID___"."accessRuleSets" AS ars ON ars."id" =  NEW."accessRuleSetId" AND racp."regionId" = ars."regionId"
						WHERE acp."deletedAt" IS NULL
							AND ars."deletedAt" IS NULL
							AND acp."organizationId" = ars."organizationId"
					) AS T
					WHERE T."row_num" = 1
				INTO device_id_list;
			END IF;
			INSERT INTO "___ORGANIZATION_ID___"."terminalChanges"
			("id", "transactionId", "deviceId", "actionDateISO", "type", "data")
			VALUES(uuid_generate_v4(), transaction_id, UNNEST(device_id_list), now(), 3, '{}'::json);
	
			IF (TG_OP = 'DELETE') THEN
				RETURN OLD;
			ELSE
				RETURN NEW;
			END IF;
	
		END;
	$BODY$;
	
	ALTER FUNCTION "___ORGANIZATION_ID___".before_up_in_delete_user_group_access_rule_sets()
		OWNER TO ${dbsuperuser};
	
	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".before_up_in_delete_user_group_access_rule_sets() TO ${dbsuperuser};
	
	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".before_up_in_delete_user_group_access_rule_sets() TO PUBLIC;
	
	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".before_up_in_delete_user_group_access_rule_sets() TO ${dbuser};
	


	-- FUNCTION: ___ORGANIZATION_ID___.after_in_up_del_anti_passback_state()
	-- DROP FUNCTION "___ORGANIZATION_ID___".after_in_up_del_anti_passback_state();
	CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_in_up_del_anti_passback_state()
		RETURNS trigger
		LANGUAGE 'plpgsql'
		COST 100
		VOLATILE NOT LEAKPROOF
	AS $BODY$
	DECLARE
				data_object		 	json;
				related_device_ids 	uuid[];
				res					text;
				region_id			uuid;
			BEGIN
				IF (TG_OP = 'DELETE') THEN
					region_id := OLD."regionId";
				ELSIF (TG_OP = 'INSERT') THEN
					region_id := NEW."regionId";
				ELSIF (TG_OP = 'UPDATE') THEN
					region_id := NEW."regionId";
				END IF;
				
				IF ((SELECT "antiPassback" FROM "___ORGANIZATION_ID___"."regions" WHERE id = region_id) != TRUE
				AND (SELECT "capacity" FROM "___ORGANIZATION_ID___"."regions" WHERE id = region_id) IS NULL) THEN
					RETURN NULL;
				END IF;
				IF (TG_OP = 'DELETE') THEN
					SELECT ARRAY_AGG(T."deviceId") FROM (
						SELECT acp."deviceId", ROW_NUMBER() OVER (PARTITION BY acp."deviceId") AS row_num
							FROM "___ORGANIZATION_ID___"."userAccessRights" as uar
							INNER JOIN "___ORGANIZATION_ID___"."accessControlPoints" AS acp ON uar."accessControlPointId" = acp."id"
							INNER JOIN "___ORGANIZATION_ID___"."regionAccessControlPoints" AS racp ON racp."accessControlPointId" = acp."id"
							WHERE racp."regionId" = OLD."regionId" AND
								acp."deletedAt" IS NULL AND
								uar."userId" = OLD."userId" AND
								uar."deletedAt" IS NULL 
					) AS T INTO related_device_ids
					WHERE T.row_num = 1;
					data_object := JSON_BUILD_OBJECT(
						'userId', OLD."userId",
						'regionId', OLD."regionId",
						'state', 0,
						'actionDateISO', now());
					UPDATE "___ORGANIZATION_ID___"."userGroupAccessRuleSets"
					SET "currentCapacityUsage" = "currentCapacityUsage" - 1, "lastUpdateAccessControlPointId" = NULL, "lastActionDateISO" = NULL
					WHERE "accessRuleSetId" = (
						SELECT ar."accessRuleSetId" FROM "___ORGANIZATION_ID___"."accessRules" AS ar
						INNER JOIN "___ORGANIZATION_ID___"."accessRuleSets" AS ars 
							ON ar."accessRuleSetId" = ars.id 
							AND ar."type" = 3
							AND ars."regionId" = OLD."regionId"
							AND ar."deletedAt" IS NULL
							AND ars."deletedAt" IS NULL
						)
						AND "userGroupId" = ANY (
							SELECT "userGroupId" FROM "___ORGANIZATION_ID___"."userGroupUserOrganizations" AS uguo
							INNER JOIN "___ORGANIZATION_ID___"."userOrganizations" AS uo
								ON uo.id = uguo."userOrganizationId"
								AND uguo."deletedAt" IS NULL
								AND uo."deletedAt" IS NULL
								AND uo."userId" = OLD."userId"
						)
						AND "currentCapacityUsage" IS NOT NULL
						AND "currentCapacityUsage" > 0;
				ELSIF (TG_OP = 'INSERT') THEN
					SELECT ARRAY_AGG(T."deviceId") FROM (
						SELECT acp."deviceId", ROW_NUMBER() OVER (PARTITION BY acp."deviceId") AS row_num
							FROM "___ORGANIZATION_ID___"."userAccessRights" as uar
							INNER JOIN "___ORGANIZATION_ID___"."accessControlPoints" AS acp ON uar."accessControlPointId" = acp."id"
							INNER JOIN "___ORGANIZATION_ID___"."regionAccessControlPoints" AS racp ON racp."accessControlPointId" = acp."id"
							WHERE racp."regionId" = NEW."regionId" AND
								acp."deletedAt" IS NULL AND
								uar."userId" = NEW."userId" AND
								uar."deletedAt" IS NULL 
						AND	acp."deviceId" != (SELECT "deviceId" FROM "___ORGANIZATION_ID___"."accessControlPoints" WHERE "id" = NEW."accessControlPointId")
					) AS T INTO related_device_ids
					WHERE T.row_num = 1;
					data_object := JSON_BUILD_OBJECT(
						'userId', NEW."userId",
						'regionId', NEW."regionId",
						'state', NEW."state",
						'actionDateISO', NEW."actionUtc",
						'entranceLockExpirationDateISO', NEW."entranceLockExpirationUtc",
						'exitLockExpirationDateISO', NEW."exitLockExpirationUtc");
				ELSIF (TG_OP = 'UPDATE') THEN
					IF (OLD."userId" != NEW."userId" OR OLD."regionId" != NEW."regionId") THEN
						RETURN NULL;
					END IF;
					SELECT ARRAY_AGG(T."deviceId") FROM (
						SELECT acp."deviceId", ROW_NUMBER() OVER (PARTITION BY acp."deviceId") AS row_num
							FROM "___ORGANIZATION_ID___"."userAccessRights" as uar
							INNER JOIN "___ORGANIZATION_ID___"."accessControlPoints" AS acp ON uar."accessControlPointId" = acp."id"
							INNER JOIN "___ORGANIZATION_ID___"."regionAccessControlPoints" AS racp ON racp."accessControlPointId" = acp."id"
							WHERE racp."regionId" = NEW."regionId" AND
								acp."deletedAt" IS NULL AND
								uar."userId" = NEW."userId" AND
								uar."deletedAt" IS NULL 
						AND
								acp."deviceId" != (SELECT "deviceId" FROM "___ORGANIZATION_ID___"."accessControlPoints" WHERE "id" = NEW."accessControlPointId")
					) AS T INTO related_device_ids
					WHERE T.row_num = 1;
		
					data_object := JSON_BUILD_OBJECT(
							'userId', NEW."userId",
							'regionId', NEW."regionId",
							'state', NEW."state",
							'actionDateISO', NEW."actionUtc",
							'entranceLockExpirationDateISO', NEW."entranceLockExpirationUtc",
							'exitLockExpirationDateISO', NEW."exitLockExpirationUtc");
				END IF;
				res :=JSON_BUILD_OBJECT(
							'data', data_object,
							'relatedDeviceIds', related_device_ids,
							'organizationId', TG_TABLE_SCHEMA)::text;
				PERFORM PG_NOTIFY('region_state_changes', res);
				RETURN NULL;
			END;
	$BODY$;

	ALTER FUNCTION "___ORGANIZATION_ID___".after_in_up_del_anti_passback_state()
		OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_in_up_del_anti_passback_state() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_in_up_del_anti_passback_state() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_in_up_del_anti_passback_state() TO ${dbuser};
	`;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    const organizationList = await client.query(`
    	SELECT id FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"
    `);
    const missingPermission = '"' + predefined_permissions_1.Permissions.user.getUser() + '"';
    for (const organization of organizationList.rows) {
        const existingRoles = await client.query(`
        	SELECT id, permissions FROM "${organization.id}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" WHERE "deletedAt" IS NULL
        `);
        for (const role of existingRoles.rows) {
            const permissions = role.permissions.replace(new RegExp("[\\[\\]]", "g"), "");
            const permissionsArray = permissions.indexOf(",") > 0 ? permissions.split(",") : [];
            let updateNeeded = false;
            if (!permissionsArray.includes(missingPermission)) {
                updateNeeded = true;
                permissionsArray.push(missingPermission);
            }
            if (updateNeeded) {
                const newPermissions = "[" + permissionsArray.join(",") + "]";
                await client.query(`
                UPDATE "${organization.id}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
                SET permissions = $1
                WHERE id = $2
                `, [newPermissions, role.id]);
            }
        }
    }
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
    const q = `
		ALTER TABLE "___ORGANIZATION_ID___"."regions" DROP COLUMN IF EXISTS "capacity";
		ALTER TABLE "___ORGANIZATION_ID___"."userGroupAccessRuleSets" 
		DROP COLUMN IF EXISTS "currentCapacityUsage",
		DROP COLUMN IF EXISTS "lastUpdateAccessControlPointId",
		DROP COLUMN IF EXISTS "lastActionDateISO",
		ALTER COLUMN "userGroupId" DROP NOT NULL,
		ALTER COLUMN "accessRuleSetId" DROP NOT NULL,
		DROP CONSTRAINT IF EXISTS "userGroupAccessRuleSets_userGroupId_accessRuleSetId_key",
		DROP CONSTRAINT IF EXISTS "userGroupAccessRuleSets_accessControlPointId_fkey"
    `;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
}
exports.down = down;
