"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.binToDec = exports.decToHex = exports.decToBin = exports.hexToDec = exports.hexToBin = exports.pad = exports.checkHex = exports.checkDec = exports.checkBin = exports.createCredentialData = exports.credentialTypeToAuthenticationFactor = exports.authenticationFactorToCredentialTypes = exports.undefinedToNull = exports.generateHash = exports.renderHtml = exports.json2HtmlTable = exports.getFirstMondayMidnightTimeAfter = exports.areLocalDatesEqual = exports.getUTCDayOfWeekForWorkingFromDate = exports.utcTimeToLocalTimeIndexFormat = exports.utcDateTimeToLocalTimeIndexFormat = exports.localDateTimeExToLocalTimeIndexFormat = exports.localDateTimeToLocalTimeIndexFormat = exports.localDateToLocalTimeIndexFormat = exports.localDateTimeToEpochTimeInMs = exports.dateToEpochTimeInSecs = exports.localDateToEpochTimeInSecs = exports.localDateToUtcDate = exports.localDateToUtcTime = exports.compareLocalDates = exports.timezoneStringToMs = exports.timezoneOffsetToString = exports.getISOStringOfNow = exports.areUtcDateTimesInSameDate = exports.MILLISECONDS_IN_A_WEEK = exports.MILLISECONDS_IN_A_DAY = exports.MILLISECONDS_IN_A_HOUR = exports.MILLISECONDS_IN_A_MINUTE = exports.SECONDS_IN_A_DAY = exports.MINUTES_IN_A_DAY = exports.getUtcStartOfDay = exports.saltHashPassword = exports.hashPasswordWithSalt = exports.genRandomString = exports.uint8ArrayToHexEncoding = exports.hexEncodingToUint8Array = exports.isUUID = exports.extendStringForUnaccent = exports.safeArray = exports.flatten = void 0;
exports.getProfileFieldsFromUserIds = exports.listenClientWrapperWithRetry = exports.getOrganizationVisitorModuleSettings = exports.checkTimeRange = exports.updateViewsAfterClone = exports.calculateNextExecutionDateOfScheduledJob = exports.momentWeekdays = exports.getSingleUserIdFromFilter = exports.upsertUserFilter = exports.generateUserFilterQuery = exports.queryForOrganizationOnlyWithVisitorModule = exports.queryForAllOrganizationSchemasPg = exports.loadSafeUserCaptions = exports.insertUpdateLogWithType = exports.insertUpdateLogToSystemStatusTable = exports.insertUpdateLog = exports.binToHex = void 0;
const crypto_1 = __importDefault(require("crypto"));
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const moment_1 = __importDefault(require("moment"));
const util_1 = require("util");
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../app.enums");
const dal_constants_1 = require("./dal.constants");
const dal_db_armon_schema_1 = require("./db/armon/dal.db.armon.schema");
const app_logs_1 = require("../app.logs");
const dal_manager_1 = require("./dal.manager");
const flatten = (arr) => arr.reduce((a, b) => a.concat(Array.isArray(b) ? (0, exports.flatten)(b) : b), []);
exports.flatten = flatten;
function safeArray(value) {
    return value ? value : [];
}
exports.safeArray = safeArray;
const diacriticsMap = [
    {
        base: "A",
        letters: "[\u0041\u24B6\uFF21\u00C0\u00C1\u00C2\u1EA6\u1EA4\u1EAA\u1EA8\u00C3\u0100\u0102\u1EB0\u1EAE\u1EB4\u1EB2\u0226\u01E0\u00C4\u01DE\u1EA2\u00C5\u01FA\u01CD\u0200\u0202\u1EA0\u1EAC\u1EB6\u1E00\u0104\u023A\u2C6F]",
    },
    { base: "AA", letters: "[\uA732]" },
    { base: "AE", letters: "[\u00C6\u01FC\u01E2]" },
    { base: "AO", letters: "[\uA734]" },
    { base: "AU", letters: "[\uA736]" },
    { base: "AV", letters: "[\uA738\uA73A]" },
    { base: "AY", letters: "[\uA73C]" },
    {
        base: "B",
        letters: "[\u0042\u24B7\uFF22\u1E02\u1E04\u1E06\u0243\u0182\u0181]",
    },
    {
        base: "C",
        letters: "[\u0043\u24B8\uFF23\u0106\u0108\u010A\u010C\u00C7\u1E08\u0187\u023B\uA73E]",
    },
    {
        base: "D",
        letters: "[\u0044\u24B9\uFF24\u1E0A\u010E\u1E0C\u1E10\u1E12\u1E0E\u0110\u018B\u018A\u0189\uA779]",
    },
    { base: "DZ", letters: "[\u01F1\u01C4]" },
    { base: "Dz", letters: "[\u01F2\u01C5]" },
    {
        base: "E",
        letters: "[\u0045\u24BA\uFF25\u00C8\u00C9\u00CA\u1EC0\u1EBE\u1EC4\u1EC2\u1EBC\u0112\u1E14\u1E16\u0114\u0116\u00CB\u1EBA\u011A\u0204\u0206\u1EB8\u1EC6\u0228\u1E1C\u0118\u1E18\u1E1A\u0190\u018E]",
    },
    { base: "F", letters: "[\u0046\u24BB\uFF26\u1E1E\u0191\uA77B]" },
    {
        base: "G",
        letters: "[\u0047\u24BC\uFF27\u01F4\u011C\u1E20\u011E\u0120\u01E6\u0122\u01E4\u0193\uA7A0\uA77D\uA77E]",
    },
    {
        base: "H",
        letters: "[\u0048\u24BD\uFF28\u0124\u1E22\u1E26\u021E\u1E24\u1E28\u1E2A\u0126\u2C67\u2C75\uA78D]",
    },
    {
        base: "I",
        letters: "[\u0049\u24BE\uFF29\u00CC\u00CD\u00CE\u0128\u012A\u012C\u0130\u00CF\u1E2E\u1EC8\u01CF\u0208\u020A\u1ECA\u012E\u1E2C\u0197]",
    },
    { base: "J", letters: "[\u004A\u24BF\uFF2A\u0134\u0248]" },
    {
        base: "K",
        letters: "[\u004B\u24C0\uFF2B\u1E30\u01E8\u1E32\u0136\u1E34\u0198\u2C69\uA740\uA742\uA744\uA7A2]",
    },
    {
        base: "L",
        letters: "[\u004C\u24C1\uFF2C\u013F\u0139\u013D\u1E36\u1E38\u013B\u1E3C\u1E3A\u0141\u023D\u2C62\u2C60\uA748\uA746\uA780]",
    },
    { base: "LJ", letters: "[\u01C7]" },
    { base: "Lj", letters: "[\u01C8]" },
    { base: "M", letters: "[\u004D\u24C2\uFF2D\u1E3E\u1E40\u1E42\u2C6E\u019C]" },
    {
        base: "N",
        letters: "[\u004E\u24C3\uFF2E\u01F8\u0143\u00D1\u1E44\u0147\u1E46\u0145\u1E4A\u1E48\u0220\u019D\uA790\uA7A4]",
    },
    { base: "NJ", letters: "[\u01CA]" },
    { base: "Nj", letters: "[\u01CB]" },
    {
        base: "O",
        letters: "[\u004F\u24C4\uFF2F\u00D2\u00D3\u00D4\u1ED2\u1ED0\u1ED6\u1ED4\u00D5\u1E4C\u022C\u1E4E\u014C\u1E50\u1E52\u014E\u022E\u0230\u00D6\u022A\u1ECE\u0150\u01D1\u020C\u020E\u01A0\u1EDC\u1EDA\u1EE0\u1EDE\u1EE2\u1ECC\u1ED8\u01EA\u01EC\u00D8\u01FE\u0186\u019F\uA74A\uA74C]",
    },
    { base: "OI", letters: "[\u01A2]" },
    { base: "OO", letters: "[\uA74E]" },
    { base: "OU", letters: "[\u0222]" },
    {
        base: "P",
        letters: "[\u0050\u24C5\uFF30\u1E54\u1E56\u01A4\u2C63\uA750\uA752\uA754]",
    },
    { base: "Q", letters: "[\u0051\u24C6\uFF31\uA756\uA758\u024A]" },
    {
        base: "R",
        letters: "[\u0052\u24C7\uFF32\u0154\u1E58\u0158\u0210\u0212\u1E5A\u1E5C\u0156\u1E5E\u024C\u2C64\uA75A\uA7A6\uA782]",
    },
    {
        base: "S",
        letters: "[\u0053\u24C8\uFF33\u1E9E\u015A\u1E64\u015C\u1E60\u0160\u1E66\u1E62\u1E68\u0218\u015E\u2C7E\uA7A8\uA784]",
    },
    {
        base: "T",
        letters: "[\u0054\u24C9\uFF34\u1E6A\u0164\u1E6C\u021A\u0162\u1E70\u1E6E\u0166\u01AC\u01AE\u023E\uA786]",
    },
    { base: "TZ", letters: "[\uA728]" },
    {
        base: "U",
        letters: "[\u0055\u24CA\uFF35\u00D9\u00DA\u00DB\u0168\u1E78\u016A\u1E7A\u016C\u00DC\u01DB\u01D7\u01D5\u01D9\u1EE6\u016E\u0170\u01D3\u0214\u0216\u01AF\u1EEA\u1EE8\u1EEE\u1EEC\u1EF0\u1EE4\u1E72\u0172\u1E76\u1E74\u0244]",
    },
    { base: "V", letters: "[\u0056\u24CB\uFF36\u1E7C\u1E7E\u01B2\uA75E\u0245]" },
    { base: "VY", letters: "[\uA760]" },
    {
        base: "W",
        letters: "[\u0057\u24CC\uFF37\u1E80\u1E82\u0174\u1E86\u1E84\u1E88\u2C72]",
    },
    { base: "X", letters: "[\u0058\u24CD\uFF38\u1E8A\u1E8C]" },
    {
        base: "Y",
        letters: "[\u0059\u24CE\uFF39\u1EF2\u00DD\u0176\u1EF8\u0232\u1E8E\u0178\u1EF6\u1EF4\u01B3\u024E\u1EFE]",
    },
    {
        base: "Z",
        letters: "[\u005A\u24CF\uFF3A\u0179\u1E90\u017B\u017D\u1E92\u1E94\u01B5\u0224\u2C7F\u2C6B\uA762]",
    },
    {
        base: "a",
        letters: "[\u0061\u24D0\uFF41\u1E9A\u00E0\u00E1\u00E2\u1EA7\u1EA5\u1EAB\u1EA9\u00E3\u0101\u0103\u1EB1\u1EAF\u1EB5\u1EB3\u0227\u01E1\u00E4\u01DF\u1EA3\u00E5\u01FB\u01CE\u0201\u0203\u1EA1\u1EAD\u1EB7\u1E01\u0105\u2C65\u0250]",
    },
    { base: "aa", letters: "[\uA733]" },
    { base: "ae", letters: "[\u00E6\u01FD\u01E3]" },
    { base: "ao", letters: "[\uA735]" },
    { base: "au", letters: "[\uA737]" },
    { base: "av", letters: "[\uA739\uA73B]" },
    { base: "ay", letters: "[\uA73D]" },
    {
        base: "b",
        letters: "[\u0062\u24D1\uFF42\u1E03\u1E05\u1E07\u0180\u0183\u0253]",
    },
    {
        base: "c",
        letters: "[\u0063\u24D2\uFF43\u0107\u0109\u010B\u010D\u00E7\u1E09\u0188\u023C\uA73F\u2184]",
    },
    {
        base: "d",
        letters: "[\u0064\u24D3\uFF44\u1E0B\u010F\u1E0D\u1E11\u1E13\u1E0F\u0111\u018C\u0256\u0257\uA77A]",
    },
    { base: "dz", letters: "[\u01F3\u01C6]" },
    {
        base: "e",
        letters: "[\u0065\u24D4\uFF45\u00E8\u00E9\u00EA\u1EC1\u1EBF\u1EC5\u1EC3\u1EBD\u0113\u1E15\u1E17\u0115\u0117\u00EB\u1EBB\u011B\u0205\u0207\u1EB9\u1EC7\u0229\u1E1D\u0119\u1E19\u1E1B\u0247\u025B\u01DD]",
    },
    { base: "f", letters: "[\u0066\u24D5\uFF46\u1E1F\u0192\uA77C]" },
    {
        base: "g",
        letters: "[\u0067\u24D6\uFF47\u01F5\u011D\u1E21\u011F\u0121\u01E7\u0123\u01E5\u0260\uA7A1\u1D79\uA77F]",
    },
    {
        base: "h",
        letters: "[\u0068\u24D7\uFF48\u0125\u1E23\u1E27\u021F\u1E25\u1E29\u1E2B\u1E96\u0127\u2C68\u2C76\u0265]",
    },
    { base: "hv", letters: "[\u0195]" },
    {
        base: "i",
        letters: "[\u0069\u24D8\uFF49\u00EC\u00ED\u00EE\u0129\u012B\u012D\u00EF\u1E2F\u1EC9\u01D0\u0209\u020B\u1ECB\u012F\u1E2D\u0268\u0131]",
    },
    { base: "j", letters: "[\u006A\u24D9\uFF4A\u0135\u01F0\u0249]" },
    {
        base: "k",
        letters: "[\u006B\u24DA\uFF4B\u1E31\u01E9\u1E33\u0137\u1E35\u0199\u2C6A\uA741\uA743\uA745\uA7A3]",
    },
    {
        base: "l",
        letters: "[\u006C\u24DB\uFF4C\u0140\u013A\u013E\u1E37\u1E39\u013C\u1E3D\u1E3B\u017F\u0142\u019A\u026B\u2C61\uA749\uA781\uA747]",
    },
    { base: "lj", letters: "[\u01C9]" },
    { base: "m", letters: "[\u006D\u24DC\uFF4D\u1E3F\u1E41\u1E43\u0271\u026F]" },
    {
        base: "n",
        letters: "[\u006E\u24DD\uFF4E\u01F9\u0144\u00F1\u1E45\u0148\u1E47\u0146\u1E4B\u1E49\u019E\u0272\u0149\uA791\uA7A5]",
    },
    { base: "nj", letters: "[\u01CC]" },
    {
        base: "o",
        letters: "[\u006F\u24DE\uFF4F\u00F2\u00F3\u00F4\u1ED3\u1ED1\u1ED7\u1ED5\u00F5\u1E4D\u022D\u1E4F\u014D\u1E51\u1E53\u014F\u022F\u0231\u00F6\u022B\u1ECF\u0151\u01D2\u020D\u020F\u01A1\u1EDD\u1EDB\u1EE1\u1EDF\u1EE3\u1ECD\u1ED9\u01EB\u01ED\u00F8\u01FF\u0254\uA74B\uA74D\u0275]",
    },
    { base: "oi", letters: "[\u01A3]" },
    { base: "ou", letters: "[\u0223]" },
    { base: "oo", letters: "[\uA74F]" },
    {
        base: "p",
        letters: "[\u0070\u24DF\uFF50\u1E55\u1E57\u01A5\u1D7D\uA751\uA753\uA755]",
    },
    { base: "q", letters: "[\u0071\u24E0\uFF51\u024B\uA757\uA759]" },
    {
        base: "r",
        letters: "[\u0072\u24E1\uFF52\u0155\u1E59\u0159\u0211\u0213\u1E5B\u1E5D\u0157\u1E5F\u024D\u027D\uA75B\uA7A7\uA783]",
    },
    {
        base: "s",
        letters: "[\u0073\u24E2\uFF53\u00DF\u015B\u1E65\u015D\u1E61\u0161\u1E67\u1E63\u1E69\u0219\u015F\u023F\uA7A9\uA785\u1E9B]",
    },
    {
        base: "t",
        letters: "[\u0074\u24E3\uFF54\u1E6B\u1E97\u0165\u1E6D\u021B\u0163\u1E71\u1E6F\u0167\u01AD\u0288\u2C66\uA787]",
    },
    { base: "tz", letters: "[\uA729]" },
    {
        base: "u",
        letters: "[\u0075\u24E4\uFF55\u00F9\u00FA\u00FB\u0169\u1E79\u016B\u1E7B\u016D\u00FC\u01DC\u01D8\u01D6\u01DA\u1EE7\u016F\u0171\u01D4\u0215\u0217\u01B0\u1EEB\u1EE9\u1EEF\u1EED\u1EF1\u1EE5\u1E73\u0173\u1E77\u1E75\u0289]",
    },
    { base: "v", letters: "[\u0076\u24E5\uFF56\u1E7D\u1E7F\u028B\uA75F\u028C]" },
    { base: "vy", letters: "[\uA761]" },
    {
        base: "w",
        letters: "[\u0077\u24E6\uFF57\u1E81\u1E83\u0175\u1E87\u1E85\u1E98\u1E89\u2C73]",
    },
    { base: "x", letters: "[\u0078\u24E7\uFF58\u1E8B\u1E8D]" },
    {
        base: "y",
        letters: "[\u0079\u24E8\uFF59\u1EF3\u00FD\u0177\u1EF9\u0233\u1E8F\u00FF\u1EF7\u1E99\u1EF5\u01B4\u024F\u1EFF]",
    },
    {
        base: "z",
        letters: "[\u007A\u24E9\uFF5A\u017A\u1E91\u017C\u017E\u1E93\u1E95\u01B6\u0225\u0240\u2C6C\uA763]",
    },
];
const diacriticsMapUtf8 = [
    { base: "A", letters: "[AⒶＡÀÁÂẦẤẪẨÃĀĂẰẮẴẲȦǠÄǞẢÅǺǍȀȂẠẬẶḀĄȺⱯ]" },
    { base: "AA", letters: "[Ꜳ]" },
    { base: "AE", letters: "[ÆǼǢ]" },
    { base: "AO", letters: "[Ꜵ]" },
    { base: "AU", letters: "[Ꜷ]" },
    { base: "AV", letters: "[ꜸꜺ]" },
    { base: "AY", letters: "[Ꜽ]" },
    { base: "B", letters: "[BⒷＢḂḄḆɃƂƁ]" },
    { base: "C", letters: "[CⒸＣĆĈĊČÇḈƇȻꜾ]" },
    { base: "D", letters: "[DⒹＤḊĎḌḐḒḎĐƋƊƉꝹ]" },
    { base: "DZ", letters: "[ǱǄ]" },
    { base: "Dz", letters: "[ǲǅ]" },
    { base: "E", letters: "[EⒺＥÈÉÊỀẾỄỂẼĒḔḖĔĖËẺĚȄȆẸỆȨḜĘḘḚƐƎ]" },
    { base: "F", letters: "[FⒻＦḞƑꝻ]" },
    { base: "G", letters: "[GⒼＧǴĜḠĞĠǦĢǤƓꞠꝽꝾ]" },
    { base: "H", letters: "[HⒽＨĤḢḦȞḤḨḪĦⱧⱵꞍ]" },
    { base: "I", letters: "[IⒾＩÌÍÎĨĪĬİÏḮỈǏȈȊỊĮḬƗ]" },
    { base: "J", letters: "[JⒿＪĴɈ]" },
    { base: "K", letters: "[KⓀＫḰǨḲĶḴƘⱩꝀꝂꝄꞢ]" },
    { base: "L", letters: "[LⓁＬĿĹĽḶḸĻḼḺŁȽⱢⱠꝈꝆꞀ]" },
    { base: "LJ", letters: "[Ǉ]" },
    { base: "Lj", letters: "[ǈ]" },
    { base: "M", letters: "[MⓂＭḾṀṂⱮƜ]" },
    { base: "N", letters: "[NⓃＮǸŃÑṄŇṆŅṊṈȠƝꞐꞤ]" },
    { base: "NJ", letters: "[Ǌ]" },
    { base: "Nj", letters: "[ǋ]" },
    { base: "O", letters: "[OⓄＯÒÓÔỒỐỖỔÕṌȬṎŌṐṒŎȮȰÖȪỎŐǑȌȎƠỜỚỠỞỢỌỘǪǬØǾƆƟꝊꝌ]" },
    { base: "OI", letters: "[Ƣ]" },
    { base: "OO", letters: "[Ꝏ]" },
    { base: "OU", letters: "[Ȣ]" },
    { base: "P", letters: "[PⓅＰṔṖƤⱣꝐꝒꝔ]" },
    { base: "Q", letters: "[QⓆＱꝖꝘɊ]" },
    { base: "R", letters: "[RⓇＲŔṘŘȐȒṚṜŖṞɌⱤꝚꞦꞂ]" },
    { base: "S", letters: "[SⓈＳẞŚṤŜṠŠṦṢṨȘŞⱾꞨꞄ]" },
    { base: "T", letters: "[TⓉＴṪŤṬȚŢṰṮŦƬƮȾꞆ]" },
    { base: "TZ", letters: "[Ꜩ]" },
    { base: "U", letters: "[UⓊＵÙÚÛŨṸŪṺŬÜǛǗǕǙỦŮŰǓȔȖƯỪỨỮỬỰỤṲŲṶṴɄ]" },
    { base: "V", letters: "[VⓋＶṼṾƲꝞɅ]" },
    { base: "VY", letters: "[Ꝡ]" },
    { base: "W", letters: "[WⓌＷẀẂŴẆẄẈⱲ]" },
    { base: "X", letters: "[XⓍＸẊẌ]" },
    { base: "Y", letters: "[YⓎＹỲÝŶỸȲẎŸỶỴƳɎỾ]" },
    { base: "Z", letters: "[ZⓏＺŹẐŻŽẒẔƵȤⱿⱫꝢ]" },
    { base: "a", letters: "[aⓐａẚàáâầấẫẩãāăằắẵẳȧǡäǟảåǻǎȁȃạậặḁąⱥɐ]" },
    { base: "aa", letters: "[Ꜷ]" },
    { base: "ae", letters: "[æǽǣ]" },
    { base: "ao", letters: "[ꜵ]" },
    { base: "au", letters: "[ꜷ]" },
    { base: "av", letters: "[ꜹꜻ]" },
    { base: "ay", letters: "[ꜽ]" },
    { base: "b", letters: "[bⓑｂḃḅḇƀƃɓ]" },
    { base: "c", letters: "[cⓒｃćĉċčçḉƈȼꜿↄ]" },
    { base: "d", letters: "[dⓓｄḋďḍḑḓḏđƌɖɗꝺ]" },
    { base: "dz", letters: "[ǳǆ]" },
    { base: "e", letters: "[eⓔｅèéêềếễểẽēḕḗĕėëẻěȅȇẹệȩḝęḙḛɇɛǝ]" },
    { base: "f", letters: "[fⓕｆḟƒꝼ]" },
    { base: "g", letters: "[gⓖｇǵĝḡğġǧģǥɠꞡᵹꝿ]" },
    { base: "h", letters: "[hⓗｈĥḣḧȟḥḩḫẖħⱨⱶɥ]" },
    { base: "hv", letters: "[ƕ]" },
    { base: "i", letters: "[iⓘｉìíîĩīĭïḯỉǐȉȋịįḭɨı]" },
    { base: "j", letters: "[jⓙｊĵǰɉ]" },
    { base: "k", letters: "[kⓚｋḱǩḳķḵƙⱪꝁꝃꝅꞣ]" },
    { base: "l", letters: "[lⓛｌŀĺľḷḹļḽḻſłƚɫⱡꝉꞁꝇ]" },
    { base: "lj", letters: "[ǉ]" },
    { base: "m", letters: "[mⓜｍḿṁṃɱɯ]" },
    { base: "n", letters: "[nⓝｎǹńñṅňṇņṋṉƞɲŉꞑꞥ]" },
    { base: "nj", letters: "[ǌ]" },
    { base: "o", letters: "[oⓞｏòóôồốỗổõṍȭṏōṑṓŏȯȱöȫỏőǒȍȏơờớỡởợọộǫǭøǿɔꝋꝍɵ]" },
    { base: "oi", letters: "[ƣ]" },
    { base: "ou", letters: "[ȣ]" },
    { base: "oo", letters: "[ꝏ]" },
    { base: "p", letters: "[pⓟｐṕṗƥᵽꝑꝓꝕ]" },
    { base: "q", letters: "[qⓠｑɋꝗꝙ]" },
    { base: "r", letters: "[rⓡｒŕṙřȑȓṛṝŗṟɍɽꝛꞧꞃ]" },
    { base: "s", letters: "[sⓢｓßśṥŝṡšṧṣṩșşȿꞩꞅẛ]" },
    { base: "t", letters: "[tⓣｔṫẗťṭțţṱṯŧƭʈⱦꞇ]" },
    { base: "tz", letters: "[ꜩ]" },
    { base: "u", letters: "[uⓤｕùúûũṹūṻŭüǜǘǖǚủůűǔȕȗưừứữửựụṳųṷṵʉ]" },
    { base: "v", letters: "[vⓥｖṽṿʋꝟʌ]" },
    { base: "vy", letters: "[ꝡ]" },
    { base: "w", letters: "[wⓦｗẁẃŵẇẅẘẉⱳ]" },
    { base: "x", letters: "[xⓧｘẋẍ]" },
    { base: "y", letters: "[yⓨｙỳýŷỹȳẏÿỷẙỵƴɏỿ]" },
    { base: "z", letters: "[zⓩｚźẑżžẓẕƶȥɀⱬꝣ]" },
];
function extendStringForUnaccent(s) {
    s = lodash_1.default.deburr(s).toUpperCase();
    for (var i = 0; i < diacriticsMap.length; i++) {
        s = s.replace(diacriticsMap[i].base, diacriticsMap[i].letters);
    }
    return s;
}
exports.extendStringForUnaccent = extendStringForUnaccent;
const uuidPattern = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-4][0-9a-f]{3}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
const regexHHmmss = new RegExp(/([0-1][0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9])/);
function isUUID(input) {
    if (!uuidPattern.test(input)) {
        return false;
    }
    return true;
}
exports.isUUID = isUUID;
function hexEncodingToUint8Array(hex) {
    if (!hex) {
        return new Uint8Array(0);
    }
    let size = hex.length / 2;
    let result = new Uint8Array(size);
    for (let i = 0; i < size; i++) {
        result[i] = parseInt(hex[i * 2] + hex[i * 2 + 1], 16);
    }
    return result;
}
exports.hexEncodingToUint8Array = hexEncodingToUint8Array;
function uint8ArrayToHexEncoding(arr) {
    let result = "";
    for (let i = 0; i < arr.length; i++) {
        result += (arr[i] < 16 ? "0" : "") + arr[i].toString(16);
    }
    return result;
}
exports.uint8ArrayToHexEncoding = uint8ArrayToHexEncoding;
function genRandomString(length) {
    return crypto_1.default
        .randomBytes(Math.ceil(length / 2))
        .toString("hex")
        .slice(0, length);
}
exports.genRandomString = genRandomString;
function hashPasswordWithSalt(password, salt) {
    let hash = crypto_1.default.createHmac("sha512", salt);
    hash.update(password);
    let value = hash.digest("hex");
    return {
        salt: salt,
        passwordHash: value,
    };
}
exports.hashPasswordWithSalt = hashPasswordWithSalt;
function saltHashPassword(userpassword) {
    return hashPasswordWithSalt(userpassword, genRandomString(32));
}
exports.saltHashPassword = saltHashPassword;
function getUtcStartOfDay(date) {
    return new Date(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate());
}
exports.getUtcStartOfDay = getUtcStartOfDay;
exports.MINUTES_IN_A_DAY = 24 * 60;
exports.SECONDS_IN_A_DAY = exports.MINUTES_IN_A_DAY * 60;
exports.MILLISECONDS_IN_A_MINUTE = 60 * 1000;
exports.MILLISECONDS_IN_A_HOUR = 60 * exports.MILLISECONDS_IN_A_MINUTE;
exports.MILLISECONDS_IN_A_DAY = exports.SECONDS_IN_A_DAY * 1000;
exports.MILLISECONDS_IN_A_WEEK = exports.MILLISECONDS_IN_A_DAY * 7;
function areUtcDateTimesInSameDate(d1, d2) {
    return Math.floor(d1.getTime() / exports.MILLISECONDS_IN_A_DAY) == Math.floor(d2.getTime() / exports.MILLISECONDS_IN_A_DAY);
}
exports.areUtcDateTimesInSameDate = areUtcDateTimesInSameDate;
function getISOStringOfNow() {
    let now = new Date();
    let timezoneOffset = now.getTimezoneOffset();
    now.setMinutes(now.getMinutes() - timezoneOffset);
    let iso = now.toISOString();
    return iso.substr(0, iso.length - 1) + timezoneOffsetToString(timezoneOffset);
}
exports.getISOStringOfNow = getISOStringOfNow;
function timezoneOffsetToString(offset) {
    let abs = Math.abs(offset);
    return (offset > 0 ? "-" : "+") + Math.floor(abs / 600) + (Math.floor(abs / 60) % 10) + ":" + Math.floor((abs % 60) / 10) + ((abs % 60) % 10);
}
exports.timezoneOffsetToString = timezoneOffsetToString;
function timezoneStringToMs(tz) {
    let s = tz.split(":");
    let h = parseInt(s[0]);
    return (h * 60 + parseInt(s[1]) * (h < 0 ? -1 : 1)) * exports.MILLISECONDS_IN_A_MINUTE;
}
exports.timezoneStringToMs = timezoneStringToMs;
function compareLocalDates(d1, d2) {
    return d1.year < d2.year ? -1 : d1.year > d2.year ? 1 : d1.month < d2.month ? -1 : d1.month > d2.month ? 1 : d1.day < d2.day ? -1 : d1.day > d2.day ? 1 : 0;
}
exports.compareLocalDates = compareLocalDates;
function localDateToUtcTime(d) {
    return localDateToUtcDate(d).getTime();
}
exports.localDateToUtcTime = localDateToUtcTime;
function localDateToUtcDate(d) {
    return new Date(`${localDateToISOString(d)}Z`);
}
exports.localDateToUtcDate = localDateToUtcDate;
function localDateToEpochTimeInSecs(d) {
    return localDateToUtcDate(d).getTime() / 1000;
}
exports.localDateToEpochTimeInSecs = localDateToEpochTimeInSecs;
function dateToEpochTimeInSecs(d) {
    return d.getTime() / 1000;
}
exports.dateToEpochTimeInSecs = dateToEpochTimeInSecs;
function localDateToISOString(d) {
    return `${d.year}-${d.month > 9 ? d.month : "0" + d.month}-${d.day > 9 ? d.day : "0" + d.day}`;
}
function localDateTimeToEpochTimeInMs(d, hours, minutes, seconds) {
    return new Date(`${localDateToISOString(d)}T${hours > 9 ? hours : "0" + hours}:${minutes > 9 ? minutes : "0" + minutes}:${seconds > 9 ? seconds : "0" + seconds}Z`).getTime();
}
exports.localDateTimeToEpochTimeInMs = localDateTimeToEpochTimeInMs;
function localDateToLocalTimeIndexFormat(d) {
    return (d.year * 10000 + d.month * 100 + d.day) * 1000000;
}
exports.localDateToLocalTimeIndexFormat = localDateToLocalTimeIndexFormat;
function localDateTimeToLocalTimeIndexFormat(d, hours, minutes, seconds) {
    return d.year * 10000000000 + d.month * 100000000 + d.day * 1000000 + hours * 10000 + minutes * 100 + seconds;
}
exports.localDateTimeToLocalTimeIndexFormat = localDateTimeToLocalTimeIndexFormat;
function localDateTimeExToLocalTimeIndexFormat(year, month, day, hours, minutes, seconds) {
    return year * 10000000000 + month * 100000000 + day * 1000000 + hours * 10000 + minutes * 100 + seconds;
}
exports.localDateTimeExToLocalTimeIndexFormat = localDateTimeExToLocalTimeIndexFormat;
function utcDateTimeToLocalTimeIndexFormat(utcDate, hours, minutes, seconds) {
    return utcDate.getUTCFullYear() * 10000000000 + (utcDate.getUTCMonth() + 1) * 100000000 + utcDate.getUTCDate() * 1000000 + hours * 10000 + minutes * 100 + seconds;
}
exports.utcDateTimeToLocalTimeIndexFormat = utcDateTimeToLocalTimeIndexFormat;
function utcTimeToLocalTimeIndexFormat(time) {
    let date = new Date(time);
    return (date.getUTCFullYear() * 10000000000 +
        (date.getUTCMonth() + 1) * 100000000 +
        date.getUTCDate() * 1000000 +
        date.getUTCHours() * 10000 +
        date.getUTCMinutes() * 100 +
        date.getUTCSeconds());
}
exports.utcTimeToLocalTimeIndexFormat = utcTimeToLocalTimeIndexFormat;
function getUTCDayOfWeekForWorkingFromDate(d) {
    let day = d.getUTCDay();
    return day === 0 ? 1 : day;
}
exports.getUTCDayOfWeekForWorkingFromDate = getUTCDayOfWeekForWorkingFromDate;
function areLocalDatesEqual(d1, d2) {
    return d1.day === d2.day && d1.month === d2.month && d1.year === d2.year;
}
exports.areLocalDatesEqual = areLocalDatesEqual;
function getFirstMondayMidnightTimeAfter(d) {
    let dutc = localDateToUtcDate(d);
    let isAlreadyMonday = dutc.getUTCDay() === dal_constants_1.DalConstants.DayOfWeek.Monday;
    if (!isAlreadyMonday) {
        while (dutc.getUTCDay() !== dal_constants_1.DalConstants.DayOfWeek.Monday) {
            dutc = new Date(dutc.setDate(dutc.getDate() + 1));
        }
    }
    return {
        isAlreadyMonday: isAlreadyMonday,
        midNightTime: dutc.getTime(),
    };
}
exports.getFirstMondayMidnightTimeAfter = getFirstMondayMidnightTimeAfter;
const urlRegExp = new RegExp(/(\b(https?|ftp|file):\/\/[-A-Z0-9+&@#\/%?=~_|!:,.;]*[-A-Z0-9+&@#\/%=~_|])/gi);
const javascriptRegExp = new RegExp(/(^javascript:[\s\S]*;$)/gi);
function json2HtmlTable(model, data) {
    return `
        <table border="1" cellpadding="1" cellspacing="1">
            ${generateHeader(model)}
            ${generateBody(model, data)}
        </table>
    `;
}
exports.json2HtmlTable = json2HtmlTable;
function generateHeader(model) {
    let columns = "";
    for (let i = 0; i < model.columns.length; i++) {
        columns += `<th>${model.columns[i].th}</th>`;
    }
    return `
        <thead>${columns}</thead>
    `;
}
function generateBody(model, data) {
    if (data.length === 0) {
        return "";
    }
    let rows = "";
    for (let i = 0; i < data.length; i++) {
        let dataRow = data[i];
        let row = "";
        for (let k = 0; k < model.columns.length; k++) {
            let column = model.columns[k];
            let keyPath = column.contextKey.split(".");
            let value = dataRow;
            for (let contextKeyIndex = 0; contextKeyIndex < keyPath.length; contextKeyIndex++) {
                let contextKeyItem = keyPath[contextKeyIndex];
                value = value[contextKeyItem];
                if (!value) {
                    value = "";
                    break;
                }
            }
            if (column.fn) {
                value = column.fn(value);
            }
            let isUrl = urlRegExp.test(value) || javascriptRegExp.test(value);
            if (isUrl) {
                row += `<td><a href="${value}">${value}</a></td>`;
            }
            else {
                row += `<td>${value}</td>`;
            }
        }
        rows += `<tr ${i % 2 === 0 ? 'class="alt"' : ""}>${row}</tr>`;
    }
    return `
        <tbody>${rows}</tbody>
    `;
}
function renderHtml(template, data) {
    let keys = Object.keys(data);
    for (let index = 0; index < keys.length; index++) {
        let key = keys[index];
        let regex = new RegExp("{{" + key + "}}", "ig");
        template = template.replace(regex, data[key]);
    }
    return template;
}
exports.renderHtml = renderHtml;
function generateHash(value) {
    var h = 5831 << 2;
    var i = 0;
    for (i = 0; i < value.length; i++) {
        var ascii = value.charCodeAt(i);
        h = (h << 3) ^ h ^ ascii;
    }
    return (h & 0xffffffffff).toString(16);
}
exports.generateHash = generateHash;
function undefinedToNull(obj) {
    let key;
    for (key in obj) {
        let value = obj[key];
        if (value === undefined) {
            obj[key] = null;
        }
        else if ((0, util_1.isObject)(value)) {
            obj[key] = undefinedToNull(value);
        }
    }
    return obj;
}
exports.undefinedToNull = undefinedToNull;
function authenticationFactorToCredentialTypes(authFactors) {
    switch (authFactors) {
        case dal_constants_1.DalConstants.AuthenticationFactor.AccessToken:
            return [dal_constants_1.DalConstants.CredentialType.AccessToken];
        case dal_constants_1.DalConstants.AuthenticationFactor.Bluetooth:
            return [dal_constants_1.DalConstants.CredentialType.BLE];
        case dal_constants_1.DalConstants.AuthenticationFactor.FingerPrint:
            return [dal_constants_1.DalConstants.CredentialType.FingerPrintISO19794];
        case dal_constants_1.DalConstants.AuthenticationFactor.MiFare:
            return [dal_constants_1.DalConstants.CredentialType.MiFare];
        case dal_constants_1.DalConstants.AuthenticationFactor.NFC:
            return [dal_constants_1.DalConstants.CredentialType.NFC];
        case dal_constants_1.DalConstants.AuthenticationFactor.ProximityCard:
            return [dal_constants_1.DalConstants.CredentialType.ProximityCard];
        case dal_constants_1.DalConstants.AuthenticationFactor.UHFRfid:
            return [dal_constants_1.DalConstants.CredentialType.UHFRfid];
        case dal_constants_1.DalConstants.AuthenticationFactor.VehiclePlate:
            return [dal_constants_1.DalConstants.CredentialType.VehiclePlate];
        case dal_constants_1.DalConstants.AuthenticationFactor.MRZ:
            return [dal_constants_1.DalConstants.CredentialType.MRZ];
        case dal_constants_1.DalConstants.AuthenticationFactor.QR_BarCode:
            return [dal_constants_1.DalConstants.CredentialType.QrCode];
        default:
            return [];
    }
}
exports.authenticationFactorToCredentialTypes = authenticationFactorToCredentialTypes;
function credentialTypeToAuthenticationFactor(type) {
    switch (type) {
        case dal_constants_1.DalConstants.CredentialType.AccessToken:
            return dal_constants_1.DalConstants.AuthenticationFactor.AccessToken;
        case dal_constants_1.DalConstants.CredentialType.BLE:
            return dal_constants_1.DalConstants.AuthenticationFactor.Bluetooth;
        case dal_constants_1.DalConstants.CredentialType.FingerPrintISO19794:
            return dal_constants_1.DalConstants.AuthenticationFactor.FingerPrint;
        case dal_constants_1.DalConstants.CredentialType.MiFare:
            return dal_constants_1.DalConstants.AuthenticationFactor.MiFare;
        case dal_constants_1.DalConstants.CredentialType.NFC:
            return dal_constants_1.DalConstants.AuthenticationFactor.NFC;
        case dal_constants_1.DalConstants.CredentialType.ProximityCard:
            return dal_constants_1.DalConstants.AuthenticationFactor.ProximityCard;
        case dal_constants_1.DalConstants.CredentialType.UHFRfid:
            return dal_constants_1.DalConstants.AuthenticationFactor.UHFRfid;
        case dal_constants_1.DalConstants.CredentialType.VehiclePlate:
            return dal_constants_1.DalConstants.AuthenticationFactor.VehiclePlate;
        case dal_constants_1.DalConstants.CredentialType.MRZ:
            return dal_constants_1.DalConstants.AuthenticationFactor.MRZ;
        case dal_constants_1.DalConstants.CredentialType.QrCode:
            return dal_constants_1.DalConstants.AuthenticationFactor.QR_BarCode;
        default:
            break;
    }
}
exports.credentialTypeToAuthenticationFactor = credentialTypeToAuthenticationFactor;
const createCredentialData = (decimalData, type) => {
    if (decimalData) {
        let hexData = (0, exports.decToHex)(decimalData.trim()).toString();
        return (0, exports.pad)(hexData, type == dal_constants_1.DalConstants.CredentialType.UHFRfid ? 8 : 4, "0").toLowerCase();
    }
    else {
        return decimalData;
    }
};
exports.createCredentialData = createCredentialData;
const checkBin = (n) => {
    return /^[01]{1,64}$/.test(n);
};
exports.checkBin = checkBin;
const checkDec = (n) => {
    return /^[0-9]{1,64}$/.test(n);
};
exports.checkDec = checkDec;
const checkHex = (n) => {
    return /^[0-9A-Fa-f]{1,64}$/.test(n);
};
exports.checkHex = checkHex;
const pad = (n, width, z) => {
    z = z || "0";
    n = n + "";
    return n.length >= width ? n : new Array(width - n.length + 1).join(z) + n;
};
exports.pad = pad;
const hexToBin = (hexData) => {
    if (!(0, exports.checkHex)(hexData)) {
        return "0";
    }
    return parseInt(hexData, 16).toString(2);
};
exports.hexToBin = hexToBin;
const hexToDec = (hexData) => {
    if (!(0, exports.checkHex)(hexData)) {
        return "0";
    }
    return convertBase(hexData.toLowerCase(), 16, 10);
};
exports.hexToDec = hexToDec;
const decToBin = (decData) => {
    if (!(0, exports.checkDec)(decData) || decData < 0) {
        return "0";
    }
    return parseInt(decData).toString(2);
};
exports.decToBin = decToBin;
const decToHex = (decStr) => {
    if (!(0, exports.checkDec)(decStr)) {
        return "0";
    }
    let hex = convertBase(decStr, 10, 16);
    return hex ?? null;
};
exports.decToHex = decToHex;
const binToDec = (binData) => {
    if (!(0, exports.checkBin)(binData)) {
        return "0";
    }
    return parseInt(binData, 2).toString(10);
};
exports.binToDec = binToDec;
const binToHex = (binData) => {
    if (!(0, exports.checkBin)(binData)) {
        return "0";
    }
    return parseInt(binData, 2).toString(16);
};
exports.binToHex = binToHex;
const convertBase = (str, fromBase, toBase) => {
    let digits = parseToDigitsArray(str, fromBase);
    if (digits === null || digits.length === 0) {
        return null;
    }
    let outArray = [];
    let power = [1];
    for (let i = 0; i < digits.length; i++) {
        if (digits[i]) {
            outArray = conversionHelperAdd(outArray, multiplyByNumber(digits[i], power, toBase), toBase);
        }
        power = multiplyByNumber(fromBase, power, toBase);
    }
    let outString = "";
    for (let i = outArray.length - 1; i >= 0; i--) {
        outString += outArray[i].toString(toBase);
    }
    if (outString === "") {
        outString = "0";
    }
    return outString;
};
const multiplyByNumber = (decimalDigit, powerArray, base) => {
    if (decimalDigit < 0) {
        return null;
    }
    if (decimalDigit == 0) {
        return [];
    }
    let result = [];
    let power = powerArray;
    while (true) {
        if (decimalDigit & 1) {
            result = conversionHelperAdd(result, power, base);
        }
        decimalDigit = decimalDigit >> 1;
        if (decimalDigit === 0) {
            break;
        }
        power = conversionHelperAdd(power, power, base);
    }
    return result;
};
const parseToDigitsArray = (decimalString, base) => {
    let decimalDigits = decimalString.split("");
    let numberValues = [];
    for (let i = decimalDigits.length - 1; i >= 0; i--) {
        let n = parseInt(decimalDigits[i], base);
        if (isNaN(n)) {
            return null;
        }
        numberValues.push(n);
    }
    return numberValues;
};
const conversionHelperAdd = (x, y, base) => {
    let result = [];
    let n = Math.max(x.length, y.length);
    let carry = 0;
    let i = 0;
    while (i < n || carry) {
        let xi = i < x.length ? x[i] : 0;
        let yi = i < y.length ? y[i] : 0;
        let zi = carry + xi + yi;
        result.push(zi % base);
        carry = Math.floor(zi / base);
        i++;
    }
    return result;
};
async function insertUpdateLog(organizationId, knex, table, data) {
    let minDate = lodash_1.default.minBy(data, (d) => new Date(d.actionUtc).getTime()).actionUtc;
    let maxDate = lodash_1.default.maxBy(data, (d) => new Date(d.actionUtc).getTime()).actionUtc;
    minDate = (0, moment_1.default)(minDate).add(-1, "hour").toDate();
    maxDate = (0, moment_1.default)(maxDate).add(1, "hour").toDate();
    let exists = await knex
        .withSchema(organizationId)
        .whereIn("id", data.map((d) => d.id))
        .from(table)
        .select("id");
    await knex
        .withSchema(organizationId)
        .from(table)
        .insert(data.filter((d) => !exists.some((e) => e.id == d.id)));
}
exports.insertUpdateLog = insertUpdateLog;
async function insertUpdateLogToSystemStatusTable(params) {
    let minDate = lodash_1.default.minBy(params.data, (d) => new Date(d.actionUtc).getTime()).actionUtc;
    let maxDate = lodash_1.default.maxBy(params.data, (d) => new Date(d.actionUtc).getTime()).actionUtc;
    minDate = (0, moment_1.default)(minDate).add(-1, "minute").toDate();
    maxDate = (0, moment_1.default)(maxDate).add(1, "minute").toDate();
    const exists = await params.trx.query(`
		SELECT id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs}"
		WHERE "actionUtc" <= $1
		AND "actionUtc" >= $2
		AND id = ANY($3::uuid[])
		AND type = $4
	`, [maxDate, minDate, params.data.map((d) => d.id), params.type]);
    const qi = [];
    const qb = [];
    let qx = 1;
    let q = `INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs}" 
	(id, "actionUtc", type, log) `;
    for (const log of params.data.filter((d) => !exists.rows.some((e) => e.id === d.id))) {
        qi.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++})`);
        qb.push(log.id, log.actionUtc, log.type, log.log);
    }
    q = q + " VALUES " + qi.join(" , ");
    if (params.data.filter((d) => !exists.rows.some((e) => e.id === d.id)).length === 0) {
        return;
    }
    await params.trx.query(q, qb);
}
exports.insertUpdateLogToSystemStatusTable = insertUpdateLogToSystemStatusTable;
async function insertUpdateLogWithType(organizationId, knex, table, data, type) {
    let minDate = lodash_1.default.minBy(data, (d) => new Date(d.actionUtc).getTime()).actionUtc;
    let maxDate = lodash_1.default.maxBy(data, (d) => new Date(d.actionUtc).getTime()).actionUtc;
    minDate = (0, moment_1.default)(minDate).add(-1, "minute").toDate();
    maxDate = (0, moment_1.default)(maxDate).add(1, "minute").toDate();
    let exists = await knex
        .withSchema(organizationId)
        .where("actionUtc", "<=", maxDate)
        .where("actionUtc", ">=", minDate)
        .whereIn("id", data.map((d) => d.id))
        .where("type", type)
        .from(table)
        .select("id");
    await knex
        .withSchema(organizationId)
        .from(table)
        .insert(data.filter((d) => !exists.some((e) => e.id == d.id)));
}
exports.insertUpdateLogWithType = insertUpdateLogWithType;
function loadSafeUserCaptions(userId, userCaptionsList) {
    let targetCaptions = userCaptionsList.find((u) => u.id == userId);
    return targetCaptions ? targetCaptions.captionLines : [];
}
exports.loadSafeUserCaptions = loadSafeUserCaptions;
async function queryForAllOrganizationSchemasPg(trx, query, schemaNameReplace) {
    const organizationIds = (await trx.query(` SELECT id FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}";`)).rows.map((r) => r.id);
    for (const organizationId of organizationIds) {
        const orgQuery = query.split(schemaNameReplace).join(`${organizationId}`);
        await trx.query(orgQuery);
    }
    const tempQuery = query
        .split('"' + schemaNameReplace + '"')
        .join(schemaNameReplace)
        .split(schemaNameReplace)
        .join(`organization`);
    await trx.query(tempQuery);
}
exports.queryForAllOrganizationSchemasPg = queryForAllOrganizationSchemasPg;
async function queryForOrganizationOnlyWithVisitorModule(trx, query, schemaNameReplace, values) {
    const organizationIds = (await trx.query(` SELECT id FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}";`)).rows.map((r) => r.id);
    for (const organizationId of organizationIds) {
        const orgModules = await trx.query(`SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules}"`);
        const visitorModule = orgModules.rows.find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.Visitor);
        if (!visitorModule) {
            continue;
        }
        await trx.query(query.split("___ORGANIZATION_ID___").join(`${organizationId}`), values);
    }
}
exports.queryForOrganizationOnlyWithVisitorModule = queryForOrganizationOnlyWithVisitorModule;
function generateUserFilterQuery(organizationId, filterId, bindingKeys) {
    bindingKeys.push(filterId);
    const bindIndex = bindingKeys.length;
    const query = `
	SELECT "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.functionNames.userfilterresult}"(
		(SELECT array_agg((id, combinator, "allIncluded")::"${organizationId}".filter_info)
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
		WHERE id = $${bindIndex}),
		(SELECT array_agg(ufu)
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUser}" ufu
		WHERE "filterId" = $${bindIndex}),
		(SELECT array_agg(ufg)
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}" ufg
		WHERE "filterId" = $${bindIndex}),
		(SELECT array_agg(ufor)
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterOrganizationRole}" ufor
		WHERE "filterId" = $${bindIndex}),
		(SELECT array_agg(ufu)
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnit}" ufu
		WHERE "filterId" = $${bindIndex}),
		(SELECT array_agg(ufur)
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnitRole}" ufur
		WHERE "filterId" = $${bindIndex}),
		(SELECT array_agg(ufwp)
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}" ufwp
		WHERE "filterId" = $${bindIndex})
	)`;
    return query;
}
exports.generateUserFilterQuery = generateUserFilterQuery;
async function upsertUserFilter(organizationId, filter, trx) {
    if (filter.id) {
        await trx.query(`DELETE FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUser}
			WHERE "filterId" = $1`, [filter.id]);
        await trx.query(`DELETE FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}
			WHERE "filterId" = $1`, [filter.id]);
        await trx.query(`DELETE FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}
			WHERE "filterId" = $1`, [filter.id]);
        await trx.query(`DELETE FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnitRole}
			WHERE "filterId" = $1`, [filter.id]);
        await trx.query(`DELETE FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnit}
			WHERE "filterId" = $1`, [filter.id]);
        await trx.query(`DELETE FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterOrganizationRole}
			WHERE "filterId" = $1`, [filter.id]);
    }
    else {
        filter.id = uuid_1.default.v4();
        await trx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}
			(id) VALUES ($1)`, [filter.id]);
    }
    if (filter.userIds?.length > 0) {
        for (const i of filter.userIds) {
            await trx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUser}
				("filterId", "userId") VALUES ($1, $2)`, [filter.id, i]);
        }
    }
    if (filter.groupIds?.length > 0) {
        for (const i of filter.groupIds) {
            await trx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}
				("filterId", "groupId") VALUES ($1, $2)`, [filter.id, i]);
        }
    }
    if (filter.roleIds?.length > 0) {
        for (const i of filter.roleIds) {
            await trx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterOrganizationRole}
				("filterId", "roleId") VALUES ($1, $2)`, [filter.id, i]);
        }
    }
    if (filter.workPlanIds?.length > 0) {
        for (const i of filter.workPlanIds) {
            await trx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}
				("filterId", "workPlanId") VALUES ($1, $2)`, [filter.id, i]);
        }
    }
    if (filter.units?.length > 0) {
        for (const u of filter.units) {
            await trx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnit}
				("filterId", "unitId", "hierarchically") VALUES ($1, $2, $3)`, [filter.id, u.id, u.hierarchically]);
            if (u.roleIds?.length > 0) {
                for (const i of u.roleIds) {
                    await trx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnitRole}
						("filterId", "unitId", "roleId") VALUES ($1, $2, $3)`, [filter.id, u.id, i]);
                }
            }
        }
    }
    return filter.id;
}
exports.upsertUserFilter = upsertUserFilter;
function getSingleUserIdFromFilter(filter) {
    return filter.userIds?.length === 1 &&
        !(filter.groupIds?.length > 0) &&
        !(filter.roleIds?.length > 0) &&
        !(filter.workPlanIds?.length > 0) &&
        !filter.state &&
        !(filter.units?.length > 0) &&
        !filter.visiteeMemberFilter &&
        !(filter.visiteeUnits?.length > 0)
        ? filter.userIds[0]
        : null;
}
exports.getSingleUserIdFromFilter = getSingleUserIdFromFilter;
exports.momentWeekdays = {
    Sunday: 0,
    Monday: 1,
    Tuesday: 2,
    Wednesday: 3,
    Thursday: 4,
    Friday: 5,
    Saturday: 6,
};
function calculateNextExecutionDateOfScheduledJob(interval, firstExecutionDate) {
    const now = (0, moment_1.default)();
    const date = (0, moment_1.default)(firstExecutionDate);
    switch (interval) {
        case 5: {
            date.set("years", now.get("years"));
            if (date.isBefore(now)) {
                date.add(1, "years");
            }
            break;
        }
        case 4: {
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            if (date.isBefore(now)) {
                date.add(1, "months");
            }
            break;
        }
        case 3: {
            const expectedDayOfWeek = date.get("weekday");
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            date.set("date", now.get("date"));
            if (date.isBefore(now)) {
                date.add(1, "days");
            }
            while (expectedDayOfWeek !== date.get("weekday")) {
                date.add(1, "days");
            }
            break;
        }
        case 2: {
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            date.set("date", now.get("date"));
            if (date.isBefore(now)) {
                date.add(1, "days");
            }
            break;
        }
        case 7: {
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            date.set("date", now.get("date"));
            if (date.isBefore(now)) {
                date.add(1, "days");
            }
            let w = date.get("weekday");
            while (w !== exports.momentWeekdays.Sunday && w !== exports.momentWeekdays.Saturday) {
                date.add(1, "days");
                w = date.get("weekday");
            }
            break;
        }
        case 6: {
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            date.set("date", now.get("date"));
            if (date.isBefore(now)) {
                date.add(1, "days");
            }
            let w = date.get("weekday");
            while (w === exports.momentWeekdays.Sunday || w === exports.momentWeekdays.Saturday) {
                date.add(1, "days");
                w = date.get("weekday");
            }
            break;
        }
        case 1: {
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            date.set("date", now.get("date"));
            date.set("hours", now.get("hours"));
            if (date.isBefore(now)) {
                date.add(1, "hours");
            }
            break;
        }
        case 8: {
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            date.set("date", now.get("date"));
            date.set("hours", now.get("hours"));
            if (date.isBefore(now)) {
                date.add(6, "hours");
            }
            break;
        }
        case 9: {
            date.set("years", now.get("years"));
            date.set("months", now.get("months"));
            date.set("date", now.get("date"));
            date.set("hours", now.get("hours"));
            if (date.isBefore(now)) {
                date.add(12, "hours");
            }
            break;
        }
        default:
            break;
    }
    return date.toDate();
}
exports.calculateNextExecutionDateOfScheduledJob = calculateNextExecutionDateOfScheduledJob;
function updateViewsAfterClone(orgId, dbuser, dbsuperuser) {
    return `
		CREATE VIEW "${orgId}".vw_user_filter
		AS
		SELECT  uf.id,
				jsonb_strip_nulls(jsonb_build_object(
									'id', uf.id,
									'name', uf.name,
									'description', uf.description,
									'combinator', uf.combinator,
									'usageType', uf."usageType",
									'allIncluded', uf."allIncluded",
									'groupIds', COALESCE(jsonb_agg(DISTINCT ufg."groupId") FILTER (WHERE ufg."groupId" IS NOT NULL), NULL::jsonb), 
									'roleIds', COALESCE(jsonb_agg(DISTINCT ufor."roleId") FILTER (WHERE ufor."roleId" IS NOT NULL), NULL::jsonb), 
									'userIds', COALESCE(jsonb_agg(DISTINCT ufu."userId") FILTER (WHERE ufu."userId" IS NOT NULL), NULL::jsonb), 
									'workPlanIds', COALESCE(jsonb_agg(DISTINCT ufw."workPlanId") FILTER (WHERE ufw."workPlanId" IS NOT NULL), NULL::jsonb), 
									'units', COALESCE(jsonb_agg(DISTINCT jsonb_build_object('id', ufn."unitId", 'hierarchically', ufn.hierarchically, 'roleIds', ufuragg."roleIds")) 
											FILTER (WHERE ufn."unitId" IS NOT NULL), NULL::jsonb))
								)AS filter
		FROM "${orgId}".user_filter uf
		LEFT JOIN "${orgId}".user_filter_group ufg ON uf.id = ufg."filterId"
		LEFT JOIN "${orgId}".user_filter_organization_role ufor ON uf.id = ufor."filterId"
		LEFT JOIN "${orgId}".user_filter_user ufu ON uf.id = ufu."filterId"
		LEFT JOIN "${orgId}".user_filter_work_plan ufw ON uf.id = ufw."filterId"
		LEFT JOIN "${orgId}".user_filter_unit ufn ON uf.id = ufn."filterId"
		LEFT JOIN ( SELECT	ufur."filterId",
							ufur."unitId",
							COALESCE(json_agg(ufur."roleId") FILTER (WHERE ufur."roleId" IS NOT NULL), NULL::json) AS "roleIds"
					FROM "${orgId}".user_filter_unit_role ufur
					GROUP BY ufur."filterId", ufur."unitId") ufuragg 
			ON ufuragg."unitId" = ufn."unitId" AND ufn."filterId" = ufuragg."filterId"
		GROUP BY uf.id;

	ALTER TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vwUserFilter}"
		OWNER TO ${dbsuperuser};
		
	CREATE OR REPLACE VIEW "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_CredentialUserProfiles}"
		AS
        SELECT uoc.id,
            uoc.id AS "credentialId",
            uop.id AS "profileId",
            uoc.data,
            uoc."userId",
            (((uop.name)::text || ' '::text) || (uop.surname)::text) AS "fullName",
            uoc."organizationId",
            uo."roleId"
        FROM "${orgId}"."userOrganizationCredentials" uoc
            JOIN "${orgId}"."userOrganizations" uo 
                ON uo."userId" = uoc."userId"
                    AND uo."organizationId" = uoc."organizationId"
                    AND uo."deletedAt" IS NULL 
                    AND uoc."deletedAt" IS NULL
            JOIN "${orgId}".roles r 
                ON r."organizationId" = uo."organizationId" 
                    AND uo."roleId" = r.id 
                    AND (
                        r."typeId" IS NULL OR
                        r."typeId" <> ALL (ARRAY['2a4ad5ff-1ced-45f8-992f-e19fc4e766d7'::uuid, '9b8eebd7-057f-4021-ad6f-f2879733bdb9'::uuid])
                    )
            JOIN "${orgId}"."userOrganizationProfiles" uop 
            ON uoc."userId" = uop."userId" 
                AND uop."deletedAt" IS NULL 
                AND uoc."deletedAt" IS NULL
        UNION ALL
            SELECT uoc.id,
            uoc.id AS "credentialId",
            ovp.id AS "profileId",
            uoc.data,
            uoc."userId",
            (((ovp.name)::text || ' '::text) || (ovp.surname)::text) AS "fullName",
            uoc."organizationId",
            uo."roleId"
            FROM "${orgId}"."userOrganizationCredentials" uoc
                JOIN "${orgId}"."userOrganizations" uo 
                    ON uo."userId" = uoc."userId" 
                        AND uo."organizationId" = uoc."organizationId"
                        AND uo."deletedAt" IS NULL
                        AND uoc."deletedAt" IS NULL
                JOIN "${orgId}".roles r 
                    ON r."organizationId" = uo."organizationId" 
                        AND uo."roleId" = r.id
                        AND r."typeId" = ANY (ARRAY['2a4ad5ff-1ced-45f8-992f-e19fc4e766d7'::uuid, '9b8eebd7-057f-4021-ad6f-f2879733bdb9'::uuid])
                JOIN "${orgId}"."organizationVisitorProfiles" ovp 
                    ON ovp."userOrganizationId" = uo.id 
                    AND ovp."deletedAt" IS NULL 
                    AND ovp."deletedAt" IS NULL;

	ALTER TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_CredentialUserProfiles}"
		OWNER TO ${dbsuperuser};
      

	CREATE OR REPLACE VIEW "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_SearchPPermissions}" AS
	SELECT p.id,
		p."organizationId",
		p."dynamicFormData",
		p."dynamicFormId",
		p."startDateTime",
		p."endDateTime",
		p.status,
		p."requestDateTime",
		p.note,
		p."requestUserId",
		up."userPPermissions",
		upa."userPPermissionApprovements",
		useranddeputy2."userAndDeputies",
		pt."ppermissionType",
		upa."nextApproverOrganizationUnitIds",
		upa."approvedOrganizationUnitIds",
		upa."approvementWaitingOrganizationUnitIds",
		up."userCount"
	FROM ((((("${orgId}"."ppermissions" p
		JOIN ( SELECT pt_1.id,
				COALESCE(json_agg(( SELECT x.*::record AS x
					FROM ( SELECT pt_1.id,
								pt_1."organizationId",
								pt_1.name,
								pt_1.description,
								pt_1."isPassive",
								pt_1."hrCanInit",
								pt_1."personnelCanInit",
								pt_1."isDailyScheduled",
								pt_1."minValue",
								pt_1."maxValue",
								pt_1."maxValueBoundaryInterval",
								pt_1."method",
								pt_1."sendNotification",
								pt_1."createdAt",
								pt_1."updatedAt",
								pt_1."deletedAt",
								pt_1."needsManagerDeputy",
								pt_1."needsUserDeputy",
								pt_1."userMustSign",
								pt_1.type,
								pt_1."showRemainingCount",
								pt_1."isUnpaidLeave",
								pt_1."notifyHR",
								pt_1."dynamicFormId",
								pt_1."notifyManagers",
								pt_1.approvements) x)) FILTER (WHERE (pt_1.id IS NOT NULL)), '[]'::json) AS "ppermissionType"
			FROM ( SELECT pt_2.id,
						pt_2."organizationId",
						pt_2.name,
						pt_2.description,
						pt_2."isPassive",
						pt_2."hrCanInit",
						pt_2."personnelCanInit",
						pt_2."isDailyScheduled",
						pt_2."minValue",
						pt_2."maxValue",
						pt_2."maxValueBoundaryInterval",
						pt_2."method",
						pt_2."sendNotification",
						pt_2."createdAt",
						pt_2."updatedAt",
						pt_2."deletedAt",
						pt_2."needsManagerDeputy",
						pt_2."needsUserDeputy",
						pt_2."userMustSign",
						pt_2.type,
						pt_2."showRemainingCount",
						pt_2."isUnpaidLeave",
						pt_2."notifyHR",
						pt_2."dynamicFormId",
						pt_2."notifyManagers",
						COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
							FROM ( SELECT pta."rootOrganizationUnitId",
										pta."order") x)) FILTER (WHERE (pta.id IS NOT NULL)), '[]'::json) AS approvements
					FROM ("${orgId}"."ppermissionTypes" pt_2
						LEFT JOIN "${orgId}"."ppermissionTypeApprovements" pta ON ((pt_2.id = pta."ppermissionTypeId")))
					GROUP BY pt_2.id) pt_1
			GROUP BY pt_1.id) pt ON ((pt.id = p."ppermissionTypeId")))
		LEFT JOIN ( SELECT p_1.id,
				COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
					FROM ( SELECT pa_1.id,
								pa_1."ppermissionId",
								pa_1.name) x)) FILTER (WHERE (pa_1.id IS NOT NULL)), '[]'::json) AS "ppermissionAttachments"
			FROM ("${orgId}"."ppermissionAttachments" pa_1
				JOIN "${orgId}"."ppermissions" p_1 ON ((p_1.id = pa_1."ppermissionId")))
			GROUP BY p_1.id) pa ON ((pa.id = p.id)))
		LEFT JOIN ( SELECT up_1."ppermissionId",
				COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
					FROM ( SELECT up_1.id,
								up_1."userId",
								up_1."ppermissionId",
								up_1.note,
								up_1.approvement,
								up_1."signedByUser",
								up_1."currentApprovementOrder") x)) FILTER (WHERE (up_1.id IS NOT NULL)), '[]'::json) AS "userPPermissions",
				count(*) AS "userCount"
			FROM "${orgId}"."userPPermissions" up_1
			GROUP BY up_1."ppermissionId") up ON ((p.id = up."ppermissionId")))
		LEFT JOIN ( SELECT upa_1."ppermissionId",
				COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
					FROM ( SELECT upa_1.id,
								upa_1."userId",
								upa_1."ppermissionId",
								upa_1."order",
								upa_1."approverUserId",
								upa_1."approvementDate",
								upa_1.note,
								upa_1.status,
								upa_1."organizationUnitId") x)) FILTER (WHERE (upa_1.id IS NOT NULL)), '[]'::json) AS "userPPermissionApprovements",
				COALESCE(json_agg(DISTINCT ( SELECT x."organizationUnitId" AS x
					FROM ( SELECT upa_1.id,
								upa_1."userId",
								upa_1."ppermissionId",
								upa_1."order",
								upa_1."approverUserId",
								upa_1."approvementDate",
								upa_1.note,
								upa_1.status,
								upa_1."organizationUnitId") x)) FILTER (WHERE ((upa_1.id IS NOT NULL) AND (upa_1."approvementDate" IS NULL))), '[]'::json) AS "approvementWaitingOrganizationUnitIds",
				COALESCE(json_agg(DISTINCT ( SELECT x."organizationUnitId" AS x
					FROM ( SELECT upa_1.id,
								upa_1."userId",
								upa_1."ppermissionId",
								upa_1."order",
								upa_1."approverUserId",
								upa_1."approvementDate",
								upa_1.note,
								upa_1.status,
								upa_1."organizationUnitId") x)) FILTER (WHERE ((upa_1.id IS NOT NULL) AND (upa_1."approvementDate" IS NOT NULL))), '[]'::json) AS "approvedOrganizationUnitIds",
				COALESCE(json_agg(DISTINCT ( SELECT x."organizationUnitId" AS x
					FROM ( SELECT upa_1.id,
								upa_1."userId",
								upa_1."ppermissionId",
								upa_1."order",
								upa_1."approverUserId",
								upa_1."approvementDate",
								upa_1.note,
								upa_1.status,
								upa_1."organizationUnitId") x)) FILTER (WHERE ((upa_1.id IS NOT NULL) AND (upa_1."order" = up_1."currentApprovementOrder") AND (upa_1."approverUserId" IS NULL))), '[]'::json) AS "nextApproverOrganizationUnitIds"
			FROM ("${orgId}"."userPPermissionApprovements" upa_1
				JOIN "${orgId}"."userPPermissions" up_1 ON (((up_1."ppermissionId" = upa_1."ppermissionId") AND (up_1."userId" = upa_1."userId"))))
			GROUP BY upa_1."ppermissionId") upa ON ((p.id = upa."ppermissionId")))
		LEFT JOIN ( SELECT useranddeputy."ppermissionId",
				COALESCE(json_agg(( SELECT x.*::record AS x
					FROM ( SELECT useranddeputy."user",
								useranddeputy."deputyUser") x)), '[]'::json) AS "userAndDeputies"
			FROM ( SELECT p_1.id AS "ppermissionId",
						p_1."organizationId",
						json_build_object('id', uop."userId", 'fullname', (((uop.name)::text || ' '::text) || (uop.surname)::text), 'uniqueId', uop."uniqueId") AS "user",
							CASE
								WHEN (uop2."userId" IS NOT NULL) THEN json_build_object('id', uop2."userId", 'fullname', (((uop2.name)::text || ' '::text) || (uop2.surname)::text), 'uniqueId', uop2."uniqueId")
								ELSE NULL::json
							END AS "deputyUser"
					FROM (((((("${orgId}"."ppermissions" p_1
						JOIN "${orgId}"."userPPermissions" up_1 ON ((up_1."ppermissionId" = p_1.id)))
						JOIN "${orgId}"."userOrganizations" uo ON (((uo."userId" = up_1."userId") AND (uo."organizationId" = p_1."organizationId"))))
						JOIN "${orgId}"."userOrganizationProfiles" uop ON ((uop."userOrganizationId" = uo.id)))
						LEFT JOIN "${orgId}"."userOrganizationDeputies" uod ON (((uod."ppermissionId" = p_1.id) AND (uod."userOrganizationId" = uo.id))))
						LEFT JOIN "${orgId}"."userOrganizations" uo2 ON ((uo2.id = uod."deputyUserOrganizationId")))
						LEFT JOIN "${orgId}"."userOrganizationProfiles" uop2 ON ((uop2."userOrganizationId" = uo2.id)))
					WHERE ((uo."deletedAt" IS NULL) AND (uop."deletedAt" IS NULL) AND (uo2."deletedAt" IS NULL) AND (uop2."deletedAt" IS NULL))) useranddeputy
			GROUP BY useranddeputy."ppermissionId") useranddeputy2 ON ((useranddeputy2."ppermissionId" = p.id)));

	ALTER TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_SearchPPermissions}"
		OWNER TO ${dbsuperuser};

	CREATE OR REPLACE VIEW "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_UserOrganizationUnitHierarchys}"
		AS
		SELECT uo."organizationId" AS id,
		uo."organizationId",
		uo."userId",
		cou.id AS "ancestorOrganizationUnitId",
		ou.id AS "descenderOrganizationUnitId"
		FROM "${orgId}"."userOrganizationOrganizationUnits" uoou
			JOIN "${orgId}"."userOrganizations" uo ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL AND uo."deletedAt" IS NULL
			JOIN "${orgId}"."organizationUnits" ou ON ou.id = uoou."organizationUnitId" AND ou."deletedAt" IS NULL AND uoou."deletedAt" IS NULL
			JOIN "${orgId}"."organizationUnits" cou ON (ou.id = cou.id OR ou."ancestorIds"::text ~~ (('%'::text || cou.id) || '%'::text)) AND uoou."deletedAt" IS NULL AND ou."deletedAt" IS NULL AND cou."deletedAt" IS NULL;

	ALTER TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_UserOrganizationUnitHierarchys}"
		OWNER TO ${dbsuperuser};
      


	CREATE OR REPLACE VIEW "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_UserHierarchys}"
	AS
	SELECT mu."organizationId" AS id,
		mu."organizationId",
		uo."userId",
		mu."ancestorOrganizationUnitId",
		mu."descenderOrganizationUnitId",
		mu."userId" AS "userIdUnderAuthority"
		FROM "${orgId}"."userOrganizationOrganizationUnits" uoou
		JOIN "${orgId}"."userOrganizations" uo ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL AND uo."deletedAt" IS NULL
		JOIN "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_UserOrganizationUnitHierarchys}" mu ON (mu."descenderOrganizationUnitId" = uoou."organizationUnitId" OR mu."ancestorOrganizationUnitId" = uoou."organizationUnitId") AND uoou."deletedAt" IS NULL;

	ALTER TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_UserHierarchys}"
		OWNER TO ${dbsuperuser};

	CREATE OR REPLACE VIEW "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_WorkPlanPeriodTstzrange}"
	AS
	SELECT
		"workPlanPeriods".id,
		"workPlanPeriods"."workPlanId",
		"workPlanPeriods".details,
		"workPlanPeriods"."offset",
		tstzrange("workPlanPeriods"."periodStartDateTime", lead("workPlanPeriods"."periodStartDateTime") OVER (PARTITION BY "workPlanPeriods"."workPlanId" ORDER BY "workPlanPeriods"."periodStartDateTime")) AS range,
		"workPlans"."organizationId"
	FROM "${orgId}"."workPlanPeriods"
	JOIN "${orgId}"."workPlans"
		ON "workPlans".id = "workPlanPeriods"."workPlanId"
	WHERE "workPlanPeriods"."deletedAt" IS NULL AND "workPlans"."deletedAt" IS NULL;

	ALTER TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_WorkPlanPeriodTstzrange}"
		OWNER TO ${dbsuperuser};


	------------------ mVW_SearchUsers ---------------------
	CREATE OR REPLACE VIEW "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.mVW_SearchUsers}"
	AS
	SELECT uo.id,
	uo."isDisabled" AS "isDisabledForUserOrganization",
	uo."userId",
	uo."organizationId",
	COALESCE(string_agg(DISTINCT uop.name::text, ' '::text)) AS "userOrganizationProfileNameForOrder",
	COALESCE(string_agg(DISTINCT uop.surname::text, ' '::text)) AS "userOrganizationProfileSurnameForOrder",
	COALESCE(string_agg(DISTINCT uop."uniqueId"::text, ' '::text)) AS "userOrganizationProfileUniqueIdForOrder",
	COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
			FROM ( SELECT u.username) x)) FILTER (WHERE u.id IS NOT NULL), '[]'::json) AS "user",
	COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
			FROM ( SELECT uop."uniqueId",
					uop.name,
					uop.surname,
					uop.email) x)) FILTER (WHERE uop.id IS NOT NULL), '[]'::json) AS "userOrganizationProfile",
	COALESCE(string_agg(DISTINCT ug.id::character varying::text, ' '::text)) AS "userGroupIds",
	COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
			FROM ( SELECT ug.id,
					ug.name,
					ug."colorCode") x)) FILTER (WHERE ug.id IS NOT NULL), '[]'::json) AS "userGroups",
	COALESCE(string_agg(DISTINCT ou.id::character varying::text, ' '::text)) AS "organizationUnitIds",
	COALESCE(string_agg(( CASE WHEN ou."ancestorIds" IS NOT NULL THEN ou."ancestorIds"::text || ','::text ELSE '' END) || ou.id, ' '::text)) AS "organizationUnitHierarchies",
	COALESCE(json_agg(DISTINCT ( SELECT x.*::record AS x
			FROM ( SELECT ou.id,
					ou.name,
					ou."typeId",
					ou."ancestorIds") x)) FILTER (WHERE ou.id IS NOT NULL), '[]'::json) AS "organizationUnits",
	(((((((setweight(to_tsvector('simple'::regconfig, unaccent(COALESCE(string_agg(uop.name::text, ' '::text) FILTER (WHERE uop.id IS NOT NULL), ''::text))), 'A'::"char") || setweight(to_tsvector('simple'::regconfig, unaccent(COALESCE(string_agg(uop.surname::text, ' '::text) FILTER (WHERE uop.id IS NOT NULL), ''::text))), 'A'::"char")) || setweight(to_tsvector('simple'::regconfig, unaccent(COALESCE(string_agg(uop.address::text, ' '::text) FILTER (WHERE uop.id IS NOT NULL), ''::text))), 'D'::"char")) || setweight(to_tsvector('simple'::regconfig, COALESCE(string_agg(uop."phoneNumber"::text, ' '::text) FILTER (WHERE uop.id IS NOT NULL), ''::text)), 'B'::"char")) || setweight(to_tsvector('simple'::regconfig, COALESCE(string_agg(uop.email::text, ' '::text) FILTER (WHERE uop.id IS NOT NULL), ''::text)), 'B'::"char")) || setweight(to_tsvector('simple'::regconfig, COALESCE(string_agg(uop."uniqueId"::text, ' '::text) FILTER (WHERE uop.id IS NOT NULL), ''::text)), 'A'::"char")) || setweight(to_tsvector('simple'::regconfig, unaccent(COALESCE(string_agg(uop."extensionFields"::text, ' '::text) FILTER (WHERE uop.id IS NOT NULL), ''::text))), 'B'::"char")) || setweight(to_tsvector('simple'::regconfig, unaccent(COALESCE(string_agg(ug.name::text, ' '::text) FILTER (WHERE ug.id IS NOT NULL), ''::text))), 'C'::"char")) || setweight(to_tsvector('simple'::regconfig, unaccent(COALESCE(string_agg(ou.name::text, ' '::text) FILTER (WHERE ou.id IS NOT NULL), ''::text))), 'C'::"char") AS "searchDocument"
	FROM "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
		JOIN "${orgId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.users} u ON uo."userId" = u.id AND uo."deletedAt" IS NULL AND u."deletedAt" IS NULL
		JOIN "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop ON uo.id = uop."userOrganizationId" AND uo."deletedAt" IS NULL AND uop."deletedAt" IS NULL
		LEFT JOIN "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou ON uo.id = uoou."userOrganizationId" AND uo."deletedAt" IS NULL AND uoou."deletedAt" IS NULL
		LEFT JOIN "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou ON ou.id = uoou."organizationUnitId" AND ou."deletedAt" IS NULL AND uoou."deletedAt" IS NULL
		LEFT JOIN "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo ON uo.id = uguo."userOrganizationId" AND uo."deletedAt" IS NULL AND uguo."deletedAt" IS NULL
		LEFT JOIN "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug ON ug.id = uguo."userGroupId" AND ug."deletedAt" IS NULL AND uguo."deletedAt" IS NULL
	GROUP BY uo.id;

	ALTER TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.mVW_SearchUsers}"
	OWNER TO ${dbsuperuser};

	GRANT ALL ON TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.mVW_SearchUsers}" TO ${dbsuperuser};
	GRANT INSERT, SELECT, UPDATE, DELETE ON TABLE "${orgId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.mVW_SearchUsers}" TO ${dbuser};
	`;
}
exports.updateViewsAfterClone = updateViewsAfterClone;
function checkTimeRange(options) {
    const now = luxon_1.DateTime.now();
    const day = now.weekday;
    let flag = false;
    if (options.days && day in options.days) {
        for (const timeRange of options.days[day]) {
            const startDate = luxon_1.DateTime.fromISO(timeRange.startTime);
            const endDate = luxon_1.DateTime.fromISO(timeRange.endTime);
            const diffFromStartTime = now.diff(startDate, ["hours", "minutes", "seconds"]);
            const diffFromEndTime = now.diff(endDate, ["hours", "minutes", "seconds"]);
            if (diffFromStartTime.minutes >= 0 &&
                diffFromStartTime.hours >= 0 &&
                diffFromStartTime.seconds >= 0 &&
                diffFromEndTime.minutes <= 0 &&
                diffFromEndTime.hours <= 0 &&
                diffFromEndTime.seconds <= 0) {
                flag = !flag;
                break;
            }
        }
    }
    else if (!options.days) {
        flag = !flag;
    }
    return flag;
}
exports.checkTimeRange = checkTimeRange;
const getOrganizationVisitorModuleSettings = async (organizationId, client) => {
    const query = `SELECT settings FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorModuleSettings}"`;
    const { rows } = await client.query(query);
    return rows[0]?.settings;
};
exports.getOrganizationVisitorModuleSettings = getOrganizationVisitorModuleSettings;
const listenClientWrapperWithRetry = async (serviceName, dbPool) => {
    let tryCount = 1;
    let isConnected = false;
    let trialInterval;
    const retryPromise = new Promise(async (resolve, reject) => {
        const retryFn = async () => {
            if (tryCount > 10) {
                app_logs_1.logger.error(`Database connection could not be established for LISTEN client, terminating process`);
                process.exit(1);
            }
            try {
                const client = await dbPool.connect();
                if (trialInterval) {
                    clearInterval(trialInterval);
                }
                app_logs_1.logger.info(`Database LISTEN connected for ${serviceName}, Try Count:${tryCount}`);
                isConnected = true;
                resolve(client);
            }
            catch (error) {
                app_logs_1.logger.error(`Database LISTEN connection is failed for ${serviceName}, Try Count:${tryCount}`);
                app_logs_1.logger.error(error);
                tryCount++;
            }
        };
        await retryFn();
        if (!isConnected) {
            trialInterval = setInterval(retryFn, 10000);
        }
    });
    const dbClient = await retryPromise;
    return dbClient;
};
exports.listenClientWrapperWithRetry = listenClientWrapperWithRetry;
const getProfileFieldsFromUserIds = async (organizationId, userIds, pagination, client) => {
    let result = [];
    const queryResult = (await client.query(`
		SELECT "userId",
				name || ' ' ||surname as fullname 
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop
		WHERE uop."userId" = ANY($1)
		ORDER BY name ASC
		OFFSET $2
		LIMIT $3
		`, [userIds, pagination.skip, pagination.take])).rows;
    result = (await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, queryResult.map((elem) => elem.userId))).map((elem) => {
        return {
            userId: elem.id,
            fullname: queryResult.find((item) => item.userId === elem.id)?.fullname,
            userCaptions: elem.captionLines,
        };
    });
    return result;
};
exports.getProfileFieldsFromUserIds = getProfileFieldsFromUserIds;
