"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessUserFilter = void 0;
const dal_access_rdb_userfilter_1 = require("../rdb/dal.access.rdb.userfilter");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const lodash_1 = __importDefault(require("lodash"));
const uuid_1 = __importDefault(require("uuid"));
const restapi_1 = require("../../../lib/es/models/restapi");
const dal_utils_1 = require("../../dal.utils");
const api_error_1 = require("../../../api/api.error");
const app_enums_1 = require("../../../app.enums");
const dal_access_psql_organization_1 = require("./dal.access.psql.organization");
const dal_manager_1 = require("../../dal.manager");
class PSQLDalAccessUserFilter extends dal_access_rdb_userfilter_1.RDBDalAccessUserFilter {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getFilterUsageType(organizationId, id, trx) {
        const result = (await trx.query(`
			SELECT "usageType" 
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
			WHERE id = $1
			`, [id])).rows[0]?.usageType;
        return result;
    }
    async getFilter(organizationId, id, trx) {
        const filter = (await trx.query(`
			SELECT filter 
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vwUserFilter}"
			WHERE id = $1
			`, [id])).rows[0]?.filter;
        const result = {
            id: filter.id,
            name: filter.name,
            description: filter.description,
            combinator: filter.combinator,
            usageType: filter.usageType,
            allIncluded: filter.allIncluded,
            settings: {
                groupIds: filter.groupIds,
                userIds: filter.userIds,
                units: filter.units,
                roleIds: filter.roleIds,
                workPlanIds: filter.workPlanIds,
            },
        };
        return result;
    }
    async getUserFilterDetail(organizationId, id, trx) {
        const filter = await this.getFilter(organizationId, id, trx);
        let result = {
            id: filter.id,
            name: filter.name,
            description: filter.description,
            combinator: filter.combinator,
            usageType: filter.usageType,
            allIncluded: filter.allIncluded,
            settings: {},
            userCount: 0,
            acpCount: 0,
        };
        if (filter.settings.groupIds?.length) {
            result.settings.groups = (await trx.query(`
				SELECT id, name
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
				WHERE "id" = ANY($1)
				`, [filter.settings.groupIds])).rows;
        }
        if (filter.settings.roleIds?.length) {
            result.settings.roles = (await trx.query(`
				SELECT id, name
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
				WHERE id = ANY($1)
				`, [filter.settings.roleIds])).rows;
        }
        if (filter.settings.workPlanIds?.length) {
            result.settings.workPlans = (await trx.query(`
				SELECT id, name 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
				WHERE id = ANY($1)
				`, [filter.settings.workPlanIds])).rows;
        }
        if (filter.settings.units?.length) {
            result.settings.units = (await trx.query(`
				SELECT id, name
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
				WHERE id = ANY($1)
				`, [filter.settings.units.map((elem) => elem.id)])).rows;
            const unitIdRolenameMappingsDbesult = await trx.query(`
				SELECT id, name
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
				WHERE id = ANY($1::UUID[])
				`, [[...new Set(filter.settings.units?.map((m) => m.roleIds || []).flat())]]);
            for (const unit of result.settings.units) {
                const feature = filter.settings.units.find((elem) => elem.id === unit.id);
                unit.hierarchically = feature.hierarchically;
                if (feature.roleIds?.length > 0) {
                    unit.roles = unitIdRolenameMappingsDbesult.rows.filter((f) => feature.roleIds.includes(f.id));
                    unit.hierarchically = feature.hierarchically;
                }
                else {
                    unit.roles = [];
                }
            }
        }
        if (filter.settings.userIds?.length) {
            const userInformations = await (0, dal_utils_1.getProfileFieldsFromUserIds)(organizationId, filter.settings.userIds, {
                skip: 0,
                take: 100000,
            }, trx);
            result.settings.users = userInformations.map((m) => {
                return {
                    id: m.userId,
                    name: m.fullname,
                };
            });
        }
        const userCountDbResult = await trx.query(`
				SELECT "userCount"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
				WHERE "id" = $1
			`, [id]);
        result.userCount = userCountDbResult.rows[0].userCount;
        if (filter.usageType === restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS) {
            const acpCountDbResult = await trx.query(`
					SELECT COUNT(*)::INTEGER AS c
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
					WHERE "filterId" = $1
				`, [id]);
            result.acpCount = acpCountDbResult.rows[0].c;
        }
        return result;
    }
    async listUserFilters(organizationId, filter, requesterUserId, trx) {
        const result = {
            pagination: {
                take: filter.pagination.take,
                skip: filter.pagination.skip,
                total: 0,
            },
            items: [],
        };
        const isUserFilterAccessRightActivated = await (0, dal_access_psql_organization_1.getOrganizationUserFilterAccessRightActivationStatus)(organizationId, trx);
        const whereConditions = [];
        const bindings = [];
        let baseQuery = `
		SELECT  DISTINCT(id),
				name,
				description,
				"usageType",
				"userCount",
				(
				SELECT 
					CASE WHEN "usageType" = ${restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS}  
						 THEN (SELECT COUNT(*)::INTEGER FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}" ufacp WHERE ufacp."filterId" = uf.id) 
					ELSE 0 END
				) AS "acpCount",
				true as "canEdit"
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}" as uf
		`;
        if (filter.name) {
            bindings.push(filter.name);
            whereConditions.push(`uf.name ILIKE '%' || $${bindings.length} || '%'`);
        }
        if (filter.description) {
            bindings.push(filter.description);
            whereConditions.push(`uf.description ILIKE '%' || $${bindings.length} || '%'`);
        }
        if (filter.allIncluded) {
            bindings.push(filter.allIncluded);
            whereConditions.push(`uf."allIncluded" = $${bindings.length}`);
        }
        if (filter.usageType?.length > 0) {
            bindings.push(filter.usageType);
            whereConditions.push(`uf."usageType" = ANY($${bindings.length})`);
        }
        if (!isUserFilterAccessRightActivated) {
            whereConditions.push(`NOT uf."usageType" = ${restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS}`);
        }
        if (filter.settings) {
            if (filter.settings.groupIds?.length > 0) {
                baseQuery += `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}" as ufg
					ON ufg."filterId" = uf.id
				`;
                bindings.push(filter.settings.groupIds);
                whereConditions.push(`ufg."groupId" = ANY($${bindings.length})`);
            }
            if (filter.settings.roleIds?.length > 0) {
                baseQuery += `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterOrganizationRole}" as ufor
					ON ufor."filterId" = uf.id
				`;
                bindings.push(filter.settings.roleIds);
                whereConditions.push(`ufor."roleId" = ANY($${bindings.length})`);
            }
            if (filter.settings.userIds?.length > 0) {
                baseQuery += `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUser}" as ufu
					ON ufu."filterId" = uf.id
				`;
                bindings.push(filter.settings.userIds);
                whereConditions.push(`ufu."userId" = ANY($${bindings.length})`);
            }
            if (filter.settings.workPlanIds?.length > 0) {
                baseQuery += `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}" as ufwp
					ON ufwp."filterId" = uf.id
				`;
                bindings.push(filter.settings.workPlanIds);
                whereConditions.push(`ufwp."workPlanId" = ANY($${bindings.length})`);
            }
            if (filter.settings.unitIds?.length > 0) {
                baseQuery += `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnit}" as ufu
					ON ufu."filterId" = uf.id
				`;
                bindings.push(filter.settings.unitIds);
                whereConditions.push(`ufu."unitId" = ANY($${bindings.length})`);
            }
        }
        baseQuery += whereConditions.length > 0 ? " WHERE " + whereConditions.join(" AND ") : "";
        result.pagination.total = (await trx.query(`SELECT COUNT(uf.id)::integer as total FROM ( ${baseQuery} ) as uf `, bindings)).rows[0].total;
        baseQuery += `
			ORDER BY uf.name ASC, uf."userCount" DESC
		`;
        bindings.push(filter.pagination.skip);
        baseQuery += ` OFFSET $${bindings.length} `;
        bindings.push(filter.pagination.take);
        baseQuery += ` LIMIT $${bindings.length} `;
        const { rows } = await trx.query(baseQuery, bindings);
        result.items = rows;
        const userRights = await this.getUserRightsOverUserFilters(organizationId, rows.map((r) => r.id), requesterUserId, trx);
        result.items.forEach((r) => {
            if (r.usageType === restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS && r.acpCount > 0) {
                r.canEdit = userRights.find((ur) => ur.filterId === r.id)?.grant || false;
            }
        });
        return result;
    }
    async deleteUserFilter(organizationId, id, trx) {
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
			WHERE "filterId" = $1
			`, [id]);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
			WHERE id = $1
			`, [id]);
        return id;
    }
    async updateUserFilter(organizationId, filter, trx) {
        const currentUserFilter = await this.getFilter(organizationId, filter.id, trx);
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
			SET "usageType" = $1
			WHERE id = $2`, [0, filter.id]);
        const deletedUserGroupFilters = currentUserFilter.settings.groupIds?.filter((elem) => !filter.settings.groupIds?.includes(elem));
        const insertedUserGroupFilters = filter.settings.groupIds?.filter((elem) => !currentUserFilter.settings.groupIds?.includes(elem));
        if (deletedUserGroupFilters?.length) {
            await trx.query(`
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}"
				WHERE   "groupId" = ANY($1) AND
						"filterId" = $2;`, [deletedUserGroupFilters, filter.id]);
        }
        if (insertedUserGroupFilters?.length) {
            const valuePlaceholders = insertedUserGroupFilters.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}" (
				"filterId", "groupId")
				VALUES ${valuePlaceholders}
				`, [...insertedUserGroupFilters.flatMap((elem) => [filter.id, elem])]);
        }
        const deletedUnitFilters = currentUserFilter.settings.units?.filter((elem) => filter.settings.units?.find((item) => elem.id === item.id && !lodash_1.default.isEqual(elem, item)) || !filter.settings.units?.map((item) => item.id).includes(elem.id));
        const insertedUnitFilters = filter.settings.units?.filter((elem) => !currentUserFilter.settings.units?.find((item) => elem.id === item.id && lodash_1.default.isEqual(elem, item)));
        if (deletedUnitFilters?.length) {
            await trx.query(`
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnit}"
				WHERE  "unitId" = ANY($1) AND
					   "filterId" = $2
				;`, [deletedUnitFilters.map((elem) => elem.id), filter.id]);
            for (const unit of deletedUnitFilters) {
                if (unit.roleIds?.length > 0) {
                    trx.query(`
					DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnitRole}"
					WHERE "roleId" = ANY($1) AND
						   "filterId" = $2
					`, [unit.roleIds, filter.id]);
                }
            }
        }
        if (insertedUnitFilters?.length) {
            const valuePlaceholders = insertedUnitFilters.map((_, index) => `($${index * 3 + 1}, $${index * 3 + 2}, $${index * 3 + 3})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnit}" (
				"filterId", "unitId", "hierarchically")
				VALUES ${valuePlaceholders}
				`, [...insertedUnitFilters.flatMap((elem) => [filter.id, elem.id, elem.hierarchically])]);
            for (const unit of insertedUnitFilters) {
                if (unit.roleIds?.length > 0) {
                    const valuePlaceholders = unit.roleIds.map((_, index) => `($${index * 3 + 1}, $${index * 3 + 2}, $${index * 3 + 3})`).join(", ");
                    trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnitRole}" (
					"filterId", "unitId", "roleId")
					VALUES ${valuePlaceholders}
					`, [...unit.roleIds.flatMap((elem) => [filter.id, unit.id, elem])]);
                }
            }
        }
        const deletedUserFilters = currentUserFilter.settings.userIds?.filter((elem) => !filter.settings.userIds?.includes(elem));
        const insertedUserFilters = filter.settings.userIds?.filter((elem) => !currentUserFilter.settings.userIds?.includes(elem));
        if (deletedUserFilters?.length) {
            await trx.query(`
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUser}"
				WHERE  "userId" = ANY($1)
						AND "filterId" = $2;`, [deletedUserFilters, filter.id]);
        }
        if (insertedUserFilters?.length) {
            const valuePlaceholders = insertedUserFilters.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUser}" (
				"filterId", "userId")
				VALUES ${valuePlaceholders}
			`, [...insertedUserFilters.flatMap((elem) => [filter.id, elem])]);
        }
        const deletedRolesFilters = currentUserFilter.settings.roleIds?.filter((elem) => !filter.settings.roleIds?.includes(elem));
        const insertedRolesFilters = filter.settings.roleIds?.filter((elem) => !currentUserFilter.settings.roleIds?.includes(elem));
        if (deletedRolesFilters?.length) {
            await trx.query(`
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterOrganizationRole}"
				WHERE "roleId" = ANY($1) AND "filterId" = $2;`, [deletedRolesFilters, filter.id]);
        }
        if (insertedRolesFilters?.length) {
            const valuePlaceholders = insertedRolesFilters.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterOrganizationRole}" (
				"filterId", "roleId")
				VALUES ${valuePlaceholders}
			`, [...insertedRolesFilters.flatMap((elem) => [filter.id, elem])]);
        }
        const deletedWorkPlansFilters = currentUserFilter.settings.workPlanIds?.filter((elem) => !filter.settings.workPlanIds?.includes(elem));
        const insertedWorkPlansFilters = filter.settings.workPlanIds?.filter((elem) => !currentUserFilter.settings.workPlanIds?.includes(elem));
        if (deletedWorkPlansFilters?.length) {
            await trx.query(`
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}"
				WHERE "workPlanId" = ANY($1) AND "filterId" = $2;`, [deletedWorkPlansFilters, filter.id]);
        }
        if (insertedWorkPlansFilters?.length) {
            const valuePlaceholders = insertedWorkPlansFilters.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}" (
				"filterId", "workPlanId")
				VALUES ${valuePlaceholders}			`, [...insertedWorkPlansFilters.flatMap((elem) => [filter.id, elem])]);
        }
        if (currentUserFilter.combinator !== filter.combinator) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
				SET combinator = $1
				WHERE "id" = $2
				`, [filter.combinator, filter.id]);
        }
        if (currentUserFilter.allIncluded !== filter.allIncluded) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
				SET "allIncluded" = $1
				WHERE "id" = $2
				`, [filter.allIncluded, filter.id]);
        }
        if (currentUserFilter.usageType !== filter.usageType) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
				SET "usageType" = $1
				WHERE "id" = $2
				`, [filter.usageType, filter.id]);
        }
        if (currentUserFilter.name !== filter.name) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
				SET name = $1
				WHERE "id" = $2
				`, [filter.name, filter.id]);
        }
        if (currentUserFilter.description !== filter.description) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
				SET description = $1
				WHERE "id" = $2
				`, [filter.description, filter.id]);
        }
        if (currentUserFilter.usageType !== filter.usageType) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.USER_FILTER.CHANGE_USAGE_TYPE", null, true);
        }
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
			SET "usageType" = $1
			WHERE id = $2`, [currentUserFilter.usageType, filter.id]);
        await trx.query(`SELECT "${organizationId}".update_filter_results($1::uuid[])`, [[filter.id]]);
        return filter.id;
    }
    async insertUserFilter(organizationId, filter, trx) {
        const id = uuid_1.default.v4();
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}" (
				id, name, description, combinator, "allIncluded", "usageType")
				VALUES ( $1, $2, $3, $4, $5, $6);
			`, [id, filter.name, filter.description, filter.combinator, filter.allIncluded, 0]);
        if (filter.settings.roleIds?.length > 0) {
            const valuePlaceholders = filter.settings.roleIds.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterOrganizationRole}"
				("filterId", "roleId")
				VALUES ${valuePlaceholders}
				`, [...filter.settings.roleIds.flatMap((elem) => [id, elem])]);
        }
        if (filter.settings.units?.length > 0) {
            const valuePlaceholders = filter.settings.units.map((_, index) => `($${index * 3 + 1}, $${index * 3 + 2}, $${index * 3 + 3})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnit}"
				("filterId", "unitId", "hierarchically")
				VALUES ${valuePlaceholders}
				`, [...filter.settings.units.flatMap((elem) => [id, elem.id, elem.hierarchically])]);
            for (const unit of filter.settings.units) {
                if (unit.roleIds?.length > 0) {
                    const valuePlaceholders = unit.roleIds.map((_, index) => `($${index * 3 + 1}, $${index * 3 + 2}, $${index * 3 + 3})`).join(", ");
                    await trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUnitRole}"
					("filterId","unitId" ,"roleId")
					VALUES ${valuePlaceholders}
					`, [...unit.roleIds.flatMap((elem) => [id, unit.id, elem])]);
                }
            }
        }
        if (filter.settings.workPlanIds?.length > 0) {
            const valuePlaceholders = filter.settings.workPlanIds.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterWorkPlan}"
				("filterId", "workPlanId")
				VALUES ${valuePlaceholders}
				`, [...filter.settings.workPlanIds.flatMap((elem) => [id, elem])]);
        }
        if (filter.settings.userIds?.length > 0) {
            const valuePlaceholders = filter.settings.userIds.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterUser}"
				("filterId", "userId")
				VALUES ${valuePlaceholders}
				`, [...filter.settings.userIds.flatMap((elem) => [id, elem])]);
        }
        if (filter.settings.groupIds?.length > 0) {
            const valuePlaceholders = filter.settings.groupIds.map((_, index) => `($${index * 2 + 1}, $${index * 2 + 2})`).join(", ");
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}"
				("filterId", "groupId")
				VALUES ${valuePlaceholders};
				`, [...filter.settings.groupIds.flatMap((elem) => [id, elem])]);
        }
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
			SET "usageType" = $1
			WHERE id = $2`, [filter.usageType, id]);
        await trx.query(`SELECT "${organizationId}".update_filter_results($1::uuid[])`, [[id]]);
        return id;
    }
    async searchUserFilterByName(organizationId, request, trx) {
        const result = {
            items: [],
            pagination: {
                take: request.pagination.take,
                skip: request.pagination.skip,
                total: 0,
            },
        };
        let baseQuery = `
		SELECT  id,
				name,
				(COUNT(id) OVER())::integer total
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
		`;
        const whereConditions = [];
        if (request.name) {
            whereConditions.push(`name ILIKE '%' || '${request.name}' || '%'`);
        }
        if (request.usageType) {
            whereConditions.push(`"usageType" = ${request.usageType}`);
        }
        else {
            whereConditions.push(`"usageType" = ANY(${request.authorizedUsageTypes})`);
        }
        const queryResult = (await trx.query(baseQuery + (whereConditions.length > 0 ? " WHERE " + whereConditions.join(" AND ") : "") + ` OFFSET ${request.pagination.skip} LIMIT ${request.pagination.take};`)).rows;
        result.items = queryResult.map((elem) => {
            return {
                id: elem.id,
                name: elem.name,
            };
        });
        result.pagination.total = queryResult[0]?.total ?? 0;
        return result;
    }
    async listUserFilterByIds(organizationId, request, requesterUserId, trx) {
        const result = {
            pagination: {
                take: request.pagination.take,
                skip: request.pagination.skip,
                total: 0,
            },
            items: [],
        };
        const total = (await trx.query(`
				SELECT COUNT(id)::integer AS total
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}"
					WHERE id = ANY($1)
				`, [request.ids])).rows[0].total;
        const queryResult = (await trx.query(`
				WITH user_filter AS (
					SELECT	uf.id,
							uf.name,
							uf.description,
							uf."usageType",
							(SELECT COUNT(*)::INTEGER FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.filterResult}" fr WHERE fr."filterId" = uf.id) AS "userCount",
							(SELECT COUNT(*)::INTEGER FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}" ufacp WHERE ufacp."filterId" = uf.id) AS "acpCount",
							true as "canEdit"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}" uf
				WHERE uf.id = ANY($1) AND uf."usageType" = ANY($2)
				OFFSET $3
				LIMIT $4
				)
				SELECT  jsonb_agg(
						jsonb_build_object(
							'id', id,
							'name', name,
							'description', description,
							'usageType', "usageType",
							'userCount', "userCount",
							'acpCount', "acpCount",
							'canEdit', "canEdit"
							)
						) as items
				FROM user_filter
				`, [request.ids, request.authorizedUsageTypes, request.pagination.skip, request.pagination.take > 0 ? request.pagination.take : total])).rows;
        result.items = queryResult[0].items;
        const userRights = await this.getUserRightsOverUserFilters(organizationId, result.items.map((r) => r.id), requesterUserId, trx);
        result.items.forEach((r) => {
            if (r.usageType === restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS && r.acpCount > 0) {
                r.canEdit = userRights.find((ur) => ur.filterId === r.id)?.grant || false;
            }
        });
        result.pagination.total = total;
        return result;
    }
    async getFilterResultByDefinition(organizationId, params, trx) {
        const result = {
            pagination: {
                take: params.pagination.take,
                skip: params.pagination.skip,
                total: 0,
            },
            items: [],
        };
        const tempFilterId = uuid_1.default.v4();
        const filterQuery = `
			FROM "${organizationId}".user_filter_result(
				(SELECT array_agg(jsonb_populate_record(null::"${organizationId}".filter_info, $1))),
				(SELECT array_agg(jsonb_populate_record(null::"${organizationId}"."user_filter_user", value)) FROM jsonb_array_elements($2)),
				(SELECT array_agg(jsonb_populate_record(null::"${organizationId}"."user_filter_group", value)) FROM jsonb_array_elements($3)),
				(SELECT array_agg(jsonb_populate_record(null::"${organizationId}"."user_filter_organization_role", value)) FROM jsonb_array_elements($4)),
				(SELECT array_agg(jsonb_populate_record(null::"${organizationId}"."user_filter_unit", value)) FROM jsonb_array_elements($5)),
				(SELECT array_agg(jsonb_populate_record(null::"${organizationId}"."user_filter_unit_role", value)) FROM jsonb_array_elements($6)),
				(SELECT array_agg(jsonb_populate_record(null::"${organizationId}"."user_filter_work_plan", value)) FROM jsonb_array_elements($7))
			) ufr`;
        const bindings = [
            JSON.stringify({
                id: tempFilterId,
                combinator: params.userFilter.combinator,
                allIncluded: params.userFilter.allIncluded,
                usageType: restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS,
            }),
            JSON.stringify((params.userFilter.settings.userIds || []).map((e) => {
                return { filterId: tempFilterId, userId: e };
            })),
            JSON.stringify((params.userFilter.settings.groupIds || []).map((e) => {
                return { filterId: tempFilterId, groupId: e };
            })),
            JSON.stringify((params.userFilter.settings.roleIds || []).map((e) => {
                return { filterId: tempFilterId, roleId: e };
            })),
            JSON.stringify((params.userFilter.settings.units || []).map((e) => {
                return { filterId: tempFilterId, unitId: e.id, hierarchically: e.hierarchically ?? false };
            })),
            JSON.stringify((params.userFilter.settings.units || [])
                .filter((u) => u.roleIds && u.roleIds.length)
                .map((u) => {
                return u.roleIds.map((r) => {
                    return { filterId: tempFilterId, unitId: u.id, roleId: r };
                });
            })
                .flat()),
            JSON.stringify((params.userFilter.settings.workPlanIds || []).map((e) => {
                return { filterId: tempFilterId, workPlanId: e };
            })),
        ];
        result.pagination.total = (await trx.query(`SELECT COUNT(*)::integer as count ` + filterQuery, bindings)).rows[0].count;
        bindings.push(params.pagination.skip, params.pagination.take);
        const filteredUsers = (await trx.query(`SELECT ufr."userId", uop.name, uop.surname
				` +
            filterQuery +
            `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					ON uop."userId" = ufr."userId"
				ORDER BY uop.name ASC, uop."surname" DESC
				OFFSET $8 LIMIT $9`, bindings)).rows;
        const userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, filteredUsers.map((f) => f.userId));
        result.items = filteredUsers.map((f) => {
            return {
                userId: f.userId,
                fullname: f.name + " " + f.surname,
                userCaptions: userCaptions.find((c) => c.id === f.userId).captionLines,
            };
        });
        return result;
    }
    async getUserFilterAccessControlPointMappings(organizationId, filterId, trx) {
        const filterDetail = await this.getUserFilterDetail(organizationId, filterId, trx);
        if (!filterDetail) {
        }
        if (filterDetail.usageType !== restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS) {
        }
        const result = {
            filter: filterDetail,
            accessControlPointMappings: undefined,
        };
        const queryResult = await trx.query(`
				SELECT ufacp."accessControlPointId" AS "id", acp."name", ufacp."read", ufacp."access" ,ufacp."remoteAccess",  ufacp."config", ufacp."grant", ufacp."snapshot"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}" ufacp
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
					ON acp.id = ufacp."accessControlPointId"
				WHERE ufacp."filterId" = $1
					AND acp."deletedAt" IS NULL
			`, [filterId]);
        result.accessControlPointMappings = queryResult.rows;
        return result;
    }
    async getAccessControlPointUserFilterMappings(organizationId, acpId, trx) {
        const result = {
            accessControlPoint: undefined,
            filters: [],
        };
        const queryResult = await trx.query(`
				SELECT  
						jsonb_agg(
							jsonb_build_object('id', uf.id,
												'read',ufacp."read", 
												'access', ufacp."access",
												'remoteAccess', ufacp."remoteAccess",
												'config', ufacp."config",
												'grant', ufacp."grant",
												'snapshot',ufacp."snapshot"
							)
						) as filters,
						ufacp."accessControlPointId" AS id,
						acp."name"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}" ufacp
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}" uf
					ON uf.id = ufacp."filterId"
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
					ON acp.id = ufacp."accessControlPointId"
				WHERE ufacp."accessControlPointId" = $1
					AND acp."deletedAt" IS NULL
				GROUP BY ufacp."accessControlPointId", acp.name
			`, [acpId]);
        if (queryResult.rows.length === 0) {
            return result;
        }
        result.filters = queryResult.rows[0].filters;
        result.accessControlPoint = { id: queryResult.rows[0].id, name: queryResult.rows[0].name };
        return result;
    }
    async updateUserFilterAccessControlPointMappings(organizationId, params, trx) {
        const accessControlPointIds = params.accessControlPoints.map((m) => m.id);
        if (accessControlPointIds.filter((item, index) => accessControlPointIds.indexOf(item) !== index)) {
        }
        const currentFilterAcpMappings = (await trx.query(`
				SELECT "accessControlPointId" AS "id", "read", "access", "remoteAccess", "config", "grant", "snapshot"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
				WHERE "filterId" = $1
			`, [params.filterId])).rows;
        const deletedAccessControlPoints = currentFilterAcpMappings.filter((elem) => !params.accessControlPoints.find((f) => f.id === elem.id));
        const insertedAccessControlPoints = params.accessControlPoints.filter((elem) => !currentFilterAcpMappings.find((f) => f.id === elem.id));
        const updatedAccessControlPoints = params.accessControlPoints.filter((elem) => currentFilterAcpMappings.find((f) => f.id === elem.id &&
            (f.access !== elem.access ||
                f.config !== elem.config ||
                f.grant !== elem.grant ||
                f.read !== elem.read ||
                f.remoteAccess !== elem.remoteAccess ||
                f.snapshot !== elem.snapshot)));
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
			("id", "filterId", "accessControlPointId", "read", "access", "remoteAccess", "config", "grant", "snapshot")
			SELECT gen_random_uuid(), $1, (sq->>'id')::UUID, (sq->>'read')::BOOLEAN, (sq->>'access')::BOOLEAN, (sq->>'remoteAccess')::BOOLEAN, (sq->>'config')::BOOLEAN, (sq->>'grant')::BOOLEAN, (sq->>'snapshot')::BOOLEAN
			FROM UNNEST ($2::JSONB[]) AS sq
			`, [params.filterId, insertedAccessControlPoints]);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
			WHERE "filterId" = $1 AND
				"accessControlPointId" = ANY($2::UUID[])
			`, [params.filterId, deletedAccessControlPoints.map((m) => m.id)]);
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}" AS ufacp
			SET "read" = (sq->>'read')::BOOLEAN,
				"access" = (sq->>'access')::BOOLEAN,
				"remoteAccess" = (sq->>'remoteAccess')::BOOLEAN,
				"config" = (sq->>'config')::BOOLEAN,
				"grant" = (sq->>'grant')::BOOLEAN,
				"snapshot" = (sq->>'snapshot')::BOOLEAN
			FROM UNNEST ($2::JSON[]) AS sq
			WHERE ufacp."filterId" = $1
				AND ufacp."accessControlPointId" = (sq->>'id')::UUID
			`, [params.filterId, updatedAccessControlPoints]);
    }
    async updateAccessControlPoinUserFiltertMappings(organizationId, params, trx) {
        const filterIds = params.filters.map((m) => m.id);
        if (filterIds.filter((item, index) => filterIds.indexOf(item) !== index)) {
        }
        const currentFilterAcpMappings = (await trx.query(`
				SELECT "filterId" AS "id", "read", "access", "remoteAccess", "config", "grant", "snapshot"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
				WHERE "accessControlPointId" = $1
			`, [params.acpId])).rows;
        const deletedFilters = currentFilterAcpMappings.filter((elem) => !params.filters.find((f) => f.id === elem.id));
        const insertedFilters = params.filters.filter((elem) => !currentFilterAcpMappings.find((f) => f.id === elem.id));
        const updatedFilters = params.filters.filter((elem) => currentFilterAcpMappings.find((f) => f.id === elem.id &&
            (f.access !== elem.access ||
                f.config !== elem.config ||
                f.grant !== elem.grant ||
                f.read !== elem.read ||
                f.remoteAccess !== elem.remoteAccess ||
                f.snapshot !== elem.snapshot)));
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
			("id", "filterId", "accessControlPointId", "read", "access", "remoteAccess", "config", "grant", "snapshot")
			SELECT gen_random_uuid(), (sq->>'id')::UUID,  $1, (sq->>'read')::BOOLEAN, (sq->>'access')::BOOLEAN, (sq->>'remoteAccess')::BOOLEAN, (sq->>'config')::BOOLEAN, (sq->>'grant')::BOOLEAN, (sq->>'snapshot')::BOOLEAN
			FROM UNNEST ($2::JSONB[]) AS sq
			`, [params.acpId, insertedFilters]);
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}" AS ufacp
			SET "read" = (sq->>'read')::BOOLEAN,
				"access" = (sq->>'access')::BOOLEAN,
				"remoteAccess" = (sq->>'remoteAccess')::BOOLEAN,
				"config" = (sq->>'config')::BOOLEAN,
				"grant" = (sq->>'grant')::BOOLEAN,
				"snapshot" = (sq->>'snapshot')::BOOLEAN
			FROM UNNEST ($2::JSON[]) AS sq
			WHERE ufacp."accessControlPointId" = $1
				AND ufacp."filterId" = (sq->>'id')::UUID
			`, [params.acpId, updatedFilters]);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}"
			WHERE "accessControlPointId" = $1 AND
				  "filterId" = ANY($2::UUID[])
			`, [params.acpId, deletedFilters.map((m) => m.id)]);
    }
    async getUserRightsOverUserFilters(organizationId, filterIds, requesterUserId, trx) {
        let query = `
			WITH user_rights AS (
				SELECT
					uar."accessControlPointId",
					BOOL_OR("read") as read,
					BOOL_OR("grant") as grant
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" uar
				WHERE "deletedAt" IS NULL AND "userId" = $1
				GROUP BY uar."accessControlPointId"
			)
			SELECT ufacp."filterId",
				BOOL_AND(COALESCE(ur.read, false)) as "read",
				BOOL_AND(COALESCE(ur.grant, false)) as "grant"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterAccessControlPoints}" ufacp
			LEFT JOIN user_rights ur
				ON ufacp."accessControlPointId" = ur."accessControlPointId"
			WHERE "ufacp"."filterId" = ANY($2)
			GROUP BY ufacp."filterId"
		`;
        const result = await trx.query(query, [requesterUserId, filterIds?.length > 0 ? filterIds : []]);
        return result.rows;
    }
}
exports.PSQLDalAccessUserFilter = PSQLDalAccessUserFilter;
