"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessRegion = void 0;
const uuid_1 = __importDefault(require("uuid"));
const dal_constants_1 = require("../../dal.constants");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_region_1 = require("../rdb/dal.access.rdb.region");
const dal_manager_1 = require("../../dal.manager");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const api_error_1 = require("../../../api/api.error");
const dal_access_psql_common_1 = require("./dal.access.psql.common");
const Cursor = require("pg-cursor");
class PSQLDalAccessRegion extends dal_access_rdb_region_1.RDBDalAccessRegion {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getRegionRedisPersistentData(params) {
        const { rows, rowCount } = await (params.trx || this._pgPool).query(`
		SELECT id, name as n, "visitorAccess" as v, "terminateVisitOnExit" t
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
		WHERE id = $1
	`, [params.regionId]);
        if (rowCount === 0) {
            return null;
        }
        return rows[0];
    }
    async getAllRegionRedisPersistentData(params) {
        const trxx = params.trx || (await this._pgPool.connect());
        const query = `
			  SELECT
			  id, name as n, "visitorAccess" v, "terminateVisitOnExit" t
			  FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
		  `;
        const cursor = trxx.query(new Cursor(query));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(100, async (err, r) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(r);
                        }
                    });
                });
                await params.onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error("Error while fetch credentials data with cursor!");
            }
            if (rows.length < 100) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            if (!params.trx) {
                trxx.release();
            }
        }
        catch (error) {
            if (!params.trx) {
                trxx?.release(error);
            }
            app_logs_1.logger.error(error);
        }
    }
    async upsertRegion(organizationId, region, trx) {
        if (region.antiPassback && (region.antiPassbackLockDuration === null || region.antiPassbackLockDuration === undefined)) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.REGION.ANTI_PASSBACK_LOCK_DURATION_NOT_SETTED");
        }
        const { rows: regionRows } = await trx.query(`
			SELECT id, name, "antiPassback", "antiPassbackLockDuration" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
			WHERE name = $1
		`, [region.name]);
        if (regionRows.length > 0 && region.id !== regionRows[0].id) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.REGION.NAME_ALREADY_INUSE");
        }
        if (region.id) {
            await this.updateRegion(organizationId, region, trx);
            if (region.antiPassback && !regionRows[0].antiPassback) {
                const regionAccessControlPointsDbResult = await this.getAccessControlPointsOfRegion(organizationId, {
                    regionId: region.id,
                    options: {
                        skip: 0,
                        take: 10000,
                    },
                }, trx);
                const regionRelatedTerminalIdsDbResult = await trx.query(`
					SELECT DISTINCT "deviceId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
					WHERE "deletedAt" IS NULL
						AND "deviceId" IS NOT NULL
						AND id = ANY($1::UUID[])
				`, [regionAccessControlPointsDbResult.items.map((m) => m.id)]);
                const now = new Date();
                for (const r of regionRelatedTerminalIdsDbResult.rows) {
                    const userIdsOfTerminal = await dal_manager_1.dbManager.accessDevice.getUserIdsOfTerminal(organizationId, r.deviceId, trx);
                    const antiPassbackStatesOfUsersForRegion = await this.listUserRegionStatesForTerminalUsers(organizationId, { userIds: userIdsOfTerminal, deviceIds: [r.deviceId] }, trx);
                    const changeItemsToInsert = antiPassbackStatesOfUsersForRegion.map((m) => {
                        return {
                            actionDateISO: now.toISOString(),
                            type: app_enums_1.enums.DeviceChangeItemType.RegionState,
                            data: {
                                actionDateISO: m.actionUtc.toISOString(),
                                regionId: region.id,
                                state: m.state,
                                userId: m.userId,
                                entranceLockExpirationUtc: m.entranceLockExpirationUtc ? m.entranceLockExpirationUtc.toISOString() : null,
                                exitLockExpirationUtc: m.exitLockExpirationUtc ? m.exitLockExpirationUtc.toISOString() : null,
                            },
                        };
                    });
                    const usersWithoutAnyStateRelationships = userIdsOfTerminal
                        .filter((f) => !antiPassbackStatesOfUsersForRegion.map((m) => m.userId).includes(f))
                        .map((m) => {
                        return {
                            actionDateISO: now.toISOString(),
                            type: app_enums_1.enums.DeviceChangeItemType.RegionState,
                            data: {
                                actionDateISO: now.toISOString(),
                                regionId: region.id,
                                state: app_enums_1.enums.libEnumsV2.AntiPassbackState.Unknown,
                                userId: m,
                                entranceLockExpirationUtc: null,
                                exitLockExpirationUtc: null,
                            },
                        };
                    });
                    changeItemsToInsert.push(...usersWithoutAnyStateRelationships);
                    await dal_manager_1.dbManager.accessDevice.addTerminalChangesBatch(organizationId, { deviceId: r.deviceId, changeItems: changeItemsToInsert }, trx);
                }
            }
        }
        else {
            region.id = await this.insertRegion(organizationId, region, trx);
        }
        return region;
    }
    async upsertRegionAccessControlPoints(organizationId, regionId, upsertItems) {
        await this.dbClient.transaction(async (trx) => {
            let addExists = upsertItems.addedIds && upsertItems.addedIds.length > 0;
            let removeExists = upsertItems.removedIds && upsertItems.removedIds.length > 0;
            if (!addExists && !removeExists) {
                return Promise.resolve([]);
            }
            let region = await this.dbClient
                .withSchema(organizationId)
                .transacting(trx)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions)
                .first("id")
                .where("id", regionId)
                .where("organizationId", organizationId);
            if (!region) {
                (0, dal_access_error_1.throwDbAccessNotFoundError)("The region is not found for the organization");
            }
            if (addExists) {
                let currentAccessControlPointIds = (await this.dbClient
                    .withSchema(organizationId)
                    .transacting(trx)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints)
                    .where("regionId", regionId)
                    .select("accessControlPointId")).map((rr) => rr.accessControlPointId);
                for (const newAcpId of upsertItems.addedIds) {
                    if (!currentAccessControlPointIds || currentAccessControlPointIds.indexOf(newAcpId) === -1) {
                        await this.dbClient.withSchema(organizationId).transacting(trx).table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints).insert({
                            id: uuid_1.default.v4(),
                            regionId: regionId,
                            accessControlPointId: newAcpId,
                        });
                    }
                }
            }
            if (removeExists) {
                await this.dbClient
                    .withSchema(organizationId)
                    .transacting(trx)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints)
                    .where("regionId", regionId)
                    .whereIn("accessControlPointId", upsertItems.removedIds)
                    .delete();
            }
        });
    }
    async insertRegion(organizationId, region, trx) {
        const id = uuid_1.default.v4();
        await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
			(id, name, "antiPassback", "organizationId", "antiPassbackLockDuration", "visitorAccess", 
			"terminateVisitOnExit", "deliverDeviceEmergencyToRegionDevices", "emergencyState", capacity)
		VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)`, [
            id,
            region.name,
            region.antiPassback,
            organizationId,
            region.antiPassbackLockDuration,
            true,
            region.terminateVisitOnExit,
            region.deliverDeviceEmergencyToRegionDevices,
            region.emergencyState,
            region.capacity,
        ]);
        return Promise.resolve(id);
    }
    async updateRegion(organizationId, region, trx) {
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
			SET name = $2,
				"antiPassback" = $3,
				"antiPassbackLockDuration" = $4,
				capacity = $5,
				"visitorAccess" = $6,
				"terminateVisitOnExit" = $7,
				"deliverDeviceEmergencyToRegionDevices" = $8,
				"emergencyState" = $9
			WHERE id = $1 `, [
            region.id,
            region.name,
            region.antiPassback,
            region.antiPassbackLockDuration,
            region.capacity,
            true,
            region.terminateVisitOnExit,
            region.deliverDeviceEmergencyToRegionDevices,
            region.emergencyState,
        ]);
    }
    async getOrganizationRegionRelatedDeviceIds(organizationId, trx) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("regionAccessControlPoints as racp")
            .innerJoin("accessControlPoints as acp", "racp.accessControlPointId", "acp.id")
            .whereNull("acp.deletedAt")
            .whereNotNull("acp.deviceId")
            .where("acp.organizationId", organizationId);
        if (trx)
            qb.transacting(trx);
        let panelIds = await qb.distinct("acp.deviceId");
        return Promise.resolve(panelIds ? panelIds.map((m) => m.deviceId) : []);
    }
    async getRegionDetailed(params) {
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(`
			SELECT id, name, "antiPassback", "antiPassbackLockDuration", capacity, "emergencyState", 
				"visitorAccess", "terminateVisitOnExit", "deliverDeviceEmergencyToRegionDevices"
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
			WHERE id = $1
		`, [params.regionId]);
        if (rowCount === 0) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.REGION.NOT_FOUND");
        }
        return rows[0];
    }
    async getRegionList(params) {
        let qx = 1;
        const qb = [];
        let qSelect = `
		SELECT r.id, r.name, r."antiPassback", r."antiPassbackLockDuration", r."emergencyState", r."capacity", count(acp.id) AS "accessControlPointCount", ra."userIds", ra."userRegionRights"
		
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" r
		LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints}" racp
			ON r.id = racp."regionId"
		LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
			ON acp.id = racp."accessControlPointId" AND acp."deletedAt" IS NULL
		LEFT JOIN
		(
			SELECT ra."regionId", array_agg(ra."userId") FILTER (WHERE ra."userId" IS NOT NULL) "userIds",
			jsonb_agg(jsonb_build_object('userId', ra."userId",
											'userRights', jsonb_build_object('read', ra.read, 'write', ra.write, 'notify', ra.notify)))
										FILTER (WHERE ra."userId" IS NOT NULL AND ra."userId" = $${qx++}) as "userRegionRights"
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" ra
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u ON u.id = ra."userId" AND u."deletedAt" IS NULL
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uo."deletedAt" IS NULL AND uo."isDisabled" = false AND uo."userId" = u.id
			GROUP BY ra."regionId"
		) ra ON ra."regionId" = r.id

		WHERE r.id = ANY($${qx++}::UUID[]) AND 
		($${qx++} = ANY("userIds"::UUID[]) OR "userIds" IS NULL)
		GROUP BY r.id, r.name, r."antiPassback", r."antiPassbackLockDuration", r."emergencyState", ra."regionId", ra."userIds", ra."userRegionRights"`;
        qb.push(params.requesterUserId, params.regionIds, params.requesterUserId);
        const total = await params.trx.query(`SELECT COUNT(DISTINCT q1.id)::SMALLINT AS c FROM (${qSelect}) q1`, qb);
        const result = {
            items: [],
            pagination: {
                total: total.rowCount > 0 ? total.rows[0].c : 0,
                skip: params.options.skip,
                take: params.options.take,
            },
        };
        qSelect += `
			OFFSET $${qx++}
			LIMIT $${qx++}
		`;
        qb.push(params.options.skip, params.options.take);
        if (result.pagination.total > 0) {
            const { rows } = await params.trx.query(qSelect, qb);
            result.items = rows.map((r) => {
                let userRights = r.userRegionRights?.find((urr) => urr.userId === params.requesterUserId)?.userRights;
                if (!userRights) {
                    userRights = {
                        notify: true,
                        read: true,
                        write: true,
                    };
                }
                return {
                    antiPassback: r.antiPassback,
                    name: r.name,
                    accessControlPointCount: r.accessControlPointCount,
                    antiPassbackLockDuration: r.antiPassbackLockDuration,
                    id: r.id,
                    capacity: r.capacity,
                    emergencyState: r.emergencyState,
                    userRights: userRights,
                };
            });
        }
        return result;
    }
    async getRegionIdName(organizationId, regionIds) {
        return this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).whereIn("id", regionIds).select("id", "name");
    }
    async removeRegion(params) {
        const region = await this.getRegionDetailed(params);
        if (!region) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.REGION.NOT_FOUND");
        }
        await params.trx.query(`DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
		    WHERE id = $1`, [params.regionId]);
    }
    async listRegions(params) {
        let result = {
            total: 0,
            items: [],
        };
        if (!params.options.regionIds?.length) {
            return result;
        }
        let qx = 1;
        const qParams = [];
        let qSelect = `
			SELECT id, name FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
			WHERE id = ANY ($${qx++}::UUID[])
		`;
        qParams.push(params.options.regionIds);
        result.total = (await params.trx.query(`SELECT COUNT(DISTINCT q1.id)::SMALLINT AS c FROM (${qSelect}) q1`, qParams)).rows[0].c;
        qSelect += `
			ORDER BY name
		`;
        if (params.options.skip) {
            qSelect += `
				OFFSET $${qx++}
			`;
            qParams.push(params.options.skip);
        }
        if (params.options.take) {
            qSelect += `
				LIMIT $${qx++}
			`;
            qParams.push(params.options.take);
        }
        const dbResult = (await params.trx.query(qSelect, qParams)).rows;
        result.items = dbResult.map((t) => {
            return {
                id: t.id,
                captionLines: [t.name],
            };
        });
        return Promise.resolve(result);
    }
    async listOrganizationRegions(organizationId, trx) {
        let qb = this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).where("organizationId", organizationId);
        if (trx)
            qb.transacting(trx);
        return await qb.select("id", "name", "antiPassback", "antiPassbackLockDuration", "visitorAccess", "terminateVisitOnExit");
    }
    async genericSearchRegion(params) {
        let result = {
            total: 0,
            items: [],
        };
        const filterExists = params.options.filter && params.options.filter.trim().length > 0;
        let regexList = [];
        let qx = 1;
        const qParams = [];
        const qFilterOrs = [];
        if (filterExists) {
            let filterTokens = params.options.filter.split(" ");
            for (const filterToken of filterTokens) {
                let token = filterToken.trim();
                regexList.push(new RegExp(params.options.filter.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&"), "ig"));
                if (token.length > 0) {
                    token = `name ilike '%${token}%'`;
                    qFilterOrs.push(token);
                }
            }
        }
        let qSelect = `
			SELECT id, name 
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" r
			LEFT JOIN
			(
				SELECT ra."regionId", array_agg(ra."userId") FILTER (WHERE ra."userId" IS NOT NULL) "userIds",
				jsonb_agg(jsonb_build_object('userId', ra."userId",
												'userRights', jsonb_build_object('read', ra.read, 'write', ra.write, 'notify', ra.notify)))
											FILTER (WHERE ra."userId" IS NOT NULL AND ra."userId" = $${qx}) as "userRegionRights"
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" ra
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u ON u.id = ra."userId" AND u."deletedAt" IS NULL
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uo."deletedAt" IS NULL AND uo."isDisabled" = false AND uo."userId" = u.id
				GROUP BY ra."regionId"
			) ra ON ra."regionId" = r.id
	
			WHERE ($${qx++} = ANY("userIds"::UUID[]) OR "userIds" IS NULL)
		`;
        qParams.push(params.requesterUserId);
        if (qFilterOrs.length) {
            let qOrs = qFilterOrs.join(` OR `);
            qSelect += `
				AND (${qOrs})
			`;
        }
        result.total = (await params.trx.query(`SELECT COUNT(DISTINCT q1.id)::SMALLINT AS c FROM (${qSelect}) q1`, qParams)).rows[0].c;
        qSelect += `
			ORDER BY name
			OFFSET $${qx++}
			LIMIT $${qx++}
		`;
        qParams.push(params.options.skip, params.options.take);
        result.items = (await params.trx.query(qSelect, qParams)).rows.map((r) => {
            return {
                id: r.id,
                captionLines: [r.name],
                matchItem: r.name,
            };
        });
        return Promise.resolve(result);
    }
    async getAccessControlPointsOfRegion(organizationId, params, trx) {
        let result = {
            total: 0,
            items: [],
        };
        let queryIndex = 1;
        const queryParams = [];
        let fromQuery = `
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints}" racp
			INNER JOIN  "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
				ON acp.id = racp."accessControlPointId"
			WHERE acp."deletedAt" IS NULL
				AND racp."regionId" = $${queryIndex++}
		`;
        queryParams.push(params.regionId);
        let filterExists = params.options.filter && params.options.filter.length > 0;
        let regexList = [];
        if (filterExists) {
            let filterTokens = params.options.filter.split(" ");
            const orConditions = [];
            for (const filterToken of filterTokens) {
                let token = filterToken.trim();
                regexList.push(new RegExp(params.options.filter.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&"), "ig"));
                if (token.length > 0) {
                    token = "%" + token + "%";
                    orConditions.push(` acp.name ilike $${queryIndex}`);
                    orConditions.push(` acp.location ilike $${queryIndex++}`);
                    queryParams.push(token);
                }
            }
            fromQuery += `
				AND (${orConditions.join(` OR `)})
			`;
        }
        result.total = (await trx.query(` SELECT COUNT(*)::SMALLINT AS c ${fromQuery} `, queryParams)).rows[0].c;
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        const selectQuery = `
			SELECT acp.id, acp.name, acp.location
			${fromQuery}
			ORDER BY acp.name ASC
			OFFSET $${queryIndex++} LIMIT $${queryIndex++}
		`;
        queryParams.push(params.options.skip);
        queryParams.push(params.options.take);
        const dbResult = await trx.query(selectQuery, queryParams);
        for (const r of dbResult.rows) {
            let captionLines = [r.name];
            if (r.location) {
                captionLines.push(r.location);
            }
            let matchItem = null;
            if (filterExists) {
                for (const regex of regexList) {
                    if (r.name.search(regex) >= 0) {
                        matchItem = r.name;
                        break;
                    }
                    else if (r.location.search(regex) >= 0) {
                        matchItem = r.location;
                        break;
                    }
                }
            }
            result.items.push({
                id: r.id,
                captionLines: captionLines,
                matchItem: matchItem,
            });
        }
        return Promise.resolve(result);
    }
    async listRegionUserStates(organizationId) {
        let regions = await this.listOrganizationRegions(organizationId);
        return this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("roles as r", "r.id", "uo.roleId")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .innerJoin("antiPassbackStates as aps", "uo.userId", "aps.userId")
            .innerJoin("accessControlPoints as acp", "acp.id", "aps.accessControlPointId")
            .where("uo.organizationId", organizationId)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt")
            .where("acp.organizationId", organizationId)
            .where("aps.state", dal_constants_1.DalConstants.AntiPassbackState.In)
            .whereIn("aps.regionId", regions.map((r) => r.id))
            .whereNull("acp.deletedAt")
            .select("aps.regionId", "uo.userId", "r.typeId");
    }
    async getRegionStateReportNew(organizationId, filter, requesterUserId) {
        let { query: userQuery, bindingKeys } = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: filter.userIds,
                userGroupIds: filter.userGroupIds,
                organizationUnitIds: filter.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: filter.applyOrganizationUnitFilterHierarchically,
                userOrganizationStatus: filter.status,
            },
            requiredOrganizationWidePermissions: ["n:r", "l:s", "i:b"],
            requiredOrganizationUnitWidePermissions: ["n:r", "l:s", "i:b"],
            bindingKeys: [filter.regionId],
            specificSelectItems: ["userId", "name", "surname", "uniqueId", "isDisabled"],
        });
        let query = `SELECT 
                uf."userId",
                aps.state, aps."actionUtc" as "lastActionUtc",
                aps."entranceLockExpirationUtc", aps."exitLockExpirationUtc",
                aps."accessControlPointId", acp.name as "accessControlPointName",
                (uf.name || ' ' || uf.surname) as "userFullname",
                json_build_object('uniqueId', uf."uniqueId", 'isDisabled', uf."isDisabled") as "userProfile"
                FROM (
					${userQuery}
				) uf
                LEFT JOIN "${organizationId}"."antiPassbackStates" AS aps 
                ON uf."userId" = aps."userId" AND aps."regionId" = $1
                LEFT JOIN "${organizationId}"."accessControlPoints" acp 
                ON acp.id = aps."accessControlPointId" AND acp."deletedAt" IS NULL
				WHERE `;
        query +=
            "(" +
                (filter.state || [null])
                    .map((s) => {
                    switch (s) {
                        case dal_constants_1.DalConstants.AntiPassbackState.Unknown:
                            bindingKeys.push(s);
                            return `aps.state = $${bindingKeys.length} OR aps.state IS NULL`;
                        case dal_constants_1.DalConstants.AntiPassbackState.In:
                        case dal_constants_1.DalConstants.AntiPassbackState.Out:
                            bindingKeys.push(s);
                            return `aps.state = $${bindingKeys.length}`;
                        case null:
                            return `TRUE`;
                    }
                })
                    .join(" OR ") +
                ")";
        let result = {
            pagination: {
                total: 0,
                take: filter.pagination.take,
                skip: filter.pagination.skip,
            },
            items: [],
        };
        result.pagination.total = parseInt((await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q1", bindingKeys)).rows[0].count);
        if (result.pagination.total > 0) {
            bindingKeys.push(filter.pagination.skip, filter.pagination.take);
            let sortType;
            let sortOrder = ["ASC", "DESC"].includes(filter.sortOrder) ? filter.sortOrder : "DESC";
            switch (filter.sortType) {
                case app_enums_1.enums.RegionReportSortType.LastAction:
                    sortType = ' "lastActionUtc" ';
                    break;
                case app_enums_1.enums.RegionReportSortType.AccessControlPointName:
                    sortType = ' "accessControlPointName" ';
                    break;
                case app_enums_1.enums.RegionReportSortType.State:
                    sortType = ' "aps"."state" ';
                    break;
                default:
                    sortType = ' "userFullname" ';
                    break;
            }
            query += `
			ORDER BY ${sortType} ${sortOrder} NULLS LAST`;
            query += `
            OFFSET $${bindingKeys.length - 1} 
            LIMIT $${bindingKeys.length}`;
            const { rows } = await this._pgPool.query(query, bindingKeys);
            result.items = rows.map((row) => {
                return {
                    user: {
                        id: row.userId,
                        fullname: row.userFullname,
                        uniqueId: row.userProfile.uniqueId,
                        organizationUnits: [],
                        userGroups: [],
                        isDisabled: row.userProfile.isDisabled,
                        isAnonymized: row.userProfile.isAnonymized,
                        hasUserAccount: false,
                        userCaptions: [],
                    },
                    lastActionUtc: row.lastActionUtc,
                    state: row.state || dal_constants_1.DalConstants.AntiPassbackState.Unknown,
                    entranceLockExpirationUtc: row.entranceLockExpirationUtc,
                    exitLockExpirationUtc: row.exitLockExpirationUtc,
                    accessControlPoint: row.accessControlPointId
                        ? {
                            id: row.accessControlPointId,
                            name: row.accessControlPointName,
                        }
                        : null,
                };
            });
        }
        if (filter.includeUserCaptions) {
            const userIds = result.items.map((u) => u.user.id);
            const userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
            for (let i = 0; i < result.items.length; i++) {
                const caption = userCaptions.find((u) => u.id == result.items[i].user.id);
                result.items[i].userCaptions = caption ? caption.captionLines : [];
            }
        }
        return result;
    }
    async getDurationInRegionReportOverview(organizationId, requesterUserId, trx, filter, reportFilter) {
        const selectedFilter = reportFilter ?? filter;
        let userQuery = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: selectedFilter.userIds,
                userGroupIds: selectedFilter.userGroupIds,
                organizationUnitIds: selectedFilter.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: selectedFilter.applyOrganizationUnitFilterHierarchically,
            },
            requiredOrganizationWidePermissions: ["n:r", "l:s", "i:b"],
            requiredOrganizationUnitWidePermissions: ["n:r", "l:s", "i:b"],
            bindingKeys: [selectedFilter.regionId, selectedFilter.range.start, selectedFilter.range.end],
            specificSelectItems: ["userId", "name", "surname"],
        });
        let sortField = "duration";
        if (selectedFilter.sortField === app_enums_1.enums.DurationInRegionSortType.UserName) {
            sortField = "username";
        }
        let sortType = "ASC";
        if (selectedFilter.sortType === app_enums_1.enums.SortType.Descending) {
            sortType = "DESC";
        }
        let query = `
		SELECT
			v_pairs."userId", SUM(EXTRACT(EPOCH FROM (upper(i_range) - lower(i_range)))) as duration
		FROM (
			SELECT *, tstzrange("actionUtc", "nextActionUtc") * tstzrange($2, $3) as i_range
			FROM (
				SELECT urs.*, uf.name || uf.surname as username,
					LAG(urs."id") OVER (PARTITION BY urs."userId", urs."regionId" ORDER BY urs."actionUtc" DESC) as "nextId",
					LAG("action") OVER (PARTITION BY urs."userId", urs."regionId" ORDER BY urs."actionUtc" DESC) as "nextAction",
					COALESCE(LAG(urs."actionUtc") OVER (PARTITION BY urs."userId", urs."regionId" ORDER BY urs."actionUtc" DESC), now()) as "nextActionUtc"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionStates}" urs
				INNER JOIN (
					${userQuery.query}
				) uf
					ON uf."userId" = urs."userId"
				WHERE "regionId" = $1`;
        if (!selectedFilter.includePasses) {
            query += ` AND action <> 0`;
        }
        query += `
			) all_pairs
			WHERE tstzrange("actionUtc", "nextActionUtc") && tstzrange($2, $3)
			AND ((all_pairs.action = 1 AND all_pairs."nextAction" = 2) -- entry -> exit
				OR (all_pairs.action = 1 AND all_pairs."nextAction" = 0) -- entry -> pass
				OR (all_pairs.action = 0 AND all_pairs."nextAction" = 2) -- pass -> exit
				OR (all_pairs.action = 1 AND all_pairs."nextAction" IS NULL)) -- last access entry
		) v_pairs
		GROUP BY "userId", "username"
		`;
        let result = reportFilter
            ? {
                pagination: {
                    total: parseInt((await trx.query("SELECT COUNT(*) FROM (" + query + ")q1", userQuery.bindingKeys)).rows[0].count),
                },
                region: (await trx.query(`SELECT id, name FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" WHERE id = $1`, [selectedFilter.regionId])).rows[0],
                items: [],
            }
            : {
                pagination: {
                    total: parseInt((await trx.query("SELECT COUNT(*) FROM (" + query + ")q1", userQuery.bindingKeys)).rows[0].count),
                    take: filter.pagination.take ?? 100,
                    skip: filter.pagination.skip ?? 0,
                },
                region: (await trx.query(`SELECT id, name FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" WHERE id = $1`, [filter.regionId])).rows[0],
                items: [],
            };
        if (result.pagination.total > 0) {
            if (reportFilter) {
                query += `
				ORDER BY ${sortField} ${sortType};`;
            }
            else {
                userQuery.bindingKeys.push(filter.pagination.skip ?? 0, filter.pagination.take ?? 100);
                query += `
				ORDER BY ${sortField} ${sortType}
				OFFSET $${userQuery.bindingKeys.length - 1} LIMIT $${userQuery.bindingKeys.length};`;
            }
            const { rows } = await trx.query(query, userQuery.bindingKeys);
            const userIds = rows.map((u) => u.userId);
            const userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
            result.items = rows.map((r) => {
                return {
                    user: userCaptions.find((uc) => uc.id === r.userId),
                    duration: r.duration,
                };
            });
        }
        return result;
    }
    async getDurationInRegionReportDetail(organizationId, filter, requesterUserId, trx) {
        let userQuery = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: [filter.userId],
            },
            requiredOrganizationWidePermissions: ["n:r", "l:s", "i:b"],
            requiredOrganizationUnitWidePermissions: ["n:r", "l:s", "i:b"],
            bindingKeys: [filter.regionId, filter.range.start, filter.range.end],
            specificSelectItems: ["userId"],
        });
        let sortField = "duration";
        if (filter.sortField === app_enums_1.enums.DurationInRegionDetailSortType.Date) {
            sortField = "actionUtc";
        }
        let sortType = "ASC";
        if (filter.sortType === app_enums_1.enums.SortType.Descending) {
            sortType = "DESC";
        }
        let query = `
		SELECT
			"accessLogId" as "startId",
			COALESCE(action, 0) as "startDirection",
			"actionUtc" as "startUtc",
			"accessControlPointId" as "acpId",
			"nextLogId" as "endId",
			COALESCE("nextAction", 0) as "endDirection",
			"nextActionUtc" as "endUtc",
			"nextAcpId" as "nextAcpId",
			CASE WHEN lower(i_range) = "actionUtc" THEN NULL ELSE lower(i_range) END as i_start,
			CASE WHEN upper(i_range) = "nextActionUtc" THEN NULL ELSE upper(i_range) END as i_end,
			EXTRACT(EPOCH FROM (upper(i_range) - lower(i_range))) as i_duration,
			EXTRACT(EPOCH FROM ("nextActionUtc" - "actionUtc")) as duration
		FROM (
			SELECT *, tstzrange("actionUtc", "nextActionUtc") * tstzrange($2, $3) as i_range
			FROM (
				SELECT urs.*,
					LAG(urs."accessLogId") OVER (PARTITION BY urs."userId", urs."regionId" ORDER BY urs."actionUtc" DESC) as "nextLogId",
					LAG("action") OVER (PARTITION BY urs."userId", urs."regionId" ORDER BY urs."actionUtc" DESC) as "nextAction",
					COALESCE(LAG(urs."actionUtc") OVER (PARTITION BY urs."userId", urs."regionId" ORDER BY urs."actionUtc" DESC), now()) as "nextActionUtc",
					LAG("accessControlPointId") OVER (PARTITION BY urs."userId", urs."regionId" ORDER BY urs."actionUtc" DESC) as "nextAcpId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionStates}" urs
				INNER JOIN (
					${userQuery.query}
				) uf
					ON uf."userId" = urs."userId"
				WHERE "regionId" = $1`;
        if (!filter.includePasses) {
            query += ` AND action <> 0`;
        }
        query += `
			) all_pairs
			WHERE tstzrange("actionUtc", "nextActionUtc") && tstzrange($2, $3)
			AND ((all_pairs.action = 1 AND all_pairs."nextAction" = 2) -- entry -> exit
				OR (all_pairs.action = 1 AND all_pairs."nextAction" = 0) -- entry -> pass
				OR (all_pairs.action = 0 AND all_pairs."nextAction" = 2) -- pass -> exit
				OR (all_pairs.action = 1 AND all_pairs."nextAction" IS NULL)) -- last access entry
		) v_pairs`;
        let result = {
            pagination: {
                total: parseInt((await trx.query("SELECT COUNT(*) FROM (" + query + ")q1", userQuery.bindingKeys)).rows[0].count),
                take: filter.pagination.take ?? 100,
                skip: filter.pagination.skip ?? 0,
            },
            region: (await trx.query(`SELECT id, name FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" WHERE id = $1`, [filter.regionId])).rows[0],
            user: (await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, [filter.userId]))[0],
            items: [],
        };
        if (result.pagination.total > 0) {
            userQuery.bindingKeys.push(filter.pagination.skip ?? 0, filter.pagination.take ?? 100);
            query += `
			ORDER BY "${sortField}" ${sortType}
			OFFSET $${userQuery.bindingKeys.length - 1} LIMIT $${userQuery.bindingKeys.length};`;
            const { rows } = await trx.query(query, userQuery.bindingKeys);
            let acps = [];
            rows.map((r) => {
                r.acpId && !acps.includes(r.acpId) && acps.push(r.acpId);
                r.nextAcpId && !acps.includes(r.nextAcpId) && acps.push(r.nextAcpId);
            });
            let acpDetails = [];
            if (acps.length > 0) {
                acpDetails = (await trx.query(`SELECT id, name FROM "${organizationId}"."accessControlPoints" WHERE id = ANY($1)`, [acps])).rows;
            }
            result.items = rows.map((r) => {
                return {
                    start: {
                        id: r.startId ?? undefined,
                        direction: r.startDirection,
                        actionUtc: r.startUtc,
                        intersectionUtc: r.i_start ?? undefined,
                        accessControlPoint: r.acpId ? acpDetails.find((a) => a.id === r.acpId) : undefined,
                    },
                    end: {
                        id: r.endId ?? undefined,
                        direction: r.endDirection,
                        actionUtc: r.endUtc,
                        intersectionUtc: r.i_end ?? undefined,
                        accessControlPoint: r.nextAcpId ? acpDetails.find((a) => a.id === r.nextAcpId) : undefined,
                    },
                    duration: r.duration,
                    intersectionDuration: r.duration !== r.i_duration ? r.i_duration : undefined,
                };
            });
        }
        return result;
    }
    async unlockUserStateForRegion(organizationId, params, trx) {
        await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.antiPassbackStates}" WHERE "userId" = $1 AND "regionId" = $2`, [params.userId, params.regionId]);
        await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionStates}"
			(id, "userId", "regionId", "accessLogId", "actionUtc", action, "accessControlPointId")
			VALUES (gen_random_uuid(), $1, $2, NULL, now(), $3, NULL);`, [params.userId, params.regionId, app_enums_1.enums.AntiPassbackState.Unknown]);
    }
    async unlockUserStateForAllRegions(organizationId, userId, trx) {
        const deletedStatesRegionIds = await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.antiPassbackStates}" WHERE "userId" = $1
			RETURNING "regionId"
			`, [userId]);
        if (deletedStatesRegionIds.rowCount) {
            await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionStates}"
					(id, "userId", "regionId", "accessLogId", "actionUtc", action, "accessControlPointId")
					SELECT gen_random_uuid(), $1, UNNEST($2::UUID[]), NULL, now(), $3, NULL;`, [userId, deletedStatesRegionIds.rows.map((m) => m.regionId) ?? [], app_enums_1.enums.AntiPassbackState.Unknown]);
        }
    }
    async getUserCurrentRegions(organizationId, userId) {
        return await this.dbClient
            .withSchema(organizationId)
            .from("antiPassbackStates as aps")
            .innerJoin("regions as r", "r.id", "aps.regionId")
            .where("aps.userId", userId)
            .where("r.organizationId", organizationId)
            .where("aps.state", dal_constants_1.DalConstants.RegionState.In)
            .select("r.name")
            .then((rows) => {
            if (rows)
                return rows.map((r) => r.name);
        });
    }
    async getUserCurrentRegionState(organizationId, userId, workPlanId) {
        let result = await this.dbClient
            .withSchema(organizationId)
            .from("antiPassbackStates as aps")
            .innerJoin("regions as r", "r.id", "aps.regionId")
            .innerJoin("workPlans as wp", "wp.regionId", "r.id")
            .where("aps.userId", userId)
            .where("r.organizationId", organizationId)
            .where("wp.id", workPlanId)
            .first("aps.state");
        if (result)
            return Promise.resolve(result.state);
        return Promise.resolve(null);
    }
    async listUsersCurrentRegionState(organizationId, userIds, workPlanId) {
        return await this.dbClient
            .withSchema(organizationId)
            .from("antiPassbackStates as aps")
            .innerJoin("regions as r", "r.id", "aps.regionId")
            .innerJoin("workPlans as wp", "wp.regionId", "r.id")
            .whereIn("aps.userId", userIds)
            .where("r.organizationId", organizationId)
            .where("wp.id", workPlanId)
            .select("aps.state", "aps.userId");
    }
    async getUserRegionLastState(organizationId, userId, regionId) {
        let result = await this.dbClient
            .withSchema(organizationId)
            .from("antiPassbackStates as aps")
            .innerJoin("regions as r", "r.id", "aps.regionId")
            .where("aps.userId", userId)
            .where("r.organizationId", organizationId)
            .where("aps.regionId", regionId)
            .first("aps.*")
            .orderBy("actionUtc", "desc");
        if (result)
            return Promise.resolve(result);
        return Promise.resolve(null);
    }
    async upsertUserRegionState(organizationId, regionState, trx) {
        let qb = this.dbClient.withSchema(organizationId).from("antiPassbackStates as aps").where("regionId", regionState.regionId).where("userId", regionState.userId).update({
            actionUtc: regionState.actionUtc,
            state: regionState.state,
        });
        if (trx)
            qb.transacting(trx);
        await qb;
    }
    async listAccessControlPointRegions(organizationId, accessControlPointId) {
        return (await this._pgPool.query(`SELECT DISTINCT
                r.id,
                r.name,
                r."antiPassback",
                r."antiPassbackLockDuration",
                r."visitorAccess",
                r."terminateVisitOnExit"
            FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.regions} r
            INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints}" racp
            ON r.id = racp."regionId"
            WHERE racp."accessControlPointId" = $1 `, [accessControlPointId])).rows;
    }
    async listUserRegionStatesForTerminalUsers(organizationId, params, trx) {
        let userAntiPassbackStatesDbResult = await trx.query(`
                SELECT aps."userId", aps."regionId", aps."actionUtc", aps."state",aps."entranceLockExpirationUtc", aps."exitLockExpirationUtc"
                FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.antiPassbackStates}" AS aps
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints}" AS racp
					ON racp."regionId" = aps."regionId"
                INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp 
					ON racp."accessControlPointId" = acp.id
                WHERE aps."userId" = ANY ($1::UUID[]) AND
                    acp."deletedAt" IS NULL AND
					acp."deviceId" = ANY($2::UUID[])
				GROUP BY aps."userId", aps."regionId", aps."actionUtc", aps."state",aps."entranceLockExpirationUtc", aps."exitLockExpirationUtc"
            `, [params.userIds, params.deviceIds]);
        return userAntiPassbackStatesDbResult.rows ? userAntiPassbackStatesDbResult.rows : [];
    }
    async getRegionTerminalIds(organizationId, regionId, trx) {
        const qFunction = async (trx) => {
            const { rows } = await trx.query(`
			SELECT DISTINCT(d.id) FROM "${organizationId}"."regionAccessControlPoints" racp
			INNER JOIN "${organizationId}"."accessControlPoints" acp
				ON racp."accessControlPointId" = acp.id
			INNER JOIN "${organizationId}".devices d
				ON acp."deviceId" = d.id
			WHERE racp."regionId" = $1
				AND d."organizationId" = $2`, [regionId, organizationId]);
            return rows.map((d) => d.id);
        };
        if (trx) {
            return await qFunction(trx);
        }
        else {
            return await dal_manager_1.dbManager.systemTransaction(qFunction);
        }
    }
    async setEmergencyForRegion(organizationId, regionId, state) {
        const { rows } = await this._pgPool.query(`
            UPDATE "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.regions} 
            SET "emergencyState" = $1
            WHERE id = $2
            AND "organizationId" = $3`, [state, regionId, organizationId]);
    }
    async upsertRegionAdministrator(organizationId, requestUserId, regionId, userRightParameters) {
        const userRegionRightId = uuid_1.default.v4();
        try {
            return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
                let writeUsers = (await trx.query(`
                SELECT array_agg("userId") as users FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" 
				WHERE "regionId" = $1 AND write`, [regionId])).rows[0].users;
                if (!writeUsers || writeUsers.length === 0) {
                    if (requestUserId !== userRightParameters.userId) {
                        await trx.query(` 
                        INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" as ra
                        ("id", "regionId", "userId", read, write, notify)
                        VALUES ($1, $2, $3, $4, $5, $6)
                        ON CONFLICT ON CONSTRAINT "region_administrators_regionId_userId_key" DO UPDATE
                        SET read = $4, write = $5, notify = $6
                        WHERE ra."regionId" = $2
                        AND ra."userId" = $3
                        `, [uuid_1.default.v4(), regionId, requestUserId, true, true, true]);
                    }
                    else {
                        userRightParameters.userRights.write = true;
                    }
                }
                if (writeUsers && writeUsers.length === 1 && writeUsers.includes(userRightParameters.userId) && !userRightParameters.userRights.write) {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.UNABLETOCHANGEREGIONRIGHTS");
                }
                const { rows } = await trx.query(` 
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" as ra
                ("id", "regionId", "userId", read, write, notify)
                VALUES ($1, $2, $3, $4, $5, $6)
                ON CONFLICT ON CONSTRAINT "region_administrators_regionId_userId_key" DO UPDATE
                SET read = $4, write = $5, notify = $6
                WHERE ra."regionId" = $2
                AND ra."userId" = $3
                `, [
                    userRegionRightId,
                    regionId,
                    userRightParameters.userId,
                    userRightParameters.userRights.read,
                    userRightParameters.userRights.write,
                    userRightParameters.userRights.notify,
                ]);
                return {
                    userRegionRightId: userRegionRightId,
                    userId: userRightParameters.userId,
                    userRights: userRightParameters.userRights,
                    userCaptions: await dal_manager_1.dbManager.accessUser.getSingleUserOrganizationCaptionLines(organizationId, userRightParameters.userId),
                };
            });
        }
        catch (error) {
            app_logs_1.logger.error("Error while upserting region administrator!");
            app_logs_1.logger.error(error);
            throw error;
        }
    }
    async removeRegionAdministrator(organizationId, userRegionRightId) {
        try {
            return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
                let users = (await trx.query(`
                SELECT 
                array_agg("userId") FILTER (WHERE write) as writers,
                array_agg("userId") FILTER (WHERE NOT write) as readers
                FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" WHERE "regionId" = (
                    SELECT "regionId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" WHERE id = $1
                ) GROUP BY "regionId"
                `, [userRegionRightId])).rows[0];
                let deletingRow = (await trx.query(`SELECT "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" 
						WHERE id = $1`, [userRegionRightId])).rows[0];
                if (users && users.writers && users.readers && deletingRow && users.writers.length === 1 && users.writers[0] === deletingRow.userId && users.readers.length > 0) {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.UNABLETOCHANGEREGIONRIGHTS");
                }
                await trx.query(`
                    DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}"
                    WHERE id = $1
                `, [userRegionRightId]);
            });
        }
        catch (error) {
            app_logs_1.logger.error("Error while removing region administrator!");
            app_logs_1.logger.error(error);
            throw error;
        }
    }
    async getUserRegionRights(params) {
        let query = `
        SELECT ra."regionId", array_agg(ra."userId") FILTER (WHERE ra."userId" IS NOT NULL) "userIds",
        jsonb_agg(jsonb_build_object('userId', ra."userId",
                                     'userRights', jsonb_build_object('read', ra.read, 'write', ra.write, 'notify', ra.notify)))
                                    FILTER (WHERE ra."userId" IS NOT NULL AND ra."userId" = $1) as "userRegionRights"
        FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" ra
        INNER JOIN "${params.organizationId}"."users" u ON u.id = ra."userId" AND u."deletedAt" IS NULL
        INNER JOIN "${params.organizationId}"."userOrganizations" uo ON uo."deletedAt" IS NULL AND uo."isDisabled" = false AND uo."organizationId" = $2 AND uo."userId" = u.id
        WHERE ra."regionId" = $3
        GROUP BY ra."regionId";
        `;
        let dbResult = await (params.trx ? params.trx : this.pgPool).query(query, [params.requestingUserId, params.organizationId, params.regionId]);
        let result = { read: true, write: true, notify: true };
        let row = dbResult.rows[0];
        if (row) {
            if (row.userIds.length === 0) {
                return result;
            }
            else if (row.userIds.includes(params.requestingUserId)) {
                result = row.userRegionRights.find((utr) => utr.userId === params.requestingUserId).userRights;
            }
            else {
                result = { read: false, write: false, notify: false };
            }
        }
        return result;
    }
    async getRegionIdForUserRegionRightId(organizationId, userRegionId) {
        let result = undefined;
        let dbResult = await this.pgPool.query(`
        SELECT "regionId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}" WHERE id = $1;`, [userRegionId]);
        if (dbResult.rows.length > 0) {
            result = dbResult.rows[0].regionId;
        }
        return result;
    }
    async getRegionAdministrators(organizationId, regionId, pagination) {
        let queryParams = [];
        let queryParamsIndex = 1;
        let query = ` 
            SELECT id, "userId", read, write, notify FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}"
            WHERE "regionId" = $${queryParamsIndex++}
        `;
        queryParams.push(regionId);
        const total = parseInt((await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q1", queryParams)).rows[0].count);
        if (pagination) {
            query += `
            OFFSET $${queryParamsIndex++}
            LIMIT $${queryParamsIndex++}`;
            queryParams.push(pagination.skip, pagination.take);
        }
        const { rows, rowCount } = await this._pgPool.query(query, queryParams);
        if (rowCount == 0) {
            return {
                pagination: {
                    total: total,
                    skip: pagination ? pagination.skip : undefined,
                    take: pagination ? pagination.take : undefined,
                },
                items: [],
            };
        }
        const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, rows.map((r) => r.userId));
        return {
            pagination: {
                total: total,
                skip: pagination ? pagination.skip : undefined,
                take: pagination ? pagination.take : undefined,
            },
            items: rows.map((item) => {
                return {
                    userRegionRightId: item.id,
                    userId: item.userId,
                    userCaptions: userCaptionList.find((uc) => uc.id === item.userId)?.captionLines,
                    userRights: {
                        read: item.read,
                        write: item.write,
                        notify: item.notify,
                    },
                };
            }),
        };
    }
    async getAdministratorsOfRegions(params) {
        let queryParams = [];
        let queryParamsIndex = 1;
        let query = ` 
            SELECT "regionId" as id, jsonb_agg(jsonb_build_object('userId', "userId",
				'userRights', jsonb_build_object('read', read, 'write', write, 'notify', notify))) as "userRegionRights"
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators}"
            WHERE "regionId" = ANY($${queryParamsIndex++}::uuid[])
			GROUP BY "regionId"
        `;
        queryParams.push(params.regionIds);
        const { rows, rowCount } = await params.trx.query(query, queryParams);
        if (rowCount == 0) {
            return [];
        }
        return rows;
    }
    async getRegionStateSummaryForDashboard(organizationId, filter, requesterUserId, trx) {
        let { query: userQuery, bindingKeys } = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: filter.userIds,
                userGroupIds: filter.userGroupIds,
                organizationUnitIds: filter.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: filter.applyOrganizationUnitFilterHierarchically,
                userOrganizationStatus: filter.status,
            },
            requiredOrganizationWidePermissions: ["n:r", "l:s", "i:b"],
            requiredOrganizationUnitWidePermissions: ["n:r", "l:s", "i:b"],
            bindingKeys: [filter.regionId, filter.state, filter.pagination?.skip ?? 0, filter.pagination?.take ?? 10],
            specificSelectItems: ["userId"],
        });
        const dbResult = await trx.query(`WITH region_users AS (
				SELECT uf."userId", 
					CASE WHEN aps."state" IS NOT NULL THEN aps."state" ELSE 0 END AS "state", -- users with unknown state 
					aps."actionUtc"
				FROM (
					${userQuery}
				) uf
				LEFT JOIN  "${organizationId}"."antiPassbackStates" aps
					ON uf."userId" = aps."userId" AND aps."regionId" = $1
			)
			SELECT
				state, COUNT(*),
				(
					SELECT jsonb_agg(to_jsonb(sq) ORDER BY "actionUtc" DESC NULLS LAST)
					FROM (
						SELECT "userId", state, "actionUtc"
						FROM region_users
						WHERE state = ANY($2)
						ORDER BY "actionUtc" DESC NULLS LAST
						OFFSET $3 LIMIT $4
					) sq
				) as users
			FROM region_users
			GROUP BY state`, bindingKeys);
        let userCaptions = null;
        const userIds = (dbResult.rows[0]?.users ?? []).map((r) => r.userId);
        if (userIds.length > 0) {
            userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        const regionDetail = await dal_manager_1.dbManager.accessRegion.getRegionDetailed({ organizationId, regionId: filter.regionId, trx });
        return {
            detail: regionDetail,
            states: dbResult.rows.map((r) => {
                return {
                    state: r.state,
                    count: r.count,
                };
            }),
            users: (dbResult.rows[0]?.users ?? []).map((r) => {
                return {
                    user: {
                        id: r.userId,
                        userCaptions: userCaptions.find((uc) => uc.id === r.userId).captionLines,
                    },
                    state: r.state,
                    actionUtc: r.actionUtc,
                };
            }),
        };
    }
    async getAccessRuleSetIdNamePairsOfRegion(organizationId, regionId, trx) {
        const dbResult = await trx.query(`
			SELECT id, name
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleSets}"
			WHERE "regionId" = $1
				AND "deletedAt" IS NULL
		`, [regionId]);
        return dbResult.rows;
    }
    async getRegionTicketIdNamePairsOfRegion(organizationId, regionId, trx) {
        const dbResult = await trx.query(`
			SELECT id, name
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets}"
			WHERE "regionId" = $1
				AND "deletedAt" IS NULL
		`, [regionId]);
        return dbResult.rows;
    }
    async isRegionUsedForAccessNotifications(organizationId, regionId, trx) {
        const dbResult = await trx.query(`
			SELECT COUNT(*)::SMALLINT AS c
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notificationAccess}"
			WHERE "regionId" = $1
		`, [regionId]);
        return dbResult.rows[0].c > 0;
    }
    async isRegionUsedForForbiddances(organizationId, regionId, trx) {
        const dbResult = await trx.query(`
			SELECT COUNT(*)::SMALLINT AS c
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}"
			WHERE "regionId" = $1
				AND "deletedAt" IS NULL
		`, [regionId]);
        return dbResult.rows[0].c > 0;
    }
}
exports.PSQLDalAccessRegion = PSQLDalAccessRegion;
