"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkAutoShiftRuleExistency = exports.PSQLDalAccessPacs2 = void 0;
const cluster = __importStar(require("cluster"));
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const moment_range_1 = require("moment-range");
const uuid = __importStar(require("uuid"));
const api_error_1 = require("../../../api/api.error");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const business_hooks_1 = require("../../../business/business.hooks");
const business_pacs_ppermission_1 = require("../../../business/pacs/business.pacs.ppermission");
const dal_constants_1 = require("../../dal.constants");
const dal_logger_1 = require("../../dal.logger");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_pacs2_1 = require("../rdb/dal.access.rdb.pacs2");
const cli_queries_1 = require("./cli-queries");
const dal_access_psql_common_1 = require("./dal.access.psql.common");
const moment = require("moment");
const Cursor = require("pg-cursor");
class PSQLDalAccessPacs2 extends dal_access_rdb_pacs2_1.RDBDalAccessPacs2 {
    constructor(knex, pgPool, listenPacsDbNotifications) {
        super(knex, pgPool);
        if (cluster.isMaster && listenPacsDbNotifications) {
            this.listenDbNotifications();
        }
    }
    async listenDbNotifications() {
        const client = await (0, dal_utils_1.listenClientWrapperWithRetry)("PACS", dal_manager_1.dbManager.poolMain);
        client.on("notification", (async (msg) => {
            switch (msg.channel) {
                case "pacs_live_recalculation":
                    if (this._liveRecalculationListener) {
                        this._liveRecalculationListener(msg.payload);
                    }
                    break;
                default:
                    break;
            }
        }).bind(this));
        client.on("error", (err) => {
            app_logs_1.logger.error("LISTEN PACS pg_notif connection error: " + err);
        });
        client.on("end", () => {
            app_logs_1.logger.debug("LISTEN PACS pg_notif connection terminated... Restarting...");
            client.release();
            this.listenDbNotifications();
        });
        client.query("LISTEN pacs_live_recalculation");
        app_logs_1.logger.info("LISTEN PACS connections started.");
    }
    registerLiveRecalculationListener(onLiveRecalculationTriggered) {
        if (cluster.isMaster) {
            this._liveRecalculationListener = onLiveRecalculationTriggered;
        }
    }
    concatIdsForWhereInRawQuery(ids) {
        return `(${ids
            .map((id) => {
            return `'${id}'`;
        })
            .join(",")})`;
    }
    isInPostgresRange(date, range) {
        const [start, end] = range
            .slice(1, -1)
            .split(",")
            .map((s) => s.trim().replace(/^"|"$/g, ""));
        const startDate = start ? new Date(start) : null;
        const endDate = end ? new Date(end) : null;
        return (!startDate || (range[0] === "[" ? date >= startDate : date > startDate)) && (!endDate || (range.slice(-1) === "]" ? date <= endDate : date < endDate));
    }
    async getApprovedPPermissionsOfUserAtDate(organizationId, userId, date) {
        let result = [];
        try {
            let query = `
            SELECT	"ppermissionId", 
					"ppermissionTypeId", 
					"startDateTime", 
					"endDateTime", 
					"lastApprovementTime"
			FROM
            (
                SELECT pp.id as "ppermissionId", pp."ppermissionTypeId", upp."userId",pp."organizationId", pp."startDateTime", pp."endDateTime",
                (
                    SELECT MAX(uppa2."approvementDate")  
					FROM "${organizationId}"."userPPermissionApprovements" uppa2 
					WHERE uppa2."ppermissionId" = pp.id AND uppa2."userId" = upp."userId" 
					GROUP BY uppa2."userId"
                ) as "lastApprovementTime"
                FROM "${organizationId}"."userPPermissions" upp
                INNER JOIN "${organizationId}"."ppermissions" pp ON upp."ppermissionId" = pp.id AND pp.status = 1
                INNER JOIN "${organizationId}"."ppermissionTypes" as pt on pt."id" = PP."ppermissionTypeId"
                WHERE pt."deletedAt" IS NULL
            ) subq
            WHERE "lastApprovementTime" < $1::timestamp with time zone
            AND "userId" = $2 AND "organizationId" = $3
            `;
            let bindings = [date, userId, organizationId];
            let dbResult = await this.pgPool.query(query, bindings);
            result = dbResult.rows;
        }
        catch (error) {
            dal_logger_1.dbLogger.error("Could not list permission usages of user: " + userId + " because: " + error);
        }
        return Promise.resolve(result);
    }
    async getUsersApprovedPPermissionsAtDate(organizationId, userIds, date, trx) {
        const result = (await trx.query(`
				SELECT	"userId",
						json_agg(json_build_object(
							'ppermissionId', "ppermissionId",
							'ppermissionTypeId', "ppermissionTypeId",
							'startDateTime', "startDateTime",
							'endDateTime', "endDateTime",
							'lastApprovementTime', "lastApprovementTime"
						)) as "approvedPPermission"
				FROM
				(
					SELECT pp.id as "ppermissionId", pp."ppermissionTypeId", upp."userId",pp."organizationId", pp."startDateTime", pp."endDateTime",
					(
						SELECT MAX(uppa2."approvementDate")  
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements}" uppa2 
						WHERE uppa2."ppermissionId" = pp.id AND uppa2."userId" = upp."userId" 
						GROUP BY uppa2."userId"
					) as "lastApprovementTime"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" upp
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" pp ON upp."ppermissionId" = pp.id AND pp.status = 1
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" as pt on pt."id" = PP."ppermissionTypeId"
					WHERE pt."deletedAt" IS NULL
				) subq
				WHERE "lastApprovementTime" < $1::timestamp with time zone
					AND "userId" = ANY($2::uuid[])
				GROUP BY "userId"
 				;`, [date, userIds])).rows;
        return result;
    }
    findDayItem(date, periodStart, routineType, dayItems) {
        let result = null;
        let mdate = moment(date).startOf("day");
        let mperiodStart = moment(periodStart).startOf("day");
        if (routineType === dal_constants_1.DalConstants.PacsWorkplanPeriodRoutineTypes.Week) {
            result = dayItems.find((di) => {
                return di.index === mdate.isoWeekday();
            });
        }
        else if (routineType === dal_constants_1.DalConstants.PacsWorkplanPeriodRoutineTypes.Day && mdate.isSameOrAfter(mperiodStart, "day")) {
            let periodLength = dayItems.length;
            let days = mdate.diff(mperiodStart, "days");
            let periodCount = Math.floor(days / periodLength);
            let dayIndex = days - periodLength * periodCount + 1;
            result = dayItems.find((di) => {
                return di.index === dayIndex;
            });
        }
        return result;
    }
    async getEmployeeWorkPlan(trx, organizationId, userId, dateRange) {
        let result = {
            userId: userId,
            userCaptions: (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId })).caption,
            range: dateRange,
            workingSchedule: [],
            memberships: [],
            holidays: [],
            ppermissions: [],
        };
        let workingHoursDbResult = await trx.query(`
		SELECT days.date,
		days.range * wpp.range * tstzrange(uwp."startDateTime", uwp."endDateTime") as intersection,
		uwp."workPlanId", wpp.id as "workPlanPeriodId",
		wp.name as "workPlanName", wp.type as "workPlanType", wp."colorCode",
		wpp."details",
		lower(wpp.range) as "periodStartDateTime"
		FROM (
			SELECT days::date as date, tstzrange(days, days + '1 day'::interval) as range FROM generate_series( $1::timestamp, $2::timestamp - '1 day'::interval, '1 day'::interval) days
			) days
		INNER JOIN "${organizationId}"."userWorkPlans" uwp ON uwp."userId" = $3
			AND tstzrange(uwp."startDateTime", uwp."endDateTime") && days.range
		INNER JOIN "${organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = uwp."workPlanId" AND wpp.range && days.range
		INNER JOIN "${organizationId}"."workPlans" wp ON wp.id = wpp."workPlanId"
		WHERE wp."organizationId" = $4
		ORDER BY date ASC
		`, [dateRange.startDateTime, dateRange.endDateTime, userId, organizationId]);
        for (const row of workingHoursDbResult.rows) {
            let dateWorkingHours = undefined;
            if (result.workingSchedule.length > 0 && moment(result.workingSchedule[result.workingSchedule.length - 1].date).isSame(moment(row.date), "day")) {
                dateWorkingHours = result.workingSchedule[result.workingSchedule.length - 1];
            }
            else {
                dateWorkingHours = {
                    date: row.date,
                    workingHours: [],
                };
                result.workingSchedule.push(dateWorkingHours);
            }
            let range = row.intersection.replace(/[\[\"\)]/g, ``).split(",");
            let intersection = new moment_range_1.DateRange(range[0], range[1]);
            switch (row.workPlanType) {
                case dal_constants_1.DalConstants.WorkPlanType.REGULAR:
                case dal_constants_1.DalConstants.WorkPlanType.HALF_FLEXIBLE:
                    {
                        let dayItem = this.findDayItem(dateWorkingHours.date, new Date(row.periodStartDateTime), row.details.routineType, row.details.dayItems);
                        if (dayItem && dayItem.workingHours) {
                            dayItem.workingHours.forEach((wh) => {
                                let dayItemRange = new moment_range_1.DateRange(moment(dateWorkingHours.date)
                                    .set("hours", parseInt(wh.startTime.substr(0, 2)))
                                    .set("minutes", parseInt(wh.startTime.substr(2, 2))), moment(dateWorkingHours.date)
                                    .set("hours", parseInt(wh.endTime.substr(0, 2)))
                                    .set("minutes", parseInt(wh.endTime.substr(2, 2))));
                                if (dayItemRange.overlaps(intersection)) {
                                    let drintersection = dayItemRange.intersect(intersection);
                                    dateWorkingHours.workingHours.push({
                                        workPlanId: row.workPlanId,
                                        workPlanName: row.workPlanName,
                                        colorCode: row.colorCode,
                                        workPlanPeriodId: row.workPlanPeriodId,
                                        periodStartDateTime: row.periodStartDateTime,
                                        range: {
                                            startDateTime: drintersection.start.toDate(),
                                            endDateTime: drintersection.end.toDate(),
                                        },
                                    });
                                }
                            });
                        }
                    }
                    break;
                case dal_constants_1.DalConstants.WorkPlanType.SHIFT:
                    {
                        const periodLength = row.details.workDuration + row.details.restDuration;
                        let range = new moment_range_1.DateRange(moment(row.periodStartDateTime), moment(row.periodStartDateTime).add(row.details.workDuration, "minutes"));
                        while (range.start.isBefore(intersection.end)) {
                            let shiftIntersection = range.intersect(intersection);
                            if (shiftIntersection) {
                                dateWorkingHours.workingHours.push({
                                    workPlanId: row.workPlanId,
                                    workPlanName: row.workPlanName,
                                    workPlanPeriodId: row.workPlanPeriodId,
                                    periodStartDateTime: row.periodStartDateTime,
                                    range: {
                                        startDateTime: shiftIntersection.start.toDate(),
                                        endDateTime: shiftIntersection.end.toDate(),
                                    },
                                });
                            }
                            range.start = range.start.add(periodLength, "minutes");
                            range.end = range.end.add(periodLength, "minutes");
                        }
                    }
                    break;
            }
        }
        let membershipDbResults = await trx.query(`
		SELECT uwp.id, wp."id" as "workPlanId", wp.name, wp."colorCode", uwp."startDateTime", uwp."endDateTime",
		tstzrange(uwp."startDateTime", uwp."endDateTime") * tstzrange($1,$2) as intersection
		FROM "${organizationId}"."userWorkPlans" uwp
		INNER JOIN "${organizationId}"."workPlans" wp ON wp.id = uwp."workPlanId"
		WHERE uwp."userId" = $3 AND
		wp."organizationId" = $4 AND
		tstzrange(uwp."startDateTime", uwp."endDateTime") && tstzrange($5,$6)
		AND wp."deletedAt" IS NULL
		ORDER BY "startDateTime";
		`, [dateRange.startDateTime, dateRange.endDateTime, userId, organizationId, dateRange.startDateTime, dateRange.endDateTime]);
        for (const row of membershipDbResults.rows) {
            let intersectingRange = row.intersection.replace(/[\[\"\)]/g, ``).split(",");
            let membership = result.memberships.find((m) => m.workPlanId === row.workPlanId);
            if (!membership) {
                membership = {
                    workPlanId: row.workPlanId,
                    workPlanName: row.name,
                    colorCode: row.colorCode,
                    ranges: [],
                };
                result.memberships.push(membership);
            }
            membership.ranges.push({
                userWorkPlanId: row.id,
                originalRange: {
                    startDateTime: row.startDateTime,
                    endDateTime: row.endDateTime,
                },
                intersectingRange: {
                    startDateTime: moment(intersectingRange[0]).toDate(),
                    endDateTime: moment(intersectingRange[1]).toDate(),
                },
            });
        }
        let holidayDbResults = await trx.query(`
		SELECT vac.id, vac.title, vac."startDateTime", vac."endDateTime",
		to_jsonb(array_agg(json_build_object('workPlanId', subq."workPlanId", 'workPlanName', subq.name, 'intersection', subq.intersection*tstzrange(vac."startDateTime", vac."endDateTime")))) as "wpIntersections"
		FROM "${organizationId}"."vacations" vac INNER JOIN
		(
			SELECT uwp."workPlanId", wp.name,
			tstzrange(uwp."startDateTime", uwp."endDateTime") * tstzrange($1,$2) as intersection
			FROM "${organizationId}"."userWorkPlans" uwp
			INNER JOIN "${organizationId}"."workPlans" wp ON wp.id = uwp."workPlanId"
			WHERE uwp."userId" = $3 AND
			wp."organizationId" = $4 AND
			tstzrange(uwp."startDateTime", uwp."endDateTime") && tstzrange($5,$6)
			AND wp."deletedAt" IS NULL AND (wp."ignoreHolidays" IS NULL OR wp."ignoreHolidays" = false)
		) subq
		ON tstzrange(vac."startDateTime", vac."endDateTime") && subq.intersection
		WHERE vac."organizationId" = $7
		GROUP BY vac.id, vac.title, vac."startDateTime", vac."endDateTime"
		`, [dateRange.startDateTime, dateRange.endDateTime, userId, organizationId, dateRange.startDateTime, dateRange.endDateTime, organizationId]);
        for (const row of holidayDbResults.rows) {
            result.holidays.push({
                id: row.id,
                name: row.title,
                originalRange: {
                    startDateTime: row.startDateTime,
                    endDateTime: row.endDateTime,
                },
                workPlanIntersections: row.wpIntersections.map((wpi) => {
                    let intersectingRange = wpi.intersection.replace(/[\[\"\)]/g, ``).split(",");
                    return {
                        workPlanId: wpi.workPlanId,
                        workPlanName: wpi.workPlanName,
                        intersectingRange: {
                            startDateTime: moment(intersectingRange[0]).toDate(),
                            endDateTime: moment(intersectingRange[1]).toDate(),
                        },
                    };
                }),
            });
        }
        let permissionDbResults = await trx.query(`
		SELECT p.id, pt.name, p."startDateTime", p."endDateTime", tstzrange(p."startDateTime", p."endDateTime") * tstzrange($1,$2) as "intersectingRange", (
			SELECT COUNT(*) FROM "${organizationId}"."userPPermissionApprovements" uppa
			WHERE uppa."ppermissionId" = p.id AND uppa."userId" = $3
			AND (uppa."approverUserId" IS NULL OR uppa.status = false)
		) = 0 as approved
		FROM "${organizationId}"."ppermissions" p 
		INNER JOIN "${organizationId}"."userPPermissions" upp ON upp."ppermissionId" = p.id AND upp."userId" = $3
		INNER JOIN "${organizationId}"."ppermissionTypes" pt ON pt.id = p."ppermissionTypeId"
		WHERE tstzrange(p."startDateTime", p."endDateTime") && tstzrange($1,$2)
		`, [dateRange.startDateTime, dateRange.endDateTime, userId]);
        for (const row of permissionDbResults.rows) {
            if (row.approved) {
                let intersectingRange = row.intersectingRange.replace(/[\[\"\)]/g, ``).split(",");
                result.ppermissions.push({
                    ppermissionId: row.id,
                    ppermissionTypeName: row.name,
                    originalRange: {
                        startDateTime: row.startDateTime,
                        endDateTime: row.endDateTime,
                    },
                    intersectingRange: {
                        startDateTime: moment(intersectingRange[0]).toDate(),
                        endDateTime: moment(intersectingRange[1]).toDate(),
                    },
                });
            }
        }
        return result;
    }
    async getPlanningCalendar(trx, organizationId, requestUserId, args) {
        let result = {
            pagination: {
                take: args.pagination.take,
                skip: args.pagination.skip,
                total: 0,
            },
            range: args.range,
            items: [],
        };
        let userQuery = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requestUserId,
            requiredOrganizationUnitWidePermissions: ["j:l"],
            requiredOrganizationWidePermissions: ["j:l"],
            bindingKeys: [],
            idBasedUserFilter: {
                applyOrganizationUnitFilterHierarchically: args.applyOrganizationUnitFilterHierarchically,
                userIds: args.userIds,
                organizationUnitIds: args.organizationUnitIds,
                userGroupIds: args.userGroupIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
            },
            specificSelectItems: ["userId", "name", "surname"],
        });
        let bindings = userQuery.bindingKeys;
        let query = `
		SELECT users."userId" FROM (
			${userQuery.query}
		) users
		LEFT JOIN "${organizationId}"."userWorkPlans" __uwp
			ON __uwp."userId" = users."userId"`;
        if (args.workPlanIds && args.workPlanIds.length > 0) {
            let bindCounter = bindings.length;
            bindings.push(args.workPlanIds, args.range.startDateTime, args.range.endDateTime);
            query += `
		WHERE __uwp."workPlanId" = ANY($${++bindCounter}) AND 
			tstzrange(__uwp."startDateTime", __uwp."endDateTime") && tstzrange($${++bindCounter}, $${++bindCounter})`;
        }
        query += `
		GROUP BY users."userId", users.name || ' ' || users.surname`;
        const count = await trx.query(`SELECT COUNT(*) as count FROM (` + query + `) sq`, bindings);
        result.pagination.total = parseInt(count.rows[0].count);
        query += `
		ORDER BY users.name || ' ' || users.surname ASC`;
        if (args.pagination.skip) {
            bindings.push(args.pagination.skip);
            query += `
		OFFSET $${bindings.length}`;
        }
        if (args.pagination.take) {
            bindings.push(args.pagination.take);
            query += `
		LIMIT $${bindings.length}`;
        }
        const userIds = (await trx.query(query, bindings)).rows.map((r) => r.userId);
        for (const userId of userIds) {
            const userResult = await this.getEmployeeWorkPlan(trx, organizationId, userId, args.range);
            result.items.push(userResult);
        }
        return result;
    }
    async workplanEmployeeListAtDate(organizationId, workplanId, timestamp, pagination, filter) {
        return this.dbClient.transaction(async (trx) => {
            let result = {
                pagination: {
                    take: pagination ? pagination.take : undefined,
                    skip: pagination ? pagination.skip : undefined,
                    total: 0,
                },
                items: [],
            };
            let qParams = [workplanId, organizationId, timestamp];
            let filterWhereQ = "";
            if (filter) {
                filterWhereQ = ` AND (unaccent(upper(uop."name")) ilike unaccent(upper(?)) OR 
								unaccent(upper(uop."surname")) ilike unaccent(upper(?)) OR 
								unaccent(upper(uop."uniqueId")) ilike unaccent(upper(?)) OR 
								(unaccent(upper(uop."name")) || ' ' || unaccent(upper(uop."surname"))) ILIKE unaccent(upper(?)))`;
                const likeFilter = `%${filter}%`;
                qParams.push(likeFilter, likeFilter, likeFilter, likeFilter);
            }
            let query = `
            SELECT uwp.id, uop."name" || ' ' || uop."surname" as name, uwp."userId", uwp."startDateTime", uwp."endDateTime"
            FROM "${organizationId}"."userWorkPlans" uwp
            INNER JOIN "${organizationId}"."workPlans" wp ON wp.id = uwp."workPlanId" AND wp.id = ? AND wp."organizationId" = ?
            INNER JOIN "${organizationId}"."users" u ON u.id = uwp."userId" AND u."deletedAt" IS NULL
            INNER JOIN "${organizationId}"."userOrganizations" uo
                ON uo."deletedAt" IS NULL AND uo."isDisabled" = false AND uo."userId" = u.id
                AND uo."organizationId" = wp."organizationId"
            INNER JOIN "${organizationId}"."userOrganizationProfiles" uop ON uop."userOrganizationId" = uo.id and uop."deletedAt" IS NULL
            WHERE tstzrange(uwp."startDateTime", uwp."endDateTime") @> ?::timestamp with time zone
			${filterWhereQ}
            ORDER BY uop."name" || uop."surname" ASC NULLS FIRST
            `;
            result.pagination.total = parseInt((await trx.raw(`SELECT COUNT(*) as count FROM (${query}) sq`, qParams)).rows[0].count);
            if (!pagination) {
                let userList = await trx.raw(query, qParams);
                for (const row of userList.rows) {
                    result.items.push({
                        id: row.id,
                        userId: row.userId,
                        name: row.name,
                        captionLines: [],
                        range: {
                            startDateTime: row.startDateTime,
                            endDateTime: row.endDateTime,
                        },
                    });
                }
            }
            else if (pagination.take && pagination.take > 0) {
                let userList = await trx.raw(query + "OFFSET ? LIMIT ?", [...qParams, pagination.skip ? pagination.skip : 0, pagination.take]);
                let captionLines = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userList.rows.map((u) => u.userId), trx);
                for (const row of userList.rows) {
                    result.items.push({
                        id: row.id,
                        userId: row.userId,
                        name: row.name,
                        captionLines: captionLines.find((cl) => cl.id === row.userId)?.captionLines,
                        range: {
                            startDateTime: row.startDateTime,
                            endDateTime: row.endDateTime,
                        },
                    });
                }
            }
            return result;
        });
    }
    async upsertRegularWorkPlanPeriod(organizationId, params) {
        let now = new Date();
        let workPlanPeriod = {
            id: params.id,
            createdAt: now,
            updatedAt: now,
            workPlanId: params.workPlanId,
            periodStartDateTime: params.periodStartDateTime,
            offset: params.offset ?? null,
            details: {
                routineType: params.routineType,
                tolerances: params.tolerances,
                dayItems: params.dayItems,
            },
        };
        return this.upsertWorkPlanPeriod(organizationId, workPlanPeriod);
    }
    async upsertHalfFlexibleWorkPlanPeriod(organizationId, params) {
        let now = new Date();
        let workPlanPeriod = {
            id: params.id,
            createdAt: now,
            updatedAt: now,
            workPlanId: params.workPlanId,
            periodStartDateTime: params.periodStartDateTime,
            offset: params.offset ?? null,
            details: {
                routineType: params.routineType,
                tolerances: params.tolerances,
                dayItems: params.dayItems,
                weeklyExpectedWorkDuration: params.weeklyExpectedWorkDuration,
            },
        };
        return this.upsertWorkPlanPeriod(organizationId, workPlanPeriod);
    }
    async upsertFullFlexibleWorkPlanPeriod(organizationId, params) {
        let now = new Date();
        let workPlanPeriod = {
            id: params.id,
            createdAt: now,
            updatedAt: now,
            workPlanId: params.workPlanId,
            periodStartDateTime: params.periodStartDateTime,
            offset: params.offset ?? null,
            details: {
                routineType: params.routineType,
                dayBreakItems: params.dayBreakItems,
                weeklyExpectedWorkDuration: params.weeklyExpectedWorkDuration,
            },
        };
        return this.upsertWorkPlanPeriod(organizationId, workPlanPeriod);
    }
    async upsertShiftWorkPlanPeriod(organizationId, params) {
        let now = new Date();
        let workPlanPeriod = {
            id: params.id,
            createdAt: now,
            updatedAt: now,
            workPlanId: params.workPlanId,
            periodStartDateTime: params.periodStartDateTime,
            offset: params.offset ?? null,
            details: {
                routineType: params.routineType,
                dayBreakItems: params.dayBreakItems,
                tolerances: params.tolerances,
                restDuration: params.restDuration,
                workDuration: params.workDuration,
            },
        };
        return this.upsertWorkPlanPeriod(organizationId, workPlanPeriod);
    }
    async insertRecalculateWorkQueueAfterAccessLogImport(organizationId, users, startDates, usersToTriggerLiveDay, trx) {
        await trx.query(`INSERT INTO "${organizationId}"."recalculateWorkQueue" 
				(id, "userId", "organizationId", reason, "startDate")
				SELECT uuid_generate_v4(), u.user_id, $1, $2, u.start_date
				FROM UNNEST($3::uuid[], $4::timestamptz[]) 
				AS u(user_id, start_date)`, [organizationId, dal_constants_1.DalConstants.RecalculateWorkReason.NewAccessLog, users, startDates]);
        if (usersToTriggerLiveDay.length > 0) {
            await trx.query(`UPDATE "${organizationId}"."employeeLiveDay" eld
				SET "updateRequiredReason" = "updateRequiredReason" | $1, "updateRequiredRaisedAt" = now()
				WHERE eld."userId" = ANY($2)`, [app_enums_1.enums.RecalculateWorkReason.NewAccessLog, usersToTriggerLiveDay]);
        }
    }
    async upsertWorkPlanPeriod(organizationId, workPlanPeriod) {
        return this.dbClient.transaction(async (trx) => {
            let workPlanPeriodConflictQb = trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods)
                .first("id")
                .where("workPlanId", workPlanPeriod.workPlanId)
                .where("periodStartDateTime", workPlanPeriod.periodStartDateTime);
            if (workPlanPeriod.id) {
                workPlanPeriodConflictQb.where("id", "<>", workPlanPeriod.id);
            }
            if (await workPlanPeriodConflictQb) {
                (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.PACS.WORKPLANPERIODSTARTDATECONFLICT");
            }
            let workPlanQb = trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans + " as wp")
                .leftOuterJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods + " as wpp", "wp.id", "wpp.workPlanId")
                .first("wp.id as workPlanId", "wp.accessCheckType", "wp.type", "wpp.id as periodId", "wpp.periodStartDateTime")
                .where("wp.id", workPlanPeriod.workPlanId)
                .where("wp.organizationId", organizationId);
            if (workPlanPeriod.id) {
                workPlanQb.where("wpp.id", workPlanPeriod.id);
            }
            let workPlan = await workPlanQb;
            if (!workPlan) {
                (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.PACS.WORKPLANPERIODNOTFOUND");
            }
            if (workPlan.accessCheckType === dal_constants_1.DalConstants.WorkPlanAccessCheckType.FirstAndLastAccess) {
                let hasOptionalBreaks = this.checkPeriodsForOptionalBreaks(workPlan.type, [workPlanPeriod.details]);
                if (hasOptionalBreaks) {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.PACS.WORKPLANHASOPTINALBREAKS");
                }
            }
            let existingPeriod = undefined;
            if (workPlanPeriod.id) {
                existingPeriod = (await trx.raw(`SELECT * FROM "${organizationId}"."workPlanPeriods" WHERE id = ?`, workPlanPeriod.id)).rows[0];
            }
            let recalculationStartDate = workPlanPeriod.id && workPlan.periodStartDateTime
                ? moment.min(moment(workPlan.periodStartDateTime), moment(workPlanPeriod.periodStartDateTime)).toDate()
                : workPlanPeriod.periodStartDateTime;
            await trx.raw(this.dbClient
                .raw(`
                INSERT INTO "${organizationId}"."recalculateWorkQueue"
                (id, "userId", "organizationId", reason, "startDate")

                SELECT uuid_generate_v4(), uwp."userId", ?, ?, lower(tstzrange(uwp."startDateTime", uwp."endDateTime") * tstzrange(?, null))

                FROM "${organizationId}"."userWorkPlans" AS uwp
                WHERE uwp."workPlanId" = ? AND
                NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime") * tstzrange(?, null))
            `, [organizationId, dal_constants_1.DalConstants.RecalculateWorkReason.WorkPlanDefinitionChange, recalculationStartDate, workPlanPeriod.workPlanId, recalculationStartDate])
                .toQuery());
            if (workPlanPeriod.id) {
                workPlanPeriod.createdAt = undefined;
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods).update(workPlanPeriod).where("id", workPlan.periodId);
            }
            else {
                workPlanPeriod.id = uuid.v4();
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods).insert(workPlanPeriod);
            }
            if ((existingPeriod && moment(existingPeriod.periodStartDateTime).isSameOrBefore(moment(), "day")) ||
                moment(workPlanPeriod.periodStartDateTime).isSameOrBefore(moment(), "day")) {
                let wpInfo = await this.getWorkPlansOfNow(trx, organizationId, [workPlanPeriod.workPlanId]);
                await this.scaffoldEmployeeLiveDataForWorkplans(trx, wpInfo, organizationId);
            }
            return workPlanPeriod.id;
        });
    }
    async applyUserSelectionSessionForWorkPlanWithMultipleRanges(params) {
        return dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            let session = await dal_manager_1.dbManager.accessUser.validateUserSelectionSession(params.organizationId, params.userId, params.sessionId);
            for (const rng of params.ranges) {
                if (rng.startDateTime === undefined) {
                    rng.startDateTime = null;
                }
                if (rng.endDateTime === undefined) {
                    rng.endDateTime = null;
                }
            }
            if (!params.enforce) {
                for (const rng of params.ranges) {
                    let result = (await trx.query(`
									SELECT count(*)::int AS count
									FROM "${params.organizationId}"."userSelectionSessionActions" AS ussa
									INNER JOIN "${params.organizationId}"."userWorkPlans" AS uwp ON uwp."userId" = ussa."userId"
									WHERE ussa."sessionId" = $1 AND ussa."action" = $2 AND
									NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime") * tstzrange($3, $4))

								`, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Added, rng.startDateTime, rng.endDateTime])).rows[0].count;
                    if (result > 0) {
                        (0, dal_access_error_1.throwDbAccessConflictError)("There are overlapped main work planning for users. Use enforce: true");
                    }
                }
            }
            const deletedUserIds = (await trx.query(`
					SELECT "userId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
					WHERE "sessionId" = $1 AND action = $2
				
				`, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed])).rows.map((elem) => elem.userId);
            const insertedUserIds = (await trx.query(`
					SELECT "userId"
					FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
					WHERE "sessionId" = $1 AND action = $2
					`, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Added])).rows.map((elem) => elem.userId);
            for (const rng of params.ranges) {
                await trx.query(`
						INSERT INTO "${params.organizationId}"."recalculateWorkQueue"
						(id, "userId", "organizationId", reason, "startDate")
						SELECT uuid_generate_v4(), "userId", $1 as "organizationId", $2 as "reason", MIN("startDateTime") as "startDate"
						FROM
						(
							SELECT ussa."userId", lower(wpp.range*tstzrange($3, $4)) as "startDateTime"
							FROM "${params.organizationId}"."userSelectionSessions" AS uss
							INNER JOIN "${params.organizationId}"."userSelectionSessionActions" AS ussa ON uss."id" = ussa."sessionId" AND ussa."action" = $5
							INNER JOIN "${params.organizationId}"."vW_WorkPlanPeriodTstzrange" as wpp ON wpp."workPlanId" = uss."relatedItemId"
								AND wpp.range&&tstzrange($3, $4)
								WHERE uss."id" = $6
							UNION ALL
							SELECT ussa."userId", lower(wpp.range*tstzrange($3, $4)*tstzrange(uwp."startDateTime", uwp."endDateTime")) as "startDateTime"
							FROM "${params.organizationId}"."userSelectionSessions" AS uss
							INNER JOIN "${params.organizationId}"."userSelectionSessionActions" AS ussa ON uss."id" = ussa."sessionId" AND ussa."action" = $5
							INNER JOIN "${params.organizationId}"."userWorkPlans" as uwp ON uwp."userId" = ussa."userId"
							INNER JOIN "${params.organizationId}"."vW_WorkPlanPeriodTstzrange" as wpp ON wpp."workPlanId" = uwp."workPlanId"
								AND (wpp.range*tstzrange($3, $4))&&tstzrange(uwp."startDateTime", uwp."endDateTime")
								WHERE uss."id" = $6
							UNION ALL
							SELECT ussa."userId", lower(wpp.range*tstzrange($3, $4)*tstzrange(uwp."startDateTime", uwp."endDateTime"))
							FROM "${params.organizationId}"."userSelectionSessions" AS uss
							INNER JOIN "${params.organizationId}"."userSelectionSessionActions" AS ussa ON uss."id" = ussa."sessionId" AND ussa."action" = $7
							INNER JOIN "${params.organizationId}"."userWorkPlans" as uwp ON uwp."userId" = ussa."userId" AND uwp."workPlanId" = uss."relatedItemId"
							INNER JOIN "${params.organizationId}"."vW_WorkPlanPeriodTstzrange" as wpp ON wpp."workPlanId" = uss."relatedItemId"
								AND (wpp.range*tstzrange($3, $4))&&tstzrange(uwp."startDateTime", uwp."endDateTime")
								WHERE uss."id" = $6
						) intersections
						WHERE "startDateTime" IS NOT NULL
						GROUP BY "userId"
						`, [
                    session.organizationId,
                    dal_constants_1.DalConstants.RecalculateWorkReason.UserWorkPlanChange,
                    rng.startDateTime,
                    rng.endDateTime,
                    dal_constants_1.DalConstants.UserSelectionSessionAction.Added,
                    session.id,
                    dal_constants_1.DalConstants.UserSelectionSessionAction.Removed,
                ]);
                for (const userId of deletedUserIds) {
                    await this.removeUserWorkplanMembershipInRange(params.organizationId, { userId: userId, range: rng }, trx);
                }
                for (const userId of insertedUserIds) {
                    await this.insertWorkPlanMembership(params.organizationId, { userId: userId, workPlanId: session.relatedItemId, range: rng }, trx);
                }
                await trx.query(`
						DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" as eld
						USING "${params.organizationId}"."userSelectionSessionActions" ussa
						WHERE ussa."sessionId" = $1 AND ussa."action" <> $2 AND ussa."userId" = eld."userId"
						`, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged]);
                let affectedUsers = (await trx.query(`
						SELECT ussa."userId" FROM "${params.organizationId}"."userSelectionSessionActions" ussa
            			WHERE ussa."sessionId" = $1 AND ussa."action" <> $2
            				`, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged])).rows.map((elem) => elem.userId);
                let workPlans = await this.getWorkPlansOfNowPg(trx, params.organizationId);
                await this.scaffoldEmployeeLiveDataForWorkplansPg(trx, workPlans, params.organizationId, affectedUsers);
                await trx.query(`
						DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
						WHERE id = $1
						`, [session.id]);
            }
            for (const userId of insertedUserIds) {
                await this.mergeUserWorkPlanMemberShip(params.organizationId, { userId: userId }, trx);
            }
        }, params.userId, params.organizationId);
    }
    async markEmployeeLiveDaysForRecalculation(organizationId) {
        let sessionId = uuid.v4();
        let count = await this._pgPool.query(`
			WITH update_query AS
			(
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
				SET "sessionId" = $1
				FROM (
					SELECT eld.id
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" AS eld
					LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.recalculateWorkQueue}" AS rwq ON rwq."userId" = eld."userId"
					WHERE "rwq"."id" IS NULL AND "eld"."updateRequiredReason" <> $2 AND eld."updateRequiredReason" & $3 = 0 AND eld."sessionId" IS NULL
					FOR UPDATE OF eld
				) sq
				WHERE "${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}".id = sq.id
				RETURNING *
			)
			SELECT COUNT(*) as "entryCount" FROM update_query
		`, [sessionId, dal_constants_1.DalConstants.RecalculateWorkReason.Standby, dal_constants_1.DalConstants.RecalculateWorkReason.Processing]);
        if (count.rowCount === 0) {
            sessionId = null;
        }
        return sessionId;
    }
    async fetchEmployeeLiveDaysForRecalculation(organizationId, sessionId, count) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let results = (await trx.query(`
					SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
					WHERE "sessionId" = $1
					ORDER BY "updateRequiredRaisedAt" ASC, "userId"
					LIMIT $2
					FOR UPDATE
				`, [sessionId, count ? count : 1])).rows;
            if (results && results.length > 0) {
                await trx.query(`
                	UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
                	SET "updateRequiredReason" = $1::smallint | $2::smallint
                	WHERE id = ANY($3)
                `, [dal_constants_1.DalConstants.RecalculateWorkReason.Processing, dal_constants_1.DalConstants.RecalculateWorkReason.Standby, results.map((r) => r.id)]);
            }
            return results;
        });
    }
    async updateLiveEmployeeDay(organizationId, params) {
        await this.pgPool.query(`
        UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
        SET (data, "sessionId", "updatedAt", "updateRequiredReason", "workPlanIds") = ($1, NULL, $2, "updateRequiredReason" & (~($3::smallint)) , $5)
        WHERE id = $4
        `, [params.employeeDayShort, new Date(), dal_constants_1.DalConstants.RecalculateWorkReason.Processing, params.id, JSON.stringify(params.employeeDayShort.w)]);
    }
    async deleteLiveDayEntry(organizationId, userId) {
        await this.pgPool.query(`
		DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" WHERE "userId" = $1;
		`, [userId]);
    }
    async updateLiveEmployeeDays(organizationId, list) {
        let valuesArray = [];
        let bindingsArray = [];
        let bindingCounter = 1;
        for (const row of list) {
            valuesArray.push(`($${bindingCounter++},$${bindingCounter++})`);
            bindingsArray.push(row.userId, JSON.stringify(row.employeeDayShort));
        }
        const updatedList = await this.pgPool.query(`UPDATE "${organizationId}"."employeeLiveDay" AS eld SET
				data = sq.data::jsonb
			FROM (VALUES
				${valuesArray.join(",")}
			) as sq("userId", "data")
			WHERE eld."userId" = sq."userId"::uuid AND eld.data->>'pt' = ((sq.data)::jsonb)->>'pt';`, bindingsArray);
        if (list.length - updatedList.rowCount > 0) {
            app_logs_1.logger.info("Skipped " + (list.length - updatedList.rowCount) + " items for report refresh because they are in the process of being updated");
        }
    }
    async scaffoldEmployeeLiveDataForWorkplans(trx, workPlans, organizationId, forUsers) {
        let now = new Date();
        for (const wp of workPlans) {
            let rawQuery = `
            INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
            ("id", "userId", "startAt", "expiredAt", "organizationId", "workPlanIds", "updateRequiredReason", "updateRequiredRaisedAt")
            SELECT uuid_generate_v4(), uwp."userId", ?, ?, ?, array_to_json(array_agg(wpp."workPlanId"::text)), ?, ?
            FROM "${organizationId}"."userWorkPlans" uwp
            INNER JOIN "${organizationId}"."userOrganizations" uo ON uo."userId" = uwp."userId" AND uo."organizationId" = ?
            INNER JOIN "${organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = uwp."workPlanId" AND wpp."organizationId" = uo."organizationId" AND uwp."workPlanId" = ?
            WHERE NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime")*wpp.range*tstzrange(?,?))
            AND NOT uo."isDisabled" AND uo."deletedAt" IS NULL
            ${forUsers ? 'AND uo."userId" IN ' + this.concatIdsForWhereInRawQuery(forUsers) : ""}
            GROUP BY uwp."userId"
            ON CONFLICT ON CONSTRAINT "employeeLiveDay_userId_organizationId_key" DO UPDATE SET ("workPlanIds", "updateRequiredReason", "updateRequiredRaisedAt") = 
            (
                (SELECT JSONB_AGG(DISTINCT "unfoldedWps"."wpList") FROM
                (SELECT JSONB_ARRAY_ELEMENTS(EXCLUDED."workPlanIds" || "employeeLiveDay"."workPlanIds") as "wpList") AS "unfoldedWps"),
                EXCLUDED."updateRequiredReason" | "employeeLiveDay"."updateRequiredReason", EXCLUDED."updateRequiredRaisedAt"
            )
            RETURNING id;`;
            let insertedIds = await trx.raw(rawQuery, [
                wp.startOfDay.toDate(),
                wp.expiredAt.toDate(),
                wp.organizationId,
                dal_constants_1.DalConstants.RecalculateWorkReason.Initialization,
                now,
                wp.organizationId,
                wp.workPlanId,
                wp.startOfDay.toDate(),
                wp.expiredAt.toDate(),
            ]);
            app_logs_1.logger.info("New live day entries are created for " + organizationId + " and workplan " + wp.workPlanId + " with " + insertedIds.rows.length);
        }
    }
    async scaffoldEmployeeLiveDataForWorkplansPg(trx, workPlans, organizationId, forUsers) {
        let now = new Date();
        for (const wp of workPlans) {
            let bindings = [
                wp.startOfDay.toDate(),
                wp.expiredAt.toDate(),
                wp.organizationId,
                dal_constants_1.DalConstants.RecalculateWorkReason.Initialization,
                now,
                wp.organizationId,
                wp.workPlanId,
            ];
            let rawQuery = `
            INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
            ("id", "userId", "startAt", "expiredAt", "organizationId", "workPlanIds", "updateRequiredReason", "updateRequiredRaisedAt")
            SELECT uuid_generate_v4(), uwp."userId", $1, $2, $3, array_to_json(array_agg(wpp."workPlanId"::text)), $4, $5
            FROM "${organizationId}"."userWorkPlans" uwp
            INNER JOIN "${organizationId}"."userOrganizations" uo ON uo."userId" = uwp."userId" AND uo."organizationId" = $6
            INNER JOIN "${organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = uwp."workPlanId" AND wpp."organizationId" = uo."organizationId" AND uwp."workPlanId" = $7
            WHERE NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime")*wpp.range*tstzrange($1,$2))
            AND NOT uo."isDisabled" AND uo."deletedAt" IS NULL`;
            if (forUsers && forUsers.length > 0) {
                rawQuery += `
				AND uo."userId" = ANY($8)`;
                bindings.push(forUsers);
            }
            rawQuery += `
			GROUP BY uwp."userId"
            ON CONFLICT ON CONSTRAINT "employeeLiveDay_userId_organizationId_key" DO UPDATE SET ("workPlanIds", "updateRequiredReason", "updateRequiredRaisedAt") = 
            (
                (SELECT JSONB_AGG(DISTINCT "unfoldedWps"."wpList") FROM
                (SELECT JSONB_ARRAY_ELEMENTS(EXCLUDED."workPlanIds" || "employeeLiveDay"."workPlanIds") as "wpList") AS "unfoldedWps"),
                EXCLUDED."updateRequiredReason" | "employeeLiveDay"."updateRequiredReason", EXCLUDED."updateRequiredRaisedAt"
            )
            RETURNING id;`;
            let insertedIds = await trx.query(rawQuery, bindings);
            app_logs_1.logger.info("New live day entries are created for " + organizationId + " and workplan " + wp.workPlanId + " with " + insertedIds.rows.length);
        }
    }
    async scaffoldEmployeeLiveData(organizationId) {
        return this.dbClient.transaction(async (trx) => {
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay).delete();
            let workPlans = await this.getWorkPlansOfNow(trx, organizationId);
            await this.scaffoldEmployeeLiveDataForWorkplans(trx, workPlans, organizationId);
        });
    }
    async rescaffoldEmployeeLiveData(organizationId) {
        return this.dbClient.transaction(async (trx) => {
            let now = new Date();
            let result = [];
            let existingLiveWorkplans = (await trx.raw(`
            SELECT DISTINCT JSONB_ARRAY_ELEMENTS("workPlanIds") as id, "expiredAt" <= ? as expired FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}";
            `, now)).rows;
            if (existingLiveWorkplans.find((lw) => lw.expired)) {
                {
                    let qb = trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay).where("expiredAt", "<=", now);
                    result = await qb.clone().select("organizationId").select("userId");
                    await qb.delete();
                }
            }
            let workPlansToScaffold = await this.getWorkPlansOfNow(trx, organizationId);
            workPlansToScaffold = workPlansToScaffold.filter((w) => {
                let existing = existingLiveWorkplans.find((e) => e.id === w.workPlanId);
                return !existing || existing.expired;
            });
            if (workPlansToScaffold.length > 0) {
                await this.scaffoldEmployeeLiveDataForWorkplans(trx, workPlansToScaffold, organizationId);
            }
            return result;
        });
    }
    async getWorkPlansOfNow(trx, organizationId, including) {
        let now = moment();
        let query = `
		SELECT * FROM (
			SELECT (
				SELECT COUNT(*) FROM "${organizationId}"."userWorkPlans" uwp
				INNER JOIN "${organizationId}"."userOrganizations" uo ON uo."userId" = uwp."userId" AND uo."deletedAt" IS NULL AND NOT uo."isDisabled"
				INNER JOIN "${organizationId}"."userOrganizationProfiles" uop ON uop."userId" = uwp."userId" AND uop."deletedAt" IS NULL
				WHERE uwp."workPlanId" = wp.id AND tstzrange(uwp."startDateTime", uwp."endDateTime") @> ?::timestamp with time zone
			) as users, wp.id, wp.timezone, wp."organizationId" FROM "${organizationId}"."workPlans" wp
			INNER JOIN "${organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = wp.id AND wpp."range" @> ?::timestamp with time zone
			WHERE wp."deletedAt" IS NULL
		) sq WHERE users > 0
        `;
        let bindings = [now.toDate(), now.toDate()];
        if (including) {
            query += `AND id IN (` + including.map((w) => `'` + w + `'`).join(`,`) + `)`;
        }
        let workPlanList = await trx.raw(query, bindings);
        return workPlanList.rows.map((row) => {
            let start = now.clone().utcOffset(row.timezone, false).startOf("day");
            return {
                workPlanId: row.id,
                organizationId: row.organizationId,
                timezone: row.timezone,
                now: now,
                startOfDay: start,
                expiredAt: start.clone().add(1, "day"),
            };
        });
    }
    async getWorkPlansOfNowPg(trx, organizationId, including) {
        let now = moment();
        let query = `
		SELECT * FROM (
			SELECT (
				SELECT COUNT(*) FROM "${organizationId}"."userWorkPlans" uwp
				INNER JOIN "${organizationId}"."userOrganizations" uo ON uo."userId" = uwp."userId" AND uo."deletedAt" IS NULL AND NOT uo."isDisabled"
				INNER JOIN "${organizationId}"."userOrganizationProfiles" uop ON uop."userId" = uwp."userId" AND uop."deletedAt" IS NULL
				WHERE uwp."workPlanId" = wp.id AND tstzrange(uwp."startDateTime", uwp."endDateTime") @> $1::timestamp with time zone
			) as users, wp.id, wp.timezone, wp."organizationId" FROM "${organizationId}"."workPlans" wp
			INNER JOIN "${organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = wp.id AND wpp."range" @> $1::timestamp with time zone
			WHERE wp."deletedAt" IS NULL
		) sq WHERE users > 0
        `;
        let bindings = [now.toDate()];
        if (including) {
            bindings.push(including);
            query += `AND id = ANY($${bindings.length})`;
        }
        let workPlanList = await trx.query(query, bindings);
        return workPlanList.rows.map((row) => {
            let start = now.clone().utcOffset(row.timezone, false).startOf("day");
            return {
                workPlanId: row.id,
                organizationId: row.organizationId,
                timezone: row.timezone,
                now: now,
                startOfDay: start,
                expiredAt: start.clone().add(1, "day"),
            };
        });
    }
    async getEmployeeDayParams(organizationId, userId, date, previousDays) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            const dayRange = new moment_range_1.DateRange(moment(date).startOf("day"), moment(date).add(1, "day").startOf("day"));
            let userOrganizationId = (await trx.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					WHERE uo."userId" = $1 AND "deletedAt" IS NULL`, [userId])).rows[0]?.id;
            if (!userOrganizationId) {
                (0, dal_access_error_1.throwDbAccessNotFoundError)("User organization mapping is not found");
            }
            const { rows: uwps, rowCount } = await trx.query(`SELECT uwp.*, wp.offset, wpp.offset as "periodOffset"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}" wp
					ON wp.id = uwp."workPlanId"
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_WorkPlanPeriodTstzrange}" wpp
					ON wp.id = wpp."workPlanId" AND wpp.range && tstzrange($2,$3)
				WHERE uwp."userId" = $1 AND tstzrange(uwp."startDateTime", uwp."endDateTime") && tstzrange($2, $3)
				ORDER BY uwp."startDateTime" ASC NULLS FIRST, LOWER(wpp.range) ASC;`, [userId, dayRange.start.toDate(), dayRange.end.toDate()]);
            const buffer = (await trx.query(`SELECT "bufferMinutes" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationPACSModuleSettings}"`)).rows[0]?.bufferMinutes ?? 0;
            let startOffset = 0;
            let endOffset = 0;
            if (rowCount > 0) {
                let calculatingDay = moment(date);
                if ((!uwps[0].startDateTime || moment(uwps[0].startDateTime).isSameOrBefore(calculatingDay.clone().startOf("day"))) &&
                    (!uwps[0].endDateTime || moment(uwps[0].endDateTime).isSameOrAfter(calculatingDay.clone().startOf("day")))) {
                    startOffset = uwps[0].periodOffset ?? uwps[0].offset;
                }
                if ((!uwps[uwps.length - 1].startDateTime || moment(uwps[uwps.length - 1].startDateTime).isSameOrBefore(calculatingDay.clone().startOf("day").add(1, "day"))) &&
                    (!uwps[uwps.length - 1].endDateTime || moment(uwps[uwps.length - 1].endDateTime).isSameOrAfter(calculatingDay.clone().startOf("day").add(1, "day")))) {
                    endOffset = uwps[uwps.length - 1].periodOffset ?? uwps[uwps.length - 1].offset;
                }
            }
            const offsetDayRange = new moment_range_1.DateRange(dayRange.start.clone().add(startOffset, "minutes"), dayRange.end.clone().add(endOffset, "minutes"));
            const bufferedDayRange = new moment_range_1.DateRange(offsetDayRange.start.clone().subtract(buffer, "minutes"), offsetDayRange.end.clone().add(buffer, "minutes"));
            const result = {
                organizationId: organizationId,
                employeeDayParams: undefined,
                flexibleWorkPlanDayParams: [],
                fixedWorkPlanDayParams: [],
            };
            const { rows: wpPeriodDetailRanges } = await trx.query(`SELECT
					day_ranges.date,
					wpp."workPlanId",
					lower(wpp.range * day_ranges.range * tstzrange(uwp."startDateTime", uwp."endDateTime")) as "intersectionStart",
					upper(wpp.range * day_ranges.range * tstzrange(uwp."startDateTime", uwp."endDateTime")) as "intersectionEnd",
					wpp.details,
					wp.type,
					wp."accessCheckType",
					wp."regionId",
					wp."offset",
					wp."allowMobileCheckins",
					wp."allowUnreliableCheckins",
					wp."permissionRequiredForExtraWorking",
					wp."ignoreHolidays",
					lower(wpp.range) as "periodStart",
					wp."ignoreUnpairedAccesses"
				FROM "${organizationId}"."userWorkPlans" uwp
				INNER JOIN (
					SELECT date_sq.date::date, tstzrange(date_sq.date, date_sq.date + INTERVAL '1 day')*tstzrange($1::timestamp with time zone, $2::timestamp with time zone) as range FROM
					(
						SELECT generate_series($3::date, $4::date, '1 day')::timestamp with time zone as date
					) date_sq
				) day_ranges ON day_ranges.range && tstzrange(uwp."startDateTime", uwp."endDateTime")
				INNER JOIN "${organizationId}"."workPlans" wp ON wp.id = uwp."workPlanId"
				INNER JOIN "${organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = uwp."workPlanId"
				WHERE uwp."userId" = $5
				AND NOT ISEMPTY(wpp.range * day_ranges.range * tstzrange(uwp."startDateTime", uwp."endDateTime"))
				ORDER BY uwp."startDateTime" ASC NULLS FIRST`, [bufferedDayRange.start.toDate(), bufferedDayRange.end.toDate(), bufferedDayRange.start.toDate(), bufferedDayRange.end.toDate(), userId]);
            const organizationPacsSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId);
            const { rows: holidaysDb } = await trx.query(`SELECT
					id,
					title,
					lower(tstzrange("startDateTime", "endDateTime")*tstzrange($1::timestamp with time zone, $2::timestamp with time zone)) as start,
					upper(tstzrange("startDateTime", "endDateTime")*tstzrange($1::timestamp with time zone, $2::timestamp with time zone)) as end
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.vacations}"
				WHERE tstzrange("startDateTime", "endDateTime") && tstzrange($1::timestamp with time zone, $2::timestamp with time zone)`, [bufferedDayRange.start.toDate(), bufferedDayRange.end.toDate()]);
            let holidays = undefined;
            if (holidaysDb?.length > 0) {
                holidays = holidaysDb.map((h) => {
                    let start = moment(h.start);
                    let end = moment(h.end);
                    if (organizationPacsSettings.middayHour !== "12:00") {
                        let middayHour = parseInt(organizationPacsSettings.middayHour.substr(0, 2));
                        let middayMinutes = parseInt(organizationPacsSettings.middayHour.substr(3, 2));
                        if (start.hour() === 12 && start.minute() === 0) {
                            start = start.set("hour", middayHour).set("minute", middayMinutes);
                        }
                        if (end.hour() === 12 && end.minute() === 0) {
                            end = end.set("hour", middayHour).set("minute", middayMinutes);
                        }
                    }
                    return {
                        id: h.id,
                        range: new moment_range_1.DateRange(start, end),
                    };
                });
            }
            for (const workPlanPeriod of wpPeriodDetailRanges) {
                const periodIntersection = new moment_range_1.DateRange(workPlanPeriod.intersectionStart, workPlanPeriod.intersectionEnd);
                const workPlanDayParams = {
                    date: moment(workPlanPeriod.date).format("YYYY-MM-DD"),
                    workPlanId: workPlanPeriod.workPlanId,
                    regionId: workPlanPeriod.regionId,
                    dateIntersection: periodIntersection,
                    offsetDayRange: offsetDayRange,
                    bufferedDayRange: bufferedDayRange,
                    tolerances: workPlanPeriod.details.tolerances,
                    type: workPlanPeriod.type,
                    attendanceControlMethod: workPlanPeriod.accessCheckType,
                    workingHours: [],
                    fixedBreaks: null,
                    claimBreak: null,
                    extraWorkBreaks: null,
                    holidays: workPlanPeriod.ignoreHolidays ? undefined : holidays,
                    allowMobileCheckins: workPlanPeriod.allowMobileCheckins,
                    allowUnreliableCheckins: workPlanPeriod.allowUnreliableCheckins,
                    permissionRequiredForExtraWorking: workPlanPeriod.permissionRequiredForExtraWorking,
                    ignoreUnpairedAccesses: workPlanPeriod.ignoreUnpairedAccesses,
                };
                if (workPlanPeriod.type === dal_constants_1.DalConstants.WorkPlanType.SHIFT || workPlanPeriod.type === dal_constants_1.DalConstants.WorkPlanType.FULL_FLEXIBLE) {
                    let relatedDayItem = undefined;
                    if (workPlanPeriod.details.dayBreakItems && workPlanPeriod.details.dayBreakItems.length > 0) {
                        relatedDayItem = this.findDayItem(moment(workPlanPeriod.date).toDate(), new Date(workPlanPeriod.periodStart), workPlanPeriod.details.routineType, workPlanPeriod.details.dayBreakItems);
                    }
                    if (relatedDayItem) {
                        workPlanDayParams.fixedBreaks = relatedDayItem.fixedBreaks;
                        workPlanDayParams.claimBreak = relatedDayItem.claimBreak;
                        workPlanDayParams.extraWorkBreaks = relatedDayItem.extraWorkBreaks;
                    }
                }
                if (workPlanPeriod.type === dal_constants_1.DalConstants.WorkPlanType.SHIFT) {
                    const periodLength = workPlanPeriod.details.workDuration + workPlanPeriod.details.restDuration;
                    workPlanDayParams.workingHours = [];
                    const range = new moment_range_1.DateRange(moment(workPlanPeriod.periodStart), moment(workPlanPeriod.periodStart).add(workPlanPeriod.details.workDuration, "minutes"));
                    while (range.start.isBefore(periodIntersection.end)) {
                        const shiftIntersection = range.intersect(periodIntersection);
                        if (shiftIntersection) {
                            workPlanDayParams.workingHours.push(shiftIntersection);
                        }
                        range.start = range.start.add(periodLength, "minutes");
                        range.end = range.end.add(periodLength, "minutes");
                    }
                }
                else if (workPlanPeriod.type !== dal_constants_1.DalConstants.WorkPlanType.FULL_FLEXIBLE) {
                    let relatedDayItem = undefined;
                    if (workPlanPeriod.details.dayItems && workPlanPeriod.details.dayItems.length > 0) {
                        relatedDayItem = this.findDayItem(moment(workPlanPeriod.date).toDate(), new Date(workPlanPeriod.periodStart), workPlanPeriod.details.routineType, workPlanPeriod.details.dayItems);
                    }
                    if (relatedDayItem) {
                        if (relatedDayItem.workingHours) {
                            workPlanDayParams.workingHours = relatedDayItem.workingHours.map((wh) => {
                                return new moment_range_1.DateRange(moment(workPlanPeriod.date)
                                    .clone()
                                    .set("hours", parseInt(wh.startTime.substr(0, 2)))
                                    .set("minutes", parseInt(wh.startTime.substr(2, 2))), moment(workPlanPeriod.date)
                                    .clone()
                                    .set("hours", parseInt(wh.endTime.substr(0, 2)))
                                    .set("minutes", parseInt(wh.endTime.substr(2, 2))));
                            });
                        }
                        workPlanDayParams.fixedBreaks = relatedDayItem.fixedBreaks;
                        workPlanDayParams.claimBreak = relatedDayItem.claimBreak;
                        workPlanDayParams.extraWorkBreaks = relatedDayItem.extraWorkBreaks;
                    }
                }
                if (workPlanPeriod.type === dal_constants_1.DalConstants.WorkPlanType.HALF_FLEXIBLE || workPlanPeriod.type === dal_constants_1.DalConstants.WorkPlanType.FULL_FLEXIBLE) {
                    workPlanDayParams.weeklyExpectedWorkDuration = workPlanPeriod.details.weeklyExpectedWorkDuration;
                    result.flexibleWorkPlanDayParams.push(workPlanDayParams);
                    workPlanPeriod.weeklyExpectedWorkDuration = workPlanPeriod.details.weeklyExpectedWorkDuration;
                }
                else {
                    result.fixedWorkPlanDayParams.push(workPlanDayParams);
                }
            }
            result.employeeDayParams = await this.getEmployeeDayParamsForDay({
                organizationId: organizationId,
                userOrganizationId: userOrganizationId,
                userId: userId,
                date: date,
                offsetDayRange: offsetDayRange,
                bufferedDayRange: bufferedDayRange,
                trx: trx,
                previousDays: previousDays,
                workPlans: wpPeriodDetailRanges,
            });
            return result;
        });
    }
    async getAccessTimesForWorkPlanIntersection(params) {
        const trxx = params.trx ?? this._pgPool;
        let qx = 1;
        const q = `
			SELECT acp.id
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionAccessControlPoints}" AS racp
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp ON racp."accessControlPointId" = acp.id
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" AS r ON r.id = racp."regionId"
			WHERE r.id = $${qx++} AND acp."deletedAt" IS NULL
		`;
        const { rows: accessControlPointIdsRows } = await trxx.query(q, [params.regionId]);
        const accessControlPointIds = accessControlPointIdsRows.map((row) => row.id);
        if (params.allowMobileCheckins) {
            qx = 1;
            const q = `
				SELECT id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
				WHERE "accessControlPointType" = $${qx++} AND "deletedAt" IS NULL
			`;
            const { rows: mobileAcp } = await trxx.query(q, [dal_constants_1.DalConstants.AccessControlPointType.VirtualMobileCheckin]);
            if (mobileAcp.length > 0) {
                accessControlPointIds.push(...mobileAcp.map((r) => r.id));
            }
        }
        const accessTimes = [];
        if (params.accessCheckType === dal_constants_1.DalConstants.WorkPlanAccessCheckType.FirstAndLastAccess) {
            const mainDay = await dal_manager_1.dbManager.accessLogPacs.getAccessTimesFirstLast({
                organizationId: params.organizationId,
                start: params.offsetDayRange.start.toDate(),
                end: params.offsetDayRange.end.toDate(),
                regionId: params.regionId,
                accessControlPointIds: accessControlPointIds,
                userId: params.userId,
                allowUnreliableCheckins: params.allowMobileCheckins ? params.allowUnreliableCheckins : undefined,
                trx: params.trx,
            });
            if (mainDay.actionUtcMin) {
                if (moment(mainDay.actionUtcMin).isSame(mainDay.actionUtcMax)) {
                    const previousDay = await dal_manager_1.dbManager.accessLogPacs.getAccessTimesFirstLast({
                        organizationId: params.organizationId,
                        start: params.offsetDayRange.start.clone().subtract(1, "day").toDate(),
                        end: params.offsetDayRange.start.toDate(),
                        regionId: params.regionId,
                        accessControlPointIds: accessControlPointIds,
                        userId: params.userId,
                        allowUnreliableCheckins: params.allowMobileCheckins ? params.allowUnreliableCheckins : undefined,
                        trx: params.trx,
                    });
                    if (previousDay.actionUtcMin && moment(previousDay.actionUtcMin).isSame(previousDay.actionUtcMax)) {
                        accessTimes.push({
                            id: null,
                            time: params.offsetDayRange.start.clone(),
                            checkIn: true,
                        }, {
                            id: mainDay.idMin,
                            time: moment(mainDay.actionUtcMin),
                            checkIn: false,
                        });
                    }
                    else {
                        accessTimes.push({
                            id: mainDay.idMin,
                            time: moment(mainDay.actionUtcMin),
                            checkIn: true,
                        }, {
                            id: null,
                            time: params.offsetDayRange.end.clone(),
                            checkIn: false,
                        });
                    }
                }
                else {
                    accessTimes.push({
                        id: mainDay.idMin,
                        time: moment(mainDay.actionUtcMin),
                        checkIn: true,
                    });
                    accessTimes.push({
                        id: mainDay.idMax,
                        time: moment(mainDay.actionUtcMax),
                        checkIn: false,
                    });
                }
            }
            else {
                const previousDay = await dal_manager_1.dbManager.accessLogPacs.getAccessTimesFirstLast({
                    organizationId: params.organizationId,
                    start: params.offsetDayRange.start.clone().subtract(1, "day").toDate(),
                    end: params.offsetDayRange.start.toDate(),
                    regionId: params.regionId,
                    accessControlPointIds: accessControlPointIds,
                    userId: params.userId,
                    allowUnreliableCheckins: params.allowMobileCheckins ? params.allowUnreliableCheckins : undefined,
                    trx: params.trx,
                });
                if (previousDay.actionUtcMin && moment(previousDay.actionUtcMin).isSame(previousDay.actionUtcMax)) {
                    accessTimes.push({
                        id: null,
                        time: params.offsetDayRange.start.clone(),
                        checkIn: true,
                    }, {
                        id: null,
                        time: params.offsetDayRange.end.clone(),
                        checkIn: false,
                    });
                }
            }
        }
        else {
            const accesses = await dal_manager_1.dbManager.accessLogPacs.getAccessTimesWithDirection({
                organizationId: params.organizationId,
                start: params.offsetDayRange.start.toDate(),
                end: params.offsetDayRange.end.toDate(),
                regionId: params.regionId,
                accessControlPointIds: accessControlPointIds,
                userId: params.userId,
                allowUnreliableCheckins: params.allowMobileCheckins ? params.allowUnreliableCheckins : undefined,
                trx: params.trx,
            });
            if (accesses && accesses.length > 0) {
                const as = accesses[0];
                if (as.direction === dal_constants_1.DalConstants.AccessDirection.Exit) {
                    accessTimes.push({
                        id: null,
                        time: params.offsetDayRange.start,
                        checkIn: true,
                    });
                }
                accessTimes.push(...accesses.map((a) => {
                    return {
                        id: a.id,
                        time: moment(a.actionUtc),
                        checkIn: a.direction === dal_constants_1.DalConstants.AccessDirection.Entrance,
                    };
                }));
                let ae = accesses[accesses.length - 1];
                if (ae.direction === dal_constants_1.DalConstants.AccessDirection.Entrance) {
                    accessTimes.push({
                        id: null,
                        time: params.offsetDayRange.end,
                        checkIn: false,
                    });
                }
            }
            else {
                const accessesOfYesterday = await dal_manager_1.dbManager.accessLogPacs.getAccessTimesWithDirection({
                    organizationId: params.organizationId,
                    start: params.offsetDayRange.start.clone().add(-1, "days").toDate(),
                    end: params.offsetDayRange.start.toDate(),
                    regionId: params.regionId,
                    accessControlPointIds: accessControlPointIds,
                    userId: params.userId,
                    allowUnreliableCheckins: params.allowMobileCheckins ? params.allowUnreliableCheckins : undefined,
                    trx: params.trx,
                });
                if (accessesOfYesterday && accessesOfYesterday.length > 0) {
                    const lastAccess = accessesOfYesterday[accessesOfYesterday.length - 1];
                    if (lastAccess.direction === dal_constants_1.DalConstants.AccessDirection.Entrance) {
                        accessTimes.push({
                            id: null,
                            time: params.offsetDayRange.start,
                            checkIn: true,
                        });
                    }
                }
            }
        }
        return accessTimes;
    }
    async getEmployeePermissionsForDay(params) {
        const trxx = params.trx ?? this._pgPool;
        const result = {
            declaredWorkPermissions: [],
            shouldBeIgnoredWorkPermissions: [],
            offTimePermissions: [],
            extraWorkPermissions: [],
        };
        let qx = 1;
        const q = `
		SELECT pp.id, pp."startDateTime", pp."endDateTime", pp."ppermissionTypeId", 
		ppt."method", ppt."applyOnHolidays", ppt."applyOnlyOnWorkingHours", ppt."isUnpaidLeave", ppt."isDailyScheduled"
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" AS pp
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" AS ppt ON pp."ppermissionTypeId" = ppt.id
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" AS uppt ON pp.id = uppt."ppermissionId"
		WHERE uppt."userId" = $${qx++} AND ppt."deletedAt" IS NULL
		AND tstzrange($${qx++}, $${qx++}) && tstzrange(pp."startDateTime", pp."endDateTime")
		AND (SELECT bool_and(uppa."approvementDate" is not null AND uppa."status" = true) 
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements}" as uppa
		WHERE pp.id = uppa."ppermissionId" AND uppa."userId" = uppt."userId") 
		`;
        const { rows: ppermissionsOfDay } = await trxx.query(q, [params.userId, params.bufferedDayRange.start.toDate(), params.bufferedDayRange.end.toDate()]);
        const typeInputMap = {};
        for (const p of ppermissionsOfDay) {
            typeInputMap[p.ppermissionTypeId] = {
                type: p.method,
                isUnpaid: p.isUnpaidLeave,
            };
        }
        const applicableDaysMap = await dal_manager_1.dbManager.accessPacs.getApplicableDaysForPPermissionTypes(params.organizationId, typeInputMap, params.userId);
        const organizationPacsSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(params.organizationId);
        for (const ppermission of ppermissionsOfDay) {
            let start = moment(ppermission.startDateTime);
            let end = moment(ppermission.endDateTime);
            if (organizationPacsSettings.middayHour !== "12:00" && ppermission.isDailyScheduled) {
                let middayHour = parseInt(organizationPacsSettings.middayHour.substr(0, 2));
                let middayMinutes = parseInt(organizationPacsSettings.middayHour.substr(3, 2));
                if (start.hour() === 12 && start.minute() === 0) {
                    start = start.set("hour", middayHour).set("minute", middayMinutes);
                }
                if (end.hour() === 12 && end.minute() === 0) {
                    end = end.set("hour", middayHour).set("minute", middayMinutes);
                }
            }
            let p = {
                id: ppermission.id,
                typeId: ppermission.ppermissionTypeId,
                range: new moment_range_1.DateRange(start, end),
                applicableDays: [],
                applyOnHolidays: ppermission.applyOnHolidays,
                applyOnlyOnWorkingHours: ppermission.applyOnlyOnWorkingHours,
            };
            p.applicableDays = applicableDaysMap.get(ppermission.ppermissionTypeId) ?? [];
            switch (ppermission.method) {
                case app_enums_1.enums.PPermissionType.DeclaredWork:
                    result.declaredWorkPermissions.push(p);
                    break;
                case app_enums_1.enums.PPermissionType.ShouldBeIgnoredWork:
                    result.shouldBeIgnoredWorkPermissions.push(p);
                    break;
                case app_enums_1.enums.PPermissionType.ExtraWork:
                    result.extraWorkPermissions.push(p);
                    break;
                case app_enums_1.enums.PPermissionType.ShouldBeIgnoredWork:
                default:
                    result.offTimePermissions.push(p);
                    break;
            }
        }
        return result;
    }
    async getEmployeeDayParamsForDay(params) {
        const result = {
            organizationId: params.organizationId,
            date: params.date,
            offsetDayRange: params.offsetDayRange,
            bufferedDayRange: params.bufferedDayRange,
            employee: {
                userId: params.userId,
                userOrganizationId: params.userOrganizationId,
            },
            declaredWorkPermissions: [],
            offTimePermissions: [],
            physicalAccessTimes: [],
            shouldBeIgnoredWorkPermissions: [],
            extraWorkPermissions: [],
        };
        const permissionsOfUser = await this.getEmployeePermissionsForDay({
            organizationId: params.organizationId,
            trx: params.trx,
            userId: params.userId,
            offsetDayRange: params.offsetDayRange,
            bufferedDayRange: params.bufferedDayRange,
        });
        result.declaredWorkPermissions = permissionsOfUser.declaredWorkPermissions;
        result.shouldBeIgnoredWorkPermissions = permissionsOfUser.shouldBeIgnoredWorkPermissions;
        result.offTimePermissions = permissionsOfUser.offTimePermissions;
        result.extraWorkPermissions = permissionsOfUser.extraWorkPermissions;
        const uniqueWorkPlans = [];
        for (const wp of params.workPlans) {
            const wpp = uniqueWorkPlans.find((t) => t.workPlanId === wp.workPlanId);
            if (!wpp) {
                uniqueWorkPlans.push(wp);
            }
        }
        for (const wpr of uniqueWorkPlans) {
            const accessTimes = await this.getAccessTimesForWorkPlanIntersection({
                organizationId: params.organizationId,
                accessCheckType: wpr.accessCheckType,
                offsetDayRange: wpr.accessCheckType === dal_constants_1.DalConstants.WorkPlanAccessCheckType.Region ? params.bufferedDayRange : params.offsetDayRange,
                regionId: wpr.regionId,
                trx: params.trx,
                userId: params.userId,
                allowMobileCheckins: wpr.allowMobileCheckins,
                allowUnreliableCheckins: wpr.allowUnreliableCheckins,
            });
            result.physicalAccessTimes.push({
                workPlanId: wpr.workPlanId,
                accessTimes: accessTimes,
            });
        }
        result.weeklyRemainingWorkDurations = [];
        for (const wp of uniqueWorkPlans) {
            if (wp.type === dal_constants_1.DalConstants.WorkPlanType.HALF_FLEXIBLE || wp.type === dal_constants_1.DalConstants.WorkPlanType.FULL_FLEXIBLE) {
                result.weeklyRemainingWorkDurations.push({
                    workPlanId: wp.workPlanId,
                    duration: await this.getWeeklyRemainingWorkDurationOfUser({
                        userId: params.userId,
                        organizationId: params.organizationId,
                        workPlanId: wp.workPlanId,
                        day: moment(params.date),
                        trx: params.trx,
                        previousDays: params.previousDays,
                        weeklyExpectedWorkDuration: wp.weeklyExpectedWorkDuration,
                    }),
                });
            }
        }
        return result;
    }
    async getWorkPlanPeriodInfo(organizationId, workPlanPeriodId, trx) {
        let transactionScope = async (trx) => {
            try {
                return trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods)
                    .where("id", workPlanPeriodId)
                    .whereNull("deletedAt")
                    .first("id", "periodStartDateTime", "workPlanId");
            }
            catch (error) {
                dal_logger_1.dbLogger.error(error);
                throw new dal_access_error_1.DbAccessError();
            }
        };
        return trx ? transactionScope(trx) : this.dbClient.transaction(transactionScope);
    }
    async removeWorkPlanPeriod(organizationId, workPlanPeriodId) {
        return this.dbClient.transaction(async (trx) => {
            let workPlanPeriod = await this.getWorkPlanPeriodInfo(organizationId, workPlanPeriodId, trx);
            if (!workPlanPeriod) {
                (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.PACS.WORKPLANPERIODNOTFOUND");
            }
            try {
                await trx.raw(this.dbClient
                    .raw(`
                INSERT INTO "${organizationId}"."recalculateWorkQueue"
                (id, "userId", "organizationId", reason, "startDate")

                SELECT uuid_generate_v4(), uwp."userId", ?, ?, lower(tstzrange(uwp."startDateTime", uwp."endDateTime")*tstzrange(wpp."periodStartDateTime", null))
                FROM "${organizationId}"."userWorkPlans" uwp INNER JOIN "${organizationId}"."workPlanPeriods" wpp ON wpp."workPlanId" = uwp."workPlanId"

                WHERE wpp."id" = ? AND wpp."deletedAt" IS NULL
                AND NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime")*tstzrange(wpp."periodStartDateTime", null))
                `, [organizationId, dal_constants_1.DalConstants.RecalculateWorkReason.WorkPlanDefinitionChange, workPlanPeriodId])
                    .toQuery());
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods).where("id", workPlanPeriodId).delete();
                if (moment(workPlanPeriod.periodStartDateTime).isSameOrBefore(moment(), "day")) {
                    await trx.raw(`
                    UPDATE "${organizationId}"."employeeLiveDay" eld SET ("workPlanIds", "updateRequiredRaisedAt", "updateRequiredReason") = ("workPlanIds" - ?, now(), ?)
                    WHERE eld."workPlanIds" \\? ?;
                    `, [workPlanPeriod.workPlanId, dal_constants_1.DalConstants.RecalculateWorkReason.WorkPlanDefinitionChange, workPlanPeriod.workPlanId]);
                    await trx.raw(`
                    DELETE FROM "${organizationId}"."employeeLiveDay" WHERE jsonb_array_length("workPlanIds") = 0;
                    `);
                    let wpInfo = await this.getWorkPlansOfNow(trx, organizationId, [workPlanPeriod.workPlanId]);
                    await this.scaffoldEmployeeLiveDataForWorkplans(trx, wpInfo, organizationId);
                }
            }
            catch (error) {
                dal_logger_1.dbLogger.error(error);
                throw new dal_access_error_1.DbAccessError();
            }
        });
    }
    async removeWorkPlan(organizationId, workPlanId, enforceRemove, trx) {
        const relatedShiftTemplates = (await trx.query(`
				SELECT DISTINCT(name)
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}" as st
				WHERE st."defaultWorkplanId" = $1
				   	OR st.id IN (
						SELECT wir."shiftTemplateId"
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanImportRules}" wir
						WHERE wir."workplanId" = $1
				);
				`, [workPlanId])).rows.map((elem) => elem.name);
        if (relatedShiftTemplates.length > 0) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR, "ERRORS.PACS.WORKPLANSHIFTTEMPLATECONFLICT", {
                shiftTemplates: relatedShiftTemplates.join(", "),
            }, true, true);
        }
        const autoShiftNames = (await trx.query(`
				SELECT DISTINCT(name) as name
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSet}" as asrs, unnest(rules) as rule
				WHERE rule->>'workPlanId' = $1 and asrs."deletedAt" IS NULL
				`, [workPlanId])).rows.map((elem) => elem.name);
        if (autoShiftNames.length > 0) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR, "ERRORS.PACS.WORKPLANAUTOSHIFTCONFLICT", {
                autoShiftNames: autoShiftNames.join(", "),
            }, true, true);
        }
        if (!enforceRemove) {
            const assignedUserCount = parseInt((await trx.query(`
						SELECT COUNT(id) as count
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}"
						WHERE "workPlanId" = $1;
						`, [workPlanId])).rows[0].count);
            if (assignedUserCount > 0) {
                (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.PACS.WORKPLANMEMBERCONFLICT");
            }
        }
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.recalculateWorkQueue}"
			(id, "userId", "organizationId", reason, "startDate")

			SELECT uuid_generate_v4(), uwp."userId", $1, $2, lower(tstzrange(uwp."startDateTime", uwp."endDateTime")*tstzrange(wpp."periodStartDateTime", null))
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp 
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods}" wpp 
				ON wpp."workPlanId" = uwp."workPlanId"
			WHERE 	uwp."workPlanId" = $3 
					AND wpp."deletedAt" IS NULL
					AND NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime")*tstzrange(wpp."periodStartDateTime", null))
			`, [organizationId, dal_constants_1.DalConstants.RecalculateWorkReason.WorkPlanDefinitionChange, workPlanId]);
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" eld 
			SET ("workPlanIds", "updateRequiredRaisedAt", "updateRequiredReason") = ("workPlanIds" - $1, now(), $2)
			WHERE eld."workPlanIds" ? $1;	
			`, [workPlanId, dal_constants_1.DalConstants.RecalculateWorkReason.WorkPlanDefinitionChange]);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
			WHERE jsonb_array_length("workPlanIds") = 0;
		`);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
			WHERE id = $1 
			`, [workPlanId]);
    }
    checkPeriodsForOptionalBreaks(workplanType, periodDetails) {
        let fbreaks = [];
        for (const detail of periodDetails) {
            switch (workplanType) {
                case app_enums_1.enums.WorkPlanType.REGULAR:
                case app_enums_1.enums.WorkPlanType.HALF_FLEXIBLE: {
                    let dayItems = detail.dayItems;
                    for (const item of dayItems) {
                        if (item.fixedBreaks) {
                            fbreaks.push(...item.fixedBreaks);
                        }
                    }
                    break;
                }
                case app_enums_1.enums.WorkPlanType.SHIFT:
                case app_enums_1.enums.WorkPlanType.FULL_FLEXIBLE: {
                    let dayItems = detail.dayBreakItems;
                    for (const item of dayItems) {
                        if (item.fixedBreaks) {
                            fbreaks.push(...item.fixedBreaks);
                        }
                    }
                    break;
                }
            }
        }
        let hasOptionalBreak = false;
        for (const fbreak of fbreaks) {
            if ((fbreak.allowedMaxBreakDuration && !fbreak.executeAutomaticallyBelowDuration) || fbreak.executeAutomaticallyBelowDuration < fbreak.allowedMaxBreakDuration) {
                hasOptionalBreak = true;
                break;
            }
        }
        return hasOptionalBreak;
    }
    async getOrganizationUnitsWithPermission(trx, organizationId, userId, permissions, hierarchical) {
        let query = `
		SELECT array_agg(uoou."organizationUnitId"::text) as ouids FROM "${organizationId}"."userOrganizations" uo 
		INNER JOIN "${organizationId}"."userOrganizationOrganizationUnits" uoou
		ON uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND uoou."deletedAt" IS NULL AND uoou."userOrganizationId" = uo.id
		INNER JOIN "${organizationId}"."roles" r ON r."deletedAt" IS NULL AND  r.id = uoou."roleId"
		WHERE uo."userId" = $1
		`;
        const bindings = [userId];
        for (const perm of permissions) {
            bindings.push(perm);
            query += `
			AND POSITION($${bindings.length}:: text IN r."permissions") > 0`;
        }
        if (hierarchical) {
            query =
                `
			SELECT array_agg(DISTINCT id) as ouids FROM "${organizationId}"."organizationUnits" oun
			WHERE oun."deletedAt" IS NULL AND array_append( string_to_array(oun."ancestorIds", ','), id::text) &&
			(` +
                    query +
                    `)`;
        }
        let result = (await trx.query(query, bindings)).rows[0];
        return result ? result.ouids : [];
    }
    async upsertWorkPlan(organizationId, requestUserId, hasGlobalPermission, args) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            if (!args.id) {
                let nameExists = await trx.query(`SELECT COUNT(*) as count FROM "${organizationId}"."workPlans" WHERE name = $1`, [args.name]);
                if (parseInt(nameExists.rows[0].count) > 0) {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.PACS.WORKPLANNAMECONFLICT");
                }
            }
            if (!hasGlobalPermission) {
                if (!args.organizationUnitId) {
                    (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.PACS.WORKPLANREQUIRESORGANIZATOINUNIT");
                }
                else {
                    const possibleOrgUnits = await this.getOrganizationUnitsWithPermission(trx, organizationId, requestUserId, ["h:w"], true);
                    if (!possibleOrgUnits.includes(args.organizationUnitId)) {
                        (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.PACS.UNAUTHORIZEDFORORGANIZATIONUNIT");
                    }
                }
            }
            let recalculateRequired = false;
            let id = args.id || uuid.v4();
            if (args.id) {
                let oldWp = (await trx.query(`SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}" WHERE id = $1`, [args.id])).rows[0];
                if (oldWp) {
                    recalculateRequired =
                        oldWp.regionId !== args.regionId ||
                            oldWp.accessCheckType !== args.accessCheckType ||
                            oldWp.ignoreHolidays !== args.ignoreHolidays ||
                            oldWp.timezone !== args.timezone ||
                            oldWp.allowMobileCheckins !== args.allowMobileCheckins ||
                            oldWp.allowUnreliableCheckins !== args.allowUnreliableCheckins ||
                            oldWp.permissionRequiredForExtraWorking !== args.permissionRequiredForExtraWorking ||
                            oldWp.ignoreUnpairedAccesses !== args.ignoreUnpairedAccesses ||
                            oldWp.offset !== args.offset;
                }
                if (oldWp.accessCheckType === dal_constants_1.DalConstants.WorkPlanAccessCheckType.Region && args.accessCheckType === dal_constants_1.DalConstants.WorkPlanAccessCheckType.FirstAndLastAccess) {
                    let oldPeriods = await trx.query(`
                    SELECT details FROM "${organizationId}"."workPlanPeriods" WHERE "workPlanId" = $1 AND "deletedAt" IS NULL`, [args.id]);
                    let details = oldPeriods.rows.map((r) => r.details);
                    let hasOptionalBreaks = await this.checkPeriodsForOptionalBreaks(oldWp.type, details);
                    if (hasOptionalBreaks) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.PACS.WORKPLANHASOPTINALBREAKS");
                    }
                }
                await trx.query(`
                UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
                SET ("organizationId","name","type","updatedAt","accessCheckType","regionId","timezone","ignoreHolidays","offset","allowMobileCheckins","mobileCheckinRequiresLocation","allowUnreliableCheckins","permissionRequiredForExtraWorking", "typedGeoLocations", "colorCode", "ignoreUnpairedAccesses", "organizationUnitId")
                = ($2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,$17,$18)
                WHERE id = $1
                `, [
                    args.id,
                    organizationId,
                    args.name,
                    args.type,
                    new Date(),
                    args.accessCheckType,
                    args.regionId === undefined ? null : args.regionId,
                    args.timezone ? args.timezone : "+03:00",
                    args.ignoreHolidays,
                    args.offset ? args.offset : 0,
                    args.allowMobileCheckins,
                    args.mobileCheckinRequiresLocation,
                    args.allowUnreliableCheckins,
                    args.permissionRequiredForExtraWorking,
                    args.typedGeoLocations
                        ? JSON.stringify(args.typedGeoLocations?.map((g) => {
                            if (g.type === dal_constants_1.DalConstants.GeoLocationType.Circle) {
                                return {
                                    t: g.type,
                                    lo: g.longitude,
                                    la: g.latitude,
                                    r: g.radius,
                                    n: g.name,
                                };
                            }
                            else {
                                return {
                                    t: g.type,
                                    n: g.name,
                                    p: g.points.map((point) => {
                                        return {
                                            lo: point.longitude,
                                            la: point.latitude,
                                        };
                                    }),
                                };
                            }
                        }))
                        : null,
                    args.colorCode,
                    args.ignoreUnpairedAccesses,
                    args.organizationUnitId ? args.organizationUnitId : null,
                ]);
            }
            else {
                await trx.query(`
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
                ("id","organizationId","name","type","createdAt","accessCheckType","regionId","timezone","ignoreHolidays","offset","allowMobileCheckins","mobileCheckinRequiresLocation", "allowUnreliableCheckins","permissionRequiredForExtraWorking", "typedGeoLocations", "colorCode", "ignoreUnpairedAccesses", "organizationUnitId")
                VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,$17,$18)
                `, [
                    id,
                    organizationId,
                    args.name,
                    args.type,
                    new Date(),
                    args.accessCheckType,
                    args.regionId === undefined ? null : args.regionId,
                    args.timezone ? args.timezone : "+03:00",
                    args.ignoreHolidays,
                    args.offset ? args.offset : 0,
                    args.allowMobileCheckins,
                    args.mobileCheckinRequiresLocation,
                    args.allowUnreliableCheckins,
                    args.permissionRequiredForExtraWorking,
                    args.typedGeoLocations
                        ? JSON.stringify(args.typedGeoLocations?.map((g) => {
                            if (g.type === dal_constants_1.DalConstants.GeoLocationType.Circle) {
                                return {
                                    t: g.type,
                                    n: g.name,
                                    lo: g.longitude,
                                    la: g.latitude,
                                    r: g.radius,
                                };
                            }
                            else {
                                return {
                                    t: g.type,
                                    n: g.name,
                                    p: g.points.map((point) => {
                                        return {
                                            lo: point.longitude,
                                            la: point.latitude,
                                        };
                                    }),
                                };
                            }
                        }))
                        : null,
                    args.colorCode,
                    args.ignoreUnpairedAccesses,
                    args.organizationUnitId ? args.organizationUnitId : null,
                ]);
            }
            if (recalculateRequired) {
                await trx.query(`
            INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.recalculateWorkQueue}"
                    (id, "userId", "organizationId", reason, "startDate")

            SELECT uuid_generate_v4(), uwp."userId", $1, $2, lower(tstzrange(uwp."startDateTime", uwp."endDateTime") * tstzrange(wpp."periodStartDateTime", null))
            FROM "${organizationId}"."userWorkPlans" uwp INNER JOIN "${organizationId}"."workPlanPeriods" wpp ON wpp."workPlanId" = uwp."workPlanId"
            WHERE uwp."workPlanId" = $3 AND wpp."deletedAt" IS NULL
            AND NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime") * tstzrange(wpp."periodStartDateTime", null))
                    `, [organizationId, dal_constants_1.DalConstants.RecalculateWorkReason.WorkPlanDefinitionChange, id]);
                await trx.query(`
            UPDATE "${organizationId}"."employeeLiveDay" eld SET "updateRequiredReason" = "updateRequiredReason" | $1, "updateRequiredRaisedAt" = now()
            WHERE eld."workPlanIds" ? $2
                    `, [app_enums_1.enums.RecalculateWorkReason.WorkPlanDefinitionChange, id]);
            }
            await dal_manager_1.dbManager.accessAccessControlPoint.createMobileAccessControlPointAndSetAdmin(organizationId, requestUserId);
            return id;
        });
    }
    async onEmployeeMobileCheckin(params) {
        const { date, organizationId, userId, trx } = params;
        await trx.query(`
            UPDATE "${organizationId}"."employeeLiveDay" eld SET "updateRequiredReason" = "updateRequiredReason" | $1, "updateRequiredRaisedAt" = now()
            WHERE eld."userId" = $2
            `, [app_enums_1.enums.RecalculateWorkReason.NewAccessLog, userId]);
        if (moment(date).isBefore(moment().startOf("day"))) {
            await trx.query(`
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.recalculateWorkQueue}"
                (id, "userId", "organizationId", reason, "startDate")

                SELECT uuid_generate_v4(), uwp."userId", $1, $2, lower(tstzrange(uwp."startDateTime", uwp."endDateTime")*tstzrange(wpp."periodStartDateTime", null)*tstzrange($3, null))
                FROM "${organizationId}"."userWorkPlans" uwp
                INNER JOIN "${organizationId}"."workPlanPeriods" wpp ON wpp."workPlanId" = uwp."workPlanId"
				WHERE uwp."userId" = $4 AND wpp."deletedAt" IS NULL
				AND lower(tstzrange(uwp."startDateTime", uwp."endDateTime")*tstzrange(wpp."periodStartDateTime", null)*tstzrange($3, null)) IS NOT NULL
                AND NOT isEmpty(tstzrange(uwp."startDateTime", uwp."endDateTime")*tstzrange(wpp."periodStartDateTime", null))
                `, [organizationId, dal_constants_1.DalConstants.RecalculateWorkReason.NewAccessLog, date, userId]);
        }
    }
    async getWeeklyRemainingWorkDurationOfUser(params) {
        const trxx = params.trx ?? this._pgPool;
        let dbQueryDate = params.day.toDate();
        let mondayDate = params.day.clone().isoWeekday(app_enums_1.enums.DayOfWeekForWorking.Monday);
        let remainingDuration = null;
        if (params.day.isoWeekday() == app_enums_1.enums.DayOfWeekForWorking.Monday) {
            remainingDuration = params.weeklyExpectedWorkDuration * 60;
        }
        else if (params.previousDays) {
            for (let index = params.previousDays.length - 1; index >= 0 && remainingDuration === null; index--) {
                const element = params.previousDays[index];
                if (moment(element.d).isSameOrAfter(mondayDate, "day")) {
                    let remaining = element.r.find((wrwd) => wrwd.w === params.workPlanId);
                    if (remaining) {
                        remainingDuration = remaining.r;
                    }
                    else {
                        dbQueryDate = element.d;
                    }
                }
            }
        }
        if (remainingDuration === null) {
            let qx = 1;
            const q = `
				SELECT * FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
				WHERE "userId" = $${qx++} AND date < $${qx++} AND date >= $${qx++} AND data -> 'w' ? $${qx++}
				ORDER BY date DESC
				LIMIT 1
			`;
            const { rows, rowCount } = await trxx.query(q, [params.userId, dbQueryDate, mondayDate.toDate(), params.workPlanId]);
            const previousAvailableDayData = rowCount > 0 ? rows[0] : null;
            const previousDaysRemainingWorkDurations = previousAvailableDayData ? previousAvailableDayData.data.r.find((e) => e.w === params.workPlanId) : null;
            if (!previousDaysRemainingWorkDurations) {
                let ratio = 8 - params.day.isoWeekday();
                remainingDuration = Math.round(params.weeklyExpectedWorkDuration * 60 * (ratio / 7.0));
            }
            else {
                remainingDuration = previousDaysRemainingWorkDurations.r;
            }
        }
        return remainingDuration;
    }
    async getFirstRecalculateTask() {
        return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            return (await trx.query(`SELECT * FROM public.select_pacs_recalculate_task() ORDER BY "startDate" DESC;`)).rows[0];
        });
    }
    insertEmployeeDaySummary(userId, organizationId, startDate, summaries, rwqIdToRemove) {
        return this.dbClient.transaction(async (trx) => {
            await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays)
                .where("userId", userId)
                .where("organizationId", organizationId)
                .where("date", ">=", startDate)
                .delete();
            await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays)
                .insert(summaries.map((s) => {
                return {
                    id: uuid.v4(),
                    userId: userId,
                    organizationId: organizationId,
                    date: s.d,
                    data: s,
                };
            }));
            if (rwqIdToRemove) {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.recalculateWorkQueue).where("id", rwqIdToRemove).delete();
            }
        });
    }
    async onAccessLogsChanged(params) {
        return this.dbClient.transaction(async (trx) => {
            if (params.organizationId && params.userId && params.acpId) {
                let boundQuery = trx
                    .raw(`
                    UPDATE "${params.organizationId}"."employeeLiveDay" eld
                    SET "updateRequiredReason" = "updateRequiredReason" | ?, "updateRequiredRaisedAt" = now()
                    FROM "${params.organizationId}"."workPlans" wp
                    INNER JOIN "${params.organizationId}"."regions" r ON wp."regionId" = r."id"
                    INNER JOIN "${params.organizationId}"."regionAccessControlPoints" AS racp ON r."id" = racp."regionId"
                    WHERE ? IN (racp."accessControlPointId")
                    AND eld."userId" = ? AND eld."organizationId" = ?
                    AND eld."workPlanIds" @> to_jsonb(wp.id)
                    AND (wp."accessCheckType" = 2 OR (wp."accessCheckType" = 1 AND (? = 1 OR ? = 2)))
                `, [params.reason, params.acpId, params.userId, params.organizationId, params.direction || 99, params.direction || 99])
                    .toQuery();
                await trx.raw(boundQuery);
                boundQuery = trx
                    .raw(`
                    INSERT INTO "${params.organizationId}"."recalculateWorkQueue"
                    ("id", "userId", "organizationId", "reason", "startDate")
                    SELECT uuid_generate_v4(), uwp."userId", wp."organizationId", ?, ?
                    FROM "${params.organizationId}"."userWorkPlans" uwp
                    INNER JOIN "${params.organizationId}"."workPlans" wp ON wp."id" = uwp."workPlanId"
                    INNER JOIN "${params.organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = wp."id" AND wpp."organizationId" = ?
                    INNER JOIN "${params.organizationId}"."userOrganizations" uo ON uo."userId" = uwp."userId" AND uo."organizationId" = wp."organizationId"
                        AND uo."deletedAt" IS NULL AND uo."isDisabled" = false
                        AND (tstzrange(uwp."startDateTime", uwp."endDateTime")*wpp."range") @> (?::timestamp with time zone)
                    INNER JOIN "${params.organizationId}"."regions" r ON wp."regionId" = r."id"
                    INNER JOIN "${params.organizationId}"."regionAccessControlPoints" AS racp ON r."id" = racp."regionId"
                    WHERE ? IN (racp."accessControlPointId")
                    AND (wp."accessCheckType" = 2 OR (wp."accessCheckType" = 1 AND (? = 1 OR ? = 2)))
                    AND uwp."userId"=? AND wp."organizationId"=?;
                    `, [
                    params.reason,
                    params.timestamp,
                    params.organizationId,
                    params.timestamp,
                    params.acpId,
                    params.direction || 99,
                    params.direction || 99,
                    params.userId,
                    params.organizationId,
                ])
                    .toQuery();
                await trx.raw(boundQuery);
            }
        });
    }
    async getBasicUserInfo(organizationId, userId, trx) {
        if (!userId)
            return Promise.resolve(null);
        let result = {
            id: "",
            fullname: "",
            uniqueId: "",
            captionLines: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.id", "uop.userOrganizationId")
            .where("uo.organizationId", organizationId)
            .where("uo.userId", userId)
            .whereNull("uop.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb
            .select(["uo.userId", "uop.name", "uop.surname", "uop.uniqueId"])
            .first()
            .then((row) => {
            if (row) {
                result.id = row.userId;
                result.fullname = (row.name ?? "") + " " + (row.surname ?? "");
                result.uniqueId = row.uniqueId;
                result.captionLines = [];
            }
        });
        return Promise.resolve(result);
    }
    async getBasicInfoOfPermissions(organizationId, permissionIds, trx) {
        let result = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions + " as pp")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes + " as ppt", "pp.ppermissionTypeId", "ppt.id")
            .whereIn("pp.id", permissionIds)
            .select("pp.id", "ppt.id as typeId", "ppt.name", "ppt.type", "pp.startDateTime", "pp.endDateTime", "ppt.method");
        if (trx)
            qb.transacting(trx);
        await qb.then((rows) => {
            if (rows && rows.length > 0) {
                result = rows.map((r) => {
                    return {
                        id: r.id,
                        typeId: r.typeId,
                        name: r.name,
                        type: r.method,
                        range: new moment_range_1.DateRange(moment(r.startDateTime), moment(r.endDateTime)),
                        usedDuration: 0,
                    };
                });
            }
        });
        return result;
    }
    async getBasicInfoOfWorkPlans(organizationId, workPlanIds) {
        let query = `
        SELECT "id", "name", "type", "colorCode"
        FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
        WHERE "deletedAt" IS NULL AND "id" = ANY ($1::uuid[])
        `;
        const { rows } = await this._pgPool.query(query, [workPlanIds]);
        return rows;
    }
    async getBasicInfoOfHolidys(organizationId, holidayIds, trx) {
        let result = [];
        await trx
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.vacations)
            .whereIn("id", holidayIds)
            .select("id", "title", "startDateTime", "endDateTime")
            .then((rows) => {
            if (rows && rows.length > 0) {
                result = rows.map((r) => {
                    return {
                        id: r.id,
                        name: r.title,
                        range: new moment_range_1.DateRange(moment(r.startDateTime), moment(r.endDateTime)),
                        usedDuration: 0,
                    };
                });
            }
        });
        return result;
    }
    async getWorkPlan(organizationId, workPlanId, trx) {
        let result = {
            id: workPlanId,
            name: "",
            type: 0,
            workPlanFullFlexiblePeriods: [],
            workPlanHalfFlexiblePeriods: [],
            workPlanRegularPeriods: [],
            workPlanShiftPeriods: [],
            regionId: "",
            regionName: "",
            accessCheckType: dal_constants_1.DalConstants.WorkPlanAccessCheckType.Region,
            offset: 0,
            allowMobileCheckins: false,
            mobileCheckinRequiresLocation: false,
            allowUnreliableCheckins: false,
            permissionRequiredForExtraWorking: false,
            colorCode: "FF0000",
            organizationUnitId: null,
            isEditable: false,
            ignoreUnpairedAccesses: false,
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("workPlans as wp")
            .leftJoin("regions as r", "r.id", "wp.regionId")
            .where("wp.id", workPlanId)
            .where("wp.organizationId", organizationId)
            .whereNull("wp.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb
            .select("wp.name", "wp.type", "wp.accessCheckType", "wp.offset", "r.id as regionId", "r.name as regionName", "wp.allowMobileCheckins", "wp.mobileCheckinRequiresLocation", "wp.allowUnreliableCheckins", "wp.permissionRequiredForExtraWorking", "wp.colorCode", "wp.ignoreUnpairedAccesses", "wp.organizationUnitId")
            .first()
            .then((row) => {
            if (row) {
                (result.name = row.name), (result.type = row.type), (result.regionId = row.regionId), (result.regionName = row.regionName);
                result.accessCheckType = row.accessCheckType;
                result.offset = row.offset;
                result.allowMobileCheckins = row.allowMobileCheckins;
                result.mobileCheckinRequiresLocation = row.mobileCheckinRequiresLocation;
                result.allowUnreliableCheckins = row.allowUnreliableCheckins;
                result.permissionRequiredForExtraWorking = row.permissionRequiredForExtraWorking;
                result.colorCode = row.colorCode;
                result.ignoreUnpairedAccesses = row.ignoreUnpairedAccesses;
                result.organizationUnitId = row.organizationUnitId;
            }
        });
        let qbx = this.dbClient.withSchema(organizationId).table("workPlanPeriods").whereNull("deletedAt").orderBy("periodStartDateTime", "DESC").where("workPlanId", workPlanId);
        if (trx)
            qbx = qbx.transacting(trx);
        await qbx.select().then((rows) => {
            if (rows) {
                rows.forEach((element) => {
                    let detail = Object.assign({
                        id: element.id,
                        workPlanId: workPlanId,
                        periodStartDateTime: element.periodStartDateTime,
                    }, element.details);
                    switch (result.type) {
                        case dal_constants_1.DalConstants.WorkPlanType.SHIFT:
                            result.workPlanShiftPeriods.push(detail);
                            break;
                        case dal_constants_1.DalConstants.WorkPlanType.REGULAR:
                            result.workPlanRegularPeriods.push(detail);
                            break;
                        case dal_constants_1.DalConstants.WorkPlanType.HALF_FLEXIBLE:
                            result.workPlanHalfFlexiblePeriods.push(detail);
                            break;
                        case dal_constants_1.DalConstants.WorkPlanType.FULL_FLEXIBLE:
                            result.workPlanFullFlexiblePeriods.push(detail);
                            break;
                        default:
                    }
                });
            }
        });
        return Promise.resolve(result);
    }
    async getCurrentWorkPlanPeriod(organizationId, workPlanId, dateUTC, trx) {
        let workPlanDetail = await this.getWorkPlan(organizationId, workPlanId, trx);
        switch (workPlanDetail.type) {
            case dal_constants_1.DalConstants.WorkPlanType.REGULAR:
                for (let detail of workPlanDetail.workPlanRegularPeriods) {
                    if (detail.periodStartDateTime <= dateUTC)
                        return Promise.resolve(detail);
                }
                break;
            case dal_constants_1.DalConstants.WorkPlanType.FULL_FLEXIBLE:
                for (let detail of workPlanDetail.workPlanFullFlexiblePeriods) {
                    if (detail.periodStartDateTime <= dateUTC)
                        return Promise.resolve(detail);
                }
                break;
            case dal_constants_1.DalConstants.WorkPlanType.HALF_FLEXIBLE:
                for (let detail of workPlanDetail.workPlanHalfFlexiblePeriods) {
                    if (detail.periodStartDateTime <= dateUTC)
                        return Promise.resolve(detail);
                }
                break;
            case dal_constants_1.DalConstants.WorkPlanType.SHIFT:
                for (let detail of workPlanDetail.workPlanShiftPeriods) {
                    if (detail.periodStartDateTime <= dateUTC)
                        return Promise.resolve(detail);
                }
                break;
            default:
                break;
        }
    }
    getReportUserFilter(params) {
        let bindingKeys = [];
        let organizationWidePermissions = "";
        if (params.requiredOrganizationWidePermissions && params.requiredOrganizationWidePermissions.length > 0) {
            organizationWidePermissions = " ( ";
            let pa = [];
            for (const p of params.requiredOrganizationWidePermissions) {
                pa.push(` POSITION(?IN "permissions") > 0 `);
                bindingKeys.push(p);
            }
            organizationWidePermissions += pa.join(" AND ") + " ) AND ";
        }
        bindingKeys.push(params.requesterUserId, params.organizationId);
        let organizationUnitWidePermissions = "";
        if (params.requiredOrganizationUnitWidePermissions && params.requiredOrganizationUnitWidePermissions.length > 0) {
            organizationUnitWidePermissions = " ( ";
            let pa = [];
            for (const p of params.requiredOrganizationUnitWidePermissions) {
                pa.push(` POSITION(?IN "permissions") > 0 `);
                bindingKeys.push(p);
            }
            organizationUnitWidePermissions += pa.join(" AND ") + " ) AND ";
        }
        bindingKeys.push(params.requesterUserId, params.organizationId, params.organizationId);
        let userIds = "";
        if (params.idBasedUserFilter.userIds && params.idBasedUserFilter.userIds.length > 0) {
            userIds = " AND ( ";
            let ui = [];
            for (const userId of params.idBasedUserFilter.userIds) {
                ui.push(` uo."userId" = ? `);
                bindingKeys.push(userId);
            }
            userIds += ui.join(" OR ") + " ) ";
        }
        let userGroupIds = "";
        let userGroupJoin = "";
        if (params.idBasedUserFilter.userGroupIds && params.idBasedUserFilter.userGroupIds.length > 0) {
            userGroupJoin = `
            INNER JOIN "${params.organizationId}"."userGroupUserOrganizations" AS uguo ON uguo."userOrganizationId" = uo."id" AND uguo."deletedAt" IS NULL
    `;
            userGroupIds = " AND ( ";
            let ug = [];
            for (const userGroupId of params.idBasedUserFilter.userGroupIds) {
                ug.push(` uguo."userGroupId" = ? `);
                bindingKeys.push(userGroupId);
            }
            userGroupIds += ug.join(" OR ") + " ) ";
        }
        let organizationUnitIds = "";
        if (params.idBasedUserFilter.organizationUnitIds && params.idBasedUserFilter.organizationUnitIds.length > 0) {
            organizationUnitIds = " AND ( ";
            let ou = [];
            for (const organizationUnitId of params.idBasedUserFilter.organizationUnitIds) {
                if (params.idBasedUserFilter.applyOrganizationUnitFilterHierarchically) {
                    ou.push(` POSITION(?IN TUSERS."organizationUnitPath") > 0 `);
                }
                else {
                    ou.push(` ou."id" = ? `);
                }
                bindingKeys.push(organizationUnitId);
            }
            organizationUnitIds += ou.join(" OR ") + " ) ";
        }
        let userOrganizationIsDisabledFilter = "";
        if (params.idBasedUserFilter.userOrganizationStatus && params.idBasedUserFilter.userOrganizationStatus !== dal_constants_1.DalConstants.IdentityStatusType.All) {
            userOrganizationIsDisabledFilter =
                params.idBasedUserFilter.userOrganizationStatus === dal_constants_1.DalConstants.IdentityStatusType.Enabled
                    ? (userOrganizationIsDisabledFilter = `AND uo."isDisabled" = false `)
                    : (userOrganizationIsDisabledFilter = `AND uo."isDisabled" = true `);
        }
        let q = `
SELECT DISTINCT * FROM(
    SELECT T1."userId" AS "requesterUserId", T2."userId", T2."userOrganizationId", T2."organizationId",
    T2."name", T2."surname", T2."uniqueId" FROM
    (
        --requesterUser
        SELECT "userOrganizationId", "userId", "organizationId", "organizationUnitId", "isOrganizationWide"
        FROM
            (SELECT uo."id" AS "userOrganizationId", uo."userId", uo."organizationId",
                r."permissions", NULL:: text AS "organizationUnitId", TRUE AS "isOrganizationWide"
        FROM "${params.organizationId}"."userOrganizations" AS uo
        INNER JOIN "${params.organizationId}"."roles" AS r ON r."id" = uo."roleId" AND r."deletedAt" IS NULL
        WHERE
        ${organizationWidePermissions}
        uo."deletedAt" IS NULL
        AND uo."userId" = ? AND uo."organizationId" = ?

                UNION

        SELECT uo."id" AS "userOrganizationId", uo."userId", uo."organizationId",
                r."permissions", uoou."organizationUnitId":: text, FALSE AS "isOrganizationWide"
        FROM "${params.organizationId}"."userOrganizations" AS uo
        INNER JOIN "${params.organizationId}"."userOrganizationOrganizationUnits" AS uoou ON uoou."userOrganizationId" = uo."id" AND uoou."deletedAt" IS NULL
        INNER JOIN "${params.organizationId}"."roles" AS r ON r."id" = uoou."roleId" AND r."deletedAt" IS NULL
        WHERE
        ${organizationUnitWidePermissions}
        uo."deletedAt" IS NULL
        AND uo."userId" = ? AND uo."organizationId" = ?

        ) AS TPERMISSION) AS T1
        INNER JOIN
        --requesting users
    (SELECT TUSERS."userOrganizationId", TUSERS."userId", TUSERS."organizationUnitPath",
        TUSERS."organizationId", uop."name", uop."surname", uop."uniqueId"
        FROM
            (SELECT uo."id" AS "userOrganizationId", uo."userId", uo."organizationId",
                STRING_AGG((CASE WHEN ou."ancestorIds" IS NULL THEN ou."id":: text ELSE ou."id":: text || ',' || ou."ancestorIds" END), ',') AS "organizationUnitPath"
        FROM "${params.organizationId}"."userOrganizations" AS uo
        LEFT JOIN "${params.organizationId}"."userOrganizationOrganizationUnits" AS uoou ON uoou."userOrganizationId" = uo."id" AND uoou."deletedAt" IS NULL
        LEFT JOIN "${params.organizationId}"."organizationUnits" AS ou ON uoou."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL
        ${userGroupJoin}
        WHERE
        uo."organizationId" = ?
        AND uo."deletedAt" IS NULL
        ${userOrganizationIsDisabledFilter}
        ${userIds}
        ${userGroupIds}
        ${!params.idBasedUserFilter.applyOrganizationUnitFilterHierarchically ? organizationUnitIds : ""}
        GROUP BY uo."id") AS TUSERS
        INNER JOIN "${params.organizationId}"."userOrganizationProfiles" AS uop ON uop."userOrganizationId" = TUSERS."userOrganizationId" AND uop."deletedAt" IS NULL
        ${params.idBasedUserFilter.applyOrganizationUnitFilterHierarchically ? organizationUnitIds : ""}
) AS T2
ON T1."isOrganizationWide" = TRUE OR POSITION(T1."organizationUnitId" IN T2."organizationUnitPath") > 0
        ) AS TDIFF
    `;
        return {
            query: q,
            bindingKeys: bindingKeys,
        };
    }
    getSortAndFilter(params) {
        let sortOrder = ["ASC", "DESC"].includes(params.sortOrder) ? params.sortOrder : "DESC";
        let reverseSortOrder = sortOrder === "DESC" ? "ASC" : "DESC";
        const sortType = params.sortType ? params.sortType : dal_constants_1.DalConstants.WorkSummarySortType.Name;
        let q = "";
        switch (sortType) {
            case dal_constants_1.DalConstants.WorkSummarySortType.Name:
                q += ` "name" ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.UniqueId:
                q += ` "uniqueId" ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.LateStart:
                q += ` (CASE WHEN(ed."data" -> 'ex'->>'s') IS NULL THEN NULL ELSE ed."data" #> '{s,ls}' END) ${sortOrder} NULLS LAST,
    (ed."data" -> 'ex'->>'s'):: timestamp with time zone ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.EarylLeave:
                q += ` (CASE WHEN(ed."data" -> 'ex'->>'e') IS NULL THEN NULL ELSE ed."data" #> '{s,el}' END)
${sortOrder} NULLS LAST, (ed."data" -> 'ex'->>'e'):: timestamp with time zone ${reverseSortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.FirstStart:
                q += ` (ed."data" -> 'ex'->>'s'):: timestamp with time zone ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.LastLeave:
                q += ` (ed."data" -> 'ex'->>'e'):: timestamp with time zone ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.ExtraWork:
                q += ` (CASE WHEN(ed."data" -> 'ex'->>'s') IS NULL THEN NULL ELSE ed."data" #> '{s,e}' END) ${sortOrder} NULLS LAST,
    (ed."data" -> 'ex'->>'e'):: timestamp with time zone ASC NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.MissingWork:
                q += ` ed."data" #> '{s,m}' ${sortOrder},
				(ed."data" -> 'ex'->>'s'):: timestamp with time zone ${reverseSortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.Date:
                q += ` date ${sortOrder} NULLS LAST`;
                break;
            default:
                break;
        }
        q += " OFFSET " + (params.skip !== null && params.skip !== undefined ? params.skip : 0);
        q += " LIMIT " + (params.take !== null && params.take !== undefined ? params.take : 100);
        return q;
    }
    collectAttendanceEmployeeTimeRangeTypePair(params) {
        let result = [];
        if (params.working === false) {
            switch (params.status) {
                case app_enums_1.enums.EmployeeStatus.AtBreak:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtHoliday:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtOffTimePermission:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtOutOfWorkingHours:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtWorkingHours:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        });
                    }
                    break;
                case undefined:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                        });
                    }
                    break;
                default:
                    break;
            }
        }
        else if (params.working === true) {
            switch (params.status) {
                case app_enums_1.enums.EmployeeStatus.AtBreak:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtHoliday:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtOffTimePermission:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtOutOfWorkingHours:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtWorkingHours:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    break;
                case undefined:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                        });
                    }
                    else {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                        });
                    }
                    break;
                default:
                    break;
            }
        }
        else {
            switch (params.status) {
                case app_enums_1.enums.EmployeeStatus.AtBreak:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    else {
                        result.push({
                            et: app_enums_1.enums.EmployeeTimeRangeType.Break,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtHoliday:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    else {
                        result.push({
                            et: app_enums_1.enums.EmployeeTimeRangeType.Holiday,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtOffTimePermission:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    else {
                        result.push({
                            et: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtOutOfWorkingHours:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    else {
                        result.push({
                            et: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        });
                    }
                    break;
                case app_enums_1.enums.EmployeeStatus.AtWorkingHours:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    else {
                        result.push({
                            et: app_enums_1.enums.EmployeeTimeRangeType.WorkingHours,
                        }, {
                            et: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        });
                    }
                    break;
                case undefined:
                    if (params.inRegion === true) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                        });
                    }
                    else if (params.inRegion === false) {
                        result.push({
                            at: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                        }, {
                            at: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                        });
                    }
                    else {
                    }
                    break;
                default:
                    break;
            }
        }
        return result;
    }
    async getDistributionOfEmployeesAtDateTime(requesterUserId, organizationId, filter) {
        let workingDistribution = {
            AtBreak: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtBreak,
            },
            AtHoliday: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtHoliday,
            },
            AtOffTimePermission: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtOffTimePermission,
            },
            AtOutOfWorkingHours: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours,
            },
            AtWorkingHours: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours,
            },
        };
        let notWorkingDistribution = {
            AtBreak: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtBreak,
            },
            AtHoliday: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtHoliday,
            },
            AtOffTimePermission: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtOffTimePermission,
            },
            AtOutOfWorkingHours: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours,
            },
            AtWorkingHours: {
                count: 0,
                inRegion: 0,
                outRegion: 0,
                type: dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours,
            },
        };
        let result = {
            filter: null,
            total: 0,
            working: 0,
            notWorking: 0,
            workingDistribution: [],
            notWorkingDistribution: [],
        };
        return this.dbClient.transaction(async (trx) => {
            if (moment(filter.dateTime).isAfter(moment())) {
                filter.dateTime = moment().toDate();
            }
            let qresult = await this.getRawStatisticsAtDateTime(requesterUserId, organizationId, filter, trx);
            for (const c of qresult) {
                if (c.expectedType === dal_constants_1.DalConstants.EmployeeTimeRangeType.Break) {
                    switch (c.actualType) {
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.Absent:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtBreak.count += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtBreak.count += c.count;
                            notWorkingDistribution.AtBreak.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                            result.working += c.count;
                            workingDistribution.AtBreak.count += c.count;
                            workingDistribution.AtBreak.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.DeclaredAttendance:
                            result.working += c.count;
                            workingDistribution.AtBreak.count += c.count;
                            workingDistribution.AtBreak.outRegion += c.count;
                            break;
                        default:
                            break;
                    }
                }
                else if (c.expectedType === dal_constants_1.DalConstants.EmployeeTimeRangeType.FlexibleWorkingHours || c.expectedType === dal_constants_1.DalConstants.EmployeeTimeRangeType.WorkingHours) {
                    switch (c.actualType) {
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.Absent:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtWorkingHours.count += c.count;
                            notWorkingDistribution.AtWorkingHours.outRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtWorkingHours.count += c.count;
                            notWorkingDistribution.AtWorkingHours.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                            result.working += c.count;
                            workingDistribution.AtWorkingHours.count += c.count;
                            workingDistribution.AtWorkingHours.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.DeclaredAttendance:
                            result.working += c.count;
                            workingDistribution.AtWorkingHours.count += c.count;
                            workingDistribution.AtWorkingHours.outRegion += c.count;
                            break;
                        default:
                            break;
                    }
                }
                else if (c.expectedType === dal_constants_1.DalConstants.EmployeeTimeRangeType.Holiday) {
                    switch (c.actualType) {
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.Absent:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtHoliday.count += c.count;
                            notWorkingDistribution.AtHoliday.outRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtHoliday.count += c.count;
                            notWorkingDistribution.AtHoliday.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                            result.working += c.count;
                            workingDistribution.AtHoliday.count += c.count;
                            workingDistribution.AtHoliday.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.DeclaredAttendance:
                            result.working += c.count;
                            workingDistribution.AtHoliday.count += c.count;
                            workingDistribution.AtHoliday.outRegion += c.count;
                            break;
                        default:
                            break;
                    }
                }
                else if (c.expectedType === dal_constants_1.DalConstants.EmployeeTimeRangeType.OffTimePermission) {
                    switch (c.actualType) {
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.Absent:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtOffTimePermission.count += c.count;
                            notWorkingDistribution.AtOffTimePermission.outRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtOffTimePermission.count += c.count;
                            notWorkingDistribution.AtOffTimePermission.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                            result.working += c.count;
                            workingDistribution.AtOffTimePermission.count += c.count;
                            workingDistribution.AtOffTimePermission.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.DeclaredAttendance:
                            result.working += c.count;
                            workingDistribution.AtOffTimePermission.count += c.count;
                            workingDistribution.AtOffTimePermission.outRegion += c.count;
                            break;
                        default:
                            break;
                    }
                }
                else if (c.expectedType === dal_constants_1.DalConstants.EmployeeTimeRangeType.OutOfWorkingHours) {
                    switch (c.actualType) {
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.Absent:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtOutOfWorkingHours.count += c.count;
                            notWorkingDistribution.AtOutOfWorkingHours.outRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                            result.notWorking += c.count;
                            notWorkingDistribution.AtOutOfWorkingHours.count += c.count;
                            notWorkingDistribution.AtOutOfWorkingHours.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                            result.working += c.count;
                            workingDistribution.AtOutOfWorkingHours.count += c.count;
                            workingDistribution.AtOutOfWorkingHours.inRegion += c.count;
                            break;
                        case dal_constants_1.DalConstants.AttendanceTimeRangeType.DeclaredAttendance:
                            result.working += c.count;
                            workingDistribution.AtOutOfWorkingHours.count += c.count;
                            workingDistribution.AtOutOfWorkingHours.outRegion += c.count;
                            break;
                        default:
                            break;
                    }
                }
            }
            result.total = result.working + result.notWorking;
            result.workingDistribution = Object.values(workingDistribution);
            result.notWorkingDistribution = Object.values(notWorkingDistribution);
            result.filter = {
                dateTime: filter.dateTime,
                applyOrganizationUnitFilterHierarchically: filter.idBasedUserFilter.applyOrganizationUnitFilterHierarchically,
            };
            if (filter.idBasedUserFilter.organizationUnitIds && filter.idBasedUserFilter.organizationUnitIds.length > 0) {
                result.filter.organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(organizationId, filter.idBasedUserFilter.organizationUnitIds, trx);
            }
            if (filter.idBasedUserFilter.userGroupIds && filter.idBasedUserFilter.userGroupIds.length > 0) {
                result.filter.userGroups = await dal_manager_1.dbManager.accessUserGroup.getIdNamePairsOfUserGroups(organizationId, filter.idBasedUserFilter.userGroupIds, trx);
            }
            if (filter.workPlanIds && filter.workPlanIds.length > 0) {
                result.filter.workPlans = await dal_manager_1.dbManager.accessPacs2.getIdNamePairsOfWorkPlans(organizationId, filter.workPlanIds, trx);
            }
            if (filter.idBasedUserFilter.userIds && filter.idBasedUserFilter.userIds.length > 0) {
                result.filter.users = [];
                for (const userId of filter.idBasedUserFilter.userIds) {
                    result.filter.users.push((await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId })).caption);
                }
            }
            return result;
        });
    }
    async getIdNamePairsOfWorkPlans(organizationId, workPlanIds, trx) {
        let transactionScope = async (trx) => {
            let organizations = [];
            await trx.withSchema(organizationId).table("workPlans").whereIn("id", workPlanIds).where("organizationId", organizationId).whereNull("deletedAt").select("id", "name");
            return Promise.resolve(organizations);
        };
        if (trx) {
            return transactionScope(trx);
        }
        else {
            return this.dbClient.transaction(transactionScope);
        }
    }
    async getRawStatisticsAtDateTime(requesterUserId, organizationId, filter, trx) {
        if (moment(filter.dateTime).isAfter(moment())) {
            filter.dateTime = moment().toDate();
        }
        let transactionScope = async (trxIn) => {
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: filter.idBasedUserFilter,
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            let date = moment(filter.dateTime).startOf("day");
            let now = moment().startOf("day");
            let isToday = date.isSame(now);
            let qdParams = userFilter.bindingKeys;
            qdParams.push(organizationId, filter.dateTime, filter.dateTime);
            let tableName = isToday ? dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay : dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays;
            if (!isToday) {
                qdParams.push(date.format("YYYY-MM-DD"));
            }
            let workPlanIdFilter = filter.workPlanIds && filter.workPlanIds.length > 0 ? ` AND "data" @> '{"w": ${JSON.stringify(filter.workPlanIds)}}' ` : "";
            let q = `
SELECT COUNT(*):: integer AS "count", (ee ->> 'at'):: smallint AS "actualType", (ee ->> 'et'):: smallint AS "expectedType"
FROM "${organizationId}"."${tableName}" AS ed
INNER JOIN(
    ${userFilter.query}
) AS T3 ON T3."userId" = ed."userId" ${workPlanIdFilter} AND ed."organizationId" = ?
                    , jsonb_array_elements(ed.data -> 'c') AS ee
WHERE(ee ->> 's'):: timestamp with time zone <= ? AND(ee ->> 'e') :: timestamp with time zone > ?
    AND ed."data" IS NOT NULL
${!isToday ? ` AND ed."date" = ? ` : ""}
GROUP BY "actualType", "expectedType"
    `;
            let rawResult = await trxIn.raw(trxIn.raw(q, qdParams).toQuery());
            return rawResult.rows;
        };
        if (trx) {
            return transactionScope(trx);
        }
        else {
            return this.dbClient.transaction(transactionScope);
        }
    }
    async listOfCumulativeEmployeesAtDateTime(requesterUserId, organizationId, filter) {
        if (moment(filter.dateTime).isAfter(moment())) {
            filter.dateTime = moment().toDate();
        }
        return this.dbClient.transaction(async (trx) => {
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: filter.idBasedUserFilter,
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            let date = moment(filter.dateTime).startOf("day");
            let now = moment().startOf("day");
            let isToday = date.isSame(now);
            let qdParams = [date.format("YYYY-MM-DD"), ...userFilter.bindingKeys, organizationId];
            let tableName = isToday ? dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay : dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays;
            if (!isToday) {
                qdParams.push(date.format("YYYY-MM-DD"));
            }
            let workPlanIdFilter = filter.workPlanIds && filter.workPlanIds.length > 0 ? ` AND "data" @> '{"w": ${JSON.stringify(filter.workPlanIds)}}' ` : "";
            let qSelect = `
SELECT	ed."userId", 
		T3."name", 
		T3."surname",
    	T3."uniqueId", 
		ed."data" as "dateData", 
		rwq."reason", 
		rwq."note",
		(ed."data" -> 'ar') as "accessRanges"
`;
            let q = `
FROM "${organizationId}"."${tableName}" AS ed
LEFT OUTER JOIN "${organizationId}"."recalculateWorkQueue" as rwq ON rwq."userId" = ed."userId" AND rwq."organizationId" = ed."organizationId"
AND "startDate" <= ?
    INNER JOIN
        (${userFilter.query}) AS T3
ON T3."userId" = ed."userId" AND ed."organizationId" = ? ${!isToday ? `AND ed."date" = ?` : ""} ${workPlanIdFilter}
                    , jsonb_array_elements(ed.data -> 'c') AS ee
WHERE(ee ->> 's'):: timestamp with time zone <= ? AND(ee ->> 'e') :: timestamp with time zone > ? AND
    `;
            let wq = "";
            switch (filter.binaryType) {
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.Work:
                    if (filter.existing) {
                        wq = `((ed."data" -> 's' ->> 'pr')::integer > 0)`;
                    }
                    else {
                        wq = `((ed."data" -> 's' ->> 'pr')::integer = 0)`;
                    }
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.ExtraWork:
                    wq = `(ed."data" -> 's' ->> 'e'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.MissingWork:
                    wq = `(ed."data" -> 's' ->> 'm'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.LateStart:
                    wq = `(ed."data" -> 's' ->> 'ls'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.EarlyLeave:
                    wq = `(ed."data" -> 's' ->> 'el'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.LateLeave:
                    wq = `(ed."data" -> 's' ->> 'll'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.EarlyStart:
                    wq = `(ed."data" -> 's' ->> 'es'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.OffTimePermission:
                    wq = `(ed."data" -> 's' ->> 'ao'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.DeclaredWork:
                    wq = `(ed."data" -> 's' ->> 'dw'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
                case dal_constants_1.DalConstants.EmployeeBinaryDistributionType.ShouldBeIgnoredWork:
                    wq = `(ed."data" -> 's' ->> 'iw'):: smallint ${filter.existing ? ">" : "="} 0`;
                    break;
            }
            qdParams.push(filter.dateTime, filter.dateTime);
            let qc = `SELECT COUNT(*) AS c ${q} ${wq} `;
            let total = 0;
            await trx.raw(this.dbClient.raw(qc, qdParams).toQuery()).then((c) => {
                total = parseInt(c.rows[0].c);
            });
            q = `${qSelect} ${q} ${wq} ORDER BY`;
            q += this.getSortAndFilter({
                sortOrder: filter.sortOrder,
                sortType: filter.sortType,
                skip: filter.paginationRequest.skip,
                take: filter.paginationRequest.take,
            });
            const queryExtractionResult = [];
            (await trx.raw(this.dbClient.raw(q, qdParams).toQuery())).rows.forEach((elem) => {
                let queryResult = elem;
                let firstEntryTime;
                queryResult.accessRanges.find((elem) => {
                    let findingElem = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                    if (findingElem) {
                        firstEntryTime = {
                            firstStart: findingElem.s,
                            firstStartId: findingElem.si,
                        };
                    }
                });
                let lastLeaveTime;
                queryResult.accessRanges.reverse();
                queryResult.accessRanges.find((elem) => {
                    elem.a.reverse();
                    let findingElem = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                    if (findingElem) {
                        lastLeaveTime = {
                            lastLeave: findingElem.e,
                            lastLeaveId: findingElem.ei,
                        };
                    }
                });
                delete queryResult.accessRanges;
                queryExtractionResult.push(Object.assign({
                    firstStart: firstEntryTime?.firstStart,
                    firstStartId: firstEntryTime?.firstStartId,
                    lastLeave: lastLeaveTime?.lastLeave,
                    lastLeaveId: lastLeaveTime?.lastLeaveId,
                }, queryResult));
            });
            let result = {
                paginationResponse: {
                    take: filter.paginationRequest.take,
                    skip: filter.paginationRequest.skip,
                    total: total,
                },
                filter: null,
                items: await this.setUnapprovedPPermissionFlags(organizationId, await this.mapEmployeeDayShortToWorkSummaryItem(organizationId, queryExtractionResult), trx),
            };
            for (const item of result.items) {
                item.userCaption = (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: item.userId })).caption;
            }
            result.filter = {
                dateTime: filter.dateTime,
                applyOrganizationUnitFilterHierarchically: filter.idBasedUserFilter.applyOrganizationUnitFilterHierarchically,
            };
            if (filter.idBasedUserFilter.organizationUnitIds && filter.idBasedUserFilter.organizationUnitIds.length > 0) {
                result.filter.organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(organizationId, filter.idBasedUserFilter.organizationUnitIds, trx);
            }
            if (filter.idBasedUserFilter.userGroupIds && filter.idBasedUserFilter.userGroupIds.length > 0) {
                result.filter.userGroups = await dal_manager_1.dbManager.accessUserGroup.getIdNamePairsOfUserGroups(organizationId, filter.idBasedUserFilter.userGroupIds, trx);
            }
            if (filter.workPlanIds && filter.workPlanIds.length > 0) {
                result.filter.workPlans = await dal_manager_1.dbManager.accessPacs2.getIdNamePairsOfWorkPlans(organizationId, filter.workPlanIds, trx);
            }
            if (filter.idBasedUserFilter.userIds && filter.idBasedUserFilter.userIds.length > 0) {
                result.filter.users = [];
                for (const userId of filter.idBasedUserFilter.userIds) {
                    result.filter.users.push((await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId })).caption);
                }
            }
            return result;
        });
    }
    async listOfDistributedEmployeesAtDateTime(requesterUserId, organizationId, filter) {
        if (moment(filter.dateTime).isAfter(moment())) {
            filter.dateTime = moment().toDate();
        }
        return this.dbClient.transaction(async (trx) => {
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: filter.idBasedUserFilter,
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            let date = moment(filter.dateTime).startOf("day");
            let now = moment().startOf("day");
            let isToday = date.isSame(now);
            let qdParams = [date.format("YYYY-MM-DD"), ...userFilter.bindingKeys, organizationId];
            let tableName = isToday ? dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay : dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays;
            if (!isToday) {
                qdParams.push(date.format("YYYY-MM-DD"));
            }
            let employeeStatusConditions = this.collectAttendanceEmployeeTimeRangeTypePair(filter.employeeStatusAtSegment);
            let workPlanIdFilter = filter.workPlanIds && filter.workPlanIds.length > 0 ? ` AND "data" @> '{"w": ${JSON.stringify(filter.workPlanIds)}}' ` : "";
            let qSelect = `
SELECT ed."userId", T3."name", T3."surname",
    T3."uniqueId", ed."data" as "dateData", rwq."reason", rwq."note",
            (ed."data" -> 'ar')as "accessRanges"
                        `;
            let q = `
FROM "${organizationId}"."${tableName}" AS ed
LEFT OUTER JOIN "${organizationId}"."recalculateWorkQueue" as rwq ON rwq."userId" = ed."userId" AND rwq."organizationId" = ed."organizationId"
AND "startDate" <= ?
    INNER JOIN
        (${userFilter.query}) AS T3
ON T3."userId" = ed."userId" AND ed."organizationId" = ? ${!isToday ? `AND ed."date" = ?` : ""} ${workPlanIdFilter}
                , jsonb_array_elements(ed.data -> 'c') AS ee
    `;
            let wq = `WHERE(ee ->> 's'):: timestamp with time zone <= ? AND(ee ->> 'e') :: timestamp with time zone > ?
    ${employeeStatusConditions.length > 0
                ? " AND (" +
                    employeeStatusConditions
                        .map((mm) => {
                        return "ee @> '" + JSON.stringify(mm) + "'::jsonb";
                    })
                        .join(" OR ") +
                    ")"
                : ""}
`;
            qdParams.push(filter.dateTime, filter.dateTime);
            let qc = `SELECT COUNT(*) AS c ${q} ${wq} `;
            let total = 0;
            await trx.raw(this.dbClient.raw(qc, qdParams).toQuery()).then((c) => {
                total = parseInt(c.rows[0].c);
            });
            q = `${qSelect} ${q} ${wq} ORDER BY `;
            q += this.getSortAndFilter({
                sortOrder: filter.sortOrder,
                sortType: filter.sortType,
                skip: filter.paginationRequest.skip,
                take: filter.paginationRequest.take,
            });
            const queryExtractionResult = [];
            (await trx.raw(this.dbClient.raw(q, qdParams).toQuery())).rows.forEach((elem) => {
                let queryResult = elem;
                let firstEntryTime;
                queryResult.accessRanges.find((elem) => {
                    let findingElem = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                    if (findingElem) {
                        firstEntryTime = {
                            firstStart: findingElem.s,
                            firstStartId: findingElem.si,
                        };
                    }
                });
                let lastLeaveTime;
                queryResult.accessRanges.reverse();
                queryResult.accessRanges.find((elem) => {
                    elem.a.reverse();
                    let findingElem = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                    if (findingElem) {
                        lastLeaveTime = {
                            lastLeave: findingElem.e,
                            lastLeaveId: findingElem.ei,
                        };
                    }
                });
                delete queryResult.accessRanges;
                queryExtractionResult.push(Object.assign({
                    firstStart: firstEntryTime?.firstStart,
                    firstStartId: firstEntryTime?.firstStartId,
                    lastLeave: lastLeaveTime?.lastLeave,
                    lastLeaveId: lastLeaveTime?.lastLeaveId,
                }, queryResult));
            });
            let result = {
                paginationResponse: {
                    take: filter.paginationRequest.take,
                    skip: filter.paginationRequest.skip,
                    total: total,
                },
                filter: null,
                items: await this.setUnapprovedPPermissionFlags(organizationId, await this.mapEmployeeDayShortToWorkSummaryItem(organizationId, queryExtractionResult), trx),
            };
            for (const item of result.items) {
                item.userCaption = (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: item.userId })).caption;
            }
            result.filter = {
                dateTime: filter.dateTime,
                applyOrganizationUnitFilterHierarchically: filter.idBasedUserFilter.applyOrganizationUnitFilterHierarchically,
            };
            if (filter.idBasedUserFilter.organizationUnitIds && filter.idBasedUserFilter.organizationUnitIds.length > 0) {
                result.filter.organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(organizationId, filter.idBasedUserFilter.organizationUnitIds, trx);
            }
            if (filter.idBasedUserFilter.userGroupIds && filter.idBasedUserFilter.userGroupIds.length > 0) {
                result.filter.userGroups = await dal_manager_1.dbManager.accessUserGroup.getIdNamePairsOfUserGroups(organizationId, filter.idBasedUserFilter.userGroupIds, trx);
            }
            if (filter.workPlanIds && filter.workPlanIds.length > 0) {
                result.filter.workPlans = await dal_manager_1.dbManager.accessPacs2.getIdNamePairsOfWorkPlans(organizationId, filter.workPlanIds, trx);
            }
            if (filter.idBasedUserFilter.userIds && filter.idBasedUserFilter.userIds.length > 0) {
                result.filter.users = [];
                for (const userId of filter.idBasedUserFilter.userIds) {
                    result.filter.users.push((await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId })).caption);
                }
            }
            return result;
        });
    }
    async cumulativeStatisticsOfOrganizationUnit(requesterUserId, organizationId, organizationUnitId, range) {
        return this.dbClient.transaction(async (trx) => {
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: {
                    organizationUnitIds: [organizationUnitId],
                },
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            let qdParams = userFilter.bindingKeys;
            qdParams.push(organizationId, range.startDateTime, range.endDateTime);
            let q = `
SELECT
ed.date,
    COUNT(*):: smallint AS "total",
        SUM(CASE WHEN((("data" -> 's' ->> 'n'):: smallint + ("data" -> 's' ->> 'e'):: smallint) = 0 AND(ed."data" -> 'ar' -> 0 -> 'a' -> 0 ->> 's') IS NULL) THEN 0 ELSE 1 END):: smallint as "work",
            SUM(CASE WHEN("data" -> 's' ->> 'e'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "extraWork",
                SUM(CASE WHEN("data" -> 's' ->> 'm'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "missingWork",
                    SUM(CASE WHEN("data" -> 's' ->> 'ls'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "lateStart",
                        SUM(CASE WHEN("data" -> 's' ->> 'el'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "earlyLeave",
                            SUM(CASE WHEN("data" -> 's' ->> 'll'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "lateLeave",
                                SUM(CASE WHEN("data" -> 's' ->> 'es'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "earlyStart",
                                    SUM(CASE WHEN("data" -> 's' ->> 'ao'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "offTimePermission",
                                        SUM(CASE WHEN("data" -> 's' ->> 'dw'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "declaredWork",
                                            SUM(CASE WHEN("data" -> 's' ->> 'iw'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "shouldBeIgnoredWork"
FROM(
    SELECT ed."userId", ed."organizationId", ed."date", ed."data" FROM
                    "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" ed
                    UNION ALL
                    SELECT eld."userId", eld."organizationId", now():: date, eld."data" FROM
                    "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" eld
) ed
INNER JOIN
    (${userFilter.query}) AS T3
ON T3."userId" = ed."userId" AND ed."organizationId" = ? AND ed."data" IS NOT NULL
WHERE ed.date >= ? AND ed.date <= ?
    GROUP BY ed.date
        `;
            let qresult = await trx.raw(q, qdParams);
            let result = {
                organizationUnitId: organizationUnitId,
                statistics: [],
            };
            for (const row of qresult.rows) {
                result.statistics.push({
                    date: new Date(row.date),
                    distribution: [
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.Work,
                            exists: row.work,
                            notExists: row.total - row.work,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.ExtraWork,
                            exists: row.extraWork,
                            notExists: row.total - row.extraWork,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.MissingWork,
                            exists: row.missingWork,
                            notExists: row.total - row.missingWork,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.LateStart,
                            exists: row.lateStart,
                            notExists: row.total - row.lateStart,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.EarlyLeave,
                            exists: row.earlyLeave,
                            notExists: row.total - row.earlyLeave,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.LateLeave,
                            exists: row.lateLeave,
                            notExists: row.total - row.lateLeave,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.EarlyStart,
                            exists: row.earlyStart,
                            notExists: row.total - row.earlyStart,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.OffTimePermission,
                            exists: row.offTimePermission,
                            notExists: row.total - row.offTimePermission,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.DeclaredWork,
                            exists: row.declaredWork,
                            notExists: row.total - row.declaredWork,
                        },
                        {
                            type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.ShouldBeIgnoredWork,
                            exists: row.shouldBeIgnoredWork,
                            notExists: row.total - row.shouldBeIgnoredWork,
                        },
                    ],
                });
            }
            return result;
        });
    }
    async cumulativeStatisticsOfEmployeesAtDate(requesterUserId, organizationId, filter) {
        if (moment(filter.date).endOf("day").isAfter(moment().endOf("day"))) {
            filter.date = moment().toDate();
        }
        return this.dbClient.transaction(async (trx) => {
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: filter.idBasedUserFilter,
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            let date = moment(filter.date);
            let isToday = date.isSame(moment(), "day");
            let qdParams = userFilter.bindingKeys;
            qdParams.push(organizationId);
            let tableName = isToday ? dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay : dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays;
            if (!isToday) {
                qdParams.push(date.format("YYYY-MM-DD"));
            }
            let workPlanIdFilter = filter.workPlanIds && filter.workPlanIds.length > 0 ? ` AND "data" @> '{"w": ${JSON.stringify(filter.workPlanIds)}}' ` : "";
            let q = `
SELECT
COUNT(*):: smallint AS "total",
    SUM(CASE WHEN((("data" -> 's' ->> 'n'):: smallint + ("data" -> 's' ->> 'e'):: smallint) = 0 AND(ed."data" -> 'ar' -> 0 -> 'a' -> 0 ->> 's') IS NULL) THEN 0 ELSE 1 END):: smallint as "work",
        SUM(CASE WHEN("data" -> 's' ->> 'e'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "extraWork",
            SUM(CASE WHEN("data" -> 's' ->> 'm'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "missingWork",
                SUM(CASE WHEN("data" -> 's' ->> 'ls'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "lateStart",
                    SUM(CASE WHEN("data" -> 's' ->> 'el'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "earlyLeave",
                        SUM(CASE WHEN("data" -> 's' ->> 'll'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "lateLeave",
                            SUM(CASE WHEN("data" -> 's' ->> 'es'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "earlyStart",
                                SUM(CASE WHEN("data" -> 's' ->> 'ao'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "offTimePermission",
                                    SUM(CASE WHEN("data" -> 's' ->> 'dw'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "declaredWork",
                                        SUM(CASE WHEN("data" -> 's' ->> 'iw'):: smallint = 0 THEN 0 ELSE 1 END):: smallint as "shouldBeIgnoredWork"
FROM "${organizationId}"."${tableName}" AS ed
INNER JOIN
    (${userFilter.query}) AS T3
ON T3."userId" = ed."userId" AND ed."organizationId" = ? AND ed."data" IS NOT NULL ${!isToday ? `AND ed."date" = ?` : ""} ${workPlanIdFilter}
`;
            let qresult = (await trx.raw(q, qdParams)).rows[0];
            let result = {
                filter: null,
                total: qresult.total,
                distribution: [
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.Work,
                        exists: qresult.work,
                        notExists: qresult.total - qresult.work,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.ExtraWork,
                        exists: qresult.extraWork,
                        notExists: qresult.total - qresult.extraWork,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.MissingWork,
                        exists: qresult.missingWork,
                        notExists: qresult.total - qresult.missingWork,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.LateStart,
                        exists: qresult.lateStart,
                        notExists: qresult.total - qresult.lateStart,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.EarlyLeave,
                        exists: qresult.earlyLeave,
                        notExists: qresult.total - qresult.earlyLeave,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.LateLeave,
                        exists: qresult.lateLeave,
                        notExists: qresult.total - qresult.lateLeave,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.EarlyStart,
                        exists: qresult.earlyStart,
                        notExists: qresult.total - qresult.earlyStart,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.OffTimePermission,
                        exists: qresult.offTimePermission,
                        notExists: qresult.total - qresult.offTimePermission,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.DeclaredWork,
                        exists: qresult.declaredWork,
                        notExists: qresult.total - qresult.declaredWork,
                    },
                    {
                        type: dal_constants_1.DalConstants.EmployeeBinaryDistributionType.ShouldBeIgnoredWork,
                        exists: qresult.shouldBeIgnoredWork,
                        notExists: qresult.total - qresult.shouldBeIgnoredWork,
                    },
                ],
            };
            result.filter = {
                dateTime: filter.date,
                applyOrganizationUnitFilterHierarchically: filter.idBasedUserFilter.applyOrganizationUnitFilterHierarchically,
            };
            if (filter.idBasedUserFilter.organizationUnitIds && filter.idBasedUserFilter.organizationUnitIds.length > 0) {
                result.filter.organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(organizationId, filter.idBasedUserFilter.organizationUnitIds, trx);
            }
            if (filter.idBasedUserFilter.userGroupIds && filter.idBasedUserFilter.userGroupIds.length > 0) {
                result.filter.userGroups = await dal_manager_1.dbManager.accessUserGroup.getIdNamePairsOfUserGroups(organizationId, filter.idBasedUserFilter.userGroupIds, trx);
            }
            if (filter.workPlanIds && filter.workPlanIds.length > 0) {
                result.filter.workPlans = await dal_manager_1.dbManager.accessPacs2.getIdNamePairsOfWorkPlans(organizationId, filter.workPlanIds, trx);
            }
            if (filter.idBasedUserFilter.userIds && filter.idBasedUserFilter.userIds.length > 0) {
                result.filter.users = [];
                for (const userId of filter.idBasedUserFilter.userIds) {
                    result.filter.users.push((await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId })).caption);
                }
            }
            return result;
        });
    }
    async mapFunctionToUpdateList(filter, operator) {
        let pgClient = await this.pgPool.connect();
        let rowLimit = 50;
        if (filter.u && filter.u.length === 1 && filter.u[0] === filter.r) {
            filter.p = undefined;
            filter.pu = undefined;
        }
        let userQuery = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: filter.o,
            requesterUserId: filter.r,
            idBasedUserFilter: {
                userIds: filter.u,
                userGroupIds: filter.g,
                organizationUnitIds: filter.n,
                userOrganizationStatus: filter.s,
                applyOrganizationUnitFilterHierarchically: filter.a,
            },
            requiredOrganizationWidePermissions: filter.p,
            requiredOrganizationUnitWidePermissions: filter.pu,
            bindingKeys: [],
            specificSelectItems: ["userId"],
        });
        try {
            const cursor = await pgClient.query(new Cursor(userQuery.query, userQuery.bindingKeys, {
                rowMode: "array",
            }));
            while (true) {
                let userIds = await new Promise((resolve, reject) => {
                    cursor.read(rowLimit, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                if (userIds.length > 0) {
                    await operator(filter.o, userIds.map((m) => m[0]));
                }
                if (userIds.length < rowLimit) {
                    break;
                }
            }
            cursor.close();
            pgClient.release();
        }
        catch (error) {
            pgClient.release(error);
            throw error;
        }
    }
    async workSummaryOfFilteredEmployeesAtDate(requesterUserId, organizationId, filter) {
        if (moment(filter.date).endOf("day").isAfter(moment().endOf("day"))) {
            filter.date = moment().toDate();
        }
        return this.dbClient.transaction(async (trx) => {
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: filter.userFilter,
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            let date = moment(filter.date).startOf("day");
            let now = moment().startOf("day");
            let isToday = date.isSame(now);
            let qdParams = userFilter.bindingKeys;
            qdParams.push(organizationId);
            let tableName = isToday ? dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay : dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays;
            if (!isToday) {
                qdParams.push(date.format("YYYY-MM-DD"));
            }
            qdParams.push(date.format("YYYY-MM-DD"));
            let workPlanIdFilter = "";
            if (filter.userFilter && filter.userFilter.workPlanIds && filter.userFilter.workPlanIds.length > 0) {
                workPlanIdFilter += `AND ( `;
                filter.userFilter.workPlanIds.forEach((workPlan) => {
                    workPlanIdFilter += ` "data" @> '{"w": [${JSON.stringify(workPlan)}]}' OR`;
                });
                workPlanIdFilter = workPlanIdFilter.substr(0, workPlanIdFilter.length - 2);
                workPlanIdFilter += ` ) `;
            }
            let qSelect = `
				SELECT	ed."userId",
						T3."name", 
						T3."surname",
						T3."uniqueId", 
						ed."data" as "dateData", 
						rwq."reason", 
						rwq."note",
						(ed."data" -> 'ar') as "accessRanges"
										`;
            let q = `
				FROM "${organizationId}"."${tableName}" AS ed
				INNER JOIN
					(${userFilter.query}) AS T3
				ON T3."userId" = ed."userId" AND ed."organizationId" = ? AND ed."data" IS NOT NULL ${!isToday ? `AND ed."date" = ?` : ""} ${workPlanIdFilter}
				LEFT OUTER JOIN "${organizationId}"."recalculateWorkQueue" as rwq ON rwq."userId" = ed."userId" AND rwq."organizationId" = ed."organizationId"
				AND "startDate" <= ?
			`;
            if (filter.userFilter.workStatusFilter && filter.userFilter.workStatusFilter.length > 0) {
                q += ` WHERE(`;
                if (filter.userFilter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.Came)) {
                    q += ` ((data -> 's' ->> 'pr')::integer > 0) OR`;
                }
                if (filter.userFilter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.NotCame)) {
                    q += `  ((data -> 's' ->> 'pr')::integer = 0)  OR`;
                }
                if (filter.userFilter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave)) {
                    q += ` (data -> 's' ->> 'el'):: integer > 0 OR`;
                }
                if (filter.userFilter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateLeave)) {
                    q += ` (data -> 's' ->> 'll'):: integer > 0 OR`;
                }
                if (filter.userFilter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart)) {
                    q += ` (data -> 's' ->> 'es'):: integer > 0 OR`;
                }
                if (filter.userFilter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateStart)) {
                    q += ` (data -> 's' ->> 'ls'):: integer > 0 OR`;
                }
                q = q.substr(0, q.length - 2) + ` )`;
            }
            let qc = `SELECT COUNT(*) AS c ${q}`;
            let total = 0;
            await trx.raw(qc, qdParams).then((c) => {
                total = parseInt(c.rows[0].c);
            });
            q = `${qSelect} ${q} ORDER BY`;
            q += this.getSortAndFilter({
                sortOrder: filter.sortOrder,
                sortType: filter.sortType,
                skip: filter.paginationRequest.skip,
                take: filter.paginationRequest.take,
            });
            const queryExtractionResult = [];
            (await trx.raw(this.dbClient.raw(q, qdParams).toQuery())).rows.forEach((elem) => {
                let queryResult = elem;
                let firstEntryTime;
                queryResult.accessRanges.find((elem) => {
                    let findingElem = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                    if (findingElem) {
                        firstEntryTime = {
                            firstStart: findingElem.s,
                            firstStartId: findingElem.si,
                        };
                    }
                });
                let lastLeaveTime;
                queryResult.accessRanges.reverse();
                queryResult.accessRanges.find((elem) => {
                    elem.a.reverse();
                    let findingElem = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                    if (findingElem) {
                        lastLeaveTime = {
                            lastLeave: findingElem.e,
                            lastLeaveId: findingElem.ei,
                        };
                    }
                });
                delete queryResult.accessRanges;
                queryExtractionResult.push(Object.assign({
                    firstStart: firstEntryTime?.firstStart,
                    firstStartId: firstEntryTime?.firstStartId,
                    lastLeave: lastLeaveTime?.lastLeave,
                    lastLeaveId: lastLeaveTime?.lastLeaveId,
                }, queryResult));
            });
            let result = {
                paginationResponse: {
                    take: filter.paginationRequest.take,
                    skip: filter.paginationRequest.skip,
                    total: total,
                },
                items: await this.setUnapprovedPPermissionFlags(organizationId, await this.mapEmployeeDayShortToWorkSummaryItem(organizationId, queryExtractionResult), trx),
            };
            for (const item of result.items) {
                item.userCaption = (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: item.userId })).caption;
            }
            return result;
        });
    }
    async workSummaryOfFilteredEmployeesInDateRange(organizationId, requesterUserId, args) {
        if (moment(args.endDate).endOf("day").isAfter(moment().endOf("day"))) {
            args.endDate = moment().toDate();
        }
        return this.dbClient.transaction(async (trx) => {
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: args.userFilter,
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            let startDate = moment(args.startDate).startOf("day");
            let endDate = moment(args.endDate).endOf("day");
            let qd = "";
            let isLastDayToday = moment(endDate).startOf("day").isSame(moment().startOf("day"));
            let extensionFieldsOfOrg;
            let qExtFieldsWhere = "";
            let hasCustomFilter = false;
            await (0, cli_queries_1.systemTransaction)(this._pgPool, async (trx) => {
                extensionFieldsOfOrg = (await trx.query(`SELECT fields
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationForms}" of
					WHERE of."organizationId" = $1`, [organizationId])).rows;
                let isMatch = extensionFieldsOfOrg[0].fields.some((row) => row.name === args.extFieldKey);
                if (isMatch) {
                    hasCustomFilter = true;
                    qExtFieldsWhere = 'WHERE uop."extensionFields"->>? = ?';
                }
                else if (!isMatch && args.extFieldKey) {
                    qExtFieldsWhere = "WHERE false";
                }
                else {
                    qExtFieldsWhere = "";
                }
            });
            if (isLastDayToday) {
                qd = `
                    (SELECT "userId", "organizationId", "data", "date" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
                    WHERE "date" >= ?
                    UNION ALL
                    SELECT "userId", "organizationId", "data", now():: date as "date" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
					WHERE "data" IS NOT NULL)
    `;
            }
            else {
                qd = `"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"`;
            }
            let qWith = `
				WITH PEOPLE AS (
				SELECT T3."userId",
       				T3."name",
       				T3."surname",
       				uop."uniqueId" AS "requesterUniqueId",
       				uop."extensionFields"
				FROM (${userFilter.query}) AS T3
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
    			ON uop."userId" = T3."userId"
				${qExtFieldsWhere}`;
            let qdParams = userFilter.bindingKeys;
            if (hasCustomFilter) {
                qdParams.push(args.extFieldKey, args.extFieldValue);
            }
            let q = `
				SELECT 
					PEOPLE."requesterUniqueId" AS "uniqueId",
       				PEOPLE."userId",
       				json_agg(ed."data") AS "aggregatedData"
				FROM PEOPLE
				LEFT JOIN
					${qd} AS ed
                ON PEOPLE."userId" = ed."userId" AND ed."organizationId" = ? AND ed."date" >= ? AND ed."date" <= ?
				GROUP BY PEOPLE."requesterUniqueId", PEOPLE."userId"
			`;
            let qc = `${qWith}) SELECT COUNT(*) AS c FROM PEOPLE`;
            let total = 0;
            await trx.raw(qc, qdParams).then((c) => {
                if (c.rows.length > 0)
                    total = parseInt(c.rows[0].c);
            });
            if (isLastDayToday) {
                qdParams.push(organizationId, startDate.format("YYYY-MM-DD"), organizationId);
            }
            qdParams.push(organizationId, startDate.format("YYYY-MM-DD"), endDate.format("YYYY-MM-DD"));
            qWith += ` ORDER BY`;
            qWith += this.getSortAndFilter({
                sortOrder: args.sortOrder,
                sortType: args.sortType,
                skip: args.skip,
                take: args.take,
            });
            let query = `${qWith}) ${q}`;
            let resItems = [];
            (await trx.raw(this.dbClient.raw(query, qdParams).toQuery())).rows.forEach((elem) => {
                let days = [];
                if (elem.aggregatedData.length > 0 && elem.aggregatedData[0]) {
                    elem.aggregatedData.forEach((data) => {
                        let day = {
                            day: luxon_1.DateTime.fromISO(data.d).toFormat("yyyy-LL-dd"),
                            normal: +data.s.eh > 0 ? false : true,
                            holiday: +data.s.eh > 0 ? true : false,
                            workDuration: data.s.n,
                            extraWorkWeekday: luxon_1.DateTime.fromISO(data.d).weekdayShort !== "Sun" ? data.s.e : 0,
                            extraWorkWeekend: luxon_1.DateTime.fromISO(data.d).weekdayShort === "Sun" ? data.s.e : 0,
                        };
                        days.push(day);
                    });
                }
                let resItem = {
                    personnelId: elem.uniqueId,
                    days: days,
                };
                resItems.push(resItem);
            });
            let result = {
                paginationResponse: {
                    take: args.take,
                    skip: args.skip,
                    total: total,
                },
                items: resItems,
            };
            return result;
        });
    }
    async scoreStatisticsOfEmployeesInDateRange(requesterUserId, organizationId, filter) {
        const now = luxon_1.DateTime.local();
        if (luxon_1.DateTime.fromJSDate(filter.dateRange.endDateTime) > now) {
            filter.dateRange.endDateTime = now.toJSDate();
        }
        return this.dbClient.transaction(async (trx) => {
            const userFilter = this.getReportUserFilter({
                requesterUserId,
                organizationId,
                idBasedUserFilter: filter.idBasedUserFilter,
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            const filterEnd = luxon_1.DateTime.fromJSDate(filter.dateRange.endDateTime).startOf("day");
            const isLastDayToday = filterEnd.equals(now.startOf("day"));
            const startDate = luxon_1.DateTime.fromJSDate(filter.dateRange.startDateTime).startOf("day").toJSDate();
            const endDate = isLastDayToday ? filterEnd.minus({ days: 1 }) : filterEnd;
            const olderRange = new moment_range_1.DateRange(startDate, endDate.toJSDate());
            const total = olderRange.duration("days") + (isLastDayToday ? 1 : 0);
            const workPlanIdFilter = (prefix) => (filter.workPlanIds?.length ? ` AND ${prefix ? prefix + "." : ""}"data" @> '{"w": ${JSON.stringify(filter.workPlanIds)}}'` : "");
            let qd;
            let qdParams = [];
            if (isLastDayToday) {
                qd = `
				(
					SELECT "userId", "organizationId", ("data"->'s'->>'s')::smallint AS "score", "date"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
					WHERE "organizationId" = ? AND "date" >= ? AND "date" <= ?${workPlanIdFilter()}
					UNION ALL
					SELECT "userId", "organizationId", ("data"->'s'->>'s')::smallint AS "score", CURRENT_DATE AS "date"
					FROM "${organizationId}"."employeeLiveDay"
					WHERE "organizationId" = ?${workPlanIdFilter()}
				)`;
                qdParams.push(organizationId, startDate, endDate.toJSDate(), organizationId);
            }
            else {
                qd = `"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"`;
            }
            qdParams = qdParams.concat(userFilter.bindingKeys);
            const dateField = isLastDayToday ? `ed."score"` : `(ed."data"->'s'->>'s')::smallint`;
            const workPlanJoinFilter = !isLastDayToday ? workPlanIdFilter("ed") : "";
            const scoreQuery = `
			WITH filtered_days AS (
				SELECT ed."date", ${dateField} AS "score"
				FROM ${qd} AS ed
				INNER JOIN (${userFilter.query}) AS T3
					ON T3."userId" = ed."userId"
				WHERE ed."organizationId" = ? 
				AND ed."date" >= ? 
				AND ed."date" <= ?
				${workPlanJoinFilter}
			)
			SELECT "date"::text, ROUND(AVG("score"), 0)::smallint AS "score"
			FROM filtered_days
			GROUP BY "date"
			ORDER BY "date"
			OFFSET ? LIMIT ?;
		`;
            qdParams.push(organizationId, startDate, filter.dateRange.endDateTime, filter.paginationRequest.skip, filter.paginationRequest.take);
            const qresult = (await trx.raw(scoreQuery, qdParams)).rows;
            const result = {
                scores: qresult,
                paginationResponse: {
                    take: filter.paginationRequest.take,
                    skip: filter.paginationRequest.skip,
                    total: total,
                },
                filter: {
                    dateRange: filter.dateRange,
                    applyOrganizationUnitFilterHierarchically: filter.idBasedUserFilter.applyOrganizationUnitFilterHierarchically,
                },
            };
            const { userGroupIds, organizationUnitIds, userIds } = filter.idBasedUserFilter;
            if (organizationUnitIds?.length) {
                result.filter.organizationUnits = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(organizationId, organizationUnitIds, trx);
            }
            if (userGroupIds?.length) {
                result.filter.userGroups = await dal_manager_1.dbManager.accessUserGroup.getIdNamePairsOfUserGroups(organizationId, userGroupIds, trx);
            }
            if (filter.workPlanIds?.length) {
                result.filter.workPlans = await dal_manager_1.dbManager.accessPacs2.getIdNamePairsOfWorkPlans(organizationId, filter.workPlanIds, trx);
            }
            if (userIds?.length) {
                const badges = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCacheBulk({
                    organizationId,
                    userIds: userIds,
                });
                result.filter.users = badges?.map((b) => b?.caption);
            }
            return result;
        });
    }
    async workSummaryOfEmployeeInDateRangeAll(options) {
        if (moment(options.dateRange.endDateTime).isAfter(moment())) {
            options.dateRange.endDateTime = moment().toDate();
        }
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(`SELECT date, data
FROM "${options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" WHERE date >= $1 AND date <= $2 AND "userId" = $3
ORDER BY date ASC; `, [options.dateRange.startDateTime, options.dateRange.endDateTime, options.userId], {
            rowMode: "array",
        }));
        let rows = [];
        while (true) {
            rows = await new Promise((resolve, reject) => {
                cursor.read(100, (err, rows) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve(rows);
                    }
                });
            });
            options.onData(rows);
            if (rows.length < 100) {
                break;
            }
        }
        await new Promise((resolve, reject) => {
            cursor.close((err) => {
                if (err) {
                    reject(err);
                }
                else {
                    resolve();
                }
            });
        });
        client.release();
    }
    async workSummaryOfEmployeeInDateRange(requesterUserId, organizationId, filter) {
        const now = luxon_1.DateTime.now();
        let endDate = luxon_1.DateTime.fromJSDate(filter.dateRange.endDateTime);
        if (endDate > now) {
            endDate = now;
        }
        return this.dbClient.transaction(async (trx) => {
            const userFilter = this.getReportUserFilter({
                requesterUserId,
                organizationId,
                idBasedUserFilter: { userIds: [filter.userId] },
                requiredOrganizationUnitWidePermissions: filter.userId !== requesterUserId ? ["j:l", "e:u", "u:b", "g:r"] : [],
                requiredOrganizationWidePermissions: filter.userId !== requesterUserId ? ["j:l", "e:u", "u:b", "g:r"] : [],
            });
            const isLastDayToday = endDate.startOf("day").equals(now.startOf("day"));
            const dateRangeStart = luxon_1.DateTime.fromJSDate(filter.dateRange.startDateTime).startOf("day");
            const dateRangeEnd = endDate.endOf("day");
            let baseQuery = "";
            const queryParams = [];
            if (isLastDayToday) {
                baseQuery = `
				(
					SELECT "userId", "organizationId", "data", "date" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
					WHERE "userId" = ? AND "organizationId" = ? AND "date" >= ?
					UNION ALL
					SELECT "userId", "organizationId", "data", now()::date as "date" FROM "${organizationId}"."employeeLiveDay"
					WHERE "userId" = ? AND "organizationId" = ? AND "data" IS NOT NULL
				)
			`;
                queryParams.push(filter.userId, organizationId, dateRangeStart.toJSDate(), filter.userId, organizationId);
            }
            else {
                baseQuery = `"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"`;
            }
            queryParams.push(...userFilter.bindingKeys);
            let query = `
				SELECT
					ed."userId",
					T3."name",
					T3."surname",
					T3."uniqueId",
					ed."data" as "dateData",
					rwq."reason",
					rwq."note",
					(ed."data" -> 'ar') AS "accessRanges",
					COUNT(*) OVER() AS total_count
				FROM ${baseQuery} AS ed
				INNER JOIN (${userFilter.query}) AS T3
					ON T3."userId" = ed."userId"
				AND ed."organizationId" = ?
				AND ed."date" >= ?
				AND ed."date" <= ?
				LEFT JOIN "${organizationId}"."recalculateWorkQueue" as rwq
					ON rwq."userId" = ed."userId"
					AND rwq."organizationId" = ed."organizationId"
					AND rwq."startDate" <= ed."date"
			`;
            queryParams.push(organizationId, dateRangeStart.toJSDate(), dateRangeEnd.toJSDate());
            const filterConditions = [];
            if (filter?.workStatusFilter?.length) {
                const statusConditions = [];
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.Came)) {
                    statusConditions.push(`((ed."data" -> 's' ->> 'pr')::integer > 0)`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.NotCame)) {
                    statusConditions.push(`((ed."data" -> 's' ->> 'pr')::integer = 0)`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave)) {
                    statusConditions.push(`((ed."data" -> 's' ->> 'el')::integer > 0)`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateLeave)) {
                    statusConditions.push(`((ed."data" -> 's' ->> 'll')::integer > 0)`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart)) {
                    statusConditions.push(`((ed."data" -> 's' ->> 'es')::integer > 0)`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateStart)) {
                    statusConditions.push(`((ed."data" -> 's' ->> 'ls')::integer > 0)`);
                }
                if (statusConditions.length > 0) {
                    filterConditions.push(`(${statusConditions.join(" OR ")})`);
                }
            }
            if (filter.onlyWorkingDays) {
                filterConditions.push(`((ed."data" -> 's' ->> 'ew')::integer > 0)`);
            }
            if (filterConditions.length) {
                query += ` WHERE ${filterConditions.join(" AND ")}`;
            }
            query += " ORDER BY ";
            query += this.getSortAndFilter({
                sortOrder: filter.sortOrder,
                sortType: filter.sortType,
                skip: filter.paginationRequest.skip,
                take: filter.paginationRequest.take,
            });
            const rows = (await trx.raw(this.dbClient.raw(query, queryParams).toQuery())).rows;
            const totalCount = rows.length > 0 ? Number(rows[0].total_count) : 0;
            const queryExtractionResult = rows.map((row) => {
                let firstStart = null;
                let firstStartId = null;
                let lastLeave = null;
                let lastLeaveId = null;
                if (row.accessRanges?.length) {
                    for (const ar of row.accessRanges) {
                        const found = ar.a.find((e) => e.s && e.si);
                        if (found) {
                            firstStart = found.s;
                            firstStartId = found.si;
                            break;
                        }
                    }
                    for (let i = row.accessRanges.length - 1; i >= 0; i--) {
                        const reversedA = [...row.accessRanges[i].a].reverse();
                        const found = reversedA.find((e) => e.e && e.ei);
                        if (found) {
                            lastLeave = found.e;
                            lastLeaveId = found.ei;
                            break;
                        }
                    }
                }
                delete row.accessRanges;
                return {
                    ...row,
                    firstStart,
                    firstStartId,
                    lastLeave,
                    lastLeaveId,
                };
            });
            const items = await this.setUnapprovedPPermissionFlags(organizationId, await this.mapEmployeeDayShortToWorkSummaryItem(organizationId, queryExtractionResult), trx);
            const summaryResult = await this.getFirstInLastOutSummaryForUsers(organizationId, requesterUserId, {
                startDate: filter.dateRange.startDateTime,
                endDate: filter.dateRange.endDateTime,
                userIds: [filter.userId],
                sortType: filter.sortType,
                sortOrder: filter.sortOrder,
                status: dal_constants_1.DalConstants.IdentityStatusType.All,
                skip: null,
                take: null,
            });
            const userCaption = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({
                organizationId,
                userId: filter.userId,
            });
            for (const item of items) {
                item.userCaption = userCaption.caption || null;
            }
            return {
                paginationResponse: {
                    take: filter.paginationRequest.take,
                    skip: filter.paginationRequest.skip,
                    total: totalCount,
                },
                items,
                summary: summaryResult.items[0]?.summary || {
                    workDaysCount: 0,
                    arrivedOnTimeDaysCount: 0,
                    earlyLeftDaysCount: 0,
                    lateArrivedDaysCount: 0,
                    lateLeftDaysCount: 0,
                    ppermissionUsageCount: 0,
                },
            };
        });
    }
    summarizeFromShort(employeeDayShort, live) {
        let s = {
            dayRange: {
                start: employeeDayShort.s.r?.s ? employeeDayShort.s.r?.s : moment(employeeDayShort.d).startOf("day").toISOString(),
                end: employeeDayShort.s.r?.e ? employeeDayShort.s.r?.e : moment(employeeDayShort.d).add(1, "days").startOf("day").toISOString(),
            },
            workplanDuration: 0,
            firstStart: null,
            lastLeave: null,
            earlyStart: 0,
            lateStart: 0,
            earlyLeave: 0,
            lateLeave: 0,
            toleratedEarlyStart: 0,
            toleratedLateStart: 0,
            toleratedEarlyLeave: 0,
            toleratedLateLeave: 0,
            usedFixedBreakDuration: 0,
            automaticallyUsedFixedBreakDuration: 0,
            expectedFixedBreakDuration: 0,
            usedClaimBreakDuration: 0,
            automaticallyUsedClaimBreakDuration: 0,
            expectedClaimBreakDuration: 0,
            automaticallyUsedExtraWorkBreakDuration: 0,
            expectedExtraWorkBreakDuration: 0,
            physicallyInRegionDuration: 0,
            normalWorkDuration: 0,
            expectedWorkDuration: 0,
            missingWorkDuration: 0,
            extraWorkDuration: 0,
            usedHolidayDuration: 0,
            expectedHolidayDuration: 0,
            accountedDeclaredWorkDuration: 0,
            accountedOffTimePermissionDuration: 0,
            expectedOffTimePermissionDuration: 0,
            accountedIgnoredWorkDuration: 0,
            hasConflicts: false,
            hasRejectedPermissions: false,
            hasWaitingPermissions: false,
            score: null,
            expectedExtraWork: 0,
            ignoredExtraWork: 0,
            usedExtraWork: 0,
        };
        if (employeeDayShort.s.ua) {
            s.unpairedAccesses = employeeDayShort.s.ua.map((item) => {
                return {
                    start: item.s,
                    end: item.e,
                };
            });
        }
        let now = moment();
        if (employeeDayShort.t && employeeDayShort.t.length > 0) {
            for (const t of employeeDayShort.t) {
                s.earlyStart += t.u.es;
                s.earlyLeave += t.u.el;
                s.lateStart += t.u.ls;
                s.lateLeave += t.u.ll;
                s.toleratedEarlyStart += t.t.es;
                s.toleratedEarlyLeave += t.t.el;
                s.toleratedLateStart += t.t.ls;
                s.toleratedLateLeave += t.t.ll;
            }
        }
        if (employeeDayShort.fb && employeeDayShort.fb.length > 0) {
            for (const fb of employeeDayShort.fb) {
                s.expectedFixedBreakDuration += new moment_range_1.DateRange(fb.d.s, fb.d.e).duration("minutes");
                s.usedFixedBreakDuration += fb.u;
                s.automaticallyUsedFixedBreakDuration += fb.a;
            }
        }
        if (employeeDayShort.cb && employeeDayShort.cb.length > 0) {
            for (const cb of employeeDayShort.cb) {
                s.expectedClaimBreakDuration += cb.c;
                s.usedClaimBreakDuration += cb.u;
                s.automaticallyUsedClaimBreakDuration += cb.a;
            }
        }
        if (employeeDayShort.ewb?.length > 0) {
            for (const cb of employeeDayShort.ewb) {
                s.automaticallyUsedExtraWorkBreakDuration += cb.u;
                s.expectedExtraWorkBreakDuration += cb.bd;
            }
        }
        if (employeeDayShort.ar && employeeDayShort.ar.length > 0) {
            for (const ar of employeeDayShort.ar) {
                if (ar.a.length > 0) {
                    s.physicallyInRegionDuration += ar.a.reduce((prev, curr) => {
                        let rng = new moment_range_1.DateRange(curr.s, curr.e);
                        if (rng.start.isSameOrAfter(now)) {
                            return prev;
                        }
                        else if (rng.contains(now)) {
                            rng.end = now;
                        }
                        return prev + rng.duration("seconds");
                    }, 0);
                }
            }
        }
        for (const ws of employeeDayShort.ws) {
            if (ws?.wt === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours) {
                s.workplanDuration += new moment_range_1.DateRange(ws.s, ws.e).duration("seconds");
            }
        }
        for (const wp of employeeDayShort.fb) {
            if (wp.d.ex && wp.d.ex > 0) {
                const brRange = new moment_range_1.DateRange(wp.d.s, wp.d.e);
                let allSegmentsValid = true;
                for (const ws of employeeDayShort.ws) {
                    if (!ws || !new moment_range_1.DateRange(ws.s, ws.e).intersect(brRange)) {
                        continue;
                    }
                    if (ws.wp !== wp.d.wp || ws.wt != app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours) {
                        allSegmentsValid = false;
                    }
                }
                if (allSegmentsValid) {
                    s.workplanDuration -= wp.d.ex * 60;
                }
            }
        }
        if (s.workplanDuration < 0) {
            s.workplanDuration = 0;
        }
        for (const c of employeeDayShort.c) {
            let cr = new moment_range_1.DateRange(c.s, c.e);
            let d = cr.duration("seconds");
            if (live) {
                if (cr.contains(now)) {
                    d = new moment_range_1.DateRange(c.s, now).duration("seconds");
                }
                else if (now.isBefore(c.s)) {
                    continue;
                }
            }
            if (c.et === app_enums_1.enums.EmployeeTimeRangeType.Break) {
                switch (c.at) {
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                        s.extraWorkDuration += c.ad?.t ? 0 : d;
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.Absent:
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                    case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                    default:
                        break;
                }
            }
            else if (c.et === app_enums_1.enums.EmployeeTimeRangeType.Holiday) {
                s.expectedHolidayDuration += d;
                switch (c.at) {
                    case app_enums_1.enums.AttendanceTimeRangeType.Absent:
                    case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                        s.usedHolidayDuration += d;
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                        s.extraWorkDuration += c.ad?.t ? 0 : d;
                        break;
                    default:
                        break;
                }
            }
            else if (c.et === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission) {
                s.expectedOffTimePermissionDuration += d;
                switch (c.at) {
                    case app_enums_1.enums.AttendanceTimeRangeType.Absent:
                    case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                        s.accountedOffTimePermissionDuration += d;
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                        s.extraWorkDuration += c.ad?.t ? 0 : d;
                        break;
                    default:
                        break;
                }
            }
            else if (c.et === app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours) {
                switch (c.at) {
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                        s.extraWorkDuration += c.ad?.t ? 0 : d;
                        break;
                    default:
                        break;
                }
            }
            else if (c.et === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours || c.et === app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) {
                s.expectedWorkDuration += d;
                if (c.at === app_enums_1.enums.AttendanceTimeRangeType.Absent) {
                    if (c.ad.t) {
                        s.normalWorkDuration += d;
                    }
                    else {
                        s.missingWorkDuration += d;
                    }
                }
                else if (c.at === app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance) {
                    s.missingWorkDuration += d;
                    s.accountedIgnoredWorkDuration += d;
                }
                else if (c.at === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance || c.at === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                    if (c.ad.t) {
                        s.missingWorkDuration += d;
                    }
                    else {
                        s.normalWorkDuration += d;
                    }
                }
            }
            if (c.at === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                s.accountedDeclaredWorkDuration += d;
            }
            s.hasConflicts = s.hasConflicts || c.cs.length > 0;
        }
        s.workplanDuration = Math.round(s.workplanDuration / 60);
        s.physicallyInRegionDuration = Math.round(s.physicallyInRegionDuration / 60);
        s.expectedWorkDuration = Math.round(s.expectedWorkDuration / 60);
        s.extraWorkDuration = Math.round(s.extraWorkDuration / 60);
        s.missingWorkDuration = Math.round(s.missingWorkDuration / 60);
        s.normalWorkDuration = Math.round(s.normalWorkDuration / 60);
        s.accountedDeclaredWorkDuration = Math.round(s.accountedDeclaredWorkDuration / 60);
        s.accountedIgnoredWorkDuration = Math.round(s.accountedIgnoredWorkDuration / 60);
        s.accountedOffTimePermissionDuration = Math.round(s.accountedOffTimePermissionDuration / 60);
        s.expectedHolidayDuration = Math.round(s.expectedHolidayDuration / 60);
        s.expectedOffTimePermissionDuration = Math.round(s.expectedOffTimePermissionDuration / 60);
        s.usedHolidayDuration = Math.round(s.usedHolidayDuration / 60);
        s.normalWorkDuration = s.normalWorkDuration < 0 ? 0 : s.normalWorkDuration;
        if (s.expectedWorkDuration > 0) {
            s.score = Math.ceil(((s.normalWorkDuration + s.extraWorkDuration) / s.expectedWorkDuration) * 100);
            if (s.score > 100) {
                s.score = 100;
            }
        }
        return s;
    }
    extractIndividualRecalculateReasons(from) {
        let result = [];
        for (const key of Object.keys(app_enums_1.enums.RecalculateWorkReason)) {
            let keyInt = parseInt(key);
            if (from & keyInt && keyInt !== app_enums_1.enums.RecalculateWorkReason.Standby) {
                result.push(keyInt);
            }
        }
        return result;
    }
    getWorkplansOfDay(segments, workplanIds, workPlans) {
        const segment = segments.find((segment) => segment?.wt === dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours);
        const selectedWpIndex = segment ? workplanIds.indexOf(segment.wp) : 0;
        return workPlans[workplanIds[selectedWpIndex]];
    }
    async setUnapprovedPPermissionFlags(organizationId, list, trx) {
        for (const row of list) {
            let ppermissionsOfUser = await dal_manager_1.dbManager.accessPacs.listUnapprovedPPermissionsOfUserForDay(organizationId, row.userId, row.date, trx);
            for (let index = 0; index < ppermissionsOfUser.length; index++) {
                const element = ppermissionsOfUser[index];
                if (element.status === dal_constants_1.DalConstants.PPermissionStatus.Rejected) {
                    row.hasRejectedPermissions = true;
                }
                else if (element.status === dal_constants_1.DalConstants.PPermissionStatus.Waiting) {
                    row.hasWaitingPermissions = true;
                }
            }
        }
        return list;
    }
    async mapEmployeeDayShortToWorkSummaryItem(organizationId, params) {
        let response = [];
        for (const w of params) {
            let result;
            if (!w.dateData) {
                result = {
                    date: new Date(),
                    dayRange: {
                        start: moment().startOf("day").toISOString(),
                        end: moment().add(1, "days").startOf("day").toISOString(),
                    },
                    userId: w.userId,
                    userCaption: null,
                    waitRecalculationFor: this.extractIndividualRecalculateReasons(w.reason),
                    usedPermissions: [],
                    workplanDuration: 0,
                    normalWorkDuration: 0,
                    expectedWorkDuration: 0,
                    missingWorkDuration: 0,
                    extraWorkDuration: 0,
                    earlyStart: 0,
                    lateStart: 0,
                    earlyLeave: 0,
                    lateLeave: 0,
                    toleratedEarlyStart: 0,
                    toleratedLateStart: 0,
                    toleratedEarlyLeave: 0,
                    toleratedLateLeave: 0,
                    usedFixedBreakDuration: 0,
                    automaticallyUsedFixedBreakDuration: 0,
                    expectedFixedBreakDuration: 0,
                    usedClaimBreakDuration: 0,
                    automaticallyUsedClaimBreakDuration: 0,
                    expectedClaimBreakDuration: 0,
                    expectedExtraWorkBreakDuration: 0,
                    automaticallyUsedExtraWorkBreakDuration: 0,
                    usedHolidayDuration: 0,
                    expectedHolidayDuration: 0,
                    accountedDeclaredWorkDuration: 0,
                    physicallyInRegionDuration: 0,
                    accountedOffTimePermissionDuration: 0,
                    expectedOffTimePermissionDuration: 0,
                    accountedIgnoredWorkDuration: 0,
                    firstStart: null,
                    lastLeave: null,
                    hasConflicts: false,
                    hasRejectedPermissions: false,
                    hasWaitingPermissions: false,
                    score: 0,
                    expectedExtraWork: 0,
                    ignoredExtraWork: 0,
                    usedExtraWork: 0,
                };
            }
            else if (w.dateData && moment(w.dateData.d).isSame(new Date(), "day")) {
                let t = this.summarizeFromShort(w.dateData, true);
                result = Object.assign(t, {
                    date: w.dateData.d,
                    userId: w.userId,
                    userCaption: null,
                    waitRecalculationFor: this.extractIndividualRecalculateReasons(w.reason),
                    usedPermissions: [],
                    firstStart: w.firstStartId ? w.firstStart : null,
                    lastLeave: w.lastLeaveId ? w.lastLeave : null,
                    hasConflicts: w.dateData.c.reduce((prev, curr) => {
                        return prev + (curr.cs ? curr.cs.length : 0);
                    }, 0) > 0,
                    hasRejectedPermissions: false,
                    hasWaitingPermissions: false,
                    score: w.dateData.s.s,
                });
            }
            else {
                result = {
                    date: w.dateData.d,
                    dayRange: {
                        start: w.dateData.s.r?.s ? w.dateData.s.r?.s : moment(w.dateData.d).startOf("day").toISOString(),
                        end: w.dateData.s.r?.e ? w.dateData.s.r?.e : moment(w.dateData.d).add(1, "days").startOf("day").toISOString(),
                    },
                    userId: w.userId,
                    userCaption: null,
                    waitRecalculationFor: this.extractIndividualRecalculateReasons(w.reason),
                    usedPermissions: [],
                    workplanDuration: w.dateData.s.wd,
                    normalWorkDuration: w.dateData.s.n,
                    expectedWorkDuration: w.dateData.s.ew,
                    missingWorkDuration: w.dateData.s.m,
                    extraWorkDuration: w.dateData.s.e,
                    earlyStart: w.dateData.s.es,
                    lateStart: w.dateData.s.ls,
                    earlyLeave: w.dateData.s.el,
                    lateLeave: w.dateData.s.ll,
                    toleratedEarlyStart: w.dateData.s.tes,
                    toleratedLateStart: w.dateData.s.tls,
                    toleratedEarlyLeave: w.dateData.s.tel,
                    toleratedLateLeave: w.dateData.s.tll,
                    usedFixedBreakDuration: w.dateData.s.ufb,
                    automaticallyUsedFixedBreakDuration: w.dateData.s.afb,
                    expectedFixedBreakDuration: w.dateData.s.efb,
                    usedClaimBreakDuration: w.dateData.s.ucb,
                    automaticallyUsedClaimBreakDuration: w.dateData.s.acb,
                    expectedClaimBreakDuration: w.dateData.s.ecb,
                    automaticallyUsedExtraWorkBreakDuration: w.dateData.s.aeb,
                    expectedExtraWorkBreakDuration: w.dateData.s.eeb,
                    usedHolidayDuration: w.dateData.s.uh,
                    expectedHolidayDuration: w.dateData.s.eh,
                    accountedDeclaredWorkDuration: w.dateData.s.dw,
                    physicallyInRegionDuration: w.dateData.s.pr,
                    accountedOffTimePermissionDuration: w.dateData.s.ao,
                    expectedOffTimePermissionDuration: w.dateData.s.eo,
                    accountedIgnoredWorkDuration: w.dateData.s.iw,
                    firstStart: w.firstStartId ? w.firstStart : null,
                    lastLeave: w.lastLeaveId ? w.lastLeave : null,
                    hasConflicts: w.dateData.c.reduce((prev, curr) => {
                        return prev + (curr.cs ? curr.cs.length : 0);
                    }, 0) > 0,
                    hasRejectedPermissions: false,
                    hasWaitingPermissions: false,
                    score: w.dateData.s.s,
                    expectedExtraWork: w.dateData.s.eew,
                    ignoredExtraWork: w.dateData.s.iew,
                    usedExtraWork: w.dateData.s.uew,
                    unpairedAccesses: w.dateData.s.ua
                        ? w.dateData.s.ua.map((item) => {
                            return { start: item.s, end: item.e };
                        })
                        : undefined,
                };
            }
            if (w.dateData && w.dateData.p && w.dateData.p.length > 0) {
                let ids = w.dateData.p.map((p) => p.p).filter((pt) => pt);
                let perms = await this.getBasicInfoOfPermissions(organizationId, ids);
                for (const pu of w.dateData.p) {
                    let perm = perms.find((p) => p.id === pu.p);
                    if (perm) {
                        perm.usedDuration = pu.u;
                        result.usedPermissions.push(perm);
                    }
                }
            }
            response.push(result);
        }
        return response;
    }
    async employeeDayDetailed(requesterUserId, organizationId, userId, date) {
        if (moment(date).endOf("day").isAfter(moment().endOf("day"))) {
            (0, dal_access_error_1.throwDbAccessBadRequestError)("Date can not be after today");
        }
        return this.dbClient.transaction(async (trx) => {
            let result = {
                userId: userId,
                userCaptions: (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId })).caption,
                workPlans: [],
                actualDurationSegments: [],
                expectedDurationSegments: [],
                workPlanDurationSegments: [],
                unapprovedPermissions: [],
                tolerances: [],
                fixedBreakUsages: [],
                claimBreakUsages: [],
                extraWorkBreakUsages: [],
                summary: null,
                waitRecalculationFor: [],
            };
            let userFilter = this.getReportUserFilter({
                requesterUserId: requesterUserId,
                organizationId: organizationId,
                idBasedUserFilter: { userIds: [userId] },
                requiredOrganizationUnitWidePermissions: userId !== requesterUserId ? ["j:l", "e:u", "u:b", "g:r"] : [],
                requiredOrganizationWidePermissions: userId !== requesterUserId ? ["j:l", "e:u", "u:b", "g:r"] : [],
            });
            let datem = moment(date).startOf("day");
            let startOfToday = moment().startOf("day");
            let isToday = datem.isSame(startOfToday);
            let tableName = isToday ? "employeeLiveDay" : dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays;
            let q = `
					SELECT	ed."userId", 
							ed."data" as "dateData", 
							T3."name", 
							T3."surname",
            				T3."uniqueId", 
							${isToday ? `ed."updateRequiredReason" as reason,` : `rwq.reason,`}
            				(ed."data"->'ar') as "accessRanges"
					FROM "${organizationId}"."${tableName}" AS ed
					INNER JOIN
					(${userFilter.query}) AS T3
					ON T3."userId" = ed."userId" AND ed."organizationId" = ? AND ed."userId"= ?`;
            userFilter.bindingKeys.push(organizationId, userId);
            if (!isToday) {
                q += ` AND ed."date" = ? 
                LEFT OUTER JOIN "${organizationId}"."recalculateWorkQueue" as rwq ON rwq."userId" = ed."userId" AND rwq."organizationId" = ed."organizationId"
                AND rwq."startDate" <= ?`;
                userFilter.bindingKeys.push(datem.format("YYYY-MM-DD"));
                userFilter.bindingKeys.push(datem.format("YYYY-MM-DD"));
            }
            let queryResult;
            let queryExtractionResult;
            await trx.raw(q, userFilter.bindingKeys).then((r) => {
                if (r && r.rows.length > 0) {
                    queryResult = r.rows[0];
                    let firstEntryTime;
                    queryResult.accessRanges.find((elem) => {
                        let findingElem = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                        if (findingElem) {
                            firstEntryTime = {
                                firstStart: findingElem.s,
                                firstStartId: findingElem.si,
                            };
                        }
                    });
                    let lastLeaveTime;
                    queryResult.accessRanges.reverse();
                    queryResult.accessRanges.find((elem) => {
                        elem.a.reverse();
                        let findingElem = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                        if (findingElem) {
                            lastLeaveTime = {
                                lastLeave: findingElem.e,
                                lastLeaveId: findingElem.ei,
                            };
                        }
                    });
                    delete queryResult.accessRanges;
                    queryExtractionResult = Object.assign({
                        firstStart: firstEntryTime?.firstStart,
                        firstStartId: firstEntryTime?.firstStartId,
                        lastLeave: lastLeaveTime?.lastLeave,
                        lastLeaveId: lastLeaveTime?.lastLeaveId,
                    }, queryResult);
                }
            });
            if (queryExtractionResult === undefined || queryExtractionResult.dateData === null) {
                return null;
            }
            let newCalculatedSegments = [];
            let now = moment();
            for (const segment of queryExtractionResult.dateData.c) {
                let segmentRange = new moment_range_1.DateRange(segment.s, segment.e);
                if (moment(segment.e).isBefore(now)) {
                    newCalculatedSegments.push(segment);
                }
                else if (segmentRange.contains(now)) {
                    let newSegment = Object.assign({}, segment);
                    newSegment.e = now.toDate();
                    newCalculatedSegments.push(newSegment);
                }
                else {
                    break;
                }
            }
            queryExtractionResult.dateData.c = newCalculatedSegments;
            let ppermissionTypes = await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(organizationId, { take: 1000, skip: 0 }, trx);
            result.unapprovedPermissions = await dal_manager_1.dbManager.accessPacs.listUnapprovedPPermissionsOfUserForDay(organizationId, userId, datem.toDate(), trx);
            result.waitRecalculationFor = this.extractIndividualRecalculateReasons(queryExtractionResult.reason);
            result.workPlans =
                queryExtractionResult.dateData.w && queryExtractionResult.dateData.w.length > 0 ? await this.getBasicInfoOfWorkPlans(organizationId, queryExtractionResult.dateData.w) : [];
            let bh = [];
            if (queryExtractionResult.dateData.h && queryExtractionResult.dateData.h.length > 0) {
                bh = await this.getBasicInfoOfHolidys(organizationId, queryExtractionResult.dateData.h.map((h) => h.h), trx);
                result.holidays = [];
                for (const h of bh) {
                    h.usedDuration = queryExtractionResult.dateData.h.find((hh) => hh.h === h.id).u;
                    result.holidays.push(h);
                }
            }
            if (queryExtractionResult.dateData.p && queryExtractionResult.dateData.p.length > 0) {
                let bp = await this.getBasicInfoOfPermissions(organizationId, queryExtractionResult.dateData.p.map((p) => p.p), trx);
                result.permissions = [];
                for (const p of bp) {
                    result.permissions.push({
                        id: p.id,
                        type: p.type,
                        name: p.name,
                        range: p.range,
                        typeId: p.typeId,
                        usedDuration: queryExtractionResult.dateData.p.find((pp) => pp.p === p.id).u,
                    });
                }
            }
            if (queryExtractionResult.dateData.fb && queryExtractionResult.dateData.fb.length > 0) {
                result.fixedBreakUsages = [];
                for (const fb of queryExtractionResult.dateData.fb) {
                    let def = {
                        endTime: moment(fb.d.e).format("HHmm"),
                        startTime: moment(fb.d.s).format("HHmm"),
                        type: fb.d.t,
                        allowedMaxBreakDuration: fb.d.al,
                        executeAutomaticallyBelowDuration: fb.d.ex,
                    };
                    if (!fb.rs || fb.rs.length === 0) {
                        result.fixedBreakUsages.push({
                            definition: def,
                            range: null,
                            intersectionRatio: fb.d.ir,
                            maxAvailableDuration: fb.d.ir * def.allowedMaxBreakDuration,
                            usedDuration: fb.u,
                        });
                    }
                    else {
                        for (const fbb of fb.rs) {
                            result.fixedBreakUsages.push({
                                definition: def,
                                range: new moment_range_1.DateRange(moment(fbb.s), moment(fbb.e)),
                                intersectionRatio: fb.d.ir,
                                maxAvailableDuration: fb.d.ir * def.allowedMaxBreakDuration,
                                usedDuration: fb.u,
                            });
                        }
                    }
                }
            }
            if (queryExtractionResult.dateData.cb && queryExtractionResult.dateData.cb.length > 0) {
                result.claimBreakUsages = [];
                for (const cb of queryExtractionResult.dateData.cb) {
                    let def = {
                        type: cb.d.t,
                        workDuration: cb.d.w,
                        breakDuration: cb.d.b,
                        executeAutomaticallyBelowDuration: cb.d.ex,
                    };
                    if (!cb.rs || cb.rs.length === 0) {
                        result.claimBreakUsages.push({
                            definition: def,
                            range: null,
                            usedDuration: cb.u,
                        });
                    }
                    else {
                        for (const cbb of cb.rs) {
                            result.claimBreakUsages.push({
                                definition: def,
                                range: new moment_range_1.DateRange(moment(cbb.s), moment(cbb.e)),
                                usedDuration: cb.u,
                            });
                        }
                    }
                }
            }
            if (queryExtractionResult.dateData.ewb && queryExtractionResult.dateData.ewb.length > 0) {
                result.extraWorkBreakUsages = [];
                for (const eb of queryExtractionResult.dateData.ewb) {
                    const def = {
                        periodEndTime: moment(eb.ps).format("HHmm"),
                        periodStartTime: moment(eb.pe).format("HHmm"),
                        type: eb.d.t,
                        breakDuration: eb.bd,
                        minWorkDuration: eb.mwd,
                    };
                    if (!eb.rs || eb.rs.length === 0) {
                        result.extraWorkBreakUsages.push({
                            definition: def,
                            range: null,
                            usedDuration: eb.u,
                            expectedBreakDuration: eb.bd,
                        });
                    }
                    else {
                        for (const ebb of eb.rs) {
                            result.extraWorkBreakUsages.push({
                                definition: def,
                                range: new moment_range_1.DateRange(moment(ebb.s), moment(ebb.e)),
                                usedDuration: eb.u,
                                expectedBreakDuration: eb.bd,
                            });
                        }
                    }
                }
            }
            if (queryExtractionResult.dateData.t && queryExtractionResult.dateData.t.length > 0) {
                result.tolerances = [];
                for (const t of queryExtractionResult.dateData.t) {
                    result.tolerances.push({
                        earlyStart: t.u.es,
                        earlyLeave: t.u.el,
                        lateStart: t.u.ls,
                        lateLeave: t.u.ll,
                        toleratedEarlyLeave: t.t.el,
                        toleratedEarlyStart: t.t.es,
                        toleratedLateLeave: t.t.ll,
                        toleratedLateStart: t.t.ls,
                        range: new moment_range_1.DateRange(moment(t.s), moment(t.e)),
                        workPlanId: t.wp,
                    });
                }
            }
            let accessLogs = [];
            if (queryExtractionResult.dateData.c && queryExtractionResult.dateData.c.length > 0) {
                result.actualDurationSegments = queryExtractionResult.dateData.c.map((a) => {
                    if (a.ad && a.ad.a) {
                        if (a.ad.a.si) {
                            accessLogs.push({
                                id: a.ad.a.si,
                                regionId: a.ad.a.r,
                            });
                        }
                        if (a.ad.a.ei) {
                            accessLogs.push({
                                id: a.ad.a.ei,
                                regionId: a.ad.a.r,
                            });
                        }
                    }
                    return {
                        range: new moment_range_1.DateRange(moment(a.s), moment(a.e)),
                        workPlanTimeRangeType: a.wt,
                        workPlanId: a.wp,
                        workPlanData: a.wd
                            ? {
                                holidayIds: a.wd.h,
                            }
                            : null,
                        employeeTimeRangeType: a.et,
                        employeeData: a.ed
                            ? {
                                offTimePermissionId: a.ed.p,
                                break: a.ed.b
                                    ? {
                                        method: a.ed.b.m,
                                        type: a.ed.b.t,
                                    }
                                    : null,
                            }
                            : null,
                        attendanceTimeRangeType: a.at,
                        attendanceData: a.ad
                            ? {
                                ignoredPhysicalAttendancePermissionId: a.ad.i ? a.ad.i.p : undefined,
                                accountedPhysicalAttendance: a.ad.a
                                    ? {
                                        regionId: a.ad.a.r,
                                        startAccessLogId: a.ad.a.si,
                                        endAccessLogId: a.ad.a.ei,
                                    }
                                    : null,
                                declaredAttendancePermissionId: a.ad.d ? a.ad.d : undefined,
                                tolerance: a.ad.t,
                                extraWorkPermissionId: a.ad.xp ? a.ad.xp : undefined,
                                extraWorkPermissionTypeId: a.ad.xt ? a.ad.xt : undefined,
                                unpairedAccess: a.ad.ua ? a.ad.ua : undefined,
                            }
                            : null,
                        status: {
                            inRegion: a.es.r,
                            working: a.es.w,
                            status: a.es.s,
                        },
                        conflictItems: a.cs
                            ? a.cs.map((ci) => {
                                let itemname = undefined;
                                if (ci.t === app_enums_1.enums.SegmentConflictItemType.Holiday) {
                                    itemname = bh.find((bhi) => bhi.id === ci.i)?.name;
                                }
                                else if (ci.t === app_enums_1.enums.SegmentConflictItemType.Permission) {
                                    itemname = ppermissionTypes.items.find((pti) => pti.id === ci.ti)?.name;
                                }
                                return {
                                    id: ci.i,
                                    type: ci.t,
                                    typeId: ci.ti,
                                    name: itemname ? itemname : "",
                                };
                            })
                            : [],
                    };
                });
            }
            if (queryExtractionResult.dateData.e && queryExtractionResult.dateData.e.length > 0) {
                result.expectedDurationSegments = queryExtractionResult.dateData.e.map((a) => {
                    return {
                        range: new moment_range_1.DateRange(moment(a.s), moment(a.e)),
                        workPlanTimeRangeType: a.wt,
                        workPlanId: a.wp,
                        workPlanData: a.wd
                            ? {
                                holidayIds: a.wd.h,
                            }
                            : null,
                        employeeTimeRangeType: a.et,
                        employeeData: a.ed
                            ? {
                                offTimePermissionId: a.ed.p,
                                break: a.ed.b
                                    ? {
                                        method: a.ed.b.m,
                                        type: a.ed.b.t,
                                    }
                                    : null,
                            }
                            : null,
                        conflictItems: a.cs
                            ? a.cs.map((ci) => {
                                let itemname = undefined;
                                if (ci.t === app_enums_1.enums.SegmentConflictItemType.Holiday) {
                                    itemname = bh.find((bhi) => bhi.id === ci.i)?.name;
                                }
                                else if (ci.t === app_enums_1.enums.SegmentConflictItemType.Permission) {
                                    itemname = ppermissionTypes.items.find((pti) => pti.id === ci.ti)?.name;
                                }
                                return {
                                    id: ci.i,
                                    type: ci.t,
                                    typeId: ci.ti,
                                    name: itemname ? itemname : "",
                                };
                            })
                            : [],
                    };
                });
            }
            if (queryExtractionResult.dateData.ws && queryExtractionResult.dateData.ws.length > 0) {
                result.workPlanDurationSegments = queryExtractionResult.dateData.ws.map((a) => {
                    return {
                        range: new moment_range_1.DateRange(moment(a.s), moment(a.e)),
                        workPlanTimeRangeType: a.wt,
                        workPlanId: a.wp,
                        workPlanData: a.wd
                            ? {
                                holidayIds: a.wd.h,
                            }
                            : null,
                    };
                });
            }
            result.summary = (await this.setUnapprovedPPermissionFlags(organizationId, await this.mapEmployeeDayShortToWorkSummaryItem(organizationId, [queryExtractionResult]), trx))[0];
            if (accessLogs.length > 0) {
                let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, requesterUserId);
                result.accessLogs = await dal_manager_1.dbManager.accessLog.getBasicAccessLogInfo(organizationId, accessLogs, userAccessRights);
            }
            return result;
        });
    }
    async employeeCurrentState(requesterUserId, organizationId, userId, atTimestamp) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let timestamp = atTimestamp ? moment(atTimestamp) : moment();
            let userSummaryInfo = await dal_manager_1.dbManager.accessIdentity.getUserSummaryInfo({ organizationId, userId, trx });
            let result = {
                date: timestamp.clone().startOf("day").toDate(),
                userId: userId,
                name: userSummaryInfo.name,
                surname: userSummaryInfo.surname,
                userCaptions: (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId })).caption,
                organizationUnitNames: [],
                currentDetails: {
                    timestamp: timestamp.toISOString(),
                    state: {
                        working: false,
                        inRegion: false,
                        status: dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours,
                    },
                    regionsUserIn: [],
                    workplan: {
                        id: "",
                        name: "",
                        colorCode: "000000",
                        type: dal_constants_1.DalConstants.WorkPlanType.REGULAR,
                        allowMobileCheckins: false,
                        mobileCheckinRequiresLocation: false,
                        allowUnreliableCheckins: false,
                    },
                },
                todayDetails: {
                    workplans: [],
                    summary: undefined,
                },
            };
            result.organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.listUserOrganizationUnitNames(organizationId, userId);
            let bindIndex = 0;
            let bindingKeys = [];
            let query = `
            SELECT 	eld."userId",
					eld."workPlanIds",
					eld."data" as "dateData",
            	   (eld."data"->'ar') as "accessRanges"
            FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" eld`;
            if (userId !== requesterUserId) {
                let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                    organizationId: organizationId,
                    requesterUserId: requesterUserId,
                    idBasedUserFilter: {
                        userIds: [userId],
                    },
                    requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                    requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                    bindingKeys: bindingKeys,
                    specificSelectItems: ["userId", "organizationId"],
                });
                bindingKeys = userFilter.bindingKeys;
                bindIndex = userFilter.bindingKeys.length;
                query += `
                INNER JOIN (${userFilter.query}) ufq ON ufq."userId" = eld."userId" AND ufq."organizationId" = eld."organizationId"`;
            }
            query += `
            WHERE eld."userId" = $${++bindIndex} AND eld."organizationId" = $${++bindIndex}
            `;
            bindingKeys.push(userId, organizationId);
            let queryResult;
            let queryExtractionResult;
            let workPlanIds;
            await trx.query(query, bindingKeys).then((r) => {
                if (r && r.rows.length > 0) {
                    workPlanIds = r.rows[0].workPlanIds;
                    queryResult = r.rows[0];
                    let firstEntryTime;
                    queryResult.accessRanges?.find((elem) => {
                        let findingElem = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                        if (findingElem) {
                            firstEntryTime = {
                                firstStart: findingElem.s,
                                firstStartId: findingElem.si,
                            };
                        }
                    });
                    let lastLeaveTime;
                    queryResult.accessRanges?.reverse();
                    queryResult.accessRanges?.find((elem) => {
                        elem.a.reverse();
                        let findingElem = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                        if (findingElem) {
                            lastLeaveTime = {
                                lastLeave: findingElem.e,
                                lastLeaveId: findingElem.ei,
                            };
                        }
                    });
                    delete queryResult.accessRanges;
                    queryExtractionResult = Object.assign({
                        firstStart: firstEntryTime?.firstStart,
                        firstStartId: firstEntryTime?.firstStartId,
                        lastLeave: lastLeaveTime?.lastLeave,
                        lastLeaveId: lastLeaveTime?.lastLeaveId,
                    }, queryResult);
                }
            });
            if (queryExtractionResult) {
                result.todayDetails.summary = (await this.setUnapprovedPPermissionFlags(organizationId, await this.mapEmployeeDayShortToWorkSummaryItem(organizationId, [queryExtractionResult])))[0];
                let workplans = await trx.query(`
                SELECT id, name, type, "allowMobileCheckins", "mobileCheckinRequiresLocation", "allowUnreliableCheckins", "typedGeoLocations", "colorCode"
                FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
                WHERE id = ANY($1::uuid[])
                `, [workPlanIds]);
                for (const row of workplans.rows) {
                    result.todayDetails.workplans.push({
                        id: row.id,
                        name: row.name,
                        type: row.type,
                        colorCode: row.colorCode,
                        allowMobileCheckins: row.allowMobileCheckins,
                        mobileCheckinRequiresLocation: row.mobileCheckinRequiresLocation,
                        allowUnreliableCheckins: row.allowUnreliableCheckins,
                        geoLocations: row.typedGeoLocations
                            ?.filter((g) => g.t === dal_constants_1.DalConstants.GeoLocationType.Circle)
                            ?.map((g) => {
                            return {
                                longitude: g.lo,
                                latitude: g.la,
                                radius: g.r,
                            };
                        }),
                        typedGeoLocations: row.typedGeoLocations?.map((g) => {
                            if (g.t === dal_constants_1.DalConstants.GeoLocationType.Circle) {
                                return {
                                    type: g.t,
                                    longitude: g.lo,
                                    latitude: g.la,
                                    radius: g.r,
                                };
                            }
                            else {
                                return {
                                    type: g.t,
                                    points: g.p.map((point) => {
                                        return {
                                            longitude: point.lo,
                                            latitude: point.la,
                                        };
                                    }),
                                };
                            }
                        }),
                    });
                }
                result.currentDetails.regionsUserIn = await dal_manager_1.dbManager.accessRegion.getUserCurrentRegions(organizationId, userId);
                let currentSegment = queryExtractionResult.dateData ? queryExtractionResult.dateData.c.find((c) => new moment_range_1.DateRange(c.s, c.e).contains(timestamp)) : undefined;
                if (currentSegment) {
                    result.currentDetails.state = {
                        inRegion: currentSegment.es.r,
                        working: currentSegment.es.w,
                        status: currentSegment.es.s,
                    };
                    result.currentDetails.workplan = result.todayDetails.workplans.find((wp) => wp.id === currentSegment.wp);
                }
                else if (result.todayDetails.workplans && result.todayDetails.workplans.length === 1) {
                    result.currentDetails.workplan = result.todayDetails.workplans[0];
                }
            }
            return result;
        });
    }
    async getFirstInLastOutSummaryForUsers(organizationId, requesterUserId, args) {
        let result = {
            pagination: {
                total: 0,
                take: args.take,
                skip: args.skip,
            },
            items: [],
        };
        const now = luxon_1.DateTime.local();
        const endDate = luxon_1.DateTime.fromJSDate(args.endDate);
        const includeLiveDay = endDate.hasSame(now, "day") || endDate > now;
        let pgClient = await this.pgPool.connect();
        await pgClient.query("BEGIN");
        let bindingKeys = [organizationId, args.startDate, args.endDate];
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId,
            requesterUserId,
            idBasedUserFilter: {
                userIds: args.userIds,
                userGroupIds: args.userGroupIds,
                organizationUnitIds: args.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: args.organizationUnitHierarchically,
                userOrganizationStatus: args.status,
            },
            requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            bindingKeys,
            specificSelectItems: ["userId", "organizationId", "name", "surname", "organizationUnitIds"],
        });
        const finalBindings = userFilter.bindingKeys;
        try {
            const cteQuery = `
				WITH employee_data AS (
					SELECT "userId", "organizationId", date, data
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
					WHERE "organizationId" = $1 AND date BETWEEN $2 AND $3
					${includeLiveDay
                ? `UNION ALL
					SELECT "userId", "organizationId", now()::date, data
					FROM "${organizationId}"."employeeLiveDay"
					WHERE "organizationId" = $1`
                : ""}
				),
				filtered_users AS (${userFilter.query})
			`;
            let filters = [];
            if (args.workStatusFilter?.includes(app_enums_1.enums.WorkStatusFilter.Came))
                filters.push(`(data->'s'->>'pr')::int > 0`);
            if (args.workStatusFilter?.includes(app_enums_1.enums.WorkStatusFilter.NotCame))
                filters.push(`(data->'s'->>'pr')::int = 0`);
            if (args.workStatusFilter?.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave))
                filters.push(`(data->'s'->>'el')::int > 0`);
            if (args.workStatusFilter?.includes(app_enums_1.enums.WorkStatusFilter.LateLeave))
                filters.push(`(data->'s'->>'ll')::int > 0`);
            if (args.workStatusFilter?.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart))
                filters.push(`(data->'s'->>'es')::int > 0`);
            if (args.workStatusFilter?.includes(app_enums_1.enums.WorkStatusFilter.LateStart))
                filters.push(`(data->'s'->>'ls')::int > 0`);
            const workPlanFilter = args.workPlanIds?.length ? `AND data -> 'w' @> ANY(ARRAY['["${args.workPlanIds.join(`"]','["`)}"]']::jsonb[])` : "";
            const mainQuery = `
				${cteQuery}
				SELECT
					uf."organizationUnitIds",
					ed."userId",
					SUM(CASE WHEN (data->'s'->>'ew')::int > 0 THEN 1 ELSE 0 END)::int AS "isWorkDay",
					SUM(CASE WHEN (data->'ar'->0->'a'->0->>'s') IS NOT NULL AND (data->'s'->>'ls')::int = 0 THEN 1 ELSE 0 END)::int AS "onTimeStart",
					SUM(CASE WHEN (data->'s'->>'ls')::int > 0 THEN 1 ELSE 0 END)::int AS "lateStart",
					SUM(CASE WHEN (data->'s'->>'el')::int > 0 THEN 1 ELSE 0 END)::int AS "earlyLeft",
					SUM(CASE WHEN (data->'s'->>'ll')::int > 0 THEN 1 ELSE 0 END)::int AS "lateLeft",
					SUM((SELECT COUNT(*) FROM jsonb_array_elements(data->'p') AS p(obj) WHERE (p.obj->>'u')::int > 0))::int AS "ppermissionUsageCount",
					COUNT(*) OVER()::int AS "total_count"
				FROM employee_data ed
				INNER JOIN filtered_users uf ON uf."userId" = ed."userId" AND uf."organizationId" = ed."organizationId"
				WHERE ed."organizationId" = $1
				${filters.length ? `AND (${filters.join(" OR ")})` : ""}
				${workPlanFilter}
				GROUP BY uf."organizationUnitIds", ed."userId", uf."name", uf."surname"
			`;
            let orderClause;
            const sortOrder = ["ASC", "DESC"].includes(args.sortOrder) ? args.sortOrder : "DESC";
            switch (args.sortType) {
                case dal_constants_1.DalConstants.WorkSummarySortType.LateStart:
                    orderClause = `"lateStart" ${sortOrder}`;
                    break;
                case dal_constants_1.DalConstants.WorkSummarySortType.EarylLeave:
                    orderClause = `"earlyLeft" ${sortOrder}`;
                    break;
                case dal_constants_1.DalConstants.WorkSummarySortType.Name:
                default:
                    orderClause = `uf."name" || uf."surname" ${sortOrder}`;
                    break;
            }
            let finalQuery = `${mainQuery} ORDER BY ${orderClause}`;
            if (args.skip) {
                finalQuery += ` OFFSET $${finalBindings.length + 1}`;
                finalBindings.push(args.skip);
            }
            if (args.take) {
                finalQuery += ` LIMIT $${finalBindings.length + 1}`;
                finalBindings.push(args.take);
            }
            const rows = await pgClient.query(finalQuery, finalBindings);
            if (rows.rows.length > 0) {
                result.pagination.total = Number(rows.rows[0].total_count);
            }
            else {
                result.pagination.total = 0;
            }
            const userIds = rows.rows.map((r) => r.userId);
            const userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
            const captionMap = new Map(userCaptions.map((u) => [u.id, u.captionLines]));
            result.items = rows.rows.map((row) => ({
                organizationUnitIds: row.organizationUnitIds,
                userId: row.userId,
                userCaptions: captionMap.get(row.userId) ?? [],
                summary: {
                    workDaysCount: Number(row.isWorkDay),
                    arrivedOnTimeDaysCount: Number(row.onTimeStart),
                    earlyLeftDaysCount: Number(row.earlyLeft),
                    lateArrivedDaysCount: Number(row.lateStart),
                    lateLeftDaysCount: Number(row.lateLeft),
                    ppermissionUsageCount: Number(row.ppermissionUsageCount),
                },
            }));
            await pgClient.query("COMMIT");
            pgClient.release();
        }
        catch (error) {
            dal_logger_1.dbLogger.info("Problem while generating report: " + error);
            await pgClient.query("ROLLBACK");
            pgClient.release();
            throw error;
        }
        return result;
    }
    async employeeDaysSum(params, trx) {
        const q = `
			SELECT	ed."userId",
					extract(year from ed.date) as year,
					extract(month from ed.date) as month,
					json_build_object(
						'id', null,
						'userId', "userId",
						'note', '',
						'isChecked', false,
						'year', extract(year from ed.date),
						'month', extract(month from ed.date),
						'expectedWorkDuration', SUM((data -> 's' ->> 'ew'):: integer),
						'missedWorkDuration', SUM((data -> 's' ->> 'm'):: integer),
						'normalWorkDuration', SUM((data -> 's' ->> 'n'):: integer),
						'extraWorkDuration', SUM((data -> 's' ->> 'e'):: integer),
						'accountedOffTimePermission', SUM((data -> 's' ->> 'ao'):: integer),
						'usedHoliday', SUM((data -> 's' ->> 'uh'):: integer),
						'weekdayExtraWorkDuration', SUM(CASE WHEN extract(isodow from date) < 6 THEN(data -> 's' ->> 'e'):: integer ELSE 0 END),
						'weekendExtraWorkDuration', SUM(CASE WHEN extract(isodow from date) > 5 THEN(data -> 's' ->> 'e'):: integer ELSE 0 END),
						'earlyLeave', SUM((data -> 's' ->> 'el'):: integer),
						'earlyStart', SUM((data -> 's' ->> 'es'):: integer),
						'lateLeave', SUM((data -> 's' ->> 'll'):: integer),
						'lateStart', SUM((data -> 's' ->> 'ls'):: integer),
						'toleratedEarlyLeave', SUM((data -> 's' ->> 'tel'):: integer),
						'toleratedEarlyStart', SUM((data -> 's' ->> 'tes'):: integer),
						'toleratedLateLeave', SUM((data -> 's' ->> 'tll'):: integer),
						'toleratedLateStart', SUM((data -> 's' ->> 'tls'):: integer),
						'expectedOffTimePermissionDuration', SUM((data -> 's' ->> 'eo'):: integer),
						'score', null,
						'accountedDeclaredWorkDuration', SUM((data -> 's' ->> 'dw'):: integer),
						'accountedIgnoredWorkDuration', SUM((data -> 's' ->> 'iw'):: integer),
						'usedFixedBreakDuration', SUM((data -> 's' ->> 'ufb'):: float),
						'automaticallyUsedFixedBreakDuration', SUM((data -> 's' ->> 'afb'):: float),
						'usedClaimBreakDuration', SUM((data -> 's' ->> 'ucb'):: float),
						'automaticallyUsedClaimBreakDuration', SUM((data -> 's' ->> 'acb'):: float),
						'expectedFixedBreakDuration', SUM((data -> 's' ->> 'efb'):: float),
						'expectedClaimBreakDuration', SUM((data -> 's' ->> 'ecb'):: float),
						'usedHolidayDuration', SUM((data -> 's' ->> 'uh'):: integer),
						'expectedHolidayDuration', SUM((data -> 's' ->> 'eh'):: integer),
						'physicallyInRegionDuration', SUM((data -> 's' ->> 'pr'):: integer)
					) as data
			FROM(
    		    SELECT	ed.id, ed."userId", ed.date, ed.data
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS "ed"
				WHERE ed."userId" = ANY($1::uuid[]) AND ed.date >= $2 AND ed.date <= $3
    		    UNION ALL
				SELECT	ed.id, ed."userId", now(), ed.data
				FROM "${params.organizationId}"."employeeLiveDay" AS "ed"
				WHERE ed."userId" = ANY($1::uuid[]) AND now() >= $2 AND now() <= $3
			) ed
			 GROUP BY ed."userId", EXTRACT(YEAR FROM ed.date), EXTRACT(MONTH FROM ed.date);
		`;
        let queryArgs = [params.userIds, params.startDate.toISODate(), params.endDate.toISODate()];
        return (await trx.query(q, queryArgs)).rows;
    }
    getEmployeeDaysDateRangeFilteredQuery(organizationId, requesterUserId, args) {
        let start = args.startDate;
        let end = args.endDate;
        let ranges = [];
        let months = [];
        let endmonth = start.endOf("month");
        while (start <= end && start <= luxon_1.DateTime.now()) {
            let startmonth = start.startOf("month");
            if (endmonth <= end && start.equals(startmonth)) {
                if (luxon_1.DateTime.now().month === startmonth.month) {
                    ranges.push({
                        start: start,
                        end: end < endmonth ? end : endmonth,
                    });
                }
                else {
                    months.push({
                        start: startmonth,
                        end: end < endmonth ? end : endmonth,
                    });
                }
            }
            else if (endmonth <= end && !start.equals(startmonth)) {
                ranges.push({
                    start: start,
                    end: end < endmonth ? end : endmonth,
                });
            }
            else if (endmonth > end) {
                ranges.push({
                    start: start,
                    end: end < endmonth ? end : endmonth,
                });
            }
            start = start.plus({ months: 1 }).startOf("month");
            endmonth = start.endOf("month");
        }
        let bindingKeys = [];
        let monthConditions = [];
        let rangeConditions = [];
        for (const month of months) {
            bindingKeys.push(month.start.toSQLDate(), month.end.toSQLDate());
            monthConditions.push(`(em."date" BETWEEN $${bindingKeys.length - 1} AND $${bindingKeys.length})`);
        }
        for (const range of ranges) {
            bindingKeys.push(range.start.toSQLDate(), range.end.toSQLDate());
            rangeConditions.push(`(ed."date" BETWEEN $${bindingKeys.length - 1} AND $${bindingKeys.length})`);
        }
        const monthConditionQuery = monthConditions.length > 0 ? monthConditions.join(" OR ") : "FALSE";
        const rangeConditionQuery = rangeConditions.length > 0 ? rangeConditions.join(" OR ") : "FALSE";
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: args.userIds,
                userGroupIds: args.userGroupIds,
                organizationUnitIds: args.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: args.organizationUnitHierarchically,
                userOrganizationStatus: args.status,
            },
            requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            bindingKeys: bindingKeys,
            specificSelectItems: ["userId", "organizationId", "name", "surname", "uniqueId", "organizationUnitIds"],
        });
        const isLastDayToday = ranges.some((range) => range.end.startOf("day").equals(luxon_1.DateTime.now().startOf("day")));
        const q = `
			WITH base_data AS (
				SELECT 
					"userId",
					"data",
					"isChecked"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeMonths}" em
				WHERE ${monthConditionQuery}
				
				UNION ALL
				
				SELECT 
					"userId", 
					jsonb_set(
						jsonb_set(
							data->'s', 
							'{p}', 
							to_jsonb(array(
								SELECT jsonb_array_elements(data->'p')->>'p'
							))
						),
						'{w}', 
						data->'w'
					) AS "data",
					false as "isChecked"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" ed
				WHERE ${rangeConditionQuery}
				${isLastDayToday
            ? `
							UNION ALL
							SELECT 
								"userId", 
								jsonb_set(
									jsonb_set(
										data->'s', 
										'{p}', 
										to_jsonb(array(
											SELECT jsonb_array_elements(data->'p')->>'p'
										))
									),
									'{w}', 
									data->'w'
								) AS "data",
								false as "isChecked"
							FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
							WHERE "data" IS NOT NULL
						`
            : ""}
			),
			filtered_base_data AS (
				SELECT bd."userId", uf."uniqueId", uf."name", uf."surname", bd."data", bd."isChecked" FROM base_data bd
				INNER JOIN (${userFilter.query}) uf on uf."userId" = bd."userId"
			),
			permissions_agg AS (
				SELECT 
					"userId",
					jsonb_agg(DISTINCT elem) AS permissions
				FROM filtered_base_data,
				LATERAL jsonb_array_elements(filtered_base_data.data->'p') AS elem
				GROUP BY "userId"
			),
			workplans_agg AS (
				SELECT 
					"userId",
					jsonb_agg(DISTINCT elem) AS workplans
				FROM filtered_base_data,
				LATERAL jsonb_array_elements(filtered_base_data.data->'w') AS elem
				GROUP BY "userId"
			)
			SELECT 
				fbd."userId",
				fbd."uniqueId",
				fbd."name",
				fbd."surname",
				SUM((fbd.data ->> 'm')::integer) AS total_m,
				SUM((fbd.data ->> 'e')::integer) AS total_e,
				jsonb_build_object(
					'e',   SUM((fbd.data ->> 'e')::integer),
					'm',   SUM((fbd.data ->> 'm')::integer),
					'n',   SUM((fbd.data ->> 'n')::integer),
					'ao',  SUM((fbd.data ->> 'ao')::integer),
					'dw',  SUM((fbd.data ->> 'dw')::integer),
					'eh',  SUM((fbd.data ->> 'eh')::integer),
					'el',  SUM((fbd.data ->> 'el')::integer),
					'eo',  SUM((fbd.data ->> 'eo')::integer),
					'es',  SUM((fbd.data ->> 'es')::integer),
					'ew',  SUM((fbd.data ->> 'ew')::integer),
					'iw',  SUM((fbd.data ->> 'iw')::integer),
					'll',  SUM((fbd.data ->> 'll')::integer),
					'ls',  SUM((fbd.data ->> 'ls')::integer),
					'pr',  SUM((fbd.data ->> 'pr')::integer),
					'uh',  SUM((fbd.data ->> 'uh')::integer),
					'acb', SUM((fbd.data ->> 'acb')::float),
					'afb', SUM((fbd.data ->> 'afb')::float),
					'ecb', SUM((fbd.data ->> 'ecb')::float),
					'efb', SUM((fbd.data ->> 'efb')::float),
					'tel', SUM((fbd.data ->> 'tel')::integer),
					'tes', SUM((fbd.data ->> 'tes')::integer),
					'tll', SUM((fbd.data ->> 'tll')::integer),
					'tls', SUM((fbd.data ->> 'tls')::integer),
					'ucb', SUM((fbd.data ->> 'ucb')::float),
					'ufb', SUM((fbd.data ->> 'ufb')::float),
					's',   CEIL((SUM((fbd.data ->> 'n')::integer) + SUM((fbd.data ->> 'e')::integer))::float / 
						NULLIF(SUM((fbd.data ->> 'ew')::integer), 0) * 100),
					'p',   COALESCE(pa.permissions, '[]'::jsonb),
					'w',   COALESCE(wa.workplans, '[]'::jsonb)
				) AS "data",
				bool_and(fbd."isChecked") AS "isChecked"
			FROM filtered_base_data fbd
			LEFT JOIN permissions_agg pa ON fbd."userId" = pa."userId"
			LEFT JOIN workplans_agg wa ON fbd."userId" = wa."userId"
			LEFT JOIN (${userFilter.query}) uf ON fbd."userId" = uf."userId"
			GROUP BY fbd."userId", pa.permissions, wa.workplans, fbd."uniqueId", fbd."name", fbd."surname"
		`;
        return { q, bindingKeys: userFilter.bindingKeys };
    }
    async getEmployeeDaysDateRangeFiltered(organizationId, requesterUserId, args, trx, onData) {
        let { q, bindingKeys } = this.getEmployeeDaysDateRangeFilteredQuery(organizationId, requesterUserId, {
            startDate: args?.startDate,
            endDate: args?.endDate,
            workPlanIds: args?.workPlanIds,
            userIds: args?.userIds,
            userGroupIds: args?.userGroupIds,
            organizationUnitIds: args?.organizationUnitIds,
            organizationUnitHierarchically: args?.organizationUnitHierarchically,
            status: args?.status,
        });
        q += ' ORDER BY fbd."name" ASC;';
        const cursor = trx.query(new Cursor(q, bindingKeys));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(100, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (rows.length < 100) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
        }
        catch (error) {
            app_logs_1.logger.error(error);
        }
    }
    async getEmployeeDaysDateRangeFilteredWithoutCursor(organizationId, requesterUserId, args, pagination, sortOrder, sortType, trx) {
        let result = {
            pagination: {
                total: 0,
                take: pagination?.take,
                skip: pagination?.skip,
            },
            items: [],
        };
        let { q, bindingKeys } = this.getEmployeeDaysDateRangeFilteredQuery(organizationId, requesterUserId, {
            startDate: args?.startDate,
            endDate: args?.endDate,
            workPlanIds: args?.workPlanIds,
            userIds: args?.userIds,
            userGroupIds: args?.userGroupIds,
            organizationUnitIds: args?.organizationUnitIds,
            organizationUnitHierarchically: args?.organizationUnitHierarchically,
            status: args?.status,
        });
        result.pagination.total += parseInt((await this._pgPool.query("SELECT COUNT(*) FROM (" + q + ")q2", bindingKeys)).rows[0].count);
        q += " ORDER BY";
        switch (sortType) {
            case dal_constants_1.DalConstants.WorkSummarySortType.Name:
                q += ` fbd."name" ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.UniqueId:
                q += ` fbd."uniqueId" ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.ExtraWork:
                q += ` total_e ${sortOrder} NULLS LAST`;
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.MissingWork:
                q += ` total_m ${sortOrder} NULLS LAST`;
                break;
            default:
                break;
        }
        if (pagination.skip) {
            q += ` OFFSET ${pagination.skip} `;
        }
        if (pagination.take) {
            q += ` LIMIT ${pagination.take} `;
        }
        const employeeSummaries = (await trx.query(q, bindingKeys)).rows;
        const userIds = employeeSummaries.map((empSum) => empSum.userId);
        const userInfoList = await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(organizationId, Array.from(userIds));
        const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, Array.from(userIds));
        const employeeDaysSum = await this.employeeDaysSum({
            organizationId,
            requesterUserId,
            pagination,
            startDate: args?.startDate,
            endDate: args?.endDate,
            userIds,
        }, trx);
        result.items = employeeSummaries.map((empSum) => {
            let userInfo = userInfoList.find((u) => u.id == empSum.userId);
            let userCaption = userCaptionList.find((u) => u.id == empSum.userId);
            userInfo.captionLines = userCaption.captionLines;
            let matchedEmpSum = employeeDaysSum.find((empDaysSum) => empDaysSum.userId === empSum.userId);
            return {
                user: userInfo,
                data: {
                    normalWorkDuration: empSum.data.n,
                    extraWorkDuration: empSum.data.e,
                    missedWorkDuration: empSum.data.m,
                    totalWorkDuration: empSum.data.n + empSum.data.e,
                },
                expectedWorkDuration: empSum.data.ew,
                isChecked: empSum.isChecked,
                rawData: {
                    normalWorkDuration: matchedEmpSum.data.normalWorkDuration,
                    extraWorkDuration: matchedEmpSum.data.extraWorkDuration,
                    missedWorkDuration: matchedEmpSum.data.missedWorkDuration,
                    totalWorkDuration: matchedEmpSum.data.normalWorkDuration + matchedEmpSum.data.extraWorkDuration,
                },
            };
        });
        return result;
    }
    async getEmployeeDaysFiltered(organizationId, requesterUserId, args, onData) {
        let qd = "";
        let qdParams = [];
        let isLastDayToday = luxon_1.DateTime.fromJSDate(args.endDate).startOf("day").equals(luxon_1.DateTime.now().startOf("day"));
        let bindingKeys = [args.startDate];
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: args.userIds,
                userGroupIds: args.userGroupIds,
                organizationUnitIds: args.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: args.organizationUnitHierarchically,
                userOrganizationStatus: args.status,
                workPlanIds: args.workPlanIds,
            },
            requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            bindingKeys: bindingKeys,
            specificSelectItems: ["userId", "organizationId", "name", "surname", "uniqueId", "organizationUnitIds"],
        });
        let qWhere = "";
        if (args.workStatusFilter && args.workStatusFilter.length > 0) {
            qWhere += isLastDayToday ? ` WHERE(` : ` AND(`;
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.Came)) {
                qWhere += ` ((ed.data -> 's' ->> 'pr')::integer > 0) OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.NotCame)) {
                qWhere += `  ((ed.data -> 's' ->> 'pr')::integer = 0)  OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave)) {
                qWhere += ` (ed.data -> 's' ->> 'el'):: integer > 0 OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateLeave)) {
                qWhere += ` (ed.data -> 's' ->> 'll'):: integer > 0 OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart)) {
                qWhere += ` (ed.data -> 's' ->> 'es'):: integer > 0 OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateStart)) {
                qWhere += ` (ed.data -> 's' ->> 'ls'):: integer > 0 OR`;
            }
            qWhere = qWhere.substr(0, qWhere.length - 2) + ` )`;
        }
        if (isLastDayToday) {
            qd = `
				SELECT uf."name", uf."surname", uf."uniqueId", ed."userId", ed."data", ed."date"
				FROM (
					SELECT "userId", "data", "date" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
					WHERE "date" >= $1
					UNION ALL
					SELECT "userId", "data", now():: date as "date" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
					WHERE "data" IS NOT NULL
					) as ed
				INNER JOIN (${userFilter.query}) uf on uf."userId" = ed."userId" ${qWhere}
				GROUP BY ed."date", ed."userId", uf."name", uf."surname", uf."uniqueId", ed."data"
				ORDER BY uf."name" ASC,  uf."surname" ASC, ed."date" ${args.sortOrder ?? "DESC"}`;
            qdParams = [...qdParams, ...userFilter.bindingKeys];
        }
        else {
            qd = `
				SELECT uf."name", uf."surname", uf."uniqueId", ed."userId", ed."data", ed."date" 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" ed
				INNER JOIN (${userFilter.query}) uf on uf."userId" = ed."userId"
				WHERE ed."date" >= $1 AND ed."date" <= $${userFilter.bindingKeys.length + 1} ${qWhere}
				GROUP BY uf."name",  uf."surname", ed."date", ed."userId", uf."uniqueId", ed."data"
				ORDER BY uf."name" ASC,  uf."surname" ASC, ed."date" ${args.sortOrder ?? "DESC"}`;
            qdParams = userFilter.bindingKeys;
            qdParams.push(args.endDate);
        }
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(qd, qdParams));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(100, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (rows.length < 100) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
    }
    async getEmployeeDaysPerEmployeeFiltered(organizationId, requesterUserId, args, onData) {
        let qd = "";
        let qdParams = [];
        let isLastDayToday = luxon_1.DateTime.fromJSDate(args.endDate).startOf("day").equals(luxon_1.DateTime.now().startOf("day"));
        let bindingKeys = [args.startDate];
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: args.userIds,
                userGroupIds: args.userGroupIds,
                organizationUnitIds: args.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: args.organizationUnitHierarchically,
                userOrganizationStatus: args.status,
                workPlanIds: args.workPlanIds,
            },
            requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            bindingKeys: bindingKeys,
            specificSelectItems: ["userId", "organizationId", "name", "surname", "uniqueId", "organizationUnitIds"],
        });
        let qWhere = "";
        if (args.workStatusFilter && args.workStatusFilter.length > 0) {
            qWhere += isLastDayToday ? ` WHERE(` : ` AND(`;
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.Came)) {
                qWhere += ` ((ed.data -> 's' ->> 'pr')::integer > 0) OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.NotCame)) {
                qWhere += `  ((ed.data -> 's' ->> 'pr')::integer = 0)  OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave)) {
                qWhere += ` (ed.data -> 's' ->> 'el'):: integer > 0 OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateLeave)) {
                qWhere += ` (ed.data -> 's' ->> 'll'):: integer > 0 OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart)) {
                qWhere += ` (ed.data -> 's' ->> 'es'):: integer > 0 OR`;
            }
            if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateStart)) {
                qWhere += ` (ed.data -> 's' ->> 'ls'):: integer > 0 OR`;
            }
            qWhere = qWhere.substr(0, qWhere.length - 2) + ` )`;
        }
        if (isLastDayToday) {
            qd = `
				SELECT uf."name", uf."surname", uf."uniqueId", ed."userId", jsonb_agg(ed."data") AS days
				FROM (
					SELECT "userId", "data", "date" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
					WHERE "date" >= $1
					UNION ALL
					SELECT "userId", "data", now():: date as "date" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
					WHERE "data" IS NOT NULL
					) as ed
				INNER JOIN (${userFilter.query}) uf on uf."userId" = ed."userId" ${qWhere}
				GROUP BY ed."userId", uf."name", uf."surname", uf."uniqueId"
				ORDER BY uf."name" ASC,  uf."surname" ASC`;
            qdParams = [...qdParams, ...userFilter.bindingKeys];
        }
        else {
            qd = `
				SELECT uf."name", uf."surname", uf."uniqueId", ed."userId", jsonb_agg(ed."data") AS days
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" ed
				INNER JOIN (${userFilter.query}) uf on uf."userId" = ed."userId"
				WHERE ed."date" >= $1 AND ed."date" <= $${userFilter.bindingKeys.length + 1} ${qWhere}
				GROUP BY uf."name",  uf."surname", ed."userId", uf."uniqueId"
				ORDER BY uf."name" ASC,  uf."surname" ASC`;
            qdParams = userFilter.bindingKeys;
            qdParams.push(args.endDate);
        }
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(qd, qdParams));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(100, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (rows.length < 100) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
    }
    async getDailyFirstInLastOutReportForUsers(organizationId, args) {
        let result = {
            pagination: {
                total: 0,
                take: args.take,
                skip: args.skip,
            },
            items: [],
            workPlans: [],
        };
        let isToday = moment(args.date).isSame(moment(), "day");
        let userList = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: organizationId,
            organizationUnitIds: args.organizationUnitIds,
            userGroupIds: args.userGroupIds,
            userIds: args.userIds,
            filterOrganizationUnitMembersHierarchically: args.organizationUnitHierarchically,
        });
        if (!userList && !args.workPlanIds) {
            return Promise.resolve(null);
        }
        let qb = undefined;
        if (isToday) {
            qb = this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay + " as ed")
                .innerJoin("userOrganizations as uo", function () {
                this.on("uo.userId", "ed.userId").andOn("uo.organizationId", "ed.organizationId");
            })
                .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
                .where("uo.organizationId", organizationId)
                .whereNotNull("ed.data");
        }
        else {
            qb = this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays + " as ed")
                .innerJoin("userOrganizations as uo", function () {
                this.on("uo.userId", "ed.userId").andOn("uo.organizationId", "ed.organizationId");
            })
                .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
                .where("uo.organizationId", organizationId)
                .where("ed.date", args.date);
        }
        if (userList && userList.length > 0)
            qb.whereIn("ed.userId", userList.map((u) => {
                return u.userId;
            }));
        if (args.workPlanIds) {
            qb.whereRaw(`ed.data -> 'w' @> ANY(ARRAY['["${args.workPlanIds.join(`"]', '["`)}"]']:: jsonb[])`);
        }
        if (args.status && args.status !== dal_constants_1.DalConstants.IdentityStatusType.All) {
            qb.where("uo.isDisabled", args.status === dal_constants_1.DalConstants.IdentityStatusType.Disabled);
        }
        if (args.workStatusFilter && args.workStatusFilter.length) {
            qb.andWhere((where) => {
                if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.Came)) {
                    where.orWhereRaw(`(ed.data -> 's' ->> 'pr'):: integer > 0`);
                }
                if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.NotCame)) {
                    where.orWhereRaw(`(ed.data -> 's' ->> 'pr'):: integer = 0`);
                }
                if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave)) {
                    where.orWhereRaw(`(ed.data -> 's' ->> 'el'):: integer > 0`);
                }
                if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateLeave)) {
                    where.orWhereRaw(`(ed.data -> 's' ->> 'll'):: integer > 0`);
                }
                if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart)) {
                    where.orWhereRaw(`(ed.data -> 's' ->> 'es'):: integer > 0`);
                }
                if (args.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateStart)) {
                    where.orWhereRaw(`(ed.data -> 's' ->> 'ls'):: integer > 0`);
                }
            });
        }
        await qb
            .clone()
            .count("ed.id")
            .then((rows) => {
            result.pagination.total = parseInt(rows[0].count);
        });
        switch (args.sortingMethod) {
            case dal_constants_1.DalConstants.WorkSummarySortType.ExtraWork:
                qb.orderByRaw(`(ed.data -> 's' ->> 'e'):: integer DESC, ed.data -> 'ex' ->> 'e' ASC NULLS LAST`);
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.MissingWork:
                qb.orderByRaw(`(ed.data -> 's' ->> 'm'):: integer DESC`);
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.FirstStart:
                qb.orderByRaw(`ed.data -> 'ex' ->> 's' ASC NULLS LAST`);
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.LastLeave:
                qb.orderByRaw(`ed.data -> 'ex' ->> 'e' DESC NULLS LAST`);
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.EarylLeave:
                qb.orderByRaw(`(ed.data -> 's' ->> 'el'):: integer DESC, ed.data -> 'ex' ->> 'e' ASC NULLS LAST`);
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.LastLeave:
                qb.orderByRaw(`(ed.data -> 's' ->> 'ls'):: integer DESC, ed.data -> 'ex' ->> 's' DESC NULLS LAST`);
                break;
            case dal_constants_1.DalConstants.WorkSummarySortType.Name:
            default:
                qb.orderBy("uop.name", "asc");
                break;
        }
        if (args.take !== 0)
            qb.limit(args.take);
        qb.offset(args.skip).select("ed.*");
        let rows = await qb;
        let userInfoList = await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(organizationId, rows.map((u) => u.userId));
        let userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, rows.map((u) => u.userId));
        let startDate = moment(args.date).startOf("day").toDate();
        let endDate = moment(args.date).endOf("day").toDate();
        let userPPermissions = await dal_manager_1.dbManager.accessPacs.listUsersCurrentPPermissionInfoWithType(organizationId, rows.map((u) => u.userId), startDate, endDate);
        let workplans = {};
        for (let row of rows) {
            for (const wpid of row.data.w) {
                if (!workplans[wpid]) {
                    let workPlan = await this.getWorkPlan(organizationId, wpid);
                    workplans[wpid] = workPlan;
                    let workPlanActivePeriod = await this.getCurrentWorkPlanPeriod(organizationId, wpid, args.date);
                    let startOfWorkUtc;
                    let endOfWorkUtc;
                    if (row.data && row.data.ws && row.data.ws.length > 0) {
                        let arr = row.data.ws.filter((ws) => ws.wt == "1");
                        if (arr && arr.length > 0) {
                            endOfWorkUtc = moment.max(arr.map((d) => moment(d.e)));
                            startOfWorkUtc = moment.min(arr.map((d) => moment(d.s)));
                        }
                    }
                    result.workPlans.push({
                        id: wpid,
                        name: workPlan.name,
                        type: workPlan.type,
                        startOfWorkUtc: startOfWorkUtc != null ? moment(startOfWorkUtc).format() : startOfWorkUtc,
                        endOfWorkUtc: endOfWorkUtc != null ? moment(endOfWorkUtc).format() : endOfWorkUtc,
                        activePeriodTolerances: workPlanActivePeriod && workPlanActivePeriod.tolerances ? workPlanActivePeriod.tolerances : null,
                    });
                }
            }
            let beginAndEndDateTimes = [];
            for (const workPlanDaySegment of row.data.ws) {
                if (workPlanDaySegment.wt == dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours) {
                    beginAndEndDateTimes.push({ start: moment(workPlanDaySegment.s).toDate(), end: moment(workPlanDaySegment.e).toDate() });
                }
            }
            let userInfo = userInfoList.find((u) => u.id == row.userId);
            let userCaption = userCaptionList.find((u) => u.id == row.userId);
            const accessRangeForLastOut = row.data.ar.length > 0 ? row.data.ar[row.data.ar.length - 1] : undefined;
            const accessRangeForFirstIn = row.data.ar.length > 0 ? row.data.ar[0] : undefined;
            const workPlanOfDay = row?.data?.w?.length ? this.getWorkplansOfDay(row.data.c, row.data.w, workplans) : undefined;
            let item = {
                user: userInfo,
                userCaptions: userCaption ? userCaption.captionLines : [],
                workPlan: workPlanOfDay,
                expectedWorkingRanges: beginAndEndDateTimes,
                entrancePenaltyTimeInSeconds: (row.data.s.ls + row.data.s.tls) * 60,
                entranceTolerance: 0,
                exitPenaltyTimeInSeconds: (row.data.s.el + row.data.s.tel) * 60,
                exitTolerance: 0,
                extraWorkInMinutes: row.data.s.e,
                missingWorkInMinutes: row.data.s.m,
            };
            let wptolerances = result.workPlans.find((w) => w.id === item.workPlan.id)?.activePeriodTolerances;
            if (wptolerances) {
                item.entranceTolerance = wptolerances.lateStart;
                item.exitTolerance = wptolerances.earlyLeave;
            }
            let firstEntryTime;
            row.data.ar.find((elem) => {
                let findingActualAccess = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                if (findingActualAccess) {
                    firstEntryTime = {
                        firstStart: findingActualAccess.s,
                        firstStartId: findingActualAccess.si,
                    };
                }
                else {
                    let findingDummyAccess = elem.a.find((innerElem) => innerElem.s);
                    if (findingDummyAccess) {
                        firstEntryTime = {
                            firstStart: findingDummyAccess.s,
                            firstStartId: findingDummyAccess.si,
                        };
                    }
                }
            });
            let lastLeaveTime;
            row.data.ar.reverse();
            row.data.ar.find((elem) => {
                elem.a.reverse();
                let findingActualAccess = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                if (findingActualAccess) {
                    lastLeaveTime = {
                        lastLeave: findingActualAccess.e,
                        lastLeaveId: findingActualAccess.ei,
                    };
                }
                else {
                    let findingActualAccess = elem.a.find((innerElem) => innerElem.e);
                    if (findingActualAccess) {
                        lastLeaveTime = {
                            lastLeave: findingActualAccess.e,
                            lastLeaveId: findingActualAccess.ei,
                        };
                    }
                }
            });
            item.firstIn = firstEntryTime
                ? {
                    actionUtc: firstEntryTime.firstStart,
                    source: dal_constants_1.DalConstants.AccessLogSourceType.PhysicalAccess,
                    accessLogId: firstEntryTime.firstStartId,
                    ppermissionId: undefined,
                }
                : undefined;
            item.lastOut = lastLeaveTime
                ? {
                    actionUtc: lastLeaveTime.lastLeave,
                    source: dal_constants_1.DalConstants.AccessLogSourceType.PhysicalAccess,
                    accessLogId: lastLeaveTime.lastLeaveId,
                    ppermissionId: undefined,
                }
                : undefined;
            let ppermissionsOfUser = userPPermissions.filter((u) => u.userId == row.userId);
            item.ppermissions = ppermissionsOfUser.map((p) => {
                return {
                    ppermissionTypeId: p.ppermissionType.id,
                    startUtc: p.startDateTime,
                    endUtc: p.endDateTime,
                    ppermissionType: p.ppermissionType.name,
                    status: p.status,
                };
            });
            let holidays = [];
            for (const h of row.data.h) {
                const holidayNames = await this.dbClient.withSchema(organizationId).select("title").from(dal_db_armon_schema_1.ArmonSchema.tableNames.vacations).where("id", h.h);
                holidays.push({
                    startUtc: h.r.s,
                    endUtc: h.r.e,
                    holidayName: holidayNames[0].title,
                });
            }
            item.holidays = holidays;
            result.items.push(item);
        }
        return Promise.resolve(result);
    }
    async commonShiftImport(organizationId, importTemplate, sheet, params) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            if (!importTemplate) {
                throw new Error("No importer defined for this organization");
            }
            let mergeRowQuery = `
			SELECT uwp1.id as "id1", uwp2.id as "id2", uwp1."startDateTime" as "start", uwp2."startDateTime" as "mid", uwp2."endDateTime" as "end"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp1
			INNER JOIN "${organizationId}"."userWorkPlans" uwp2
			ON uwp1."userId" = $1 AND uwp1."userId" = uwp2."userId"
			AND uwp1."workPlanId" = uwp2."workPlanId" AND uwp1."endDateTime" = uwp2."startDateTime";`;
            let rows = [];
            let header;
            let cycleTemplateDayCount;
            try {
                if (sheet && sheet.rowCount > 0) {
                    rows = sheet.getRows(1, sheet.rowCount);
                    if (rows?.length > 0) {
                        header = rows[0];
                        let headerMomentDates = [];
                        if (params.type == dal_constants_1.DalConstants.WorkPlanImportTemplateType.DateRange) {
                            header.eachCell((cell, cellNumber) => {
                                if (cellNumber > 2)
                                    headerMomentDates.push(moment(cell.value.toString(), "DD/MM/YYYY"));
                            });
                        }
                        else if (params.type == dal_constants_1.DalConstants.WorkPlanImportTemplateType.Cyclical) {
                            cycleTemplateDayCount = header.cellCount - 2;
                            const startDate = moment(params.startDate);
                            const endDate = moment(params.endDate);
                            let currentDate = startDate.clone();
                            while (currentDate.isSameOrBefore(endDate, "day")) {
                                headerMomentDates.push(currentDate.clone());
                                currentDate.add(1, "day");
                            }
                        }
                        const importBegin = headerMomentDates[0];
                        const importEnd = headerMomentDates[headerMomentDates.length - 1].clone().add(1, "day");
                        const uniqueIds = [];
                        rows.forEach((row) => {
                            if (row.cellCount >= 3) {
                                uniqueIds.push(row.getCell(2).value);
                            }
                        });
                        const userIds = (await trx.query(`SELECT "userId" FROM "${organizationId}"."userOrganizationProfiles" WHERE "uniqueId" = ANY($1);`, [uniqueIds])).rows.map((row) => row.userId);
                        for (let index = 1; index < rows.length; ++index) {
                            const rowData = rows[index];
                            if (rowData.cellCount < 3) {
                                app_logs_1.logger.info("Empty data row, file might be completed.");
                                continue;
                            }
                            const username = rowData.getCell(1).value;
                            const sicil = rowData.getCell(2).value;
                            if (!sicil) {
                                app_logs_1.logger.error("UniqueId at row " + index + " seems to be empty, skipping");
                                continue;
                            }
                            const userId = (await trx.query(`SELECT "userId" FROM "${organizationId}"."userOrganizationProfiles" WHERE "uniqueId" = $1;`, [sicil])).rows[0].userId;
                            if (!userId) {
                                app_logs_1.logger.error("User " + username + " with sicil " + sicil + " could not be found in the database");
                                continue;
                            }
                            app_logs_1.logger.info("Processing " + username);
                            if (importTemplate.defaultWorkPlanId) {
                                await dal_manager_1.dbManager.accessPacs2.insertWorkPlanMembership(organizationId, {
                                    userId,
                                    workPlanId: importTemplate.defaultWorkPlanId,
                                    range: {
                                        startDateTime: importBegin.toDate(),
                                        endDateTime: importEnd.toDate(),
                                    },
                                }, trx);
                            }
                            let loopIndex = 1;
                            for (const date of headerMomentDates) {
                                try {
                                    let matchedRule;
                                    if (params.type == dal_constants_1.DalConstants.WorkPlanImportTemplateType.DateRange) {
                                        matchedRule = importTemplate.workPlanImportRules.find((rule) => rule.keyword === rowData.getCell(loopIndex + 2).value?.toString());
                                    }
                                    else if (params.type == dal_constants_1.DalConstants.WorkPlanImportTemplateType.Cyclical) {
                                        matchedRule = importTemplate.workPlanImportRules.find((rule) => rule.keyword === rowData.getCell(((loopIndex - 1) % cycleTemplateDayCount) + 3).value?.toString());
                                    }
                                    const dayImportData = {
                                        start: date
                                            .clone()
                                            .startOf("day")
                                            .add(matchedRule?.membershipRange.from.days ?? 0, "days")
                                            .add(matchedRule?.membershipRange.from.hours ?? 0, "hours")
                                            .add(matchedRule?.membershipRange.from.minutes ?? 0, "minutes"),
                                        end: date
                                            .clone()
                                            .startOf("day")
                                            .add(matchedRule?.membershipRange.to.days ?? 0, "days")
                                            .add(matchedRule?.membershipRange.to.hours ?? 0, "hours")
                                            .add(matchedRule?.membershipRange.to.minutes ?? 0, "minutes"),
                                        workplan: matchedRule?.workPlanId,
                                    };
                                    if (dayImportData?.workplan) {
                                        await dal_manager_1.dbManager.accessPacs2.insertWorkPlanMembership(organizationId, {
                                            userId,
                                            workPlanId: dayImportData.workplan,
                                            range: {
                                                startDateTime: dayImportData.start.toDate(),
                                                endDateTime: dayImportData.end.toDate(),
                                            },
                                        }, trx);
                                    }
                                    loopIndex++;
                                }
                                catch (error) {
                                    app_logs_1.logger.info("Shift import error for " +
                                        username +
                                        " on date " +
                                        headerMomentDates[loopIndex - 1].format("YYYY-MM-DD") +
                                        " -" +
                                        rowData.getCell(loopIndex) +
                                        "-" +
                                        error);
                                    process.exit(1);
                                }
                            }
                            app_logs_1.logger.info("Merging " + username);
                            let rowsToMerge = (await trx.query(mergeRowQuery, [userId])).rows[0];
                            while (rowsToMerge) {
                                await trx.query(`UPDATE "${organizationId}"."userWorkPlans"
							SET "endDateTime" = $1 WHERE id = $2;`, [rowsToMerge.end, rowsToMerge.id1]);
                                await trx.query(`DELETE FROM "${organizationId}"."userWorkPlans"
							WHERE id = $1`, [rowsToMerge.id2]);
                                rowsToMerge = (await trx.query(mergeRowQuery, [userId])).rows[0];
                            }
                            await trx.query(`INSERT INTO "${organizationId}"."recalculateWorkQueue"
						(id, "userId", "organizationId", reason, "startDate")
						VALUES (gen_random_uuid(), $1, '${organizationId}', $2, $3);`, [userId, dal_constants_1.DalConstants.RecalculateWorkReason.UserWorkPlanChange, importBegin.toDate()]);
                            await trx.query(`UPDATE "${organizationId}"."employeeLiveDay" eld
						SET "updateRequiredReason" = "updateRequiredReason" | $1, "updateRequiredRaisedAt" = now()
						WHERE eld."userId" = $2`, [app_enums_1.enums.RecalculateWorkReason.UserWorkPlanChange, userId]);
                            const now = moment();
                            if (importBegin <= now && now <= importEnd) {
                                let workPlans = await this.getWorkPlansOfNowPg(trx, organizationId);
                                await this.scaffoldEmployeeLiveDataForWorkplansPg(trx, workPlans, organizationId, userIds);
                            }
                        }
                    }
                    else {
                        app_logs_1.logger.error("Could not split csv file");
                    }
                }
                else {
                    app_logs_1.logger.error("CSV File is empty!");
                }
            }
            catch (error) {
                app_logs_1.logger.error("Error while importing workplans: " + error);
            }
            app_logs_1.logger.info("Importing workplan memberships completed.");
        });
    }
    async getDailySummaryReportForSingleUser(organizationId, userId, filter) {
        return this.dbClient.transaction(async (trx) => {
            let result = {
                pagination: {
                    take: filter.pagination.take,
                    skip: filter.pagination.skip,
                    total: 0,
                },
                items: [],
                workPlans: [],
                summary: {
                    arrivedOnTimeDaysCount: 0,
                    earlyLeftDaysCount: 0,
                    lateArrivedDaysCount: 0,
                    lateLeftDaysCount: 0,
                    workDaysCount: 0,
                    ppermissionUsageList: [],
                },
                userCaptions: [],
                userFullnameAndUniqueIds: [],
            };
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getSingleUserOrganizationCaptionLines(organizationId, userId, trx);
            result.userFullnameAndUniqueIds = await dal_manager_1.dbManager.accessUser.getUsersFullnamesAndUniqueIdsKnex(organizationId, [userId], trx);
            let hasToday = moment(filter.dateRange.endDateTime).isSameOrAfter(moment(), "day");
            let range = new moment_range_1.DateRange(filter.dateRange.startDateTime, hasToday ? moment().subtract(1, "day").endOf("day") : filter.dateRange.endDateTime);
            let orderByClause = `ORDER BY date DESC`;
            let additionalWhere = "";
            if (filter.workStatusFilter && filter.workStatusFilter.length > 0) {
                additionalWhere += ` AND(`;
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.Came)) {
                    additionalWhere += ` (ed.data -> 's' ->> 'pr'):: integer > 0 OR`;
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.NotCame)) {
                    additionalWhere += ` (ed.data -> 's' ->> 'pr'):: integer = 0 OR`;
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave)) {
                    additionalWhere += ` (ed.data -> 's' ->> 'el'):: integer > 0 OR`;
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateLeave)) {
                    additionalWhere += ` (ed.data -> 's' ->> 'll'):: integer > 0 OR`;
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart)) {
                    additionalWhere += ` (ed.data -> 's' ->> 'es'):: integer > 0 OR`;
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateStart)) {
                    additionalWhere += ` (ed.data -> 's' ->> 'ls'):: integer > 0 OR`;
                }
                additionalWhere = additionalWhere.substr(0, additionalWhere.length - 2) + ` )`;
            }
            let bindings = [userId, organizationId, range.start.toDate(), range.end.toDate()];
            let rawQuery = `
					SELECT ed."userId", uop."name", ed."date", ed."data"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS ed
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo 
						ON uo."userId" = ed."userId" AND uo."organizationId" = ed."organizationId"
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop 
						ON uop."userOrganizationId" = uo."id"
					WHERE ed."userId" = ? AND uo."organizationId" = ? AND uop."deletedAt" IS NULL
					AND ed.date >= ? AND ed.date <= ?
					${additionalWhere} `;
            if (hasToday) {
                rawQuery += `
					UNION ALL
					SELECT ed."userId", uop."name", now():: date, ed."data"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" AS ed
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo 
						ON uo."userId" = ed."userId" AND uo."organizationId" = ed."organizationId"
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop 
						ON uop."userOrganizationId" = uo."id"
					WHERE ed."userId" = ? AND ed."organizationId" = ? AND uop."deletedAt" IS NULL AND ed."data" IS NOT NULL
					${additionalWhere}
					`;
                bindings.push(...[userId, organizationId]);
            }
            rawQuery += ` ${orderByClause} `;
            let countQuery = trx.raw(`SELECT count(*) FROM(${rawQuery})sq`, bindings).toQuery();
            let count = await trx.raw(countQuery);
            result.pagination.total = count.rows ? count.rows[0].count : 0;
            let paginatedQuery = rawQuery;
            if (filter.pagination.skip) {
                paginatedQuery += `
					OFFSET ${filter.pagination.skip} `;
            }
            if (filter.pagination.take) {
                paginatedQuery += `
					LIMIT ${filter.pagination.take} `;
            }
            let boundQuery = trx.raw(paginatedQuery, bindings);
            let dayResults = await boundQuery;
            for (const dayResult of dayResults.rows) {
                let data = dayResult.data;
                let beginAndEndDateTimes = [];
                let holidays = [];
                for (const h of data.h) {
                    const holidayNames = await trx.withSchema(organizationId).select("title").from(dal_db_armon_schema_1.ArmonSchema.tableNames.vacations).where("id", h.h);
                    holidays.push({
                        startUtc: h.r.s,
                        endUtc: h.r.e,
                        holidayName: holidayNames[0].title,
                    });
                }
                for (const workPlanDaySegment of data.ws) {
                    if (workPlanDaySegment.wt == dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours) {
                        beginAndEndDateTimes.push({ start: moment(workPlanDaySegment.s).toDate(), end: moment(workPlanDaySegment.e).toDate() });
                    }
                }
                let workPlan = {
                    id: undefined,
                    name: undefined,
                    type: undefined,
                    startOfWorkUtc: undefined,
                    endOfWorkUtc: undefined,
                };
                let workPlanActivePeriod = undefined;
                if (data.w && data.w.length > 0) {
                    let wpqb = trx
                        .withSchema(organizationId)
                        .table(dal_db_armon_schema_1.ArmonSchema.viewNames.vW_WorkPlanPeriodTstzrange + " as wppr")
                        .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans + " as wp", "wp.id", "wppr.workPlanId")
                        .where("wppr.workPlanId", data.w[0])
                        .where("wppr.organizationId", organizationId)
                        .whereRaw(`wppr.range @> ?:: timestamp with time zone`, dayResult.date)
                        .select("wppr.*")
                        .select(this.dbClient.raw('lower(wppr.range) as "periodStart"'))
                        .select("wp.name")
                        .select("wp.type");
                    workPlanActivePeriod = await this.getCurrentWorkPlanPeriod(organizationId, data.w[0], new Date(data.d), trx);
                    let workPlanDetails = await wpqb;
                    if (workPlanDetails.length > 0) {
                        workPlan.id = workPlanDetails[0].id;
                        workPlan.name = workPlanDetails[0].name;
                        workPlan.type = workPlanDetails[0].type;
                        let wpDetails = workPlanDetails[0].details;
                        if (wpDetails) {
                            if (wpDetails.dayItems) {
                                let dayItem = this.findDayItem(new Date(dayResult.date), new Date(workPlanDetails[0].periodStart), wpDetails.routineType, wpDetails.dayItems);
                                if (dayItem && dayItem.workingHours && dayItem.workingHours.length > 0) {
                                    workPlan.startOfWorkUtc = moment(dayResult.date)
                                        .startOf("day")
                                        .hour(dayItem.workingHours[0].startTime.replace(":", "").substr(0, 2))
                                        .minute(dayItem.workingHours[0].startTime.replace(":", "").substr(2, 2))
                                        .toDate();
                                    workPlan.endOfWorkUtc = moment(dayResult.date)
                                        .startOf("day")
                                        .hour(dayItem.workingHours[dayItem.workingHours.length - 1].endTime.replace(":", "").substr(0, 2))
                                        .minute(dayItem.workingHours[dayItem.workingHours.length - 1].endTime.replace(":", "").substr(2, 2))
                                        .toDate();
                                }
                            }
                        }
                    }
                }
                const accessRangeForLastOut = data.ar.length > 0 ? data.ar[data.ar.length - 1] : undefined;
                const accessRangeForFirstIn = data.ar.length > 0 ? data.ar[0] : undefined;
                const nextDayOfEndDate = new Date(filter.dateRange.endDateTime);
                nextDayOfEndDate.setDate(nextDayOfEndDate.getDate() + 1);
                let userPPermissions = await dal_manager_1.dbManager.accessPacs.listUsersCurrentPPermissionInfoWithType(organizationId, [userId], filter.dateRange.startDateTime, nextDayOfEndDate);
                let firstEntryTime;
                data.ar.find((elem) => {
                    let findingActualAccess = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                    if (findingActualAccess) {
                        firstEntryTime = {
                            firstStart: findingActualAccess.s,
                            firstStartId: findingActualAccess.si,
                        };
                    }
                    else {
                        let findingDummyAccess = elem.a.find((innerElem) => innerElem.s);
                        if (findingDummyAccess) {
                            firstEntryTime = {
                                firstStart: findingDummyAccess.s,
                                firstStartId: findingDummyAccess.si,
                            };
                        }
                    }
                });
                let lastLeaveTime;
                data.ar.reverse();
                data.ar.find((elem) => {
                    elem.a.reverse();
                    let findingActualAccess = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                    if (findingActualAccess) {
                        lastLeaveTime = {
                            lastLeave: findingActualAccess.e,
                            lastLeaveId: findingActualAccess.ei,
                        };
                    }
                    else {
                        let findingActualAccess = elem.a.find((innerElem) => innerElem.e);
                        if (findingActualAccess) {
                            lastLeaveTime = {
                                lastLeave: findingActualAccess.e,
                                lastLeaveId: findingActualAccess.ei,
                            };
                        }
                    }
                });
                let item = {
                    date: dayResult.date,
                    entrancePenaltyTimeInSeconds: (data.s.ls + data.s.tls) * 60,
                    entranceTolerance: 0,
                    exitPenaltyTimeInSeconds: (data.s.el + data.s.tel) * 60,
                    exitTolerance: 0,
                    expectedWorkDuration: data.s.ew,
                    normalWorkDuration: data.s.n,
                    extraWorkInMinutes: data.s.e,
                    usedFixedBreakDuration: data.s.ufb,
                    automaticallyUsedFixedBreakDuration: data.s.afb,
                    firstIn: firstEntryTime
                        ? {
                            actionUtc: firstEntryTime.firstStart,
                            source: dal_constants_1.DalConstants.AccessLogSourceType.PhysicalAccess,
                            accessLogId: firstEntryTime.firstStartId,
                            ppermissionId: undefined,
                        }
                        : undefined,
                    lastOut: moment(dayResult.date).isSame(moment(), "day")
                        ? undefined
                        : lastLeaveTime
                            ? {
                                actionUtc: lastLeaveTime.lastLeave,
                                source: dal_constants_1.DalConstants.AccessLogSourceType.PhysicalAccess,
                                accessLogId: lastLeaveTime.lastLeaveId,
                                ppermissionId: undefined,
                            }
                            : undefined,
                    missingWorkInMinutes: data.s.m,
                    workPlan: workPlan,
                    physicallyInRegionDuration: data.s.pr,
                    expectedWorkingRanges: beginAndEndDateTimes,
                    normalWork: data.s.n,
                    holidays: holidays,
                };
                if (workPlanActivePeriod && workPlanActivePeriod.tolerances) {
                    item.entranceTolerance = workPlanActivePeriod.tolerances.lateStart;
                    item.exitTolerance = workPlanActivePeriod.tolerances.earlyLeave;
                }
                item.ppermissions = userPPermissions
                    .filter((p) => {
                    return new moment_range_1.DateRange(p.startDateTime, p.endDateTime).overlaps(new moment_range_1.DateRange(moment(dayResult.date).startOf("day"), moment(dayResult.date).add(1, "days")));
                })
                    .map((p) => {
                    let intersect = new moment_range_1.DateRange(p.startDateTime, p.endDateTime).intersect(new moment_range_1.DateRange(moment(dayResult.date).startOf("day"), moment(dayResult.date).add(1, "days")));
                    return {
                        startUtc: intersect.start.toDate(),
                        endUtc: intersect.end.toDate(),
                        ppermissionTypeName: p.ppermissionType.name,
                        status: p.status,
                    };
                });
                result.items.push(item);
            }
            let permissionUsages = [];
            let summaryQueryResult = await trx.raw(`SELECT date, (data ->> 'p'):: jsonb as permissions, (data ->> 's'):: jsonb as summary, (data ->> 'ex'):: jsonb as extra FROM(${rawQuery})sq`, bindings);
            for (const row of summaryQueryResult.rows) {
                for (const permission of row.permissions) {
                    let usage = permissionUsages.find((p) => p.permissionId === permission.p);
                    if (!usage) {
                        usage = {
                            permissionId: permission.p,
                            usage: permission.u,
                        };
                        permissionUsages.push(usage);
                    }
                    else {
                        usage.usage += permission.u;
                    }
                }
                if (row.summary.ls === 0 && row.extra.s)
                    result.summary.arrivedOnTimeDaysCount++;
                if (row.summary.el > 0)
                    result.summary.earlyLeftDaysCount++;
                if (row.summary.ls > 0)
                    result.summary.lateArrivedDaysCount++;
                if (row.summary.ll > 0)
                    result.summary.lateLeftDaysCount++;
                if (moment(row.date).isoWeekday() < 6)
                    result.summary.workDaysCount++;
            }
            if (permissionUsages.length > 0) {
                let permissionTypes = await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions + " as pp")
                    .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes + " as ppt", "ppt.id", "pp.ppermissionTypeId")
                    .whereIn("pp.id", permissionUsages.map((u) => u.permissionId))
                    .select(["pp.id", "ppt.name", "ppt.isDailyScheduled"]);
                for (const usage of permissionUsages) {
                    let type = permissionTypes.find((pt) => pt.id === usage.permissionId);
                    if (usage.usage > 0 && type) {
                        let ids = permissionUsages.map((p) => p.permissionId).filter((pt) => pt);
                        let perms = await this.getBasicInfoOfPermissions(organizationId, ids);
                        let perm = perms.find((p) => p.id === usage.permissionId);
                        if (perm) {
                            result.summary.ppermissionUsageList.push(perm);
                        }
                    }
                }
            }
            return result;
        });
    }
    async getDailySummaryReportForMultipleUser(organizationId, filter) {
        return this.dbClient.transaction(async (trx) => {
            const userIds = filter.userIds;
            let result = [];
            const [usersCaptions, usersFullnameAndUniqueIds] = await Promise.all([
                dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds, trx),
                dal_manager_1.dbManager.accessUser.getUsersFullnamesAndUniqueIdsKnex(organizationId, userIds, trx),
            ]);
            const hasToday = luxon_1.DateTime.fromJSDate(filter.dateRange.endDateTime).startOf("day") >= luxon_1.DateTime.now().startOf("day");
            const range = new moment_range_1.DateRange(filter.dateRange.startDateTime, hasToday ? luxon_1.DateTime.now().minus({ days: 1 }).endOf("day").toJSDate() : filter.dateRange.endDateTime);
            const clauses = [];
            if (filter?.workStatusFilter?.length) {
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.Came)) {
                    clauses.push(`(ed.data -> 's' ->> 'pr')::integer > 0`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.NotCame)) {
                    clauses.push(`(ed.data -> 's' ->> 'pr')::integer = 0`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyLeave)) {
                    clauses.push(`(ed.data -> 's' ->> 'el')::integer > 0`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateLeave)) {
                    clauses.push(`(ed.data -> 's' ->> 'll')::integer > 0`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.EarlyStart)) {
                    clauses.push(`(ed.data -> 's' ->> 'es')::integer > 0`);
                }
                if (filter.workStatusFilter.includes(app_enums_1.enums.WorkStatusFilter.LateStart)) {
                    clauses.push(`(ed.data -> 's' ->> 'ls')::integer > 0`);
                }
            }
            const additionalWhere = clauses.length > 0 ? ` AND (${clauses.join(" OR ")})` : "";
            const orderBy = ` ORDER BY date DESC;`;
            const bindings = [userIds, organizationId, range.start.toDate(), range.end.toDate()];
            let rawQuery = `
				SELECT 
					ed."userId", 
					uop."name", 
					ed."date", 
					ed."data"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS ed
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo 
					ON uo."userId" = ed."userId" AND uo."organizationId" = ed."organizationId"
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop 
					ON uop."userOrganizationId" = uo."id"
				WHERE 
					ed."userId" = ANY (?) 
					AND uo."organizationId" = ? 
					AND uop."deletedAt" IS NULL
					AND ed.date >= ? 
					AND ed.date <= ?
					${additionalWhere}
			`;
            if (hasToday) {
                rawQuery += `
					UNION ALL
					SELECT 
						ed."userId", 
						uop."name", 
						now()::date, 
						ed."data"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" AS ed
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo 
						ON uo."userId" = ed."userId" 
						AND uo."organizationId" = ed."organizationId"
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop 
						ON uop."userOrganizationId" = uo."id"
					WHERE 
						ed."userId" = ANY (?) 
						AND ed."organizationId" = ? 
						AND uop."deletedAt" IS NULL 
						AND ed."data" IS NOT NULL
						${additionalWhere}
				`;
                bindings.push(userIds, organizationId);
            }
            rawQuery += orderBy;
            const dayResults = await trx.raw(rawQuery, bindings);
            const userDayResults = [];
            const allHolidayIdSet = new Set();
            const allPermissionIdSet = new Set();
            const workPlanIds = new Set();
            const workPlanDateMap = new Map();
            for (const row of dayResults.rows) {
                const { userId, data, date } = row;
                let userGroup = userDayResults.find((u) => u.userId === userId);
                if (!userGroup) {
                    userGroup = { userId, records: [] };
                    userDayResults.push(userGroup);
                }
                userGroup.records.push(row);
                (data.h || []).forEach((h) => h?.h && allHolidayIdSet.add(h.h));
                (data.p || []).forEach((p) => p?.p && allPermissionIdSet.add(p.p));
                if (data.w?.length) {
                    const workPlanId = data.w[0];
                    workPlanIds.add(workPlanId);
                    workPlanDateMap.set(`${workPlanId}_${date.toISOString()}`, date);
                }
            }
            const allHolidayIds = Array.from(allHolidayIdSet);
            const allPermissionIds = Array.from(allPermissionIdSet);
            const holidayNames = allHolidayIds.length
                ? (await trx.raw(`SELECT "id", "title" 
								FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.vacations}" 
								WHERE "id" = ANY (?)`, [allHolidayIds])).rows
                : [];
            let basicPermissionsMap = new Map();
            if (allPermissionIds.length) {
                const basicPermissions = await this.getBasicInfoOfPermissions(organizationId, allPermissionIds);
                basicPermissionsMap = new Map(basicPermissions.map((p) => [p.id, p]));
            }
            const workPlanPeriodsMap = new Map();
            if (workPlanIds.size > 0) {
                for (const workPlanId of workPlanIds) {
                    for (const [key, date] of workPlanDateMap.entries()) {
                        if (key.startsWith(workPlanId)) {
                            const period = await this.getCurrentWorkPlanPeriod(organizationId, workPlanId, date, trx);
                            workPlanPeriodsMap.set(key, period);
                        }
                    }
                }
            }
            const workPlanDetailsMap = new Map();
            if (workPlanIds.size > 0) {
                const dates = Array.from(workPlanDateMap.values());
                const workPlanDetailsResult = await trx.raw(`SELECT 
						wppr.*, 
						lower(wppr.range) AS "periodStart", 
						wp.name, 
						wp.type
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_WorkPlanPeriodTstzrange}" AS wppr
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}" AS wp 
						ON wp.id = wppr."workPlanId"
					WHERE wppr."workPlanId" = ANY (?) 
						AND wppr."organizationId" = ? 
						AND (wppr.range @> ANY(?::timestamptz[]))`, [Array.from(workPlanIds), organizationId, dates]);
                for (const wpd of workPlanDetailsResult.rows) {
                    for (const date of dates) {
                        if (this.isInPostgresRange(date, wpd.range)) {
                            workPlanDetailsMap.set(`${wpd.workPlanId}_${date.toISOString()}`, wpd);
                        }
                    }
                }
            }
            const nextDayOfEndDate = new Date(filter.dateRange.endDateTime);
            nextDayOfEndDate.setDate(nextDayOfEndDate.getDate() + 1);
            const allUserPPermissions = await dal_manager_1.dbManager.accessPacs.listUsersCurrentPPermissionInfoWithType(organizationId, userIds, filter.dateRange.startDateTime, nextDayOfEndDate);
            const userPPermissionsMap = new Map();
            for (const { userId } of userDayResults) {
                const permissions = allUserPPermissions.filter((p) => p.userId === userId).sort((a, b) => a.startDateTime.getTime() - b.startDateTime.getTime());
                userPPermissionsMap.set(userId, permissions);
            }
            for (const userDayResult of userDayResults) {
                const userId = userDayResult.userId;
                let resultItem = {
                    userId: userId,
                    items: [],
                    workPlans: [],
                    summary: {
                        arrivedOnTimeDaysCount: 0,
                        earlyLeftDaysCount: 0,
                        lateArrivedDaysCount: 0,
                        lateLeftDaysCount: 0,
                        workDaysCount: 0,
                        ppermissionUsageList: [],
                    },
                    userCaptions: usersCaptions.find((item) => item.id === userDayResult.userId).captionLines,
                    userFullnameAndUniqueIds: [usersFullnameAndUniqueIds.find((item) => item.userId === userDayResult.userId)],
                };
                const permissionUsageMap = new Map();
                const dayResults = userDayResult.records;
                for (const dayResult of dayResults) {
                    let data = dayResult.data;
                    for (const permission of data.p) {
                        const existingUsage = permissionUsageMap.get(permission.p) || 0;
                        permissionUsageMap.set(permission.p, existingUsage + permission.u);
                    }
                    const { ls = 0, el = 0, ll = 0 } = data.s || {};
                    const arrivedOnTime = ls === 0 && data.ex?.s;
                    if (arrivedOnTime)
                        resultItem.summary.arrivedOnTimeDaysCount++;
                    if (el > 0)
                        resultItem.summary.earlyLeftDaysCount++;
                    if (ls > 0)
                        resultItem.summary.lateArrivedDaysCount++;
                    if (ll > 0)
                        resultItem.summary.lateLeftDaysCount++;
                    if (luxon_1.DateTime.fromJSDate(dayResult.date).weekday < 6) {
                        resultItem.summary.workDaysCount++;
                    }
                    const holidays = (data.h || []).map((h) => {
                        const holidayName = holidayNames.find((item) => item.id === h.h);
                        return {
                            startUtc: h.r.s,
                            endUtc: h.r.e,
                            holidayName,
                        };
                    });
                    const expectedWorkingRanges = (data.ws || [])
                        .filter((segment) => segment.wt === dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours)
                        .map((segment) => ({
                        start: new Date(segment.s),
                        end: new Date(segment.e),
                    }));
                    let workPlan = {
                        id: undefined,
                        name: undefined,
                        type: undefined,
                        startOfWorkUtc: undefined,
                        endOfWorkUtc: undefined,
                    };
                    let workPlanActivePeriod = undefined;
                    if (data.w?.length) {
                        const workPlanId = data.w[0];
                        const key = `${workPlanId}_${dayResult.date.toISOString()}`;
                        workPlanActivePeriod = workPlanPeriodsMap.get(key);
                        const wp = workPlanDetailsMap.get(key);
                        if (wp) {
                            Object.assign(workPlan, { id: wp.id, name: wp.name, type: wp.type });
                            const wpDetails = wp.details;
                            const dayItem = wpDetails?.dayItems
                                ? this.findDayItem(new Date(dayResult.date), new Date(wp.periodStart), wpDetails.routineType, wpDetails.dayItems)
                                : undefined;
                            if (dayItem?.workingHours?.length) {
                                const startTimeStr = dayItem.workingHours[0].startTime.replace(":", "");
                                const startHour = parseInt(startTimeStr.substr(0, 2), 10);
                                const startMinute = parseInt(startTimeStr.substr(2, 2), 10);
                                const endTimeStr = dayItem.workingHours[dayItem.workingHours.length - 1].endTime.replace(":", "");
                                const endHour = parseInt(endTimeStr.substr(0, 2), 10);
                                const endMinute = parseInt(endTimeStr.substr(2, 2), 10);
                                workPlan.startOfWorkUtc = luxon_1.DateTime.fromJSDate(dayResult.date).startOf("day").set({ hour: startHour, minute: startMinute }).toJSDate();
                                workPlan.endOfWorkUtc = luxon_1.DateTime.fromJSDate(dayResult.date).startOf("day").set({ hour: endHour, minute: endMinute }).toJSDate();
                            }
                        }
                    }
                    const userPPermissions = userPPermissionsMap.get(userId) || [];
                    let firstEntryTime;
                    data.ar.find((elem) => {
                        let findingActualAccess = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                        if (findingActualAccess) {
                            firstEntryTime = {
                                firstStart: findingActualAccess.s,
                                firstStartId: findingActualAccess.si,
                            };
                        }
                        else {
                            let findingDummyAccess = elem.a.find((innerElem) => innerElem.s);
                            if (findingDummyAccess) {
                                firstEntryTime = {
                                    firstStart: findingDummyAccess.s,
                                    firstStartId: findingDummyAccess.si,
                                };
                            }
                        }
                    });
                    let lastLeaveTime;
                    data.ar.reverse();
                    data.ar.find((elem) => {
                        elem.a.reverse();
                        let findingActualAccess = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                        if (findingActualAccess) {
                            lastLeaveTime = {
                                lastLeave: findingActualAccess.e,
                                lastLeaveId: findingActualAccess.ei,
                            };
                        }
                        else {
                            let findingActualAccess = elem.a.find((innerElem) => innerElem.e);
                            if (findingActualAccess) {
                                lastLeaveTime = {
                                    lastLeave: findingActualAccess.e,
                                    lastLeaveId: findingActualAccess.ei,
                                };
                            }
                        }
                    });
                    let item = {
                        date: dayResult.date,
                        entrancePenaltyTimeInSeconds: (data.s.ls + data.s.tls) * 60,
                        entranceTolerance: workPlanActivePeriod?.tolerances?.lateStart ?? 0,
                        exitPenaltyTimeInSeconds: (data.s.el + data.s.tel) * 60,
                        exitTolerance: workPlanActivePeriod?.tolerances?.earlyLeave ?? 0,
                        expectedWorkDuration: data.s.ew,
                        normalWorkDuration: data.s.n,
                        extraWorkInMinutes: data.s.e,
                        usedFixedBreakDuration: data.s.ufb,
                        automaticallyUsedFixedBreakDuration: data.s.afb,
                        firstIn: firstEntryTime
                            ? {
                                actionUtc: firstEntryTime.firstStart,
                                source: dal_constants_1.DalConstants.AccessLogSourceType.PhysicalAccess,
                                accessLogId: firstEntryTime.firstStartId,
                                ppermissionId: undefined,
                            }
                            : undefined,
                        lastOut: luxon_1.DateTime.fromJSDate(dayResult.date).hasSame(luxon_1.DateTime.now(), "day")
                            ? undefined
                            : lastLeaveTime
                                ? {
                                    actionUtc: lastLeaveTime.lastLeave,
                                    source: dal_constants_1.DalConstants.AccessLogSourceType.PhysicalAccess,
                                    accessLogId: lastLeaveTime.lastLeaveId,
                                    ppermissionId: undefined,
                                }
                                : undefined,
                        missingWorkInMinutes: data.s.m,
                        workPlan: workPlan,
                        physicallyInRegionDuration: data.s.pr,
                        expectedWorkingRanges,
                        normalWork: data.s.n,
                        holidays: holidays,
                        ppermissions: userPPermissions
                            .filter((p) => {
                            const startOfDay = luxon_1.DateTime.fromJSDate(dayResult.date).startOf("day");
                            const endOfDay = startOfDay.plus({ days: 1 });
                            const permissionStart = luxon_1.DateTime.fromJSDate(p.startDateTime);
                            const permissionEnd = luxon_1.DateTime.fromJSDate(p.endDateTime);
                            return permissionStart < endOfDay && permissionEnd > startOfDay;
                        })
                            .map((p) => {
                            const startOfDay = luxon_1.DateTime.fromJSDate(dayResult.date).startOf("day");
                            const endOfDay = startOfDay.plus({ days: 1 });
                            const permissionStart = luxon_1.DateTime.fromJSDate(p.startDateTime);
                            const permissionEnd = luxon_1.DateTime.fromJSDate(p.endDateTime);
                            const intersectStart = permissionStart > startOfDay ? permissionStart : startOfDay;
                            const intersectEnd = permissionEnd < endOfDay ? permissionEnd : endOfDay;
                            return {
                                startUtc: intersectStart.toJSDate(),
                                endUtc: intersectEnd.toJSDate(),
                                ppermissionTypeName: p.ppermissionType.name,
                                status: p.status,
                            };
                        }),
                    };
                    resultItem.items.push(item);
                }
                if (permissionUsageMap.size > 0) {
                    for (const [permissionId, usage] of permissionUsageMap.entries()) {
                        if (usage > 0) {
                            const perm = basicPermissionsMap.get(permissionId);
                            if (perm) {
                                resultItem.summary.ppermissionUsageList.push(perm);
                            }
                        }
                    }
                }
                result.push(resultItem);
            }
            return result;
        });
    }
    async listMonthlyReport(params) {
        return this.dbClient.transaction(async (trx) => {
            let result = {
                pagination: {
                    total: 0,
                    take: params.pagination.take,
                    skip: params.pagination.skip,
                },
                items: [],
            };
            let userFilter = this.getReportUserFilter({
                requesterUserId: params.requesterUserId,
                organizationId: params.organizationId,
                idBasedUserFilter: params.idBasedUserFilter,
                requiredOrganizationWidePermissions: params.requiredOrganizationWidePermissions,
                requiredOrganizationUnitWidePermissions: params.requiredOrganizationUnitWidePermissions,
            });
            let rawQuery = `
SELECT range.*,
    CASE WHEN fr.id IS NOT NULL THEN fr.data ELSE sumq.data END
FROM
    (
        SELECT	extract('year' from "dateRange".date):: integer AS year,
				extract('month' from "dateRange".date):: integer AS month
		FROM(
            SELECT generate_series(
                date_trunc('month', ?:: timestamp with time zone), ?, '1 month'
				) AS date
            ) "dateRange"
	) range
LEFT JOIN(
    SELECT	em.id as id,
			extract(year from em.date) as year,
    		extract(month from em.date) as "month",
			json_build_object(
				'id', em.id,
				'userId', uf."userId",
				'name', uf.name,
				'note', em.note,
				'isChecked', em."isChecked",
				'year', extract(year from em.date),
				'month', extract(month from em.date),
				'expectedWorkDuration', (em."data" ->> 'ew'):: integer,
				'missedWorkDuration', (em."data" ->> 'm'):: integer,
				'normalWorkDuration', (em."data" ->> 'n'):: integer,
				'extraWorkDuration', (em."data" ->> 'e'):: integer,
				'accountedOffTimePermissionDuration', (em."data" ->> 'ao'):: integer,
				'usedHoliday', (em."data" ->> 'uh'):: integer,
				'workPlanList', em."data" -> 'w',
				'ppermissionList', em."data" -> 'p',
				'earlyLeave', (em."data" ->> 'el'):: integer,
				'earlyStart', (em."data" ->> 'es'):: integer,
				'lateLeave', (em."data" ->> 'll'):: integer,
				'lateStart', (em."data" ->> 'ls'):: integer,
				'toleratedEarlyLeave', (em."data" ->> 'tel'):: integer,
				'toleratedEarlyStart', (em."data" ->> 'tes'):: integer,
				'toleratedLateLeave', (em."data" ->> 'tll'):: integer,
				'toleratedLateStart', (em."data" ->> 'tls'):: integer,
				'expectedOffTimePermissionDuration', (em."data" ->> 'eo'):: integer,
				'score', null,
				'accountedDeclaredWorkDuration', (em."data" ->> 'dw'):: integer,
				'accountedIgnoredWorkDuration', (em."data" ->> 'iw'):: integer,
				'usedFixedBreakDuration', (em."data" ->> 'ufb'):: float,
				'automaticallyUsedFixedBreakDuration', (em."data" ->> 'afb'):: float,
				'usedClaimBreakDuration', (em."data" ->> 'ucb'):: float,
				'automaticallyUsedClaimBreakDuration', (em."data" ->> 'acb'):: float,
				'expectedFixedBreakDuration', (em."data" ->> 'efb'):: float,
				'expectedClaimBreakDuration', (em."data" ->> 'ecb'):: float,
				'usedHolidayDuration', (em."data" ->> 'uh'):: integer,
				'expectedHolidayDuration', (em."data" ->> 'eh'):: integer,
				'physicallyInRegionDuration', (em."data" ->> 'pr'):: integer
				) as data
	FROM "${params.organizationId}"."monthlySummaryReports" msr
	INNER JOIN "${params.organizationId}"."employeeMonths" em ON em."monthlyReportId" = msr.id AND msr."organizationId" = ?
    INNER JOIN(${userFilter.query}) uf ON uf."userId" = em."userId" AND uf."organizationId" = msr."organizationId"
			${params.workPlanIds && params.workPlanIds.length > 0 ? `WHERE em.data->'w' @> ANY (ARRAY ['["${params.workPlanIds.join(`"]', '["`)}"]']::jsonb[])` : ``}
	ORDER BY uf.name DESC
	) fr ON fr.year = range.year AND fr.month = range.month
LEFT JOIN(
    SELECT * FROM
	(
		SELECT	extract(year from date) as year,
			extract(month from date) as month,
			json_build_object(
				'id', null,
				'userId', "userId",
				'name', name,
				'note', '',
				'isChecked', false,
				'year', extract(year from date),
				'month', extract(month from date),
				'expectedWorkDuration', SUM((data -> 's' ->> 'ew'):: integer),
				'missedWorkDuration', SUM((data -> 's' ->> 'm'):: integer),
				'normalWorkDuration', SUM((data -> 's' ->> 'n'):: integer),
				'extraWorkDuration', SUM((data -> 's' ->> 'e'):: integer),
				'accountedOffTimePermission', SUM((data -> 's' ->> 'ao'):: integer),
				'usedHoliday', SUM((data -> 's' ->> 'uh'):: integer),
				'weekdayExtraWorkDuration', SUM(CASE WHEN extract(isodow from date) < 6 THEN(data -> 's' ->> 'e'):: integer ELSE 0 END),
				'weekendExtraWorkDuration', SUM(CASE WHEN extract(isodow from date) > 5 THEN(data -> 's' ->> 'e'):: integer ELSE 0 END),
				'earlyLeave', SUM((data -> 's' ->> 'el'):: integer),
				'earlyStart', SUM((data -> 's' ->> 'es'):: integer),
				'lateLeave', SUM((data -> 's' ->> 'll'):: integer),
				'lateStart', SUM((data -> 's' ->> 'ls'):: integer),
				'toleratedEarlyLeave', SUM((data -> 's' ->> 'tel'):: integer),
				'toleratedEarlyStart', SUM((data -> 's' ->> 'tes'):: integer),
				'toleratedLateLeave', SUM((data -> 's' ->> 'tll'):: integer),
				'toleratedLateStart', SUM((data -> 's' ->> 'tls'):: integer),
				'expectedOffTimePermissionDuration', SUM((data -> 's' ->> 'eo'):: integer),
				'score', null,
				'accountedDeclaredWorkDuration', SUM((data -> 's' ->> 'dw'):: integer),
				'accountedIgnoredWorkDuration', SUM((data -> 's' ->> 'iw'):: integer),
				'usedFixedBreakDuration', SUM((data -> 's' ->> 'ufb'):: float),
				'automaticallyUsedFixedBreakDuration', SUM((data -> 's' ->> 'afb'):: float),
				'usedClaimBreakDuration', SUM((data -> 's' ->> 'ucb'):: float),
				'automaticallyUsedClaimBreakDuration', SUM((data -> 's' ->> 'acb'):: float),
				'expectedFixedBreakDuration', SUM((data -> 's' ->> 'efb'):: float),
				'expectedClaimBreakDuration', SUM((data -> 's' ->> 'ecb'):: float),
				'usedHolidayDuration', SUM((data -> 's' ->> 'uh'):: integer),
				'expectedHolidayDuration', SUM((data -> 's' ->> 'eh'):: integer),
				'physicallyInRegionDuration', SUM((data -> 's' ->> 'pr'):: integer),
				'ppermissionList', array_remove(array_agg(DISTINCT pp.list), NULL),
				'workPlanList', json_agg(DISTINCT d.list)
			) as data,
			sq.name
	FROM	(
        SELECT	ed.id, ed."userId", uf.name, ed.date, ed.data
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS "ed"
		INNER JOIN(${userFilter.query}) uf ON uf."userId" = ed."userId" AND uf."organizationId" = ed."organizationId" AND ed.date >= ? AND ed.date <= ?
        ${params.workPlanIds && params.workPlanIds.length > 0 ? `WHERE ed.data->'w' @> ANY (ARRAY ['["${params.workPlanIds.join(`"]', '["`)}"]']::jsonb[])` : ``}
                    UNION ALL
		SELECT	ed.id, ed."userId", uf.name, now(), ed.data
		FROM "${params.organizationId}"."employeeLiveDay" AS "ed"
		INNER JOIN(${userFilter.query}) uf ON uf."userId" = ed."userId" AND uf."organizationId" = ed."organizationId" AND now() >= ? AND now() <= ?
        ${params.workPlanIds && params.workPlanIds.length > 0 ? `WHERE ed.data->'w' @> ANY (ARRAY ['["${params.workPlanIds.join(`"]', '["`)}"]']::jsonb[])` : ``}
    ) sq
	CROSS JOIN LATERAL(
        SELECT string_agg(d.elem:: text, ', ') AS list
		FROM jsonb_array_elements_text(data -> 'w') AS d(elem)
    ) d
    LEFT JOIN LATERAL (
        select unnest(array_agg(DISTINCT ppermissions -> 'p'))  as list from jsonb_array_elements(data -> 'p') as x(ppermissions)
    ) pp ON TRUE
	GROUP BY "userId", "name", "year", "month" ) group_sum
	ORDER BY group_sum.name DESC
) sumq 
ON fr.id IS NULL AND sumq.year = range.year AND sumq.month = range.month
WHERE fr.data IS NOT NULL OR sumq.data IS NOT NULL
    `;
            let queryArgs = [
                params.startMonth.toDate(),
                params.endMonth.toDate(),
                params.organizationId,
                ...userFilter.bindingKeys,
                ...userFilter.bindingKeys,
                params.startMonth.toDate(),
                params.endMonth.toDate(),
                ...userFilter.bindingKeys,
                params.startMonth.toDate(),
                params.endMonth.toDate(),
            ];
            await trx.raw(this.dbClient.raw(`SELECT COUNT(*) as count FROM(${rawQuery}) subq`, queryArgs).toQuery()).then((countResult) => {
                result.pagination.total = parseInt(countResult.rows[0].count);
            });
            if (params.pagination.skip) {
                rawQuery += ` OFFSET ` + params.pagination.skip;
            }
            if (params.pagination.take) {
                rawQuery += ` LIMIT ` + params.pagination.take + " ";
            }
            let dbResult = await trx.raw(this.dbClient.raw(rawQuery, queryArgs).toQuery());
            let userIds = new Set();
            for (const row of dbResult.rows) {
                userIds.add(row.data.userId);
            }
            let userInfoList = await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(params.organizationId, Array.from(userIds));
            let userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(params.organizationId, Array.from(userIds));
            let rawMonthlyDate = await this.getSummaryOfEmployeesDateRange(params.organizationId, {
                start: params.startMonth.toDate(),
                end: params.endMonth.toDate(),
                userIds: Array.from(userIds),
            });
            let ids = [];
            for (let row of dbResult.rows) {
                if (row.data.id) {
                    ids.push(row.data.id);
                }
                let workPlanIdList = [];
                for (const workPlan of row.data.workPlanList) {
                    let seperated = workPlan.split(", ");
                    for (const id of seperated) {
                        if (!workPlanIdList.includes(id)) {
                            workPlanIdList.push(id);
                        }
                    }
                }
                let ppermissionList = [];
                if (row.data.ppermissionList && row.data.ppermissionList.length > 0) {
                    for (const ppermission of row.data.ppermissionList) {
                        let seperated = ppermission?.split(", ");
                        for (const id of seperated) {
                            if (!ppermissionList.includes(id)) {
                                ppermissionList.push(id);
                            }
                        }
                    }
                }
                let userInfo = userInfoList.find((u) => u.id == row.data.userId);
                let userCaption = userCaptionList.find((u) => u.id == row.data.userId);
                userInfo.captionLines = userCaption.captionLines;
                let userRawData = rawMonthlyDate.items.find((u) => u.user.id === row.data.userId);
                let workPlans = await this.getBasicInfoOfWorkPlans(params.organizationId, workPlanIdList);
                result.items.push({
                    user: userInfo,
                    date: moment({ year: row.year, month: row.month - 1 })
                        .startOf("month")
                        .toDate(),
                    workPlans: workPlans,
                    editedWorkingDurations: {
                        extraWorkDuration: parseInt(row.data.extraWorkDuration),
                        missedWorkDuration: parseInt(row.data.missedWorkDuration),
                        normalWorkDuration: parseInt(row.data.normalWorkDuration),
                    },
                    rawWorkingDurations: {
                        extraWorkDuration: userRawData?.rawWorkingDurations?.extraWorkDuration,
                        missedWorkDuration: userRawData?.rawWorkingDurations?.missedWorkDuration,
                        normalWorkDuration: userRawData?.rawWorkingDurations?.normalWorkDuration,
                    },
                    monthlyControlled: row.data.isChecked,
                    miscellaneous: {
                        expectedWorkDuration: parseInt(row.data.expectedWorkDuration),
                        earlyStart: parseInt(row.data.earlyStart),
                        lateStart: parseInt(row.data.lateStart),
                        earlyLeave: parseInt(row.data.earlyLeave),
                        lateLeave: parseInt(row.data.lateLeave),
                        toleratedEarlyStart: parseInt(row.data.toleratedEarlyStart),
                        toleratedLateStart: parseInt(row.data.toleratedLateStart),
                        toleratedEarlyLeave: parseInt(row.data.toleratedEarlyLeave),
                        toleratedLateLeave: parseInt(row.data.toleratedLateLeave),
                        usedFixedBreakDuration: parseInt(row.data.usedFixedBreakDuration),
                        automaticallyUsedFixedBreakDuration: parseInt(row.data.automaticallyUsedFixedBreakDuration),
                        expectedFixedBreakDuration: parseInt(row.data.expectedFixedBreakDuration),
                        usedClaimBreakDuration: parseInt(row.data.usedClaimBreakDuration),
                        automaticallyUsedClaimBreakDuration: parseInt(row.data.automaticallyUsedClaimBreakDuration),
                        expectedClaimBreakDuration: parseInt(row.data.expectedClaimBreakDuration),
                        usedHolidayDuration: parseInt(row.data.usedHolidayDuration),
                        expectedHolidayDuration: parseInt(row.data.expectedHolidayDuration),
                        accountedDeclaredWorkDuration: parseInt(row.data.accountedDeclaredWorkDuration),
                        physicallyInRegionDuration: parseInt(row.data.physicallyInRegionDuration),
                        accountedOffTimePermissionDuration: parseInt(row.data.accountedOffTimePermissionDuration),
                        expectedOffTimePermissionDuration: parseInt(row.data.expectedOffTimePermissionDuration),
                        accountedIgnoredWorkDuration: parseInt(row.data.accountedIgnoredWorkDuration),
                        score: parseInt(row.data.score),
                        usedPermissions: await this.getBasicInfoOfPermissions(params.organizationId, ppermissionList),
                        waitRecalculationFor: [],
                    },
                });
            }
            return result;
        });
    }
    async getSummaryOfEmployeesDateRange(organizationId, args) {
        let result = {
            pagination: {
                total: 0,
                take: args?.pagination?.take,
                skip: args?.pagination?.skip,
            },
            items: [],
        };
        let queryParams = [moment(args.start).format("YYYY-MM-DD"), moment(args.end).format("YYYY-MM-DD")];
        let query = `
        SELECT
        ed."userId",
        json_build_object(
			'ppermissionList', array_agg(ed.data -> 'p'),
			'workPlanIds', array_agg(ed.data -> 'w'),
            'expectedWorkDuration', SUM((ed.data -> 's' -> 'ew'):: integer),
            'normalWorkDuration', SUM((ed.data -> 's' -> 'n'):: integer),
            'extraWorkDuration', SUM((ed.data -> 's' -> 'e'):: integer),
            'missingWorkDuration', SUM((ed.data -> 's' -> 'm'):: integer),
            'earlyLeave', SUM((ed.data -> 's' -> 'el'):: integer),
            'earlyStart', SUM((ed.data -> 's' -> 'es'):: integer),
            'lateLeave', SUM((ed.data -> 's' -> 'll'):: integer),
            'lateStart', SUM((ed.data -> 's' -> 'ls'):: integer),
            'toleratedEarlyLeave', SUM((ed.data -> 's' -> 'tel'):: integer),
            'toleratedEarlyStart', SUM((ed.data -> 's' -> 'tes'):: integer),
            'toleratedLateLeave', SUM((ed.data -> 's' -> 'tll'):: integer),
            'toleratedLateStart', SUM((ed.data -> 's' -> 'tls'):: integer),
            'accountedOffTimePermissionDuration', SUM((ed.data -> 's' -> 'ao'):: integer),
            'expectedOffTimePermissionDuration', SUM((ed.data -> 's' -> 'eo'):: integer),
            'accountedDeclaredWorkDuration', SUM((ed.data -> 's' -> 'dw'):: integer),
            'accountedIgnoredWorkDuration', SUM((ed.data -> 's' -> 'iw'):: integer),
            'usedFixedBreakDuration', SUM((ed.data -> 's' -> 'ufb'):: float),
            'automaticallyUsedFixedBreakDuration', SUM((ed.data -> 's' -> 'afb'):: float),
            'expectedFixedBreakDuration', SUM((ed.data -> 's' -> 'efb'):: float),
            'usedClaimBreakDuration', SUM((ed.data -> 's' -> 'ucb'):: float),
            'automaticallyUsedClaimBreakDuration', SUM((ed.data -> 's' -> 'acb'):: float),
            'expectedClaimBreakDuration', SUM((ed.data -> 's' -> 'ecb'):: float),
            'usedHolidayDuration', SUM((ed.data -> 's' -> 'uh'):: integer),
            'expectedHolidayDuration', SUM((ed.data -> 's' -> 'eh'):: integer),
            'physicallyInRegionDuration', SUM((ed.data -> 's' -> 'pr'):: integer),
            'score', SUM(CASE WHEN ed.data->'s'->'s' is NULL THEN 0 ELSE (ed.data->'s'->>'s')::integer END) / SUM(1)
        ) as "calculatedData"
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS "ed"
                WHERE ed.date >= $1 AND ed.date <= $2`;
        if (args.userIds && args.userIds.length > 0) {
            queryParams.push(args.userIds);
            query += `
			AND ed."userId" = ANY ($${queryParams.length}::uuid[])`;
        }
        query += `
        GROUP BY "userId", "organizationId", extract(year from ed.date), extract(month from ed.date)
                ORDER BY "userId", "organizationId"   
        `;
        result.pagination.total += parseInt((await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q2", queryParams)).rows[0].count);
        if (args.pagination?.skip) {
            queryParams.push(args.pagination.skip);
            query += `
			OFFSET $${queryParams.length}`;
        }
        if (args.pagination?.take) {
            queryParams.push(args.pagination.take);
            query += `
			LIMIT $${queryParams.length}`;
        }
        const { rows } = await this._pgPool.query(query, queryParams);
        let userIds = new Set();
        for (const row of rows) {
            userIds.add(row.userId);
        }
        let userInfoList = await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(organizationId, Array.from(userIds));
        let userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, Array.from(userIds));
        let ids = [];
        for (let row of rows) {
            if (row.userId) {
                ids.push(row.userId);
            }
            let workPlanIdList = [];
            row.calculatedData.workPlanIds.forEach((elem) => workPlanIdList.push(...elem));
            const setOfWorkPlanIdList = workPlanIdList.filter((element, index) => workPlanIdList.indexOf(element) === index);
            let ppermissionList = [];
            row.calculatedData.ppermissionList.forEach((elem) => ppermissionList.push(...elem.map((perm) => perm.p)));
            const setOfPermissionList = ppermissionList.filter((element, index) => ppermissionList.indexOf(element) === index);
            let userInfo = userInfoList.find((u) => u.id == row.userId);
            let userCaption = userCaptionList.find((u) => u.id == row.userId);
            userInfo.captionLines = userCaption.captionLines;
            let workPlans = await this.getBasicInfoOfWorkPlans(organizationId, setOfWorkPlanIdList);
            result.items.push({
                user: userInfo,
                workPlans: workPlans,
                editedWorkingDurations: {
                    extraWorkDuration: parseInt(row.calculatedData.extraWorkDuration),
                    missedWorkDuration: parseInt(row.calculatedData.missingWorkDuration),
                    normalWorkDuration: parseInt(row.calculatedData.normalWorkDuration),
                },
                rawWorkingDurations: {
                    extraWorkDuration: parseInt(row.calculatedData.extraWorkDuration),
                    missedWorkDuration: parseInt(row.calculatedData.missingWorkDuration),
                    normalWorkDuration: parseInt(row.calculatedData.normalWorkDuration),
                },
                monthlyControlled: undefined,
                miscellaneous: {
                    expectedWorkDuration: parseInt(row.calculatedData.expectedWorkDuration),
                    earlyStart: parseInt(row.calculatedData.earlyStart),
                    lateStart: parseInt(row.calculatedData.lateStart),
                    earlyLeave: parseInt(row.calculatedData.earlyLeave),
                    lateLeave: parseInt(row.calculatedData.lateLeave),
                    toleratedEarlyStart: parseInt(row.calculatedData.toleratedEarlyStart),
                    toleratedLateStart: parseInt(row.calculatedData.toleratedLateStart),
                    toleratedEarlyLeave: parseInt(row.calculatedData.toleratedEarlyLeave),
                    toleratedLateLeave: parseInt(row.calculatedData.toleratedLateLeave),
                    usedFixedBreakDuration: parseInt(row.calculatedData.usedFixedBreakDuration),
                    automaticallyUsedFixedBreakDuration: parseInt(row.calculatedData.automaticallyUsedFixedBreakDuration),
                    expectedFixedBreakDuration: parseInt(row.calculatedData.expectedFixedBreakDuration),
                    usedClaimBreakDuration: parseInt(row.calculatedData.usedClaimBreakDuration),
                    automaticallyUsedClaimBreakDuration: parseInt(row.calculatedData.automaticallyUsedClaimBreakDuration),
                    expectedClaimBreakDuration: parseInt(row.calculatedData.expectedClaimBreakDuration),
                    usedHolidayDuration: parseInt(row.calculatedData.usedHolidayDuration),
                    expectedHolidayDuration: parseInt(row.calculatedData.expectedHolidayDuration),
                    accountedDeclaredWorkDuration: parseInt(row.calculatedData.accountedDeclaredWorkDuration),
                    physicallyInRegionDuration: parseInt(row.calculatedData.physicallyInRegionDuration),
                    accountedOffTimePermissionDuration: parseInt(row.calculatedData.accountedOffTimePermissionDuration),
                    expectedOffTimePermissionDuration: parseInt(row.calculatedData.expectedOffTimePermissionDuration),
                    accountedIgnoredWorkDuration: parseInt(row.calculatedData.accountedIgnoredWorkDuration),
                    score: parseInt(row.calculatedData.score),
                    usedPermissions: await this.getBasicInfoOfPermissions(organizationId, setOfPermissionList),
                    waitRecalculationFor: [],
                },
            });
        }
        return result;
    }
    async getEmployeeMonthlyReportHistory(organizationId, year, month, userId, pagination) {
        const date = moment({ year: year, month: month - 1, day: 1 }).format("YYYY-MM-DD");
        const { rows, rowCount } = await this._pgPool.query(`
            SELECT "id" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeMonths}"
            WHERE "organizationId" = $1
            AND "date" = $2
            AND "userId" = $3      
        `, [organizationId, date, userId]);
        if (!rowCount || rowCount < 1)
            return;
        return await dal_manager_1.dbManager.accessLogPacs.listMonthlyReportHistoryOp(organizationId, rows[0].id, pagination);
    }
    async listMonthlyExtraWorks(params) {
        return this.dbClient.transaction(async (trx) => {
            let result = [];
            let userFilter = this.getReportUserFilter({
                requesterUserId: params.requesterUserId,
                organizationId: params.organizationId,
                idBasedUserFilter: params.idBasedUserFilter,
                requiredOrganizationWidePermissions: params.requiredOrganizationWidePermissions,
                requiredOrganizationUnitWidePermissions: params.requiredOrganizationUnitWidePermissions,
            });
            let rawQuery = `
SELECT
"userId",
    name,
    date,
    (data -> 's' ->> 'e'):: integer AS "extraWorkDuration"
FROM(
    SELECT
                ed."userId", uf.name, ed.date, ed.data
            FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS "ed"
            INNER JOIN(${userFilter.query}) uf ON uf."userId" = ed."userId" AND uf."organizationId" = ed."organizationId"
            UNION ALL
            SELECT
                ed."userId", uf.name, now(), ed.data
            FROM "${params.organizationId}"."employeeLiveDay" AS "ed"
            INNER JOIN(${userFilter.query}) uf ON uf."userId" = ed."userId" AND uf."organizationId" = ed."organizationId"
) sq
CROSS JOIN LATERAL(
    SELECT string_agg(d.elem:: text, ', ') AS list
                FROM jsonb_array_elements_text(data -> 'w') AS d(elem)
) d
WHERE date >= ? AND date < ?
    ${params.workPlanIds && params.workPlanIds.length > 0 ? `AND data->'w' @> ANY (ARRAY ['["${params.workPlanIds.join(`"]', '["`)}"]']::jsonb[])` : ``}
ORDER BY "name" ASC, date DESC
    `;
            let queryArgs = [...userFilter.bindingKeys];
            queryArgs.push(...userFilter.bindingKeys);
            queryArgs.push(...[params.startMonth, params.endMonth]);
            let dbResult = await trx.raw(this.dbClient.raw(rawQuery, queryArgs).toQuery());
            let userInfoList = await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(params.organizationId, dbResult.rows.map((u) => u.userId));
            for (const row of dbResult.rows) {
                let resultItem = result.find((ri) => ri.user.id === row.userId);
                if (!resultItem) {
                    resultItem = {
                        user: userInfoList.find((u) => u.id == row.userId),
                        extraWorkList: [],
                    };
                    result.push(resultItem);
                }
                resultItem.extraWorkList.push({
                    date: row.date,
                    extraWorkDuration: parseInt(row.extraWorkDuration),
                    eliminatedDuration: 0,
                });
            }
            return result;
        });
    }
    async updateMonthlyReportOfUser(organizationId, year, month, userId, args) {
        await this.dbClient.transaction(async (trx) => {
            let actionUtc = new Date();
            let summary = (await trx.raw(`
SELECT em.* FROM
"${organizationId}"."monthlySummaryReports" msr
INNER JOIN "${organizationId}"."employeeMonths" em ON em."monthlyReportId" = msr.id
WHERE
msr."organizationId" = ? AND msr.year = ? AND msr.month = ? AND em."userId" = ?
    `, [organizationId, year, month, userId])).rows[0];
            if (!summary) {
                return Promise.resolve(false);
            }
            if (args.missedWorkDuration !== summary.data.m ||
                args.extraWorkDuration !== summary.data.e ||
                args.normalWorkDuration !== summary.data.n ||
                args.note !== summary.note ||
                args.checked !== summary.isChecked) {
                summary.updatedAt = actionUtc;
                summary.updatedByUserId = args.updateUserId;
                summary.note = args.note ? args.note : null;
                let oldData = summary.data;
                oldData.m = args.missedWorkDuration ? args.missedWorkDuration : oldData.m;
                oldData.e = args.extraWorkDuration ? args.extraWorkDuration : oldData.e;
                oldData.n = args.normalWorkDuration ? args.normalWorkDuration : oldData.n;
                await trx.raw(`
                UPDATE "${organizationId}"."employeeMonths" SET(data, "isChecked", note, "updatedAt", "updatedByUserId") = (?, ?, ?, ?, ?)
WHERE id = ?
    `, [oldData, args.checked ? args.checked : summary.isChecked, args.note ? args.note : summary.note, actionUtc, args.updateUserId, summary.id]);
                await trx.raw(`
                UPDATE "${organizationId}"."monthlySummaryReports" SET("isFinalized", note, "updatedDate", "updatedById") = (?, ?, ?, ?)
WHERE id = ?
    `, [false, args.note ? args.note : "", actionUtc, args.updateUserId, summary.monthlyReportId]);
                let monthlyReportData = (await trx.raw(`
                SELECT * FROM "${organizationId}"."monthlySummaryReports" WHERE id = ?
    `, summary.monthlyReportId)).rows[0];
                await trx.raw(`
                    INSERT INTO "${organizationId}"."user_monthly_summary_history"(id, "organizationId", "actionUtc", "originalId", "userId", log) VALUES(?, ?, ?, ?, ?, ?)
    `, [uuid.v4(), organizationId, actionUtc, summary.id, summary.userId, summary]);
                await trx.raw(`
INSERT INTO "${organizationId}"."monthly_summary_history"(id, "organizationId", "actionUtc", "originalId", log) VALUES(?, ?, ?, ?, ?)
    `, [uuid.v4(), organizationId, actionUtc, monthlyReportData.id, monthlyReportData]);
            }
        });
        return Promise.resolve(true);
    }
    async getUnactedMonthlyReportDate(organizationId) {
        let unactedReportDates = (await this.dbClient.raw(`
			SELECT mr2.* FROM
			(
				SELECT extract('year' FROM mr.month) as year, extract('month' FROM mr.month) as month FROM
				(
					SELECT date_trunc('month', generate_series(om."enableUtc", COALESCE(om."disableUtc", date_trunc('month', now())), '1 month'::interval)) as month
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules}" om WHERE module = 3
				) mr
				WHERE mr.month <> date_trunc('month', now())
			) mr2
			LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.monthlySummaryReports}" msr
				ON msr.year = mr2.year AND msr.month = mr2.month
			WHERE msr.id IS NULL
			ORDER BY year ASC, month ASC
			`)).rows;
        return Promise.resolve(unactedReportDates);
    }
    async generateMonthlyReports(organizationId, year, month) {
        return this.dbClient.transaction(async (trx) => {
            let actionUtc = new Date();
            let insertedMonthlyReports = await trx.raw(`
                INSERT INTO "${organizationId}"."monthlySummaryReports"
                    ("id", "organizationId", "runDateTime", "isFinalized", "year", "month")
                SELECT
                uuid_generate_v4(), "organizationId", NULL as "runDateTime", false as "isFinalized", ? as year, ? as month
                FROM
                    (SELECT org.id as "organizationId", msr.id as "reportId", COUNT(ed.id) as "empDayCount"
                            FROM "${organizationId}"."organizations" org
                            LEFT JOIN "${organizationId}"."monthlySummaryReports" msr ON msr."organizationId" = org.id AND msr.year =? AND msr.month =?
                        LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" ed ON ed."organizationId" = org.id AND extract(year from ed."date") =? AND extract(month from ed."date") =?
                            WHERE msr.id IS NULL AND org."deletedAt" IS NULL
                            GROUP BY org.id, msr.id) subquery
                WHERE "empDayCount" > 0
                RETURNING *
            `, [year, month, year, month, year, month]);
            let insertedMonthlyUserSummaries;
            let hook = business_hooks_1.armonHookManager.getMonthlyReportHook(organizationId);
            if (hook) {
                await hook(organizationId, year, month - 1, trx);
            }
            else {
                insertedMonthlyUserSummaries = await trx.raw(`
                INSERT INTO "${organizationId}"."employeeMonths"
                    (id, "date", "userId", "organizationId", "updatedAt", "isChecked", "data", "monthlyReportId")
                SELECT subq.*, msr.id as "monthlyReportId" FROM
                    (
                        SELECT
                                uuid_generate_v4() as id,
                        format('%s-%s-1', extract(year from ed.date), extract(month from ed.date)):: date as date,
                        ed."userId", ed."organizationId",
                        now() as "createdAt",
                        false as "isChecked",
                        json_build_object(
                            'p', array_remove(array_agg(DISTINCT pp.list), NULL),
                            'w', array_to_json(array_agg(DISTINCT d.list)),
                            'ew', SUM((ed.data -> 's' -> 'ew'):: integer),
                            'n', SUM((ed.data -> 's' -> 'n'):: integer),
                            'e', SUM((ed.data -> 's' -> 'e'):: integer),
                            'm', SUM((ed.data -> 's' -> 'm'):: integer),
                            'el', SUM((ed.data -> 's' -> 'el'):: integer),
                            'es', SUM((ed.data -> 's' -> 'es'):: integer),
                            'll', SUM((ed.data -> 's' -> 'll'):: integer),
                            'ls', SUM((ed.data -> 's' -> 'ls'):: integer),
                            'tel', SUM((ed.data -> 's' -> 'tel'):: integer),
                            'tes', SUM((ed.data -> 's' -> 'tes'):: integer),
                            'tll', SUM((ed.data -> 's' -> 'tll'):: integer),
                            'tls', SUM((ed.data -> 's' -> 'tls'):: integer),
                            'ao', SUM((ed.data -> 's' -> 'ao'):: integer),
                            'eo', SUM((ed.data -> 's' -> 'eo'):: integer),
                            'dw', SUM((ed.data -> 's' -> 'dw'):: integer),
                            'iw', SUM((ed.data -> 's' -> 'iw'):: integer),
                            'ufb', SUM((ed.data -> 's' -> 'ufb'):: float),
                            'afb', SUM((ed.data -> 's' -> 'afb'):: float),
                            'efb', SUM((ed.data -> 's' -> 'efb'):: float),
                            'ucb', SUM((ed.data -> 's' -> 'ucb'):: float),
                            'acb', SUM((ed.data -> 's' -> 'acb'):: float),
                            'ecb', SUM((ed.data -> 's' -> 'ecb'):: float),
                            'uh', SUM((ed.data -> 's' -> 'uh'):: integer),
                            'eh', SUM((ed.data -> 's' -> 'eh'):: integer),
                            'pr', SUM((ed.data -> 's' -> 'pr'):: integer),
                            's', SUM(CASE WHEN ed.data->'s'->'s' is NULL THEN 0 ELSE (ed.data->'s'->>'s')::integer END) / SUM(1)
                        ) as "calculatedData"
                                FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS "ed"
                        CROSS JOIN LATERAL(
                            SELECT string_agg(d.elem:: text, ', ') AS list
                                    FROM jsonb_array_elements_text(ed.data -> 'w') AS d(elem)
                        ) d
                        LEFT JOIN LATERAL (
                            select unnest(array_agg(DISTINCT ppermissions -> 'p'))  as list from jsonb_array_elements(ed.data -> 'p') as x(ppermissions)
                        ) pp ON TRUE
                                WHERE extract(year from ed.date) = ? AND extract(month from ed.date) = ?
                        GROUP BY "userId", "organizationId", extract(year from ed.date), extract(month from ed.date)
                                ORDER BY "userId", "organizationId", "date"
                    ) subq
                INNER JOIN "${organizationId}"."monthlySummaryReports" msr ON msr."organizationId" = subq."organizationId"
                AND msr.year = extract(year from subq.date) AND msr.month = extract(month from subq.date)
                RETURNING *
                    `, [year, month]);
            }
            if (insertedMonthlyReports?.rows && insertedMonthlyReports.rows.length > 0) {
                let monthlyBindings = [];
                let monthlyQuery = `
INSERT INTO "${organizationId}"."monthly_summary_history"("id", "organizationId", "actionUtc", "originalId", "log") VALUES
    ` +
                    insertedMonthlyReports.rows
                        .map((summary) => {
                        monthlyBindings.push(...[uuid.v4(), summary.organizationId, actionUtc, summary.id, summary]);
                        return `(?,?,?,?,?)`;
                    })
                        .join(`, `);
                await trx.raw(monthlyQuery, monthlyBindings);
            }
            if (insertedMonthlyUserSummaries?.rows && insertedMonthlyUserSummaries.rows.length > 0) {
                let userMonthlyBindings = [];
                let userMonthlyQuery = `
INSERT INTO "${organizationId}"."user_monthly_summary_history"("id", "organizationId", "actionUtc", "originalId", "userId", "log") VALUES
    ` +
                    insertedMonthlyUserSummaries.rows
                        .map((summary) => {
                        userMonthlyBindings.push(...[uuid.v4(), summary.organizationId, actionUtc, summary.id, summary.userId, summary]);
                        return `(?,?,?,?,?,?)`;
                    })
                        .join(`, `);
                await trx.raw(userMonthlyQuery, userMonthlyBindings);
            }
        });
    }
    async createOrInvalidateMonthlyReport(organizationId, date) {
        return this.dbClient.transaction(async (trx) => {
            let existingReport = (await trx.raw(`
SELECT * FROM "${organizationId}"."monthlySummaryReports" msr WHERE msr."organizationId" = ? AND msr.year = ? AND msr.month = ?
    `, [organizationId, date.year(), date.month() + 1])).rows[0];
            if (!existingReport) {
                existingReport = (await trx.raw(`
                INSERT INTO "${organizationId}"."monthlySummaryReports"
    ("id", "organizationId", "runDateTime", "isFinalized", "year", "month") VALUES(?, ?, NULL, false, ?, ?) RETURNING * `, [uuid.v4(), organizationId, date.year(), date.month() + 1])).rows[0];
            }
            else {
                existingReport = (await trx.raw(`
UPDATE "${organizationId}"."monthlySummaryReports" SET("isFinalized", note, "updatedDate", "updatedById") = (false, NULL, now(), NULL:: uuid)
WHERE id = ? RETURNING *
    `, [existingReport.id])).rows[0];
            }
            await trx.raw(`
                INSERT INTO "${organizationId}"."monthly_summary_history"("id", "organizationId", "actionUtc", "originalId", "log") VALUES(?,?, now(),?,?)`, [uuid.v4(), organizationId, existingReport.id, existingReport]);
        });
    }
    async upsertUserMonthlyReport(organizationId, userId, date) {
        return this.dbClient.transaction(async (trx) => {
            let existingReport = (await trx.raw(`
                SELECT em.* FROM
                "${organizationId}"."monthlySummaryReports" msr
                INNER JOIN "${organizationId}"."employeeMonths" em ON em."monthlyReportId" = msr.id
                WHERE
                msr."organizationId" = ? AND msr.year = ? AND msr.month = ? AND em."userId" = ?
            `, [organizationId, date.year(), date.month() + 1, userId])).rows[0];
            const startDate = date.clone().startOf("month");
            const endDate = date.clone().endOf("month");
            const hook = business_hooks_1.armonHookManager.getMonthlyReportSingleUserHook(organizationId);
            if (existingReport) {
                if (hook) {
                    existingReport = await hook(organizationId, startDate, endDate, trx, { userId: userId, insertToDb: true, update: true, existingReportId: existingReport.id });
                }
                else {
                    existingReport = (await trx.raw(`
                    UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeMonths}" 
						SET (data, "isChecked", note, "updatedAt", "updatedByUserId") = (sq2."calculatedData", sq2."isChecked", sq2."note", sq2."updatedAt", sq2."updatedByUserId")
					FROM (
						SELECT "calculatedData", false AS "isChecked", NULL AS note, now() AS "updatedAt", NULL::UUID AS "updatedByUserId" FROM
						(
							WITH base_data AS (
								SELECT
									EXTRACT(YEAR FROM ed.date) AS year,
									EXTRACT(MONTH FROM ed.date) AS month,
									ed."userId",
									ed."organizationId",
									ed.data
								FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays} ed
								WHERE EXTRACT(YEAR FROM ed.date) = ?
									AND EXTRACT(MONTH FROM ed.date) = ?
									AND ed."userId" = ?
									AND ed."organizationId" = ?
							),
							permission_list AS (
								SELECT
									"userId",
									"organizationId",
									year,
									month,
									ARRAY_REMOVE(ARRAY_AGG(DISTINCT p.value ->> 'p'), NULL) AS p_list
								FROM base_data
								LEFT JOIN LATERAL JSONB_ARRAY_ELEMENTS(data -> 'p') AS p(value) ON TRUE
								GROUP BY "userId", "organizationId", year, month
							),
							work_list AS (
								SELECT
									"userId",
									"organizationId",
									year,
									month,
									ARRAY_AGG(DISTINCT w.value::TEXT) AS w_list
								FROM base_data
								LEFT JOIN LATERAL JSONB_ARRAY_ELEMENTS_TEXT(data -> 'w') AS w(value) ON TRUE
								GROUP BY "userId", "organizationId", year, month
							)
							SELECT
								UUID_GENERATE_V4() AS id,
								MAKE_DATE(bd.year::INT, bd.month::INT, 1) AS date,
								bd."userId",
								bd."organizationId",
								JSON_BUILD_OBJECT(
									'p',  pl.p_list,
									'w',  wl.w_list,
									'ew',  SUM((bd.data -> 's' ->> 'ew')::INT),
									'n',   SUM((bd.data -> 's' ->> 'n')::INT),
									'e',   SUM((bd.data -> 's' ->> 'e')::INT),
									'm',   SUM((bd.data -> 's' ->> 'm')::INT),
									'el',  SUM((bd.data -> 's' ->> 'el')::INT),
									'es',  SUM((bd.data -> 's' ->> 'es')::INT),
									'll',  SUM((bd.data -> 's' ->> 'll')::INT),
									'ls',  SUM((bd.data -> 's' ->> 'ls')::INT),
									'tel', SUM((bd.data -> 's' ->> 'tel')::INT),
									'tes', SUM((bd.data -> 's' ->> 'tes')::INT),
									'tll', SUM((bd.data -> 's' ->> 'tll')::INT),
									'tls', SUM((bd.data -> 's' ->> 'tls')::INT),
									'ao',  SUM((bd.data -> 's' ->> 'ao')::INT),
									'eo',  SUM((bd.data -> 's' ->> 'eo')::INT),
									'dw',  SUM((bd.data -> 's' ->> 'dw')::INT),
									'iw',  SUM((bd.data -> 's' ->> 'iw')::INT),
									'ufb', SUM((bd.data -> 's' ->> 'ufb')::FLOAT),
									'afb', SUM((bd.data -> 's' ->> 'afb')::FLOAT),
									'efb', SUM((bd.data -> 's' ->> 'efb')::FLOAT),
									'ucb', SUM((bd.data -> 's' ->> 'ucb')::FLOAT),
									'acb', SUM((bd.data -> 's' ->> 'acb')::FLOAT),
									'ecb', SUM((bd.data -> 's' ->> 'ecb')::FLOAT),
									'uh',  SUM((bd.data -> 's' ->> 'uh')::INT),
									'eh',  SUM((bd.data -> 's' ->> 'eh')::INT),
									'pr',  SUM((bd.data -> 's' ->> 'pr')::INT),
									's',   SUM(COALESCE((bd.data -> 's' ->> 's')::INT, 0)) / COUNT(*)
								) AS "calculatedData"
							FROM base_data bd
							JOIN permission_list pl ON pl."userId" = bd."userId" AND pl."organizationId" = bd."organizationId" AND pl.year = bd.year AND pl.month = bd.month
							JOIN work_list wl ON wl."userId" = bd."userId" AND wl."organizationId" = bd."organizationId" AND wl.year = bd.year AND wl.month = bd.month
							GROUP BY bd."userId", bd."organizationId", bd.year, bd.month, pl.p_list, wl.w_list
							ORDER BY bd."userId", bd."organizationId", date
						) subq
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.monthlySummaryReports}" msr ON msr."organizationId" = subq."organizationId"
						AND msr.year = extract(year from subq.date) AND msr.month = extract(month from subq.date)
                    )sq2  WHERE id = ? RETURNING *; `, [date.year(), date.month() + 1, userId, organizationId, existingReport.id])).rows[0];
                }
            }
            else {
                if (hook) {
                    existingReport = await hook(organizationId, startDate, endDate, trx, { userId: userId, insertToDb: true });
                }
                else {
                    existingReport = (await trx.raw(`
                    INSERT INTO "${organizationId}"."employeeMonths"
                        (id, "date", "userId", "organizationId", "updatedAt", "isChecked", "data", "monthlyReportId")
                    SELECT subq.*, msr.id as "monthlyReportId" FROM
					(
						WITH base_data AS (
							SELECT
								EXTRACT(YEAR FROM ed.date) AS year,
								EXTRACT(MONTH FROM ed.date) AS month,
								ed."userId",
								ed."organizationId",
								ed.data
							FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays} ed
							WHERE EXTRACT(YEAR FROM ed.date) = ?
								AND EXTRACT(MONTH FROM ed.date) = ?
								AND ed."userId" = ?
								AND ed."organizationId" = ?
						),
						permission_list AS (
							SELECT
								"userId",
								"organizationId",
								year,
								month,
								ARRAY_REMOVE(ARRAY_AGG(DISTINCT p.value ->> 'p'), NULL) AS p_list
							FROM base_data
							LEFT JOIN LATERAL JSONB_ARRAY_ELEMENTS(data -> 'p') AS p(value) ON TRUE
							GROUP BY "userId", "organizationId", year, month
						),
						work_list AS (
							SELECT
								"userId",
								"organizationId",
								year,
								month,
								ARRAY_AGG(DISTINCT w.value::TEXT) AS w_list
							FROM base_data
							LEFT JOIN LATERAL JSONB_ARRAY_ELEMENTS_TEXT(data -> 'w') AS w(value) ON TRUE
							GROUP BY "userId", "organizationId", year, month
						)
						SELECT
							UUID_GENERATE_V4() AS id,
							MAKE_DATE(bd.year::INT, bd.month::INT, 1) AS date,
							bd."userId",
							bd."organizationId",
							now() as "createdAt",
                            false as "isChecked",
							JSON_BUILD_OBJECT(
								'p',  pl.p_list,
								'w',  wl.w_list,
								'ew',  SUM((bd.data -> 's' ->> 'ew')::INT),
								'n',   SUM((bd.data -> 's' ->> 'n')::INT),
								'e',   SUM((bd.data -> 's' ->> 'e')::INT),
								'm',   SUM((bd.data -> 's' ->> 'm')::INT),
								'el',  SUM((bd.data -> 's' ->> 'el')::INT),
								'es',  SUM((bd.data -> 's' ->> 'es')::INT),
								'll',  SUM((bd.data -> 's' ->> 'll')::INT),
								'ls',  SUM((bd.data -> 's' ->> 'ls')::INT),
								'tel', SUM((bd.data -> 's' ->> 'tel')::INT),
								'tes', SUM((bd.data -> 's' ->> 'tes')::INT),
								'tll', SUM((bd.data -> 's' ->> 'tll')::INT),
								'tls', SUM((bd.data -> 's' ->> 'tls')::INT),
								'ao',  SUM((bd.data -> 's' ->> 'ao')::INT),
								'eo',  SUM((bd.data -> 's' ->> 'eo')::INT),
								'dw',  SUM((bd.data -> 's' ->> 'dw')::INT),
								'iw',  SUM((bd.data -> 's' ->> 'iw')::INT),
								'ufb', SUM((bd.data -> 's' ->> 'ufb')::FLOAT),
								'afb', SUM((bd.data -> 's' ->> 'afb')::FLOAT),
								'efb', SUM((bd.data -> 's' ->> 'efb')::FLOAT),
								'ucb', SUM((bd.data -> 's' ->> 'ucb')::FLOAT),
								'acb', SUM((bd.data -> 's' ->> 'acb')::FLOAT),
								'ecb', SUM((bd.data -> 's' ->> 'ecb')::FLOAT),
								'uh',  SUM((bd.data -> 's' ->> 'uh')::INT),
								'eh',  SUM((bd.data -> 's' ->> 'eh')::INT),
								'pr',  SUM((bd.data -> 's' ->> 'pr')::INT),
								's',   SUM(COALESCE((bd.data -> 's' ->> 's')::INT, 0)) / COUNT(*)
							) AS "calculatedData"
						FROM base_data bd
						JOIN permission_list pl ON pl."userId" = bd."userId" AND pl."organizationId" = bd."organizationId" AND pl.year = bd.year AND pl.month = bd.month
						JOIN work_list wl ON wl."userId" = bd."userId" AND wl."organizationId" = bd."organizationId" AND wl.year = bd.year AND wl.month = bd.month
						GROUP BY bd."userId", bd."organizationId", bd.year, bd.month, pl.p_list, wl.w_list
						ORDER BY bd."userId", bd."organizationId", date
					) subq
                    INNER JOIN "${organizationId}"."monthlySummaryReports" msr ON msr."organizationId" = subq."organizationId"
                    AND msr.year = extract(year from subq.date) AND msr.month = extract(month from subq.date)
                    RETURNING * `, [date.year(), date.month() + 1, userId, organizationId])).rows[0];
                }
            }
            if (existingReport.id) {
                await trx.raw(`INSERT INTO "${organizationId}"."user_monthly_summary_history"
					("id", "organizationId", "actionUtc", "originalId", "userId", "log")
					VALUES(?,?,?,?,?,?)`, [uuid.v4(), organizationId, existingReport.updatedAt, existingReport.id, userId, existingReport]);
            }
        });
    }
    async customMonthlyPerformanceReportForGubretas(options) {
        if (moment(options.dateRange.endDateTime).isAfter(moment())) {
            options.dateRange.endDateTime = moment().toDate();
        }
        const client = await this._pgPool.connect();
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: options.organizationId,
            requesterUserId: options.requesterUserId,
            idBasedUserFilter: {
                userIds: options.userIds,
                userGroupIds: options.userGroupIds,
                organizationUnitIds: options.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: options.applyOrganizationHierarchically,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
            },
            requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            bindingKeys: [],
            specificSelectItems: ["userId", "name", "surname", "uniqueId"],
        });
        let paramIndex = userFilter.bindingKeys.length;
        let reportQuery = `
            SELECT UFQ."uniqueId", UFQ."name", UFQ."surname" AS "surName", json_agg(json_build_object(
                'date', ed."date",
                'employeeDay', ed."data"
            ) ORDER BY ed."date" ASC) AS "monthlyData"
            FROM "${options.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS ed
            INNER JOIN(
                ${userFilter.query}
            ) UFQ ON UFQ."userId" = ed."userId" AND ed."organizationId" = $${++paramIndex}
            WHERE ed."date" >= $${++paramIndex} AND ed."date" <= $${++paramIndex}
            ${options.workPlanIds && options.workPlanIds.length > 0 ? `AND ed.data->'w' ?| array['` + options.workPlanIds.join(`','`) + `']` : ""}
            GROUP BY ed."userId", UFQ."uniqueId", UFQ."name", UFQ."surname" ORDER BY UFQ."uniqueId" ASC
        `;
        userFilter.bindingKeys.push(...[options.organizationId, options.dateRange.startDateTime, options.dateRange.endDateTime]);
        const cursor = client.query(new Cursor(reportQuery, userFilter.bindingKeys, {
            rowMode: "array",
        }));
        let rows = [];
        while (true) {
            rows = await new Promise((resolve, reject) => {
                cursor.read(100, (err, rows) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve(rows);
                    }
                });
            });
            await options.onData(rows);
            if (rows.length < 100) {
                break;
            }
        }
        await new Promise((resolve, reject) => {
            cursor.close((err) => {
                if (err) {
                    reject(err);
                }
                else {
                    resolve();
                }
            });
        });
        client.release();
    }
    async getPermissionPermissionTypeIdMapping(organizationId, permissionIds) {
        return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let boundQuery = `
                SELECT "id", "ppermissionTypeId" FROM "${organizationId}"."ppermissions"
                WHERE "organizationId" = $1 AND "id" = ANY($2:: UUID[])
            `;
            let dbResult = await trx.query(boundQuery, [organizationId, permissionIds]);
            return dbResult.rows;
        });
    }
    async getOrganizationUnitsWithParentInfo(organizationId, organizationUnitIds) {
        return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let boundQuery = `
                SELECT ou."id", ou."name", (SELECT "name" FROM "${organizationId}"."organizationUnits" WHERE "id" = ou."parentId") AS "parentName" FROM "${organizationId}"."organizationUnits" as ou
                WHERE ou."id" = ANY($1:: UUID[]) AND ou."organizationId" = $2
            `;
            let dbResult = await trx.query(boundQuery, [organizationUnitIds, organizationId]);
            return dbResult.rows;
        });
    }
    async getWorksForUsers(organizationId, requesterUserId, filter) {
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: organizationId,
            requesterUserId: requesterUserId,
            idBasedUserFilter: {
                userIds: filter.userIds,
                userGroupIds: filter.userGroupIds,
                organizationUnitIds: filter.organizationUnitIds,
                applyOrganizationUnitFilterHierarchically: filter.filterOrganizationUnitMembersHierarchically,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
            },
            requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            bindingKeys: [],
            specificSelectItems: ["userId", "organizationId"],
        });
        if (!filter.type) {
            return;
        }
        const selectionData = filter.type === app_enums_1.enums.EmployeeBinaryDistributionType.MissingWork ? "m" : "e";
        let bindIndex = userFilter.bindingKeys.length;
        let rawQuery = `
            SELECT sq."userId", sq."organizationId", rwq.reason as "waitRecalculationFor", SUM((sq.data -> 's' -> '${selectionData}'):: integer) as "workDuration", COUNT(*) as "workDays", NULL as "ppermissionIds"
            FROM
                (
                    SELECT
                            ed."userId", ed."organizationId", ed.date, ed.data
                        FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" AS "ed"
                        INNER JOIN(${userFilter.query}) uf ON uf."userId" = ed."userId" AND uf."organizationId" = ed."organizationId"
                        WHERE $${++bindIndex} <= ed.date AND ed.date <= $${++bindIndex}
                    UNION ALL
                        SELECT
                            ed."userId", ed."organizationId", now(), ed.data
                        FROM "${organizationId}"."employeeLiveDay" AS "ed"
                        INNER JOIN(${userFilter.query}) uf ON uf."userId" = ed."userId" AND uf."organizationId" = ed."organizationId"
                        WHERE $${++bindIndex} <= now() AND now() <= $${++bindIndex}
            ) sq
            LEFT OUTER JOIN "${organizationId}"."recalculateWorkQueue" rwq ON rwq."userId" = sq."userId" AND rwq."organizationId" = sq."organizationId" AND rwq."startDate" <= $${++bindIndex} 
        `;
        switch (filter.type) {
            case app_enums_1.enums.EmployeeBinaryDistributionType.MissingWork:
                rawQuery += ` WHERE (sq.data -> 's' ->> 'ew'):: integer > 0 AND(sq.data -> 's' -> 'm'):: integer > 0 `;
                break;
            case app_enums_1.enums.EmployeeBinaryDistributionType.ExtraWork:
                rawQuery += ` WHERE (sq.data -> 's' -> 'e'):: integer > 0 `;
                break;
            default:
                break;
        }
        rawQuery += ` 
            ${filter.workPlanIds && filter.workPlanIds.length > 0 ? `AND sq.data->'w' @> ANY (ARRAY ['["${filter.workPlanIds.join(`"]', '["`)}"]']::jsonb[])` : ``}
            GROUP BY sq."userId", sq."organizationId", rwq.reason
            ORDER BY "workDuration" DESC, "workDays" DESC
        `;
        let dateRange = [];
        if (filter.range) {
            dateRange.push(filter.range.startDateTime, filter.range.endDateTime);
        }
        else {
            try {
                let startDate = await this.getPacsActivedDate(organizationId);
                dateRange.push(startDate, new Date());
            }
            catch {
                let oneMonthRangeStart = moment().subtract(1, "month");
                dateRange.push(oneMonthRangeStart, new Date());
            }
        }
        let queryArgs = userFilter.bindingKeys;
        queryArgs.push(...dateRange, ...dateRange, dateRange[1]);
        const count = (await this._pgPool.query("SELECT COUNT(*) FROM (" + rawQuery + ")q2", queryArgs)).rows[0].count;
        if (filter.pagination.skip) {
            rawQuery += `
                OFFSET $${++bindIndex}
            `;
            queryArgs.push(filter.pagination.skip);
        }
        if (filter.pagination.take) {
            rawQuery += `
                LIMIT $${++bindIndex}
            `;
            queryArgs.push(filter.pagination.take);
        }
        let dbResult = await this.pgPool.query(rawQuery, queryArgs);
        const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, dbResult.rows.map((r) => r.userId));
        return {
            items: dbResult.rows.map((row) => {
                return {
                    id: row.userId,
                    userCaptions: userCaptionList.find((uc) => uc.id === row.userId)?.captionLines,
                    workDays: parseInt(row.workDays),
                    workDuration: parseInt(row.workDuration),
                    waitRecalculationFor: this.extractIndividualRecalculateReasons(row.waitRecalculationFor),
                    warnings: {
                        pendingPermissionIds: row.ppermissionIds ? row.ppermissionIds : [],
                    },
                };
            }),
            pagination: {
                total: parseInt(count),
                skip: filter.pagination.skip,
                take: filter.pagination.take,
            },
        };
    }
    async getPacsActivedDate(organizationId) {
        const { rows, rowCount } = await this._pgPool.query(`
            SELECT * FROM "${organizationId}"."organizationModules"
            WHERE "organizationId" = $1
            AND module = 3
            AND status = 1
            AND "disableUtc" IS NULL
            ORDER BY "enableUtc" desc
        `, [organizationId]);
        if (rowCount === 0) {
            throw "Pacs Module is disable";
        }
        return rows[0].enableUtc;
    }
    async getUserWorkSegments(organizationId, userId, type, pagination, dateTimeRange) {
        if (!type) {
            return;
        }
        const selectionData = type === app_enums_1.enums.EmployeeBinaryDistributionType.MissingWork ? "m" : "e";
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
        SELECT day, "work", segments, rwq."reason",
            (
                SELECT array_agg(pp.id)
                FROM "${organizationId}"."userPPermissions" upp
                INNER JOIN "${organizationId}"."ppermissions" pp ON pp.id = upp."ppermissionId" AND pp."organizationId" = q1."organizationId" AND pp.status = $${queryParamIndex++}
                INNER JOIN "${organizationId}"."ppermissionTypes" ppt ON ppt.id = pp."ppermissionTypeId" AND ppt."deletedAt" IS NULL
                AND tstzrange(pp."startDateTime", pp."endDateTime") && tstzrange((segments -> 'ws' -> (0) ->> 's'):: timestamp with time zone, (segments -> 'ws' -> (jsonb_array_length(Segments -> 'ws') - 1) ->> 'e'):: timestamp with time zone)
                WHERE upp."userId" = q1."userId"
                GROUP BY upp."userId"
            ) as "ppermissionIds",
            (segments -> 'ar' -> (jsonb_array_length(segments -> 'ar') - 1) -> 'a' -> (jsonb_array_length(segments -> 'ar' -> (jsonb_array_length(segments -> 'ar') - 1) -> 'a') - 1) ->> 'e'):: timestamp with time zone as "lastLeave",
            (segments -> 'ar' -> 0 -> 'a' -> 0 ->> 's'):: timestamp with time zone as "firstStart"
        FROM (
            (
                SELECT ed."userId", ed."organizationId", date as day, (data -> 's' ->> '${selectionData}'):: int "work", data as segments
                FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}" ed
                WHERE ed."organizationId" = $${queryParamIndex++} AND ed."userId" = $${queryParamIndex++}
                AND(data -> 's' ->> 'm'):: int > 0 AND(data -> 's' ->> 'ew'):: int != 0
            )
            UNION
            (
                SELECT eld."userId", eld."organizationId", NOW() as day, (data -> 's' ->> '${selectionData}'):: int "work", data as segments
                FROM "${organizationId}"."employeeLiveDay" eld         
                WHERE eld."organizationId" = $${queryParamIndex++} and eld."userId" = $${queryParamIndex++} `;
        switch (type) {
            case app_enums_1.enums.EmployeeBinaryDistributionType.MissingWork:
                query += ` AND (data -> 's' ->> 'm'):: int > 0 AND (data -> 's' ->> 'ew'):: int != 0 `;
                break;
            case app_enums_1.enums.EmployeeBinaryDistributionType.ExtraWork:
                query += ` AND (data -> 's' ->> 'e'):: int > 0 `;
                break;
            default:
                break;
        }
        query += `
                )
            ) q1
        `;
        queryParams.push(dal_constants_1.DalConstants.PPermissionStatus.Waiting, organizationId, userId, organizationId, userId);
        query += `
        LEFT OUTER JOIN "${organizationId}"."recalculateWorkQueue" rwq
        ON rwq."userId" = q1."userId" AND rwq."organizationId" = q1."organizationId" AND rwq."startDate" <= day
        WHERE day > $${queryParamIndex++} AND day < $${queryParamIndex++}
`;
        if (dateTimeRange) {
            queryParams.push(dateTimeRange.startDateTime, dateTimeRange.endDateTime);
        }
        else {
            let startDate;
            try {
                startDate = await this.getPacsActivedDate(organizationId);
                queryParams.push(startDate, new Date());
            }
            catch (error) {
                startDate = (await dal_manager_1.dbManager.accessUser.getUserOrganizationProfileEmploymentInfo(organizationId, userId))?.employmentStartUtc;
                queryParams.push(startDate, new Date());
            }
        }
        query += `
        ORDER BY day DESC`;
        const count = (await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q1", queryParams)).rows[0].count;
        if (pagination.skip) {
            query += `
            OFFSET $${queryParamIndex++}`;
            queryParams.push(pagination.skip);
        }
        if (pagination.take) {
            query += `
            LIMIT $${queryParamIndex++}`;
            queryParams.push(pagination.take);
        }
        let result = {
            pagination: {
                total: parseInt(count),
                skip: pagination.skip,
                take: pagination.take,
            },
            items: [],
        };
        const now = moment();
        const dbResult = await this._pgPool.query(query, queryParams);
        for (const row of dbResult.rows) {
            let item = {
                day: moment(row.day).format("YYYY-MM-DD"),
                work: row.work,
                segments: [],
                waitRecalculationFor: this.extractIndividualRecalculateReasons(row.reason),
                summary: this.summarizeFromShort(row.segments),
                warnings: {
                    pendingPermissionIds: row.ppermissionIds ? row.ppermissionIds : [],
                },
            };
            item.summary.firstStart = row.firstStart;
            item.summary.lastLeave = row.lastLeave;
            item.segments = row.segments.c
                .filter((c) => {
                let check = false;
                switch (type) {
                    case app_enums_1.enums.EmployeeBinaryDistributionType.MissingWork:
                        if (c.et === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours || c.et === app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) {
                            if (c.at === app_enums_1.enums.AttendanceTimeRangeType.Absent) {
                                if (!c.ad?.t) {
                                    check = true;
                                }
                            }
                            else if (c.at === app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance) {
                                check = true;
                            }
                            else if (c.at === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance || c.at === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                                if (c.ad?.t) {
                                    check = true;
                                }
                            }
                            if (check && moment(c.e).isAfter(now)) {
                                check = false;
                                if (moment(c.s) < now) {
                                    let d = new moment_range_1.DateRange(moment(c.s), moment()).duration("minutes");
                                    item.work = item.work - d;
                                }
                            }
                        }
                        break;
                    case app_enums_1.enums.EmployeeBinaryDistributionType.ExtraWork:
                        if (c.et === app_enums_1.enums.EmployeeTimeRangeType.Break) {
                            if (c.at === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance) {
                                if (!c.ad?.t) {
                                    check = true;
                                }
                            }
                        }
                        else if (c.et === app_enums_1.enums.EmployeeTimeRangeType.Holiday) {
                            if (c.at === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance || c.at === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                                if (!c.ad?.t) {
                                    check = true;
                                }
                            }
                        }
                        else if (c.et === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission) {
                            if (c.at === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance || c.at === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                                if (!c.ad?.t) {
                                    check = true;
                                }
                            }
                        }
                        else if (c.et === app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours) {
                            if (c.at === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance || c.at === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                                if (!c.ad?.t) {
                                    check = true;
                                }
                            }
                        }
                        break;
                }
                return check;
            })
                .map((c) => {
                return {
                    startDateTime: c.s,
                    endDateTime: c.e,
                };
            });
            result.items.push(item);
        }
        return result;
    }
    mergeEmployeeWorkPlanMemberships(organizationId) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let rowQuery = `
SELECT uwp1.id as "id1", uwp2.id as "id2", uwp1."startDateTime" as "start", uwp2."startDateTime" as "mid", uwp2."endDateTime" as "end" FROM
"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp1
INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp2 ON uwp1."userId" = uwp2."userId" AND uwp1."workPlanId" = uwp2."workPlanId" AND uwp1."endDateTime" = uwp2."startDateTime";
`;
            let rowsToMerge = (await trx.query(rowQuery)).rows[0];
            let mergeOperations = 0;
            while (rowsToMerge) {
                await trx.query(`
UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" SET "endDateTime" = $1 WHERE id = $2;
`, [rowsToMerge.end, rowsToMerge.id1]);
                await trx.query(`
DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" WHERE id = $1
    `, [rowsToMerge.id2]);
                mergeOperations++;
                rowsToMerge = (await trx.query(rowQuery)).rows[0];
            }
            dal_logger_1.dbLogger.info("User work plan entry merging completed after performing " + mergeOperations + " operations");
        });
    }
    async mergeUserWorkPlanMemberShip(organizationId, params, trx) {
        let mergeRowQuery = `
		SELECT uwp1.id as "id1", uwp2.id as "id2", uwp1."startDateTime" as "start", uwp2."startDateTime" as "mid", uwp2."endDateTime" as "end"
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp1
		INNER JOIN "${organizationId}"."userWorkPlans" uwp2
		ON uwp1."userId" = $1 AND uwp1."userId" = uwp2."userId"
		AND uwp1."workPlanId" = uwp2."workPlanId" AND uwp1."endDateTime" = uwp2."startDateTime";`;
        let rowsToMerge = (await trx.query(mergeRowQuery, [params.userId])).rows[0];
        let mergeOperations = 0;
        while (rowsToMerge) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" SET "endDateTime" = $1 WHERE id = $2;
				`, [rowsToMerge.end, rowsToMerge.id1]);
            await trx.query(`
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" WHERE id = $1
				`, [rowsToMerge.id2]);
            mergeOperations++;
            rowsToMerge = (await trx.query(mergeRowQuery, [params.userId])).rows[0];
        }
        dal_logger_1.dbLogger.info("User work plan entry merging completed after performing " + mergeOperations + " operations");
    }
    async getAllPermissionsBetweenDates(organizationId, requesterUserId, startDate, endDate, idBasedUserFilter, onData) {
        let queryParams = [];
        let query = "";
        let queryParamIndex = 1;
        query += `
        SELECT allData."userId" AS "userId", json_agg(allData."permissions") as permissions, array_agg(allData."totalDays")::int[] as totalDays, array_agg(allData."workedDays")::int[] as workedDays
        FROM (
            SELECT "userId" AS "userId",
                json_agg(
                    json_build_object(
                        'date', date,
                        'permission', data -> 'p',
                        'summary', data -> 's'
                    ) ORDER BY date ASC
                ) AS "permissions",
                SUM(CASE WHEN ( (data->'s'->>'ew')::integer > 0 OR (data->'s'->>'eo')::integer > 0 ) THEN 1 ELSE 0 END) as "totalDays",
                SUM(CASE WHEN ( (data->'s'->>'ew')::integer > 0 AND (data->'s'->>'n')::integer > 0 ) THEN 1 ELSE 0 END) as "workedDays"
                FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
                WHERE date >= $${queryParamIndex++} and date <= $${queryParamIndex++}
                GROUP BY "userId"
        `;
        queryParams.push(startDate.format("YYYY-MM-DD"), endDate.format("YYYY-MM-DD"));
        let now = moment();
        let end = endDate;
        if (now.isSameOrBefore(end, "day")) {
            query += `
            UNION ALL
            SELECT "userId" AS "userId",
                json_agg(
                    json_build_object(
                        'date', now(),
                        'permission', data -> 'p',
                        'summary', data -> 's'
                    )
                ) AS "permissions",
                SUM(CASE WHEN ( (data->'s'->>'ew')::integer > 0 OR (data->'s'->>'eo')::integer > 0 ) THEN 1 ELSE 0 END) as "totalDays",
                SUM(CASE WHEN ( (data->'s'->>'ew')::integer > 0 AND (data->'s'->>'n')::integer > 0 ) THEN 1 ELSE 0 END) as "workedDays"
            FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}"
            WHERE data IS NOT NULL
            GROUP BY "userId"
            `;
        }
        query += ` ) as allData `;
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            requesterUserId: requesterUserId,
            organizationId: organizationId,
            idBasedUserFilter: idBasedUserFilter,
            requiredOrganizationUnitWidePermissions: [],
            requiredOrganizationWidePermissions: [],
            bindingKeys: queryParams,
        });
        query += `
            INNER JOIN (${userFilter.query}) userFilter ON userFilter."userId" = allData."userId"
        `;
        queryParams = userFilter.bindingKeys;
        queryParamIndex = userFilter.bindingKeys.length;
        query += `
            group by allData."userId"
        `;
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(query, queryParams));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(1000, async (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error("Error while fetch custom report data with cursor!");
            }
            if (rows.length < 1000) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
        return Promise.resolve();
    }
    getCurrentPPermissionOverview(organizationId, requesterUserId, listAs) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let result = {
                waitingApprovalPPermissionCount: 0,
                waitingMyApprovalPPermissionCount: 0,
                hasPPermissionCount: 0,
                authorizedOrganizationUnits: [],
                isOrganizationWide: listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR,
            };
            result.authorizedOrganizationUnits = await dal_manager_1.dbManager.accessPacs.getOrganizationUnitsOfManager(organizationId, requesterUserId);
            let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: organizationId,
                requesterUserId: requesterUserId,
                bindingKeys: [],
                idBasedUserFilter: {
                    applyOrganizationUnitFilterHierarchically: false,
                    organizationUnitIds: listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager ? result.authorizedOrganizationUnits : undefined,
                },
            });
            let bindingKeys = userFilter.bindingKeys;
            let bindIndex = bindingKeys.length;
            let approveCheck = "";
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                approveCheck = `uppa."organizationUnitId" IS NULL`;
            }
            else if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager) {
                approveCheck = `uppa."organizationUnitId"::text = ANY(array['${result.authorizedOrganizationUnits.join(`','`)}'])`;
            }
            else {
                approveCheck = `false`;
            }
            let query = `
            SELECT SUM(CASE
                WHEN uppa."approvementDate" IS NOT NULL AND tstzrange(pp."startDateTime", pp."endDateTime") @> now()
                THEN 1 ELSE 0
            END) as "hasPPermissionCount",
            SUM(CASE
                WHEN uppa."approvementDate" IS NULL AND uppa.order = up."currentApprovementOrder" THEN 1 ELSE 0
            END) as "waitingApprovalPPermissionCount",
            SUM(CASE
                WHEN uppa."approvementDate" IS NULL AND uppa.order = up."currentApprovementOrder" AND uppa."organizationUnitId" IS NULL AND ${approveCheck} THEN 1 ELSE 0
            END) as "waitingMyApprovalPPermissionCount"
            FROM "${organizationId}"."userPPermissionApprovements" uppa
            INNER JOIN "${organizationId}"."userPPermissions" up ON uppa."ppermissionId" = up."ppermissionId" AND uppa."userId" = up."userId"
            INNER JOIN "${organizationId}"."ppermissions" pp ON pp.id = up."ppermissionId"
            INNER JOIN "${organizationId}"."ppermissionTypes" ppt ON ppt.id = pp."ppermissionTypeId" AND ppt."deletedAt" IS NULL
            INNER JOIN (
                ${userFilter.query}
            ) ufq ON ufq."userId" = up."userId"
            WHERE (uppa."approvementDate" IS NULL AND uppa.order = up."currentApprovementOrder") OR (uppa."approvementDate" IS NOT NULL)
            `;
            let dbResult = await trx.query(query, bindingKeys);
            if (dbResult.rows[0]) {
                result.hasPPermissionCount = dbResult.rows[0].hasPPermissionCount;
                result.waitingApprovalPPermissionCount = dbResult.rows[0].waitingApprovalPPermissionCount;
                result.waitingMyApprovalPPermissionCount = dbResult.rows[0].waitingMyApprovalPPermissionCount;
            }
            return result;
        });
    }
    listCurrentPPermissionOverview(organizationId, requesterUserId, listAs, pagination, type, organizationUnits) {
        return dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let result = {
                pagination: {
                    take: pagination.take,
                    skip: pagination.skip,
                    total: 0,
                },
                authorizedOrganizationUnits: [],
                items: [],
            };
            result.authorizedOrganizationUnits = await dal_manager_1.dbManager.accessPacs.getOrganizationUnitsOfManager(organizationId, requesterUserId);
            let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: organizationId,
                requesterUserId: requesterUserId,
                bindingKeys: [],
                idBasedUserFilter: {
                    applyOrganizationUnitFilterHierarchically: false,
                    organizationUnitIds: organizationUnits ? organizationUnits : listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager ? result.authorizedOrganizationUnits : undefined,
                },
            });
            let bindingKeys = userFilter.bindingKeys;
            let bindIndex = bindingKeys.length;
            let approveCheck = "";
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                approveCheck = `uppa."organizationUnitId" IS NULL`;
            }
            else if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager) {
                approveCheck = `uppa."organizationUnitId"::text = ANY(array['${result.authorizedOrganizationUnits.join(`','`)}'])`;
            }
            else {
                approveCheck = `false`;
            }
            let query = `
            SELECT uppa."userId", ufq.name, ufq.surname
            FROM "${organizationId}"."userPPermissionApprovements" uppa
            INNER JOIN "${organizationId}"."userPPermissions" up ON uppa."ppermissionId" = up."ppermissionId" AND uppa."userId" = up."userId"
            INNER JOIN "${organizationId}"."ppermissions" pp ON pp.id = up."ppermissionId"
            INNER JOIN "${organizationId}"."ppermissionTypes" ppt ON ppt.id = pp."ppermissionTypeId" AND ppt."deletedAt" IS NULL
            INNER JOIN (
                ${userFilter.query}
            ) ufq ON ufq."userId" = up."userId"
            WHERE (uppa."approvementDate" IS NULL AND uppa.order = up."currentApprovementOrder") OR (uppa."approvementDate" IS NOT NULL)
            `;
            if (type === dal_constants_1.DalConstants.CurrentPPermissionInfoType.HasPPermission) {
                query += `
                AND uppa."approvementDate" IS NOT NULL AND tstzrange(pp."startDateTime", pp."endDateTime") @> now()`;
            }
            else if (type === dal_constants_1.DalConstants.CurrentPPermissionInfoType.WaitingApproval) {
                query += `
                AND uppa."approvementDate" IS NULL AND uppa.order = up."currentApprovementOrder"`;
            }
            else if (type === dal_constants_1.DalConstants.CurrentPPermissionInfoType.WaitingMyApproval) {
                query += `
                uppa."approvementDate" IS NULL AND uppa.order = up."currentApprovementOrder" AND uppa."organizationUnitId" IS NULL AND ${approveCheck}`;
            }
            result.pagination.total = (await trx.query(`SELECT COUNT(*) as cnt FROM (${query}) sq`, bindingKeys)).rows[0].cnt;
            if (pagination.skip) {
                query += `
                OFFSET $${++bindIndex}`;
                bindingKeys.push(pagination.skip);
            }
            if (pagination.take) {
                query += `
                LIMIT $${++bindIndex}`;
                bindingKeys.push(pagination.take);
            }
            let dbResult = await trx.query(query, bindingKeys);
            for (const row of dbResult.rows) {
                const userCaptions = (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: row.userId })).caption;
                let item = {
                    userId: row.userId,
                    userCaptions: userCaptions,
                    userFullName: (row.name ?? "") + " " + (row.surname ?? ""),
                };
            }
            return result;
        });
    }
    async getDataSapCustomReports(params) {
        let queryParams = [];
        let query = "";
        let queryParamIndex = 1;
        query += `
        SELECT ed."userId", ed.date, json_agg(ed.data) as data
        FROM (
            SELECT "userId", date,  data 
            from "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeDays}"
            where "organizationId" = $${queryParamIndex++} and date >= $${queryParamIndex++} AND date <= $${queryParamIndex++}
        ) ed
        `;
        queryParams.push(params.organizationId, params.startDate.format("YYYY-MM-DD"), params.endDate.format("YYYY-MM-DD"));
        let userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            requesterUserId: params.requesterUserId,
            organizationId: params.organizationId,
            idBasedUserFilter: {
                applyOrganizationUnitFilterHierarchically: params.organizationUnitHierarchically,
                organizationUnitIds: params.organizationUnitIds ? params.organizationUnitIds : null,
                userGroupIds: params.userGroups,
                userIds: null,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
            },
            requiredOrganizationUnitWidePermissions: [],
            requiredOrganizationWidePermissions: [],
            bindingKeys: queryParams,
        });
        query += `
            INNER JOIN (${userFilter.query}) userFilter ON userFilter."userId" = ed."userId"
            group by ed."userId", ed.date
        `;
        queryParams = userFilter.bindingKeys;
        queryParamIndex = userFilter.bindingKeys.length;
        const client = await this._pgPool.connect();
        const { rows } = await client.query(query, queryParams);
        client.release();
        return Promise.resolve(rows);
    }
    async updateRemainingAnnualPPermissionOfUser(organizationId, userId, remainingAnnualPPermission) {
        let userOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId, userId });
        await this.pgPool.query(`
            UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
            SET "manuallyRemainedAnnualPermission" = $1::float
            WHERE "userOrganizationId" = $2 AND "userId" = $3
        `, [remainingAnnualPPermission, userOrganizationId, userId]);
    }
    async updateRemainingAnnualPPermissionOfUserByApprove(organizationId, userId, ppermissionTypeId, ppermissionStartDateTime, ppermissionEndDateTime, isAddition) {
        let userOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId, userId });
        let dayCount = await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, ppermissionStartDateTime, ppermissionEndDateTime, undefined, userId);
        await this.pgPool.query(`
            UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
            SET "manuallyRemainedAnnualPermission" = "manuallyRemainedAnnualPermission" ${isAddition ? "+" : "-"} $1::float
            WHERE "userOrganizationId" = $2 AND "userId" = $3
        `, [dayCount, userOrganizationId, userId]);
    }
    async updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, userIds, ppermissionTypeId, ppermissionStartDateTime, ppermissionEndDateTime, trx, isAddition) {
        if (userIds.length === 0)
            return;
        const updates = await Promise.all(userIds.map(async (userId) => {
            const userOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId, userId });
            const dayCount = await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, ppermissionStartDateTime, ppermissionEndDateTime, undefined, userId);
            return { userId, userOrganizationId, dayCount };
        }));
        const caseLines = [];
        const whereLines = [];
        const values = [];
        for (const { userId, userOrganizationId, dayCount } of updates) {
            values.push(userOrganizationId, userId, dayCount);
            caseLines.push(`WHEN "userOrganizationId" = ? AND "userId" = ? THEN ?::float`);
        }
        for (const { userId, userOrganizationId } of updates) {
            values.push(userOrganizationId, userId);
            whereLines.push(`("userOrganizationId" = ? AND "userId" = ?)`);
        }
        const operator = isAddition ? "+" : "-";
        const query = `
		UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
		SET "manuallyRemainedAnnualPermission" = "manuallyRemainedAnnualPermission" ${operator} CASE
			${caseLines.join("\n")}
			ELSE 0
		END
		WHERE ${whereLines.join(" OR ")}
	`;
        await trx.raw(query, values);
    }
    async updateRemainingAnnualPPermissionOfUserMultiplePPermissions(organizationId, userId, ppermissionTypeId, usage) {
        let userOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId, userId });
        await this.pgPool.query(`
            UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
            SET "manuallyRemainedAnnualPermission" = "manuallyRemainedAnnualPermission" - $1::float
            WHERE "userOrganizationId" = $2 AND "userId" = $3
        `, [usage, userOrganizationId, userId]);
    }
    async getAbsentUserIds(params) {
        const { rows } = await params.trx.query(`
            SELECT "userId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
            WHERE "userId" = any($1::uuid[])
            AND (data ->'s'->'ew')::integer > 0 AND (data ->'s'->'ew')::integer = (data ->'s'->'m')::integer
        `, [params.userIds]);
        return rows.map((r) => r.userId);
    }
    async getOverworkedUserIds(params) {
        const { rows } = await params.trx.query(`
            SELECT "userId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
            WHERE "userId" = any($1::uuid[])
            AND (data ->'s'->'ll')::integer > 0
			AND ((data->'c')::json->>-1)::json->>'at' = '8'
            AND ((((data->'c')::json->>-1)::json->>'es')::json->>'w')::boolean = TRUE;
        `, [params.userIds]);
        return rows.map((r) => r.userId);
    }
    async getEarlyStartUserIds(params) {
        const { rows } = await params.trx.query(`
        SELECT "userId", (data ->'s'->'es') as "earlyStartAmount" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
        WHERE "userId" = any($1::uuid[])
        AND (data ->'s'->'ew')::integer > 0 AND (data ->'s'->'es')::integer > 0 
        `, [params.userIds]);
        return rows;
    }
    async getEarlyLeaveUserIds(params) {
        const { rows } = await params.trx.query(`
        SELECT "userId", (data ->'s'->'el') as "earlyLeaveAmount" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
        WHERE "userId" = any($1::uuid[])
        AND (data ->'s'->'ew')::integer > 0 AND (data ->'s'->'el')::integer > 0 
        `, [params.userIds]);
        return rows;
    }
    async getLateLeaveUserIds(params) {
        const { rows } = await params.trx.query(`
        SELECT "userId", (data ->'s'->'ll') as "lateLeaveAmount" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
        WHERE "userId" = any($1::uuid[])
        AND (data ->'s'->'ew')::integer > 0 AND (data ->'s'->'ll')::integer > 0 
        `, [params.userIds]);
        return rows;
    }
    async getLateStartUserIds(params) {
        const { rows } = await params.trx.query(`
        SELECT "userId", (data ->'s'->'ls') as "lateStartAmount" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
        WHERE "userId" = any($1::uuid[])
        AND (data ->'s'->'ew')::integer > 0 AND (data ->'s'->'ls')::integer > 0 
        `, [params.userIds]);
        return rows;
    }
    async getDailyMissingWorkUserIds(params) {
        const { rows } = await this.pgPool.query(`
        SELECT "userId", (data ->'s'->'m') AS "missingWork" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.employeeLiveDay}" 
        WHERE "userId" = ANY($1::uuid[])
        AND (data ->'s'->'ew')::integer > 0 AND (data ->'s'->'m')::integer > 0
        `, [params.userIds]);
        return rows.map((r) => {
            return {
                userId: r.userId,
                missingWork: r.missingWork,
            };
        });
    }
    async getMonthlyReport(organizationId, requesterUserId, filter) {
        let result = {
            pagination: {
                skip: filter.pagination?.skip,
                take: filter.pagination?.take,
                total: 0,
            },
            items: [],
        };
        let requiredPermissions = ["j:l", "e:u", "u:b", "g:r"];
        if (filter.userFilter && filter.userFilter.userIds && filter.userFilter.userIds.length === 1 && filter.userFilter.userIds[0] === requesterUserId) {
            requiredPermissions = undefined;
        }
        let begin;
        let end;
        let dateFilter = filter.dateFilter;
        if (dateFilter.group === app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Annually && dateFilter.year) {
            begin = moment().startOf("year").year(dateFilter.year);
            end = begin.clone().endOf("year");
        }
        else if (dateFilter.group === app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Monthly && dateFilter.month) {
            begin = moment()
                .startOf("month")
                .year(dateFilter.year)
                .month(dateFilter.month - 1);
            end = begin.clone().endOf("month");
        }
        else if (dateFilter.group === app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Whole && dateFilter.range && dateFilter.range.startDateTime && dateFilter.range.endDateTime) {
            begin = moment(dateFilter.range.startDateTime);
            end = moment(dateFilter.range.endDateTime);
        }
        switch (dateFilter.group) {
            case app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Annually:
                let endMonth = begin.clone().endOf("month");
                while (begin.isSameOrBefore(end) && begin.isSameOrBefore(moment())) {
                    let dbResult = await this.listMonthlyReport({
                        organizationId: organizationId,
                        requesterUserId: requesterUserId,
                        pagination: {
                            take: filter.pagination?.take,
                            skip: filter.pagination?.skip,
                        },
                        startMonth: begin,
                        endMonth: endMonth,
                        idBasedUserFilter: {
                            applyOrganizationUnitFilterHierarchically: filter.userFilter?.applyOrganizationUnitFilterHierarchically,
                            organizationUnitIds: filter.userFilter?.organizationUnitIds,
                            userGroupIds: filter.userFilter?.userGroupIds,
                            userIds: filter.userFilter?.userIds,
                        },
                        workPlanIds: filter.userFilter?.workPlanIds,
                        requiredOrganizationWidePermissions: requiredPermissions,
                        requiredOrganizationUnitWidePermissions: requiredPermissions,
                    });
                    if (result.items.length === 0) {
                        result = dbResult;
                    }
                    else {
                        for (const item of result.items) {
                            let newData = dbResult.items.find((i) => i.user.id === item.user.id);
                            if (!newData)
                                continue;
                            let ppermissions = item.miscellaneous.usedPermissions.concat(newData.miscellaneous.usedPermissions);
                            let recalculation = item.miscellaneous.waitRecalculationFor.concat(newData.miscellaneous.waitRecalculationFor);
                            item.editedWorkingDurations = lodash_1.default.mergeWith({}, item.editedWorkingDurations, newData.editedWorkingDurations, lodash_1.default.add);
                            item.rawWorkingDurations = lodash_1.default.mergeWith({}, item.rawWorkingDurations, newData.rawWorkingDurations, lodash_1.default.add);
                            item.miscellaneous = lodash_1.default.mergeWith({}, item.miscellaneous, newData.miscellaneous, lodash_1.default.add);
                            item.miscellaneous.usedPermissions = ppermissions;
                            item.miscellaneous.waitRecalculationFor = recalculation;
                        }
                    }
                    begin = begin.clone().add(1, "months");
                    endMonth = begin.clone().endOf("month");
                }
                break;
            case app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Whole:
                let start = begin.clone();
                let ranges = [];
                let months = [];
                let endmonth = start.clone().endOf("month");
                while (start.isSameOrBefore(end) && start.isSameOrBefore(moment())) {
                    let startmonth = start.clone().startOf("month");
                    if (endmonth.isSameOrBefore(end, "day") && start.isSame(startmonth)) {
                        months.push({
                            start: startmonth,
                            end: end.isBefore(endmonth) ? end : endmonth,
                        });
                    }
                    else if (endmonth.isSameOrBefore(end) && !start.isSame(startmonth)) {
                        ranges.push({
                            start: start,
                            end: end.isBefore(endmonth) ? end : endmonth,
                        });
                    }
                    else if (endmonth.isAfter(end)) {
                        ranges.push({
                            start: start,
                            end: end.isBefore(endmonth) ? end : endmonth,
                        });
                    }
                    start = start.clone().add(1, "month").startOf("month");
                    endmonth = start.clone().endOf("month");
                }
                for (const month of months) {
                    let dbResult = await this.listMonthlyReport({
                        organizationId: organizationId,
                        requesterUserId: requesterUserId,
                        pagination: {
                            take: filter.pagination?.take,
                            skip: filter.pagination?.skip,
                        },
                        startMonth: month.start,
                        endMonth: month.end,
                        idBasedUserFilter: {
                            applyOrganizationUnitFilterHierarchically: filter.userFilter?.applyOrganizationUnitFilterHierarchically,
                            organizationUnitIds: filter.userFilter?.organizationUnitIds,
                            userGroupIds: filter.userFilter?.userGroupIds,
                            userIds: filter.userFilter?.userIds,
                        },
                        workPlanIds: filter.userFilter?.workPlanIds,
                        requiredOrganizationWidePermissions: requiredPermissions,
                        requiredOrganizationUnitWidePermissions: requiredPermissions,
                    });
                    if (result.items.length === 0) {
                        result = dbResult;
                    }
                    else {
                        for (const item of result.items) {
                            let newData = dbResult.items.find((i) => i.user.id === item.user.id);
                            if (!newData)
                                continue;
                            let ppermissions = item.miscellaneous.usedPermissions.concat(newData.miscellaneous.usedPermissions);
                            let recalculation = item.miscellaneous.waitRecalculationFor.concat(newData.miscellaneous.waitRecalculationFor);
                            item.editedWorkingDurations = lodash_1.default.mergeWith({}, item.editedWorkingDurations, newData.editedWorkingDurations, lodash_1.default.add);
                            item.rawWorkingDurations = lodash_1.default.mergeWith({}, item.rawWorkingDurations, newData.rawWorkingDurations, lodash_1.default.add);
                            item.miscellaneous = lodash_1.default.mergeWith({}, item.miscellaneous, newData.miscellaneous, lodash_1.default.add);
                            item.miscellaneous.usedPermissions = ppermissions;
                            item.miscellaneous.waitRecalculationFor = recalculation;
                        }
                    }
                }
                let userIds = [];
                if (result?.items?.length > 0) {
                    userIds = result.items.map((i) => i.user.id);
                }
                else {
                    let userQuery = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                        organizationId: organizationId,
                        requesterUserId,
                        bindingKeys: [],
                        idBasedUserFilter: {
                            applyOrganizationUnitFilterHierarchically: filter.userFilter?.applyOrganizationUnitFilterHierarchically,
                            organizationUnitIds: filter.userFilter?.organizationUnitIds,
                            userGroupIds: filter.userFilter?.userGroupIds,
                            userIds: filter.userFilter?.userIds,
                            workPlanIds: filter.userFilter?.workPlanIds,
                        },
                        specificSelectItems: ["userId"],
                        requiredOrganizationUnitWidePermissions: requiredPermissions,
                        requiredOrganizationWidePermissions: requiredPermissions,
                    });
                    userIds = (await this.pgPool.query(userQuery.query, userQuery.bindingKeys)).rows.map((r) => r.userId);
                }
                for (const range of ranges) {
                    let rawMonthlyData = await this.getSummaryOfEmployeesDateRange(organizationId, {
                        start: range.start.toDate(),
                        end: range.end.toDate(),
                        userIds: userIds,
                        pagination: {
                            take: filter.pagination?.take,
                            skip: filter.pagination?.skip,
                        },
                    });
                    if (!result.pagination.total) {
                        result.pagination.total = rawMonthlyData.pagination.total;
                    }
                    if (!rawMonthlyData) {
                        continue;
                    }
                    if (result.items.length === 0) {
                        result = rawMonthlyData;
                    }
                    else {
                        for (const item of result.items) {
                            let newData = rawMonthlyData.items.find((i) => i.user.id === item.user.id);
                            if (!newData)
                                continue;
                            let ppermissions = item.miscellaneous.usedPermissions.concat(newData.miscellaneous.usedPermissions);
                            let recalculation = item.miscellaneous.waitRecalculationFor.concat(newData.miscellaneous.waitRecalculationFor);
                            item.editedWorkingDurations = lodash_1.default.mergeWith({}, item.editedWorkingDurations, newData.editedWorkingDurations, lodash_1.default.add);
                            item.rawWorkingDurations = lodash_1.default.mergeWith({}, item.rawWorkingDurations, newData.rawWorkingDurations, lodash_1.default.add);
                            item.miscellaneous = lodash_1.default.mergeWith({}, item.miscellaneous, newData.miscellaneous, lodash_1.default.add);
                            item.miscellaneous.usedPermissions = ppermissions;
                            item.miscellaneous.waitRecalculationFor = recalculation;
                        }
                    }
                }
                break;
            case app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Monthly:
                let dbResult = await this.listMonthlyReport({
                    organizationId: organizationId,
                    requesterUserId: requesterUserId,
                    pagination: {
                        take: filter.pagination?.take,
                        skip: filter.pagination?.skip,
                    },
                    startMonth: begin,
                    endMonth: end,
                    idBasedUserFilter: {
                        applyOrganizationUnitFilterHierarchically: filter.userFilter?.applyOrganizationUnitFilterHierarchically,
                        organizationUnitIds: filter.userFilter?.organizationUnitIds,
                        userGroupIds: filter.userFilter?.userGroupIds,
                        userIds: filter.userFilter?.userIds,
                    },
                    workPlanIds: filter.userFilter?.workPlanIds,
                    requiredOrganizationWidePermissions: requiredPermissions,
                    requiredOrganizationUnitWidePermissions: requiredPermissions,
                });
                result = dbResult;
                break;
            default:
                break;
        }
        for (const item of result.items) {
            if (item.miscellaneous.expectedWorkDuration > 0) {
                item.miscellaneous.score = Math.ceil(((item.editedWorkingDurations.normalWorkDuration + item.editedWorkingDurations.extraWorkDuration) / item.miscellaneous.expectedWorkDuration) * 100);
                if (item.miscellaneous.score > 100) {
                    item.miscellaneous.score = 100;
                }
            }
        }
        return result;
    }
    async removeUserWorkplanMembership(trx, organizationId, uwpId) {
        let existing = (await trx.query(`SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" WHERE id = $1`, [uwpId])).rows[0];
        if (!existing) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.PACS.WORKPLANMEMBERSHIPNOTFOUND");
        }
        else {
            await trx.query(`INSERT INTO "${organizationId}"."recalculateWorkQueue"
				(id, "userId", "organizationId", reason, "startDate")
				SELECT uuid_generate_v4(), uwp."userId", '${organizationId}'::uuid, $1::smallint, lower(wpp.range*tstzrange(uwp."startDateTime", uwp."endDateTime"))
				FROM "${organizationId}"."userWorkPlans" uwp
				INNER JOIN "${organizationId}"."vW_WorkPlanPeriodTstzrange" as wpp ON wpp."workPlanId" = uwp."workPlanId"
				AND uwp.id = $2 AND wpp.range&&tstzrange(uwp."startDateTime", uwp."endDateTime")
				WHERE LOWER(wpp.range*tstzrange(uwp."startDateTime", uwp."endDateTime")) IS NOT NULL`, [dal_constants_1.DalConstants.RecalculateWorkReason.UserWorkPlanChange, uwpId]);
            await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" WHERE id = $1`, [uwpId]);
            await trx.query(`DELETE FROM "${organizationId}"."employeeLiveDay" WHERE "userId" = $1`, [existing.userId]);
            let workPlans = await this.getWorkPlansOfNowPg(trx, organizationId);
            await this.scaffoldEmployeeLiveDataForWorkplansPg(trx, workPlans, organizationId, [existing.userId]);
        }
    }
    async removeUserWorkplanMembershipInRange(organizationId, params, trx) {
        await trx.query(`
			WITH uwp AS (
				SELECT uwp.id, "workPlanId" as wp, "startDateTime", "endDateTime" 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp
				WHERE  tstzrange("startDateTime", "endDateTime") && tstzrange($1, $2)
						AND uwp."userId" = $3
			),
			entry_info AS (
				SELECT *
				FROM (
					SELECT *, row_number() OVER (PARTITION BY id, isempty(range) ) as op
					FROM (
						SELECT uwp.id, uwp.wp, tstzrange(uwp."startDateTime", uwp."endDateTime") - tstzrange($1,null) as range, true as exists
						FROM uwp
						UNION ALL
						SELECT uwp.id, uwp.wp, tstzrange(uwp."startDateTime", uwp."endDateTime") - tstzrange(null,$2) as range, true as exists
						FROM uwp
					) sq
				) sq2
			),delete_entry AS (
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp
				USING entry_info as ei
				WHERE uwp.id = ei.id AND ei.exists AND ei.op = 2 AND isempty(range)
			), update_entry AS (
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp
				SET "startDateTime" = LOWER(range), "endDateTime" = UPPER(range) 
				FROM entry_info as ei
				WHERE uwp.id = ei.id AND ei.exists AND ei.op = 1 AND NOT isempty(range)
			)
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp(
			id, "workPlanId", "userId", "startDateTime", "endDateTime", "createdAt")
			SELECT gen_random_uuid(), ei.wp, $3, LOWER(ei.range), UPPER(ei.range), now()
			FROM entry_info as ei
			WHERE ei.exists AND op = 2 AND NOT isempty(ei.range);
			`, [params.range.startDateTime, params.range.endDateTime, params.userId]);
    }
    async insertWorkPlanMembership(organizationId, params, trx) {
        await trx.query(`
			WITH uwp AS (
				SELECT uwp.id, "workPlanId" as wp, "startDateTime", "endDateTime" 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp
				WHERE  tstzrange("startDateTime", "endDateTime") && tstzrange($1, $2)
						AND uwp."userId" = $3
			),
			entry_info AS (
				SELECT *
				FROM (
					SELECT *, row_number() OVER (PARTITION BY id, isempty(range) ) as op
					FROM (
						SELECT uwp.id, uwp.wp, tstzrange(uwp."startDateTime", uwp."endDateTime") - tstzrange($1,null) as range, true as exists
						FROM uwp
						UNION ALL
						SELECT uwp.id, uwp.wp, tstzrange(uwp."startDateTime", uwp."endDateTime") - tstzrange(null,$2) as range, true as exists
						FROM uwp
						UNION ALL
						SELECT gen_random_uuid() as id, $4 as wp, tstzrange($1,$2) as range, false as exists
					) sq
				) sq2
			),delete_entry AS (
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp
				USING entry_info as ei
				WHERE uwp.id = ei.id AND ei.exists AND ei.op = 2 AND isempty(range)
			), update_entry AS (
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp
				SET "startDateTime" = LOWER(range), "endDateTime" = UPPER(range) 
				FROM entry_info as ei
				WHERE uwp.id = ei.id AND ei.exists AND ei.op = 1 AND NOT isempty(range)
			)
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp(
			id, "workPlanId", "userId", "startDateTime", "endDateTime", "createdAt")
			SELECT gen_random_uuid(), ei.wp, $3, LOWER(ei.range), UPPER(ei.range), now()
			FROM entry_info as ei
			WHERE (ei.exists AND op = 2 AND NOT isempty(ei.range)) OR (NOT ei.exists AND NOT isempty(ei.range) AND op = 1);
			`, [params.range.startDateTime, params.range.endDateTime, params.userId, params.workPlanId]);
    }
    async updateUserWorkplanMembership(trx, organizationId, uwpId, range) {
        let existing = (await trx.query(`
				SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" WHERE id = $1
				`, [uwpId])).rows[0];
        if (!existing) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.PACS.WORKPLANMEMBERSHIPNOTFOUND");
        }
        else {
            await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" WHERE id = $1`, [uwpId]);
            await this.insertWorkPlanMembership(organizationId, { userId: existing.userId, workPlanId: existing.workPlanId, range }, trx);
            await trx.query(`
			INSERT INTO "${organizationId}"."recalculateWorkQueue"
			(id, "userId", "organizationId", reason, "startDate")
			SELECT uuid_generate_v4(), $1, '${organizationId}'::uuid, $2::smallint, LEAST(LOWER(range), $3, $4)
			FROM "${organizationId}"."vW_WorkPlanPeriodTstzrange"
			WHERE "workPlanId" = $5 AND LEAST(LOWER(range), $3, $4) IS NOT NULL
			`, [existing.userId, dal_constants_1.DalConstants.RecalculateWorkReason.UserWorkPlanChange, existing.startDateTime ?? null, range.startDateTime ?? null, existing.workPlanId]);
            await trx.query(`
			DELETE FROM "${organizationId}"."employeeLiveDay" WHERE "userId" = $1
			`, [existing.userId]);
            let workPlans = await this.getWorkPlansOfNowPg(trx, organizationId);
            await this.scaffoldEmployeeLiveDataForWorkplansPg(trx, workPlans, organizationId, [existing.userId]);
        }
    }
    async updateUserWorkplanMembershipInRange(trx, organizationId, args) {
        await this.removeUserWorkplanMembershipInRange(organizationId, { userId: args.userId, range: args.range }, trx);
        for (const item of args.items) {
            await this.insertWorkPlanMembership(organizationId, { userId: args.userId, workPlanId: item.workplanId, range: item.range }, trx);
        }
        await this.mergeUserWorkPlanMemberShip(organizationId, { userId: args.userId }, trx);
        await trx.query(`
		INSERT INTO "${organizationId}"."recalculateWorkQueue"
		(id, "userId", "organizationId", reason, "startDate")
		VALUES (uuid_generate_v4(), $1, '${organizationId}'::uuid, $2, $3)
		`, [args.userId, dal_constants_1.DalConstants.RecalculateWorkReason.UserWorkPlanChange, args.range.startDateTime ?? null]);
        await trx.query(`
		DELETE FROM "${organizationId}"."employeeLiveDay" WHERE "userId" = $1
		`, [args.userId]);
        let workPlans = await this.getWorkPlansOfNowPg(trx, organizationId);
        await this.scaffoldEmployeeLiveDataForWorkplansPg(trx, workPlans, organizationId, [args.userId]);
    }
    async upsertAutoShift(organizationId, params, trx) {
        const { id, name, groupIds, regionIds, startDateTime, endDateTime, supportMobileCheckin, supportManualLog, useOnlyFirstAccess } = params;
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSet}"
			(id, "name", "startDateTime", "endDateTime", "supportMobileCheckin", "supportManualLog", "useOnlyFirstAccess")
			VALUES
			($1, $2, $3, $4, $5, $6, $7)
			ON CONFLICT (id) DO UPDATE
			SET name = $2, "startDateTime" = $3, "endDateTime" = $4, "supportMobileCheckin" = $5, "supportManualLog" = $6, "useOnlyFirstAccess" = $7, "deletedAt" = NULL
			`, [id, name, startDateTime, endDateTime, supportMobileCheckin, supportManualLog, useOnlyFirstAccess]);
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetRegion}"
			(id, "regionId", "ruleSetId")
			SELECT gen_random_uuid(), id, $1 FROM unnest($2::uuid[]) as g(id)
			ON CONFLICT("regionId","ruleSetId") DO NOTHING
		`, [id, regionIds]);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetRegion}"
			WHERE "regionId" <> ALL($1::uuid[]) AND "ruleSetId" = $2
			`, [regionIds, id]);
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetGroup}"
			(id, "groupId", "ruleSetId")
			SELECT gen_random_uuid(), id, $1 FROM unnest($2::uuid[]) as g(id)
			ON CONFLICT("groupId","ruleSetId") DO NOTHING
		`, [id, groupIds]);
        await trx.query(`
				DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetGroup}"
				WHERE "groupId" <> ALL($1::uuid[]) AND "ruleSetId" = $2
				`, [groupIds, id]);
        return { id: id };
    }
    async upsertAutoShiftRule(organizationId, params, trx) {
        const { id, rules } = params;
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSet}"
			SET rules = $2
			WHERE id = $1
			`, [id, rules]);
        return { id: id };
    }
    async deleteAutoShift(organizationId, params, trx) {
        let deletedAt = luxon_1.DateTime.now();
        await trx.query(`
            UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSet}"
            SET "deletedAt" = $2
            WHERE id = $1
            `, [params.id, deletedAt]);
        return { id: params.id };
    }
    async listAutoShift(organizationId, params, trx) {
        const { pagination, filter } = params;
        let bindings = [];
        let result = { pagination: { take: pagination?.take ?? 0, skip: pagination?.skip ?? 0, total: 0 }, items: [] };
        let mainQuery = `
		SELECT
			asrs.id AS id,
			asrs.name,
			asrs."startDateTime",
			asrs."endDateTime",
			asrs."rules",
			asrs."supportMobileCheckin",
			asrs."supportManualLog",
			asrs."useOnlyFirstAccess",
			array_agg(DISTINCT jsonb_build_object('id', r.id, 'name', r.name)) AS regions,
			array_agg(DISTINCT jsonb_build_object('id', g.id, 'name', g.name)) AS groups
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSet}" AS asrs
		INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetRegion}" AS asrsr
			ON asrs.id = asrsr."ruleSetId"
		INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" AS r
			ON r.id = asrsr."regionId"
		INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetGroup}" AS asrsg
			ON asrs.id = asrsg."ruleSetId"
		INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" AS g
			ON g.id = asrsg."groupId"
		`;
        let whereQuery = [`asrs."deletedAt" IS NULL`];
        if (filter.startDateTime) {
            bindings.push(filter.startDateTime.toSQL());
            whereQuery.push(`asrs."startDateTime" <= $${bindings.length}`);
        }
        if (filter.endDateTime) {
            bindings.push(filter.endDateTime.toSQL());
            whereQuery.push(`asrs."endDateTime" >= $${bindings.length} OR asrs."endDateTime" IS NULL`);
        }
        if (filter.regionIds) {
            bindings.push(filter.regionIds);
            whereQuery.push(`asrsr."regionId" = ANY($${bindings.length})`);
        }
        if (filter.groupIds) {
            bindings.push(filter.groupIds);
            whereQuery.push(`asrsg."groupId" = ANY($${bindings.length})`);
        }
        if (filter.name) {
            bindings.push(`%${filter.name}%`);
            whereQuery.push(`asrs.name ILIKE $${bindings.length}`);
        }
        mainQuery += `WHERE ` + whereQuery.join(` AND `) + ` GROUP BY asrs.id`;
        result.pagination.total = parseInt((await trx.query(`SELECT COUNT(*) AS count FROM (${mainQuery}) AS sq`, bindings)).rows[0].count);
        if (pagination && pagination.take > 0) {
            bindings.push(pagination.take, pagination.skip ?? 0);
            mainQuery += `
			LIMIT $${bindings.length - 1}
			OFFSET $${bindings.length}`;
            let autoShiftItems = (await trx.query(mainQuery, bindings)).rows;
            result.items.push(...autoShiftItems);
        }
        else {
            let autoShiftItems = (await trx.query(mainQuery, bindings)).rows;
            result.items.push(...autoShiftItems);
        }
        return result;
    }
    async reCalculateShift(organizationId, params, trx) {
        const { id, start, end } = params;
        await trx.query(`SELECT "${organizationId}".assign_auto_shift($1, $2, $3, $4, $5)`, [
            null,
            id,
            null,
            luxon_1.DateTime.fromISO(start).toSQLDate(),
            end ? luxon_1.DateTime.fromISO(end).plus({ day: 1 }).toSQLDate() : null,
        ]);
        return { id: id };
    }
}
exports.PSQLDalAccessPacs2 = PSQLDalAccessPacs2;
async function checkAutoShiftRuleExistency(organizationId, trx) {
    const result = (await trx.query(`
		SELECT COUNT(id) as c
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSet}"
		WHERE "deletedAt" IS NULL
		`)).rows[0].c;
    return { existency: result > 0 ? true : false };
}
exports.checkAutoShiftRuleExistency = checkAutoShiftRuleExistency;
