"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessOAuth = void 0;
const uuid_1 = __importDefault(require("uuid"));
const api_error_1 = require("../../../api/api.error");
const dal_constants_1 = require("../../dal.constants");
const dal_logger_1 = require("../../dal.logger");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_oAuthClients_1 = require("../../db/predefined/predefined.oAuthClients");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_oauth_1 = require("../rdb/dal.access.rdb.oauth");
class PSQLDalAccessOAuth extends dal_access_rdb_oauth_1.RDBDalAccessOAuth {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getClientById(organizationId, clientId, trx) {
        return (await trx.query(`
				SELECT * 
				FROM 
				"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthClients}"
				WHERE id = $1
			`, [clientId])).rows[0];
    }
    async hasRevokedToken(organizationId, tokenId, trx) {
        const dbFn = async (trx) => {
            return trx.query(`
				SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
				WHERE id = $1
			`, [tokenId]);
        };
        let tokenDbResult = null;
        if (trx) {
            tokenDbResult = await dbFn(trx);
        }
        else {
            tokenDbResult = await dal_manager_1.dbManager.systemTransaction(dbFn);
        }
        if (!tokenDbResult || !tokenDbResult.rowCount || tokenDbResult.rows[0].deletedAt) {
            return true;
        }
        else {
            return false;
        }
    }
    async upsertMobileClientDetails(organizationId, userId, tokenId, clientId, publicKey, notificationToken, notificationTokenType, mobileInstallId, userAgent) {
        let now = new Date();
        return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            let armonApplicationUser = await this.getArmonApplicationUser(organizationId, userId, trx);
            if (!armonApplicationUser?.userOrganization?.length) {
                (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.USERNAMEORPASSWORDWRONG");
            }
            let user = this.mapToArmonApplicationUser(armonApplicationUser);
            let qParamIndex = 1;
            const qParams = [];
            let q = `
				SELECT id
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
				WHERE "id" = $${qParamIndex++}
					AND "userId" = $${qParamIndex++}
					AND "deletedAt" IS NULL
			`;
            qParams.push(tokenId);
            qParams.push(userId);
            if (mobileInstallId) {
                q += `
					AND "mobileInstallId" = $${qParamIndex++}
				`;
                qParams.push(mobileInstallId);
            }
            q += ` LIMIT 1 `;
            const dbTokenResult = await trx.query(q, qParams);
            await this.revokePreviouslyLoggedInDeviceTokens(organizationId, userId, clientId, mobileInstallId, trx);
            if (dbTokenResult.rowCount) {
                await trx.query(`
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
						SET "oAuthClientId" = $2,
							"userAgent" = $3,
							"updatedAt" = $4
					WHERE "id"= $1
				`, [dbTokenResult.rows[0].id, clientId, userAgent, now]);
                await trx.query(`
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
						SET "publicKey" = $2,
							"notificationToken" = $3,
							"notificationTokenType" = $4
						WHERE id = $1
				`, [user.id, publicKey, notificationToken, notificationTokenType]);
            }
            else {
                dal_logger_1.dbLogger.debug("oAuth Token not found while uploading device details! This case should not happen!!");
            }
            return Promise.resolve();
        });
    }
    async upsertClientToken(organizationId, params, trx) {
        let now = new Date();
        let time = now.getTime();
        let token = null;
        let qParamIndex = 1;
        const qParams = [];
        let q = `
				SELECT id 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
					WHERE "deletedAt" IS NULL
						AND "userId" = $${qParamIndex++}
			`;
        qParams.push(params.user.id);
        if (params.clientId) {
            q += `
					AND "oauthClientId" = $${qParamIndex++}
				`;
            qParams.push(params.clientId);
        }
        if (params.mobileInstallId || params.userAgentUniqueId) {
            q += `
					AND "mobileInstallId" = $${qParamIndex++}
				`;
            qParams.push(params.mobileInstallId ?? params.userAgentUniqueId);
        }
        else {
            q += `
					AND "userAgent" = $${qParamIndex++}
					AND "ip" = $${qParamIndex++}
				`;
            qParams.push(params.userAgent);
            qParams.push(params.ip);
        }
        q += `
				LIMIT 1
			`;
        const accessTokenExpireAt = params.userAgent.startsWith("mobilemanager")
            ? new Date(time + 3600 * 24 * 7 * 1000)
            : params.custom?.accessTokenExpireAt
                ? new Date(params.custom.accessTokenExpireAt * 1000)
                : new Date(time + 3600000);
        const refreshTokenExpireAt = params.userAgent.startsWith("mobilemanager")
            ? new Date(time + 3600 * 24 * 7 * 4 * 1000)
            : params.custom?.refreshTokenExpireAt
                ? new Date(params.custom.refreshTokenExpireAt * 1000)
                : new Date(time + 3600000 * 24 * 7);
        let dbResult = await trx.query(q, qParams);
        let tokenId = null;
        if (dbResult.rowCount) {
            tokenId = dbResult.rows[0].id;
            await trx.query(`
						UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
						SET "accessTokenExpiresOn" = $2,
							"refreshTokenExpiresOn" = $3,
							"userId" = $4,
							"grantedAt" = $5,
							"ip" = $6,
							"userAgent" = $7,
							"organizationAuths" = $8,
							"scopes" = $9,
							"oAuthClientId" = $10,
							"mobileInstallId" = $11,
							"updatedAt" = $12,
							"refreshToken" = $13
						WHERE id = $1
					`, [
                tokenId,
                accessTokenExpireAt,
                refreshTokenExpireAt,
                params.user.id,
                now,
                params.ip,
                params.userAgent,
                JSON.stringify(params.user.organizationAuths),
                params.scopes ? JSON.stringify(params.scopes) : null,
                params.clientId,
                params.mobileInstallId ?? params.userAgentUniqueId,
                now,
                uuid_1.default.v4(),
            ]);
        }
        else {
            tokenId = uuid_1.default.v4();
            await trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
						(	
							id, 
							"createdAt", 
							"updatedAt", 
							"accessTokenExpiresOn", 
							"refreshToken", 
							"refreshTokenExpiresOn", 
							"userId", 
							"grantedAt", 
							ip, 
							"userAgent", 
							"organizationAuths", 
							scopes, 
							"oAuthClientId", 
							"mobileInstallId"
						)
						VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14);
					`, [
                tokenId,
                now,
                now,
                accessTokenExpireAt,
                uuid_1.default.v4(),
                refreshTokenExpireAt,
                params.user.id,
                now,
                params.ip,
                params.userAgent,
                JSON.stringify(params.user.organizationAuths),
                params.scopes ? JSON.stringify(params.scopes) : null,
                params.clientId,
                params.mobileInstallId ?? params.userAgentUniqueId,
            ]);
        }
        token = (await trx.query(`
					SELECT * 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
					WHERE id = $1
				`, [tokenId])).rows[0];
        if (token) {
            return Promise.resolve(token);
        }
        else {
            throw new Error(`Token Not found while upsertClientToken, probably query error`);
        }
    }
    async getUserIdFromRefreshToken(organizationId, refreshToken, trx) {
        const userId = await trx.query(`SELECT "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}" WHERE "refreshToken" = $1`, [refreshToken]);
        return userId.rowCount > 0 ? userId.rows[0].userId : null;
    }
    async refreshToken(organizationId, params, trx) {
        let datetime = new Date();
        let token = null;
        token = (await trx.query(`
				SELECT *
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
				WHERE id = $1
					AND "refreshToken" = $2
				LIMIT 1;
			`, [params.authTokenId, params.refreshToken])).rows[0];
        if (!token) {
            (0, dal_access_error_1.throwDbAccessBadRequestError)("Unable to find refresh token");
        }
        else if (token.refreshTokenExpiresOn < datetime) {
            (0, dal_access_error_1.throwDbAccessUnauthorizedError)("Refresh token timed out, login again");
        }
        if ((!params.mobileInstallId && !params.userAgentUniqueId && token.userAgent != params.userAgent) ||
            (params.mobileInstallId &&
                token.mobileInstallId &&
                params.mobileInstallId !== token.mobileInstallId) ||
            (params.userAgentUniqueId &&
                token.mobileInstallId &&
                params.userAgentUniqueId !== token.mobileInstallId)) {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
						SET "deletedAt" = now(), reason = 'intruder alert'
						WHERE id = $1;`, [token.id]);
            });
            throw (0, api_error_1.generateForbiddenError)({ message: "Intruder alert" });
        }
        let revoke = await this.hasRevokedToken(organizationId, token.id);
        if (revoke) {
            (0, dal_access_error_1.throwDbAccessUnauthorizedError)("The token is revoked! Login again!");
        }
        const newRefreshToken = uuid_1.default.v4();
        await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
				SET "refreshedAt" = $2,
					"mobileInstallId" = $3,
					"accessTokenExpiresOn" = $4,
					"refreshTokenExpiresOn" = $5,
					"refreshToken" = $6
				WHERE id = $1
			`, [
            token.id,
            datetime,
            params.mobileInstallId ?? params.userAgentUniqueId,
            new Date(params.expirations.accessTokenExpireAt * 1000),
            new Date(params.expirations.refreshTokenExpireAt * 1000),
            newRefreshToken,
        ]);
        token.refreshToken = newRefreshToken;
        let armonApplicationUser = await this.getArmonApplicationUser(organizationId, token.userId, trx);
        if (!armonApplicationUser?.userOrganization?.length) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.USERNAMEORPASSWORDWRONG");
        }
        let user = this.mapToArmonApplicationUser(armonApplicationUser);
        const result = {
            token: token,
            user: user,
        };
        return result;
    }
    async getArmonApplicationUser(organizationId, userId, trx) {
        let userDbResult = await trx.query(`
				SELECT u."id", u."username", u."accountEnabled", u."settings", up."name", up."surname", up."email", up."phoneNumber", up."mfaSettings"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
				LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up
					ON u.id = up."userId"
				WHERE u."deletedAt" IS NULL
					AND up."deletedAt" IS NULL
					AND u.id = $1
			`, [userId]);
        if (!userDbResult.rowCount) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("The user is not found");
        }
        const user = userDbResult.rows[0];
        let result = {
            id: user.id,
            username: user.username,
            userProfile: {
                name: user.name,
                surname: user.surname,
                locale: user.settings.locale,
                isDisabled: !user.accountEnabled,
                mfaSettings: user.mfaSettings,
                email: user.email,
                phoneNumber: user.phoneNumber,
            },
            userOrganization: [],
        };
        const userOrganizationDataDbResult = await trx.query(`
			SELECT 
				uo.id,
				uop."name",
				uop."surname",
				uo."roleId",
				u.settings->>'locale' AS "locale",
				r."permissions",
				uo."extraPermissions",
				uo."isDisabled",
				o.id as "organizationId",
				o.name as "organizationName",
				o.code as "organizationCode",
				o."hasEmergencyManagement",
				o."availableCredentialTypes",
				o."customCredentialTypes",
				o."credentialTypes",
				o."uniqueIdFieldCaption",
				opms."deputySelectionEnabled",
				opms."canUseSystemPPermissionPrint",
				o."userOrganizationProfileSettings",
				o."settings",
				(
					case 
						when length(uo."extraPermissions") > 2 then replace(r."permissions" || uo."extraPermissions", '][', ',') 
							else r."permissions" end
				)::json as "userOrganizationPermissions",
				df.id as "organizationUnitDynamicFormId",
				o."userFilterAccessRightActivation"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" AS u
				ON uo."userId" = u.id AND uo."deletedAt" IS NULL AND u."deletedAt" IS NULL AND uo."isDisabled" IS FALSE
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}" AS o
				ON uo."organizationId" = o.id AND o."deletedAt" IS NULL
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" AS r
				ON uo."roleId" = r.id
			LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.dynamicForms}" AS df
				ON df."id" = o."organizationUnitDynamicFormId"
			LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationPACSModuleSettings}" AS opms
				ON opms."organizationId" = o.id
			LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
				ON uop."userOrganizationId" = uo.id
			WHERE uo."userId" = $1
			`, [userId]);
        const userOrganizationData = userOrganizationDataDbResult.rows[0];
        if (!userOrganizationData) {
            return null;
        }
        const organizationModulesDbResult = await trx.query(`
				SELECT "module", "enableUtc", TRUE AS "enabled"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules}"
				WHERE "status" = $1
			`, [dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active]);
        const organizationUnitDynamicForm = await dal_manager_1.dbManager.accessSystem.getDynamicForm({
            organizationId: organizationId,
            dynamicFormId: userOrganizationData.organizationUnitDynamicFormId,
            trx,
        });
        const visitorModuleSettingsDbResult = await trx.query(`
				SELECT settings
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorModuleSettings}"
			`, []);
        let visitorModuleSettings = visitorModuleSettingsDbResult.rows[0]?.settings ?? {};
        let visitorRegistrationPoints = await dal_manager_1.dbManager.accessVisitor.listRegistrationPointsDetailed(organizationId, userId, trx);
        const organizationFormsDbResult = await trx.query(`
				SELECT type, fields
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationForms}"
			`);
        result.userOrganization.push({
            id: userOrganizationData.id,
            organizationProfile: {
                name: userOrganizationData.name,
                surname: userOrganizationData.surname,
                locale: userOrganizationData.locale,
            },
            roleId: userOrganizationData.roleId,
            userOrganizationPermissions: userOrganizationData.userOrganizationPermissions,
            isDisabled: userOrganizationData.isDisabled,
            organization: {
                id: userOrganizationData.organizationId,
                name: userOrganizationData.organizationName,
                code: userOrganizationData.organizationCode,
                userExpressionType: userOrganizationData.settings.userExpression,
                forms: organizationFormsDbResult.rows,
                hasEmergencyManagement: userOrganizationData.hasEmergencyManagement,
                availableCredentialTypes: userOrganizationData.availableCredentialTypes,
                credentialTypes: userOrganizationData.credentialTypes,
                uniqueIdFieldCaption: userOrganizationData.uniqueIdFieldCaption,
                integrations: userOrganizationData.settings?.integrations,
                modules: organizationModulesDbResult.rows.map((i) => i.module),
                organizationUnitDynamicForm: organizationUnitDynamicForm,
                visitorModuleSettings: {
                    visitorProfileFormFields: visitorModuleSettings.visitorProfileFormFields,
                    availableCredentials: visitorModuleSettings.availableCredentials ? visitorModuleSettings.availableCredentials.availableCredentialTypes : null,
                    visitFormFields: visitorModuleSettings.visitFormFields,
                    visitorProfileFormFieldSearchEndPoint: visitorModuleSettings.visitorProfileFormFieldSearchEndPoint,
                    visitorProfileFormSearchEndPoint: visitorModuleSettings.visitorProfileFormSearchEndPoint,
                    visitFormFieldSearchEndPoint: visitorModuleSettings.visitFormFieldSearchEndPoint,
                    preRegistrationAvailable: visitorModuleSettings.preRegistrationAvailable,
                    forceToSelectRegistrationPoint: visitorModuleSettings.forceToSelectRegistrationPoint,
                    maxExpectedDayCount: visitorModuleSettings.maxExpectedDayCount,
                    visitorRegistrationPoints: visitorRegistrationPoints.map((m) => {
                        const registrationPoint = {
                            accessControlPoints: m.accessControlPoints,
                            isGDPRRequired: m.tabletDeviceId ? true : false,
                            name: m.name,
                            id: m.id,
                            deviceName: undefined,
                            isConnected: undefined,
                        };
                        if (m.tabletDeviceId) {
                            (registrationPoint.deviceName = m.deviceName), (registrationPoint.isConnected = m.isConnected);
                        }
                        return registrationPoint;
                    }),
                    visitFormFinalizedEndPoints: visitorModuleSettings.visitFormFinalizedEndPoints,
                    gdprEnabled: visitorModuleSettings.gdprEnabled,
                    qrCodeEnabled: visitorModuleSettings.qrCodeEnabled === true,
                    printableVisitorCard: visitorModuleSettings.printableVisitorCard,
                    anonymizationDuration: visitorModuleSettings.gdprSettings ? visitorModuleSettings.gdprSettings.anonymizationDuration : 0,
                    canBeDeleted: visitorModuleSettings.gdprSettings ? visitorModuleSettings.gdprSettings.canBeDeleted : false,
                    approvementTimeout: visitorModuleSettings.gdprSettings ? visitorModuleSettings.gdprSettings.approvementTimeout : 0,
                },
                pacsModuleSettings: {
                    enabledUtc: organizationModulesDbResult.rows.some((i) => i.module === dal_constants_1.DalConstants.ArmonApplicationModule.PACS && i.enabled)
                        ? organizationModulesDbResult.rows.find((i) => i.module === dal_constants_1.DalConstants.ArmonApplicationModule.PACS).enableUtc
                        : null,
                    canUseSystemPPermissionPrint: userOrganizationData.canUseSystemPPermissionPrint,
                    deputySelectionEnabled: userOrganizationData.deputySelectionEnabled,
                },
                userOrganizationProfileSettings: userOrganizationData.userOrganizationProfileSettings
                    ? userOrganizationData.userOrganizationProfileSettings
                    : dal_constants_1.DalConstants.defaultUserProfileSettings,
                customCredentialTypes: userOrganizationData.customCredentialTypes ? userOrganizationData.customCredentialTypes : [],
                userFilterAccessRightActivation: userOrganizationData.userFilterAccessRightActivation,
                multiFactorAuthenticationSettings: userOrganizationData.settings.multiFactorAuthenticationSettings,
            },
            organizationUnits: await this.getUserOrganizationUnits(userOrganizationData.organizationId, userOrganizationData.id, trx),
        });
        return result;
    }
    async getUserOrganizationUnits(organizationId, userOrganizationId, trx) {
        let organizationUnits = [];
        let result = await trx.query(`
			SELECT 
				uoou."organizationUnitId" AS id,
				ou.name,
				"uoou"."roleId", 
				"r"."permissions"::json, 
				S."deputyId", 
				S."principalPermissions"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as "uoou"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
				ON ou.id = uoou."organizationUnitId" 
					AND ou."deletedAt" IS NULL 
					AND uoou."deletedAt" IS NULL
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" as "r" 
                ON "r"."id" = "uoou"."roleId" 
					AND r."deletedAt" IS NULL
			LEFT JOIN (
				SELECT  
                	"uod"."id" as "deputyId", 
					"r2"."permissions"::json as "principalPermissions", 
					"uod"."deputyUserOrganizationId", 
					"uod"."organizationUnitId" 
                FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationDeputies}" as "uod" 
                LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as "uoou2" 
					ON "uoou2"."userOrganizationId" = "uod"."userOrganizationId"
						AND uoou2."deletedAt" IS NULL
                LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as "ou2" 
					ON "ou2"."id" = "uoou2"."organizationUnitId"
						AND ou2."deletedAt" IS NULL
                LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" as "r2" 
					ON "r2"."id" = "uoou2"."roleId"
						AND r2."deletedAt" IS NULL
                WHERE uod."startDateTime" <= now()  
					AND uod."endDateTime" >= now()
			) AS S
				ON S."deputyUserOrganizationId" = "uoou"."userOrganizationId"
           			AND  "uoou"."organizationUnitId" = S."organizationUnitId"
			WHERE "uoou"."userOrganizationId" = $1
			`, [userOrganizationId]);
        if (result.rowCount) {
            for (let row of result.rows) {
                organizationUnits.push({
                    id: row.id,
                    name: row.name,
                    roleId: row.roleId,
                    permissions: this.mergePermissions(row.permissions, row.principalPermissions),
                    hasDeputy: row.deputyId ? true : false,
                });
            }
        }
        return organizationUnits;
    }
    mergePermissions(permissions, principalPermissions) {
        if (!principalPermissions)
            return permissions;
        if (!permissions)
            return [];
        return permissions.concat(principalPermissions).filter((v, i, a) => a.indexOf(v) === i);
    }
    mapToArmonApplicationUser(user) {
        return {
            id: user.id,
            userProfile: !user.userProfile
                ? null
                : {
                    name: user.userProfile.name,
                    surname: user.userProfile.surname,
                    email: user.userProfile.email,
                    phoneNumber: user.userProfile.phoneNumber,
                    isDisabled: user.userProfile.isDisabled,
                    locale: user.userProfile.locale,
                    mfaSettings: user.userProfile.mfaSettings,
                },
            expirationUtc: user.expirationUtc,
            organizations: !user.userOrganization
                ? []
                : user.userOrganization.map((uo) => {
                    return {
                        organizationProfile: {
                            name: uo.organizationProfile.name,
                            surname: uo.organizationProfile.surname,
                            locale: uo.organizationProfile.locale,
                        },
                        organization: {
                            id: uo.organization.id,
                            name: uo.organization.name,
                            code: uo.organization.code,
                            options: {
                                userExpression: uo.organization.userExpressionType,
                            },
                            hasEmergencyManagement: uo.organization.hasEmergencyManagement,
                            forms: !uo.organization.forms
                                ? []
                                : uo.organization.forms.map((uof) => {
                                    return {
                                        type: uof.type,
                                        fields: uof.fields,
                                    };
                                }),
                            usernamePolicy: dal_constants_1.DalConstants.OrganizationUsernamePolicy.FreeAlphanumeric,
                            availableCredentialTypes: uo.organization.availableCredentialTypes,
                            credentialTypes: uo.organization.credentialTypes,
                            uniqueIdFieldCaption: uo.organization.uniqueIdFieldCaption,
                            integrations: uo.organization.integrations,
                            modules: uo.organization.modules,
                            visitorModuleSettings: uo.organization.visitorModuleSettings,
                            pacsModuleSettings: uo.organization.pacsModuleSettings,
                            userOrganizationProfileSettings: uo.organization.userOrganizationProfileSettings,
                            customCredentialTypes: uo.organization.customCredentialTypes,
                            organizationUnitDynamicForm: uo.organization.organizationUnitDynamicForm,
                            userFilterAccessRightActivation: uo.organization.userFilterAccessRightActivation,
                            multiFactorAuthenticationSettings: uo.organization.multiFactorAuthenticationSettings,
                        },
                        isDisabled: uo.isDisabled,
                        roleId: uo.roleId,
                        extraPermissions: [],
                        organizationUnits: !uo.organizationUnits
                            ? []
                            : uo.organizationUnits.map((apuso) => {
                                return {
                                    id: apuso.id,
                                    name: apuso.name,
                                    roleId: apuso.roleId,
                                    hasDeputy: apuso.hasDeputy,
                                    extraPermissions: [],
                                };
                            }),
                    };
                }),
            organizationAuths: user.userOrganization.map((oas) => {
                return {
                    i: oas.organization.id,
                    d: oas.isDisabled,
                    p: oas.userOrganizationPermissions || [],
                    r: oas.roleId,
                    u: !oas.organizationUnits
                        ? []
                        : oas.organizationUnits.map((uuo) => {
                            return {
                                i: uuo.id,
                                p: uuo.permissions || [],
                            };
                        }),
                };
            }),
            applicationSettings: null,
            scopes: [],
        };
    }
    async getArmonApplicationUserByPassword(organizationId, username, password, trx) {
        const userDbResult = await trx.query(`
				SELECT * FROM
				"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
				WHERE lower("username") = lower($1);
			`, [username]);
        if (!userDbResult.rowCount) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.USERNAMEORPASSWORDWRONG");
        }
        const user = userDbResult.rows[0];
        let saltedPassword = (0, dal_utils_1.hashPasswordWithSalt)(password, user.salt);
        if (saltedPassword.passwordHash !== user.hashedPassword) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.USERNAMEORPASSWORDWRONG");
        }
        let armonApplicationUser = await this.getArmonApplicationUser(organizationId, user.id, trx);
        if (!armonApplicationUser?.userOrganization?.length) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.USERNAMEORPASSWORDWRONG");
        }
        const result = this.mapToArmonApplicationUser(armonApplicationUser);
        return result;
    }
    async getArmonApplicationUserByUsername(organizationId, username, trx) {
        const userDbResult = await trx.query(`
			SELECT * 
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
			WHERE "accountEnabled" IS TRUE
				AND lower (username) = lower($1)
			`, [username]);
        if (!userDbResult.rowCount) {
            throw (0, api_error_1.generateBadRequestApiError)({ message: "Username or password is wrong!" }, username);
        }
        let armonApplicationUser = await this.getArmonApplicationUser(organizationId, userDbResult.rows[0].id, trx);
        if (!armonApplicationUser?.userOrganization?.length) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.USERNAMEORPASSWORDWRONG");
        }
        let user = this.mapToArmonApplicationUser(armonApplicationUser);
        return user;
    }
    async revokePreviouslyLoggedInDeviceTokens(organizationId, userId, clientId, mobileInstallId, trx) {
        if (mobileInstallId) {
            let oAuthTokenResults = await trx.query(`SELECT oat."id" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}" AS oat
				WHERE "oAuthClientId" IS NOT NULL AND
					oat."deletedAt" IS NULL AND
					oat."userId" = $1 AND 
					(oat."mobileInstallId" != $2 OR oat."oAuthClientId" != $3)`, [userId, mobileInstallId, clientId]);
            if (oAuthTokenResults.rowCount > 0) {
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
					SET "deletedAt" = now(), reason = 'mobile install id / client id mismatch'
					WHERE id = ANY ($1::UUID[]);`, [oAuthTokenResults.rows.map((r) => r.id)]);
            }
        }
    }
    async getLatestMobileInstallIdentifier(organizationId, userId, trx) {
        const deviceIds = await trx.query(`SELECT "mobileInstallId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
			WHERE "userId" = $1 AND ("oAuthClientId" = $2 OR "oAuthClientId" = $3)
			ORDER BY "createdAt" DESC LIMIT 1;`, [userId, predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileIOSClient.id, predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileAndroidClient.id]);
        return deviceIds.rowCount > 0 ? deviceIds.rows[0].mobileInstallId : null;
    }
    async clearMobileInstallIdentifiers(organizationId, userId, trx) {
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
			SET "mobileInstallId" = NULL
			WHERE "userId" = $1;`, [userId]);
    }
}
exports.PSQLDalAccessOAuth = PSQLDalAccessOAuth;
