"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessMobileClient = void 0;
const dal_constants_1 = require("../../dal.constants");
const dal_access_rdb_mobileClient_1 = require("../rdb/dal.access.rdb.mobileClient");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const crypto_1 = __importDefault(require("crypto"));
const dal_access_error_1 = require("../dal.access.error");
class PSQLDalAccessMobileClient extends dal_access_rdb_mobileClient_1.RDBDalAccessMobileClient {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getDevicesOfOrganization(organizationId) {
        let devices = await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.devices)
            .where("organizationId", organizationId)
            .whereNull("deletedAt")
            .whereNull("adapterId")
            .whereIn("brand", [dal_constants_1.DalConstants.DeviceBrand.Armon, dal_constants_1.DalConstants.DeviceBrand.AssaAbloy, dal_constants_1.DalConstants.DeviceBrand.HikVision])
            .whereIn("model", dal_constants_1.DalConstants.terminalModels)
            .select("id", "brand", "model", "name", "location", "publicKey", "certExpDate");
        let organization = await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations).whereNull("deletedAt").where("id", organizationId).first();
        if (!organization) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("organization is not found");
        }
        let result = [];
        for (const device of devices) {
            let hash = crypto_1.default.createECDH("prime256v1");
            hash.setPrivateKey(Buffer.from(organization.privateKey, "base64"));
            let secret = hash.computeSecret(Buffer.from(device.publicKey, "base64"));
            result.push(Object.assign(device, { organizationPublicKey: organization.publicKey }, { secret: secret.toString("base64") }));
        }
        return result;
    }
    async getAccessRights(organizationId, userId, pagination, filterAuthenticationFactors) {
        return this.dbClient.transaction(async (trx) => {
            let result = {
                pagination: {
                    take: pagination.take,
                    skip: pagination.skip,
                    total: 0,
                },
                items: [],
            };
            let qb = this.dbClient
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights + " as uar")
                .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints + " as acp", "uar.accessControlPointId", "acp.id")
                .whereNull("acp.deletedAt")
                .whereNull("uar.deletedAt")
                .where("uar.access", true)
                .where("acp.organizationId", organizationId)
                .where("uar.userId", userId)
                .whereNull("uar.deletedAt")
                .whereNull("acp.deletedAt")
                .transacting(trx);
            if (filterAuthenticationFactors && filterAuthenticationFactors.length > 0) {
                qb.where((qbOr) => {
                    let first = false;
                    for (let factor of filterAuthenticationFactors) {
                        if (factor === dal_constants_1.DalConstants.MobileAuthenticationFactor.AccessToken) {
                            if (first) {
                                qbOr = qbOr.where("remoteAccess", true);
                            }
                            else {
                                qbOr = qbOr.orWhere("remoteAccess", true);
                            }
                        }
                        else {
                            if (first) {
                                qbOr = qbOr.whereRaw(`"authenticationFactors"::text SIMILAR TO '%("factor":${factor})%'`);
                            }
                            else {
                                qbOr = qbOr.orWhereRaw(`"authenticationFactors"::text SIMILAR TO '%("factor":${factor})%'`);
                            }
                        }
                    }
                });
            }
            let count = await qb
                .clone()
                .count("uar.id")
                .then((rows) => {
                return parseInt(rows[0].count);
            });
            result.pagination.total = count;
            let selectColumns = [
                "uar.id",
                "uar.userId",
                "uar.accessControlPointId",
                "uar.remoteAccess",
                "acp.organizationId",
                "acp.deviceId",
                "acp.authenticationFactors",
                "acp.name as accessControlPointName",
                "acp.location as accessControlPointLocation",
                "d.brand",
                "d.name as deviceName",
                "d.location as deviceLocation",
                "d.model as deviceModel",
                "d.publicKey",
            ];
            qb.leftOuterJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.devices + " as d", "d.id", "acp.deviceId")
                .clearSelect()
                .select(selectColumns)
                .orderBy("acp.name")
                .offset(pagination.skip)
                .limit(pagination.take);
            result.items = await qb.then((rows) => {
                return rows.map((row) => {
                    return {
                        accessControlPoint: {
                            authenticationFactors: row.authenticationFactors,
                            id: row.accessControlPointId,
                            location: row.accessControlPointLocation,
                            name: row.accessControlPointName,
                        },
                        device: {
                            brand: row.brand,
                            id: row.deviceId,
                            location: row.deviceLocation,
                            model: row.deviceModel,
                            name: row.deviceName,
                            publicKey: row.publicKey,
                        },
                    };
                });
            });
            return Promise.resolve(result);
        });
    }
}
exports.PSQLDalAccessMobileClient = PSQLDalAccessMobileClient;
