"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.assignAutoShift = exports.PSQLDalAccessLog = void 0;
const enums_1 = require("../../../lib/access-model/v2/enums");
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const moment_1 = __importDefault(require("moment"));
const sharp_1 = __importDefault(require("sharp"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const business_device_1 = require("../../../business/business.device");
const report_util_1 = require("../../../business/report/report.util");
const dynamicFormCurrent_1 = require("../../../lib/es/models/dynamicFormCurrent");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_models_1 = require("../dal.access.models");
const dal_access_cache_redis_1 = require("../redis/dal.access.cache.redis");
const restapi_1 = require("../../../lib/es/models/restapi");
const Cursor = require("pg-cursor");
class PSQLDalAccessLog {
    constructor(knex, pgPool) {
        this._dbClient = knex;
        this._pgPool = pgPool;
    }
    get pgPool() {
        return this._pgPool;
    }
    get dbClient() {
        return this._dbClient;
    }
    async addMobileCheckinAccessLog(params) {
        const { logRequest, organizationId, trx } = params;
        const basicUserInfo = await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, logRequest.userId);
        const log = {
            id: uuid_1.default.v4(),
            rm: app_enums_1.enums.LogReceiveMethod.Mobile,
            u: logRequest.actionUtc.toISOString(),
            oId: organizationId,
            a: logRequest.accessControlPointId,
            r: logRequest.reason,
            o: logRequest.userId,
            i: logRequest.accessRightId,
            c: null,
            ci: null,
            cx: [
                {
                    t: dal_constants_1.DalConstants.libEnumsV3.CredentialType.AccessToken,
                    d: "Mobile",
                },
            ],
            re: null,
            rx: null,
            cid: null,
            d: logRequest.direction,
            an: logRequest.accessControlPointName || null,
            on: basicUserInfo.fullname,
            di: null,
            ir: false,
            iq: logRequest.isQrTriggered,
            m: logRequest.requesterUserId || null,
            ul: logRequest.userRoleId,
            rg: null,
            s: true,
            v: false,
            ss: null,
            cs: null,
            lt: logRequest.location?.latitude || null,
            ln: logRequest.location?.longitude || null,
            lr: logRequest.location?.isReliable !== undefined ? logRequest.location?.isReliable : null,
        };
        await Promise.all([
            trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" 
				(id, "actionUtc", "organizationId", log, "credentialData")
				VALUES ($1, $2, $3, $4, $5);
				`, [log.id, log.u, organizationId, log, null]),
            trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory}" 
				(id, "insertionUtc", "actionUtc", type, data)
				VALUES ($1, $2, $3, $4, $5);
				`, [log.id, new Date().toISOString(), log.u, dal_constants_1.DalConstants.LogHistoryType.AccessLog, null]),
        ]);
        return log.id;
    }
    async addArventoDriverAccessLog(organizationId, logRequest, trx) {
        const basicUserInfo = await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, logRequest.userId);
        const log = {
            id: uuid_1.default.v4(),
            rm: app_enums_1.enums.LogReceiveMethod.Mobile,
            u: logRequest.actionUtc.toISOString(),
            oId: organizationId,
            a: logRequest.accessControlPointId,
            r: logRequest.reason,
            o: logRequest.userId,
            i: null,
            c: null,
            ci: null,
            cx: [
                {
                    t: dal_constants_1.DalConstants.libEnumsV3.CredentialType.AccessToken,
                    d: "Mobile",
                },
            ],
            re: null,
            rx: null,
            cid: null,
            d: logRequest.direction,
            an: logRequest.accessControlPointName || null,
            on: basicUserInfo.fullname,
            di: null,
            ir: false,
            iq: logRequest.isQrTriggered,
            m: logRequest.requesterUserId || null,
            ul: null,
            rg: null,
            s: true,
            v: false,
            ss: null,
            cs: null,
            lt: logRequest.location?.latitude || null,
            ln: logRequest.location?.longitude || null,
            lr: logRequest.location?.isReliable !== undefined ? logRequest.location?.isReliable : null,
            lp: logRequest.licencePlate,
        };
        await Promise.all([
            trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" 
				(id, "actionUtc", "organizationId", log, "credentialData")
				VALUES ($1, $2, $3, $4, $5);
				`, [log.id, log.u, organizationId, log, null]),
            trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogsArvento}" 
				("accessLogId", "actionUtc", "updatedAt", type, "isSynchronized", data, "accessControlPointId")
				VALUES ($1, $2, $2, $3, $4, $5, $6);
				`, [log.id, log.u, restapi_1.ArventoAccessLogType.Arvento, false, { licencePlate: log?.lp }, logRequest.accessControlPointId]),
            trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory}" 
				(id, "insertionUtc", "actionUtc", type, data)
				VALUES ($1, $2, $3, $4, $5);
				`, [log.id, new Date().toISOString(), log.u, dal_constants_1.DalConstants.LogHistoryType.AccessLog, null]),
        ]);
        return log.id;
    }
    async addArventoDriverAccessLogBulk(organizationId, logRequests, trx) {
        const now = new Date().toISOString();
        const accessLogTuples = logRequests.map((req) => [req.userId, req.actionUtc.toISOString(), req.direction]);
        const existing = await trx.query(`
				SELECT a.log->>'o' as "userId", a."actionUtc", a.log->>'d' as "direction"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" a
				JOIN unnest($1::text[], $2::timestamptz[], $3::text[]) 
				AS t("userId", "actionUtc", "direction")
				ON a.log->>'o' = t."userId"
				AND a."actionUtc" = t."actionUtc"
				AND a.log->>'d' = t."direction"
			`, [
            accessLogTuples.map((t) => t[0]),
            accessLogTuples.map((t) => t[1]),
            accessLogTuples.map((t) => t[2]),
        ]);
        const existingKeySet = new Set(existing.rows.map((row) => `${row.userId}_${row.actionUtc.toISOString()}_${row.direction}`));
        const newRequests = logRequests.filter((req) => {
            const key = `${req.userId}_${req.actionUtc.toISOString()}_${req.direction}`;
            return !existingKeySet.has(key);
        });
        if (newRequests.length === 0)
            return true;
        const userIds = [...new Set(newRequests.map((r) => r.userId))];
        const infos = await dal_manager_1.dbManager.accessUser.getBasicUserInfoPg({ organizationId, userIds, trx });
        const userInfoMap = new Map(infos.map((u) => [u.id, u]));
        const accessLogs = [];
        const arventoLogs = [];
        const logHistories = [];
        for (const req of newRequests) {
            const id = uuid_1.default.v4();
            const fullname = userInfoMap.get(req.userId)?.fullname || null;
            const actionUtc = req.actionUtc.toISOString();
            const log = {
                id,
                rm: app_enums_1.enums.LogReceiveMethod.Mobile,
                u: actionUtc,
                oId: organizationId,
                a: req.accessControlPointId,
                r: req.reason,
                o: req.userId,
                i: null,
                c: null,
                ci: null,
                cx: [
                    {
                        t: dal_constants_1.DalConstants.libEnumsV3.CredentialType.AccessToken,
                        d: "Mobile",
                    },
                ],
                re: null,
                rx: null,
                cid: null,
                d: req.direction,
                an: req.accessControlPointName || null,
                on: fullname,
                di: null,
                ir: false,
                iq: req.isQrTriggered,
                m: req.requesterUserId || null,
                ul: null,
                rg: null,
                s: true,
                v: false,
                ss: null,
                cs: null,
                lt: req.location?.latitude || null,
                ln: req.location?.longitude || null,
                lr: req.location?.isReliable !== undefined ? req.location?.isReliable : null,
                lp: req.licencePlate,
            };
            accessLogs.push([id, actionUtc, organizationId, log, null]);
            arventoLogs.push([id, actionUtc, actionUtc, restapi_1.ArventoAccessLogType.Arvento, false, { licencePlate: log.lp }, req.accessControlPointId]);
            logHistories.push([id, now, actionUtc, dal_constants_1.DalConstants.LogHistoryType.AccessLog, null]);
        }
        await Promise.all([
            trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" (id, "actionUtc", "organizationId", log, "credentialData")
					SELECT * FROM UNNEST ($1::uuid[], $2::timestamptz[], $3::uuid[], $4::jsonb[], $5::jsonb[])
				`, [accessLogs.map((r) => r[0]), accessLogs.map((r) => r[1]), accessLogs.map((r) => r[2]), accessLogs.map((r) => r[3]), accessLogs.map((r) => r[4])]),
            trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogsArvento}" 
						("accessLogId", "actionUtc", "updatedAt", type, "isSynchronized", data, "accessControlPointId")
					SELECT * FROM UNNEST ($1::uuid[], $2::timestamptz[], $3::timestamptz[], $4::int[], $5::boolean[], $6::jsonb[], $7::uuid[])
				`, [
                arventoLogs.map((r) => r[0]),
                arventoLogs.map((r) => r[1]),
                arventoLogs.map((r) => r[2]),
                arventoLogs.map((r) => r[3]),
                arventoLogs.map((r) => r[4]),
                arventoLogs.map((r) => r[5]),
                arventoLogs.map((r) => r[6]),
            ]),
            trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory}" 
						(id, "insertionUtc", "actionUtc", type, data)
					SELECT * FROM UNNEST ($1::uuid[], $2::timestamptz[], $3::timestamptz[], $4::int[], $5::jsonb[])
				`, [logHistories.map((r) => r[0]), logHistories.map((r) => r[1]), logHistories.map((r) => r[2]), logHistories.map((r) => r[3]), logHistories.map((r) => r[4])]),
        ]);
        return true;
    }
    async getArventoAccessLogsWithNodeId(organizationId, trx) {
        const result = await trx.query(`
				SELECT ala.*, acp."arventoConfiguration"->>'nodeId' as "arventoNodeId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogsArvento}" ala
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp ON acp.id = ala."accessControlPointId"
				WHERE "isSynchronized" = false;
			`);
        return result.rows;
    }
    async getArventoAccessLogs(organizationId, trx, type, isSynchronized) {
        let query = `
			SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogsArvento}" ala
		`;
        const queryParams = [];
        let whereClauses = [];
        if (type !== null) {
            whereClauses.push("type = $1");
            queryParams.push(type);
        }
        if (isSynchronized !== null) {
            whereClauses.push('"isSynchronized" = $2');
            queryParams.push(isSynchronized);
        }
        if (whereClauses.length > 0) {
            query += " WHERE " + whereClauses.join(" AND ");
        }
        const result = await trx.query(query, queryParams);
        return result.rows;
    }
    async updateArventoAccessLog(organizationId, accessLogId, region, trx) {
        await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogsArvento}"
				SET
					"isSynchronized" = true,
					"data" = jsonb_set("data", '{region}', to_jsonb($1::text), true),
					"updatedAt" = now(),
					"type" = $3
				WHERE id = $2;
			`, [region, accessLogId, restapi_1.ArventoAccessLogType.ArventoRegion]);
    }
    async getAccessLogForSensorReport(params) {
        const trxx = params.trx || this.pgPool;
        const result = (await trxx.query(`
					SELECT log FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
					WHERE id = $3
				`, [params.l])).rows.map((r) => r.log);
        return Promise.resolve(result[0]);
    }
    async getAccessLogsForSensorReport(params) {
        const result = (await params.trx.query(`
			SELECT log FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
			WHERE "actionUtc" >= $1 AND
				"actionUtc" <= $2 AND
				id = ANY($3::UUID[])
		`, [params.dateRange.startDateTime, params.dateRange.endDateTime, params.ids])).rows.map((r) => r.log);
        return Promise.resolve(result);
    }
    async getLatestSensorLogs(params) {
        const trxx = params.trx ?? this.pgPool;
        let startOfDay = (0, moment_1.default)(new Date()).endOf("day");
        let controlMaxDay = (0, moment_1.default)(startOfDay).add(-30, "days");
        const statusSensors = params.sensors.filter((s) => s.type === dal_constants_1.DalConstants.DryContactTypeV2.StatusSensor);
        const counterSensors = params.sensors.filter((s) => s.type === dal_constants_1.DalConstants.DryContactTypeV2.CounterSensor);
        const tamperSwitches = params.sensors.filter((s) => s.type === dal_constants_1.DalConstants.DryContactTypeV2.TamperSwitch);
        const sensorResult = await trxx.query(`
			SELECT S.log, S.id FROM 
			(
				SELECT ssl."log", ssl."log"->>'i' AS "id", ROW_NUMBER() OVER(PARTITION BY ssl."log"->>'i' ORDER BY ssl."actionUtc", ssl."log"->>'s' DESC) AS rn
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs}" AS ssl
				WHERE ssl."actionUtc" >= $1
					AND ssl."actionUtc" <= $2
					AND ssl."log"->>'i' = ANY ($3::text[])
					
			) AS S
			WHERE S.rn = 1
			UNION ALL

			SELECT C.log, C.id FROM 
			(
				SELECT csl."log", csl."log"->>'i' AS "id", ROW_NUMBER() OVER(PARTITION BY csl."log"->>'i' ORDER BY csl."actionUtc" DESC) AS rn
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs}" AS csl
				WHERE csl."actionUtc" >= $1
					AND csl."actionUtc" <= $2
					AND csl."log"->>'i' = ANY ($4::text[])
			) AS C
			WHERE C.rn = 1

			UNION ALL

			SELECT T.log, T.id FROM 
			(
				SELECT tsl.log, ddci.id::text, ROW_NUMBER() OVER(PARTITION BY tsl."log"->>'ci' ORDER BY tsl."actionUtc" DESC) AS rn
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.TamperSwitchLogs}" AS tsl
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceDryContactInputs}" AS ddci ON ddci."deviceId"::text = tsl."log"->>'ci'
				WHERE tsl."actionUtc" >= $1
					AND tsl."actionUtc" <= $2
					AND ddci.id = ANY ($5::uuid[])
			) AS T
			WHERE T.rn = 1;
		`, [controlMaxDay.toDate(), startOfDay.toDate(), statusSensors.map((ss) => ss.id), counterSensors.map((cs) => cs.id), tamperSwitches.map((ts) => ts.id)]);
        return Promise.resolve(sensorResult.rows);
    }
    async getStatusSensorLogsForReport(organizationId, dryContactInputIds, dateRange, reason, state) {
        let statusSensorLogsResult = dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let params = [];
            let paramIndex = 1;
            let query = `
            SELECT "actionUtc", "log" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs}"
            WHERE "log"->>'i' = ANY($${paramIndex++}) AND
                "actionUtc" >= $${paramIndex++} AND "actionUtc" <= $${paramIndex++}
            `;
            params.push(dryContactInputIds);
            params.push(dateRange.startDateTime);
            params.push(dateRange.endDateTime);
            if (reason !== null && reason !== undefined) {
                query += `
                AND "log"->>'r' = $${paramIndex++}
                `;
                params.push(reason);
            }
            if (state !== null && state !== undefined) {
                query += `
                AND "log"->>'s' = $${paramIndex++}
                `;
                params.push(state);
            }
            let dbResult = await trx.query(query, params);
            return dbResult.rows;
        });
        return statusSensorLogsResult;
    }
    async getTamperSensorLogsReport(organizationId, deviceIds, dateRange, state) {
        let tamperSensorLogsResult = dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let params = [];
            let paramIndex = 1;
            let query = `
            SELECT "actionUtc", "log" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.TamperSwitchLogs}"
            WHERE "log"->>'ci' = ANY($${paramIndex++}) AND
                "actionUtc" >= $${paramIndex++} AND "actionUtc" <= $${paramIndex++}
            `;
            params.push(deviceIds);
            params.push(dateRange.startDateTime);
            params.push(dateRange.endDateTime);
            if (state !== null && state !== undefined) {
                query += `
                AND "log"->>'s' = $${paramIndex++}
                `;
                params.push(state);
            }
            let dbResult = await trx.query(query, params);
            return dbResult.rows;
        });
        return tamperSensorLogsResult;
    }
    async getCounterSensorLogsForReport(organizationId, dryContactInputIds, dateRange, reason) {
        let counterSensorLogsResult = dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let params = [];
            let paramIndex = 1;
            let query = `
            SELECT "actionUtc", "log" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs}"
            WHERE "log"->>'i' = ANY($${paramIndex++}) AND
                "actionUtc" >= $${paramIndex++} AND "actionUtc" <= $${paramIndex++}
            `;
            params.push(dryContactInputIds);
            params.push(dateRange.startDateTime);
            params.push(dateRange.endDateTime);
            if (reason !== null && reason !== undefined) {
                query += `
                AND "log"->>'r' = $${paramIndex++}
                `;
                params.push(reason);
            }
            let dbResult = await trx.query(query, params);
            return dbResult.rows;
        });
        return counterSensorLogsResult;
    }
    async uploadControlPanelAccessLogs(organizationId, rLogs) {
        let accessLogs = [];
        let insertionLogs = [];
        let now = new Date();
        try {
            for (const log of rLogs) {
                let credentialData = null;
                if (log.c) {
                    credentialData = [log.c.toUpperCase()];
                }
                else if (log.cx && log.cx.length > 0) {
                    credentialData = log.cx.filter((c) => c.d).map((c) => c.d.toUpperCase());
                }
                log.id = log.id || uuid_1.default.v4();
                let item = {
                    id: log.id,
                    actionUtc: log.u,
                    organizationId: log.oId,
                    credentialData: JSON.stringify(credentialData),
                    log: JSON.stringify(log),
                };
                accessLogs.push(item);
                insertionLogs.push({
                    id: log.id,
                    insertionUtc: now.toISOString(),
                    actionUtc: log.u,
                    type: dal_constants_1.DalConstants.LogHistoryType.AccessLog,
                    data: null,
                });
            }
            try {
                await this._dbClient.transaction(async (trx) => {
                    await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs).insert(accessLogs);
                    await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory).insert(insertionLogs);
                });
                for (const accessLog of accessLogs) {
                    const jsObjLog = JSON.parse(accessLog.log);
                    if (jsObjLog.ir !== true && jsObjLog.d && jsObjLog.d === enums_1.EnumsV2.AccessDirection.Entrance && jsObjLog.s === true) {
                        await assignAutoShift(organizationId, {
                            credentialOwnerUserId: jsObjLog.o,
                            generationTime: jsObjLog.u,
                            redisCache: dal_manager_1.dbManager.accessRedisCache,
                            logId: jsObjLog.id,
                        });
                    }
                }
            }
            catch (error) {
                await (0, dal_utils_1.insertUpdateLog)(organizationId, this._dbClient, dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs, accessLogs);
            }
        }
        catch (error) {
            app_logs_1.logger.error(error);
            return Promise.reject();
        }
    }
    async insertManualLog(organizationId, requestUserId, logrequest, logTransaction) {
        let basicUserInfo = await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, logrequest.userId);
        let log = {
            id: uuid_1.default.v4(),
            rm: app_enums_1.enums.LogReceiveMethod.ManuelLog,
            u: logrequest.actionUtc.toISOString(),
            oId: organizationId,
            a: logrequest.accessControlPointId,
            r: dal_constants_1.DalConstants.AccessLogReason.Manual,
            o: logrequest.userId,
            i: null,
            ci: null,
            cx: [
                {
                    t: dal_constants_1.DalConstants.libEnumsV3.CredentialType.AccessToken,
                    d: "Mobile",
                },
            ],
            re: null,
            rx: null,
            cid: null,
            d: logrequest.direction,
            an: logrequest.accessControlPointName || null,
            on: basicUserInfo.fullname,
            c: null,
            di: null,
            ir: null,
            iq: false,
            m: requestUserId,
            ul: null,
            rg: null,
            s: true,
            v: false,
            ss: null,
            cs: null,
        };
        let qb_1 = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs).insert({
            id: log.id,
            actionUtc: log.u,
            organizationId: organizationId,
            log: log,
            credentialData: null,
        });
        let qb_2 = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory).insert({
            id: log.id,
            insertionUtc: new Date().toISOString(),
            actionUtc: log.u,
            type: dal_constants_1.DalConstants.LogHistoryType.AccessLog,
            data: null,
        });
        if (logTransaction) {
            qb_1.transacting(logTransaction);
            qb_2.transacting(logTransaction);
            await qb_1;
            await qb_2;
        }
        else {
            await this._dbClient.transaction(async (trx) => {
                await qb_1.transacting(trx);
                await qb_2.transacting(trx);
            });
        }
        return Promise.resolve(log.id);
    }
    async excelUploadManualLog(organizationId, requestUserId, users, sheet, accessControlPoints, trx, defaultAcpId, dateTimeCombined) {
        let logs = [];
        let defaultAcp = accessControlPoints.find((acp) => acp.id === defaultAcpId);
        let defaultAcpName = defaultAcp.name;
        sheet.eachRow((row, rowNumber) => {
            let date;
            let logDate;
            let time;
            let acpId = defaultAcpId;
            let userId;
            let direction;
            let acpName = defaultAcpName;
            let userFullName;
            row.eachCell((cell, colNumber) => {
                let cellValue = cell.text;
                if (rowNumber > 1 && colNumber === 1) {
                    if (!cellValue) {
                        console.log("Skipping empty row in the sheet.");
                        return;
                    }
                    let user = users.find((user) => user.uniqueId === cellValue);
                    if (user) {
                        userId = user.userId;
                        userFullName = `${user.name} ${user.surname}`;
                    }
                }
                if (dateTimeCombined) {
                    if (rowNumber > 1 && colNumber === 2) {
                        logDate = luxon_1.DateTime.fromJSDate(new Date(cell.value.toString()), { zone: "Etc/Utc" }).setZone("local", { keepLocalTime: true }).toJSDate();
                    }
                }
                else {
                    if (rowNumber > 1 && colNumber === 2) {
                        date = luxon_1.DateTime.fromJSDate(cell.value, { zone: "Etc/Utc" });
                    }
                    if (rowNumber > 1 && colNumber === 3) {
                        time = luxon_1.DateTime.fromJSDate(cell.value, { zone: "Etc/Utc" });
                        if (!time) {
                            time = luxon_1.DateTime.fromISO(cell.text).toUTC();
                        }
                        const combinedDateTime = time
                            .set({
                            year: date.year,
                            month: date.month,
                            day: date.day,
                        })
                            .setZone("local", { keepLocalTime: true });
                        logDate = combinedDateTime.toJSDate();
                    }
                }
                if (rowNumber > 1 && colNumber === 4 - +dateTimeCombined) {
                    if (cellValue === "Giriş") {
                        direction = dal_constants_1.DalConstants.AccessDirection.Entrance;
                    }
                    else if (cellValue === "Çıkış") {
                        direction = dal_constants_1.DalConstants.AccessDirection.Exit;
                    }
                    else if (cellValue === "") {
                        direction = dal_constants_1.DalConstants.AccessDirection.All;
                    }
                }
                if (rowNumber > 1 && colNumber === 5 - +dateTimeCombined) {
                    let acp = accessControlPoints.find((acp) => acp.name.trim() === cellValue.trim());
                    acpId = acp.id;
                    acpName = acp.name;
                }
            });
            if (rowNumber > 1) {
                let log = {
                    id: uuid_1.default.v4(),
                    rm: app_enums_1.enums.LogReceiveMethod.ExcelImport,
                    u: logDate.toISOString(),
                    oId: organizationId,
                    a: acpId,
                    r: dal_constants_1.DalConstants.AccessLogReason.Manual,
                    o: userId,
                    i: null,
                    ci: null,
                    cx: [
                        {
                            t: dal_constants_1.DalConstants.libEnumsV3.CredentialType.AccessToken,
                            d: "Mobile",
                        },
                    ],
                    re: null,
                    rx: null,
                    cid: null,
                    d: direction,
                    an: acpName || null,
                    on: userFullName,
                    c: null,
                    di: null,
                    ir: null,
                    iq: false,
                    m: requestUserId,
                    ul: null,
                    rg: null,
                    s: true,
                    v: false,
                    ss: null,
                    cs: null,
                };
                logs.push(log);
            }
        });
        const chunks = [];
        for (let i = 0; i < logs.length; i += 10000) {
            chunks.push(logs.slice(i, i + 10000));
        }
        const operationDate = new Date().toISOString();
        for (const chunk of chunks) {
            const qiLog = [];
            const qbLog = [];
            let qx = 1;
            let insertAccessLogQuery = `INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" (
    								id, "actionUtc", "organizationId", log, "credentialData") `;
            for (const log of chunk) {
                qiLog.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++})`);
                qbLog.push(log.id, log.u, organizationId, log, null);
            }
            insertAccessLogQuery = insertAccessLogQuery + " VALUES " + qiLog.join(" , ");
            const qiHistory = [];
            const qbHistory = [];
            qx = 1;
            let insertLogHistoryQuery = `INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory}" (
    								id, "insertionUtc", "actionUtc", type, data) `;
            for (const log of chunk) {
                qiHistory.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++})`);
                qbHistory.push(log.id, operationDate, log.u, dal_constants_1.DalConstants.LogHistoryType.AccessLog, null);
            }
            insertLogHistoryQuery = insertLogHistoryQuery + " VALUES " + qiHistory.join(" , ");
            await trx.query(insertAccessLogQuery, qbLog);
            await trx.query(insertLogHistoryQuery, qbHistory);
        }
        return Promise.resolve();
    }
    async addAccessLogFromDevice(organizationId, log, trx) {
        const rLog = dal_db_armon_schema_1.ArmonSchema.Models.generatePSQLNoSqlAccessLogForDevice(organizationId, log);
        if (rLog.o === dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId && log.c) {
            const user = await dal_manager_1.dbManager.accessUser.getUserInfoByCredentialData(organizationId, { credentialData: log.cx[0]?.d, credentialType: log.cx[0]?.t }, trx);
            if (user) {
                rLog.on = user.fullName;
                rLog.o = user.userId;
                rLog.ci = user.credentialId;
                rLog.ul = user.roleId;
                log.on = user.fullName;
                log.o = user.userId;
                log.ci = user.credentialId;
                log.ul = user.roleId;
            }
        }
        let credentialData = null;
        if (rLog.c) {
            credentialData = [rLog.c.toUpperCase()];
        }
        else if (rLog.cx && rLog.cx.length > 0) {
            credentialData = rLog.cx.filter((c) => c.d).map((c) => c.d.toUpperCase());
        }
        await Promise.all([
            trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
			(id, "actionUtc", "organizationId", log, "credentialData")
			VALUES ($1, $2, $3, $4, $5)`, [rLog.id, log.u, organizationId, rLog, JSON.stringify(credentialData)]),
            trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory}"
			(id, "insertionUtc", "actionUtc", type, data)
			VALUES ($1, $2, $3, $4, $5)`, [log.id, new Date().toISOString(), log.u, dal_constants_1.DalConstants.LogHistoryType.AccessLog, null]),
        ]);
        return rLog;
    }
    async updateAccessLogForSensorData(params) {
        const trxx = params.trx ?? this._pgPool;
        const { rows: accessLogRows, rowCount: accessLogRowCount } = await trxx.query(`
		SELECT id, "actionUtc", log FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
		WHERE id = $1`, [params.logId]);
        if (accessLogRowCount === 0) {
            app_logs_1.logger.warn("Sensor data received but no related AccessLog found, forwarded to deadLetter");
            return Promise.reject();
        }
        const accessLog = accessLogRows[0];
        if (params.statusSensorLogId) {
            if (!accessLog.log.ss) {
                accessLog.log.ss = [params.statusSensorLogId];
            }
            else {
                accessLog.log.ss.push(params.statusSensorLogId);
            }
        }
        if (params.counterSensorLogId) {
            if (!accessLog.log.cs) {
                accessLog.log.cs = [params.counterSensorLogId];
            }
            else {
                accessLog.log.cs.push(params.counterSensorLogId);
            }
        }
        await trxx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
				SET log = $1
				WHERE id = $2`, [accessLog.log, params.logId]);
    }
    async getLatestAccessLogs(organizationId, take) {
        let startOfDay = (0, moment_1.default)(new Date()).endOf("day");
        let controlMaxDay = (0, moment_1.default)(startOfDay).add(-10, "days");
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .where("actionUtc", ">=", controlMaxDay.toDate())
            .where("actionUtc", "<=", startOfDay.toDate())
            .where("organizationId", organizationId)
            .orderBy("actionUtc", "desc")
            .limit(take);
        let items = (await qb.select("log"))
            .map((row) => row.log)
            .map((logItem) => {
            let t = !logItem.di ? dal_constants_1.DalConstants.CardActivityTimelineItemType.AccessAttempt : dal_constants_1.DalConstants.CardActivityTimelineItemType.ExitButton;
            return {
                i: logItem.id,
                u: new Date(logItem.u),
                t: t,
                a: t === dal_constants_1.DalConstants.CardActivityTimelineItemType.AccessAttempt
                    ? {
                        i: logItem.id,
                        r: logItem.r,
                        d: logItem.d,
                        u: logItem.o,
                        n: logItem.on,
                        ai: logItem.a,
                        an: logItem.an,
                        di: logItem.di,
                        ir: logItem.ir,
                        sid: logItem.sid,
                        re: logItem.re,
                        cid: logItem.cid,
                        s: logItem.s,
                        v: logItem.v,
                        cx: logItem.cx,
                        cl: [],
                    }
                    : null,
                e: t === dal_constants_1.DalConstants.CardActivityTimelineItemType.ExitButton
                    ? {
                        i: logItem.id,
                        d: logItem.d,
                        di: logItem.di,
                        ai: logItem.a,
                        an: logItem.an,
                    }
                    : null,
            };
        });
        for (const item of items) {
            if (item.a && item.a.u) {
                let userCaptionLines = await dal_manager_1.dbManager.accessUser.getSingleUserOrganizationCaptionLines(organizationId, item.a.u);
                item.a.cl = userCaptionLines;
            }
        }
        return Promise.resolve(items);
    }
    async getBasicAccessLogInfo(organizationId, accessLogs, userAccessRights) {
        if (!accessLogs || accessLogs.length === 0) {
            return [];
        }
        let logs = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .whereIn("id", accessLogs.map((m) => m.id))
            .orderBy("actionUtc", "asc")
            .select("id", "actionUtc", "log");
        let regionIds = [];
        let snapshotIds = [];
        let accessControlPointIds = [];
        let result = logs.map((a) => {
            let regionId = accessLogs.find((aa) => aa.id === a.id).regionId;
            let accessControlPointId = a.log.a;
            let uar = userAccessRights.find((uar) => uar.accessControlPointId === accessControlPointId);
            if (regionId) {
                regionIds.push(regionId);
            }
            if (accessControlPointId) {
                accessControlPointIds.push(accessControlPointId);
            }
            if (a.log.sid) {
                for (const sid of a.log.sid) {
                    if (!snapshotIds.includes(sid) && uar && uar.snapshot) {
                        snapshotIds.push(sid);
                    }
                }
            }
            return {
                id: a.id,
                direction: a.log.d,
                region: regionId
                    ? {
                        id: regionId,
                        name: null,
                    }
                    : null,
                accessControlPoint: {
                    id: a.log.a,
                    name: null,
                },
                dateTime: a.actionUtc,
                snapshots: uar && uar.snapshot && a.log.sid
                    ? a.log.sid.map((s) => {
                        return {
                            accessLogId: a.id,
                            snapshotId: s,
                            status: 1,
                        };
                    })
                    : [],
                location: {
                    latitude: a.log.lt,
                    longitude: a.log.ln,
                    isLocationReliable: a.log.lr,
                },
            };
        });
        if (regionIds.length > 0) {
            let regions = await dal_manager_1.dbManager.accessRegion.getRegionIdName(organizationId, regionIds);
            for (const iterator of result) {
                if (iterator.region && iterator.region.id) {
                    let region = regions.find((r) => r.id === iterator.region.id);
                    if (region) {
                        iterator.region.name = region.name;
                    }
                }
            }
        }
        if (accessControlPointIds.length > 0) {
            let acps = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointIdName(organizationId, accessControlPointIds);
            for (const iterator of result) {
                if (iterator.accessControlPoint && iterator.accessControlPoint.id) {
                    let acp = acps.find((a) => a.id === iterator.accessControlPoint.id);
                    if (acp) {
                        iterator.accessControlPoint.name = acp.name;
                    }
                }
            }
        }
        let snapshotQb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots)
            .whereIn("id", snapshotIds)
            .select("id as sid", this._dbClient.raw("log->'r' as r"));
        let snapshots = await snapshotQb;
        if (snapshots.length > 0) {
            for (const row of result) {
                for (const rowsnap of row.snapshots) {
                    let snap = snapshots.find((sn) => sn.sid === rowsnap.snapshotId);
                    rowsnap.status = snap ? snap.r : 3;
                }
            }
        }
        return result;
    }
    async getManualLogById(organizationId, logId) {
        let row = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs).where("id", logId).first();
        if (!row)
            return Promise.reject("log not found");
        let manualLog = {
            userId: row.log.o,
            actionUtc: row.log.u,
            accessControlPointId: row.log.a,
            direction: row.log.d,
            credentialId: row.log.ci,
            wiegandReaderId: null,
        };
        return Promise.resolve(manualLog);
    }
    async deleteManualLogById(organizationId, logId) {
        let row = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs).where("id", logId).delete("id");
        if (row) {
            return Promise.resolve(true);
        }
        return Promise.resolve(false);
    }
    async listPreviousVisitorProfileIds(organizationId, userId, trx) {
        let query = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("organizationId", organizationId)
            .where("visitedUserId", userId)
            .select("visitorProfileId");
        if (trx) {
            return (await query.transacting(trx)).map((m) => m.visitorProfileId);
        }
        else {
            return (await query).map((m) => m.visitorProfileId);
        }
    }
    async insertActiveToNoSql(organizationId, visitDetailed, organizationVisitorModuleSettings) {
        visitDetailed.fields["visitorRegistrationPointId"] = visitDetailed.visitorRegistrationPointId;
        const dynamicSnapshotFieldName = organizationVisitorModuleSettings.visitFormFields.find((elem) => elem.type === dynamicFormCurrent_1.FormFieldType.ImageCapture)?.name;
        let cameraSnapshot;
        if (dynamicSnapshotFieldName) {
            cameraSnapshot = visitDetailed.fields[dynamicSnapshotFieldName]?.data;
            delete visitDetailed.fields[dynamicSnapshotFieldName];
        }
        let now = new Date();
        let log = {
            id: visitDetailed.id,
            su: visitDetailed.startUtc.toISOString(),
            eu: now.toISOString(),
            pi: visitDetailed.profile.id,
            oId: organizationId,
            v: visitDetailed.fields,
            p: visitDetailed.profile,
            ui: visitDetailed.userId,
        };
        await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .insert({
            id: log.id,
            startUtc: visitDetailed.startUtc,
            endUtc: now,
            organizationId: organizationId,
            visitorProfileId: log.pi,
            visitedUserId: log.v.visitedPerson ? log.v.visitedPerson.userId : null,
            visitorRegistrationPointId: visitDetailed.visitorRegistrationPointId,
            log: log,
            cameraSnapshot: cameraSnapshot,
        });
    }
    async insertPreregisteredVisitToNoSql(organizationId, visitDetailed, trx) {
        let now = new Date();
        let log = {
            id: uuid_1.default.v4(),
            vid: visitDetailed.id,
            su: visitDetailed.expectedStartUtc ? visitDetailed.expectedStartUtc.toISOString() : null,
            eu: visitDetailed.expectedEndUtc ? visitDetailed.expectedEndUtc.toISOString() : null,
            pi: visitDetailed.profile.id,
            oId: organizationId,
            v: visitDetailed.fields,
            p: visitDetailed.profile,
            ui: visitDetailed.userId,
            r: visitDetailed.requestUserId,
            u: now,
        };
        let query = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitorPreregistrationHistory).insert({
            id: log.id,
            organizationId: organizationId,
            actionUtc: now,
            visitorProfileId: log.pi,
            log: log,
        });
        if (trx) {
            await query.transacting(trx);
        }
        else {
            await query;
        }
    }
    async getPreregisteredVisitToNoSql(organizationId, visitId) {
        let logItem = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitorPreregistrationHistory).whereRaw(`log->>'vid' = ?`, visitId).first();
        return Promise.resolve(logItem ? logItem.log : null);
    }
    async updateVisitorProfileId(organizationId, originalVisitorProfileId, replacedVisitorProfileId) {
        return this._dbClient.transaction(async (trx) => {
            await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
                .where("visitorProfileId", originalVisitorProfileId)
                .update("visitorProfileId", replacedVisitorProfileId)
                .update("log", this._dbClient.raw(`jsonb_set(log, '{"pi"}', '"${replacedVisitorProfileId}"')`));
            await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitorPreregistrationHistory)
                .where("visitorProfileId", originalVisitorProfileId)
                .update("visitorProfileId", replacedVisitorProfileId)
                .update("log", this._dbClient.raw(`jsonb_set(log, '{"pi"}', '"${replacedVisitorProfileId}"')`));
        });
    }
    async makeVisitorProfileHistoryAnonym(organizationId, visitorProfileIds) {
        let visitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
        let updateFields = [];
        for (const visitorField of visitorModuleSettings.visitorProfileFormFields) {
            if (visitorField.gdpr && visitorField.name) {
                updateFields.push('"' + visitorField.name + '":"****"');
            }
        }
        let updateFieldsQuery = `{${updateFields.join(",")}}`;
        await this._dbClient.raw(`
            update "${organizationId}"."visit_history"
            set log = jsonb_set(log,'{p,fields}',?)
            where "organizationId" = ?
            and "visitorProfileId" IN (${"'" + visitorProfileIds.join("','") + "'"})
            `, [updateFieldsQuery, organizationId]);
    }
    async deleteVisitorProfileHistory(organizationId, visitorProfileId) {
        await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("organizationId", organizationId)
            .where("visitorProfileId", visitorProfileId)
            .delete();
    }
    async listTerminateVisits(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId) {
        let result = {
            options: options,
            total: 0,
            items: [],
        };
        let now = new Date();
        let startUtc = options.startUtc || new Date(now.getFullYear() - 1, now.getMonth(), now.getDate());
        let endUtc = options.endUtc || now;
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(`${dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory} as vh`)
            .leftJoin("visitorRegistrationPoints as vrp", (join) => {
            join.on("vh.visitorRegistrationPointId", "vrp.id");
        })
            .where("vh.startUtc", ">=", startUtc)
            .where("vh.startUtc", "<=", endUtc)
            .whereNull("vrp.deletedAt");
        let logFilter = {};
        if (visitorRegistrationPointId) {
            if (!logFilter.v) {
                logFilter.v = {};
            }
            logFilter.v.visitorRegistrationPointId = visitorRegistrationPointId;
        }
        if (options.visitorProfileIds && options.visitorProfileIds.length > 0) {
            qb.whereIn("vh.visitorProfileId", options.visitorProfileIds);
        }
        let splitted = [];
        if (options.visitorProfileFilterFields && options.visitorProfileFilterFields.length > 0) {
            logFilter.p = { fields: {} };
            for (const field of options.visitorProfileFilterFields) {
                for (const settingsField of visitorModuleSettings.visitorProfileFormFields) {
                    if (settingsField.name === field.name) {
                        switch (settingsField.type) {
                            case dal_constants_1.DalConstants.FormFieldType.Checkbox:
                                if (field.value) {
                                    logFilter.p.fields[field.name] = "true";
                                }
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Date:
                            case dal_constants_1.DalConstants.FormFieldType.DateTime:
                                logFilter.p.fields[field.name] = new Date(field.value).toISOString();
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Number:
                                logFilter.p.fields[field.name] = parseInt(field.value);
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Select:
                            case dal_constants_1.DalConstants.FormFieldType.RadioGroup:
                                logFilter.p.fields[field.name] = field.value;
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Text:
                            case dal_constants_1.DalConstants.FormFieldType.Textarea:
                                splitted = field.value
                                    .trim()
                                    .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                                    .split(" ");
                                if (splitted.length > 0) {
                                    qb.whereWrapped((where) => {
                                        for (const s of splitted) {
                                            if (s.length > 0) {
                                                let textOptions = settingsField.options;
                                                if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                                                    where.orWhereRaw(`unaccent (upper(vh.log->'p'->'fields'->>'${field.name}')) ilike unaccent(upper('${s}%')) AND unaccent(upper((vh.log)::text)) ilike unaccent(upper('%${s}%'))`);
                                                }
                                                else {
                                                    where.orWhereRaw(`unaccent (upper (vh.log->'p'->'fields'->>'${field.name}')) ilike unaccent (upper('%${s}%')) AND unaccent(upper((vh.log)::text)) ilike unaccent(upper('%${s}%'))`);
                                                }
                                            }
                                        }
                                    });
                                }
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Custom:
                            default:
                                break;
                        }
                    }
                }
            }
        }
        if (options.visitFilterFields && options.visitFilterFields.length > 0) {
            if (!logFilter.v) {
                logFilter.v = {};
            }
            if (visitorModuleSettings.visitFormFields.some((value) => value.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit)) {
                logFilter.v = {
                    ...logFilter.v,
                    visitedOrganizationUnit: {
                        id: null,
                    },
                };
            }
            for (const field of options.visitFilterFields) {
                for (const settingsField of visitorModuleSettings.visitFormFields) {
                    if (settingsField.name === field.name) {
                        switch (settingsField.name) {
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                                let acpArr = [];
                                if (typeof field.value === "string") {
                                    acpArr = [field.value];
                                }
                                else {
                                    acpArr = field.value.map((l) => l.id);
                                }
                                if (acpArr && acpArr.length > 0) {
                                    qb.whereRaw(`vh.log->'v'->'accessControlPoints' @> '${JSON.stringify(acpArr)}'::jsonb`);
                                }
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                                logFilter.v.escortPerson.userId = field.value;
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                                if ((0, dal_utils_1.isUUID)(field.value)) {
                                    qb.where("vh.visitedUserId", field.value);
                                }
                                else {
                                    splitted = field.value
                                        .trim()
                                        .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                                        .split(" ");
                                    if (splitted.length > 0) {
                                        qb.whereWrapped((where) => {
                                            for (const s of splitted) {
                                                if (s.length > 0) {
                                                    where.orWhereRaw(`unaccent (upper (vh.log->'v'->'visitedPerson'->>'name')) ilike unaccent(upper('%${s}%')) AND unaccent(upper((vh.log)::text)) ilike unaccent(upper('%${s}%'))`);
                                                    where.orWhereRaw(`unaccent (upper (vh.log->'v'->'visitedPerson'->>'surname')) ilike unaccent(upper('%${s}%')) AND unaccent(upper((vh.log)::text)) ilike unaccent(upper('%${s}%'))`);
                                                }
                                            }
                                        });
                                    }
                                }
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                                logFilter.v.visitedOrganizationUnit.id = field.value;
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials:
                                break;
                            default:
                                switch (settingsField.type) {
                                    case dal_constants_1.DalConstants.FormFieldType.Checkbox:
                                        if (field.value) {
                                            logFilter.v[field.name] = true;
                                        }
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Date:
                                    case dal_constants_1.DalConstants.FormFieldType.DateTime:
                                        logFilter.v[field.name] = new Date(field.value).toISOString();
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Number:
                                    case dal_constants_1.DalConstants.FormFieldType.Select:
                                        logFilter.v[field.name] = parseInt(field.value);
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.RadioGroup:
                                        logFilter.v[field.name] = parseInt(field.value);
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Text:
                                    case dal_constants_1.DalConstants.FormFieldType.Textarea:
                                        splitted = field.value
                                            .trim()
                                            .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                                            .split(" ");
                                        if (splitted.length > 0) {
                                            qb.whereWrapped((where) => {
                                                for (const s of splitted) {
                                                    if (s.length > 0) {
                                                        let textOptions = settingsField.options;
                                                        if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                                                            where.orWhereRaw(`unaccent (upper (vh.log->'v'->>'${field.name}')) ilike unaccent (upper('${s}%')) AND unaccent(upper((vh.log)::text)) ilike unaccent (upper('%${s}%'))`);
                                                        }
                                                        else {
                                                            where.orWhereRaw(`unaccent (upper (vh.log->'v'->>'${field.name}')) ilike unaccent (upper('%${s}%')) AND unaccent(upper((vh.log)::text)) ilike unaccent (upper('%${s}%'))`);
                                                        }
                                                    }
                                                }
                                            });
                                        }
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Custom:
                                    default:
                                        break;
                                }
                                break;
                        }
                    }
                }
            }
        }
        if (Object.keys(logFilter).length > 0) {
            qb.whereRaw(`vh.log @> ?::jsonb`, JSON.stringify(logFilter));
        }
        let relatedOrganizationUnitIds = [];
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            relatedOrganizationUnitIds = await dal_manager_1.dbManager.accessOrganizationUnit.getAncestorAndChildOrganizationUnitIds(organizationId, userId, trx);
        }, userId, organizationId);
        qb.where(function () {
            if (relatedOrganizationUnitIds.length > 0) {
                this.whereIn("vrp.organizationUnitId", relatedOrganizationUnitIds).orWhereNull("vrp.organizationUnitId");
            }
        });
        let qbCount = qb.clone().count().first();
        result.total = parseInt((await qbCount).count);
        if (result.total > 0) {
            qb.orderBy("vh.startUtc", options.sortStartUtcAsc ? "asc" : "desc");
            if (options.skip && options.skip > 0) {
                qb.offset(options.skip);
            }
            if (options.take && options.take > 0) {
                qb.limit(options.take);
            }
            qb.select("vh.id", "vh.startUtc", "vh.endUtc", "vh.log", this.dbClient.raw('convert_from(vh."cameraSnapshot", \'UTF8\') as "cameraSnapshot"'));
            const dalConfigBackup = app_config_1.appConfig.db.log.backup;
            const snapsCleanedDay = luxon_1.DateTime.now()
                .minus({ months: dalConfigBackup.tableSpecific?.access_snapshots ?? dalConfigBackup.expiredBeforeLastMonthsCount })
                .valueOf();
            const dynamicSnapshotFieldName = visitorModuleSettings.visitFormFields.find((elem) => elem.type === dynamicFormCurrent_1.FormFieldType.ImageCapture)?.name;
            result.items = (await qb).map((row) => {
                if (dynamicSnapshotFieldName)
                    row.log.v = Object.assign(row.log.v, { [dynamicSnapshotFieldName]: { data: row.cameraSnapshot } });
                const actionTime = luxon_1.DateTime.fromISO(row.log.eu).valueOf();
                if (actionTime < snapsCleanedDay && row.log.v.cameraSnapshot) {
                    row.log.v.cameraSnapshot = { data: dal_constants_1.DalConstants.imageThatUsedForArchivedData };
                }
                return {
                    visitorProfileId: row.log.pi,
                    visitorUserId: row.log.ui,
                    visitId: row.log.id,
                    visitStartUtc: row.startUtc,
                    visitEndUtc: row.endUtc,
                    visitorProfileFields: row.log.p.fields,
                    visitFields: row.log.v,
                };
            });
        }
        return Promise.resolve(result);
    }
    async listTerminateVisitsWithCursor(organizationId, userId, options, request, visitorModuleSettings, onData) {
        var _a;
        let now = new Date();
        let startUtc = options.startUtc || new Date(now.getFullYear() - 1, now.getMonth(), now.getDate());
        let endUtc = options.endUtc || now;
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(`${dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory} as vh`)
            .leftJoin("visitorRegistrationPoints as vrp", (join) => {
            join.on("vh.visitorRegistrationPointId", "vrp.id");
        })
            .where("vh.startUtc", ">=", startUtc)
            .where("vh.startUtc", "<=", endUtc)
            .whereNull("vrp.deletedAt");
        let logFilter = {};
        if (request.visitorRegistrationPointId) {
            logFilter.v = {};
            logFilter.v.visitorRegistrationPointId = request.visitorRegistrationPointId;
        }
        if (options.visitorProfileIds && options.visitorProfileIds.length > 0) {
            qb.whereIn("vh.visitorProfileId", options.visitorProfileIds);
        }
        let splitted = [];
        if (options.visitorProfileFilterFields && options.visitorProfileFilterFields.length > 0) {
            logFilter.p = { fields: {} };
            for (const field of options.visitorProfileFilterFields) {
                for (const settingsField of visitorModuleSettings.visitorProfileFormFields) {
                    if (settingsField.name === field.name) {
                        switch (settingsField.type) {
                            case dal_constants_1.DalConstants.FormFieldType.Checkbox:
                                if (field.value) {
                                    logFilter.p.fields[field.name] = "true";
                                }
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Date:
                            case dal_constants_1.DalConstants.FormFieldType.DateTime:
                                logFilter.p.fields[field.name] = new Date(field.value).toISOString();
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Number:
                                logFilter.p.fields[field.name] = parseInt(field.value);
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Select:
                            case dal_constants_1.DalConstants.FormFieldType.RadioGroup:
                                logFilter.p.fields[field.name] = field.value;
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Text:
                            case dal_constants_1.DalConstants.FormFieldType.Textarea:
                                splitted = field.value
                                    .trim()
                                    .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                                    .split(" ");
                                if (splitted.length > 0) {
                                    qb.whereWrapped((where) => {
                                        for (const s of splitted) {
                                            if (s.length > 0) {
                                                let textOptions = settingsField.options;
                                                if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                                                    where.orWhereRaw(`unaccent (vh.log->'p'->'fields'->>'${field.name}')  ilike unaccent ('${s}%') and (vh.log)::text ilike unaccent ('%${s}%')`);
                                                }
                                                else {
                                                    where.orWhereRaw(`unaccent (vh.log->'p'->'fields'->>'${field.name}') ilike unaccent ('%${s}%') and (vh.log)::text ilike unaccent ('%${s}%')`);
                                                }
                                            }
                                        }
                                    });
                                }
                                break;
                            case dal_constants_1.DalConstants.FormFieldType.Custom:
                            default:
                                break;
                        }
                    }
                }
            }
        }
        if (options.visitFilterFields && options.visitFilterFields.length > 0) {
            logFilter.v = {};
            for (const field of options.visitFilterFields) {
                for (const settingsField of visitorModuleSettings.visitFormFields) {
                    if (settingsField.name === field.name) {
                        switch (settingsField.name) {
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                                let acpArr = [];
                                if (typeof field.value === "string") {
                                    acpArr = [field.value];
                                }
                                else {
                                    acpArr = field.value.map((l) => l.id);
                                }
                                if (acpArr && acpArr.length > 0) {
                                    qb.whereRaw(`vh.log->'v'->'accessControlPoints' @> '${JSON.stringify(acpArr)}'::jsonb`);
                                }
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                                logFilter.v.escortPerson.userId = field.value;
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                                if ((0, dal_utils_1.isUUID)(field.value)) {
                                    qb.where("vh.visitedUserId", field.value);
                                }
                                else {
                                    splitted = field.value
                                        .trim()
                                        .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                                        .split(" ");
                                    if (splitted.length > 0) {
                                        qb.whereWrapped((where) => {
                                            for (const s of splitted) {
                                                if (s.length > 0) {
                                                    where.orWhereRaw(` unaccent (vh.log->'v'->'visitedPerson'->>'name') ilike unaccent ('%${s}%') and (vh.log)::text ilike unaccent ('%${s}%')`);
                                                    where.orWhereRaw(` unaccent (vh.log->'v'->'visitedPerson'->>'surname') ilike unaccent ('%${s}%') and (vh.log)::text ilike unaccent ('%${s}%')`);
                                                }
                                            }
                                        });
                                    }
                                }
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                                (_a = logFilter.v).visitedOrganizationUnit || (_a.visitedOrganizationUnit = {});
                                logFilter.v.visitedOrganizationUnit.id = field.value;
                                break;
                            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials:
                                break;
                            default:
                                switch (settingsField.type) {
                                    case dal_constants_1.DalConstants.FormFieldType.Checkbox:
                                        if (field.value) {
                                            logFilter.v[field.name] = true;
                                        }
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Date:
                                    case dal_constants_1.DalConstants.FormFieldType.DateTime:
                                        logFilter.v[field.name] = new Date(field.value).toISOString();
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Number:
                                    case dal_constants_1.DalConstants.FormFieldType.Select:
                                        logFilter.v[field.name] = parseInt(field.value);
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.RadioGroup:
                                        logFilter.v[field.name] = parseInt(field.value);
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Text:
                                    case dal_constants_1.DalConstants.FormFieldType.Textarea:
                                        splitted = field.value
                                            .trim()
                                            .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                                            .split(" ");
                                        if (splitted.length > 0) {
                                            qb.whereWrapped((where) => {
                                                for (const s of splitted) {
                                                    if (s.length > 0) {
                                                        let textOptions = settingsField.options;
                                                        if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                                                            where.orWhereRaw(`unaccent (vh.log->'v'->>'${field.name}') ilike unaccent ('${s}%') and (vh.log)::text ilike unaccent ('%${s}%')`);
                                                        }
                                                        else {
                                                            where.orWhereRaw(`unaccent (vh.log->'v'->>'${field.name}') ilike unaccent ('%${s}%') and (vh.log)::text ilike unaccent ('%${s}%')`);
                                                        }
                                                    }
                                                }
                                            });
                                        }
                                        break;
                                    case dal_constants_1.DalConstants.FormFieldType.Custom:
                                    default:
                                        break;
                                }
                                break;
                        }
                    }
                }
            }
        }
        if (Object.keys(logFilter).length > 0) {
            qb.whereRaw(`vh.log @> ?::jsonb`, JSON.stringify(logFilter));
        }
        let relatedOrganizationUnitIds = [];
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            relatedOrganizationUnitIds = await dal_manager_1.dbManager.accessOrganizationUnit.getAncestorAndChildOrganizationUnitIds(organizationId, userId, trx);
        }, userId, organizationId);
        qb.where(function () {
            if (relatedOrganizationUnitIds.length > 0) {
                this.whereIn("vrp.organizationUnitId", relatedOrganizationUnitIds).orWhereNull("vrp.organizationUnitId");
            }
        });
        qb.orderBy("vh.startUtc", options.sortStartUtcAsc ? "asc" : "desc");
        let columns = [
            { id: "vh.id" },
            { startUtc: "vh.startUtc" },
            { endUtc: "vh.endUtc" },
            { log: "vh.log" },
            { cameraSnapshot: this.dbClient.raw("convert_from(vh.\"cameraSnapshot\", 'UTF8')") },
        ];
        qb.columns(columns);
        let queryResult = qb.toSQL().toNative();
        let queryParams = queryResult.bindings;
        let query = queryResult.sql;
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(query, queryParams));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(500, async (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                const dynamicSnapshotFieldName = visitorModuleSettings.visitFormFields.find((elem) => elem.type === dynamicFormCurrent_1.FormFieldType.ImageCapture)?.name;
                rows = rows.map((row) => {
                    if (dynamicSnapshotFieldName)
                        row.log.v = Object.assign(row.log.v, { [dynamicSnapshotFieldName]: { data: row.cameraSnapshot } });
                    return {
                        visitorProfileId: row.log.pi,
                        visitorUserId: row.log.ui,
                        visitId: row.log.id,
                        visitStartUtc: row.startUtc,
                        visitEndUtc: row.endUtc,
                        visitorProfileFields: row.log.p.fields,
                        visitFields: row.log.v,
                    };
                });
                (0, report_util_1.hideSensitiveFieldsOfVisitsForReport)(request.sensitiveFields.visitFields, request.sensitiveFields.visitorFields, rows, {
                    hideUniqueIdForVisitor: request.sensitiveFields.hideUniqueIdForVisitor,
                    showNumberOfInitial: request.sensitiveFields.showNumberOfInitial,
                    showNumberOfLast: request.sensitiveFields.showNumberOfLast,
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error("Error while fetch terminated visits data with cursor!");
            }
            if (rows.length < 500) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
        return Promise.resolve();
    }
    async searchVisitHistoryFormField(organizationId, visitorModuleSettings, options) {
        let result = {
            total: 0,
            items: [],
        };
        let now = new Date();
        let startUtc = new Date(now.getFullYear() - 1, now.getMonth(), now.getDate());
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("startUtc", ">=", startUtc)
            .where("endUtc", "<=", now)
            .where("organizationId", organizationId);
        let logFilter = {};
        let splitted = [];
        switch (options.formField.name) {
            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                break;
            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                break;
            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                splitted = options.filter
                    .trim()
                    .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                    .split(" ");
                if (splitted.length > 0) {
                    qb.whereWrapped((where) => {
                        for (const s of splitted) {
                            if (s.length > 0) {
                                where.orWhereRaw(`unaccent (log->'v'->'visitedPerson'->>'name') ilike unaccent ('%${s}%') and (log)::text ilike unaccent ('%${s}%')`);
                                where.orWhereRaw(`unaccent (log->'v'->'visitedPerson'->>'surname') ilike unaccent ('%${s}%') and (log)::text ilike unaccent ('%${s}%')`);
                            }
                        }
                    });
                }
                break;
            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                break;
            case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials:
                break;
            default:
                logFilter.v = {};
                switch (options.formField.type) {
                    case dal_constants_1.DalConstants.FormFieldType.Checkbox:
                        if (options.filter) {
                            logFilter.v[options.formField.name] = true;
                        }
                        break;
                    case dal_constants_1.DalConstants.FormFieldType.Date:
                    case dal_constants_1.DalConstants.FormFieldType.DateTime:
                        logFilter.v[options.formField.name] = new Date(options.filter).toISOString();
                        break;
                    case dal_constants_1.DalConstants.FormFieldType.Select:
                    case dal_constants_1.DalConstants.FormFieldType.Number:
                        logFilter.v[options.formField.name] = parseInt(options.filter);
                        break;
                    case dal_constants_1.DalConstants.FormFieldType.RadioGroup:
                        logFilter.v[options.formField.name] = parseInt(options.filter);
                        break;
                    case dal_constants_1.DalConstants.FormFieldType.Text:
                    case dal_constants_1.DalConstants.FormFieldType.Textarea:
                        splitted = options.filter
                            .trim()
                            .replace(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g, " ")
                            .split(" ");
                        if (splitted.length > 0) {
                            qb.whereWrapped((where) => {
                                for (const s of splitted) {
                                    if (s.length > 0) {
                                        let textOptions = options.formField.options;
                                        if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                                            where.orWhereRaw(`unaccent (log->'v'->>'${options.formField.name}') ilike unaccent ('${s}%') and (log)::text ilike unaccent ('%${s}%')`);
                                        }
                                        else {
                                            where.orWhereRaw(`unaccent (log->'v'->>'${options.formField.name}') ilike unaccent ('%${s}%') and (log)::text ilike unaccent ('%${s}%')`);
                                        }
                                    }
                                }
                            });
                        }
                        break;
                    case dal_constants_1.DalConstants.FormFieldType.Custom:
                    default:
                        break;
                }
                break;
        }
        if (Object.keys(logFilter).length > 0) {
            qb.whereRaw(`log @> ?::jsonb`, JSON.stringify(logFilter));
        }
        let qbCount = qb.clone().count().first();
        result.total = parseInt((await qbCount).count);
        if (result.total > 0) {
            let items = (await qb).map((row) => {
                let captionLines = [];
                let value = "";
                switch (options.formField.name) {
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                        captionLines.push(row.log.v.visitedPerson.name + " " + row.log.v.visitedPerson.surname);
                        value = row.log.v.visitedPerson.userId;
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials:
                        break;
                    default:
                        captionLines.push(row.log.v[options.formField.name]);
                        value = row.log.v[options.formField.name];
                        break;
                }
                return {
                    captionLines: captionLines,
                    value: value,
                };
            });
            for (let item of items) {
                if (!result.items.some((v) => v.value == item.value)) {
                    result.items.push(item);
                }
            }
        }
        return Promise.resolve(result);
    }
    async getLastTerminatedVisit(organizationId, visitorProfileId, organizationVisitorModuleSettings) {
        const dynamicSnapshotFieldName = organizationVisitorModuleSettings.visitFormFields.find((elem) => elem.type === dynamicFormCurrent_1.FormFieldType.ImageCapture)?.name;
        let result = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("visitorProfileId", visitorProfileId)
            .orderBy("endUtc", "desc")
            .first("log", this.dbClient.raw('convert_from("cameraSnapshot", \'UTF8\') as "cameraSnapshot"'));
        if (result) {
            if (dynamicSnapshotFieldName) {
                result.log.v = Object.assign(result.log.v, { [dynamicSnapshotFieldName]: { data: result.cameraSnapshot } });
            }
            return Promise.resolve(result.log);
        }
        return Promise.resolve(null);
    }
    async uploadStatusSensorLogsPg(params) {
        const sensorLogsToInsert = [];
        for (const log of params.sensorLogs) {
            log.id = uuid_1.default.v4();
            sensorLogsToInsert.push({
                id: log.id,
                actionUtc: log.u,
                organizationId: params.organizationId,
                log: log,
            });
        }
        const qi = [];
        const qb = [];
        let qx = 1;
        let q = `INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs}"
		(id, "actionUtc", "organizationId", log) `;
        for (const log of sensorLogsToInsert) {
            qi.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++})`);
            qb.push(log.id, log.actionUtc, log.organizationId, log.log);
        }
        q = q + " VALUES " + qi.join(" , ");
        await params.trx.query(q, qb);
        return sensorLogsToInsert.map((s) => s.id);
    }
    async uploadStatusSensorLogs(organizationId, sensorLogs) {
        let sensorLogsToInsert = [];
        for (const log of sensorLogs) {
            log.id = uuid_1.default.v4();
            let item = {
                id: log.id,
                actionUtc: log.u,
                organizationId: organizationId,
                log: log,
            };
            sensorLogsToInsert.push(item);
        }
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs).insert(sensorLogsToInsert);
        return Promise.resolve(sensorLogsToInsert.map((s) => s.id));
    }
    async uploadTamperSwitchLogsPg(params) {
        const tamperSwitchLogsToInsert = [];
        for (const log of params.tamperSwitchLogs) {
            log.id = uuid_1.default.v4();
            tamperSwitchLogsToInsert.push({
                id: log.id,
                actionUtc: log.u,
                organizationId: params.organizationId,
                log: log,
            });
        }
        const qi = [];
        const qb = [];
        let qx = 1;
        let q = `INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.TamperSwitchLogs}"
		(id, "actionUtc", "organizationId", log) `;
        for (const log of tamperSwitchLogsToInsert) {
            qi.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++})`);
            qb.push(log.id, log.actionUtc, log.organizationId, log.log);
        }
        q = q + " VALUES " + qi.join(" , ");
        await params.trx.query(q, qb);
    }
    async uploadTamperSwitchLogs(organizationId, tamperSwitchLogs) {
        let tamperSwitchLogsToInsert = [];
        for (const log of tamperSwitchLogs) {
            log.id = uuid_1.default.v4();
            let item = {
                id: log.id,
                actionUtc: log.u,
                organizationId: organizationId,
                log: log,
            };
            tamperSwitchLogsToInsert.push(item);
        }
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.TamperSwitchLogs).insert(tamperSwitchLogsToInsert);
    }
    async uploadCounterLogs(params) {
        const counterSensorLogsToInsert = [];
        const trxx = params.trx ?? this._pgPool;
        for (const log of params.logs) {
            log.id = uuid_1.default.v4();
            const item = {
                id: log.id,
                actionUtc: log.u,
                organizationId: params.organizationId,
                log: log,
            };
            counterSensorLogsToInsert.push(item);
        }
        const qi = [];
        const qb = [];
        let qx = 1;
        let q = `INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs}"
		(id, "actionUtc", "organizationId", log) `;
        for (const log of counterSensorLogsToInsert) {
            qi.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++})`);
            qb.push(log.id, log.actionUtc, log.organizationId, log.log);
        }
        q = q + " VALUES " + qi.join(" , ");
        await trxx.query(q, qb);
        return counterSensorLogsToInsert.map((s) => s.id);
    }
    async insertOnlineControlPanelTamperSwitchLog(organizationId, log) {
        await this.uploadTamperSwitchLogs(organizationId, [log]);
    }
    async insertOnlineCounterLog(params) {
        let result = await this.uploadCounterLogs({ organizationId: params.organizationId, logs: [params.log], trx: params.trx });
        return Promise.resolve(result[0]);
    }
    async uploadDeviceSystemStatusLogs(organizationId, deviceId, logs, previousBatteryLevel) {
        let systemStatusLogsToInsert = [];
        for (const log of logs) {
            let item = {
                id: deviceId,
                actionUtc: log.u,
                type: dal_constants_1.DalConstants.SystemStatusLogType.TerminalHealth,
                log: JSON.stringify(log),
            };
            if (log.ic !== null && log.ic !== undefined) {
                item.type = log.ic ? dal_constants_1.DalConstants.SystemStatusLogType.TerminaleConnected : dal_constants_1.DalConstants.SystemStatusLogType.TerminalDisconnected;
                item.log = JSON.stringify({});
            }
            if (log.b !== null && log.b !== undefined && log.b !== previousBatteryLevel) {
                await (0, business_device_1.notifyUsersAboutAperioBatteryStatus)(organizationId, {
                    lockDeviceId: log.did,
                    batteryLevel: log.b,
                    timestamp: (log.u ? new Date(log.u) : new Date()).toLocaleString(),
                });
            }
            systemStatusLogsToInsert.push(item);
        }
        await (0, dal_utils_1.insertUpdateLogWithType)(organizationId, this._dbClient, dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs, systemStatusLogsToInsert, dal_constants_1.DalConstants.SystemStatusLogType.TerminalHealth);
        return Promise.resolve();
    }
    async insertAccessLogFromDevice(organizationId, log) {
        let rLog = dal_db_armon_schema_1.ArmonSchema.Models.generatePSQLNoSqlAccessLogForDevice(organizationId, log);
        if (rLog.o === dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId && rLog.c) {
            let user = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return dal_manager_1.dbManager.accessUser.getUserInfoByCredentialData(organizationId, { credentialData: rLog.cx[0]?.d, credentialType: rLog.cx[0]?.t }, trx);
            });
            if (user) {
                rLog.on = user.fullName;
                rLog.o = user.userId;
                rLog.ci = user.credentialId;
                rLog.ul = user.roleId;
            }
        }
        let credentialData = null;
        if (rLog.c) {
            credentialData = [rLog.c.toUpperCase()];
        }
        else if (rLog.cx && rLog.cx.length > 0) {
            credentialData = rLog.cx.filter((c) => c.d).map((c) => c.d.toUpperCase());
        }
        rLog.id = uuid_1.default.v4();
        await Promise.all([
            this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs).insert({
                id: rLog.id,
                actionUtc: log.u,
                organizationId: organizationId,
                log: rLog,
                credentialData: credentialData,
            }),
            this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory).insert({
                id: rLog.id,
                insertionUtc: new Date().toISOString(),
                actionUtc: log.u,
                type: dal_constants_1.DalConstants.LogHistoryType.AccessLog,
                data: null,
            }),
        ]);
        if (rLog.ir !== true && rLog.d && rLog.d === enums_1.EnumsV2.AccessDirection.Entrance && rLog.s === true) {
            await assignAutoShift(organizationId, {
                credentialOwnerUserId: rLog.o,
                generationTime: rLog.u,
                redisCache: dal_manager_1.dbManager.accessRedisCache,
                logId: rLog.id,
            });
        }
        return Promise.resolve(rLog);
    }
    async getAccessLogsNew(organizationId, filter, onData) {
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
            SELECT * FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}
            WHERE "actionUtc" >= $${queryParamIndex++}
            AND "actionUtc" <= $${queryParamIndex++}
            `;
        queryParams.push(filter.startUtc || new Date(new Date().getTime() - 24 * 60 * 60 * 1000));
        queryParams.push(filter.endUtc || new Date());
        let logFilter = {};
        if (filter.direction) {
            logFilter.d = filter.direction;
        }
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            query += ` AND log->>'a' IN (${"'" + filter.accessControlPointIds.join("','") + "'"}) `;
        }
        if (filter.regionIds?.length) {
            let regionAcpIds = [];
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                regionAcpIds.push(...(await dal_manager_1.dbManager.accessAccessControlPoint.listRegionsAccesControlPointIds(organizationId, filter.regionIds, trx)));
            });
            if (regionAcpIds && regionAcpIds.length > 0) {
                query += ` AND log->>'a' IN (${"'" + regionAcpIds.join("','") + "'"}) `;
            }
        }
        if (filter.userIds && filter.userIds.length > 0) {
            query += ` AND log->>'o' IN (${"'" + filter.userIds.join("','") + "'"}) `;
        }
        if (filter.showOnlyManuallyInserted) {
            query += ` AND log->>'m' IS NOT NULL `;
        }
        if (filter.manualRecordCreatorUserId) {
            query += ` AND log->>'m' = '${filter.manualRecordCreatorUserId}' `;
        }
        if (filter.showOnlyRemoteAccess) {
            logFilter.ir = true;
        }
        if (filter.showOnlyVisitors) {
            logFilter.v = true;
        }
        if (filter.accessResult) {
            if (filter.accessResult === dal_constants_1.DalConstants.AccessReportFilterAccessResultType.Success) {
                logFilter.s = true;
            }
            else if (filter.accessResult === dal_constants_1.DalConstants.AccessReportFilterAccessResultType.Fail) {
                query += `AND (log->'s' = 'null' or log->'s' = 'false') `;
            }
        }
        if (filter.showOnlyExitButtons) {
            query += `AND log->>'di' is not null `;
        }
        if (filter.reasons && filter.reasons.length > 0) {
            query += `AND (log->>'r')::INTEGER IN (${filter.reasons.join(",")}) `;
        }
        if (filter.credentialData) {
            query += `AND "credentialData" @> '["${filter.credentialData.trim().toUpperCase()}"]' `;
        }
        if (filter.credentialTypes) {
            query += `AND EXISTS (SELECT 1 FROM jsonb_array_elements(COALESCE(log->'cx', '[]')) AS cx_element WHERE jsonb_typeof(log->'cx') = 'array' AND (cx_element->>'t')::integer = ANY(array[${filter.credentialTypes.join(",")}])) `;
        }
        if (Object.keys(logFilter).length > 0) {
            query += `AND log @> '${JSON.stringify(logFilter)}':: jsonb `;
        }
        query += `ORDER BY "actionUtc" ${filter.sortDateDesc ? "DESC" : "ASC"} `;
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(query, queryParams));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(1000, async (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            const manuelLogs = rows.filter((row) => row.log.r === enums_1.EnumsV2.AccessLogReason.Manual);
                            let manuelLogInsertionDates = [];
                            if (manuelLogs.length > 0) {
                                manuelLogInsertionDates = await this.dbClient
                                    .withSchema(organizationId)
                                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory)
                                    .whereIn("id", manuelLogs.map((l) => l.log.id))
                                    .select("id", "insertionUtc");
                            }
                            const items = rows.map((r) => {
                                return { ...r, insertionDate: manuelLogInsertionDates.find((q) => q.id === r.log.id)?.insertionUtc || undefined };
                            });
                            resolve(items);
                        }
                    });
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error("Error while fetch accesslog data with cursor!" + error);
            }
            if (rows.length < 1000) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
    }
    async getAccessLogsReport(organizationId, filter, shouldReturnSnapshots, userAccessRights) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .where("actionUtc", ">=", filter.startUtc || new Date(new Date().getTime() - 24 * 60 * 60 * 1000))
            .where("actionUtc", "<=", filter.endUtc || new Date())
            .where("organizationId", organizationId);
        let logFilter = {};
        if (filter.direction) {
            logFilter.d = filter.direction;
        }
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            qb.whereRaw(`log->>'a' IN (${"'" + filter.accessControlPointIds.join("','") + "'"})`);
        }
        if (filter.regionIds?.length) {
            let regionAcpIds = [];
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                regionAcpIds.push(...(await dal_manager_1.dbManager.accessAccessControlPoint.listRegionsAccesControlPointIds(organizationId, filter.regionIds, trx)));
            });
            if (regionAcpIds && regionAcpIds.length > 0) {
                qb.whereRaw(`log->>'a' IN (${"'" + regionAcpIds.join("','") + "'"})`);
            }
        }
        if (filter.userIds?.length === 0) {
            if (!filter.organizationUnitIds && !filter.userGroupIds) {
                filter.userIds = null;
            }
        }
        if (filter.userIds) {
            qb.whereRaw(filter.userIds.length ? `log->>'o' IN (${"'" + filter.userIds.join("','") + "'"})` : "1 = 0");
        }
        if (filter.showOnlyManuallyInserted) {
            qb.whereRaw(`log->>'m' IS NOT NULL`);
        }
        if (filter.manualRecordCreatorUserId) {
            qb.whereRaw(`log->>'m' = '${filter.manualRecordCreatorUserId}'`);
        }
        if (filter.showOnlyRemoteAccess) {
            logFilter.ir = true;
        }
        if (filter.showOnlyVisitors) {
            logFilter.v = true;
        }
        if (filter.accessResult) {
            if (filter.accessResult === dal_constants_1.DalConstants.AccessReportFilterAccessResultType.Success) {
                logFilter.s = true;
            }
            else if (filter.accessResult === dal_constants_1.DalConstants.AccessReportFilterAccessResultType.Fail) {
                qb.whereRaw(`(log->'s' = 'null' OR log->'s' = 'false')`);
            }
        }
        if (filter.showOnlyExitButtons) {
            qb.whereRaw(`log->>'di' is not null`);
        }
        if (filter.reasons && filter.reasons.length > 0) {
            qb.whereRaw(`(log->>'r')::INTEGER IN (${filter.reasons.join(",")})`);
        }
        if (filter.credentialData) {
            qb.whereRaw(`"credentialData" @> '["${filter.credentialData.trim().toUpperCase()}"]'`);
        }
        if (filter.credentialTypes) {
            qb.whereRaw(`EXISTS (SELECT 1 FROM jsonb_array_elements(COALESCE(log->'cx', '[]')) AS cx_element WHERE jsonb_typeof(log->'cx') = 'array' AND (cx_element->>'t')::integer = ANY(array[${filter.credentialTypes.join(",")}]))`);
        }
        if (Object.keys(logFilter).length > 0) {
            qb.whereRaw(`log @> ?:: jsonb`, JSON.stringify(logFilter));
        }
        let qbCount = qb.clone().count().first();
        let totalCount = parseInt((await qbCount).count);
        if (totalCount == 0) {
            return Promise.resolve({
                logs: [],
                totalCount: 0,
            });
        }
        qb.orderBy("actionUtc", filter.sortDateDesc ? "desc" : "asc");
        if (filter.skip) {
            qb.offset(filter.skip);
        }
        qb.limit(filter.take);
        let logs = await qb.select("actionUtc", "log");
        let minDate = filter.sortDateDesc ? logs[logs.length - 1].actionUtc : logs[0].actionUtc;
        let maxDate = filter.sortDateDesc ? logs[0].actionUtc : logs[logs.length - 1].actionUtc;
        minDate = (0, moment_1.default)(minDate).add(-1, "hour").toDate();
        maxDate = (0, moment_1.default)(maxDate).add(1, "hour").toDate();
        let snapshots = [];
        const manuelLogs = logs.filter((row) => row.log.r === enums_1.EnumsV2.AccessLogReason.Manual);
        let manuelLogInsertionDates = [];
        if (manuelLogs.length > 0) {
            manuelLogInsertionDates = await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory)
                .whereIn("id", manuelLogs.map((log) => log.log.id))
                .select("id", "insertionUtc");
        }
        const items = logs.map((l) => {
            return { ...l.log, iUtc: manuelLogInsertionDates.find((q) => q.id === l.log.id)?.insertionUtc || undefined, actionUtc: l.actionUtc };
        });
        if (shouldReturnSnapshots) {
            let snapshotIds = [];
            snapshotIds = logs
                .filter((l) => userAccessRights.find((uar) => uar.accessControlPointId === l.log.a)?.snapshot === true)
                .filter((l) => l.log.sid)
                .map((l) => l.log.sid);
            snapshotIds = [].concat.apply([], snapshotIds);
            let snapshotQb = this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots)
                .where("actionUtc", ">=", minDate)
                .where("actionUtc", "<=", maxDate)
                .whereIn("id", snapshotIds)
                .select("id as sid", this._dbClient.raw("log->'r' as r"));
            snapshots = await snapshotQb;
            for (let log of items) {
                if (log.sid && shouldReturnSnapshots && snapshotIds.some((s) => log.sid.indexOf(s) > -1)) {
                    let snapshotList = snapshots.filter((s) => log.sid.indexOf(s.sid) > -1);
                    if (snapshotList && snapshotList.length > 0) {
                        log.snapshots = snapshotList.map((s) => {
                            return {
                                snapshotId: s.sid,
                                status: s.r,
                            };
                        });
                    }
                    else {
                        const dalConfigBackup = app_config_1.appConfig.db.log.backup;
                        log.snapshots = log.sid.map((s) => {
                            const snapsCleanedDay = luxon_1.DateTime.now()
                                .minus({ months: dalConfigBackup.tableSpecific?.access_snapshots ?? dalConfigBackup.expiredBeforeLastMonthsCount })
                                .valueOf();
                            const actionTime = luxon_1.DateTime.fromISO((0, moment_1.default)(log.actionUtc).toISOString()).valueOf();
                            if (actionTime > snapsCleanedDay) {
                                return {
                                    snapshotId: s,
                                    status: 1,
                                };
                            }
                            else {
                                return {
                                    snapshotId: s,
                                    status: 98,
                                };
                            }
                        });
                    }
                }
            }
        }
        return Promise.resolve({
            logs: items,
            totalCount: totalCount,
        });
    }
    async getLogDetailInfoById(organizationId, logId) {
        let result = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs).where("organizationId", organizationId).where("id", logId).first();
        if (!result) {
            return Promise.reject("log not found");
        }
        let snapshotInfo = [];
        if (result.log.sid && result.log.sid.length > 0) {
            let intervalStart = (0, moment_1.default)(result.actionUtc).add(-1, "hour").toDate();
            let intervalEnd = (0, moment_1.default)(result.actionUtc).add(1, "hour").toDate();
            let snapshotQb = this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots)
                .where("actionUtc", ">=", intervalStart)
                .where("actionUtc", "<=", intervalEnd)
                .whereIn("id", result.log.sid)
                .select("id as sid", this._dbClient.raw("log->'r' as r"));
            let snapshots = await snapshotQb;
            if (snapshots && snapshots.length > 0) {
                snapshotInfo = snapshots
                    .filter((s) => ~result.log.sid.indexOf(s.sid))
                    .map((s) => {
                    return {
                        snapshotId: s.sid,
                        status: s.r,
                    };
                });
            }
            else {
                snapshotInfo = result.log.sid.map((s) => {
                    return {
                        snapshotId: s,
                        status: 1,
                    };
                });
            }
        }
        return Promise.resolve({
            log: result.log,
            snapshotInfo: snapshotInfo,
        });
    }
    async getFirstAndLastAccessLogsForUsers(organizationId, dateRange, userIds, accessControlPointIds) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .where("actionUtc", ">=", dateRange.startDateTime)
            .where("actionUtc", "<=", dateRange.endDateTime)
            .where("organizationId", organizationId)
            .whereRaw(`log ->> 'o' IN(${"'" + userIds.join("','") + "'"})`);
        if (accessControlPointIds && accessControlPointIds.length > 0) {
            qb.whereRaw(`log ->> 'a' IN(${"'" + accessControlPointIds.join("','") + "'"})`);
        }
        let rows = await qb
            .groupByRaw(`log -> 'o'`)
            .min("actionUtc as min")
            .max("actionUtc as max")
            .select(this._dbClient.raw(`log -> 'o' as "userId"`));
        return rows.map((m) => {
            return {
                userId: m.userId,
                firstUtc: m.min,
                lastUtc: m.max,
            };
        });
    }
    async getFirstAndLastAccessLogsForUsersWithAcpId(organizationId, dateRange, userIds, accessControlPointIds) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .where("actionUtc", ">=", dateRange.startDateTime)
            .where("actionUtc", "<=", dateRange.endDateTime)
            .where("organizationId", organizationId)
            .whereRaw(`log ->> 'o' IN(${"'" + userIds.join("','") + "'"})`);
        if (accessControlPointIds && accessControlPointIds.length > 0) {
            qb.whereRaw(`log ->> 'a' IN(${"'" + accessControlPointIds.join("','") + "'"})`);
        }
        let minRows = await qb.groupByRaw(`log -> 'id'`).groupByRaw(`log -> 'o'`).groupByRaw(`log->'a'`).min("actionUtc as min").select(this._dbClient.raw(`log -> 'id' as "accessLogId", log -> 'o' as "userId", log->'a' as "acpId"`));
        let maxRows = await qb.groupByRaw(`log -> 'id'`).groupByRaw(`log -> 'o'`).groupByRaw(`log->'a'`).max("actionUtc as max").select(this._dbClient.raw(`log -> 'id' as "accessLogId", log -> 'o' as "userId", log->'a' as "acpId"`));
        let minItems = {};
        for (const row of minRows) {
            if (!minItems[row.userId]) {
                minItems[row.userId] = {
                    min: row.min,
                    acpId: row.acpId,
                    accessLogId: row.accessLogId,
                };
            }
            if (minItems[row.userId] && (0, moment_1.default)(minItems[row.userId].min).isAfter((0, moment_1.default)(row.min))) {
                minItems[row.userId] = {
                    min: row.min,
                    acpId: row.acpId,
                    accessLogId: row.accessLogId,
                };
            }
        }
        let maxItems = {};
        for (const row of maxRows) {
            if (!maxItems[row.userId]) {
                maxItems[row.userId] = {
                    max: row.max,
                    acpId: row.acpId,
                    accessLogId: row.accessLogId,
                };
            }
            if (maxItems[row.userId] && (0, moment_1.default)(maxItems[row.userId].max).isBefore((0, moment_1.default)(row.max))) {
                maxItems[row.userId] = {
                    max: row.max,
                    acpId: row.acpId,
                    accessLogId: row.accessLogId,
                };
            }
        }
        let result = [];
        for (const userId of userIds) {
            let log = {
                firstAccessLogId: null,
                firstAcpId: null,
                userId: userId,
                firstUtc: null,
                lastUtc: null,
                lastAcpId: null,
                lastAccessLogId: null,
            };
            let minVal = minItems[userId];
            if (minVal) {
                log.firstAcpId = minVal.acpId;
                log.firstAccessLogId = minVal.accessLogId;
                log.firstUtc = minVal.min;
            }
            let maxVal = maxItems[userId];
            if (maxVal) {
                log.lastAcpId = maxVal.acpId;
                log.lastUtc = maxVal.max;
                log.lastAccessLogId = maxVal.accessLogId;
            }
            result.push(log);
        }
        return Promise.resolve(result);
    }
    async addUploadLogId(organizationId, controlPanelId, uploadId, count) {
        let log = dal_db_armon_schema_1.ArmonSchema.Models.generatePSQLUploadLogHistory(organizationId, controlPanelId, uploadId, count);
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.UploadLogHistory).insert({
            id: uuid_1.default.v4(),
            actionUtc: log.u,
            organizationId: organizationId,
            uploadId: uploadId,
            log: log,
        });
    }
    async checkUploadLogIdExits(organizationId, uploadId) {
        return new Promise(async (resolve, reject) => {
            let exist = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.UploadLogHistory).where("uploadId", uploadId).first();
            if (exist) {
                reject(new dal_access_error_1.DbAccessError("That upload operation is already done!"));
            }
            resolve();
        });
    }
    async getAccessControlPointSummaryReport(organizationId, accessControlPointId, request, acp) {
        if (!acp) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("Access control point is not found!");
        }
        let intervalStart = new Date(request.startDt.year, request.startDt.month - 1, request.startDt.day, request.startDt.hours, request.startDt.minutes, request.startDt.seconds);
        let intervalEnd = new Date(request.endDt.year, request.endDt.month - 1, request.endDt.day, request.endDt.hours, request.endDt.minutes, request.endDt.seconds);
        let result = {
            statusSensorSummary: null,
            exitButtonAccessSummary: null,
            counterSensorSummary: null,
            readerAccessSummary: null,
            remoteAccessSummary: null,
        };
        if (request.readerAccessLogs || request.exitButtonAccessLogs || request.remoteAccessLogs) {
            let logs = (await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
                .where("actionUtc", ">=", intervalStart)
                .where("actionUtc", "<=", intervalEnd)
                .whereRaw(`log ->> 'a' = ?`, accessControlPointId)
                .select("log")).map((l) => l.log);
            if (request.readerAccessLogs) {
                let readerAccessSummary = {
                    entranceSuccessCount: 0,
                    exitSuccessCount: 0,
                    undirectedSuccessCount: 0,
                    entranceFailCount: 0,
                    exitFailCount: 0,
                    undirectedFailCount: 0,
                    items: [],
                };
                for (const reader of acp.readers) {
                    let readerLogs = logs.filter((r) => r.cid == reader.deviceId && r.re == reader.number);
                    let item = {
                        id: reader.id,
                        name: reader.name,
                        entranceSuccessCount: readerLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Entrance && r.s).length,
                        exitSuccessCount: readerLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Exit && r.s).length,
                        undirectedSuccessCount: readerLogs.filter((r) => !r.d && r.s).length,
                        entranceFailCount: readerLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Entrance && !r.s).length,
                        exitFailCount: readerLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Exit && !r.s).length,
                        undirectedFailCount: readerLogs.filter((r) => !r.d && !r.s).length,
                    };
                    readerAccessSummary.entranceSuccessCount += item.entranceSuccessCount;
                    readerAccessSummary.exitSuccessCount += item.exitSuccessCount;
                    readerAccessSummary.undirectedSuccessCount += item.undirectedSuccessCount;
                    readerAccessSummary.entranceFailCount += item.entranceFailCount;
                    readerAccessSummary.exitFailCount += item.exitFailCount;
                    readerAccessSummary.undirectedFailCount += item.undirectedFailCount;
                }
                result.readerAccessSummary = readerAccessSummary;
            }
            if (request.exitButtonAccessLogs) {
                let exitButtonAccessSummary = {
                    entranceCount: 0,
                    exitCount: 0,
                    undirectedCount: 0,
                    items: [],
                };
                for (const exitButton of acp.exitButtons) {
                    let exitButtonLogs = logs.filter((r) => r.cid == exitButton.deviceId && exitButton.dryContactInputNumber == r.di);
                    let relay = acp.relays.find((r) => r.id === exitButton.deviceRelayId) ? acp.relays.find((r) => r.id === exitButton.deviceRelayId) : null;
                    let managesTo = relay ? relay.managesTo : null;
                    let item = {
                        id: exitButton.id,
                        managesTo: managesTo,
                        entranceCount: exitButtonLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Entrance && r.s).length,
                        exitCount: exitButtonLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Exit && r.s).length,
                        undirectedCount: exitButtonLogs.filter((r) => r.s).length,
                    };
                    exitButtonAccessSummary.entranceCount += item.entranceCount;
                    exitButtonAccessSummary.exitCount += item.exitCount;
                    exitButtonAccessSummary.undirectedCount += item.undirectedCount;
                }
                result.exitButtonAccessSummary = exitButtonAccessSummary;
            }
            if (request.remoteAccessLogs) {
                let remoteAccessSummary = {
                    entranceCount: 0,
                    exitCount: 0,
                    undirectedCount: 0,
                    items: [],
                };
                let item = {
                    entranceCount: logs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Entrance && r.s && r.ir).length,
                    exitCount: logs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Exit && r.s && r.ir).length,
                    undirectedCount: logs.filter((r) => r.s && r.ir).length,
                };
                remoteAccessSummary.entranceCount += item.entranceCount;
                remoteAccessSummary.exitCount += item.exitCount;
                remoteAccessSummary.undirectedCount += item.undirectedCount;
                result.remoteAccessSummary = remoteAccessSummary;
            }
        }
        if (request.counterSensorLogs) {
            result.counterSensorSummary = {
                entranceCount: 0,
                exitCount: 0,
                undirectedCount: 0,
                items: [],
            };
            let logs = (await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs)
                .where("actionUtc", ">=", intervalStart)
                .where("actionUtc", "<=", intervalEnd)
                .whereRaw(`log ->> 'a' = ?`, accessControlPointId)
                .select("log")).map((l) => l.log);
            for (const counterSensor of acp.counterSensors) {
                let counterSensorLogs = logs.filter((r) => r.i == counterSensor.id);
                let item = {
                    id: counterSensor.id,
                    countsTo: counterSensor.countsTo,
                    entranceCount: counterSensorLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Entrance).length,
                    exitCount: counterSensorLogs.filter((r) => r.d == dal_constants_1.DalConstants.AccessDirection.Exit).length,
                    undirectedCount: counterSensorLogs.filter((r) => r.d != dal_constants_1.DalConstants.AccessDirection.Exit && r.d != dal_constants_1.DalConstants.AccessDirection.Entrance).length,
                };
                result.counterSensorSummary.entranceCount += item.entranceCount;
                result.counterSensorSummary.exitCount += item.exitCount;
                result.counterSensorSummary.undirectedCount += item.undirectedCount;
            }
        }
        if (request.statusSensorLogs) {
            result.statusSensorSummary = {
                closedCount: 0,
                opennedCount: 0,
                timeOutCount: 0,
                items: [],
            };
            let logs = (await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs)
                .where("actionUtc", ">=", intervalStart)
                .where("actionUtc", "<=", intervalEnd)
                .whereRaw(`log ->> 'a' = ?`, accessControlPointId)
                .select("log")).map((l) => l.log);
            for (const statusSensor of acp.statusSensors) {
                let statusSensorLogs = logs.filter((r) => r.i == statusSensor.id);
                let item = {
                    id: statusSensor.id,
                    senseTo: statusSensor.senseTo,
                    opennedCount: statusSensorLogs.filter((r) => r.s == dal_constants_1.DalConstants.StatusSensorState.Open).length,
                    closedCount: statusSensorLogs.filter((r) => r.s == dal_constants_1.DalConstants.StatusSensorState.Closed).length,
                    timeOutCount: statusSensorLogs.filter((r) => r.d == dal_constants_1.DalConstants.StatusSensorState.TimeOut).length,
                };
                result.statusSensorSummary.opennedCount += item.opennedCount;
                result.statusSensorSummary.closedCount += item.closedCount;
                result.statusSensorSummary.timeOutCount += item.timeOutCount;
            }
        }
        return Promise.resolve(result);
    }
    async insertStatusSensorNotificationDevice(organizationId, log) {
        let result = await this.uploadStatusSensorLogs(organizationId, [log]);
        return Promise.resolve(result[0]);
    }
    async insertSystemStatusLogForTerminalAction(organizationId, deviceId, log) {
        let item = {
            id: deviceId,
            actionUtc: new Date().toISOString(),
            type: dal_constants_1.DalConstants.SystemStatusLogType.TerminalUserAction,
            log: log,
        };
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs).insert(item);
        return Promise.resolve();
    }
    async insertSystemStatusLogForConnection(organizationId, deviceId, connected) {
        let item = {
            id: deviceId,
            actionUtc: new Date().toISOString(),
            type: connected ? dal_constants_1.DalConstants.SystemStatusLogType.TerminaleConnected : dal_constants_1.DalConstants.SystemStatusLogType.TerminalDisconnected,
            log: {},
        };
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs).insert(item);
        return Promise.resolve();
    }
    async getTerminalLastConnectionTime(params) {
        const { rows } = await (params.trx ?? this._pgPool).query(`
            SELECT "actionUtc" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs}"
            WHERE id = $1 
            AND type = $2
            ORDER BY "actionUtc" DESC limit 1
        `, [params.terminalId, params.type]);
        return rows[0].actionUtc;
    }
    async insertSystemHealthLog(organizationId, deviceId, log) {
        let item = {
            id: deviceId,
            actionUtc: new Date().toISOString(),
            type: dal_constants_1.DalConstants.SystemStatusLogType.TerminalHealth,
            log: log,
        };
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs).insert(item);
        return Promise.resolve();
    }
    async uploadSnapshot(organizationId, deviceId, snapshotId, success, actionUtc, snapshot) {
        try {
            let snapshotLog = {
                s: snapshot ? snapshot : null,
                r: success ? 2 : 3,
                sid: snapshotId,
            };
            await (0, dal_utils_1.insertUpdateLog)(organizationId, this._dbClient, dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots, [
                {
                    id: snapshotId,
                    actionUtc: actionUtc ? actionUtc : new Date().toISOString(),
                    log: JSON.stringify(snapshotLog),
                },
            ]);
            return Promise.resolve(true);
        }
        catch (error) {
            app_logs_1.logger.error(error);
            return Promise.resolve(false);
        }
    }
    async getAccessSnapshot(organizationId, snapshotId) {
        let snapshotLog;
        let result = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots).where("id", snapshotId).first("log");
        if (!result)
            (0, dal_access_error_1.throwDbAccessNotFoundError)("snapshot not found");
        snapshotLog = result.log;
        let jpegSnapshot = (await (0, sharp_1.default)(new Buffer(snapshotLog.s), { failOnError: false }).toFormat("jpeg").toBuffer()).toString("base64");
        return Promise.resolve({
            snapshot: jpegSnapshot,
            success: snapshotLog.r === 2,
        });
    }
    async getLogStats(organizationId, sinceLastMonth, sinceLastSixMonth) {
        let row = await this._pgPool.query(`SELECT (json_build_object('totalAccessLogsInLastMonth', (SELECT count(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" WHERE "actionUtc" > $1),
            'totalAccessLogsInLastSixMonths', (SELECT count(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" WHERE "actionUtc" > $2),
            'totalSnapshotsTakenInLastMonth', (SELECT count(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots}" WHERE "actionUtc" > $1),
            'totalSnapshotsTakenInLastSixMonths', (SELECT count(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots}" WHERE "actionUtc" > $2))) AS row
        `, [sinceLastMonth, sinceLastSixMonth]);
        let data = row.rows[0].row;
        return {
            totalAccessLogsInLastMonth: data.totalAccessLogsInLastMonth,
            totalAccessLogsInLastSixMonths: data.totalAccessLogsInLastSixMonths,
            totalSnapshotsTakenInLastSixMonths: data.totalSnapshotsTakenInLastSixMonths,
            totalSnapshotsTakenInLastMonth: data.totalSnapshotsTakenInLastMonth,
        };
    }
    async getAccessSnapshotWithLogId(organizationId, requesterUserId, logId, snapshotId, backupPolicy) {
        let log = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs).where("id", logId).first();
        if (!log) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("log is not fou nd");
        }
        let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, requesterUserId);
        let uar = userAccessRights.find((u) => u.accessControlPointId === log.log.a);
        if (!uar || !uar.snapshot) {
            return Promise.resolve({
                isSuccess: false,
                isArchived: false,
                snapshot: null,
                isUnauthorized: true,
            });
        }
        if (backupPolicy && backupPolicy.expiredBeforeLastMonthsCount) {
            let checkDate = backupPolicy.expiredBeforeLastMonthsCount;
            if (backupPolicy.tableSpecific && backupPolicy.tableSpecific[dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots]) {
                checkDate = parseInt(backupPolicy.tableSpecific[dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots]);
            }
            if ((0, moment_1.default)(new Date()).add(-checkDate, "month").isAfter(log.actionUtc, "month")) {
                return Promise.resolve({
                    isSuccess: false,
                    isArchived: true,
                    snapshot: null,
                    isUnauthorized: false,
                });
            }
        }
        let result = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessSnapshots)
            .where("id", snapshotId)
            .where("actionUtc", ">=", (0, moment_1.default)(log.actionUtc).add("hour", -1).toDate())
            .where("actionUtc", "<=", (0, moment_1.default)(log.actionUtc).add("hour", 1).toDate())
            .first("log");
        if (!result)
            (0, dal_access_error_1.throwDbAccessNotFoundError)("snapshot not found");
        let snapshotLog = result.log;
        let jpegSnapshot = (await (0, sharp_1.default)(new Buffer(snapshotLog.s), { failOnError: false }).toFormat("jpeg").toBuffer()).toString("base64");
        return Promise.resolve({
            snapshot: jpegSnapshot,
            isSuccess: snapshotLog.r === 2,
            isArchived: false,
            isUnauthorized: false,
        });
    }
    async getCounterSensorReport(organizationId, filter) {
        let result = {
            pagination: {
                total: 0,
                take: filter.pagination.take,
                skip: filter.pagination.skip,
            },
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs)
            .where("actionUtc", ">=", filter.dateRange.startDateTime)
            .where("actionUtc", "<=", filter.dateRange.endDateTime)
            .where("organizationId", organizationId);
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            qb.whereRaw(`log ->> 'a' IN(${"'" + filter.accessControlPointIds.join("','") + "'"})`);
        }
        if (filter.userIds && filter.userIds.length > 0) {
            qb.whereRaw(`log ->> 'o' IN(${"'" + filter.userIds.join("','") + "'"})`);
        }
        if (filter.filterEvents && filter.filterEvents.length > 0) {
            qb.whereRaw(`log ->> 'r' IN(${"'" + filter.filterEvents.join("','") + "'"})`);
        }
        result.pagination.total = parseInt((await qb.clone().count().first()).count);
        let sortFieldName = "u";
        if (filter.sortField) {
            switch (filter.sortField) {
                case dal_constants_1.DalConstants.SortField.Date:
                    sortFieldName = "u";
                    break;
                case dal_constants_1.DalConstants.SortField.UserName:
                    sortFieldName = "on";
                    break;
                case dal_constants_1.DalConstants.SortField.AccessControlPoint:
                    sortFieldName = "an";
                    break;
            }
        }
        if (!filter.sortType)
            filter.sortType = dal_constants_1.DalConstants.SortType.Descending;
        if (filter.sortType) {
            switch (filter.sortType) {
                case dal_constants_1.DalConstants.SortType.Descending:
                    qb.orderBy(`log -> '${sortFieldName}'`, "desc");
                    break;
                case dal_constants_1.DalConstants.SortType.Ascending:
                    qb.orderBy(`log -> '${sortFieldName}'`, "asc");
                    break;
            }
        }
        if (filter.pagination.skip) {
            qb.offset(filter.pagination.skip);
        }
        if (filter.pagination.take) {
            qb.limit(filter.pagination.take);
        }
        let logs = (await qb.select("log")).map((r) => r.log);
        let userIds = logs.filter((l) => l.o).map((l) => l.o);
        let userUniqueIds = await dal_manager_1.dbManager.accessUser.listUserUniqueIds(organizationId, userIds);
        for (let log of logs) {
            let userInfo = userUniqueIds.find((s) => s.userId === log.o);
            result.items.push({
                accessControlPointId: log.a,
                actionUtc: log.u,
                accessControlPointName: log.an,
                counterSensorReason: log.r,
                userFullName: log.on,
                userId: log.o,
                userUniqueId: userInfo ? userInfo.uniqueId : null,
            });
        }
        return Promise.resolve(result);
    }
    async getLatestAccessLog(organizationId, userId, accessControlPointIds, date) {
        let startOfDay = (0, moment_1.default)(new Date()).endOf("day");
        if (date) {
            startOfDay = (0, moment_1.default)(date).endOf("day");
        }
        let controlMaxDay = (0, moment_1.default)(startOfDay).add(-3, "days");
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .where("actionUtc", ">=", controlMaxDay.toDate())
            .where("actionUtc", "<=", startOfDay.toDate())
            .where("organizationId", organizationId)
            .whereRaw(`log ->> 'o' = ?`, userId)
            .orderBy("actionUtc", "desc");
        if (accessControlPointIds && accessControlPointIds.length > 0) {
            qb.whereRaw(`log ->> 'a' IN(${"'" + accessControlPointIds.join("','") + "'"})`);
        }
        let lastLog = await qb.first();
        if (lastLog) {
            return Promise.resolve(lastLog.log);
        }
        return Promise.resolve(null);
    }
    async uploadUserTicketTransactionHistory(organizationId, logs, trx) {
        let ticketLogs = [];
        for (const log of logs) {
            log.id = uuid_1.default.v4();
            let result = {
                id: log.id,
                actionUtc: log.u,
                organizationId: log.oId,
                actionUserId: log.r,
                log: log,
            };
            ticketLogs.push(result);
        }
        if (ticketLogs.length > 0) {
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserTicketTransactionHistory}"
				(id, "organizationId", "actionUtc", "actionUserId", log)
				SELECT (u->>'id')::UUID, (u->>'organizationId')::UUID, (u->>'actionUtc')::TIMESTAMP WITH TIME ZONE, (u->>'actionUserId')::UUID, u->'log'
				FROM UNNEST ($1::JSONB[]) AS u
			`, [ticketLogs]);
        }
    }
    async listRegionTicketReport(organizationId, pagination, dateRange, userIds, regionTicketIds) {
        let paginationResponse = {
            total: 0,
            take: pagination.take,
            skip: pagination.skip,
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .where("actionUtc", ">=", dateRange.startDateTime)
            .where("actionUtc", "<=", dateRange.endDateTime)
            .where("organizationId", organizationId)
            .whereRaw(`log->>'rg' ilike '%rti%'  `);
        if (userIds && userIds.length > 0) {
            qb.whereRaw(`log ->>'o' IN(${"'" + userIds.join("','") + "'"})`);
        }
        if (regionTicketIds && regionTicketIds.length > 0) {
            qb.whereRaw(`log ->'rg' @> '${JSON.stringify(regionTicketIds.map((rt) => {
                return { rti: { i: rt } };
            }))}':: jsonb`);
        }
        qb.whereRaw(`log @> ?:: jsonb`, JSON.stringify({ s: true }));
        paginationResponse.total = parseInt((await qb.clone().count().first()).count);
        qb.orderBy("actionUtc", "desc");
        if (pagination.skip)
            qb.offset(pagination.skip);
        if (pagination.take)
            qb.limit(pagination.take);
        let results = await qb.select("log");
        return Promise.resolve({
            pagination: paginationResponse,
            items: results.map((r) => r.log),
        });
    }
    async listRegionTicketTransactions(organizationId, pagination, dateRange, actionUserId, targetUserId) {
        let paginationResponse = {
            total: 0,
            take: pagination.take,
            skip: pagination.skip,
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.UserTicketTransactionHistory)
            .where("actionUtc", ">=", dateRange.startDateTime)
            .where("actionUtc", "<=", dateRange.endDateTime)
            .where("organizationId", organizationId);
        if (actionUserId) {
            qb.where("actionUserId", actionUserId);
        }
        if (targetUserId) {
            qb.whereRaw(`log -> 'o' @> '["${targetUserId}"]':: jsonb`);
        }
        paginationResponse.total = parseInt((await qb.clone().count().first()).count);
        qb.orderBy("actionUtc", "desc");
        if (pagination.skip)
            qb.offset(pagination.skip);
        if (pagination.take)
            qb.limit(pagination.take);
        let resultLogs = (await qb.select("log")).map((r) => r.log);
        let items = [];
        for (const log of resultLogs) {
            items.push({
                actionUser: await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(organizationId, log.r),
                date: new Date(log.u),
                amount: log.a,
                id: log.id,
                note: log.n,
            });
        }
        return Promise.resolve({
            pagination: paginationResponse,
            items: items,
        });
    }
    async listRegionTicketTransactionDetails(organizationId, transactionId, pagination) {
        let paginationResponse = {
            total: 0,
            take: pagination.take,
            skip: pagination.skip,
        };
        let result = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.UserTicketTransactionHistory).where("id", transactionId).first("log");
        if (!result) {
            return Promise.resolve({
                pagination: paginationResponse,
                users: [],
            });
        }
        let transaction = result.log;
        paginationResponse.total = transaction.o.length;
        let userList = await dal_manager_1.dbManager.accessPacs.getBasicUserInfoList(organizationId, transaction.o);
        let startIndex = 0;
        let endIndex = 0;
        if (pagination.skip)
            startIndex = pagination.skip;
        if (pagination.take)
            endIndex = startIndex + pagination.take;
        if (endIndex > transaction.o.length)
            endIndex = userList.length;
        userList.sort((a, b) => (a.fullname < b.fullname ? -1 : 1));
        let items = userList.slice(startIndex, endIndex);
        return Promise.resolve({
            pagination: paginationResponse,
            users: items,
        });
    }
    async listRegionTicketRawTransactions(organizationId, filter) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.UserTicketTransactionHistory)
            .where("actionUtc", ">=", filter.dateRange.startDateTime)
            .where("actionUtc", "<=", filter.dateRange.endDateTime)
            .where("organizationId", organizationId);
        if (filter.actionUserId) {
            qb.where("actionUserId", filter.actionUserId);
        }
        if (filter.targetUserId) {
            qb.whereRaw(`log -> 'o' @> '["${filter.targetUserId}"]':: jsonb`);
        }
        qb.orderBy("actionUtc", "desc");
        let resultLogs = (await qb.select("log")).map((r) => r.log);
        let items = [];
        for (const log of resultLogs) {
            let userIds = filter.targetUserId ? [filter.targetUserId] : log.o;
            items.push({
                actionUserId: log.r,
                actionUtc: new Date(log.u),
                transactionId: log.id,
                userIds: userIds,
                amount: log.a,
                note: log.n,
            });
        }
        return Promise.resolve(items);
    }
    async listAccessRuleHistory(organizationId, trx, userIds) {
        let result = [];
        let endDate = new Date();
        let startDate = (0, moment_1.default)(endDate).startOf("week").toDate();
        let logFilter = { s: true };
        const qParams = [startDate, endDate, JSON.stringify(logFilter)];
        let q = `
			SELECT "log"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
			WHERE "actionUtc" >= $1
				AND "actionUtc" <= $2
				AND "log"->'rg' IS NOT NULL
				AND "log" @> $3::JSONB
				AND "log"->>'rg' LIKE '%ru%'
		`;
        if (userIds) {
            qParams.push(userIds);
            q += ` AND ("log"->>'o')::UUID = ANY($${qParams.length}::UUID[])`;
        }
        q += `
			ORDER BY "actionUtc" DESC
		`;
        const dbResult = await trx.query(q, qParams);
        let lastItems = {};
        let items = [];
        for (const log of dbResult.rows.map((m) => m.log)) {
            if (log.rg)
                for (const regionLog of log.rg) {
                    if (regionLog.ru) {
                        let lastDate = lastItems[regionLog.ru.i + log.o];
                        if (!lastDate)
                            lastItems[regionLog.ru.i + log.o] = log.u;
                        else if (new Date(lastDate) <= new Date(log.u))
                            lastItems[regionLog.ru.i + log.o] = log.u;
                        else
                            continue;
                        items.push({
                            actionDate: new Date(log.u).toISOString(),
                            userId: log.o,
                            count: regionLog.ru.c,
                            ruleId: regionLog.ru.i,
                            startDateTime: regionLog.ru.sd,
                            endDateTime: regionLog.ru.ed,
                        });
                    }
                }
        }
        for (const item of items) {
            let index = result.findIndex((r) => r.userId == item.userId);
            if (index > -1) {
                result[index].history.push({
                    actionDateISO: item.actionDate,
                    count: item.count,
                    ruleId: item.ruleId,
                    startDateTime: item.startDateTime,
                    endDateTime: item.endDateTime,
                    weekday: new Date(item.startDateTime).getDay(),
                });
            }
            else {
                result.push({
                    userId: item.userId,
                    history: [
                        {
                            actionDateISO: item.actionDate,
                            count: item.count,
                            ruleId: item.ruleId,
                            startDateTime: item.startDateTime,
                            endDateTime: item.endDateTime,
                            weekday: new Date(item.startDateTime).getDay(),
                        },
                    ],
                });
            }
        }
        return Promise.resolve(result);
    }
    async addOrganizationFeedback(feedback) {
        let result = {
            id: uuid_1.default.v4(),
            actionUtc: feedback.u,
            organizationId: feedback.oId,
            threadId: feedback.tid,
            log: feedback,
        };
        await this.dbClient.withSchema(feedback.oId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.OrganizationFeedbacks).insert(result);
    }
    async addOrganizationFeedbackUpdate(feedback) {
        let result = {
            id: uuid_1.default.v4(),
            actionUtc: feedback.u,
            organizationId: feedback.oId,
            threadId: feedback.tid,
            log: feedback,
        };
        await this.dbClient.withSchema(feedback.oId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.OrganizationFeedbackUpdates).insert(result);
    }
    async listOrganizationFeedback(organizationId, threadIds) {
        return (await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.OrganizationFeedbacks).whereIn("threadId", threadIds).orderBy("actionUtc", "desc").select("log")).map((s) => s.log);
    }
    async listOrganizationFeedbackUpdates(organizationId, threadIds) {
        return (await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.OrganizationFeedbackUpdates).whereIn("threadId", threadIds).orderBy("actionUtc", "desc").select("log")).map((s) => s.log);
    }
    async addUserActionHistoryItemPg(organizationId, requestUserId, trx, log) {
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
			(id, "organizationId", "actionUtc", "userId", category, type, log)
			VALUES(uuid_generate_v4(), $1, $2, $3, $4, $5, $6);`, [organizationId, luxon_1.DateTime.fromJSDate(log.u).toSQL(), requestUserId, log.c, log.t, JSON.stringify(log)]);
    }
    async addMultipleUserActionHistoryItemsPg(organizationId, requestUserId, trx, logs) {
        let q = `
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
			(id, "organizationId", "actionUtc", "userId", category, type, log)
			VALUES 
		`;
        const qParams = [];
        let qIndex = 1;
        const logQ = [];
        for (const log of logs) {
            logQ.push(` (uuid_generate_v4(), $${qIndex++}, $${qIndex++}, $${qIndex++}, $${qIndex++}, $${qIndex++}, $${qIndex++}) `);
            qParams.push(organizationId);
            qParams.push(luxon_1.DateTime.fromJSDate(log.u).toSQL());
            qParams.push(requestUserId);
            qParams.push(log.c);
            qParams.push(log.t);
            qParams.push(JSON.stringify(log));
        }
        await trx.query(q + logQ.join(", "), qParams);
    }
    async addUserActionHistoryItem(log, trx) {
        let result = {
            id: uuid_1.default.v4(),
            organizationId: log.oId,
            actionUtc: log.u,
            userId: log.o,
            category: log.c,
            type: log.t,
            log: log,
        };
        let query = this.dbClient.withSchema(log.oId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory).insert(result);
        if (trx) {
            await query.transacting(trx);
        }
        else {
            await query;
        }
    }
    async listUserActionHistory(organizationId, relationId, relatedField, category, eventTypes) {
        let results = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory)
            .where("organizationId", organizationId)
            .whereIn("type", eventTypes)
            .where("category", category)
            .whereRaw(this._dbClient.raw(` log -> 'd' ->> '${relatedField}' =  ?`, relationId))
            .orderBy("actionUtc", "desc");
        let logs = (await qb.select("log")).map((r) => r.log);
        let users = await dal_manager_1.dbManager.accessPacs.getBasicUserInfoList(organizationId, logs.map((l) => l.o));
        for (const log of logs) {
            results.push({
                actionUser: users.find((u) => u.id == log.o),
                actionUtc: log.u,
                action: log.t,
                category: category,
                changes: log.d && log.d.c
                    ? JSON.parse(log.d.c).map((c) => {
                        return {
                            field: c.f,
                            oldValue: c.o,
                            newValue: c.n,
                            needTranslate: c.i,
                        };
                    })
                    : [],
            });
        }
        return Promise.resolve(results);
    }
    async listMultipleUserActionHistory(organizationId, relationIds, relatedField, category, eventTypes) {
        let results = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory)
            .where("organizationId", organizationId)
            .whereIn("type", eventTypes)
            .where("category", category)
            .whereRaw(` log -> 'd' ->> '${relatedField}' IN(${"'" + relationIds.join("','") + "'"})`)
            .orderBy("actionUtc", "desc");
        let logs = await qb.select("userId", "log");
        let userLogs = logs.map((r) => ({
            userId: r.userId,
            log: r.log,
        }));
        let users = await dal_manager_1.dbManager.accessPacs.getBasicUserInfoList(organizationId, userLogs.map((l) => l.log.o));
        for (const log of userLogs) {
            results.push({
                visitId: log.log.d.visitId,
                actionUser: users.find((u) => u.id == log.log.o),
                actionUtc: log.log.u,
                action: log.log.t,
                category: category,
                changes: log.log.d && log.log.d.c
                    ? JSON.parse(log.log.d.c).map((c) => {
                        return {
                            field: c.f,
                            oldValue: c.o,
                            newValue: c.n,
                            needTranslate: c.i,
                        };
                    })
                    : [],
            });
        }
        return Promise.resolve(results);
    }
    async listUsersTerminalActionHistory(organizationId, category, eventTypes, filter, terminals) {
        let results = [];
        let params = await this.listUsersTerminalActionHistoryQuery(organizationId, category, eventTypes, filter, terminals);
        const { rows } = await this._pgPool.query(params.query, params.queryParams);
        let terminalNames = await dal_manager_1.dbManager.accessDevice.getDeviceBasics(organizationId, rows.map((l) => l.log.d.id));
        const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, lodash_1.default.union(rows.map((l) => l.userId), rows.map((l) => l.log.d.id)));
        for (const row of rows) {
            results.push({
                actionUserId: userCaptionList.find((uc) => uc.id === row.userId)?.id,
                terminalId: row.log.d.id,
                terminalName: terminalNames.find((device) => device.id == row.log.d.id)?.name,
                actionUserCaptions: userCaptionList.find((uc) => uc.id === row.userId)?.captionLines,
                actionUtc: row.log.u,
                action: row.log.t,
                category: category,
                changes: row.log.d && row.log.d.c
                    ? JSON.parse(row.log.d.c).map((c) => {
                        return {
                            field: c.f,
                            oldValue: c.o,
                            newValue: c.n,
                            needTranslate: c.i,
                        };
                    })
                    : [],
            });
        }
        return {
            items: results,
            pagination: {
                skip: filter.pagination.skip,
                take: filter.pagination.take,
                total: params.count,
            },
        };
    }
    async listUsersTerminalActionHistoryQuery(organizationId, category, eventTypes, filter, terminals) {
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
            SELECT "actionUtc", "userId", type, log FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
            WHERE "organizationId" = $${queryParamIndex++}
            AND type = ANY($${queryParamIndex++}::int[])
            AND category = $${queryParamIndex++}`;
        queryParams.push(organizationId, eventTypes, category);
        if (filter.actionUserIds && filter.actionUserIds.length > 0) {
            query += `
            AND "userId" = ANY($${queryParamIndex++}::uuid[])`;
            queryParams.push(filter.actionUserIds);
        }
        if (filter.dateRange) {
            query += `
            AND "actionUtc" >= $${queryParamIndex++}
            AND "actionUtc" <= $${queryParamIndex++}
                `;
            queryParams.push(filter.dateRange.startDateTime, filter.dateRange.endDateTime);
        }
        query += `
            AND log->'d'->>'id' = ANY($${queryParamIndex++}::text[])`;
        queryParams.push(filter.terminalIds ? filter.terminalIds : terminals);
        const total = parseInt((await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q1", queryParams)).rows[0].count);
        query += `
        ORDER BY "actionUtc" DESC `;
        if (filter.pagination && (filter.pagination.skip || filter.pagination.skip === 0) && filter.pagination.take) {
            query += `
            OFFSET $${queryParamIndex++}
            LIMIT $${queryParamIndex++}
            `;
            queryParams.push(filter.pagination.skip, filter.pagination.take);
        }
        return {
            query: query,
            queryParams: queryParams,
            count: total,
        };
    }
    async fetchAuditTerminalsData(params, organizationId, category, onData) {
        let rows;
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(params.query, params.queryParams));
        while (true) {
            try {
                let results = [];
                rows = await new Promise((resolve, reject) => {
                    cursor.read(1000, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                let terminalNames = await dal_manager_1.dbManager.accessDevice.getDeviceBasics(organizationId, rows.map((l) => l.log.d.id));
                const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, lodash_1.default.union(rows.map((l) => l.userId), rows.map((l) => l.log.d.id)));
                for (const row of rows) {
                    results.push({
                        actionUserId: userCaptionList.find((uc) => uc.id === row.userId)?.id,
                        terminalId: row.log.d.id,
                        terminalName: terminalNames.find((device) => device.id == row.log.d.id)?.name,
                        actionUserCaptions: userCaptionList.find((uc) => uc.id === row.userId)?.captionLines,
                        actionUtc: row.log.u,
                        action: row.log.t,
                        category: category,
                        changes: row.log.d && row.log.d.c
                            ? JSON.parse(row.log.d.c).map((c) => {
                                return {
                                    field: c.f,
                                    oldValue: c.o,
                                    newValue: c.n,
                                    needTranslate: c.i,
                                };
                            })
                            : [],
                    });
                }
                await onData(results);
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (rows.length < 1000) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
        return params.count;
    }
    async listUsersAccessControlPointActionHistory(organizationId, category, eventTypes, filter, userAccessRightsAcpList) {
        let results = [];
        let params = await this.listUsersAccessControlPointActionHistoryQuery(organizationId, category, eventTypes, filter, userAccessRightsAcpList);
        let { rows } = await this._pgPool.query(params.query, params.queryParams);
        let accessControlPointBasics = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasics(organizationId, rows.map((l) => l.log.d.id));
        if (filter.regionIds) {
            const regionAccesControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getRegionsAccesControlPointIds(organizationId, filter.regionIds);
            rows = rows.filter((row) => {
                return regionAccesControlPoint.includes(row.log.d.id);
            });
        }
        const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, lodash_1.default.union(rows.map((l) => l.userId), rows.map((l) => l.log.d.id)));
        for (const row of rows) {
            results.push({
                actionUserId: userCaptionList.find((uc) => uc.id === row.userId)?.id,
                accessControlPointId: row.log.d.id,
                accessControlPointName: accessControlPointBasics.find((device) => device.id == row.log.d.id)?.name,
                actionUserCaptions: userCaptionList.find((uc) => uc.id === row.userId)?.captionLines,
                actionUtc: row.log.u,
                action: row.log.t,
                category: category,
                changes: row.log.d && row.log.d.c
                    ? JSON.parse(row.log.d.c).map((c) => {
                        return {
                            field: c.f,
                            oldValue: c.o,
                            newValue: c.n,
                            needTranslate: c.i,
                        };
                    })
                    : [],
            });
        }
        return {
            items: results,
            pagination: {
                skip: filter.pagination.skip,
                take: filter.pagination.take,
                total: params.count,
            },
        };
    }
    async listUsersAccessControlPointActionHistoryQuery(organizationId, category, eventTypes, filter, userAccessRightsAcpList) {
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
            SELECT "actionUtc", "userId", type, log FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
            WHERE "organizationId" = $${queryParamIndex++}
            AND type = ANY($${queryParamIndex++}::int[])
            AND category = $${queryParamIndex++}`;
        queryParams.push(organizationId, eventTypes, category);
        if (filter.actionUserIds && filter.actionUserIds.length > 0) {
            query += `
            AND "userId" = ANY($${queryParamIndex++}::uuid[])`;
            queryParams.push(filter.actionUserIds);
        }
        query += `  
            AND log->'d'->>'id' = ANY($${queryParamIndex++}::text[])`;
        queryParams.push(filter.acpIds ? filter.acpIds : userAccessRightsAcpList);
        if (filter.dateRange) {
            query += `
            AND "actionUtc" >= $${queryParamIndex++}
            AND "actionUtc" <= $${queryParamIndex++}
                `;
            queryParams.push(filter.dateRange.startDateTime, filter.dateRange.endDateTime);
        }
        query += `
        ORDER BY "actionUtc" DESC `;
        const total = parseInt((await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q1", queryParams)).rows[0].count);
        if (filter.pagination && (filter.pagination.skip || filter.pagination.skip === 0) && filter.pagination.take) {
            query += `
            OFFSET $${queryParamIndex++}
            LIMIT $${queryParamIndex++}
            `;
            queryParams.push(filter.pagination.skip, filter.pagination.take);
        }
        return {
            query: query,
            queryParams: queryParams,
            count: total,
        };
    }
    async fetchAuditAccessControlPointsData(params, organizationId, category, filter, onData) {
        let rows;
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(params.query, params.queryParams));
        while (true) {
            try {
                let results = [];
                rows = await new Promise((resolve, reject) => {
                    cursor.read(1000, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                let accessControlPointBasics = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasics(organizationId, rows.map((l) => l.log.d.id));
                if (filter.regionIds) {
                    const regionAccesControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getRegionsAccesControlPointIds(organizationId, filter.regionIds);
                    rows = rows.filter((row) => {
                        return regionAccesControlPoint.includes(row.log.d.id);
                    });
                }
                const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, lodash_1.default.union(rows.map((l) => l.userId), rows.map((l) => l.log.d.id)));
                for (const row of rows) {
                    results.push({
                        actionUserId: userCaptionList.find((uc) => uc.id === row.userId)?.id,
                        accessControlPointId: row.log.d.id,
                        accessControlPointName: accessControlPointBasics.find((device) => device.id == row.log.d.id)?.name,
                        actionUserCaptions: userCaptionList.find((uc) => uc.id === row.userId)?.captionLines,
                        actionUtc: row.log.u,
                        action: row.log.t,
                        category: category,
                        changes: row.log.d && row.log.d.c
                            ? JSON.parse(row.log.d.c).map((c) => {
                                return {
                                    field: c.f,
                                    oldValue: c.o,
                                    newValue: c.n,
                                    needTranslate: c.i,
                                };
                            })
                            : [],
                    });
                }
                await onData(results);
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (rows.length < 1000) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
        return params.count;
    }
    async listUsersIdentityActionHistory(organizationId, category, eventTypes, filter) {
        let results = [];
        let params = await this.listUsersIdentityActionHistoryQuery(organizationId, category, eventTypes, filter);
        const { rows } = await this._pgPool.query(params.query, params.queryParams);
        const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, lodash_1.default.union(rows.map((l) => l.userId), rows.map((l) => l.log.d.id)));
        for (const row of rows) {
            results.push({
                actionUserId: userCaptionList.find((uc) => uc.id === row.userId)?.id,
                actionUserCaptions: userCaptionList.find((uc) => uc.id === row.userId)?.captionLines,
                modifiedUserId: userCaptionList.find((uc) => uc.id === row.log.d.id)?.id,
                modifiedUserCaptions: userCaptionList.find((uc) => uc.id === row.log.d.id)?.captionLines,
                actionUtc: row.log.u,
                action: row.log.t,
                category: category,
                changes: row.log.d && row.log.d.c
                    ? JSON.parse(row.log.d.c).map((c) => {
                        return {
                            field: c.f,
                            oldValue: c.o,
                            newValue: c.n,
                            needTranslate: c.i,
                        };
                    })
                    : [],
            });
        }
        return {
            items: results,
            pagination: {
                skip: filter.pagination.skip,
                take: filter.pagination.take,
                total: params.count,
            },
        };
    }
    async listUsersIdentityActionHistoryQuery(organizationId, category, eventTypes, filter) {
        let filterFlag = false;
        let modifiedUserIds = [];
        if (filter.modifiedUserIds || filter.organizationUnitIds || filter.userGroupIds || filter.filterOrganizationUnitMembersHierarchically) {
            modifiedUserIds = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
                organizationId: organizationId,
                userIds: filter.modifiedUserIds,
                organizationUnitIds: filter.organizationUnitIds ?? [],
                userGroupIds: filter.userGroupIds,
                filterOrganizationUnitMembersHierarchically: filter.filterOrganizationUnitMembersHierarchically,
            });
            filterFlag = true;
        }
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
            SELECT "actionUtc", "userId", type, log FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
            WHERE "organizationId" = $${queryParamIndex++}
            AND type = ANY($${queryParamIndex++}::int[])
            AND category = $${queryParamIndex++}`;
        queryParams.push(organizationId, eventTypes, category);
        if (filter.actionUserIds && filter.actionUserIds.length > 0) {
            const actionUserIds = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
                organizationId: organizationId,
                userIds: filter.actionUserIds,
                organizationUnitIds: filter.organizationUnitIds ?? [],
                userGroupIds: filter.userGroupIds,
                filterOrganizationUnitMembersHierarchically: filter.filterOrganizationUnitMembersHierarchically,
            });
            query += `
            AND "userId" = ANY($${queryParamIndex++}::uuid[])`;
            queryParams.push((actionUserIds && actionUserIds.length) > 0 ? actionUserIds.map((f) => f.userId) : []);
        }
        if (filterFlag) {
            query += `
            AND log->'d'->>'id' = ANY($${queryParamIndex++}::text[])`;
            queryParams.push((modifiedUserIds && modifiedUserIds.length) > 0 ? modifiedUserIds.map((f) => f.userId) : []);
        }
        if (filter.dateRange) {
            query += `
            AND "actionUtc" >= $${queryParamIndex++}
            AND "actionUtc" <= $${queryParamIndex++}
                `;
            queryParams.push(filter.dateRange.startDateTime, filter.dateRange.endDateTime);
        }
        query += `
            ORDER BY "actionUtc" DESC
            `;
        const total = parseInt((await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q1", queryParams)).rows[0].count);
        if (filter.pagination && (filter.pagination.skip || filter.pagination.skip === 0) && filter.pagination.take) {
            query += `
            OFFSET $${queryParamIndex++}
            LIMIT $${queryParamIndex++}
            `;
            queryParams.push(filter.pagination.skip, filter.pagination.take);
        }
        return {
            query: query,
            queryParams: queryParams,
            count: total,
        };
    }
    async fetchAuditIdentityData(params, organizationId, category, onData) {
        const roles = await dal_manager_1.dbManager.accessUser.listRolesOfOrganization(organizationId);
        let rows;
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(params.query, params.queryParams));
        while (true) {
            try {
                let results = [];
                rows = await new Promise((resolve, reject) => {
                    cursor.read(1000, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                const userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, lodash_1.default.union(rows.map((l) => l.userId), rows.map((l) => l.log.d.id)));
                for (const row of rows) {
                    results.push({
                        actionUserId: userCaptionList.find((uc) => uc.id === row.userId)?.id,
                        actionUserCaptions: userCaptionList.find((uc) => uc.id === row.userId)?.captionLines,
                        modifiedUserId: userCaptionList.find((uc) => uc.id === row.log.d.id)?.id,
                        modifiedUserCaptions: userCaptionList.find((uc) => uc.id === row.log.d.id)?.captionLines,
                        actionUtc: row.log.u,
                        action: row.log.t,
                        category: category,
                        changes: row.log.d && row.log.d.c
                            ? JSON.parse(row.log.d.c).map((c) => {
                                return {
                                    field: c.f,
                                    oldValue: c.o,
                                    newValue: c.n,
                                    needTranslate: c.i,
                                };
                            })
                            : [],
                    });
                }
                await onData(results);
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (rows.length < 1000) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
        return params.count;
    }
    async getVisitorTotalVisitCounts(organizationId, visitorProfileIds) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("organizationId", organizationId)
            .whereRaw(`log -> 'p' ->> 'id' IN(${"'" + visitorProfileIds.join("','") + "'"})`)
            .groupByRaw(`log -> 'p' -> 'id'`)
            .count()
            .select(this._dbClient.raw(`log -> 'p' -> 'id' as group`));
        let counts = await qb;
        let result = counts.map((r) => {
            return {
                visitorProfileId: r.group,
                totalVisitCount: parseInt(r.count),
            };
        });
        return Promise.resolve(result);
    }
    async addAuditLog(message) {
        await this.dbClient.withSchema("public").table(dal_db_armon_schema_1.ArmonSchema.tableNames.AuditLogs).insert({
            id: uuid_1.default.v4(),
            actionUtc: new Date(),
            log: message,
        });
    }
    async clearOldAuditLogs(targetDate) {
        let count = (await this.dbClient.withSchema("public").table(dal_db_armon_schema_1.ArmonSchema.tableNames.AuditLogs).where("actionUtc", "<=", targetDate).first().count()).count;
        await this.dbClient.withSchema("public").table(dal_db_armon_schema_1.ArmonSchema.tableNames.AuditLogs).where("actionUtc", "<=", targetDate).delete();
        return Promise.resolve(parseInt(count));
    }
    async listAuditLogs(startDate, endDate) {
        let rows = await this.dbClient.withSchema("public").table(dal_db_armon_schema_1.ArmonSchema.tableNames.AuditLogs).where("actionUtc", ">=", startDate).where("actionUtc", "<=", endDate).select("log");
        return Promise.resolve(rows.map((r) => r.log));
    }
    async listStatusSensorLogs(organizationId, statusSensorLogIds) {
        return Promise.resolve((await this._dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs).whereIn("id", statusSensorLogIds).select("log")).map((s) => s.log));
    }
    async listCounterSensorLogs(organizationId, counterSensorLogIds) {
        return Promise.resolve((await this._dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs).whereIn("id", counterSensorLogIds).select("log")).map((s) => s.log));
    }
    async listTerminalEventsReport(params) {
        let result = {
            pagination: {
                take: params.filter.pagination.take,
                skip: params.filter.pagination.skip,
                total: 0,
            },
            items: [],
        };
        let columns = ["log", "actionUtc"];
        let devices = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return await dal_manager_1.dbManager.accessDevice.listOrganizationDevicesThatUserCanSee({ organizationId: params.organizationId, trx, userId: params.requesterUserId });
        }, params.requesterUserId, params.organizationId);
        if (!params.filter.terminalIds || params.filter.terminalIds.length < 1) {
            params.filter.terminalIds = devices.map((d) => d.id);
        }
        let qb = this._dbClient
            .withSchema(params.organizationId)
            .where("actionUtc", ">=", params.filter.dateRange.startDateTime)
            .where("actionUtc", "<=", params.filter.dateRange.endDateTime);
        switch (params.filter.event) {
            case dal_constants_1.DalConstants.TerminalEventType.TerminalUserAction:
                {
                    qb.table(dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs).whereIn("type", [dal_constants_1.DalConstants.SystemStatusLogType.TerminalUserAction]).whereIn("id", params.filter.terminalIds);
                    columns.push(this._dbClient.raw(`id as "terminalId"`), this._dbClient.raw("type"));
                }
                break;
            case dal_constants_1.DalConstants.TerminalEventType.Connection:
                {
                    qb.table(dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs)
                        .whereIn("type", [dal_constants_1.DalConstants.SystemStatusLogType.TerminaleConnected, dal_constants_1.DalConstants.SystemStatusLogType.TerminalDisconnected])
                        .whereIn("id", params.filter.terminalIds);
                    columns.push(this._dbClient.raw(`id as "terminalId"`), this._dbClient.raw("type"));
                }
                break;
            case dal_constants_1.DalConstants.TerminalEventType.SystemHealthReceived:
                {
                    qb.table(dal_db_armon_schema_1.ArmonSchema.tableNames.SystemStatusLogs).where("type", dal_constants_1.DalConstants.SystemStatusLogType.TerminalHealth).whereIn("id", params.filter.terminalIds);
                    columns.push(this._dbClient.raw(`id as "terminalId"`), this._dbClient.raw("type"));
                }
                break;
            case dal_constants_1.DalConstants.TerminalEventType.UploadOfflineLogs:
                {
                    qb.table(dal_db_armon_schema_1.ArmonSchema.tableNames.UploadLogHistory)
                        .where("organizationId", params.organizationId)
                        .whereRaw(`log->>'cid' IN (${"'" + params.filter.terminalIds.join("','") + "'"})`);
                    columns.push(this._dbClient.raw(`log->>'cid' as "terminalId"`));
                }
                break;
            case dal_constants_1.DalConstants.TerminalEventType.CounterSensorLogReceived:
                {
                    qb.table(dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs)
                        .where("organizationId", params.organizationId)
                        .whereRaw(`log->>'ci' IN (${"'" + params.filter.terminalIds.join("','") + "'"})`);
                    columns.push(this._dbClient.raw(`log->>'ci' as "terminalId"`));
                }
                break;
            case dal_constants_1.DalConstants.TerminalEventType.StatusSensorLogReceived:
                {
                    qb.table(dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs)
                        .where("organizationId", params.organizationId)
                        .whereRaw(`log->>'ci' IN (${"'" + params.filter.terminalIds.join("','") + "'"})`);
                    columns.push(this._dbClient.raw(`log->>'ci' as "terminalId"`));
                }
                break;
            case dal_constants_1.DalConstants.TerminalEventType.TamperLogReceived: {
                qb.table(dal_db_armon_schema_1.ArmonSchema.tableNames.TamperSwitchLogs)
                    .where("organizationId", params.organizationId)
                    .whereRaw(`log->>'ci' IN (${"'" + params.filter.terminalIds.join("','") + "'"})`);
                columns.push(this._dbClient.raw(`log->>'ci' as "terminalId"`));
            }
        }
        result.pagination.total = parseInt((await qb.clone().count().first()).count);
        qb.offset(params.filter.pagination.skip).limit(params.filter.pagination.take).orderBy("actionUtc", "desc");
        let dbItems = await qb.columns(columns).select();
        let deviceIds = lodash_1.default.uniq(dbItems.filter((d) => d.terminalId).map((d) => d.terminalId));
        let extensions = await dal_manager_1.dbManager.accessDevice.listArmonControlPanelExtensions(params.organizationId, deviceIds);
        let sensorDetails = [];
        if ([dal_constants_1.DalConstants.TerminalEventType.TamperLogReceived, dal_constants_1.DalConstants.TerminalEventType.StatusSensorLogReceived, dal_constants_1.DalConstants.TerminalEventType.CounterSensorLogReceived].indexOf(params.filter.event) > -1) {
            sensorDetails = await dal_manager_1.dbManager.accessDevice.listSensorDetails(params.organizationId, deviceIds, null);
        }
        for (const dbItem of dbItems) {
            let device = devices.find((t) => t.id == dbItem.terminalId);
            let item = {
                actionUtc: dbItem.actionUtc,
                data: dbItem.log,
                eventType: params.filter.event,
                terminalId: dbItem.terminalId,
                terminalName: device ? device.name : "",
            };
            if (device.brand === dal_constants_1.DalConstants.DeviceBrand.Armon && !dal_constants_1.DalConstants.terminalModels.includes(device.model)) {
                item.extensionName = device.name;
                let extensionInfo = extensions?.find((d) => d.deviceId == device.id);
                let mainControlPanel;
                if (extensionInfo && devices.some((d) => d.id == extensionInfo.mainControlPanelId)) {
                    mainControlPanel = devices.find((d) => d.id == extensionInfo.mainControlPanelId);
                }
                item.terminalName = mainControlPanel?.name;
            }
            if ([dal_constants_1.DalConstants.TerminalEventType.TamperLogReceived, dal_constants_1.DalConstants.TerminalEventType.StatusSensorLogReceived, dal_constants_1.DalConstants.TerminalEventType.CounterSensorLogReceived].indexOf(params.filter.event) > -1) {
                let sensorInfo = sensorDetails.find((s) => s.id == dbItem.log.i);
                if (sensorInfo) {
                    item.accessControlPointId = sensorInfo.accessControlPointId;
                    item.accessControlPointName = sensorInfo.accessControlPointName;
                    item.sensorName = sensorInfo.name;
                    item.sensorNumber = sensorInfo.number;
                }
            }
            if (dal_constants_1.DalConstants.TerminalEventType.Connection == params.filter.event) {
                item.data = { s: dbItem.type == dal_constants_1.DalConstants.SystemStatusLogType.TerminaleConnected };
            }
            if (dal_constants_1.DalConstants.TerminalEventType.TerminalUserAction == params.filter.event) {
                item.data = dbItem.log;
            }
            result.items.push(item);
        }
        return Promise.resolve(result);
    }
    async listUserPastVisits(organizationId, dateRange, visitedUserId) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("organizationId", organizationId)
            .where("startUtc", ">=", dateRange.startDateTime)
            .where("startUtc", "<=", dateRange.endDateTime);
        if (visitedUserId) {
            qb.where("visitedUserId", visitedUserId);
        }
        return qb.select("id", "startUtc", "endUtc");
    }
    async listPregistrationReport(organizationId, dateRange) {
        let pLogs = await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitorPreregistrationHistory)
            .where("organizationId", organizationId)
            .where("actionUtc", ">=", dateRange.startDateTime)
            .where("actionUtc", "<=", dateRange.endDateTime)
            .select();
        let vLogs = await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("organizationId", organizationId)
            .where("startUtc", ">=", dateRange.startDateTime)
            .where("startUtc", "<=", dateRange.endDateTime)
            .whereIn("id", pLogs.map((p) => p.log.vid))
            .select("id");
        return Promise.resolve(pLogs.map((p) => {
            return {
                actionUtc: p.actionUtc,
                id: p.id,
                turnedIntoVisit: vLogs.some((v) => v == p.log.vid),
            };
        }));
    }
    async listVisitStats(organizationId, dateRange) {
        let vLogs = await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory)
            .where("organizationId", organizationId)
            .where("endUtc", ">=", dateRange.startDateTime)
            .where("endUtc", "<=", dateRange.endDateTime)
            .select("id", "endUtc");
        let pLogs = await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.VisitorPreregistrationHistory)
            .where("organizationId", organizationId)
            .where("actionUtc", ">=", dateRange.startDateTime)
            .where("actionUtc", "<=", dateRange.endDateTime)
            .whereRaw(`log->>'vid' IN (${"'" + vLogs.map((v) => v.id).join("','") + "'"})`)
            .select(this._dbClient.raw(`log->>'vid' as vid`));
        return Promise.resolve(vLogs.map((p) => {
            return {
                date: p.endUtc,
                id: p.id,
                fromPreregistration: pLogs.some((v) => v.vid == p.id),
            };
        }));
    }
    async addLogHistoryItem(organizationId, referenceId, type, data, actionUtc, trx) {
        let qb = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory).insert({
            id: referenceId,
            insertionUtc: new Date().toISOString(),
            actionUtc: actionUtc,
            type: type,
            data: data,
        });
        if (trx) {
            qb.transacting(trx);
        }
        await qb;
    }
    async addProcessLogHistoryItem(organizationId, processLogItem, trx) {
        let qb = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogProcessHistory).insert(processLogItem);
        if (trx) {
            qb.transacting(trx);
        }
        await qb;
    }
    async listLogHistoryItem(organizationId, startUtc, endUtc, maxLogCount) {
        return this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory)
            .where("insertionUtc", ">=", startUtc)
            .where("insertionUtc", "<=", endUtc)
            .orderBy("id", "desc")
            .limit(maxLogCount)
            .select();
    }
    async getLastProcessLogHistoryItem(organizationId) {
        return this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.LogProcessHistory).orderBy("startUtc", "desc").first();
    }
    async getAccessLogsById(organizationId, startUtc, endUtc, logIds) {
        let logs = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs)
            .where("actionUtc", ">=", startUtc)
            .where("actionUtc", "<=", endUtc)
            .whereIn("id", logIds)
            .orderBy("actionUtc", "asc")
            .select("log");
        return Promise.resolve(logs.map((l) => l.log));
    }
    async getVisitCountsByGenericFilter(organizationId, filter) {
        const client = await this._pgPool.connect();
        let queryParamsIndex = 1;
        let queryParams = [];
        let query = ``;
        if (filter.endDay) {
            filter.endDay = (0, moment_1.default)(filter.endDay).add(1, "day").toDate();
        }
        switch (filter.filterType) {
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitorId:
                query += `
                SELECT "log"->'p'->'fields'->'name' AS "name", "visitorProfileId" AS "id", COUNT (*) AS count FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory}"
                WHERE "organizationId" = $${queryParamsIndex++}
                `;
                queryParams.push(organizationId);
                if (filter.filterValue) {
                    query += ` AND "visitorProfileId" = $${queryParamsIndex++}`;
                    queryParams.push(filter.filterValue);
                }
                if (filter.startDay) {
                    query += ` AND "startUtc" >= $${queryParamsIndex++}`;
                    queryParams.push(filter.startDay);
                }
                if (filter.endDay) {
                    query += ` AND "endUtc" <= $${queryParamsIndex++}`;
                    queryParams.push((0, moment_1.default)(filter.endDay).add(1, "day").toDate());
                }
                query += `
                GROUP BY ("visitorProfileId", "log"->'p'->'fields'->'name')
                `;
                break;
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitReason:
                query += `
                SELECT "log"->'v'->'reason'::TEXT AS "name", COUNT (*) AS count FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory}"
                WHERE "organizationId" = $${queryParamsIndex++}`;
                queryParams.push(organizationId);
                if (filter.startDay) {
                    query += ` AND "startUtc" >= $${queryParamsIndex++}`;
                    queryParams.push(filter.startDay);
                }
                if (filter.endDay) {
                    query += ` AND "endUtc" <= $${queryParamsIndex++}`;
                    queryParams.push(filter.endDay);
                }
                query += `
                GROUP BY ("log"->'v'->'reason')
                `;
                break;
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitedPersonId:
                query += `
                SELECT (("log"->'v'->'visitedPerson'->>'name') || ' ' || ("log"->'v'->'visitedPerson'->>'surname'))  AS "name",
                "visitedUserId" AS "id", COUNT (*) AS count FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory}"
                WHERE "organizationId" = $${queryParamsIndex++} AND ("visitedUserId") IS NOT NULL
                `;
                queryParams.push(organizationId);
                if (filter.filterValue) {
                    query += ` AND "visitedUserId" = $${queryParamsIndex++}`;
                    queryParams.push(filter.filterValue);
                }
                if (filter.startDay) {
                    query += ` AND "startUtc" >= $${queryParamsIndex++}`;
                    queryParams.push(filter.startDay);
                }
                if (filter.endDay) {
                    query += ` AND "endUtc" <= $${queryParamsIndex++}`;
                    queryParams.push(filter.endDay);
                }
                query += `
                GROUP BY ("visitedUserId", "log"->'v'->'visitedPerson'->>'name', "log"->'v'->'visitedPerson'->>'surname')
                `;
                break;
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByLicensePlate:
                query += `
                    SELECT ("log" -> 'p'->'fields'->>'licencePlate')  AS "name",
                    COUNT (*) AS count FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory}"
                    WHERE "organizationId" = $${queryParamsIndex++} AND ("log" -> 'p'->'fields'->>'licencePlate') IS NOT NULL
                    `;
                queryParams.push(organizationId);
                if (filter.filterValue) {
                    query += ` AND "log" -> 'p'->'fields'->>'licencePlate' ILIKE '%' || $${queryParamsIndex++} || '%' AND (log)::text ILIKE '%' || $${queryParamsIndex++} || '%'`;
                    queryParams.push(filter.filterValue.toUpperCase());
                }
                if (filter.startDay) {
                    query += ` AND "startUtc" >= $${queryParamsIndex++}`;
                    queryParams.push(filter.startDay);
                }
                if (filter.endDay) {
                    query += ` AND "endUtc" <= $${queryParamsIndex++}`;
                    queryParams.push(filter.endDay);
                }
                query += `
                    GROUP BY ("log" -> 'p'->'fields'->>'licencePlate')
                    `;
                break;
            default:
                break;
        }
        query += `
        ORDER BY ("count") DESC`;
        let totalCountResult = (await client.query(`SELECT COUNT (DISTINCT "name") FROM (${query}) AS T`, queryParams)).rows;
        query += `
        OFFSET $${queryParamsIndex++} LIMIT $${queryParamsIndex++}`;
        queryParams.push(filter.pagination.skip);
        queryParams.push(filter.pagination.take);
        let rows = (await client.query(query, queryParams)).rows;
        let result = {
            pagination: {
                take: filter.pagination.take,
                skip: filter.pagination.skip,
                total: totalCountResult[0] ? parseInt(totalCountResult[0].count) : 0,
            },
            items: rows,
        };
        client.release();
        return result;
    }
    async getUserDailyAccessLogs(organizationId, userId) {
        let startOfDay = (0, moment_1.default)(new Date()).startOf("day");
        let endOfDay = (0, moment_1.default)(new Date()).endOf("day");
        return (await this._pgPool.query(`
            SELECT log FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}
            WHERE "actionUtc" >= ${startOfDay.toDate()}
            AND "actionUtc" <= ${endOfDay.toDate()}
            AND log->>'o' = $1
            AND "organizationId" = $2`, [userId, organizationId])).rows;
    }
    async getAccessLogs(organizationId, args) {
        let filterUsers = (await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: organizationId,
            userIds: args.userIds,
            organizationUnitIds: args.organizationUnitIds,
            userGroupIds: args.userGroupIds,
            filterOrganizationUnitMembersHierarchically: args.filterOrganizationUnitMembersHierarchically,
        })).map((f) => f.userId);
        let queryParams = [];
        let queryParamIndex = 1;
        let query = `SELECT log FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
        WHERE "organizationId" = $${queryParamIndex++}`;
        queryParams.push(organizationId);
        if (args.direction) {
            query += `
            AND log ->> 'd' = $${queryParamIndex++}`;
            queryParams.push(args.direction);
        }
        if (args.status !== app_enums_1.enums.AccessReportFilterAccessResultType.All) {
            if (args.status === app_enums_1.enums.AccessReportFilterAccessResultType.Success) {
                query += `
                AND log ->> 's' = $${queryParamIndex++}`;
                queryParams.push("true");
            }
            else if (args.status === app_enums_1.enums.AccessReportFilterAccessResultType.Fail) {
                query += `
                AND log ->> 's' IS NULL`;
            }
        }
        if (filterUsers.length > 0) {
            query += `
            AND log ->> 'o' IN('${filterUsers.join("','")}')`;
        }
        if (args.accessControlPointIds && args.accessControlPointIds.length > 0) {
            query += `
            AND log ->> 'a' IN('${args.accessControlPointIds.join("','")}')`;
        }
        if (args.recordReasons && args.recordReasons.length > 0) {
            query += `
            AND log ->> 'r' IN('${args.recordReasons.join("','")}')`;
        }
        query += `
			AND "actionUtc" > $${queryParamIndex++}
            ORDER BY("actionUtc") DESC
            LIMIT $${queryParamIndex++} `;
        queryParams.push(new Date(new Date(new Date().setHours(0)).setMinutes(0)));
        queryParams.push(args.take);
        let result = await this._pgPool.query(query, queryParams);
        return result.rows.map((r) => r.log);
    }
    async getVisitorIdAndRegistrationPointIdFromHistory(params) {
        const visitHistory = await (params.trx ?? this._pgPool).query(`
		SELECT id, "organizationId", "startUtc", "endUtc", "visitorProfileId", "visitedUserId", "visitorRegistrationPointId", log
		FROM "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.VisitHistory}
		WHERE id = $1
		`, [params.visitId]);
        if (visitHistory.rowCount === 0) {
            return null;
        }
        const visitHistoryRow = visitHistory.rows[0];
        return {
            date: visitHistoryRow.startUtc,
            visitorProfileId: visitHistoryRow.visitorProfileId,
            visitorRegistrationPointId: visitHistoryRow.visitorRegistrationPointId,
        };
    }
    async getUsersAccessLogsActionTimesWithDirection(params, filters) {
        let innerQuery = `
			SELECT log->>'o' as "userId",
					jsonb_build_object('direction',(log->>'d')::integer,'actionTime', "actionUtc") as "actionTime" 
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
			WHERE "actionUtc" > $1 AND "actionUtc" < $2 				   
		`;
        const bindings = [filters.startDate.toISO(), filters.endDate.toISO()];
        if (filters.userIds?.length) {
            innerQuery += `
			AND log->>'o' = ANY($3)
			`;
            bindings.push(filters.userIds);
        }
        let finalQuery = `
			SELECT act."userId", jsonb_agg(act."actionTime") as "actionTimeWithDirection" FROM 
			(
			` +
            innerQuery +
            ' ORDER BY "userId","actionUtc" ) as act GROUP BY "userId"';
        const result = await (params.trx ?? this._pgPool).query(finalQuery, bindings);
        return result.rows;
    }
}
exports.PSQLDalAccessLog = PSQLDalAccessLog;
async function assignAutoShift(organizationId, params, trx) {
    if (trx) {
        const autoShiftsRules = await (0, dal_access_cache_redis_1.getAutoShiftCache)({ organizationId, trx, redisCache: params.redisCache });
        if (autoShiftsRules?.existency) {
            await trx.query(`SELECT "${organizationId}".assign_auto_shift($1, null, $2, $3, $4)`, [
                params.credentialOwnerUserId,
                params.logId,
                luxon_1.DateTime.fromISO(params.generationTime).toSQLDate(),
                luxon_1.DateTime.fromISO(params.generationTime).plus({ day: 1 }).toSQLDate(),
            ]);
        }
    }
    else {
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const autoShiftsRules = await (0, dal_access_cache_redis_1.getAutoShiftCache)({ organizationId, trx, redisCache: params.redisCache });
            if (autoShiftsRules?.existency) {
                await trx.query(`SELECT "${organizationId}".assign_auto_shift($1, null, $2, $3, $4)`, [
                    params.credentialOwnerUserId,
                    params.logId,
                    luxon_1.DateTime.fromISO(params.generationTime).toSQLDate(),
                    luxon_1.DateTime.fromISO(params.generationTime).plus({ day: 1 }).toSQLDate(),
                ]);
            }
        });
    }
}
exports.assignAutoShift = assignAutoShift;
