"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessCamera = void 0;
const lodash_1 = __importDefault(require("lodash"));
const uuid_1 = __importDefault(require("uuid"));
const dal_constants_1 = require("../../dal.constants");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const dal_access_rdb_1 = require("../rdb/dal.access.rdb");
class PSQLDalAccessCamera extends dal_access_rdb_1.RDBAccess {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async removeCamera(organizationId, cameraId) {
        await this.dbClient.transaction(async (trx) => {
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameras).where("id", cameraId).update({
                deletedAt: new Date(),
            });
        });
    }
    async upsertCamera(organizationId, args) {
        let id = args.id || uuid_1.default.v4();
        await this.dbClient.transaction(async (trx) => {
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameraActions).where("cameraId", id).delete();
            if (args.id) {
                await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameras)
                    .where("id", args.id)
                    .update({
                    name: args.name,
                    userName: args.userName ? args.userName : null,
                    proxyTerminalId: args.proxyTerminalId ? args.proxyTerminalId : null,
                    snapshotUrl: args.snapshotUrl ? args.snapshotUrl : null,
                    updatedAt: new Date(),
                });
            }
            else {
                await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameras)
                    .insert({
                    id: id,
                    organizationId: organizationId,
                    name: args.name,
                    userName: args.userName ? args.userName : null,
                    proxyTerminalId: args.proxyTerminalId ? args.proxyTerminalId : null,
                    snapshotUrl: args.snapshotUrl ? args.snapshotUrl : null,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                });
            }
            if (args.password) {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameras).where("id", id).update({ password: args.password });
            }
            if (args.mappings) {
                let dryContactInputIds = args.mappings.filter((t) => t.type == dal_constants_1.DalConstants.CameraActionType.Sensor).map((d) => d.dryContactInputId);
                dryContactInputIds = lodash_1.default.uniq(dryContactInputIds);
                let dryContactInputs = (await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.deviceDryContactInputs)
                    .whereNull("deletedAt")
                    .whereIn("id", dryContactInputIds)
                    .select("id", "name", "accessControlPointId", "deviceId", "number", "nc", "type", "settings"));
                let cameraActions = [];
                for (const actionMapping of args.mappings) {
                    switch (actionMapping.type) {
                        case dal_constants_1.DalConstants.CameraActionType.Auth:
                        case dal_constants_1.DalConstants.CameraActionType.RemoteAccess:
                            {
                                cameraActions.push({
                                    cameraId: id,
                                    settings: {
                                        direction: actionMapping.direction,
                                        result: actionMapping.result,
                                    },
                                    type: actionMapping.type,
                                    accessControlPointId: actionMapping.accessControlPointId,
                                });
                            }
                            break;
                        case dal_constants_1.DalConstants.CameraActionType.Sensor:
                            {
                                cameraActions.push({
                                    cameraId: id,
                                    settings: {
                                        dryContactInputId: actionMapping.dryContactInputId,
                                        filterEvents: actionMapping.statusSensorEvents || actionMapping.counterSensorEvents || [],
                                    },
                                    type: actionMapping.type,
                                    deviceId: dryContactInputs.find((d) => d.id == actionMapping.dryContactInputId).deviceId,
                                });
                            }
                            break;
                        case dal_constants_1.DalConstants.CameraActionType.View:
                            {
                                cameraActions.push({
                                    cameraId: id,
                                    settings: null,
                                    type: actionMapping.type,
                                    accessControlPointId: actionMapping.accessControlPointId,
                                });
                            }
                            break;
                    }
                }
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameraActions).insert(cameraActions);
            }
        });
        return Promise.resolve(id);
    }
    async getCameraActions(organizationId, cameraId, trx) {
        let cameraActions = await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameraActions).where("cameraId", cameraId).select();
        let devices = await trx
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.devices)
            .whereIn("id", cameraActions.filter((a) => a.deviceId).map((a) => a.deviceId))
            .select("id", "name");
        let dryContactInputs = (await trx
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.deviceDryContactInputs)
            .whereNull("deletedAt")
            .whereIn("id", cameraActions
            .filter((a) => a.deviceId && a.type == dal_constants_1.DalConstants.CameraActionType.Sensor)
            .map((a) => a.settings.dryContactInputId))
            .select("id", "name", "accessControlPointId", "deviceId", "number", "nc", "type", "settings"));
        let accessControlPointInfo = (await trx
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints)
            .whereIn("id", cameraActions
            .filter((a) => a.accessControlPointId)
            .map((a) => a.accessControlPointId)
            .concat(dryContactInputs.map((d) => d.accessControlPointId)))
            .select("id", "name", "deviceId")
            .whereNull("deletedAt"));
        let cameraActionDetaileds = [];
        for (const action of cameraActions) {
            switch (action.type) {
                case dal_constants_1.DalConstants.CameraActionType.Auth:
                case dal_constants_1.DalConstants.CameraActionType.RemoteAccess:
                case dal_constants_1.DalConstants.CameraActionType.View:
                    {
                        let acp = accessControlPointInfo.find((a) => a.id == action.accessControlPointId);
                        cameraActionDetaileds.push({
                            cameraId: action.cameraId,
                            accessControlPointId: action.accessControlPointId,
                            accessControlPointName: acp ? acp.name : null,
                            type: action.type,
                            direction: action.settings ? action.settings.direction : null,
                            result: action.settings ? action.settings.result : null,
                        });
                    }
                    break;
                case dal_constants_1.DalConstants.CameraActionType.Sensor:
                    {
                        let dryContactInput = dryContactInputs.find((a) => a.id == action.settings.dryContactInputId);
                        let acp = accessControlPointInfo.find((a) => a.id == dryContactInput.accessControlPointId);
                        let _dryContactInputType = dryContactInputs.find((a) => a.id == action.settings.dryContactInputId).type;
                        cameraActionDetaileds.push({
                            cameraId: action.cameraId,
                            terminalId: action.deviceId,
                            accessControlPointId: acp ? acp.id : "",
                            accessControlPointName: acp ? acp.name : "",
                            terminalName: devices.find((d) => d.id == action.deviceId).name,
                            dryContactInputId: action.settings.dryContactInputId,
                            dryContactInputName: dryContactInputs.find((a) => a.id == action.settings.dryContactInputId).name,
                            type: action.type,
                            dryContactInputType: _dryContactInputType,
                            counterSensorEvents: _dryContactInputType === dal_constants_1.DalConstants.libEnumsV2.DryContactType.CounterSensor
                                ? action.settings.filterEvents
                                : [],
                            statusSensorEvents: _dryContactInputType === dal_constants_1.DalConstants.libEnumsV2.DryContactType.StatusSensor
                                ? action.settings.filterEvents
                                : [],
                        });
                    }
                    break;
            }
        }
        return Promise.resolve(cameraActionDetaileds);
    }
    async getCamera(organizationId, cameraId) {
        return await this.dbClient.transaction(async (trx) => {
            let qResult = await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameras).where("id", cameraId).whereNull("deletedAt").first();
            let returnValue = {
                id: qResult.id,
                organizationId: qResult.organizationId,
                name: qResult.name,
                userName: qResult.userName,
                password: qResult.password,
                proxyTerminalId: qResult.proxyTerminalId,
                snapshotUrl: qResult.snapshotUrl,
                mappings: await this.getCameraActions(organizationId, cameraId, trx),
            };
            return Promise.resolve(returnValue);
        });
    }
    async getCameraForAccessControlPoint(organizationId, accessControlPointId) {
        return await this.dbClient.transaction(async (trx) => {
            let result = [];
            let cameras = await trx
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.cameras + " as c")
                .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.cameraActions + " as ca", "ca.cameraId", "c.id")
                .where("ca.accessControlPointId", accessControlPointId)
                .where("ca.type", dal_constants_1.DalConstants.CameraActionType.View)
                .whereNull("c.deletedAt")
                .select("c.*");
            for (let row of cameras) {
                let cameraInfo = {
                    id: row.id,
                    organizationId: row.organizationId,
                    name: row.name,
                    userName: row.userName,
                    password: row.password,
                    proxyTerminalId: row.proxyTerminalId,
                    snapshotUrl: row.snapshotUrl,
                };
                if (!result.some((c) => c.id == row.id)) {
                    result.push(cameraInfo);
                }
            }
            return Promise.resolve(result);
        });
    }
    async listCameras(organizationId, args) {
        let result = {
            take: args.take,
            skip: args.skip,
            total: 0,
            items: [],
        };
        let qb = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.cameras).where("organizationId", organizationId).whereNull("deletedAt");
        if (args.name)
            qb.where("name", "ilike", "%" + args.name + "%");
        result.total = parseInt(await qb
            .clone()
            .count()
            .then((r) => r[0].count));
        if (args.skip)
            qb.offset(args.skip);
        if (args.take)
            qb.limit(args.take);
        await this.dbClient.transaction(async (trx) => {
            await qb
                .select()
                .orderBy("name")
                .then(async (rows) => {
                for (let row of rows) {
                    let cameraInfo = {
                        id: row.id,
                        organizationId: row.organizationId,
                        name: row.name,
                        userName: row.userName,
                        proxyTerminalId: row.proxyTerminalId,
                        snapshotUrl: row.snapshotUrl,
                        mappings: await this.getCameraActions(row.organizationId, row.id, trx),
                    };
                    result.items.push(cameraInfo);
                }
            });
        });
        return Promise.resolve(result);
    }
    async listCamerasForDeviceV2(organizationId, deviceIds, trx) {
        let result = [];
        let cameraInforFromProxyTerminal = (await trx
            .withSchema(organizationId)
            .table("cameras as c")
            .leftJoin("cameraActions as ca", "ca.cameraId", "c.id")
            .whereIn("c.proxyTerminalId", deviceIds)
            .whereNull("deletedAt")
            .select("c.id", "c.name", "c.snapshotUrl", "c.userName", "c.password", "ca.*"));
        let remoteAccessCameraActions = (await trx
            .withSchema(organizationId)
            .table("cameras as c")
            .innerJoin("cameraActions as ca", "ca.cameraId", "c.id")
            .innerJoin("accessControlPoints as acp", "acp.id", "ca.accessControlPointId")
            .whereNull("c.deletedAt")
            .whereNull("acp.deletedAt")
            .whereIn("ca.type", [dal_constants_1.DalConstants.CameraActionType.Auth, dal_constants_1.DalConstants.CameraActionType.RemoteAccess])
            .whereIn("acp.deviceId", deviceIds)
            .select("c.id", "c.name", "c.snapshotUrl", "c.userName", "c.password", "ca.*"));
        let deviceCameraActions = (await trx
            .withSchema(organizationId)
            .table("cameras as c")
            .innerJoin("cameraActions as ca", "ca.cameraId", "c.id")
            .whereNull("c.deletedAt")
            .where("ca.type", dal_constants_1.DalConstants.CameraActionType.Sensor)
            .whereIn("ca.deviceId", deviceIds)
            .select("c.id", "c.name", "c.snapshotUrl", "c.userName", "c.password", "ca.*"));
        let cameras = lodash_1.default.uniqBy(remoteAccessCameraActions.concat(deviceCameraActions).concat(cameraInforFromProxyTerminal), "id");
        for (const camera of cameras) {
            let cameraActions = remoteAccessCameraActions.filter((c) => c.id == camera.id).concat(deviceCameraActions.filter((d) => d.id == camera.id));
            result.push({
                id: camera.id,
                snapshotUrl: camera.snapshotUrl,
                userName: camera.userName,
                password: camera.password,
                mappings: cameraActions.map((c) => {
                    return {
                        accessControlPointId: c.accessControlPointId,
                        cameraId: c.cameraId,
                        deviceId: c.deviceId,
                        settings: c.settings,
                        type: c.type,
                    };
                }),
            });
        }
        return Promise.resolve(result);
    }
    async listCameraActionsForDevice(organizationId, deviceIds, trx) {
        let result = [];
        let remoteAccessCameraActions = (await trx
            .withSchema(organizationId)
            .table("cameras as c")
            .innerJoin("cameraActions as ca", "ca.cameraId", "c.id")
            .innerJoin("accessControlPoints as acp", "acp.id", "ca.accessControlPointId")
            .whereNull("c.deletedAt")
            .whereNull("acp.deletedAt")
            .whereIn("ca.type", [dal_constants_1.DalConstants.CameraActionType.Auth, dal_constants_1.DalConstants.CameraActionType.RemoteAccess])
            .whereIn("acp.deviceId", deviceIds)
            .select("c.id", "c.name", "c.snapshotUrl", "c.userName", "c.password", "ca.*"));
        let deviceCameraActions = (await trx
            .withSchema(organizationId)
            .table("cameras as c")
            .innerJoin("cameraActions as ca", "ca.cameraId", "c.id")
            .whereNull("c.deletedAt")
            .where("ca.type", dal_constants_1.DalConstants.CameraActionType.Sensor)
            .whereIn("ca.deviceId", deviceIds)
            .select("c.id", "c.name", "c.snapshotUrl", "c.userName", "c.password", "ca.*"));
        let cameraActions = remoteAccessCameraActions.concat(deviceCameraActions);
        for (const action of cameraActions) {
            switch (action.type) {
                case dal_constants_1.DalConstants.CameraActionType.Auth:
                case dal_constants_1.DalConstants.CameraActionType.RemoteAccess:
                case dal_constants_1.DalConstants.CameraActionType.View:
                    {
                        result.push({
                            accessControlPointId: action.accessControlPointId,
                            type: action.type,
                            direction: action.settings ? action.settings.direction : null,
                            result: action.settings ? action.settings.result : null,
                        });
                    }
                    break;
                case dal_constants_1.DalConstants.CameraActionType.Sensor:
                    {
                        result.push({
                            dryContactInputId: action.accessControlPointId,
                            type: action.type,
                            counterSensorEvents: action.settings.filterEvents,
                            statusSensorEvents: action.settings.filterEvents,
                        });
                    }
                    break;
            }
        }
        return Promise.resolve(result);
    }
}
exports.PSQLDalAccessCamera = PSQLDalAccessCamera;
