"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.anonymizeOrganization = exports.uploadDeviceSystemStatusLogsPg = exports.getUserOrganizationRedisPersistentData = exports.getOrganization = exports.getOrganizationRedisPersistentData = exports.getRegionRedisPersistentData = exports.getAccessPointRedisPersistentData = exports.getDeviceNotificationInfoPersistentData = exports.getDeviceStatePersistentData = exports.uploadDeviceSystemStatusLogsNew = exports.uploadTamperSwitchLogsPg = exports.uploadStatusSensorLogsPg = exports.onAccessLogsChanged = exports.deleteDeadLetterOfflineLog = exports.insertUnhandledDeviceLog = exports.updateAccessLogForSensorData = exports.uploadCounterLogs = exports.updateUserRegionTicketUnits = exports.upsertAccessRuleHistory = exports.upsertRegionStates = exports.getUserInfoByCredentialData = exports.addAccessLogFromDevice = exports.getOrganizationIdNamePairs = exports.updateUsername = exports.createOrganization = exports.systemTransaction = void 0;
const crypto_1 = __importDefault(require("crypto"));
const lodash_1 = __importDefault(require("lodash"));
const moment_1 = __importDefault(require("moment"));
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const business_device_1 = require("../../../business/business.device");
const dal_constants_1 = require("../../dal.constants");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_oAuthClients_1 = require("../../db/predefined/predefined.oAuthClients");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_cache_1 = require("../access-interfaces/dal.access.cache");
const dal_access_error_1 = require("../dal.access.error");
const luxon_1 = require("luxon");
const partition_manager_1 = require("../../../components/pgmaintenance/db/partition-manager");
async function systemTransaction(pool, trx) {
    const client = await pool.connect();
    try {
        await client.query(`BEGIN`);
        const result = await trx(client);
        await client.query("COMMIT");
        client.release();
        return result;
    }
    catch (error) {
        await client.query("ROLLBACK");
        client.release(error);
        throw error;
    }
}
exports.systemTransaction = systemTransaction;
async function createOrganization(params, pool) {
    const organizationDbRes = await params.trx.query(`
                SELECT id FROM public."organizationList"
                WHERE name = $1
            `, [params.request.name]);
    if (organizationDbRes.rowCount > 0) {
        throw {
            errorCode: dal_access_error_1.DbAccessErrorCode.BAD_REQUEST,
            message: JSON.stringify({
                en: "This organization is already exists",
                tr: "Bu organizasyon zaten tanımlı",
            }),
        };
    }
    const userDbRes = await params.trx.query(`
                SELECT id FROM public."userList"
                WHERE lower(username) = lower($1)
            `, [params.request.administratorAccount.username]);
    if (userDbRes.rowCount > 0) {
        throw {
            errorCode: dal_access_error_1.DbAccessErrorCode.BAD_REQUEST,
            message: JSON.stringify({
                en: "This username is already exists",
                tr: "Bu kullanıcı adı zaten tanımlı.",
            }),
        };
    }
    const opTime = new Date();
    const ecdh = crypto_1.default.createECDH("prime256v1");
    const publicKey = ecdh.generateKeys("base64");
    const privateKey = ecdh.getPrivateKey("base64");
    const systemAdminRoleId = uuid_1.default.v4();
    const organizationId = params.request.id;
    await params.trx.query(`SELECT public.clone_schema('organization','${organizationId}', FALSE);`);
    await params.trx.query((0, dal_utils_1.updateViewsAfterClone)(organizationId, params.dbuser, params.dbsuperuser));
    await params.trx.query(`
		GRANT USAGE ON SCHEMA "${organizationId}" TO ${params.dbuser};
		ALTER DEFAULT PRIVILEGES IN SCHEMA "${organizationId}" GRANT INSERT, SELECT, UPDATE, DELETE ON TABLES TO ${params.dbuser};
		ALTER DEFAULT PRIVILEGES IN SCHEMA "${organizationId}" GRANT SELECT, UPDATE, USAGE ON SEQUENCES TO ${params.dbuser};				
		ALTER DEFAULT PRIVILEGES IN SCHEMA "${organizationId}" GRANT EXECUTE ON FUNCTIONS TO ${params.dbuser};
	`);
    await params.trx.query(`COMMIT`);
    await (0, partition_manager_1.createForwardPartitions)(pool, 1);
    await params.trx.query(`BEGIN`);
    let credentialTypes = [];
    for (const credentialType of params.request.credentialTypes) {
        if (credentialType === dal_constants_1.DalConstants.libEnumsV2.CredentialType.MiFare ||
            credentialType === dal_constants_1.DalConstants.libEnumsV2.CredentialType.ProximityCard ||
            credentialType === dal_constants_1.DalConstants.libEnumsV2.CredentialType.UHFRfid ||
            credentialType === dal_constants_1.DalConstants.libEnumsV2.CredentialType.VehiclePlate ||
            credentialType === dal_constants_1.DalConstants.libEnumsV2.CredentialType.MRZ) {
            credentialTypes.push({
                type: credentialType,
                dataConverter: "return r;",
                oneTimeAssignable: false,
            });
        }
        else {
            credentialTypes.push({
                type: credentialType,
                oneTimeAssignable: false,
            });
        }
    }
    await params.trx.query(`
                INSERT INTO public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}" (id, name)
                VALUES ($1, $2)
            `, [organizationId, params.request.name]);
    await params.trx.query(`
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}" 
                (id, "createdAt", "updatedAt", name, "usernamePolicy", 
                    "emergencyState", "hasEmergencyManagement", "credentialTypes", "publicKey", "privateKey", "settings", "availableCredentialTypes")
                VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
            `, [
        organizationId,
        opTime,
        params.request.name,
        params.request.usernamePolicy,
        dal_constants_1.DalConstants.EmergencyState.Normal,
        params.request.hasEmergencyManagement,
        JSON.stringify(credentialTypes),
        publicKey,
        privateKey,
        JSON.stringify(params.request.settings),
        JSON.stringify(params.request.credentialTypes),
    ]);
    await params.trx.query(`
                INSERT INTO public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationAuthenticationMethods}" 
                (id, "createdAt", "updatedAt", "organizationId", "authenticationMethod", "grantType", "isDefault", "isHidden")
                VALUES ($1, $2, $2, $3, $4, $5, $6, $7)
            `, [uuid_1.default.v4(), opTime, organizationId, dal_constants_1.DalConstants.AuthenticationMethod.Native, dal_constants_1.DalConstants.AuthGrantType.UsernamePassword, true, false]);
    await generatePredefinedOrganizationUnitTypes(organizationId, params.trx);
    await generatePredefinedRolesForOrganization(organizationId, opTime, systemAdminRoleId, params.trx);
    await generatePredefinedOauthClientsForOrganization(organizationId, opTime, params.trx);
    await params.trx.query(`COMMIT`);
    await params.trx.query(`BEGIN`);
    await generatePredefinedSystemUserInfoForOrganization(organizationId, opTime, systemAdminRoleId, params.trx);
    const hash = (0, dal_utils_1.saltHashPassword)(params.request.administratorAccount.password);
    const userId = uuid_1.default.v4();
    await params.trx.query(`
                INSERT INTO "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}" 
                (id, "username")
                VALUES ($1, $2)
            `, [userId, params.request.administratorAccount.username]);
    await params.trx.query(`
                INSERT INTO  "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" 
                (id, "createdAt", "updatedAt", "accountEnabled", "username", "hashedPassword", "salt", "settings")
                VALUES ($1, $2, $2, $3, $4, $5, $6, $7)
            `, [
        userId,
        opTime,
        true,
        params.request.administratorAccount.username,
        hash.passwordHash,
        hash.salt,
        JSON.stringify({
            locale: params.request.settings.locale,
        }),
    ]);
    await params.trx.query(`
                INSERT INTO  "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" 
                (id, "createdAt", "updatedAt", "name", "surname", "userId")
                VALUES ($1, $2, $2, $3, $4, $5)
            `, [userId, opTime, params.request.administratorAccount.name, params.request.administratorAccount.surname, userId]);
    const userOrganizationId = uuid_1.default.v4();
    await params.trx.query(`
                INSERT INTO  "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping}" 
                (id, "userId", "organizationId")
                VALUES ($1, $2, $3)
            `, [userOrganizationId, userId, organizationId]);
    await params.trx.query(`
                INSERT INTO  "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" 
                (id, "createdAt", "updatedAt", "userId", "organizationId", "roleId", "isDisabled", "extraPermissions", "settings")
                VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8)
            `, [
        userOrganizationId,
        opTime,
        userId,
        organizationId,
        systemAdminRoleId,
        false,
        "",
        JSON.stringify({
            notification: {
                mediumSettings: params.request.settings.notification.mediumSettings,
            },
        }),
    ]);
    let userOrganizationProfileId = uuid_1.default.v4();
    await params.trx.query(`
                INSERT INTO  "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" 
                (id, "createdAt", "updatedAt", "userId", "userOrganizationId", "name", "surname", "uniqueId")
                VALUES ($1, $2, $2, $3, $4, $5, $6, $7)
            `, [
        userOrganizationProfileId,
        opTime,
        userId,
        userOrganizationId,
        params.request.administratorAccount.name,
        params.request.administratorAccount.surname,
        params.request.administratorAccount.uniqueId,
    ]);
    await params.trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
					(id, "createdT", type, "receiverFilterId", email, sms, "pushNotification", web, "settings")
					VALUES (uuid_generate_v4(), now(), UNNEST ($1::int[]), null, true, false, true, true, null)`, [dal_constants_1.DalConstants.organizationBaseModuleNotificationTypesWithSingleInstances]);
    return organizationId;
}
exports.createOrganization = createOrganization;
async function generatePredefinedOauthClientsForOrganization(organizationId, opTime, trx) {
    await trx.query(`
		INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthClients}" 
		(id, "createdAt", "updatedAt", "name", profile, "clientSecret", "redirectUris")
		VALUES ($1, $2, $2, $3, $4, $5, $6);
	`, [
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonCPServer.id,
        opTime,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonCPServer.name,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonCPServer.profile,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonCPServer.clientSecret,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonCPServer.redirectUris,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthClients}" 
		(id, "createdAt", "updatedAt", "name", profile, "clientSecret", "redirectUris")
		VALUES ($1, $2, $2, $3, $4, $5, $6);
	`, [
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileAndroidClient.id,
        opTime,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileAndroidClient.name,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileAndroidClient.profile,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileAndroidClient.clientSecret,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileAndroidClient.redirectUris,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthClients}" 
		(id, "createdAt", "updatedAt", "name", profile, "clientSecret", "redirectUris")
		VALUES ($1, $2, $2, $3, $4, $5, $6);
	`, [
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileIOSClient.id,
        opTime,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileIOSClient.name,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileIOSClient.profile,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileIOSClient.clientSecret,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileIOSClient.redirectUris,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthClients}" 
		(id, "createdAt", "updatedAt", "name", profile, "clientSecret", "redirectUris")
		VALUES ($1, $2, $2, $3, $4, $5, $6);
	`, [
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonWebClient.id,
        opTime,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonWebClient.name,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonWebClient.profile,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonWebClient.clientSecret,
        predefined_oAuthClients_1.PredefinedOAuthClients.ArmonWebClient.redirectUris,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthClients}" 
		(id, "createdAt", "updatedAt", "name", profile, "clientSecret", "redirectUris")
		VALUES ($1, $2, $2, $3, $4, $5, $6);
	`, [
        predefined_oAuthClients_1.PredefinedOAuthClients.SwaggerDocs.id,
        opTime,
        predefined_oAuthClients_1.PredefinedOAuthClients.SwaggerDocs.name,
        predefined_oAuthClients_1.PredefinedOAuthClients.SwaggerDocs.profile,
        predefined_oAuthClients_1.PredefinedOAuthClients.SwaggerDocs.clientSecret,
        predefined_oAuthClients_1.PredefinedOAuthClients.SwaggerDocs.redirectUris,
    ]);
    return Promise.resolve();
}
async function generatePredefinedSystemUserInfoForOrganization(organizationId, opTime, systemAdminRoleId, trx) {
    const systemUserOrganizationMappingId = uuid_1.default.v4();
    await trx.query(`INSERT INTO "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
			VALUES ($1, $2) ON CONFLICT (id) DO NOTHING`, [dal_constants_1.DalConstants.SystemUserId, dal_constants_1.DalConstants.SystemUserName]);
    await trx.query(`INSERT INTO "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping}"
			VALUES ($1, $2, $3)`, [systemUserOrganizationMappingId, dal_constants_1.DalConstants.SystemUserId, organizationId]);
    await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
			(id, "createdAt", "updatedAt", "accountEnabled", username, "fromLDAP", settings)
			VALUES ($1, $2, $2, $3, $4, $5, $6) ON CONFLICT (id) DO NOTHING`, [dal_constants_1.DalConstants.SystemUserId, opTime, false, dal_constants_1.DalConstants.SystemUserName, false, { locale: "en" }]);
    await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
			(id, "createdAt", "updatedAt", "userId", "organizationId", "roleId", "isDisabled", "extraPermissions", "settings")
			VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8)`, [
        systemUserOrganizationMappingId,
        opTime,
        dal_constants_1.DalConstants.SystemUserId,
        organizationId,
        systemAdminRoleId,
        true,
        null,
        JSON.stringify({
            notification: {
                mediumSettings: {
                    general: {
                        email: false,
                        pushNotification: false,
                        sms: false,
                        web: true,
                    },
                },
            },
        }),
    ]);
    await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
			(id, "createdAt", "updatedAt", "userId", "userOrganizationId", "name", "surname", "uniqueId")
			VALUES ($1, $2, $2, $3, $4, $5, $6, $7)`, [uuid_1.default.v4(), opTime, dal_constants_1.DalConstants.SystemUserId, systemUserOrganizationMappingId, "Sistem", "", "Armon Sistem Kullanıcısı"]);
    return Promise.resolve();
}
async function generatePredefinedOrganizationUnitTypes(organizationId, trx) {
    const companyId = uuid_1.default.v4();
    const officeId = uuid_1.default.v4();
    const teamId = uuid_1.default.v4();
    const campusId = uuid_1.default.v4();
    const buildingId = uuid_1.default.v4();
    const regionId = uuid_1.default.v4();
    const departmentId = uuid_1.default.v4();
    const organizationUserDefinedListOrganizationUnitTypeId = uuid_1.default.v4();
    const buildingData = {
        caption_tr: "Bina",
        caption_en: "Building",
        color: "#009688",
    };
    const campusData = {
        caption_tr: "Kampüs",
        caption_en: "Campus",
        color: "#03A9f4",
    };
    const companyData = {
        caption_tr: "Firma",
        caption_en: "Company",
        color: "#AD1457",
    };
    const departmentData = {
        caption_tr: "Birim",
        caption_en: "Department",
        color: "#FF5722",
    };
    const officeData = {
        caption_tr: "Ofis",
        caption_en: "Office",
        color: "#FFA000",
    };
    const regionData = {
        caption_tr: "Bölge",
        caption_en: "Region",
        color: "#607D8B",
    };
    const teamData = {
        caption_tr: "Takım",
        caption_en: "Team",
        color: "#BA68C8",
    };
    await trx.query(`	INSERT INTO "${organizationId}"."organizationUserDefinedLists"
	(id, "organizationId", type, caption, "valueType")
		VALUES ($1, $2, $3, $4, $5);`, [
        organizationUserDefinedListOrganizationUnitTypeId,
        organizationId,
        app_enums_1.enums.OrganizationUserDefinedListType.OrganizationUnitType,
        "Birim Türü",
        dal_constants_1.DalConstants.OrganizationUserDefinedListDataType.JSON,
    ]);
    await trx.query(`				INSERT INTO "${organizationId}"."organizationUserDefinedListValues"
			(id, "userDefinedListId", value)
			VALUES ($1, $2, $3), ($4, $5, $6), ($7, $8, $9), ($10, $11, $12), ($13, $14, $15), ($16, $17, $18), ($19, $20, $21)`, [
        buildingId,
        organizationUserDefinedListOrganizationUnitTypeId,
        buildingData,
        campusId,
        organizationUserDefinedListOrganizationUnitTypeId,
        campusData,
        companyId,
        organizationUserDefinedListOrganizationUnitTypeId,
        companyData,
        departmentId,
        organizationUserDefinedListOrganizationUnitTypeId,
        departmentData,
        officeId,
        organizationUserDefinedListOrganizationUnitTypeId,
        officeData,
        regionId,
        organizationUserDefinedListOrganizationUnitTypeId,
        regionData,
        teamId,
        organizationUserDefinedListOrganizationUnitTypeId,
        teamData,
    ]);
}
async function generatePredefinedRolesForOrganization(organizationId, opTime, systemAdminRoleId, trx) {
    await trx.query(`
		INSERT INTO "${organizationId}"."roles" 
		(id, "createdAt", "updatedAt", "isOrganizationWide", name, permissions, "organizationId", "typeId","minPermissions")
		VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8);
	`, [
        systemAdminRoleId,
        opTime,
        predefined_roles_1.PredefinedRoles.SystemAdministrator.isOrganizationWide,
        predefined_roles_1.PredefinedRoles.SystemAdministrator.name,
        predefined_roles_1.PredefinedRoles.SystemAdministrator.permissions,
        organizationId,
        predefined_roles_1.PredefinedRoles.SystemAdministrator.id,
        predefined_roles_1.PredefinedRoles.SystemAdministrator.permissions,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."roles" 
		(id, "createdAt", "updatedAt", "isOrganizationWide", name, permissions, "organizationId", "typeId","minPermissions")
		VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8);
	`, [
        uuid_1.default.v4(),
        opTime,
        predefined_roles_1.PredefinedRoles.UnitAdministrator.isOrganizationWide,
        predefined_roles_1.PredefinedRoles.UnitAdministrator.name,
        predefined_roles_1.PredefinedRoles.UnitAdministrator.permissions,
        organizationId,
        predefined_roles_1.PredefinedRoles.UnitAdministrator.id,
        predefined_roles_1.PredefinedRoles.UnitAdministrator.permissions,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."roles" 
		(id, "createdAt", "updatedAt", "isOrganizationWide", name, permissions, "organizationId", "typeId","minPermissions")
		VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8);
	`, [
        uuid_1.default.v4(),
        opTime,
        predefined_roles_1.PredefinedRoles.StandartUserOrganization.isOrganizationWide,
        predefined_roles_1.PredefinedRoles.StandartUserOrganization.name,
        predefined_roles_1.PredefinedRoles.StandartUserOrganization.permissions,
        organizationId,
        predefined_roles_1.PredefinedRoles.StandartUserOrganization.id,
        predefined_roles_1.PredefinedRoles.StandartUserOrganization.permissions,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."roles" 
		(id, "createdAt", "updatedAt", "isOrganizationWide", name, permissions, "organizationId", "typeId","minPermissions")
		VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8);
	`, [
        uuid_1.default.v4(),
        opTime,
        predefined_roles_1.PredefinedRoles.StandartUser.isOrganizationWide,
        predefined_roles_1.PredefinedRoles.StandartUser.name,
        predefined_roles_1.PredefinedRoles.StandartUser.permissions,
        organizationId,
        predefined_roles_1.PredefinedRoles.StandartUser.id,
        predefined_roles_1.PredefinedRoles.StandartUser.permissions,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."roles" 
		(id, "createdAt", "updatedAt", "isOrganizationWide", name, permissions, "organizationId", "typeId","minPermissions")
		VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8);
	`, [
        uuid_1.default.v4(),
        opTime,
        predefined_roles_1.PredefinedRoles.OrganizationVisitor.isOrganizationWide,
        predefined_roles_1.PredefinedRoles.OrganizationVisitor.name,
        predefined_roles_1.PredefinedRoles.OrganizationVisitor.permissions,
        organizationId,
        predefined_roles_1.PredefinedRoles.OrganizationVisitor.id,
        predefined_roles_1.PredefinedRoles.OrganizationVisitor.permissions,
    ]);
    await trx.query(`
		INSERT INTO "${organizationId}"."roles" 
		(id, "createdAt", "updatedAt", "isOrganizationWide", name, permissions, "organizationId", "typeId","minPermissions")
		VALUES ($1, $2, $2, $3, $4, $5, $6, $7, $8);
	`, [
        uuid_1.default.v4(),
        opTime,
        predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.isOrganizationWide,
        predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.name,
        predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.permissions,
        organizationId,
        predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.id,
        predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.permissions,
    ]);
    return Promise.resolve();
}
async function updateUsername(params) {
    try {
        const dbResultUser = await params.trx.query(`
			SELECT uom."userId", uom."organizationId" FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}" AS ul
			INNER JOIN public."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping}" AS uom
				ON ul.id = uom."userId"
			WHERE ul."username" = $1
		`, [params.oldUsername]);
        if (!dbResultUser.rowCount) {
            app_logs_1.logger.warn("Specified username cannot be found in database!");
        }
        else {
            if (params.organizationId) {
                const uoMapping = dbResultUser.rows.find((uom) => uom.organizationId === params.organizationId);
                await params.trx.query(`
					UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
					SET username = $1
					WHERE id = $2
				`, [params.newUsername, uoMapping.userId]);
                await params.trx.query(`
					UPDATE "${uoMapping.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
					SET username = $1
					WHERE id = $2
				`, [params.newUsername, uoMapping.userId]);
                app_logs_1.logger.info("Username successfully updated!!");
            }
            else {
                for (const uoMapping of dbResultUser.rows) {
                    await params.trx.query(`
						UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
						SET username = $1
						WHERE id = $2
					`, [params.newUsername, uoMapping.userId]);
                    await params.trx.query(`
						UPDATE "${uoMapping.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
						SET username = $1
						WHERE id = $2
					`, [params.newUsername, uoMapping.userId]);
                    app_logs_1.logger.info(`Username successfully updated for organization ${uoMapping.organizationId}!!`);
                }
            }
        }
    }
    catch (error) {
        app_logs_1.logger.error(error);
        throw error;
    }
}
exports.updateUsername = updateUsername;
async function getOrganizationIdNamePairs(trx) {
    try {
        const dbResult = await trx.query(`
			SELECT id, name FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"
		`);
        if (dbResult.rowCount) {
            return Promise.resolve(dbResult.rows);
        }
        else {
            return Promise.resolve([]);
        }
    }
    catch (error) {
        app_logs_1.logger.error("Error in getOrganizationIdNamePairs function");
        app_logs_1.logger.error(error);
        return null;
    }
}
exports.getOrganizationIdNamePairs = getOrganizationIdNamePairs;
async function addAccessLogFromDevice(params) {
    const rLog = dal_db_armon_schema_1.ArmonSchema.Models.generatePSQLNoSqlAccessLogForDevice(params.organizationId, params.log);
    if (rLog.o === dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId && params.log.c) {
        const user = await getUserInfoByCredentialData({ organizationId: params.organizationId, credentialData: params.log.c, trx: params.trx });
        if (user) {
            rLog.on = user.fullName;
            rLog.o = user.userId;
            rLog.ci = user.credentialId;
            rLog.ul = user.roleId;
            params.log.on = user.fullName;
            params.log.o = user.userId;
            params.log.ci = user.credentialId;
            params.log.ul = user.roleId;
        }
    }
    let credentialData = null;
    if (rLog.c) {
        credentialData = [rLog.c.toUpperCase()];
    }
    else if (rLog.cx && rLog.cx.length > 0) {
        credentialData = rLog.cx.filter((c) => c.d).map((c) => c.d.toUpperCase());
    }
    const { rowCount } = await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
		(id, "actionUtc", "organizationId", log, "credentialData")
		VALUES ($1, $2, $3, $4, $5) 
			ON CONFLICT ON CONSTRAINT access_logs_pkey DO NOTHING`, [rLog.id, params.log.u, params.organizationId, rLog, JSON.stringify(credentialData)]);
    if (rowCount) {
        await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.LogHistory}"
			(id, "insertionUtc", "actionUtc", type, data)
			VALUES ($1, $2, $3, $4, $5)`, [params.log.id, new Date().toISOString(), params.log.u, dal_constants_1.DalConstants.LogHistoryType.AccessLog, null]);
    }
    else {
        app_logs_1.logger.debug("Duplicate log, discarding...");
        return null;
    }
    return rLog;
}
exports.addAccessLogFromDevice = addAccessLogFromDevice;
async function getUserInfoByCredentialData(params) {
    const { rows, rowCount } = await params.trx.query(`SELECT "credentialId", "userId", "organizationId", data, "fullName", "roleId", "profileId" 
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_CredentialUserProfiles}"
			WHERE data = $1`, [params.credentialData]);
    return rowCount > 0 ? rows[0] : null;
}
exports.getUserInfoByCredentialData = getUserInfoByCredentialData;
async function upsertRegionStates(organizationId, params, trx) {
    const regionEntryParams = params.map((m) => {
        return { regionId: m.log.i, userId: m.log.o };
    });
    const antiPassbackStatesResult = await trx.query(`
		SELECT aps."regionId", aps."userId", aps."actionUtc"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.antiPassbackStates}" aps
			INNER JOIN UNNEST ($1::JSONB[]) sq
				ON  (sq->>'regionId')::UUID = aps."regionId" AND (sq->>'userId')::UUID = aps."userId"
			ORDER BY aps."regionId" asc , aps."userId" asc
			FOR UPDATE
`, [regionEntryParams]);
    const regionExistanceCheckResult = (await trx.query(`
			SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
			WHERE id  = ANY($1::UUID[])
		  `, [params.map((m) => m.log.i)])).rows.map((m) => m.id);
    const filteredParams = params.filter((f) => regionExistanceCheckResult.includes(f.log.i));
    const antiPassbackStatesToUpdate = filteredParams.filter((f) => antiPassbackStatesResult.rows.find((p) => f.log.i === p.regionId && f.log.o === p.userId && new Date(f.log.t) > p.actionUtc));
    let reducedAntiPassbackStatesToUpdate = Object.values(antiPassbackStatesToUpdate.reduce((a, e) => {
        if (!a[e.log.i + e.log.o]) {
            a[e.log.i + e.log.o] = { ...e };
        }
        if (Date.parse(e.log.t) > Date.parse(a[e.log.i + e.log.o].log.t)) {
            a[e.log.i + e.log.o] = { ...e };
        }
        return a;
    }, {}));
    if (reducedAntiPassbackStatesToUpdate.length) {
        reducedAntiPassbackStatesToUpdate = reducedAntiPassbackStatesToUpdate
            .sort((a, b) => {
            return a.log.i > b.log.i ? 1 : a.log.i < b.log.i ? -1 : a.log.o > b.log.o ? 1 : a.log.o < b.log.o ? -1 : 0;
        });
    }
    const updateParams = reducedAntiPassbackStatesToUpdate.map((entry) => ({
        regionId: entry.log.i,
        userId: entry.log.o,
        actionUtc: entry.log.t,
        state: entry.log.s,
        entranceLockExpirationUtc: entry.log.ee ?? null,
        exitLockExpirationUtc: entry.log.xe ?? null,
        accessControlPointId: entry.accessControlPointId,
    }));
    await trx.query(`
		WITH updates AS (
		  SELECT
			(data->>'regionId')::UUID AS "regionId",
			(data->>'userId')::UUID AS "userId",
			(data->>'actionUtc')::timestamp with time zone AS "actionUtc",
			(data->>'state')::INTEGER AS "state",
			(data->>'entranceLockExpirationUtc')::TIMESTAMPTZ AS "entranceLockExpirationUtc",
			(data->>'exitLockExpirationUtc')::TIMESTAMPTZ AS "exitLockExpirationUtc",
			(data->>'accessControlPointId')::UUID AS "accessControlPointId"
		  FROM UNNEST($1::JSONB[]) AS data
		)
		UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.antiPassbackStates}" aps
		SET
		  "actionUtc" = updates."actionUtc",
		  state = updates."state",
		  "entranceLockExpirationUtc" = updates."entranceLockExpirationUtc",
		  "exitLockExpirationUtc" = updates."exitLockExpirationUtc",
		  "accessControlPointId" = updates."accessControlPointId"
		FROM updates
		WHERE aps."regionId" = updates."regionId"
		  AND aps."userId" = updates."userId"
		  -- To avoid db trigger on update
		  AND (
			aps."actionUtc" IS DISTINCT FROM updates."actionUtc" OR
			aps.state IS DISTINCT FROM updates.state OR
			aps."entranceLockExpirationUtc" IS DISTINCT FROM updates."entranceLockExpirationUtc" OR
			aps."exitLockExpirationUtc" IS DISTINCT FROM updates."exitLockExpirationUtc" OR
			aps."accessControlPointId" IS DISTINCT FROM updates."accessControlPointId"
		  )
		`, [updateParams]);
    const antiPassbackStatesToInsert = filteredParams.filter((f) => !antiPassbackStatesResult.rows.find((p) => f.log.i === p.regionId && f.log.o === p.userId));
    let reducedAntiPassbackStatesToInsert = Object.values(antiPassbackStatesToInsert.reduce((a, e) => {
        if (!a[e.log.i + e.log.o]) {
            a[e.log.i + e.log.o] = { ...e };
        }
        if (Date.parse(e.log.t) > Date.parse(a[e.log.i + e.log.o].log.t)) {
            a[e.log.i + e.log.o] = { ...e };
        }
        return a;
    }, {}));
    const insertParams = reducedAntiPassbackStatesToInsert.map((entry) => ({
        regionId: entry.log.i,
        userId: entry.log.o,
        actionUtc: entry.log.t,
        state: entry.log.s,
        entranceLockExpirationUtc: entry.log.ee ?? null,
        exitLockExpirationUtc: entry.log.xe ?? null,
        accessControlPointId: entry.accessControlPointId,
    }));
    const query = `
		INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.antiPassbackStates}"
		(id, "regionId", "userId", "actionUtc", state, "entranceLockExpirationUtc", "exitLockExpirationUtc", "accessControlPointId")
		SELECT
			gen_random_uuid(),
			(data->>'regionId')::UUID,
			(data->>'userId')::UUID,
			(data->>'actionUtc')::TIMESTAMPTZ,
			(data->>'state')::INTEGER,
			(data->>'entranceLockExpirationUtc')::TIMESTAMPTZ,
			(data->>'exitLockExpirationUtc')::TIMESTAMPTZ,
			(data->>'accessControlPointId')::UUID
		FROM UNNEST($1::JSONB[]) AS data;
		`;
    await trx.query(query, [insertParams]);
}
exports.upsertRegionStates = upsertRegionStates;
async function upsertAccessRuleHistory(organizationId, params, trx) {
    const { accessControlPointId, accessRuleId, actionDateISO, count, date, userId, endDateTime, startDateTime } = params;
    const { rows, rowCount } = await trx.query(`
			SELECT id, count FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleHistory}"
			WHERE "createdDate" = $1 AND "accessRuleId" = $2 AND "userId" = $3
		`, [date, accessRuleId, userId]);
    const options = {
        startDateTime: startDateTime || luxon_1.DateTime.fromISO(actionDateISO).startOf("day").toSQL(),
        endDateTime: endDateTime || luxon_1.DateTime.fromISO(actionDateISO).endOf("day").toSQL(),
        dayOfWeek: params.weekDay || luxon_1.DateTime.fromISO(actionDateISO).weekday,
    };
    if (rowCount > 0 && rows[0].count <= params.count) {
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleHistory}"
				SET "count" = $1, "lastUpdateAccessControlPointId" = $2, "lastActionDateISO" = $3, options = $4
				WHERE id = $5`, [count, accessControlPointId, actionDateISO, options, rows[0].id]);
    }
    else if (rowCount === 0) {
        await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleHistory}"
				("id", "createdDate", "accessRuleId", "userId", "count", "lastUpdateAccessControlPointId", "lastActionDateISO", options)
				VALUES ($1, $2, $3, $4, $5, $6, $7, $8);`, [uuid_1.default.v4(), date, accessRuleId, userId, count, accessControlPointId, actionDateISO, options]);
    }
}
exports.upsertAccessRuleHistory = upsertAccessRuleHistory;
async function updateUserRegionTicketUnits(organizationId, params, trx) {
    const now = new Date();
    const userRegionTicket = (await trx.query(`SELECT id, "remainingUnits", "totalSpentUnits" 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits}"
			WHERE "userId" = $1
				AND "remainingUnits" != $2`, [params.userId, dal_constants_1.DalConstants.SpecialRegionTicketNumber])).rows[0];
    await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits}"
		SET  "updatedAt" = $1,
			"remainingUnits" = $2,
			"lastUpdateAccessControlPointId" = $3, 
			"totalSpentUnits" = $4
		WHERE id = $5`, [
        now,
        +userRegionTicket.remainingUnits - +params.decrementUnit,
        params.accessControlPointId,
        userRegionTicket.totalSpentUnits ? +userRegionTicket.totalSpentUnits + +params.decrementUnit : +params.decrementUnit,
        userRegionTicket.id,
    ]);
}
exports.updateUserRegionTicketUnits = updateUserRegionTicketUnits;
async function uploadCounterLogs(params) {
    const counterSensorLogsToInsert = [];
    for (const log of params.logs) {
        log.id = uuid_1.default.v4();
        const item = {
            id: log.id,
            actionUtc: log.u,
            organizationId: params.organizationId,
            log: log,
        };
        counterSensorLogsToInsert.push(item);
    }
    const qi = [];
    const qb = [];
    let qx = 1;
    let q = `INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CounterSensorLogs}"
	(id, "actionUtc", "organizationId", log) `;
    for (const log of counterSensorLogsToInsert) {
        qi.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++})`);
        qb.push(log.id, log.actionUtc, log.organizationId, log.log);
    }
    q = q + " VALUES " + qi.join(" , ");
    await params.trx.query(q, qb);
    return counterSensorLogsToInsert.map((s) => s.id);
}
exports.uploadCounterLogs = uploadCounterLogs;
async function updateAccessLogForSensorData(params) {
    const { rows: accessLogRows, rowCount: accessLogRowCount } = await params.trx.query(`
	SELECT id, "actionUtc", log FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
	WHERE id = $1`, [params.logId]);
    if (accessLogRowCount === 0) {
        app_logs_1.logger.warn("Sensor data received but no related AccessLog found, forwarded to deadLetter");
        return Promise.reject();
    }
    const accessLog = accessLogRows[0];
    if (params.statusSensorLogId) {
        if (!accessLog.log.ss) {
            accessLog.log.ss = [params.statusSensorLogId];
        }
        else {
            accessLog.log.ss.push(params.statusSensorLogId);
        }
    }
    if (params.counterSensorLogId) {
        if (!accessLog.log.cs) {
            accessLog.log.cs = [params.counterSensorLogId];
        }
        else {
            accessLog.log.cs.push(params.counterSensorLogId);
        }
    }
    await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
			SET log = $1
			WHERE id = $2`, [accessLog.log, params.logId]);
}
exports.updateAccessLogForSensorData = updateAccessLogForSensorData;
async function insertUnhandledDeviceLog(params) {
    await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.unhandledDeviceLogs}"
	(id, "createdT", "logReceiveMethod", "deviceId", type, log, note)
	VALUES ($1, $2, $3, $4, $5, $6, $7)`, [uuid_1.default.v4(), new Date(), params.logReceiveMethod, params.deviceId, params.type, params.data, params.note]);
}
exports.insertUnhandledDeviceLog = insertUnhandledDeviceLog;
async function deleteDeadLetterOfflineLog(params) {
    await params.trx.query(`
		DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.unhandledDeviceLogs}"
		WHERE id = $1`, [params.logId]);
}
exports.deleteDeadLetterOfflineLog = deleteDeadLetterOfflineLog;
async function onAccessLogsChanged(params) {
    if (params.organizationId && params.userId && params.acpId) {
        await params.trx.query(`
				UPDATE "${params.organizationId}"."employeeLiveDay" eld
				SET "updateRequiredReason" = "updateRequiredReason" | $1, "updateRequiredRaisedAt" = now()
				FROM "${params.organizationId}"."workPlans" wp
				INNER JOIN "${params.organizationId}"."regions" r ON wp."regionId" = r."id"
				INNER JOIN "${params.organizationId}"."regionAccessControlPoints" AS racp ON r."id" = racp."regionId"
				WHERE $2 IN (racp."accessControlPointId")
				AND eld."userId" = $3 AND eld."organizationId" = $4
				AND eld."workPlanIds" @> to_jsonb(wp.id)
				AND (wp."accessCheckType" = 2 OR (wp."accessCheckType" = 1 AND ($5 = 1 OR $6 = 2)))
			`, [params.reason, params.acpId, params.userId, params.organizationId, params.direction || 99, params.direction || 99]);
        if ((0, moment_1.default)(params.timestamp).isBefore((0, moment_1.default)().startOf("day"))) {
            await params.trx.query(`
				INSERT INTO "${params.organizationId}"."recalculateWorkQueue"
				("id", "userId", "organizationId", "reason", "startDate")
				SELECT uuid_generate_v4(), uwp."userId", wp."organizationId", $1, $2
				FROM "${params.organizationId}"."userWorkPlans" uwp
				INNER JOIN "${params.organizationId}"."workPlans" wp ON wp."id" = uwp."workPlanId"
				INNER JOIN "${params.organizationId}"."vW_WorkPlanPeriodTstzrange" wpp ON wpp."workPlanId" = wp."id" AND wpp."organizationId" = $3
				INNER JOIN "${params.organizationId}"."userOrganizations" uo ON uo."userId" = uwp."userId" AND uo."organizationId" = wp."organizationId"
					AND uo."deletedAt" IS NULL AND uo."isDisabled" = false
					AND (tstzrange(uwp."startDateTime", uwp."endDateTime")*wpp."range") @> ($4::timestamp with time zone)
				INNER JOIN "${params.organizationId}"."regions" r ON wp."regionId" = r."id"
				INNER JOIN "${params.organizationId}"."regionAccessControlPoints" AS racp ON r."id" = racp."regionId"
				WHERE $5 IN (racp."accessControlPointId")
				AND (wp."accessCheckType" = 2 OR (wp."accessCheckType" = 1 AND ($6 = 1 OR $7 = 2)))
				AND uwp."userId"=$8 AND wp."organizationId"=$9;
				`, [params.reason, params.timestamp, params.organizationId, params.timestamp, params.acpId, params.direction || 99, params.direction || 99, params.userId, params.organizationId]);
        }
    }
}
exports.onAccessLogsChanged = onAccessLogsChanged;
async function uploadStatusSensorLogsPg(params) {
    const sensorLogsToInsert = [];
    for (const log of params.sensorLogs) {
        log.id = uuid_1.default.v4();
        sensorLogsToInsert.push({
            id: log.id,
            actionUtc: log.u,
            organizationId: params.organizationId,
            log: log,
        });
    }
    const qi = [];
    const qb = [];
    let qx = 1;
    let q = `INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.StatusSensorLogs}"
	(id, "actionUtc", "organizationId", log) `;
    for (const log of sensorLogsToInsert) {
        qi.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++})`);
        qb.push(log.id, log.actionUtc, log.organizationId, log.log);
    }
    q = q + " VALUES " + qi.join(" , ");
    await params.trx.query(q, qb);
    return sensorLogsToInsert.map((s) => s.id);
}
exports.uploadStatusSensorLogsPg = uploadStatusSensorLogsPg;
async function uploadTamperSwitchLogsPg(params) {
    const tamperSwitchLogsToInsert = [];
    for (const log of params.tamperSwitchLogs) {
        log.id = uuid_1.default.v4();
        tamperSwitchLogsToInsert.push({
            id: log.id,
            actionUtc: log.u,
            organizationId: params.organizationId,
            log: log,
        });
    }
    const qi = [];
    const qb = [];
    let qx = 1;
    let q = `INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.TamperSwitchLogs}"
	(id, "actionUtc", "organizationId", log) `;
    for (const log of tamperSwitchLogsToInsert) {
        qi.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++})`);
        qb.push(log.id, log.actionUtc, log.organizationId, log.log);
    }
    q = q + " VALUES " + qi.join(" , ");
    await params.trx.query(q, qb);
}
exports.uploadTamperSwitchLogsPg = uploadTamperSwitchLogsPg;
async function uploadDeviceSystemStatusLogsNew(params) {
    let systemStatusLogsToInsert = [];
    for (const log of params.logs) {
        let item = {
            id: params.deviceId,
            actionUtc: log.u,
            type: dal_constants_1.DalConstants.SystemStatusLogType.TerminalHealth,
            log: JSON.stringify(log),
        };
        if (log.ic !== null && log.ic !== undefined) {
            item.type = log.ic ? dal_constants_1.DalConstants.SystemStatusLogType.TerminaleConnected : dal_constants_1.DalConstants.SystemStatusLogType.TerminalDisconnected;
            item.log = JSON.stringify({});
        }
        systemStatusLogsToInsert.push(item);
    }
    await (0, dal_utils_1.insertUpdateLogToSystemStatusTable)({
        organizationId: params.organizationId,
        type: dal_constants_1.DalConstants.SystemStatusLogType.TerminalHealth,
        data: systemStatusLogsToInsert,
        trx: params.trx,
    });
    return Promise.resolve();
}
exports.uploadDeviceSystemStatusLogsNew = uploadDeviceSystemStatusLogsNew;
async function getDeviceStatePersistentData(params) {
    const { rows, rowCount } = await params.trx.query(`
		SELECT health->'isConnected' as i, health->'warnings' as w, array_agg(ca.id) as "c"
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}" d
		LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.cameras}" ca
		ON ca."proxyTerminalId" = d.id
		WHERE d.id = $1
		AND d."deletedAt" IS NULL
		GROUP BY d.id, d.health->>'isConnected', d.health->>'warnings'`, [params.deviceId]);
    return rowCount > 0 ? rows[0] : null;
}
exports.getDeviceStatePersistentData = getDeviceStatePersistentData;
async function getDeviceNotificationInfoPersistentData(params) {
    const { rows, rowCount } = await params.trx.query(`
		SELECT brand as b, model as m, name as n, location as l FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}"
		WHERE id = $1
		AND "deletedAt" IS NULL`, [params.deviceId]);
    return rowCount > 0 ? rows[0] : null;
}
exports.getDeviceNotificationInfoPersistentData = getDeviceNotificationInfoPersistentData;
async function getAccessPointRedisPersistentData(params) {
    const { rows, rowCount } = await params.trx.query(`
	SELECT name AS n FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
	WHERE id = $1 AND "deletedAt" IS NULL
	`, [params.accessPointId]);
    return rowCount > 0 ? rows[0] : null;
}
exports.getAccessPointRedisPersistentData = getAccessPointRedisPersistentData;
async function getRegionRedisPersistentData(params) {
    const { rows, rowCount } = await params.trx.query(`
	SELECT id, name as n, "visitorAccess" as v, "terminateVisitOnExit" t
	FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
	WHERE id = $1
`, [params.regionId]);
    if (rowCount === 0) {
        return null;
    }
    return rows[0];
}
exports.getRegionRedisPersistentData = getRegionRedisPersistentData;
async function getOrganizationRedisPersistentData(params) {
    const { rows, rowCount } = await params.trx.query(`
		SELECT name as n FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
		WHERE id = $1
		AND "deletedAt" IS NULL
	`, [params.organizationId]);
    return rowCount > 0 ? rows[0] : null;
}
exports.getOrganizationRedisPersistentData = getOrganizationRedisPersistentData;
async function getOrganization(params) {
    const { rows, rowCount } = await params.trx.query(`
		SELECT  id,
				name,
				settings,
				"availableCredentialTypes",
				"credentialTypes",
				"smtpOptions",
				"usernamePolicy",
				"emergencyState",
				"hasEmergencyManagement",
				"uniqueIdFieldCaption",
				"emailAddresses",
				"publicKey",
				"privateKey",
				"scheduledJobParameters",
				"annualPPermissionUpdateIntervals",
				"userOrganizationProfileSettings"
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
		WHERE id = $1
		AND "deletedAt" IS NULL`, [params.organizationId]);
    return rowCount > 0 ? rows[0] : null;
}
exports.getOrganization = getOrganization;
async function getUserOrganizationRedisPersistentData(params) {
    const { rows, rowCount } = await params.trx.query(`SELECT
		  u.settings->>'locale' as locale,
		  uo.settings->'notification' as "notificationSettings",
		  uop.name || ' ' || uop.surname as fullname,
		  uop."uniqueId",
		  uo."roleId",
		  uo."isDisabled",
		  (CASE WHEN uop."email" IS NULL THEN up.email ELSE uop.email END) as email,
		  (CASE WHEN uop."phoneNumber" IS NULL THEN up."phoneNumber" ELSE uop."phoneNumber" END) as "phoneNumber",
		  u."notificationToken",
		  u."notificationTokenType"
		  FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
		  INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo 
			ON uo."userId" = u."id" 
			AND u."deletedAt" IS NULL 
			AND uo."deletedAt" IS NULL
			AND u."id" = $1
		  LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up 
			ON uo."userId" = up."userId" 
			AND up."deletedAt" IS NULL 
			AND uo."deletedAt" IS NULL                
		  LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop 
			ON uop."userOrganizationId" = uo."id" 
			AND uop."deletedAt" IS NULL`, [params.userId]);
    if (rowCount > 0) {
        const row = rows[0];
        return {
            f: row.fullname,
            l: row.locale,
            s: row.notificationSettings,
            u: row.uniqueId,
            d: row.isDisabled,
            r: row.roleId,
            e: row.email,
            m: row.phoneNumber,
            n: row.notificationToken,
            z: row.notificationTokenType,
        };
    }
    return null;
}
exports.getUserOrganizationRedisPersistentData = getUserOrganizationRedisPersistentData;
async function getUserOrganizationCaptionLinesPg(params) {
    const { organizationId, trx, userIds } = params;
    const result = [];
    const userProfileSettings = await getOrganizationUserProfileSettings({ organizationId, trx });
    const now = new Date();
    let qx = 1;
    const qb = [];
    const qfrom = [
        `FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo 
				ON uo."userId" = u.id AND u."deletedAt" IS NULL AND uo."deletedAt" IS NULL`,
    ];
    const columns = ["u.id"];
    const fieldTypes = new Set(userProfileSettings.captionLineFields.map((c) => c.type));
    for (const field of fieldTypes) {
        switch (field) {
            case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                {
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop2
						ON uop2."userOrganizationId" = uo.id AND uop2."deletedAt" IS NULL`);
                    columns.push("uop2.thumbnail");
                }
                break;
            case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                {
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop 
					ON uop."userOrganizationId" = uo.id AND uop."deletedAt" IS NULL`);
                    columns.push(`
						json_build_object(
							'userId', uop."userId",
							'uniqueId', uop."uniqueId",
							'name', uop."name",
							'surname', uop."surname",
							'fullname',uop."name" || ' ' || uop."surname",
							'email',uop."email",
							'extensionFields',uop."extensionFields"
						) as "profile"`);
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" ovp 
					ON ovp."userOrganizationId" = uo.id AND ovp."deletedAt" IS NULL`);
                    columns.push(`
						json_build_object(
							'userId', uo."userId",
							'uniqueId', ovp."uniqueId",
							'name', ovp."name",
							'surname', ovp."surname",
							'fullname', COALESCE(ovp."name" || ' ', '')  || ' ' || COALESCE(ovp."surname" || ' ', ''),
							'email', '',
							'extensionFields', ovp."extensionFields"
						) as "visitorProfile"
						`);
                }
                break;
            case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                {
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo  
							ON uguo."userOrganizationId" = uo.id  AND uguo."deletedAt" IS NULL
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug 
							ON ug.id = uguo."userGroupId" IS NULL AND ug."deletedAt" IS NULL`);
                    columns.push(`
						json_build_object(
							'id', ug."id",
							'name', ug."name",
							'colorCode', ug."colorCode"
						) as "userGroup"
						`);
                }
                break;
            case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                {
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou 
						ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL
					LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou 
						ON ou.id = uoou."organizationUnitId" AND ou."deletedAt" IS NULL `);
                    columns.push(`
						json_build_object(
							'id', org."id",
							'name', org."name"
						) as "organizationUnit"
					`);
                }
                break;
            case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                {
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc 
						ON uoc."userId" = uo."userId"
						AND (uoc."expiresOn" >= $${qx++} OR uoc."expiresOn" IS NULL)
						AND uoc."deletedAt" IS NULL
					`);
                    qb.push(now);
                    columns.push(`
						json_build_object(
							'id', uoc."id",
							'type', uoc."type",
							'data', uoc."data",
							'extensionFields',uoc."extensionFields"
						) as "credential"
					`);
                }
                break;
            case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                {
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}" uof 
						ON uof."userId" = uo."userId" 
						AND (uof."startUtc" <= $${qx}  OR uof."startUtc" IS NULL) 
						AND (uof."endUtc" >= $${qx++} OR uof."endUtc" IS NULL)
						AND uof."deletedAt" IS NULL
				`);
                    qb.push(now);
                    columns.push(`
						json_build_object(
							'id', uof."id"
						) as "forbiddance"
						`);
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" ovp2 
						ON ovp2."userOrganizationId" = uo.id AND ovp2."deletedAt" IS NULL
					`);
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" ovsf 
						ON ovp2."id" = uo.id AND ovsf."organizationVisitorProfileId" IS NULL
						AND (ovsf."startUtc" <= $${qx}  OR ovsf."startUtc" IS NULL) AND (ovsf."endUtc" >= $${qx++} OR ovsf."endUtc" IS NULL)
						AND  ovsf."deletedAt" IS NULL
						AND ovsf.state = $${qx++}
					`);
                    qb.push(dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit);
                    columns.push(`
						json_build_object(
							'id', ovsf."id"
						) as "visitorForbiddance"
					`);
                    qfrom.push(` LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" ovsw
					ON ovp2.id = ovsw."organizationVisitorProfileId" 
					AND (ovsw."startUtc" <= $${qx}  or ovsw."startUtc" IS NULL) ABD (ovsw."endUtc" >= $${qx++} OR ovsw."endUtc" IS NULL)
					AND ovsw."deletedAt" IS NULL
					AND ovsw.state = $${qx++}
				`);
                    qb.push(now, dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit);
                    columns.push(`
					json_build_object(
						'id', ovsw."id"
						) as "visitorPermit"
					`);
                }
                break;
            default:
                break;
        }
    }
    const q = `SELECT ${columns.join(" , ")} ${qfrom.join(" ")} WHERE uo."userId" = ANY($${qx++}::uuid[]) `;
    qb.push(params.userIds);
    const { rows: captionResults } = await params.trx.query(q, qb);
    for (const userId of userIds) {
        const captionResult = captionResults.filter((r) => r.id == userId);
        if (!captionResult || captionResult.length < 1) {
            result.push({ id: userId, captionLines: [] });
            continue;
        }
        let captions = [];
        for (let captionItem = 0; captionItem < userProfileSettings.captionLineFields.length; captionItem++) {
            const captionField = userProfileSettings.captionLineFields[captionItem];
            const texts = [];
            const colors = [];
            let profilePhoto = null;
            let translate = false;
            switch (captionField.type) {
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                    {
                        let thumbnails = captionResult.filter((c) => c.thumbnail).map((c) => c.thumbnail);
                        if (thumbnails.length > 0) {
                            profilePhoto = Buffer.from(thumbnails[0]).toString();
                        }
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                    const profiles = lodash_1.default.uniqBy(captionResult.map((c) => c.profile).concat(captionResult.map((c) => c.visitorProfile)), (x) => x.userId);
                    for (const profile of profiles) {
                        if (captionField.isExtension) {
                            if (profile.extensionFields) {
                                let field = profile.extensionFields[captionField.name];
                                if (field && lodash_1.default.isArray(field)) {
                                    texts.push(field.join("-"));
                                }
                                else {
                                    texts.push(field);
                                }
                            }
                        }
                        else {
                            profile[captionField.name] ? texts.push(profile[captionField.name]) : "";
                        }
                        if (captionField.colorCode) {
                            colors.push(captionField.colorCode);
                        }
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                    const userGroups = lodash_1.default.uniqBy(captionResult.map((c) => c.userGroup), function (x) {
                        return x.id;
                    });
                    for (const userGroup of userGroups) {
                        texts.push(userGroup[captionField.name]);
                        if (captionField.colorCode) {
                            colors.push(captionField.colorCode);
                        }
                        else {
                            colors.push(userGroup.colorCode);
                        }
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                    const organizationUnits = lodash_1.default.uniqBy(captionResult.map((c) => c.organizationUnit), function (x) {
                        return x.id;
                    });
                    for (const organizationUnit of organizationUnits) {
                        texts.push(organizationUnit[captionField.name]);
                        if (captionField.colorCode) {
                            colors.push(captionField.colorCode);
                        }
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                    const credentials = lodash_1.default.uniqBy(captionResult.map((c) => c.credential), function (x) {
                        return x.id;
                    });
                    for (const credential of credentials) {
                        if (captionField.filterTypes && captionField.filterTypes.length > 0) {
                            if (!captionField.filterTypes.some((a) => parseInt(a) == parseInt(credential["type"])))
                                continue;
                        }
                        if (captionField.isExtension) {
                            let field = credential.extensionFields ? credential.extensionFields[captionField.name] : null;
                            if (field)
                                texts.push(field.value);
                        }
                        else {
                            texts.push(credential[captionField.name]);
                        }
                        if (captionField.colorCode) {
                            colors.push(captionField.colorCode);
                        }
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                    {
                        const userForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.forbiddance).filter((c) => c.id), (x) => {
                            return x.id;
                        });
                        const userVisitorForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorForbiddance).filter((c) => c.id), (x) => {
                            return x.id;
                        });
                        const userVisitorPermit = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorPermit).filter((c) => c.id), (x) => {
                            return x.id;
                        });
                        if (userForbiddance && userForbiddance.length > 0) {
                            texts.push(dal_constants_1.DalConstants.UserCaptionKeys.USER_FORBIDDEN);
                            colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                            translate = true;
                        }
                        if (userVisitorForbiddance && userVisitorForbiddance.length > 0) {
                            texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_FORBIDDEN);
                            colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                            translate = true;
                        }
                        if (userVisitorPermit && userVisitorPermit.length > 0) {
                            texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_PERMITTED);
                            colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                            translate = true;
                        }
                    }
                    break;
                default:
                    break;
            }
            if (captionField.type != dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto) {
                if (texts.length == 0) {
                    captions.push(null);
                }
                else {
                    const filteredTexts = lodash_1.default.compact(texts);
                    if (filteredTexts.length == 0) {
                        captions.push(null);
                    }
                    else {
                        if (translate) {
                            captions.push({
                                text: texts,
                                colorCode: colors,
                                translate: true,
                            });
                        }
                        else {
                            captions.push({
                                text: texts,
                                colorCode: colors,
                            });
                        }
                    }
                }
            }
            else {
                captions.push({
                    avatar: profilePhoto ? profilePhoto : dal_constants_1.DalConstants.DefaultAvatar,
                    text: ["__avatar"],
                });
            }
        }
        for (let captionIndex = 0; captionIndex < captions.length; ++captionIndex) {
            if (!captions[captionIndex]) {
                captions[captionIndex] = {
                    text: [""],
                };
            }
        }
        captions = lodash_1.default.compact(captions);
        result.push({ id: userId, captionLines: captions });
    }
    return result;
}
async function getOrganizationUserProfileSettings(params) {
    const { rows } = await params.trx.query(`
		SELECT "userOrganizationProfileSettings" 
		FROM "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}
		WHERE id = $1 
		AND "deletedAt" IS NULL
	`, [params.organizationId]);
    return rows[0]?.userOrganizationProfileSettings ?? dal_constants_1.DalConstants.defaultUserProfileSettings;
}
async function updateDeviceHealth(params) {
    await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}"
			SET health = $1,
				"updatedAt" = $2
			WHERE id = $3`, [JSON.stringify(params.health), new Date(), params.deviceId]);
}
async function getDeviceHealth(params) {
    const { rows } = await params.trx.query(`
            SELECT health FROM "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}
            WHERE id = $1`, [params.deviceId]);
    return rows[0]?.health ? rows[0].health : null;
}
async function getDeviceStatus(params) {
    const result = {
        connected: 0,
        disconnected: 0,
        warning: {
            critical: 0,
            medium: 0,
            high: 0,
        },
    };
    const { rows: healthList, rowCount: terminalCount } = await params.trx.query(`
				SELECT health FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}" d
				LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.adapters}" ad ON ad.id = d."adapterId"			
				WHERE d."deletedAt" IS NULL
				AND d.brand = ANY($1)
				AND d.model = ANY($2)
				`, [[dal_constants_1.DalConstants.DeviceBrand.Armon, dal_constants_1.DalConstants.DeviceBrand.AssaAbloy, dal_constants_1.DalConstants.DeviceBrand.HikVision], [dal_constants_1.DalConstants.terminalModels]]);
    result.disconnected = terminalCount;
    const devicehealthList = lodash_1.default.compact(healthList.filter((d) => d?.health));
    for (const { health } of devicehealthList) {
        if (health.isConnected) {
            result.connected++;
            result.disconnected--;
        }
        if (health.warnings) {
            if (health.warnings.find((d) => d.level == dal_constants_1.DalConstants.TerminalWarningLevel.Critical)) {
                result.warning.critical++;
            }
            if (health.warnings.find((d) => d.level == dal_constants_1.DalConstants.TerminalWarningLevel.High)) {
                result.warning.high++;
            }
            if (health.warnings.find((d) => d.level == dal_constants_1.DalConstants.TerminalWarningLevel.Medium)) {
                result.warning.medium++;
            }
        }
    }
    return result;
}
async function uploadDeviceSystemStatusLogsPg(params) {
    const { deviceId, redis, deviceLogs, organizationId, trx } = params;
    try {
        if (deviceLogs.length < 1) {
            return null;
        }
        await uploadDeviceSystemStatusLogsNew({ organizationId, deviceId, logs: deviceLogs, trx });
        let deviceHealth = await getDeviceHealth({ organizationId, deviceId, trx });
        if (!deviceHealth) {
            deviceHealth = {
                temperature: null,
                availableStorage: null,
                isConnected: false,
                lastUpdateUtc: null,
                power: null,
                usedStorage: null,
                warnings: [],
                ip: null,
                onBoardComputerFreeStorage: null,
                deviceId: deviceId,
                onBoardComputerModel: null,
                onBoardComputerTotalStorage: 0,
                version: null,
                batteryLevel: 0,
            };
        }
        let sortedLogs = deviceLogs.sort((x, y) => (x.u < y.u ? 1 : -1));
        let lastLog = sortedLogs[0];
        if (!deviceHealth.lastUpdateUtc || (deviceHealth.lastUpdateUtc && (0, moment_1.default)(lastLog.u).isAfter(deviceHealth.lastUpdateUtc))) {
            let warnings = (0, business_device_1.generateWarnings)(sortedLogs[0], deviceHealth.availableStorage);
            if (!deviceHealth.version) {
                deviceHealth.version = {};
            }
            deviceHealth = {
                temperature: lastLog.t || deviceHealth.temperature,
                availableStorage: lastLog.dt || deviceHealth.availableStorage,
                lastUpdateUtc: lastLog.u ? new Date(lastLog.u) : deviceHealth.lastUpdateUtc,
                power: lastLog.ib ? app_enums_1.enums.DevicePowerState.OnBattery : app_enums_1.enums.DevicePowerState.OnPower,
                usedStorage: lastLog.dt && lastLog.df ? lastLog.dt - lastLog.df : deviceHealth.usedStorage,
                warnings: warnings,
                mac: lastLog.mac ?? deviceHealth.mac,
                ip: lastLog.ip || deviceHealth.ip,
                onBoardComputerFreeStorage: lastLog.df || deviceHealth.onBoardComputerFreeStorage,
                onBoardComputerModel: lastLog.cm || deviceHealth.onBoardComputerModel,
                onBoardComputerTotalStorage: lastLog.dt || deviceHealth.onBoardComputerTotalStorage,
                isConnected: lastLog.ic === null || lastLog.ic === undefined ? deviceHealth.isConnected : lastLog.ic,
                batteryLevel: lastLog.b || deviceHealth.batteryLevel,
                emergencyStatus: deviceHealth.emergencyStatus || app_enums_1.enums.EmergencyState.Normal,
                version: {
                    hardwareHwVersion: lastLog.hhv || deviceHealth.version?.hardwareHwVersion,
                    hardwareSwVersion: lastLog.hsv || deviceHealth.version?.hardwareSwVersion,
                    onBoardComputerVersion: lastLog.cv || deviceHealth.version?.onBoardComputerVersion,
                    serviceVersion: lastLog.sv || deviceHealth.version?.serviceVersion,
                    libAccessVersion: lastLog.lav || deviceHealth.version?.libAccessVersion,
                    backendApiVersion: lastLog.bav || deviceHealth.version?.backendApiVersion,
                    operatingSystem: lastLog.os || deviceHealth.version?.operatingSystem,
                },
            };
        }
        await updateDeviceHealth({ organizationId, deviceId, health: deviceHealth, trx });
        if (deviceHealth.warnings.length > 0) {
            let deviceStatus = await getDeviceStatus({ organizationId, trx });
            await redis.setValue(dal_access_cache_1.CacheConstantKeys.DeviceStatus + organizationId, JSON.stringify(deviceStatus));
        }
        deviceHealth.deviceId = deviceId;
        return deviceHealth;
    }
    catch (error) {
        app_logs_1.logger.error("error while inserting health log coming from device", error);
        throw error;
    }
}
exports.uploadDeviceSystemStatusLogsPg = uploadDeviceSystemStatusLogsPg;
async function anonymizeOrganization(data, trx) {
    await trx.query(`UPDATE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"
		SET name = $1, alias = NULL, code = NULL
		WHERE id = $2`, [data.newOrganizationName, data.organizationId]);
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
		SET name = $1, alias = NULL, code = NULL
		WHERE id = $2`, [data.newOrganizationName, data.organizationId]);
    if (data.anonymizeLogins) {
        const loginData = await trx.query(`SELECT ul.id, ul.username, 'armon_' || (row_number() OVER ())::text || '@armongate.com' as newusername, gen_random_uuid() as swapusername
			FROM public."userList" ul
			INNER JOIN public."userOrganizationMapping" uom
				ON ul.id = uom."userId" AND ul.username IS NOT NULL
			WHERE uom."organizationId" = $1`, [data.organizationId]);
        await trx.query(`UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}" ulist
			SET username = newlogins.swapusername
			FROM (
				SELECT (data->>'id')::uuid as id, data->>'username' as username, data->>'newusername' as newusername, data->>'swapusername' as swapusername
				FROM jsonb_array_elements($1::jsonb) as data
			) newlogins
			WHERE ulist.id = newlogins.id AND ulist.username = newlogins.username`, [JSON.stringify(loginData.rows)]);
        await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" users
			SET username = newlogins.swapusername
			FROM (
				SELECT (data->>'id')::uuid as id, data->>'username' as username, data->>'newusername' as newusername, data->>'swapusername' as swapusername
				FROM jsonb_array_elements($1::jsonb) as data
			) newlogins
			WHERE users.id = newlogins.id AND users.username = newlogins.username`, [JSON.stringify(loginData.rows)]);
        await trx.query(`UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}" ulist
			SET username = newlogins.newusername
			FROM (
				SELECT (data->>'id')::uuid as id, data->>'username' as username, data->>'newusername' as newusername, data->>'swapusername' as swapusername
				FROM jsonb_array_elements($1::jsonb) as data
			) newlogins
			WHERE ulist.id = newlogins.id AND ulist.username = newlogins.swapusername`, [JSON.stringify(loginData.rows)]);
        await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" users
			SET username = newlogins.newusername
			FROM (
				SELECT (data->>'id')::uuid as id, data->>'username' as username, data->>'newusername' as newusername, data->>'swapusername' as swapusername
				FROM jsonb_array_elements($1::jsonb) as data
			) newlogins
			WHERE users.id = newlogins.id AND users.username = newlogins.swapusername`, [JSON.stringify(loginData.rows)]);
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
		SET name = mock_data.name, surname = mock_data.surname
		FROM
		(
			SELECT row_number() OVER () as "index", id
			FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
		) users
		LEFT JOIN (
			SELECT row_number() OVER () as "index", data->>'name' as name, data->>'surname' as surname
			FROM jsonb_array_elements($1::jsonb) as data
		) mock_data
		ON
			users."index" = mock_data."index"
		WHERE uop.id = users.id
		RETURNING uop."userId"`, [JSON.stringify(data.users)]);
    for (const login of data.addLogins ?? []) {
        const userId = uuid_1.default.v4();
        const { rowCount } = await trx.query(`SELECT id FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
			WHERE lower(username) = lower($1)
			AND id != $2
			AND "deletedAt" IS NULL;`, [login.username, userId]);
        if (rowCount > 0) {
            app_logs_1.logger.info(`Admin user ${login.username} already exists, skipping...`);
            continue;
        }
        let hash = (0, dal_utils_1.saltHashPassword)(login.password);
        await trx.query(`INSERT INTO "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
			(id, username)
			VALUES($1, $2);`, [userId, login.username]);
        await trx.query(`INSERT INTO "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
			(id, "createdAt", "updatedAt", "accountEnabled", username, salt, "hashedPassword", settings)
			VALUES($1, now(), now(), $2, $3, $4, $5, $6);`, [userId, true, login.username, hash.salt, hash.passwordHash, JSON.stringify({ locale: "tr" })]);
        const userProfileId = uuid_1.default.v4();
        await trx.query(`INSERT INTO "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}"
			(id, "createdAt", "updatedAt", "userId", name, surname, email)
			VALUES($1, now(), now(), $2, $3, $4, $5);`, [userProfileId, userId, login.name, login.surname, login.email ?? null]);
        const userOrganizationId = uuid_1.default.v4();
        await trx.query(`INSERT INTO "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping}"
			(id, "userId", "organizationId")
			VALUES($1, $2, $3);`, [userOrganizationId, userId, data.organizationId]);
        let systemAdminRoleId = (await trx.query(`SELECT id FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			WHERE "typeId" = $1`, [predefined_roles_1.PredefinedRoles.SystemAdministrator.id])).rows[0].id;
        await trx.query(`INSERT INTO "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
			(id, "createdAt", "updatedAt", "userId", "organizationId", "roleId", "isDisabled", "extraPermissions", settings)
			VALUES($1, now(), now(), $2, $3, $4, $5, $6, $7);`, [
            userOrganizationId,
            userId,
            data.organizationId,
            systemAdminRoleId,
            false,
            null,
            JSON.stringify({
                notification: {
                    mediumSettings: {
                        general: {
                            email: false,
                            pushNotification: false,
                            sms: false,
                            web: true,
                        },
                    },
                },
            }),
        ]);
        const userOrganizationProfileId = uuid_1.default.v4();
        await trx.query(`INSERT INTO "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
			(id, "createdAt", "updatedAt", "userId", "userOrganizationId", name, surname, "phoneNumber", email, address, "uniqueId", "employmentStartUtc", 
				"employmentEndUtc", "previousServiceDuration", "extensionFields", "pacsEnabledRemainedAnnualPPermission", "birthDateUtc")
			VALUES($1, now(), now(), $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15);`, [
            userOrganizationProfileId,
            userId,
            userOrganizationId,
            login.name,
            login.surname,
            login.phoneNumber ?? null,
            login.email ?? null,
            null,
            uuid_1.default.v4(),
            null,
            null,
            null,
            null,
            null,
            null,
        ]);
        await trx.query(`INSERT INTO "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
			("id", "createdAt", "updatedAt", "userId", "accessControlPointId", "remoteAccess", "access", "read", "grant", "config", "snapshot")
			SELECT uuid_generate_v4(), now(), now(), $1, acp.id, true, true, true, true, true, true
			FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp`, [userId]);
    }
    const acpData = await trx.query(`SELECT id, name, location FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of acpData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
            row.location = row.location?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
		SET name = uacp.name, location = uacp.location
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name, data->>'location' as location
			FROM jsonb_array_elements($1::jsonb) as data
		) uacp
		WHERE acp.id = uacp.id`, [JSON.stringify(acpData.rows)]);
    const deviceData = await trx.query(`SELECT id, name, location, caption FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of deviceData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
            row.location = row.location?.replace(rgx, replacementRule.replacement);
            row.caption = row.caption?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}" d
		SET name = nd.name, location = nd.location, caption = nd.caption
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name, data->>'location' as location, data->>'caption' as caption
			FROM jsonb_array_elements($1::jsonb) as data
		) nd
		WHERE d.id = nd.id`, [JSON.stringify(deviceData.rows)]);
    const readerData = await trx.query(`SELECT id, name, location FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceReaders}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of readerData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
            row.location = row.location?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceReaders}" d
		SET name = nd.name, location = nd.location
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name, data->>'location' as location
			FROM jsonb_array_elements($1::jsonb) as data
		) nd
		WHERE d.id = nd.id`, [JSON.stringify(readerData.rows)]);
    const relayData = await trx.query(`SELECT id, name, location FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceRelays}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of relayData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
            row.location = row.location?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceRelays}" d
		SET name = nd.name, location = nd.location
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name, data->>'location' as location
			FROM jsonb_array_elements($1::jsonb) as data
		) nd
		WHERE d.id = nd.id`, [JSON.stringify(relayData.rows)]);
    const dryContactData = await trx.query(`SELECT id, name FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceDryContactInputs}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of dryContactData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceDryContactInputs}" d
		SET name = nd.name
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name
			FROM jsonb_array_elements($1::jsonb) as data
		) nd
		WHERE d.id = nd.id`, [JSON.stringify(dryContactData.rows)]);
    const wiegandData = await trx.query(`SELECT id, name, location FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceWiegandReaders}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of wiegandData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
            row.location = row.location?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceWiegandReaders}" d
		SET name = nd.name, location = nd.location
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name, data->>'location' as location
			FROM jsonb_array_elements($1::jsonb) as data
		) nd
		WHERE d.id = nd.id`, [JSON.stringify(wiegandData.rows)]);
    const cameraData = await trx.query(`SELECT id, name FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.cameras}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of cameraData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.cameras}" c
		SET name = nc.name
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name
			FROM jsonb_array_elements($1::jsonb) as data
		) nc
		WHERE c.id = nc.id`, [JSON.stringify(cameraData.rows)]);
    const mseriesData = await trx.query(`SELECT "deviceId", caption, extensions::text FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.controlPanelMSeriesSettings}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of mseriesData.rows) {
            row.caption = row.caption?.replace(rgx, replacementRule.replacement);
            row.extensions = row.extensions?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.controlPanelMSeriesSettings}" d
		SET caption = nd.caption, extensions = nd.extensions::jsonb
		FROM (
			SELECT (data->>'deviceId')::uuid as "deviceId", data->>'caption' as caption, (data->>'extensions')::json as extensions
			FROM jsonb_array_elements($1::jsonb) as data
		) nd
		WHERE d."deviceId" = nd."deviceId"`, [JSON.stringify(mseriesData.rows)]);
    const unitData = await trx.query(`SELECT id, name FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of unitData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" u
		SET name = uu.name
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name
			FROM jsonb_array_elements($1::jsonb) as data
		) uu
		WHERE u.id = uu.id`, [JSON.stringify(unitData.rows)]);
    const workplanData = await trx.query(`SELECT id, name FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of workplanData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}" wp
		SET name = nwp.name
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name
			FROM jsonb_array_elements($1::jsonb) as data
		) nwp
		WHERE wp.id = nwp.id`, [JSON.stringify(workplanData.rows)]);
    const regionData = await trx.query(`SELECT id, name FROM "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}";`);
    for (const replacementRule of data.entityNameReplacements) {
        const rgx = new RegExp(replacementRule.regex, replacementRule.regexOptions);
        for (const row of regionData.rows) {
            row.name = row.name?.replace(rgx, replacementRule.replacement);
        }
    }
    await trx.query(`UPDATE "${data.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" r
		SET name = nr.name
		FROM (
			SELECT (data->>'id')::uuid as id, data->>'name' as name
			FROM jsonb_array_elements($1::jsonb) as data
		) nr
		WHERE r.id = nr.id`, [JSON.stringify(regionData.rows)]);
}
exports.anonymizeOrganization = anonymizeOrganization;
