"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTerminalConnectionTimeout = exports.setTerminalConnectionTimeout = exports.getBase64Logo = exports.invalidateCacheMailTransporter = exports.getCacheMailTransporter = exports.getAvailableMediumsOfOrganizationForNotificationType = exports.setCacheTerminalConnectionStatus = exports.getCacheTerminalConnectionStatus = exports.invalidateOrganizationSettings = exports.invalidateUniqueNotificationIdOfOrganization = exports.getCacheUniqueNotificationIdOfOrganization = exports.getCacheOrganizationSettings = exports.initMemCache = void 0;
const nodemailer_1 = __importDefault(require("nodemailer"));
const app_config_1 = require("../../app.config");
const app_constants_1 = require("../../app.constants");
const app_logs_1 = require("../../app.logs");
const dal_manager_1 = require("../dal.manager");
const dal_utils_1 = require("../dal.utils");
const organizationSettings = {};
const organizationUniqueNotificationIds = {};
async function initMemCache() {
    const client = await (0, dal_utils_1.listenClientWrapperWithRetry)("MEM-CACHE", dal_manager_1.dbManager.poolMain);
    client.on("notification", (async (msg) => {
        let listener = null;
        switch (msg.channel) {
            case "organization_settings_changed":
                listener = (organizationId) => {
                    invalidateOrganizationSettings(organizationId);
                };
                break;
            default:
                break;
        }
        if (listener) {
            listener(msg.payload);
        }
    }).bind(this));
    client.on("error", (err) => {
        app_logs_1.logger.error("LISTEN MEM-CACHE connection error: " + err);
    });
    client.on("end", () => {
        app_logs_1.logger.info("LISTEN MEM-CACHE connection terminated... Restarting...");
        client.release();
        initMemCache();
    });
    client.query("LISTEN organization_settings_changed");
    app_logs_1.logger.info("LISTEN MEM-CACHE connections started.");
}
exports.initMemCache = initMemCache;
async function getCacheOrganizationSettings(organizationId, trx) {
    if (organizationSettings[organizationId]) {
        return organizationSettings[organizationId];
    }
    let settings = null;
    if (trx) {
        settings = await dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx, returnSmtpPassword: true });
    }
    else {
        settings = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            return dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx, returnSmtpPassword: true });
        });
    }
    if (!settings.notification.smtpSettings) {
        settings.notification.smtpSettings = app_config_1.appConfig.smtp;
    }
    organizationSettings[organizationId] = settings;
    return settings;
}
exports.getCacheOrganizationSettings = getCacheOrganizationSettings;
async function getCacheUniqueNotificationIdOfOrganization(organizationId, type, trx) {
    if (organizationUniqueNotificationIds[organizationId]) {
        return organizationUniqueNotificationIds[organizationId][type];
    }
    const ids = await dal_manager_1.dbManager.accessOrganization.getUniqueNotificationIdsOfOrganization({ organizationId, trx });
    organizationUniqueNotificationIds[organizationId] = ids;
    return ids[type];
}
exports.getCacheUniqueNotificationIdOfOrganization = getCacheUniqueNotificationIdOfOrganization;
function invalidateUniqueNotificationIdOfOrganization(organizationId) {
    if (organizationUniqueNotificationIds[organizationId]) {
        organizationUniqueNotificationIds[organizationId] = null;
    }
}
exports.invalidateUniqueNotificationIdOfOrganization = invalidateUniqueNotificationIdOfOrganization;
function invalidateOrganizationSettings(organizationId) {
    if (organizationSettings[organizationId]) {
        organizationSettings[organizationId] = null;
    }
    invalidateCacheMailTransporter(organizationId);
}
exports.invalidateOrganizationSettings = invalidateOrganizationSettings;
const terminalConnectionStatus = {};
function getCacheTerminalConnectionStatus(terminalId) {
    return terminalConnectionStatus[terminalId];
}
exports.getCacheTerminalConnectionStatus = getCacheTerminalConnectionStatus;
function setCacheTerminalConnectionStatus(terminalId, connected) {
    terminalConnectionStatus[terminalId] = connected;
}
exports.setCacheTerminalConnectionStatus = setCacheTerminalConnectionStatus;
async function getAvailableMediumsOfOrganizationForNotificationType(organizationId, notificationType, trx) {
    const settings = (await getCacheOrganizationSettings(organizationId, trx)).notification.mediumSettings;
    if (settings.custom) {
        return settings.custom[notificationType] ?? settings.general;
    }
    else {
        return settings.general;
    }
}
exports.getAvailableMediumsOfOrganizationForNotificationType = getAvailableMediumsOfOrganizationForNotificationType;
const organizationTransporter = {};
async function getCacheMailTransporter(organizationId, smtpSettings) {
    if (!organizationTransporter[organizationId]) {
        const transporter = nodemailer_1.default.createTransport({ ...smtpSettings.config, pool: true, maxConnections: 3 });
        await transporter.verify();
        organizationTransporter[organizationId] = transporter;
    }
    return organizationTransporter[organizationId];
}
exports.getCacheMailTransporter = getCacheMailTransporter;
function invalidateCacheMailTransporter(organizationId) {
    if (organizationTransporter[organizationId]) {
        organizationTransporter[organizationId] = null;
    }
}
exports.invalidateCacheMailTransporter = invalidateCacheMailTransporter;
async function getBase64Logo(organizationId, trx) {
    const organizationLogo = (await getCacheOrganizationSettings(organizationId, trx)).logo;
    return organizationLogo ?? app_config_1.appConfig.base64Logo ?? app_constants_1.armonBase64Logo;
}
exports.getBase64Logo = getBase64Logo;
const terminalConnectionTimeout = {};
function setTerminalConnectionTimeout(terminalId, timeout) {
    terminalConnectionTimeout[terminalId] = timeout;
}
exports.setTerminalConnectionTimeout = setTerminalConnectionTimeout;
function getTerminalConnectionTimeout(terminalId) {
    return terminalConnectionTimeout[terminalId];
}
exports.getTerminalConnectionTimeout = getTerminalConnectionTimeout;
