"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TabletConrollerV1 = void 0;
const express_1 = __importDefault(require("express"));
const crypto_1 = __importDefault(require("crypto"));
const api_util_1 = require("../../api/api.util");
const cli_queries_1 = require("../../dal/access/psql/cli-queries");
const dal_db_armon_schema_1 = require("../../dal/db/armon/dal.db.armon.schema");
const app_logs_1 = require("../../app.logs");
const app_constants_1 = require("../../app.constants");
const luxon_1 = require("luxon");
const app_enums_1 = require("../../app.enums");
const business_device_1 = require("../../business/business.device");
const uuid_1 = __importDefault(require("uuid"));
const dal_constants_1 = require("../../dal/dal.constants");
const dal_manager_1 = require("../../dal/dal.manager");
const dal_access_psql_log_1 = require("../../dal/access/psql/dal.access.psql.log");
const models_v1_1 = require("./models.v1");
const joi_1 = __importDefault(require("joi"));
const TAG = "[armon-tablet] ";
class TabletConrollerV1 {
    constructor(app) {
        this._router = express_1.default.Router();
        this._router.get("/", (req, res, next) => {
            res.status(200).send("OK");
        });
        this._router.post("/auth/register", (0, api_util_1.globalRouteHandler)(this.registerTablet));
        this._router.post("/auth/checkapprovement", (0, api_util_1.globalRouteHandler)(this.checkApprovement));
        this._router.post("/accesscontrolpoints", this.validateTablet, (0, api_util_1.globalRouteHandler)(this.listAccessControlPoints));
        this._router.post("/cardread", this.validateTablet, (0, api_util_1.globalRouteHandler)(this.cardRead));
        app.use("/tablet/v1", this._router);
    }
    async registerTablet(req, res, next) {
        const validation = models_v1_1.TabletRegisterRequestJoiValidation.validate(req.body, {
            allowUnknown: true,
        });
        if (validation.error) {
            throw validation.error;
        }
        const request = validation.value;
        const { id, publicKey, referenceCode, approvementCode, orgShortCode } = request;
        await (0, cli_queries_1.systemTransaction)(dal_manager_1.dbManager.poolMain, async (trx) => {
            const organizaitonIdDbResult = await trx.query(`
				SELECT id AS "organizationId"
				FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"
				WHERE code = $1
				`, [orgShortCode]);
            if (!organizaitonIdDbResult.rowCount) {
                const errorLog = `Organization with short code ${orgShortCode} not found!`;
                app_logs_1.logger.error(TAG + errorLog);
                res.status(404).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.ORGANIZATION_NOT_FOUND });
                return;
            }
            const organizaitonId = organizaitonIdDbResult.rows[0].organizationId;
            const duplicateCheckDbResult = await trx.query(`
				SELECT id, "creationT", name, "publicKey", "referenceCode", "approvementCode", "isApproved"
				FROM "${organizaitonId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CheckInTablet}"
				WHERE "id" = $1
					OR "publicKey" = $2
					OR "referenceCode" = $3
					OR "approvementCode" = $4
				`, [id, publicKey, referenceCode, approvementCode]);
            if (duplicateCheckDbResult.rowCount) {
                const errorLog = `One of the supplied parameters already exists in database!`;
                app_logs_1.logger.error(TAG + errorLog);
                res.status(409).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.TABLET_INFO_DUPLICATE });
                return;
            }
            else {
                await trx.query(`
					INSERT INTO "${organizaitonId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CheckInTablet}"
					(id, "creationT", "publicKey", "referenceCode", "approvementCode", "isApproved")
					VALUES($1, now(), $2, $3, $4, false);
					`, [id, publicKey, referenceCode, approvementCode]);
                res.status(200).send({
                    success: true,
                });
            }
        });
    }
    async checkApprovement(req, res, next) {
        const validation = models_v1_1.TabletCheckApprovementRequestJoiValidation.validate(req.body, {
            allowUnknown: true,
        });
        if (validation.error) {
            throw validation.error;
        }
        const request = validation.value;
        const { id, orgShortCode } = request;
        await (0, cli_queries_1.systemTransaction)(dal_manager_1.dbManager.poolMain, async (trx) => {
            const organizationIdDbResult = await trx.query(`
				SELECT id AS "organizationId"
				FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"
				WHERE code = $1
				`, [orgShortCode]);
            if (!organizationIdDbResult.rowCount) {
                const errorLog = `Organization with short code ${orgShortCode} not found!`;
                app_logs_1.logger.error(TAG + errorLog);
                res.status(404).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.ORGANIZATION_NOT_FOUND });
                return;
            }
            const organizationId = organizationIdDbResult.rows[0].organizationId;
            const approvementCheckDbResult = await trx.query(`
				SELECT "id", "isApproved"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CheckInTablet}"
				WHERE "id" = $1
				`, [id]);
            if (!approvementCheckDbResult.rowCount) {
                const errorLog = `Tablet with id ${id} is not registered to this system.`;
                app_logs_1.logger.error(TAG + errorLog);
                res.status(404).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.TABLET_NOT_FOUND });
                return;
            }
            else if (approvementCheckDbResult.rows[0].isApproved) {
                let organizationLogo = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoFile(organizationId);
                let organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(organizationId);
                res.status(200).send({
                    organizationName: organization.name,
                    organizationId: organization.id,
                    publicKey: organization.publicKey,
                    logo: organizationLogo ? Buffer.from(organizationLogo).toString() : undefined,
                });
            }
            else {
                const errorLog = `Tablet with id ${id} is not approved yet.`;
                app_logs_1.logger.error(TAG + errorLog);
                res.status(401).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.TABLET_NOT_APPROVED_YET });
            }
        });
    }
    async validateTablet(req, res, next) {
        const validation = joi_1.default.object({
            [app_constants_1.ArmonHeaders.X_Armon_DeviceId]: joi_1.default.string().uuid().required(),
            [app_constants_1.ArmonHeaders.X_Armon_OrganizationId]: joi_1.default.string().uuid().required(),
            [app_constants_1.ArmonHeaders.X_Armon_Timestamp]: joi_1.default.string().length(14).required(),
            [app_constants_1.ArmonHeaders.X_Armon_Signature]: joi_1.default.string().alphanum().required(),
        }).validate(req.headers, {
            allowUnknown: true,
        });
        if (validation.error) {
            throw validation.error;
        }
        const optime = luxon_1.DateTime.now();
        const tabletId = req.header(app_constants_1.ArmonHeaders.X_Armon_DeviceId);
        const timestamp = req.header(app_constants_1.ArmonHeaders.X_Armon_Timestamp);
        const signature = req.header(app_constants_1.ArmonHeaders.X_Armon_Signature);
        const organizationId = req.header(app_constants_1.ArmonHeaders.X_Armon_OrganizationId);
        let sharedSecret = null;
        await (0, cli_queries_1.systemTransaction)(dal_manager_1.dbManager.poolMain, async (trx) => {
            const devicePublicKeyDbResult = await trx.query(`
				SELECT id, "publicKey"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.CheckInTablet}"
				WHERE id = $1
				`, [tabletId]);
            const organizationPrivateKeyDbResult = await trx.query(`
				SELECT "privateKey"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
				WHERE id = $1
				`, [organizationId]);
            let ecdh = crypto_1.default.createECDH("prime256v1");
            ecdh.setPrivateKey(organizationPrivateKeyDbResult.rows[0].privateKey, "base64");
            sharedSecret = ecdh.computeSecret(devicePublicKeyDbResult.rows[0].publicKey, "base64");
        });
        const computedSignature = crypto_1.default
            .createHash("sha256")
            .update(Buffer.concat([Buffer.from(tabletId + timestamp), sharedSecret]))
            .digest("hex");
        if (computedSignature.toLowerCase() === signature.toLowerCase()) {
            if (Math.abs(luxon_1.DateTime.fromFormat(timestamp, "ddMMyyyyhhmmss").diff(optime, "seconds").seconds) <= 5) {
                req.organizationId = organizationId;
                req.tabletId = tabletId;
                next();
            }
            else {
                const errorLog = `Request time mismatch`;
                app_logs_1.logger.error(TAG + errorLog);
                res.status(403).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.REQUEST_TIME_MISMATCH });
            }
        }
        else {
            const errorLog = `Signature validation failed`;
            app_logs_1.logger.error(TAG + errorLog);
            res.status(403).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.TABLET_SIGNATURE_MISMATCH });
        }
    }
    async listAccessControlPoints(req, res, next) {
        const validation = models_v1_1.TabletACPListRequestJoiValidation.validate(req.body, {
            allowUnknown: true,
        });
        if (validation.error) {
            throw validation.error;
        }
        const request = validation.value;
        const { pagination } = request;
        const organizationId = req.organizationId;
        const tabletId = req.tabletId;
        let total = 0;
        const accessControlPointsDbResult = await (0, cli_queries_1.systemTransaction)(dal_manager_1.dbManager.poolMain, async (trx) => {
            const acpCountDbResult = await trx.query(`
				SELECT count(*)::integer AS c FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" WHERE "deletedAt" IS NULL
				`);
            total = acpCountDbResult.rows[0].c;
            return trx.query(`
				SELECT 
					acp.id, 
					acp.name, 
					acp."accessControlPointType" AS type, 
					COALESCE(JSON_AGG (JSONB_BUILD_OBJECT(
						'relayNumber', dr.number,
						'direction', dr.direction
					)) FILTER (WHERE dr.number IS NOT NULL), NULL, '[]') AS relays
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.deviceRelays}" dr
					ON dr."deletedAt" IS NULL AND dr."accessControlPointId" = acp.id
				WHERE acp."deletedAt" IS NULL
				GROUP BY (acp.name, acp.id, acp."accessControlPointType")
				ORDER BY acp.name
				OFFSET $1 LIMIT $2
				`, [pagination.skip, pagination.take]);
        });
        res.status(200).json({
            pagination: {
                take: pagination.take,
                skip: pagination.skip,
                total,
            },
            accessControlPoints: accessControlPointsDbResult.rows,
        });
    }
    async cardRead(req, res, next) {
        const validation = models_v1_1.TabletCardReadRequestJoiValidation.validate(req.body, {
            allowUnknown: true,
        });
        if (validation.error) {
            throw validation.error;
        }
        const request = validation.value;
        const { cardData, accessControlPointId, direction, location } = request;
        const organizationId = req.organizationId;
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            const credentialDbResult = await trx.query(`
					SELECT id, "userId", "expiresOn" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
					WHERE "deletedAt" IS NULL
						AND "data" = $1
						AND type = $2
					`, [cardData, app_enums_1.enums.CredentialType.MiFare]);
            if (!credentialDbResult.rowCount) {
                const errorLog = `Credential data (${cardData}) not found on database`;
                app_logs_1.logger.error(errorLog);
                app_logs_1.logger.error(TAG + errorLog);
                res.status(404).json({ success: false, message: errorLog, errorCode: models_v1_1.ErrorCodes.USER_NOT_FOUND });
                return;
            }
            const logItem = await (0, business_device_1.updateLogFromMobileDevice)(organizationId, {
                id: uuid_1.default.v4(),
                v: false,
                o: credentialDbResult.rows[0].userId,
                a: accessControlPointId,
                u: new Date().toISOString(),
                c: cardData,
                cx: [
                    {
                        t: app_enums_1.enums.CredentialType.MiFare,
                        d: cardData,
                        i: credentialDbResult.rows[0].id,
                    },
                ],
                d: direction,
                oId: organizationId,
                r: null,
                s: null,
                ul: null,
                rm: app_enums_1.enums.LogReceiveMethod.Mobile,
                lt: location?.latitude,
                ln: location?.longitude,
                lr: location?.isReliable,
            });
            const insertedLog = await dal_manager_1.dbManager.accessLog.addAccessLogFromDevice(organizationId, logItem, trx);
            let now = new Date();
            if (insertedLog.rg && insertedLog.rg.length > 0) {
                for (const regionLog of insertedLog.rg) {
                    let regionNotification = {
                        i: regionLog.i,
                        s: regionLog.s.valueOf(),
                        ee: regionLog.ee,
                        xe: regionLog.xe,
                        o: insertedLog.o,
                        t: insertedLog.u,
                    };
                    (0, cli_queries_1.upsertRegionStates)(organizationId, [{ accessControlPointId: insertedLog.a, log: regionNotification }], trx);
                    if (regionLog.ru && regionLog.ru.c) {
                        (0, cli_queries_1.upsertAccessRuleHistory)(organizationId, {
                            date: now,
                            accessRuleId: regionLog.ru.i,
                            userId: insertedLog.o,
                            count: regionLog.ru.c,
                            accessControlPointId: insertedLog.a,
                            actionDateISO: insertedLog.u,
                        }, trx);
                    }
                    if (regionLog.rti) {
                        (0, cli_queries_1.updateUserRegionTicketUnits)(organizationId, { userId: insertedLog.o, accessControlPointId: insertedLog.a, decrementUnit: regionLog.rti.c, regionTicketId: regionLog.rti.i }, trx);
                    }
                }
            }
            await dal_manager_1.dbManager.accessPacs2.onAccessLogsChanged({
                reason: app_enums_1.enums.RecalculateWorkReason.NewAccessLog,
                organizationId: organizationId,
                userId: logItem.o,
                acpId: logItem.a,
                timestamp: new Date(logItem.u),
                direction: logItem.d,
            });
            if (insertedLog.d && insertedLog.d === app_enums_1.enums.AccessDirection.Entrance && insertedLog.s === true) {
                await (0, dal_access_psql_log_1.assignAutoShift)(organizationId, {
                    credentialOwnerUserId: insertedLog.o,
                    generationTime: insertedLog.u,
                    redisCache: dal_manager_1.dbManager.accessRedisCache,
                    logId: insertedLog.id,
                });
            }
            res.json({
                userId: logItem?.o,
                fullname: logItem?.on,
                captionLines: logItem?.o ? await dal_manager_1.dbManager.accessUser.getSingleUserOrganizationCaptionLines(organizationId, logItem?.o) : undefined,
                reason: logItem?.r,
                success: logItem?.s,
            });
            return;
        }, dal_constants_1.DalConstants.SystemUserId, organizationId);
    }
}
exports.TabletConrollerV1 = TabletConrollerV1;
