"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const elastic_apm_node_1 = __importDefault(require("elastic-apm-node"));
if (process.env.ELASTIC_APM_SERVER_URL &&
    process.env.ELASTIC_APM_SECRET_TOKEN &&
    process.env.ELASTIC_APM_PREFIX &&
    ((process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true" && process.env.ELASTIC_APM_SERVER_CA_CERT_FILE) || process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "false")) {
    elastic_apm_node_1.default.start({
        serviceName: `${process.env.ELASTIC_APM_PREFIX}-pgredis`,
        serverUrl: process.env.ELASTIC_APM_SERVER_URL,
        secretToken: process.env.ELASTIC_APM_SECRET_TOKEN,
        environment: "production",
        serverCaCertFile: process.env.ELASTIC_APM_SERVER_CA_CERT_FILE ?? null,
        verifyServerCert: process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true",
    });
}
else {
    console.warn(`Elastic APM is not enabled. Please set ELASTIC_APM_SERVER_URL, ELASTIC_APM_SECRET_TOKEN, ELASTIC_APM_PREFIX, ELASTIC_APM_VERIFY_SERVER_CERT, and ELASTIC_APM_SERVER_CA_CERT_FILE environment variables.`);
}
const app_config_1 = require("../../app.config");
const app_constants_1 = require("../../app.constants");
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const dal_constants_1 = require("../../dal/dal.constants");
const dal_manager_1 = require("../../dal/dal.manager");
const messageBroker_manager_1 = require("../../messageBroker/messageBroker.manager");
const dal_utils_1 = require("../../dal/dal.utils");
async function start() {
    try {
        app_config_1.appConfig.init(app_constants_1.ServiceNames.PGRedisService);
        app_logs_1.appLogger.init(app_config_1.appConfig.appLogDirectory);
        app_logs_1.logger.info("DBs are initializing...");
        await dal_manager_1.dbManager.init({
            main: app_config_1.appConfig.db.main,
            log: app_config_1.appConfig.db.log,
            environment: app_config_1.appConfig.nodeEnv,
            logDirectory: app_config_1.appConfig.dbLogDirectory,
            redis: app_config_1.appConfig.db.redis,
            targetMigration: null,
        }, false, true, false, false);
        await messageBroker_manager_1.messageBrokerManager.initForPgRedisService();
        app_logs_1.logger.info("Redis is initializing...");
        await listen();
        app_logs_1.logger.info("armon-pgredis started");
    }
    catch (error) {
        app_logs_1.logger.error(error || error.message);
        process.exit(1);
    }
}
const listen = async () => {
    const client = await (0, dal_utils_1.listenClientWrapperWithRetry)("PG-REDIS", dal_manager_1.dbManager.poolMain);
    client.on("notification", async (msg) => {
        switch (msg.channel) {
            case "terminal_notification_sensor":
                {
                    const data = JSON.parse(msg.payload);
                    let oldHash = null;
                    let newHash = null;
                    if (data.new) {
                        if (data.new.sensorType === app_enums_1.enums.NotificationSensorType.Status) {
                            const hashEntropy = {
                                ...data.new.options,
                                statusSensorId: data.new.deviceDryContactInputId,
                                accessControlPointId: data.new.accessControlPointId,
                            };
                            newHash = dal_manager_1.dbManager.accessNotifications.generateHashKeyFromEntropy(hashEntropy, app_enums_1.enums.NotificationType.StatusSensor);
                        }
                        else if (data.new.sensorType === app_enums_1.enums.NotificationSensorType.Counter) {
                            const hashEntropy = {
                                ...data.new.options,
                                counterSensorId: data.new.deviceDryContactInputId,
                                accessControlPointId: data.new.accessControlPointId,
                            };
                            newHash = dal_manager_1.dbManager.accessNotifications.generateHashKeyFromEntropy(hashEntropy, app_enums_1.enums.NotificationType.CounterSensor);
                        }
                    }
                    if (data.old) {
                        if (data.old.sensorType === app_enums_1.enums.NotificationSensorType.Status) {
                            const hashEntropy = {
                                ...data.old.options,
                                statusSensorId: data.old.deviceDryContactInputId,
                                accessControlPointId: data.old.accessControlPointId,
                            };
                            oldHash = dal_manager_1.dbManager.accessNotifications.generateHashKeyFromEntropy(hashEntropy, app_enums_1.enums.NotificationType.StatusSensor);
                        }
                        else if (data.old.sensorType === app_enums_1.enums.NotificationSensorType.Counter) {
                            const hashEntropy = {
                                ...data.old.options,
                                counterSensorId: data.old.deviceDryContactInputId,
                                accessControlPointId: data.old.accessControlPointId,
                            };
                            oldHash = dal_manager_1.dbManager.accessNotifications.generateHashKeyFromEntropy(hashEntropy, app_enums_1.enums.NotificationType.CounterSensor);
                        }
                    }
                    if (!newHash || !oldHash || newHash !== oldHash) {
                        dal_manager_1.dbManager.accessNotifications.registerTerminalGeneratedNotificationToCache(data.new?.id, newHash);
                        dal_manager_1.dbManager.accessNotifications.unregisterTerminalGeneratedNotificationFromCache(data.old?.id, oldHash);
                    }
                }
                break;
            case "terminal_notification_access":
                {
                    const data = JSON.parse(msg.payload);
                    let oldHash = null;
                    let newHash = null;
                    if (data.new) {
                        const hashEntropy = {
                            direction: data.new.options?.direction ?? null,
                            success: data.new.options?.success ?? null,
                            regionId: data.new.regionId,
                            targetUserId: data.new.targetUserId,
                        };
                        newHash = dal_manager_1.dbManager.accessNotifications.generateHashKeyFromEntropy(hashEntropy, app_enums_1.enums.NotificationType.IdentityAccess);
                    }
                    if (data.old) {
                        const hashEntropy = {
                            direction: data.old.options?.direction ?? null,
                            success: data.old.options?.success ?? null,
                            regionId: data.old.regionId,
                            targetUserId: data.old.targetUserId,
                        };
                        oldHash = dal_manager_1.dbManager.accessNotifications.generateHashKeyFromEntropy(hashEntropy, app_enums_1.enums.NotificationType.IdentityAccess);
                    }
                    if (!newHash || !oldHash || newHash !== oldHash) {
                        dal_manager_1.dbManager.accessNotifications.registerTerminalGeneratedNotificationToCache(data.new?.id, newHash);
                        dal_manager_1.dbManager.accessNotifications.unregisterTerminalGeneratedNotificationFromCache(data.old?.id, oldHash);
                    }
                }
                break;
            case "update_terminal_state_cache":
                {
                    const data = JSON.parse(msg.payload);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.CameraProxyTerminal, data.id);
                }
                break;
            case "update_device_state_cache":
                {
                    const data = JSON.parse(msg.payload);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.DeviceState, data.id);
                }
                break;
            case "update_device_notification_info_cache":
                {
                    const data = JSON.parse(msg.payload);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.DeviceNotificationInfo, data.id);
                }
                break;
            case "update_access_point_info_cache":
                {
                    const data = JSON.parse(msg.payload);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.AccessPoint, data.id);
                }
                break;
            case "update_user_info_cache":
                {
                    const data = JSON.parse(msg.payload);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation, data.id);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.UserBadge, data.id);
                }
                break;
            case "organization_settings_changed":
                {
                    const organizationId = msg.payload;
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(organizationId, dal_constants_1.DalConstants.NewDataCacheType.Organization, organizationId);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(organizationId, dal_constants_1.DalConstants.NewDataCacheType.SnapshotSettings, organizationId);
                }
                break;
            case "update_region_info_cache":
                {
                    const data = JSON.parse(msg.payload);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.Region, data.id);
                }
                break;
            case "organization_unit_delete_update":
                {
                    const data = JSON.parse(msg.payload);
                    const unitUserMappings = await dal_manager_1.dbManager.accessOrganizationUnit.listUsersWithOrganizationUnits(data.organizationId, [data.organizationUnitId]);
                    if (unitUserMappings[0] && unitUserMappings[0].userIds?.length) {
                        await Promise.all(unitUserMappings[0].userIds.map(async (userId) => {
                            dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation, userId);
                            dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.UserBadge, userId);
                        }));
                    }
                }
                break;
            case "auto_shift_rule_change_event":
                {
                    const data = JSON.parse(msg.payload);
                    await dal_manager_1.dbManager.accessRedisCache.deleteDirtyCachedData(data.organizationId, dal_constants_1.DalConstants.NewDataCacheType.ShiftRule, data.organizationId);
                }
                break;
            default:
                return;
        }
    });
    client.on("error", (err) => {
        app_logs_1.logger.error("Error on PG-REDIS listener: " + err);
    });
    client.on("end", () => {
        app_logs_1.logger.info("pg listener for PG-REDIS ended, restarting...");
        client.release();
        listen();
    });
    await Promise.all([
        client.query("LISTEN update_region_info_cache"),
        client.query("LISTEN update_user_info_cache"),
        client.query("LISTEN terminal_notification_sensor"),
        client.query("LISTEN terminal_notification_access"),
        client.query("LISTEN update_device_state_cache"),
        client.query("LISTEN update_device_notification_info_cache"),
        client.query("LISTEN update_access_point_info_cache"),
        client.query("LISTEN organization_settings_changed"),
        client.query("LISTEN organization_unit_delete_update"),
        client.query("LISTEN auto_shift_rule_change_event"),
    ]);
    app_logs_1.logger.info("LISTEN PG-REDIS connections started.");
};
start();
